#ifndef SWU_COMMON_PLATFORM_WDG_H
#define SWU_COMMON_PLATFORM_WDG_H

#include <iostream>
#include <fstream>
#include "asf/core/Timer.h"


namespace ai_sw_update {
namespace common {


/**
 * \brief Platform watchdog control
 *
 * Controls (starts, stops) the watchdog device (/dev/watchdog).
 * In Application Mode this is done by Systemd.
 * In Recovery Mode for SW Update this is done by this class.
 *
 * Common documentation: http://fe0vm142.de.bosch.com/cgi-bin/gitweb.cgi?p=linux-2.6-imx.git;a=blob;f=Documentation/watchdog/watchdog-api.txt;h=eb7132ed8bbcb7650bdc20e6a43e3c9222d6621b;hb=refs/heads/linux-38-gen3
 * Code example: http://fe0vm142.de.bosch.com/cgi-bin/gitweb.cgi?p=linux-2.6-imx.git;a=tree;f=Documentation/watchdog/src;h=cadb6c3a23d8bafbc76c5d4b2f0975b39d68c4a5;hb=refs/heads/linux-38-gen3
 */
class WDGControl
: public ::asf::core::TimerCallbackIF
{
public:
	WDGControl();
	~WDGControl();

    /**
     * Default watchdog trigger interval in ms.
     * 16.06.14: V850 supervision will trigger a system reset in cause it did not receive any life sign for 10 seconds.
     */
	static const uint32 DEFAULT_WATCHDOG_TRIGGER_INTERVAL_MS = 1000;

	/**
	 * Start periodic watchdog triggering
	 * @param inIntervalMS Interval to trigger watchdog in ms.
	 */
	void startTriggering(const uint32 inIntervalMS=DEFAULT_WATCHDOG_TRIGGER_INTERVAL_MS);

	/**
	 * Stop periodic watchdog triggering
	 */
	void stopTriggering();

	/**
	 * Expired timer callback
	 * @param timer ID of timer for distinguishing multiple timers (not used here).
	 * @param data User specific data attached to timer (not used here).
	 */
	virtual void onExpired(::asf::core::Timer& timer, boost::shared_ptr< ::asf::core::TimerPayload > data);

private:

	::asf::core::Timer _wdgTimer; //!< ASF timer
	std::ofstream _wdgDevice;     //!< Watchdog device file
};

} // namespace ai_sw_update {
} // namespace common {


#endif
