/******************************************************************
 *FILE: mcp_fsmMCPUpdActions.cpp
 *SW-COMPONENT: SWUPDATE
 *DESCRIPTION: FSM for MCP-Updater
 *COPYRIGHT: © 2017 Robert Bosch GmbH
 *
 *The reproduction, distribution and utilization of this file as
 *well as the communication of its contents to others without express
 *authorization is prohibited. Offenders will be held liable for the
 *payment of damages. All rights reserved in the event of the grant
 *of a patent, utility model or design.
 
* History:
* 31.12.2019 - Initial version - RamaGopalReddy.Bommireddy@in.bosch.com
* 28.02.2020 - Revision 1.0    - Surparaju.Pavankumar@in.bosch.com
 ******************************************************************/

//--------------------------------------------------------------------------------------------
// includes
//--------------------------------------------------------------------------------------------
#include <string>
#include <stdlib.h>
#include <fstream>
#include <iostream>
#include <iterator>
#include <algorithm>
#include <vector>
#include <boost/assign/list_of.hpp>
#include <string.h>
#include "mcp_fsmMCPUpdActions.h"
#include "mcp_SecurityAlgorithm.h"

#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_MCP
#define ETG_I_FILE_PREFIX 				mcp_fsmMCPUpdActions::
#include "trcGenProj/Header/mcp_fsmMCPUpdActions.cpp.trc.h"
#endif

#define ETG_ENABLED
#ifndef __SW_UPDATE_UNIT_TESTING__
#include "trace_interface.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
/* Needed for Trace */
#define ETG_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etg_if.h"
#endif

#include <unistd.h>

const char *strenMcpCtrlState[] = {
		"MCP APPCTRL EXIT SUCCESS",
		"MCP APPCTRL UPDATE SUCCESS",
		"MCP APPCTRL VERSION OK",
		"MCP APPCTRL VERSION ERROR",
		"MCP APPCTRL SWUPDATE PARSE CMDARGS FAILED",
		"MCP APPCTRL NULL POINTER ERROR",
		"MCP APPCTRL CLIENT THREAD CREATION FAILED",
		"MCP DEFAULT SESSION FAILED",
		"MCP EXTENDED SESSION FAILED",
		"MCP PROGRAMMING SESSION FAILED",
		"MCP REQUEST SEED FAILED",
		"MCP REQUEST KEY FAILED",
		"MCP SENT NEW VERSION FAILED",
		"MCP REQUEST ERASE MEMORY FAILED",
		"MCP REQUEST DOWNLOAD FAILED",
		"MCP TRANSMIT DATA FAILED",
		"MCP TIMEOUT FAILED",
		"MCP RQUEST LENGTH ERROR",
		"MCP 10SEC TIMER IS ZERO",
		"MCP NOT IN PROGRAMMING SESSION",
		"MCP KEY IS NOT MATCHED",
		"MCP DID NOT MATCHED",
		"MCP NO SECUTIRY ACCESS",
		"MCP SUB FUNCTION IS NOT CORRECT",
		"MCP INSTALLED VERSION IS HIGHER",
		"MCP UPDATE ACTIONS IS NULLPTR",
		"MCP MAX RETRIES REACHED",
		"MCP APPCTRL NONE"
};

#ifndef __SW_UPDATE_UNIT_TESTING__
// Implementation of the methods getInstanceOfmcp_fsmMCPUpdActions and releaseInstanceOfmcp_fsmMCPUpdActions
// In case of __SW_UPDATE_UNIT_TESTING__ they are implemented in mock!
SWU_IMPL_GET_AND_RELEASE_INSTANCE(mcp_fsmMCPUpdActions);
#endif

// Implementation of the singleton methods
SWU_IMPL_SINGLETON(mcp_fsmMCPUpdActions);

/************************************************************************
 * FUNCTION: mcp_fsmMCPUpdActions()
 *
 * DESCRIPTION: Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/	

mcp_fsmMCPUpdActions::mcp_fsmMCPUpdActions():
   pFsm(nullptr),
   bufUDSResponse(),
   bufUDSRequest(),
   bufUDSRequestCopy(),
   bufResultBuffer(),
   bufResultBuffer_Negative(),
   iopKeyArray{0x00},
   u32Length(0),
   pmcp_FileHandler(nullptr),
   pmcpCsmProxy(nullptr),
   pmcpUpdaterProxy(nullptr),
   m_updatecompleted(false),
   iNoOfTimeout(0),
   isNRC27Received(false),
   lastPreDistributeSuccess(false),
   updateFailed(false),
   bUDSRequestFailed(false),
   NRC27Count(0),
   iNoOfTrials(0),
   iErrorCode(ENMCPCTRLSTATE::MCP_APPCTRL_NONE),
   pRequestSendReq(nullptr),
   u16LengthSendReq(0),
   sendrequestRetries(0)
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::Constructor"));
   ETG_I_REGISTER_FILE();

   // Create the state machine object
   pFsm = new swupd_fsmMCPUpd::Fsm(this);
   //For Initializing the State Machine
   if ( pFsm != nullptr)
   {
      pFsm->init();
   }
   pmcpCsmProxy = new mcp_CsmProxy();
   
   if(pmcpCsmProxy!=nullptr)
	   pmcpCsmProxy->vInit();

   pmcp_FileHandler = getInstanceOfmcp_FileHandler();
   pmcpUpdaterProxy = getInstanceOfmcp_UpdaterProxy();
}

/************************************************************************
 * FUNCTION: ~mcp_fsmMCPUpdActions()
 *
 * DESCRIPTION: Destruction
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
mcp_fsmMCPUpdActions::~mcp_fsmMCPUpdActions()
{
   ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions Destructor !!"));

   if(pmcpUpdaterProxy != nullptr){
	  ETG_TRACE_USR4(("MCPINFO:mcpUpdaterProxy destructor called"));
	  releaseInstanceOfmcp_UpdaterProxy();
	  pmcpUpdaterProxy = nullptr;
   }
   
   if(pmcp_FileHandler != nullptr){
	  ETG_TRACE_USR4(("MCPINFO:mcp_FileHandler destructor called"));
	  releaseInstanceOfmcp_FileHandler();
	  pmcp_FileHandler = nullptr;
   }
   
   if(pmcpCsmProxy!=nullptr)
   {
	  ETG_TRACE_USR4(("MCPINFO:pmcpCsmProxy vDeInit called"));
	  pmcpCsmProxy->vDeInit();
	  delete pmcpCsmProxy;
	  pmcpCsmProxy = nullptr;   
   }

   if(pFsm != nullptr)
   {
	ETG_TRACE_USR4(("MCPINFO:pFsm destructor called"));
	delete pFsm;
	pFsm = nullptr;
   }

}

/************************************************************************
 * FUNCTION: vAcceptEvent()
 *
 * DESCRIPTION: Accepts the FSM event
 *
 * PARAMETER: swupd_fsmMCPUpd::FsmEvent,void*
 *
 * RETURNVALUE: None
 *************************************************************************/

void mcp_fsmMCPUpdActions::vAcceptEvent ( swupd_fsmMCPUpd::FsmEvent event, void* pArg )
{
   ETG_TRACE_USR2(("MCPINFO:mcp_fsmMCPUpdActions::vAcceptEvent(event = %d,pArg = %p)",event,pArg));
   vAcceptEventInternal(event, pArg);
}

/************************************************************************
 * FUNCTION: vAcceptEventInternal()
 *
 * DESCRIPTION: Accepts the FSM event
 *
 * PARAMETER: swupd_fsmMCPUpd::FsmEvent,void*
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::vAcceptEventInternal ( swupd_fsmMCPUpd::FsmEvent event, void* pArg )
{
   if(pFsm != nullptr)
   {
      ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::vAcceptEventInternal event=%s",getEventName(event)));
      actPrintCurrentState(pArg);
      pFsm->acceptEvent(event, pArg);
   }
}

/************************************************************************
 * FUNCTION: vGetStateName()
 *
 * DESCRIPTION: Get the current state name
 *
 * PARAMETER: None
 *
 * RETURNVALUE: const char*
 *************************************************************************/
const char* mcp_fsmMCPUpdActions::vGetStateName ( void )
{
   static const char empty[] = "";
   if(pFsm != nullptr)
   {
      return pFsm->getStateName();
   }
   else
   {
      return empty;
   }
}

/************************************************************************
 * FUNCTION: vUdsMessageReceived()
 *
 * DESCRIPTION: Receives the UDS message
 *
 * PARAMETER: tPCU8,tU32
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::vUdsMessageReceived(tPCU8 bufUDSMessage, tU32 u32Length)
{
   ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::vUdsMessageReceived(u32Length = %u)",u32Length));
   bufUDSResponse.clear();
   std::copy(&bufUDSMessage[0], &bufUDSMessage[u32Length], back_inserter(bufUDSResponse));
   ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::vUdsMessageReceived bufUDSResponse vector"));
   vPrintVector(bufUDSResponse,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
}

/************************************************************************
 * FUNCTION: vPrintVector()
 *
 * DESCRIPTION: Prints the vector data
 *
 * PARAMETER: std::vector<unsigned char> , int
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::vPrintVector (std::vector<unsigned char> buffer, int printOnLevel)
{
   switch (printOnLevel)
   {
      case PRINT_VECTOR_ON_DEBUG_LEVEL_2:
         ETG_TRACE_USR2(("MCPINFO: mcp_fsmMCPUpdActions::vPrintVector(...,%d)",printOnLevel));
         break;
      case PRINT_VECTOR_ON_DEBUG_LEVEL_4:
         ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::vPrintVector(...,%d)",printOnLevel));
         break;
      default:
         break;
   }

   ETG_TRACE_USR4(("HEX="));
   for (std::vector<unsigned char>::iterator it=buffer.begin(); it!=buffer.end(); ++it)
   {
      ETG_TRACE_USR4(("%02x", *it));
   }
   
   ETG_TRACE_USR4(("ASCII="));
   for (std::vector<unsigned char>::iterator it=buffer.begin(); it!=buffer.end(); ++it)
   {
      ETG_TRACE_USR4(("%c", *it));
   }

   std::string stringBuffer(buffer.begin(), buffer.end());
   switch (printOnLevel)
   {
      case PRINT_VECTOR_ON_DEBUG_LEVEL_2:
         ETG_TRACE_USR2(("ASCII=%s", stringBuffer.c_str()));
         ETG_TRACE_USR2(("HEX=%02x", ETG_LIST_LEN(buffer.size()), ETG_LIST_PTR_T8(&buffer[0])));
         break;
      case PRINT_VECTOR_ON_DEBUG_LEVEL_4:
         ETG_TRACE_USR4(("ASCII=%s", stringBuffer.c_str()));
         ETG_TRACE_USR4(("HEX=%02x", ETG_LIST_LEN(buffer.size()), ETG_LIST_PTR_T8(&buffer[0])));
         break;
      default:
         break;
   }

}

/************************************************************************
 * FUNCTION: bRequestEqualToResponse()
 *
 * DESCRIPTION: Compares the two vectors data
 *
 * PARAMETER: std::vector<unsigned char>,std::vector<unsigned char>,tU32
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::bRequestEqualToResponse(std::vector<unsigned char> request, std::vector<unsigned char> response, tU32 u32Length )
{
   ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::bRequestEqualToResponse"));
   // The comparsion cannot be done for the bytes before BUF_POS_CMD_1
   if( equal(request.begin(), request.begin()+u32Length, response.begin()) == false)
   {
	ETG_TRACE_FATAL(("MCPFATAL: bRequestEqualToResponse 1st request != 2nd response"));
	ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::bRequestEqualToResponse request vector"));
	vPrintVector(request,PRINT_VECTOR_ON_DEBUG_LEVEL_4);
	ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::bRequestEqualToResponse response vector"));
	vPrintVector(response,PRINT_VECTOR_ON_DEBUG_LEVEL_4);
	return false;
   }
   else
   {
	ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::bRequestEqualToResponse Request and Response is same"));
	return true;
   }
}

/************************************************************************
 * FUNCTION: actDecNRC27Count()
 *
 * DESCRIPTION: Decrements the NRC27 Count
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actDecNRC27Count ( void* /* pArg */ )
{
	NRC27Count-- ;
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actDecNRC27Count NRC27Count = %u", NRC27Count));
}

/************************************************************************
 * FUNCTION: actEvalPreDistributeStatus()
 *
 * DESCRIPTION: Evaluates the last predistributes status
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actEvalPreDistributeStatus ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actEvalPreDistributeStatus lastPreDistributeSuccess = %u", lastPreDistributeSuccess));
	if(lastPreDistributeSuccess)	
	{
		pmcp_FileHandler->IncrcurrentIteration();
		ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actEvalPreDistributeStatus incrCurrentIteration "));
	}
}

/************************************************************************
 * FUNCTION: actEvaluateRequestResponse()
 *
 * DESCRIPTION: Evaluates the Resquest and Response buffers
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actEvaluateRequestResponse ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actEvaluateRequestResponse start"));
	ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::actEvaluateRequestResponse bufUDSRequest vector"));
	vPrintVector(bufUDSRequest,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
	
	ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::actEvaluateRequestResponse bufResultBuffer vector"));
	vPrintVector(bufResultBuffer,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
	
	ETG_TRACE_USR4(("MCPINFO: mcp_fsmMCPUpdActions::actEvaluateRequestResponse bufUDSResponse vector"));
	vPrintVector(bufUDSResponse,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
	
	tU32 noBytesToVerify = 0;
	switch(bufUDSRequest[0])
	{
		case MCP_SESSION:
		{
			switch(bufUDSRequest[1])
			{
				case MCP_DEFAULT_SESSION:
					noBytesToVerify = TWO_BYTES_TO_VERIFY;
					if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_DEFAULT_SESSION_FAILED);
						if(bRequestEqualToResponse(bufResultBuffer_Negative,  bufUDSResponse, THREE_BYTES_TO_VERIFY) == true)
						{
							setMCPErrorCode(ENMCPCTRLSTATE::MCP_RQUEST_LENGTH_ERROR);
						}
						setupdateFailed(true);  
					}
				    	break;
				case MCP_PROGRAMMING_SESSION:
					noBytesToVerify = TWO_BYTES_TO_VERIFY;
					if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_PROGRAMMING_SESSION_FAILED);
						if(bRequestEqualToResponse(bufResultBuffer_Negative,  bufUDSResponse, THREE_BYTES_TO_VERIFY) == true)
						{
							setMCPErrorCode(ENMCPCTRLSTATE::MCP_RQUEST_LENGTH_ERROR);
						}
						setupdateFailed(true);  
					}
					break;
					
				case MCP_EXTENDED_SESSION:
					noBytesToVerify = TWO_BYTES_TO_VERIFY;
					if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_EXTENDED_SESSION_FAILED);
						if(bRequestEqualToResponse(bufResultBuffer_Negative,  bufUDSResponse, THREE_BYTES_TO_VERIFY) == true)
						{
							setMCPErrorCode(ENMCPCTRLSTATE::MCP_RQUEST_LENGTH_ERROR);
						}
						setupdateFailed(true);  
					}
					break;
			}
		}
		break;
		  

		case MCP_SEED_KEY_REQUEST:
		{
			switch(bufUDSRequest[1])
			{
			  
				case MCP_SEED_REQUEST:
					noBytesToVerify = TWO_BYTES_TO_VERIFY;
					if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_REQUEST_SEED_FAILED);
						if(bRequestEqualToResponse(bufResultBuffer_Negative,  bufUDSResponse, TWO_BYTES_TO_VERIFY) == true)
						{
							if(bufUDSResponse[2] == MCP_REQUEST_LENGTH_ERROR)
							{
								setMCPErrorCode(ENMCPCTRLSTATE::MCP_RQUEST_LENGTH_ERROR);
							}
							else if(bufUDSResponse[2] == MCP_10SEC_TIMER_ERROR)
							{
								setMCPErrorCode(ENMCPCTRLSTATE::MCP_10SEC_TIMER_IS_ZERO);
							}
						}
						setupdateFailed(true);  
					}
					break;
			
				case MCP_KEY_REQUEST:
					noBytesToVerify = TWO_BYTES_TO_VERIFY;
					if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_REQUEST_KEY_FAILED);
						if(bRequestEqualToResponse(bufResultBuffer_Negative,  bufUDSResponse, TWO_BYTES_TO_VERIFY) == true)
						{
							if( bufUDSResponse[2] == MCP_REQUEST_LENGTH_ERROR)
							{
								setMCPErrorCode(ENMCPCTRLSTATE::MCP_RQUEST_LENGTH_ERROR);
								setupdateFailed(true); 
							}
							else if(bufUDSResponse[2] == MCP_CAN_KEY_NEGATIVE_RESPONSE)
							{
								setMCPErrorCode(ENMCPCTRLSTATE::MCP_NOT_IN_PROG_SESSION);
								setupdateFailed(true); 
							}
							else if(bufUDSResponse[2] == MAX_KEY_RETRIES_REACHED)
							{
								setMCPErrorCode(ENMCPCTRLSTATE::MCP_KEY_IS_NOT_MATCHED);
								setupdateFailed(true); 
							}
							else if(bufUDSResponse[2] == KEY_RETRY_REQUIRED)
							{
								setNRC27Status(true);
							}
						}
					}
					break;
			}
		  }
		  break;
		  
		case MCP_SEND_NEW_SW_VERSION:
			noBytesToVerify = THREE_BYTES_TO_VERIFY;
			if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
			{
				setMCPErrorCode(ENMCPCTRLSTATE::MCP_SENT_NEW_VERSION_FAILED);
				if(bRequestEqualToResponse(bufResultBuffer_Negative,  bufUDSResponse, TWO_BYTES_TO_VERIFY) == true)
				{
					if(bufUDSResponse[2] == MCP_REQUEST_LENGTH_ERROR)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_RQUEST_LENGTH_ERROR);
					}
					else if(bufUDSResponse[2] == DID_NOT_MATCHED)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_DID_NOT_MATCHED);
					}
					else if(bufUDSResponse[2] == NO_SECURITY_ACCESS)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_NO_SECUTIRY_ACCESS);
					}
				}
				setupdateFailed(true);
			}
			break;
			

 		case MCP_ERASE_MEMORY_REQUEST:
	  		noBytesToVerify = ONE_BYTE_TO_VERIFY;
 			if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
			{
				setMCPErrorCode(ENMCPCTRLSTATE::MCP_REQUEST_ERASE_MEMORY_FAILED);
				if(bRequestEqualToResponse(bufResultBuffer_Negative,  bufUDSResponse, TWO_BYTES_TO_VERIFY) == true)
				{
					if(bufUDSResponse[2] == MCP_REQUEST_LENGTH_ERROR)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_RQUEST_LENGTH_ERROR);
					}
					else if(bufUDSResponse[2] == MCP_SUB_FUNCTION_ERROR)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_SUB_FUNC_ERROR);
					}
					else if(bufUDSResponse[2] == MCP_VERSION_ERROR)
					{
						setMCPErrorCode(ENMCPCTRLSTATE::MCP_INSTALLED_VERSION_HIGHER);
					}
				}
				setupdateFailed(true); 
			}
 			break;
			
		case MCP_DOWNLOAD_START_REQUEST:
			noBytesToVerify = FOUR_BYTES_TO_VERIFY;
 			if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
			{
				setMCPErrorCode(ENMCPCTRLSTATE::MCP_REQUEST_DOWNLOAD_FAILED);
				setupdateFailed(true); 
			}
			break;
			
	  case MCP_TRANSMIT_IMAGE_DATA_REQUEST:
			noBytesToVerify = TWO_BYTES_TO_VERIFY;
			lastPreDistributeSuccess = false;
			if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
			{
				setMCPErrorCode(ENMCPCTRLSTATE::MCP_TRANSMIT_DATA_FAILED);
				setupdateFailed(true); 
			}
			else
			{
				lastPreDistributeSuccess = true;
			}
			break;
			
	  case MCP_READ_SW_VERSION:
			noBytesToVerify = THREE_BYTES_TO_VERIFY;
			if (bRequestEqualToResponse(bufResultBuffer,  bufUDSResponse, noBytesToVerify) == false)
			{
				setMCPErrorCode(ENMCPCTRLSTATE::MCP_APPCTRL_VERSION_ERROR);
				setupdateFailed(true);
			}
			break;	
	default:
		noBytesToVerify = 0;
		break;          
   	}
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actEvaluateRequestResponse end"));
	
}

/************************************************************************
 * FUNCTION: actEvaluateTimeout()
 *
 * DESCRIPTION: Evaluates time out request
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actEvaluateTimeout ( void* /* pArg */ )
{
   ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actEvaluateTimeout"));
   if (iNoOfTimeout < MAX_NUMBER_OF_TIMEOUT)
   {
      iNoOfTimeout++;
   }
   setupdateFailed(true);
   ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actEvaluateTimeout updateFailed = %u", updateFailed ));
   UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actEvaluateTimeout updateFailed = %u", DL_LOG_INFO, updateFailed);
}

/************************************************************************
 * FUNCTION: actExit()
 *
 * DESCRIPTION: Stops the update
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actExit ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actExit"));
	setUpdateCompleted(true);
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::actExit m_updatecompleted = %u",m_updatecompleted));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actExit m_updatecompleted = %u", DL_LOG_INFO, m_updatecompleted);
}

/************************************************************************
 * FUNCTION: actFillBufferForDefaultSessionRequest()
 *
 * DESCRIPTION: Fills Default Session Request buffer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForDefaultSessionRequest (void* /* pArg */ )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForDefaultSessionReques)"));

   bufUDSRequest.clear();
   bufUDSRequest.push_back(MCP_SESSION);
   bufUDSRequest.push_back(MCP_DEFAULT_SESSION);
   
   bufResultBuffer.clear();
   bufResultBuffer.push_back(MCP_SESSION_POSITIVE_RESPONSE);
   bufResultBuffer.push_back(MCP_DEFAULT_SESSION);
   
   bufResultBuffer_Negative.clear();
   bufResultBuffer_Negative.push_back(MCP_CAN_KEY_NEGATIVE_RESPONSE);
   bufResultBuffer_Negative.push_back(MCP_SESSION);
   bufResultBuffer_Negative.push_back(MCP_REQUEST_LENGTH_ERROR);
   
   UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForDefaultSessionRequest", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actFillBufferForExtendedSessionRequest()
 *
 * DESCRIPTION: Fills Extended Session Request buffer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForExtendedSessionRequest ( void* /* pArg */ )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForExtendedSessionRequest"));
   bufUDSRequest.clear();
   bufUDSRequest.push_back(MCP_SESSION);
   bufUDSRequest.push_back(MCP_EXTENDED_SESSION);
   
   bufResultBuffer.clear();
   bufResultBuffer.push_back(MCP_SESSION_POSITIVE_RESPONSE);
   bufResultBuffer.push_back(MCP_EXTENDED_SESSION);
   
   bufResultBuffer_Negative.clear();
   bufResultBuffer_Negative.push_back(MCP_CAN_KEY_NEGATIVE_RESPONSE);
   bufResultBuffer_Negative.push_back(MCP_SESSION);
   bufResultBuffer_Negative.push_back(MCP_REQUEST_LENGTH_ERROR);
   
   UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForExtendedSessionRequest", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actFillBufferForPreDistribute()
 *
 * DESCRIPTION: Fills buffer For PreDistribute request
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForPreDistribute ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForPreDistribute start"));
    	bufUDSRequest.clear();
	bufUDSRequest.push_back(MCP_TRANSMIT_IMAGE_DATA_REQUEST);

	unsigned int currentIteration = pmcp_FileHandler->getcurrentIteration();
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForPreDistribute() currentIteration: %u", currentIteration));
    	bufUDSRequest.push_back(static_cast<unsigned char>(currentIteration+1));
	
	bufResultBuffer.clear();
    	bufResultBuffer.push_back(MCPTRANSMIT_IMAGE_DATA_POSITIVE_RESPONSE);
    	bufResultBuffer.push_back(static_cast<unsigned char>(currentIteration+1));
   
    	vector<unsigned char> bufReadBinary;
	unsigned int bytesread =0;
    	bufReadBinary.clear();
    	if (pmcp_FileHandler->ReadBuffer(bufReadBinary,(currentIteration*BYTES_TO_READ ), bytesread))
	{
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForPreDistribute() bufReadBinary vector"));
		vPrintVector(bufReadBinary,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
		
		if(bytesread < MAX_BYTES)
		{
			ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForPreDistribute() last iteration "));
			vector<unsigned char> bufUDSDefault(MAX_BYTES);
			bufUDSDefault.resize(static_cast<size_t> (MAX_BYTES));
			fill_n(bufUDSDefault.begin(),MAX_BYTES, 0xFF);
		
			auto bufReadBinary_iterator = bufReadBinary.begin() + bytesread; 
			std::copy_n(bufUDSDefault.begin(), (MAX_BYTES-bytesread), std::inserter(bufReadBinary, bufReadBinary_iterator));
			
			bufReadBinary.resize(static_cast<size_t> (MAX_BYTES));
			std::copy_n(bufReadBinary.begin(), MAX_BYTES, std::back_inserter(bufUDSRequest));
		}
		else
			std::copy_n(bufReadBinary.begin(), bytesread, std::back_inserter(bufUDSRequest));
		
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForPreDistribute() bufUDSRequest vector"));
		vPrintVector(bufUDSRequest,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForPreDistribute() end"));
	}
	else
	{
		setUpdateCompleted(true);
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ReadBuffer File is empty"));
	}
}

/************************************************************************
 * FUNCTION: actFillBufferForProgrammingSessionRequest()
 *
 * DESCRIPTION: Fills Programming Session Request buffer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForProgrammingSessionRequest ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForProgrammingSessionRequest"));
	bufUDSRequest.clear();
	bufUDSRequest.push_back(MCP_SESSION);
	bufUDSRequest.push_back(MCP_PROGRAMMING_SESSION);
	
	bufResultBuffer.clear();
	bufResultBuffer.push_back(MCP_SESSION_POSITIVE_RESPONSE);
	bufResultBuffer.push_back(MCP_PROGRAMMING_SESSION);
	
	bufResultBuffer_Negative.clear();
	bufResultBuffer_Negative.push_back(MCP_CAN_KEY_NEGATIVE_RESPONSE);
	bufResultBuffer_Negative.push_back(MCP_SESSION);
	bufResultBuffer_Negative.push_back(MCP_REQUEST_LENGTH_ERROR);
   
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForProgrammingSessionRequest", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actFillBufferForRequestEraseMemoryRequest()
 *
 * DESCRIPTION: Fills request erase memory request buffer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForRequestEraseMemoryRequest ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForRequestEraseMemoryRequest"));
	bufUDSRequest.clear();
	bufUDSRequest.push_back(MCP_ERASE_MEMORY_REQUEST);
	bufUDSRequest.push_back(DID_BYTE_01);
	bufUDSRequest.push_back(DID_BYTE_02);
	bufUDSRequest.push_back(DID_BYTE_03);
	bufUDSRequest.push_back(LOGICAL_ADDRESS_01);
	bufUDSRequest.push_back(LOGICAL_ADDRESS_02);
	bufUDSRequest.push_back(LOGICAL_ADDRESS_03);
	bufUDSRequest.push_back(LOGICAL_ADDRESS_04);
	
	unsigned int fileSize = pmcp_FileHandler->calculateFileSize();
	
	if( fileSize > 0)
	{
		bufUDSRequest.push_back((unsigned char)((fileSize>>24) & 0xFF));
		bufUDSRequest.push_back((unsigned char)((fileSize>>16) & 0xFF));
		bufUDSRequest.push_back((unsigned char)((fileSize>>8) & 0xFF));
		bufUDSRequest.push_back((unsigned char)((fileSize) & 0xFF)); 
	}
	else
	{
		setUpdateCompleted(true);
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ReadBuffer File is empty"));
	}
	
	bufResultBuffer.clear();
	bufResultBuffer.push_back(MCP_ERASE_MEMORY_POSITIVE_RESPONSE);
	
	bufResultBuffer_Negative.clear();
	bufResultBuffer_Negative.push_back(MCP_CAN_KEY_NEGATIVE_RESPONSE);
	bufResultBuffer_Negative.push_back(MCP_ERASE_MEMORY_REQUEST);
	
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForRequestEraseMemoryRequest", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actFillBufferForRequestKeyRequest()
 *
 * DESCRIPTION: Fills request key buffer 
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForRequestKeyRequest ( void* /* pArg */ )
{

   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForRequestKeyRequest"));
   bufUDSRequest.clear();
   bufUDSRequest.push_back(MCP_SEED_KEY_REQUEST);
   bufUDSRequest.push_back(MCP_KEY_REQUEST);
   
   bufResultBuffer.clear();
   bufResultBuffer.push_back(MCP_SEED_KEY_REQUEST_POSITIVE_RESPONSE);
   bufResultBuffer.push_back(MCP_KEY_REQUEST);

   bufResultBuffer_Negative.clear();
   bufResultBuffer_Negative.push_back(MCP_CAN_KEY_NEGATIVE_RESPONSE);
   bufResultBuffer_Negative.push_back(MCP_SEED_KEY_REQUEST);
   
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForRequestKeyRequest iopKeyArray "));
   for (unsigned int keybyte=0;keybyte<MAX_KEY_BYTES;keybyte++)
   {
	ETG_TRACE_USR4(("%u  ",iopKeyArray[keybyte] ));
	bufUDSRequest.push_back(iopKeyArray[keybyte]);
   } 
   
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForRequestKeyRequest", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actFillBufferForRequestSeedRequest()
 *
 * DESCRIPTION: Fills seed request buffer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForRequestSeedRequest ( void* /* pArg */ )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForRequestSeedRequest"));
   bufUDSRequest.clear();
   bufUDSRequest.push_back(MCP_SEED_KEY_REQUEST);
   bufUDSRequest.push_back(MCP_SEED_REQUEST);
 
   bufResultBuffer.clear();
   bufResultBuffer.push_back(MCP_SEED_KEY_REQUEST_POSITIVE_RESPONSE);
   bufResultBuffer.push_back(MCP_SEED_REQUEST);
 
   bufResultBuffer_Negative.clear();
   bufResultBuffer_Negative.push_back(MCP_CAN_KEY_NEGATIVE_RESPONSE);
   bufResultBuffer_Negative.push_back(MCP_SEED_KEY_REQUEST);
   
   UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForRequestSeedRequest", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actFillBufferForSendNewVersion()
 *
 * DESCRIPTION: Send the version details of new binary to flash.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
 
void mcp_fsmMCPUpdActions::actFillBufferForSendNewVersion ( void* pArg )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForSendNewVersion start"));
   bufUDSRequest.clear();
   bufUDSRequest.push_back(MCP_SEND_NEW_SW_VERSION);
   bufUDSRequest.push_back(MCP_SEND_NEW_SW_VER_BYTE1);
   bufUDSRequest.push_back(MCP_SEND_NEW_SW_VER_BYTE2);
   
   string updatePath = getUpdateFilePath();
   pmcp_FileHandler->setFilePath(updatePath);
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForSendNewVersion updatePath = %s",updatePath.c_str()));
   UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForSendNewVersion updatePath = %s", DL_LOG_INFO,updatePath.c_str());
   
   unsigned int filesize = pmcp_FileHandler->calculateFileSize();
   if(filesize > 0)
   {
	vector<unsigned char> bufReadBinary;
	unsigned int bytesread =MCP_NEW_SW_VERSION_LEN;
	bufReadBinary.clear();
	pmcp_FileHandler->ReadBuffer(bufReadBinary,MCP_NEW_SW_ADD_VER_READ, bytesread);
		
	bufReadBinary.resize(static_cast<size_t> (MCP_NEW_SW_VERSION_LEN));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForSendNewVersion() bufReadBinary vector"));
	vPrintVector(bufReadBinary,PRINT_VECTOR_ON_DEBUG_LEVEL_2);

	binaryVersionInfo.clear();
	binaryVersionInfo.assign(bufReadBinary.begin(), bufReadBinary.begin()+MCP_NEW_SW_VERSION_LEN);
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForSendNewVersion SWVersion  : %s",binaryVersionInfo.c_str()));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForSendNewVersion SWVersion : %s", DL_LOG_INFO,binaryVersionInfo.c_str() );	

	std::copy_n(bufReadBinary.begin(), MCP_NEW_SW_VERSION_LEN, std::back_inserter(bufUDSRequest));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForSendNewVersion() bufUDSRequest vector"));
	vPrintVector(bufUDSRequest,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
   }
   else
   {
	setUpdateCompleted(true);
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ReadBuffer File is empty"));
   }
   
   bufResultBuffer.clear();
   bufResultBuffer.push_back(MCP_SEND_NEW_SW_VERSION_POSITIVE);
   bufResultBuffer.push_back(MCP_SEND_NEW_SW_VER_BYTE1);
   bufResultBuffer.push_back(MCP_SEND_NEW_SW_VER_BYTE2);
 
   bufResultBuffer_Negative.clear();
   bufResultBuffer_Negative.push_back(MCP_CAN_KEY_NEGATIVE_RESPONSE);
   bufResultBuffer_Negative.push_back(MCP_SEND_NEW_SW_VERSION);
   
   UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForSendNewVersion", DL_LOG_INFO);
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForSendNewVersion end"));
}

/************************************************************************
 * FUNCTION: actFillBufferForRequestDownloadStartRequest()
 *
 * DESCRIPTION: Fills download start request buffer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForRequestDownloadStartRequest(void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForRequestDownloadStartRequest start"));
    	bufUDSRequest.clear();
	bufUDSRequest.push_back(MCP_DOWNLOAD_START_REQUEST);
	bufUDSRequest.push_back(FORMATE_IDENTIFIER_1);
	bufUDSRequest.push_back((LOGICAL_ADDRESS_LENGTH<<4)|(FILE_SIZE_LENGTH));

	bufUDSRequest.push_back(LOGICAL_ADDRESS_01);
	bufUDSRequest.push_back(LOGICAL_ADDRESS_02);
	bufUDSRequest.push_back(LOGICAL_ADDRESS_03);
	bufUDSRequest.push_back(LOGICAL_ADDRESS_04);
	
	unsigned int fileSize = pmcp_FileHandler->calculateFileSize();

	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForRequestDownloadStartRequest start = %u", fileSize));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForRequestDownloadStartRequest fileSize = %u", DL_LOG_INFO,fileSize);
	
	if( fileSize > 0)
	{
		bufUDSRequest.push_back((unsigned char)((fileSize>>24) & 0xFF));
		bufUDSRequest.push_back((unsigned char)((fileSize>>16) & 0xFF));
		bufUDSRequest.push_back((unsigned char)((fileSize>>8) & 0xFF));
		bufUDSRequest.push_back((unsigned char)((fileSize) & 0xFF)); 
	}
	else
	{
		setUpdateCompleted(true);
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ReadBuffer File is empty"));
	}
	bufResultBuffer.clear();
	bufResultBuffer.push_back(MCP_DOWNLOAD_START_POSITIVE_RESPONSE);
	bufResultBuffer.push_back(FORMATE_IDENTIFIER);
	bufResultBuffer.push_back(MAX_NO_BLOCK_LEN_BYTE1);
	bufResultBuffer.push_back(MAX_NO_BLOCK_LEN_BYTE2);

	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForRequestDownloadStartRequest", DL_LOG_INFO);
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForRequestDownloadStartRequest end"));
}

/************************************************************************
 * FUNCTION: actHandlePreDistributeError()
 *
 * DESCRIPTION: Handles pre distribute error
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actHandlePreDistributeError ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actHandlePreDistributeError"));
	setMCPErrorCode(ENMCPCTRLSTATE::MCP_TRANSMIT_DATA_FAILED);
	setupdateFailed(true); 
	setUpdateCompleted(true);
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actHandlePreDistributeError", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actOnCriticalError()
 *
 * DESCRIPTION: Stops the update if critical error occurs
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actOnCriticalError ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actOnCriticalError"));
	setMCPErrorCode(ENMCPCTRLSTATE::MCP_TRANSMIT_DATA_FAILED);
	setupdateFailed(true); 
	setUpdateCompleted(true);
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actOnCriticalError", DL_LOG_INFO);
}

/************************************************************************
 * FUNCTION: actPreparePreDistribute()
 *
 * DESCRIPTION: Before pre distribute calculates the max number of iterations
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actPreparePreDistribute ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actPreparePreDistribute"));
	if(pmcp_FileHandler->CalculateMaxIterations())
	{
		unsigned int maxIterations = pmcp_FileHandler->getMaxIteration();
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actPreparePreDistribute maxIterations = %u",maxIterations));
		UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actPreparePreDistribute maxIterations = %u", DL_LOG_INFO,maxIterations);
	}
}

/************************************************************************
 * FUNCTION: actPrintCurrentState()
 *
 * DESCRIPTION: Prints the current state
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actPrintCurrentState ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actPrintCurrentState : %s" , vGetStateName() ));
}

/************************************************************************
 * FUNCTION: actRestartTimerLong()
 *
 * DESCRIPTION: Stops the timer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actRestartTimerLong ( void* /* pArg */ )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actRestartTimerLong"));
   if (pmcpUpdaterProxy != nullptr)
   {
      pmcpUpdaterProxy->stopTimer();
   }
}

/************************************************************************
 * FUNCTION: actSendRequest()
 *
 * DESCRIPTION: Sends UDS request
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actSendRequest ( void* /* pArg */ )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actSendRequest start"));
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actSendRequest bufUDSRequest"));
   vPrintVector(bufUDSRequest,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
   
   std::stringstream UDSRequest;
   for (unsigned int udsdata=0; udsdata < bufUDSRequest.size() ; udsdata++)
   {
      UDSRequest << std::hex << int(bufUDSRequest[udsdata]) << " ";
   }
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actSendRequest UDSRequest : %s",UDSRequest.str().c_str()));
   pRequestSendReq = bufUDSRequest.data();
   u16LengthSendReq = static_cast<tU16>(bufUDSRequest.size()); 

   pmcpCsmProxy->sendUdsDataMessage(pRequestSendReq, u16LengthSendReq);

   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actSendRequest end"));
}

/************************************************************************
 * FUNCTION: actStartTimer()
 *
 * DESCRIPTION: Starts the timer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actStartTimer ( void* /* pArg */ )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStartTimer"));
   if (pmcpUpdaterProxy != nullptr)
   {
      pmcpUpdaterProxy->startTimer();
   }
}

/************************************************************************
 * FUNCTION: actStopDistributeAndReturnError()
 *
 * DESCRIPTION: Handles pre distribute error
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actStopDistributeAndReturnError ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStopDistributeAndReturnError"));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStopDistributeAndReturnError lastPreDistributeSuccess = %u" , lastPreDistributeSuccess));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actStopDistributeAndReturnError lastPreDistributeSuccess = %u", DL_LOG_INFO,lastPreDistributeSuccess);
	if(!lastPreDistributeSuccess)
	{
		setMCPErrorCode(ENMCPCTRLSTATE::MCP_REQUEST_DOWNLOAD_FAILED);
		setUpdateCompleted(true);
	}
}

/************************************************************************
 * FUNCTION: actStopTimer()
 *
 * DESCRIPTION: Stops the timer
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actStopTimer ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStopTimer"));
	if (pmcpUpdaterProxy != nullptr)
		pmcpUpdaterProxy->stopTimer();
}

/************************************************************************
 * FUNCTION: actStoreDistribute()
 *
 * DESCRIPTION: Stores binary file path
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actStoreDistribute ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreDistribute"));
	std::string updatePath = getUpdateFilePath();
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreDistribute updateFilePath : %s" ,updatePath.c_str() ));
}

/************************************************************************
 * FUNCTION: actUDSRequestFailed()
 *
 * DESCRIPTION: Handles UDS request failure
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actUDSRequestFailed ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actUDSRequestFailed"));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actUDSRequestFailed", DL_LOG_INFO);
	setMCPErrorCode(ENMCPCTRLSTATE::MCP_REQUEST_DOWNLOAD_FAILED);
	lastPreDistributeSuccess = false;
	bUDSRequestFailed = true;
	
}

/************************************************************************
 * FUNCTION: actWaitForJumpToBoot()
 *
 * DESCRIPTION: Delays the timer to jump MCP to boot mode
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actWaitForJumpToBoot ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actWaitForJumpToBoot"));
	pmcpUpdaterProxy->waitTimerForJumpToBoot(DELAY_MILLISEC_TO_JUMP_BOOT);
}

/************************************************************************
 * FUNCTION: IfError()
 *
 * DESCRIPTION: If error occurs stops the update
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::IfError ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::IfError updateFailed : %u",updateFailed));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions IfError updateFailed : %u", DL_LOG_INFO,updateFailed);
	if(updateFailed)
	{
		return true;
	}
	else
		return false;

}

/************************************************************************
 * FUNCTION: ifExistNextPreDistribute()
 *
 * DESCRIPTION: checks the current iteration is less than max iteration then 
				send the data to flash.
				Else update is completed.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifExistNextPreDistribute ( void* /* pArg */ )
{
        
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifExistNextPreDistribute"));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifExistNextPreDistribute currentIteration = %u ",pmcp_FileHandler->getcurrentIteration()));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifExistNextPreDistribute max iteration = %u ",pmcp_FileHandler->getMaxIteration()));
	if( pmcp_FileHandler->getcurrentIteration() < pmcp_FileHandler->getMaxIteration() )
	{
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifExistNextPreDistribute current iteration is less then max iteration"));
		return true;
	}
	else
	{
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifExistNextPreDistribute current iteration is not less then max iteration"));
		setUpdateCompleted(true);
		return false;
	}
}

/************************************************************************
 * FUNCTION: ifLastPreDistributeSuccess()
 *
 * DESCRIPTION: if last predistribute success then send remaining data to flash
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifLastPreDistributeSuccess ( void* /* pArg */ )
{	
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifLastPreDistributeSuccess : %u", lastPreDistributeSuccess));
	return lastPreDistributeSuccess;
}

/************************************************************************
 * FUNCTION: ifLessThanMaxNoOfTimeout()
 *
 * DESCRIPTION: check the timeouts
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifLessThanMaxNoOfTimeout ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifLessThanMaxNoOfTimeout : %u", (iNoOfTimeout <= MAX_NUMBER_OF_TIMEOUT)));
	return (iNoOfTimeout < MAX_NUMBER_OF_TIMEOUT);
}

/************************************************************************
 * FUNCTION: ifLessThanMaxNoOfTrials()
 *
 * DESCRIPTION: checks the number of trails
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifLessThanMaxNoOfTrials ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifLessThanMaxNoOfTrials"));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifLastPreDistributeSuccess isNRC27Received : %u", isNRC27Received));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifLastPreDistributeSuccess iNoOfTrials : %u", iNoOfTrials));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifLastPreDistributeSuccess MAX_NUMBER_OF_TRIALS : %u", MAX_NUMBER_OF_TRIALS));
	if (isNRC27Received && (iNoOfTrials <= MAX_NUMBER_OF_TRIALS))
	{
		setMCPErrorCode(ENMCPCTRLSTATE::MCP_APPCTRL_NONE);
		return true;
	}
	else if( isNRC27Received && (iNoOfTrials == (MAX_NUMBER_OF_TRIALS+1)))
	{
		setUpdateCompleted(true);
		return false;
	}
	else
		return false;
}

/************************************************************************
 * FUNCTION: ifNotNRC27Received()
 *
 * DESCRIPTION: checks status of NRC27 received.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifNotNRC27Received ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifNotNRC27Received : %u", isNRC27Received));
	return (!isNRC27Received);
}

/************************************************************************
 * FUNCTION: mcp_ClientHandler()
 *
 * DESCRIPTION: Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifNRC27CountNotZero ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifNRC27CountNotZero NRC27Count : %u", NRC27Count));
	return (NRC27Count != 0);
}

/************************************************************************
 * FUNCTION: mcp_ClientHandler()
 *
 * DESCRIPTION: Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifNRC27Received ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifNRC27Received isNRC27Received : %u", isNRC27Received));
	if(isNRC27Received)
	{
		iNoOfTrials ++;
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifNRC27Received iNoOfTrials : %u", iNoOfTrials));
		NRC27Count ++ ;
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifNRC27Received NRC27Count : %u", NRC27Count));
	}
	return isNRC27Received;
	
}

/************************************************************************
 * FUNCTION: ifPreDistributeSuccess()
 *
 * DESCRIPTION: If predistribute is success then update is done
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifPreDistributeSuccess ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifPreDistributeSuccess m_updatecompleted : %u", m_updatecompleted));
	setMCPErrorCode(ENMCPCTRLSTATE::MCP_APPCTRL_UPDATE_SUCCESS);
	return m_updatecompleted ;
}

/************************************************************************
 * FUNCTION: ifResponseWasForLastRequest()
 *
 * DESCRIPTION: checks response of last request
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifResponseWasForLastRequest ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifResponseWasForLastRequest updateFailed : %u", updateFailed));
	if(updateFailed)
	{
		return false;
	}
	else
		return true;
}

/************************************************************************
 * FUNCTION: ifRetryRequired()
 *
 * DESCRIPTION: Check retry is required to send the UDS request again
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifRetryRequired ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifRetryRequired "));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifRetryRequired isNRC27Received : %u", isNRC27Received));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifRetryRequired iNoOfTrials : %u", iNoOfTrials));
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifRetryRequired MAX_NUMBER_OF_TRIALS : %u", MAX_NUMBER_OF_TRIALS));
	if(isNRC27Received && (iNoOfTrials <= MAX_NUMBER_OF_TRIALS))
	{
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifRetryRequired is true"));
		return true;
	}
	else
	{
		ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifRetryRequired is false"));
		return false;
	}
}

/************************************************************************
 * FUNCTION: actNRC27Received()
 *
 * DESCRIPTION: Handles NRC27 received
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actNRC27Received ( void* /* pArg */ )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actNRC27Received "));
	isNRC27Received = false;
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actNRC27Received isNRC27Received : %u", isNRC27Received));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actNRC27Received isNRC27Received : %u", DL_LOG_INFO,isNRC27Received);
}

/************************************************************************
 * FUNCTION: actStoreSeed()
 *
 * DESCRIPTION: Stores seed data
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::actStoreSeed(void* pArg)
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreSeed start"));
	unsigned char ipSeedArray[MAX_SEED_BYTES];
	unsigned int keysize = 0; 

	std::vector<unsigned char> seedKey;
	seedKey.clear();
	for (unsigned int extractSeed=2;extractSeed<(MAX_SEED_BYTES+2);extractSeed++)
	{
		seedKey.push_back(bufUDSResponse[extractSeed]);
	} 
	
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreSeed bufUDSResponse vector"));
	vPrintVector(bufUDSResponse,PRINT_VECTOR_ON_DEBUG_LEVEL_2);
	
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreSeed seedKey vector"));
	vPrintVector(seedKey,PRINT_VECTOR_ON_DEBUG_LEVEL_2);

	std::copy(seedKey.begin(),seedKey.end(),ipSeedArray);

	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreSeed ipSeedArray"));
	for(int seedbyte=0;seedbyte<MAX_SEED_BYTES;seedbyte++)
	{

		ETG_TRACE_USR4(("%u ",ipSeedArray[seedbyte]));
	} 

   	GenerateKeyExOpt(ipSeedArray,MAX_SEED_BYTES,iopKeyArray,MAX_KEY_BYTES,keysize);
	
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreSeed iopKeyArray"));
	for(int keybyte=0;keybyte<MAX_KEY_BYTES;keybyte++)
	{
		ETG_TRACE_USR4(("%u ",iopKeyArray[keybyte]));
	}
	
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStoreSeed end"));
}

/************************************************************************
 * FUNCTION: setUpdateFilePath()
 *
 * DESCRIPTION: Set the binary file path
 *
 * PARAMETER: std::string
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::setUpdateFilePath(std::string path)
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::setUpdateFilePath : %s", path.c_str()));
	updateFilePath=path;
}

/************************************************************************
 * FUNCTION: getUpdateFilePath()
 *
 * DESCRIPTION: Get the binary file path
 *
 * PARAMETER: None
 *
 * RETURNVALUE: std::string
 *************************************************************************/
std::string mcp_fsmMCPUpdActions::getUpdateFilePath(void)
{
	return updateFilePath;
}

/************************************************************************
 * FUNCTION: getMCPErrorCode()
 *
 * DESCRIPTION: Get the error code
 *
 * PARAMETER: None
 *
 * RETURNVALUE: int_least32_t
 *************************************************************************/
int_least32_t mcp_fsmMCPUpdActions::getMCPErrorCode(){
	return iErrorCode;
}

/************************************************************************
 * FUNCTION: setMCPErrorCode()
 *
 * DESCRIPTION: Set the error code
 *
 * PARAMETER: int_least32_t
 *
 * RETURNVALUE: None
 *************************************************************************/
void mcp_fsmMCPUpdActions::setMCPErrorCode(int_least32_t errorcode){
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::setMCPErrorCode errorcode = %u", errorcode));
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::setMCPErrorCode errorcode str = %s", strenMcpCtrlState[errorcode]));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions update failed. errorcode = %s", DL_LOG_INFO,strenMcpCtrlState[errorcode]);
	iErrorCode = errorcode;
}

/************************************************************************
 * FUNCTION: ifFlashFileNotExists()
 *
 * DESCRIPTION: checks the binary file exists or not
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::ifFlashFileNotExists(void* pArg)
{
	bool fileExists = pmcp_FileHandler->ifFileExists();
	ETG_TRACE_USR4(("MCPINFO:mcp_fsmMCPUpdActions::ifFlashFileNotExists = %u", !fileExists));
	return !fileExists;
}

/************************************************************************
 * FUNCTION: IfUDSRequestFailed()
 *
 * DESCRIPTION: Handles UDS request failed
 *
 * PARAMETER: None
 *
 * RETURNVALUE: bool
 *************************************************************************/
bool mcp_fsmMCPUpdActions::IfUDSRequestFailed ( void* pArg )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::IfUDSRequestFailed bUDSRequestFailed : %u", bUDSRequestFailed));
	if(bUDSRequestFailed )
		return true;
	else
		return false;
}
/************************************************************************
* FUNCTION: actFillBufferForReadMCPSWVersion()
*
* DESCRIPTION: Fills Buffer For Read MCP SWVersion
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
void mcp_fsmMCPUpdActions::actFillBufferForReadMCPSWVersion ( void* pArg )
{
   ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actFillBufferForReadMCPSWVersion"));
   bufUDSRequest.clear();
   bufUDSRequest.push_back(MCP_READ_SW_VERSION);
   bufUDSRequest.push_back(MCP_READ_SW_VERSION_BYTE1);
   bufUDSRequest.push_back(MCP_READ_SW_VERSION_BYTE2);
   
   bufResultBuffer.clear();
   bufResultBuffer.push_back(MCP_READ_SW_VERSION_POSITIVE_RESPONSE);
   bufResultBuffer.push_back(MCP_READ_SW_VERSION_BYTE1);
   bufResultBuffer.push_back(MCP_READ_SW_VERSION_BYTE2);
   
   UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actFillBufferForReadMCPSWVersion", DL_LOG_INFO);
}

/************************************************************************
* FUNCTION: actCompleteReadMCPSWVersion()
*
* DESCRIPTION: Complete Read MCP SWVersion
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
void mcp_fsmMCPUpdActions::actCompleteReadMCPSWVersion ( void* pArg )
{
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actCompleteReadMCPSWVersion"));
	
	mcpSWVersion.clear();
	mcpSWVersion.assign(bufUDSResponse.begin()+3, bufUDSResponse.begin()+17);
	ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actCompleteReadMCPSWVersion SWVersion  : %s",mcpSWVersion.c_str()));
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actCompleteReadMCPSWVersion SWVersion : %s", DL_LOG_INFO,mcpSWVersion.c_str() );

	setMCPErrorCode(ENMCPCTRLSTATE::MCP_APPCTRL_VERSION_OK);
	setUpdateCompleted(true);
	UpdateLog::vUpdateLogTrace ("%s mcp_fsmMCPUpdActions actCompleteReadMCPSWVersion", DL_LOG_INFO);

}
/************************************************************************
* FUNCTION: actResetRetrycounter()
*
* DESCRIPTION: ResetRetryCounter
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/

void mcp_fsmMCPUpdActions::actResetRetrycounter ( void* pArg )
{
    ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actResetRetrycounter"));
    sendrequestRetries = 0;
}

/************************************************************************
* FUNCTION: ifRetryCountNotExceeded()
*
* DESCRIPTION: ChecksForRetryCountNotExceeded 
*
* PARAMETER: None
*
* RETURNVALUE: bool
*************************************************************************/

bool mcp_fsmMCPUpdActions::ifRetryCountNotExceeded ( void* pArg )
{
    ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::ifRetryCountNotExceeded=%d",sendrequestRetries));
    bool res = false;
    if(sendrequestRetries < MAX_RESEND_RETRIES)
    {
        sendrequestRetries++;
        res = true;
    }
    else
    {
        setMCPErrorCode(ENMCPCTRLSTATE::MCP_MAX_RETRIES_REACHED);
    }
    return res;    

}   
/************************************************************************
* FUNCTION: actStopUpdate()
*
* DESCRIPTION: StopUpdate
*
* PARAMETER: None
*
* RETURNVALUE: None
*************************************************************************/
void mcp_fsmMCPUpdActions::actStopUpdate(void*) 
{
    ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::actStopUpdate start")); 	   
    bufUDSRequest.clear();
    bufResultBuffer.clear();
    setUpdateCompleted(true);
    
}

/************************************************************************
* FUNCTION: setUpdateCompleted
*
* DESCRIPTION: set the update completed flag
*
* PARAMETER: bool
*
* RETURNVALUE: None
*************************************************************************/

void mcp_fsmMCPUpdActions::setUpdateCompleted(bool updateCompleted)
{
    ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::setUpdateCompleted m_updatecompleted : %u", updateCompleted));
    m_updatecompleted = updateCompleted;
    
}
/************************************************************************
* FUNCTION: setupdateFailed()
*
* DESCRIPTION: setupdateFailed
*
* PARAMETER: bool
*
* RETURNVALUE: None
*************************************************************************/   
void mcp_fsmMCPUpdActions::setupdateFailed(bool updateStatus)
{
    ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::setupdateFailed updateFailed : %u", updateStatus));
    updateFailed = updateStatus;	
}

/************************************************************************
* FUNCTION: setNRC27Status()
*
* DESCRIPTION: setNRC27Status
*
* PARAMETER: bool
*
* RETURNVALUE: None
*************************************************************************/
void mcp_fsmMCPUpdActions::setNRC27Status(bool NRC27Status)
{ 
    ETG_TRACE_USR4(("mcp_fsmMCPUpdActions::setNRC27Status isNRC27Received : %u", NRC27Status)); 
    isNRC27Received = NRC27Status;
}

/************************************************************************
* FUNCTION: getMCPUpdateStatus()
*
* DESCRIPTION: get MCP Update Status
*
* PARAMETER: bool
*
* RETURNVALUE: None
*************************************************************************/
bool mcp_fsmMCPUpdActions::getMCPUpdateStatus()
{
	return m_updatecompleted;
}
