/************************************************************************
 * File: mcp_ClientHandler.cpp
 * SW-Component:
 *
 * Description:
 *	Calls APIs of MCP library and handles the responses
 *
 * Author:
 *   RamaGopalReddy.Bommireddy@in.bosch.com
 * Copyright:
 *   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
 *
 * History:
 * 31.12.2019 - Initial version - RamaGopalReddy.Bommireddy@in.bosch.com
 * 28.02.2020 - Revision 1.0    - Surparaju.Pavankumar@in.bosch.com
 ***********************************************************************/
#include "mcp_ClientHandler.h"

using namespace std;
#include <memory>
#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_MCP
#define ETG_I_FILE_PREFIX 				mcp_ClientHandler::
#include "trcGenProj/Header/mcp_ClientHandler.cpp.trc.h"
#endif

#ifdef DEBUG_MCP
#define PR printf("File - %s ;function:  %s  Line : %d\n",__FILE__,__FUNCTION__,__LINE__)
#define PR_ENTRY printf("File - %s ; ENTRY : function:  %s  Line : %d\n",__FILE__,__FUNCTION__,__LINE__)
#define PR_EXIT printf("File - %s; EXIT : function:  %s  Line : %d\n",__FILE__,__FUNCTION__,__LINE__)
#endif


mcp_QueueHandler* mcp_ClientHandler::m_pQueueHandle = nullptr;

/************************************************************************
 * FUNCTION: mcp_ClientHandler()
 *
 * DESCRIPTION: Constructor
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
mcp_ClientHandler::mcp_ClientHandler():pmcp_fsmMCPUpdActions(nullptr),
iErrorCode(ENMCPCTRLSTATE::MCP_APPCTRL_NONE),
mcpLastError(ENMCPCTRLSTATE::MCP_APPCTRL_NONE)
{
	ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler constructor called"));
	m_pQueueHandle = new mcp_QueueHandler();
	pmcp_fsmMCPUpdActions = getInstanceOfmcp_fsmMCPUpdActions();
	ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler pmcp_fsmMCPUpdActions : %p",pmcp_fsmMCPUpdActions));
}

/************************************************************************
 * FUNCTION: ~mcp_ClientHandler
 * DESCRIPTION: Destruction
 * PARAMETER: None
 * RETURNVALUE: None
 *************************************************************************/
mcp_ClientHandler::~mcp_ClientHandler()
{ 
    if(pmcp_fsmMCPUpdActions != nullptr)	{
	releaseInstanceOfmcp_fsmMCPUpdActions();
	ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler destructor called"));
    }
    if(m_pQueueHandle)
    {
        delete m_pQueueHandle;
        m_pQueueHandle = nullptr;
    }
}


/************************************************************************
 * FUNCTION: update
 * DESCRIPTION: update MCP Firmware
 * PARAMETER: None
 * RETURNVALUE: void
 *************************************************************************/
auto mcp_ClientHandler::update()->void
{
	if(pmcp_fsmMCPUpdActions != nullptr)
	{
		UpdateLog::vSetLogFile ("update_mcp.log");
		UpdateLog::vUpdateLogOpen ();
		pmcp_fsmMCPUpdActions->setUpdateFilePath(filePath);
		ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler::update start"));
		UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::update start", DL_LOG_INFO);
		mcp_Request msg(swupd_fsmMCPUpd::evDefaultSession);
        	mcp_ClientHandler::pclGetQueueHandle()->bPushQueue(msg);
	}
	else
	{
		setLastError(ENMCPCTRLSTATE::MCP_UPDATE_ACTS_NULL);
		ETG_TRACE_ERR(("MCPINFO: mcp_ClientHandler::update(): pmcp_fsmMCPUpdActions is nullptr"));
		UpdateLog::vUpdateLogTrace ("%s MCPINFO: mcp_ClientHandler::update(): pmcp_fsmMCPUpdActions is nullptr. Error code", DL_LOG_ERROR);
	}
}

/************************************************************************
 * FUNCTION:ReadVersion
 * DESCRIPTION: get version and print it on console
 * PARAMETER: None
 * RETURNVALUE: void
 *************************************************************************/
auto mcp_ClientHandler::ReadVersion()->void
{
    if(pmcp_fsmMCPUpdActions != nullptr)
	{
		UpdateLog::vSetLogFile ("update_mcp.log");
		UpdateLog::vUpdateLogOpen ();
		ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler::ReadVersion start"));
		UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::ReadVersion start", DL_LOG_INFO);
		mcp_Request msg(swupd_fsmMCPUpd::evReadMCPSWVersion);
		mcp_ClientHandler::pclGetQueueHandle()->bPushQueue(msg);

	}
	else
	{
		setLastError(ENMCPCTRLSTATE::MCP_UPDATE_ACTS_NULL);
		ETG_TRACE_ERR(("MCPINFO: mcp_ClientHandler::ReadVersion(): pmcp_fsmMCPUpdActions is nullptr"));
		UpdateLog::vUpdateLogTrace ("%s MCPINFO: mcp_ClientHandler::ReadVersion(): pmcp_fsmMCPUpdActions is nullptr. Error code", DL_LOG_ERROR);
	}
}

/************************************************************************
 * FUNCTION:sendEventToFSM
 * DESCRIPTION: send FSM event to update actions
 * PARAMETER: swupd_fsmMCPUpd::FsmEvent
 * RETURNVALUE: None
 *************************************************************************/

void mcp_ClientHandler::sendEventToFSM(swupd_fsmMCPUpd::FsmEvent event)
{
   if(pmcp_fsmMCPUpdActions != nullptr)
   {
      ETG_TRACE_USR4(("MCPINFO: mcp_ClientHandler::sendEventToFSM(%d)", event));
      pmcp_fsmMCPUpdActions->vAcceptEvent(event, 0);
   }
   else
   {
      ETG_TRACE_ERR(("MCPINFO: mcp_ClientHandler::sendEventToFSM(): pmcp_fsmMCPUpdActions is nullptr"));
   }
}

/************************************************************************
 * FUNCTION: createMCPClientHandlerThread
 * DESCRIPTION: creates the MCP client handler thread
 *
 * PARAMETER: None
 * RETURNVALUE: None
 *************************************************************************/
auto mcp_ClientHandler::createMCPClientHandlerThread() -> void
{
	 m_clientthread = std::thread(&mcp_ClientHandler::vLibInterfaceWorkerThread, this);  
}


/************************************************************************
 * FUNCTION: mcpJoinThreads
 * DESCRIPTION: Joins main and Client Handler Thread
 * 				action = MCP_ACTION_UPDATE
 * 						 MCP_ACTION_VERSION
 * PARAMETER: None
 * RETURNVALUE: None
 *************************************************************************/
auto mcp_ClientHandler::mcpJoinThreads () -> void
{
    if (mcp_ClientHandler::m_clientthread.joinable())
    {
        mcp_ClientHandler::m_clientthread.join();
	ETG_TRACE_USR4(("MCPINFO: mcp_ClientHandler::m_clientthread is joined"));
    }
    else
    {
	mcpLastError = ENMCPCTRLSTATE::MCP_APPCTRL_CLIENT_THREAD_CREATION_FAILED;
	ETG_TRACE_ERR(("ERROR: mcp_ClientHandler::m_clientthread is not joined : %u", mcpLastError));
    }
}


/************************************************************************
 * FUNCTION: getLastError
 * DESCRIPTION: Return The most recent return code
 * PARAMETER: None
 * RETURNVALUE: enMcpCtrlState
 *************************************************************************/

auto mcp_ClientHandler::getLastError() -> int_least32_t
{
	return mcpLastError;
}

/************************************************************************
 * FUNCTION: vLibInterfaceWorkerThread
 * DESCRIPTION: Run the thread in a loop until proper status is received
 * 				from MCP Library.
 * PARAMETER: None
 * RETURNVALUE: None
 *************************************************************************/
auto mcp_ClientHandler::vLibInterfaceWorkerThread() ->void
{
   mcp_QueueHandler* pclQueueHandle = mcp_ClientHandler::pclGetQueueHandle();
   bool updatestatus = pmcp_fsmMCPUpdActions->getMCPUpdateStatus();
   while(!updatestatus)
   {
	mcp_Request pReq = pclQueueHandle->vPopQueue();
        vProcessCSMEvent(pReq);
	updatestatus = pmcp_fsmMCPUpdActions->getMCPUpdateStatus();
   }
   setLastError(pmcp_fsmMCPUpdActions->getMCPErrorCode());
}

/************************************************************************
 * FUNCTION: setFilePath
 * DESCRIPTION: Setting the update binary path.
 * PARAMETER: std::string
 * RETURNVALUE: None
 *************************************************************************/
void mcp_ClientHandler::setFilePath(std::string path)
{
	filePath=path;
	ETG_TRACE_USR4(("MCPINFO: mcp_ClientHandler::setUpdateFilePath : %s",path.c_str() ));
}

/************************************************************************
 * FUNCTION: setLastError
 * DESCRIPTION: Setting the last error code.
 * PARAMETER: int_least32_t
 * RETURNVALUE: None
 *************************************************************************/
void mcp_ClientHandler::setLastError(int_least32_t error_code)
{
   mcpLastError = error_code;
   switch(mcpLastError)
   {
	   	case ENMCPCTRLSTATE::MCP_APPCTRL_VERSION_OK:
			ETG_TRACE_USR4(("MCPINFO: mcp_ClientHandler::read SW version status : %d",error_code ));
			ETG_TRACE_USR4(("MCPINFO:mcp_ClientHandler::read SW version status = %s", strenMcpCtrlState[error_code]));
			UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler read SW version status = %s", DL_LOG_INFO,strenMcpCtrlState[error_code]);
	   		ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler::ReadVersion ends"));
			UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::ReadVersion ends", DL_LOG_INFO);
			UpdateLog::vUpdateLogClose();
			break;
			
		case ENMCPCTRLSTATE::MCP_APPCTRL_EXIT_SUCCESS:
		case ENMCPCTRLSTATE::MCP_APPCTRL_UPDATE_SUCCESS:
			ETG_TRACE_USR4(("MCPINFO: mcp_ClientHandler::update status : %d",error_code ));
			ETG_TRACE_USR4(("MCPINFO:mcp_ClientHandler::update status = %s", strenMcpCtrlState[error_code]));
			UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler update status = %s", DL_LOG_INFO,strenMcpCtrlState[error_code]);
	   		ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler::update ends"));
			UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::update ends", DL_LOG_INFO);
			UpdateLog::vUpdateLogClose();
			break;

	   	case ENMCPCTRLSTATE::MCP_APPCTRL_VERSION_ERROR:
			ETG_TRACE_USR4(("MCPINFO: mcp_ClientHandler::read SW version status : %d",error_code ));
			ETG_TRACE_USR4(("MCPINFO:mcp_ClientHandler::read SW version status = %s", strenMcpCtrlState[error_code]));
			UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler read SW version status = %s", DL_LOG_INFO,strenMcpCtrlState[error_code]);
	   		ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler::ReadVersion failed"));
			UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::ReadVersion failed", DL_LOG_INFO);
			UpdateLog::vUpdateLogClose();
			break;
			
		default:
			ETG_TRACE_USR4(("MCPINFO: mcp_ClientHandler::update status : %d",error_code ));
			ETG_TRACE_USR4(("MCPINFO:mcp_ClientHandler::update status = %s", strenMcpCtrlState[error_code]));
			UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler update status = %s", DL_LOG_INFO,strenMcpCtrlState[error_code]);
			ETG_TRACE_USR4 (("MCPINFO: mcp_ClientHandler::update failed"));
			UpdateLog::vUpdateLogClose();
			break;
   }

}

/************************************************************************
* FUNCTION: pclGetQueueHandle
* DESCRIPTION: Get the m_pQueueHandle
* PARAMETER: None
* RETURNVALUE: mcp_QueueHandler
*************************************************************************/
mcp_QueueHandler* mcp_ClientHandler::pclGetQueueHandle()
{
    return (m_pQueueHandle);
}

/************************************************************************
* FUNCTION: vProcessCSMEvent
* DESCRIPTION: ProcessFsmEvent
* PARAMETER: pReq
* RETURNVALUE: None
*************************************************************************/
void mcp_ClientHandler::vProcessCSMEvent(mcp_Request& pReq)
{
    swupd_fsmMCPUpd::FsmEvent dataEvent =  pReq.getEvent();
    
    switch(dataEvent)
    {
    case swupd_fsmMCPUpd::evDataSentConfirmation:
        sendEventToFSM(swupd_fsmMCPUpd::evDataSentConfirmation);        
        break;
    case swupd_fsmMCPUpd::evDataNotSentConfirmation:
        UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::vProcessCSMEvent(%d)", DL_LOG_INFO,swupd_fsmMCPUpd::evDataNotSentConfirmation);
        sendEventToFSM(swupd_fsmMCPUpd::evDataNotSentConfirmation);  
        break;
    case swupd_fsmMCPUpd::evResponseReceived:
        pmcp_fsmMCPUpdActions->vUdsMessageReceived(static_cast<tPCU8>(&pReq.getData().front()),static_cast<tU32>(pReq.getDataLength()));
        sendEventToFSM(swupd_fsmMCPUpd::evResponseReceived);
        break;
    case swupd_fsmMCPUpd::evTimeout:
        setLastError(ENMCPCTRLSTATE::MCP_TIMEOUT_FAILED);
        UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::vProcessCSMEvent(%d)", DL_LOG_INFO,swupd_fsmMCPUpd::evTimeout);
        sendEventToFSM(swupd_fsmMCPUpd::evTimeout);
        break;
    case swupd_fsmMCPUpd::evDefaultSession:
        UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::vProcessCSMEvent(%d)", DL_LOG_INFO,swupd_fsmMCPUpd::evDefaultSession);
        sendEventToFSM(swupd_fsmMCPUpd::evDefaultSession);
        break;
    case swupd_fsmMCPUpd::evReadMCPSWVersion:
        UpdateLog::vUpdateLogTrace ("%s mcp_ClientHandler::vProcessCSMEvent(%d)", DL_LOG_INFO,swupd_fsmMCPUpd::evReadMCPSWVersion);
        sendEventToFSM(swupd_fsmMCPUpd::evReadMCPSWVersion);
        break;
		
    default:   
        break;       
    }
}





