/************************************************************************
* File: MCPUpdaCtrlApp.cpp
* SW-Component:
*
* Description:
*	A CLI Application for performing MCP Module Update
*
* Author:
*   RamaGopalReddy.Bommireddy@in.bosch.com
* Copyright:
*   Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
* History:
* 31.12.2019 - Initial version - RamaGopalReddy.Bommireddy@in.bosch.com
* 28.02.2020 - Revision 1.0    - Surparaju.Pavankumar@in.bosch.com
***********************************************************************/


using namespace std;
#include <iostream>
#include <string>
#include <stdlib.h>
#include <cstdlib>
#include <getopt.h>
#include <memory>
#include "ErrorHandler.h"
#include "mcp_ClientHandler.h"
#include "mcp_Defines.h"
#include "ai_sw_update/common/base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_MCP
  #include "trcGenProj/Header/mcpUpdateCtrlApp.cpp.trc.h"
#endif

#define OSAL_EXIT_WITH_CODE(n) 	{								\
									OSAL_vSetProcessExitCode(n);\
									OSAL_vProcessExit();		\
								}

int_least32_t iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_NONE;
auto help () ->void;
auto vParseCmdArgs (int& c) ->void;
auto vInitClientHandler(std::string data="")->void;
void validateInputFile(const string& sFirmwareFileName);
enMcpAction enAction = enMcpAction::MCP_ACTION_NONE;
std::string strFirmwarePath;
void vWriteErrorLogToErrmem(void);

/************************************************************************
 * FUNCTION: main()
 *
 * DESCRIPTION: Main function
 *
 * PARAMETER: argc,argv
 *
 * RETURNVALUE: int
 *************************************************************************/
 
int main(int argc, char *argv[])
{
    //opening and registering for TTFis traces
    vInitPlatformEtg();
    ETG_I_REGISTER_CHN();

    // To initialize SWUpdate Error Memory
    ai_sw_update::common::iInitializeSwUpdateErrorArray();
    if (argc > 1) 
    {

        DEBUG_TRACE (TRACE_LEVEL_1, "IF INFO: MCPUpdateCtrlApp started");
        // evaluate command line options
        auto indexes = 0;
        auto  c =0;

        static struct option long_options[]=
        {
            {"update"         ,required_argument  ,0,  'u'},	    // Update mcp
			{"help"           ,no_argument        ,0,  'h'},		//display cmd line opts & usage
            {"version"   	  ,no_argument        ,0,  'v'},		//to fetch version of firmware flashed in mcp module
            {0                ,0                  ,0,   0 }
        };

        ETG_TRACE_USR4 (("MCPINFO: MCPUpdateCtrlApp started."));


        while ((c = getopt_long_only (argc, argv, "u:hv",\
                    long_options, &indexes)) != -1) 
                vParseCmdArgs (c);

    }
    else
    {
        DEBUG_TRACE (TRACE_LEVEL_1, "ERROR: No command line option provided for MCPUpdateCtrlApp");
        // no command line options supplied, continue with default behaviour
        ETG_TRACE_ERR (("ERROR: No command line option provided for MCPUpdateCtrlApp"));
        help ();
        iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_SWUPDATE_PARSE_CMDARGS_FAILED;       
    }
    //closing and unregistering for TTFis traces
    ETG_I_UNREGISTER_CHN ();

    vWriteErrorLogToErrmem();
    //In Gen3 projects OSAL layer is exists. Using the OSAL EXIT
    OSAL_EXIT_WITH_CODE (iEXitCode);
}

/************************************************************************
 * FUNCTION: vParseCmdArgs()
 *
 * DESCRIPTION: Parseing the Cmd Args
 *
 * PARAMETER: int
 *
 * RETURNVALUE: None
 *************************************************************************/

auto vParseCmdArgs (int& c)->void
{
    switch (c)
    {
        case 'h':       //display help
        iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_EXIT_SUCCESS;
        help();
        break;

        case 'v':       // fetch MCP firmware version
        enAction = enMcpAction::MCP_ACTION_VERSION;
        vInitClientHandler();
        break;

        case 'u':       // return MCP app version
        enAction = enMcpAction::MCP_ACTION_UPDATE;
        strFirmwarePath = optarg;
        vInitClientHandler();
        break;
        case ':':       //arguments missing
        case '?':       //unrecognized option
        default:
        iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_SWUPDATE_PARSE_CMDARGS_FAILED;
        help ();
        break;
    }
}
/************************************************************************
 * FUNCTION: vInitClientHandler()
 *
 * DESCRIPTION: Calls vInitClientHandler
 *
 * PARAMETER: std::string
 *
 * RETURNVALUE: None
 *************************************************************************/

auto vInitClientHandler(std::string data)->void
{
    auto pMCPHandler = std::unique_ptr<mcp_ClientHandler>(new(std::nothrow) mcp_ClientHandler());

    bool retVal = false;
    if(pMCPHandler)
    {
      switch(enAction)
      {
        case enMcpAction::MCP_ACTION_VERSION:
		    pMCPHandler->createMCPClientHandlerThread();
			ETG_TRACE_USR4 (("MCPINFO: MCPUpdateCtrlApp: MCPClientHandlerThread is created."));
			if (ENMCPCTRLSTATE::MCP_APPCTRL_CLIENT_THREAD_CREATION_FAILED != pMCPHandler->getLastError())
			{
				pMCPHandler->ReadVersion();
				pMCPHandler->mcpJoinThreads();
			}
			if((pMCPHandler->getLastError()) == ENMCPCTRLSTATE::MCP_APPCTRL_VERSION_OK)
				iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_EXIT_SUCCESS;
			vWriteErrorLogToErrmem();
			ETG_TRACE_USR4 (("MCPINFO: MCPUpdateCtrlApp iEXitCode : %u",iEXitCode ));
		break;

		case enMcpAction::MCP_ACTION_UPDATE:
			validateInputFile(strFirmwarePath);
			pMCPHandler->createMCPClientHandlerThread();
			ETG_TRACE_USR4 (("MCPINFO: MCPUpdateCtrlApp: MCPClientHandlerThread is created."));
			pMCPHandler->setFilePath(strFirmwarePath);
			if (ENMCPCTRLSTATE::MCP_APPCTRL_CLIENT_THREAD_CREATION_FAILED != pMCPHandler->getLastError())
			{
				pMCPHandler->update();
				pMCPHandler->mcpJoinThreads();
			}
			ETG_TRACE_USR4 (("MCPINFO: MCPUpdateCtrlApp vInitClientHandler retVal : %u",retVal ));
			if((pMCPHandler->getLastError() == ENMCPCTRLSTATE::MCP_APPCTRL_EXIT_SUCCESS) ||
			(pMCPHandler->getLastError() == ENMCPCTRLSTATE::MCP_APPCTRL_UPDATE_SUCCESS))
				iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_EXIT_SUCCESS;
			else
				iEXitCode = pMCPHandler->getLastError();
                        
                        vWriteErrorLogToErrmem();
			ETG_TRACE_USR4 (("MCPINFO: MCPUpdateCtrlApp iEXitCode : %u",iEXitCode ));

                break;
        default:
                break;
      }
    }
    else
    {
        //client interface creation failed
        iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_NULL_POINTER_ERROR;
        vWriteErrorLogToErrmem();
        ETG_TRACE_ERR (("ERROR:pMCPHandler object creation failed"));
    }
   
}

/************************************************************************
 * FUNCTION: validateInputFile()
 *
 * DESCRIPTION: Validates the input binary file
 *
 * PARAMETER: string
 *
 * RETURNVALUE: None
 *************************************************************************/
void validateInputFile(const string& sFirmwareFileName) {

    // If firmware file is not specified exit the program
    if (sFirmwareFileName.empty()) 
    {
        ETG_TRACE_ERR (("ERROR: No command line option provided for MCPUpdateCtrlApp"));
        help();
        iEXitCode = ENMCPCTRLSTATE::MCP_APPCTRL_SWUPDATE_PARSE_CMDARGS_FAILED; 
        //closing and unregistering for TTFis traces
        ETG_I_UNREGISTER_CHN ();
        vWriteErrorLogToErrmem();
        OSAL_EXIT_WITH_CODE(iEXitCode);
    }

}
/************************************************************************
 * FUNCTION: help()
 *
 * DESCRIPTION: Displays command usage
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None
 *************************************************************************/
auto help ()->void
{
    //print the usage of MCP update control
    cout <<"\n \
        Options supported:\n \
            1. -version (-v)\n \
            2. -update (-u)\n \
            3. -help (-h)\n \
        Usage:\n \
            1. Updates of mcp Update control :\n \
                ./swu_common_MCP_app_out.out -u\n \
            2. Display options & usage:\n \
                ./swu_common_MCP_app_out.out -h\n \
            3. Returns version of firmware flashed in MCP module:\n\
                ./swu_common_MCP_app_out.out -v\n " ;

    vWriteErrorLogToErrmem();
    OSAL_EXIT_WITH_CODE (iEXitCode);
}

void vWriteErrorLogToErrmem(void)
{
	// to capture mcp error exit code in Errmem
	if ( (iEXitCode != ENMCPCTRLSTATE::MCP_APPCTRL_EXIT_SUCCESS) && \
	     (iEXitCode != ENMCPCTRLSTATE::MCP_APPCTRL_UPDATE_SUCCESS) &&\
             (iEXitCode != ENMCPCTRLSTATE::MCP_APPCTRL_VERSION_OK) )

	{
		// To fix CID 2750548
		try{
			iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_MCP,
                        (int)SW_UPDATE_ERROR_TYPE_VERIFY,
                        "MCP EXIT Code:",
                        strenMcpCtrlState[iEXitCode]);
		   }
		catch(std::bad_alloc const& e)
		   {
		   }
	}
}
