
/******************************************************************************************
*FILE					: swuhmi_drmrender.cpp
*SW-COMPONENT			: SW-Update
*DESCRIPTION			: DRM (Direct Render Manager).
*DESCRIPTION			: It Creates FrameBuffer from which widgets are drawn on it.
*AUTHOR					: VRI7COB (RBEI/ECA)
*COPYRIGHT				: (c) 1996 – 2000 Blaupunkt Werke GmbH
*HISTORY				:
*Date 		| Rev. 		| Author 			| Modification
---------------------------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
13-07-2017	| 0.2		| ALA5COB			| CMG3G-14492-Improvements in the recovery hmi
*********************************************************************************************/

#include "swuhmi_drmrender.h"

#include "util/swu_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "SWUHMI_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_SWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_HMI
#include "trcGenProj/Header/swuhmi_drmrender.cpp.trc.h"
#endif

static struct modeset_dev *modeset_list = NULL;

/************************************************************************
*FUNCTION			: drmrender
*DESCRIPTION		: Constructor of drmrender
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
drmrender::drmrender() {
	m_FrameBufferfd = 0;
	m_pDRMDevice = NULL;
	m_pDRMBuff = NULL;

}

/************************************************************************
*FUNCTION			: drmrender
*DESCRIPTION		: Destructor of drmrender
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
drmrender::~drmrender() {
	// TODO Auto-generated destructor stub
	if(m_pDRMDevice != NULL){
		delete m_pDRMDevice;
		m_pDRMDevice = NULL;
	}
	if(m_pDRMBuff != NULL){
		delete m_pDRMBuff;
		m_pDRMBuff = NULL;
	}
}

/************************************************************************
*FUNCTION			: getDRMDevice
*DESCRIPTION		: Fill the correct DRM device
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
24-01-2018	| 0.1		| RRD7KOR			| Initial
************************************************************************/

void drmrender::getDRMDevice(const char **drmDevice)
{
    struct udev *udev;
    struct udev_enumerate *enumerate;
    struct udev_list_entry *dev_list_entry;
    struct udev_device *device;
    drmModeRes *res;

    /* Create the udev object */
    udev = udev_new();
    if (!udev) {
        fprintf(stderr,"Can't create udev\n");
        exit(1);
    }

    enumerate = udev_enumerate_new(udev);
    udev_enumerate_add_match_subsystem(enumerate, "drm");
    udev_enumerate_add_match_sysname(enumerate, "card[0-9]*");
    udev_enumerate_scan_devices(enumerate);
    udev_list_entry_foreach(dev_list_entry, udev_enumerate_get_list_entry(enumerate)) {
        const char *path;
        const char *filename;
        const char *sysnum;
        int fd = -1;
        int id;
        path = udev_list_entry_get_name(dev_list_entry);
        device = udev_device_new_from_syspath(udev, path);
        filename = udev_device_get_devnode(device);
        sysnum = udev_device_get_sysnum(device);

        fd = open(filename, O_RDWR);
	//Coverity fix 173922
	if(-1 == fd)
	{
		fprintf(stderr,"open system call is failed\n");
		return;
	}
        res = drmModeGetResources(fd);
        if (!res)
        {
            close(fd);
            continue;
        }
        if (res->count_crtcs <= 0 || res->count_connectors <= 0 ||
            res->count_encoders <= 0)
	{
	    close(fd);   //Coverity fix for 173926
            continue;
	}

        if (sysnum)
            id = atoi(sysnum);
        if (!sysnum || id < 0) {
	    close(fd);  //Coverity fix for 173926
            continue;
        }
        *drmDevice = filename;
        fprintf(stderr,"drmDevice=%s\n",*drmDevice);
        close(fd);
        break;
    }
}


/************************************************************************
*FUNCTION			: InitializeDRMRender
*DESCRIPTION		: Initialize the DRM Render
*PARAMETER			: None
*RETURNVALUE		: int
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
int drmrender::InitializeDRMRender()
{
	ETG_TRACE_USR4(("InitializeDRMRender"));
	int ret = -1;   //Coverity fix for 194566

	const char *card = NULL;
   
    struct modeset_dev *iter;
    struct modeset_buf *buf;
   	
#ifndef VARIANT_S_FTR_ENABLE_G4G
    card = "/dev/dri/card0";
#else
    getDRMDevice(&card);
#endif

    fprintf(stderr,"card=%s\n",card);

    if(card == NULL)
    {
        ETG_TRACE_USR4(("Error in finding the Card"));
        goto out_return;
    }

	/* open the DRM device */
	ret = OpenDRM(&m_FrameBufferfd, card);
	if (ret)
	{
		ETG_TRACE_USR4(("Error in OpenDRM"));
		/*lint -e{801} suppress Use of goto is deprecated*/
		goto out_return;	
	}

	/* prepare all connectors and CRTCs */
	ret = PrepareDRM(m_FrameBufferfd);
	if (ret)
	{
		ETG_TRACE_USR4(("Error in PrepareDRM"));
		/*lint -e{801} suppress Use of goto is deprecated*/
		goto out_close;
	}

	/* perform actual modesetting on each found connector+CRTC */
	for (iter = modeset_list; iter; iter = iter->next) {
		iter->saved_crtc = drmModeGetCrtc(m_FrameBufferfd, iter->crtc);
		buf = &iter->bufs[iter->front_buf];
		ret = drmModeSetCrtc(m_FrameBufferfd, iter->crtc, buf->fb, 0, 0,
					 &iter->conn, 1, &iter->mode);
		if (ret)
			fprintf(stderr, "cannot set CRTC for connector %u (%d): %m\n",
				iter->conn, errno);
	}

	ret =0;
	return ret;

out_close:
	close(m_FrameBufferfd);
out_return:
	if (ret) {
		errno = -ret;
		fprintf(stderr, "modeset failed with error %d: %m\n", errno);
	} /*else {
		fprintf(stderr, "exiting\n");
	}*/  //Removing else condition for coverity fix 21775

	return ret;

}

/************************************************************************
*FUNCTION			: OpenDRM
*DESCRIPTION		: Open the DRM Render.
*PARAMETER			: int (O)
*PARAMETER				FileDescriptor.
*PARAMETER			  const char(I)
*PARAMETER			  	Node
*RETURNVALUE		: int
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
int drmrender::OpenDRM(int *out, const char *node)
{
	ETG_TRACE_USR4(("OpenDRM"));
	int fd, ret;
	uint64_t has_dumb;

	fd = open(node, O_RDWR | O_CLOEXEC);
	if (fd < 0) {
		ret = -errno;
		fprintf(stderr, "cannot open '%s': %m\n", node);
		return ret;
	}

	if (drmGetCap(fd, (tU64)DRM_CAP_DUMB_BUFFER, &has_dumb) < 0 ||
		!has_dumb) {
		fprintf(stderr, "drm device '%s' does not support dumb buffers\n",
			node);
		close(fd);
		return -EOPNOTSUPP;
	}

	*out = fd;
	return 0;
}

/************************************************************************
*FUNCTION			: PrepareDRM
*DESCRIPTION		: prepare the DRM Render.
*PARAMETER			: int (I)
*PARAMETER				FileDescriptor.
*RETURNVALUE		: int
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
int drmrender::PrepareDRM(int fd)
{
	ETG_TRACE_USR4(("PrepareDRM"));
	drmModeRes *res;
	drmModeConnector *conn;
	int i;	// gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions: At next for loop on comparing with res->count_connectors
	struct modeset_dev *dev;
	int ret;

	/* retrieve resources */
	res = drmModeGetResources(fd);
	if (!res) {
		fprintf(stderr, "cannot retrieve DRM resources (%d): %m\n",
			errno);
		return -errno;
	}

	/* iterate all connectors */
	/*lint -e{574} suppress 574 for Signed-unsigned mix with relational*/
	for (i = 0; i < res->count_connectors; ++i)
	{
		/* get information for each connector */
		conn = drmModeGetConnector(fd, res->connectors[i]);
		if (!conn) {
			fprintf(stderr, "cannot retrieve DRM connector %u:%u (%d): %m\n",
				i, res->connectors[i], errno);
			continue;
		}

		/* create a device structure */
		dev = (modeset_dev*)malloc(sizeof(*dev));
		if(dev != NULL)
		{
			memset(dev, 0, sizeof(*dev));
			dev->conn = conn->connector_id;

			/* call helper function to prepare this connector */
			ret = SetUpDRM(fd, res, conn, dev);

			if (ret) 
			{
				if (ret != -ENOENT) 
				{
					errno = -ret;
					fprintf(stderr, "cannot setup device for connector %u:%u (%d): %m\n",
						i, res->connectors[i], errno);
				}
				free(dev);
				drmModeFreeConnector(conn);
				continue;
			}

			/* free connector data and link device into global list */
			drmModeFreeConnector(conn);
			dev->next = modeset_list;
			modeset_list = dev;
		}
	}

	/* free resources again */
	drmModeFreeResources(res);
	return 0;
}

/************************************************************************
*FUNCTION			: SetUpDRM
*DESCRIPTION		: Setup the DRM Render.
*PARAMETER			: int (I)
*PARAMETER				FileDescriptor.
*PARAMETER			  pointer(I)
*PARAMETER			  	drmModeRes
*PARAMETER			  pointer(I)
*PARAMETER			  	drmModeConnector
*PARAMETER			  struct(I)
*PARAMETER			    modeset_dev
*RETURNVALUE		: int
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
int drmrender::SetUpDRM(int fd, drmModeRes *res, drmModeConnector *conn,struct modeset_dev *dev)
{	 
	ETG_TRACE_USR4(("SetUpDRM"));
	int ret;

	/* check if a monitor is connected */
	if (conn->connection != DRM_MODE_CONNECTED) {
		fprintf(stderr, "ignoring unused connector %u\n",
			conn->connector_id);
		return -ENOENT;
	}

	/* check if there is at least one valid mode */
	if (conn->count_modes == 0) {
		fprintf(stderr, "no valid mode for connector %u\n",
			conn->connector_id);
		return -EFAULT;
	}

	/* copy the mode information into our device structure and into both
		 * buffers */
	memcpy(&dev->mode, &conn->modes[0], sizeof(dev->mode));
	dev->bufs[0].width = conn->modes[0].hdisplay;
	dev->bufs[0].height = conn->modes[0].vdisplay;
	dev->bufs[1].width = conn->modes[0].hdisplay;
	dev->bufs[1].height = conn->modes[0].vdisplay;
	fprintf(stderr, "mode for connector %u is %ux%u\n",
		conn->connector_id, dev->bufs[0].width, dev->bufs[0].height);

	/* find a crtc for this connector */
	ret = FindCRTC(fd, res, conn, dev);
	if (ret) {
		fprintf(stderr, "no valid crtc for connector %u\n",
			conn->connector_id);
		return ret;
	}

	/* create framebuffer #1 for this CRTC */
	ret = CreateFrameBuff(fd, &dev->bufs[0]);
	if (ret) {
		fprintf(stderr, "cannot create framebuffer for connector %u\n",
			conn->connector_id);
		return ret;
	}

	/* create framebuffer #2 for this CRTC */
	ret = CreateFrameBuff(fd, &dev->bufs[1]);
	if (ret) {
		fprintf(stderr, "cannot create framebuffer for connector %u\n",
			conn->connector_id);
		DestroyFrameBuff(fd, &dev->bufs[0]);
		return ret;
	}

	return 0;
	/*lint -e{429} suppress 429 for Custodial pointer*/
}

/************************************************************************
*FUNCTION			: FindCRTC
*DESCRIPTION		: FindCRTC
*PARAMETER			: int (I)
*PARAMETER				FileDescriptor.
*PARAMETER			  pointer(I)
*PARAMETER			  	drmModeRes
*PARAMETER			  pointer(I)
*PARAMETER			  	drmModeConnector
*RETURNVALUE		: int
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
int drmrender::FindCRTC(int fd, drmModeRes *res, drmModeConnector *conn,
			     struct modeset_dev *dev)
{
	ETG_TRACE_USR4(("FindCRTC"));
	drmModeEncoder *enc;
	int i, j;	// gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions: since it is used for comparing with conn->count_encoders, conn->count_critics
	int32_t crtc;
	struct modeset_dev *iter;

	/* first try the currently conected encoder+crtc */
	if (conn->encoder_id)
		enc = drmModeGetEncoder(fd, conn->encoder_id);
	else
		enc = NULL;

	if (enc) {
		if (enc->crtc_id) {
			crtc = enc->crtc_id;
			for (iter = modeset_list; iter; iter = iter->next) {
				if (static_cast<int32_t> (iter->crtc) == crtc) {	// gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
					crtc = -1;
					break;
				}
			}

			if (crtc >= 0) {
				drmModeFreeEncoder(enc);
				dev->crtc = crtc;
				return 0;
			}
		}

		drmModeFreeEncoder(enc);
	}

	/* If the connector is not currently bound to an encoder or if the
	 * encoder+crtc is already used by another connector (actually unlikely
	 * but lets be safe), iterate all other available encoders to find a
	 * matching CRTC. */
	/*lint -e{574} suppress 574 for Signed-unsigned mix with relational*/
	for (i = 0; i < conn->count_encoders; ++i) {
		enc = drmModeGetEncoder(fd, conn->encoders[i]);
		if (!enc) {
			fprintf(stderr, "cannot retrieve encoder %u:%u (%d): %m\n",
				i, conn->encoders[i], errno);
			continue;
		}

		/* iterate all global CRTCs */
		/*lint -e{574} suppress 574 for Signed-unsigned mix with relational*/
		for (j = 0; j < res->count_crtcs; ++j) {
			/* check whether this CRTC works with the encoder */
			if (!(enc->possible_crtcs & (1 << j)))
				continue;

			/* check that no other device already uses this CRTC */
			crtc = res->crtcs[j];
			for (iter = modeset_list; iter; iter = iter->next) {
				if (static_cast<int32_t> (iter->crtc) == crtc) {	// gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
					crtc = -1;
					break;
				}
			}

			/* we have found a CRTC, so save it and return */
			if (crtc >= 0) {
				drmModeFreeEncoder(enc);
				dev->crtc = crtc;
				return 0;
			}
		}

		drmModeFreeEncoder(enc);
	}

	fprintf(stderr, "cannot find suitable CRTC for connector %u\n",
		conn->connector_id);
	return -ENOENT;
	/*lint -e{429} suppress 429 for Custodial pointer*/
}

/************************************************************************
*FUNCTION			: CreateFrameBuff
*DESCRIPTION		: CreateFrameBuff
*PARAMETER			: int (I)
*PARAMETER				FileDescriptor.
*PARAMETER			  struct(I)
*PARAMETER			  	modeset_buf
*RETURNVALUE		: int
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
int drmrender::CreateFrameBuff(int fd, struct modeset_buf *buf)
{
	ETG_TRACE_USR4(("CreateFrameBuff"));
	struct drm_mode_create_dumb creq;
	struct drm_mode_destroy_dumb dreq;
	struct drm_mode_map_dumb mreq;
	int ret;

	/* create dumb buffer */
	memset(&creq, 0, sizeof(creq));
	creq.width = buf->width;
	creq.height = buf->height;
	creq.bpp = 32;
	ret = drmIoctl(fd, DRM_IOCTL_MODE_CREATE_DUMB, &creq);
	if (ret < 0) {
		fprintf(stderr, "cannot create dumb buffer (%d): %m\n",
			errno);
		return -errno;
	}
	buf->stride = creq.pitch;
	buf->size =static_cast<uint32_t> (creq.size);
	buf->handle = creq.handle;

	/* create framebuffer object for the dumb-buffer */
	ret = drmModeAddFB(fd, buf->width, buf->height, 24, 32, buf->stride,
			   buf->handle, &buf->fb);
	if (ret) {
		fprintf(stderr, "cannot create framebuffer (%d): %m\n",
			errno);
		ret = -errno;
		/*lint -e{801} suppress Use of goto is deprecated*/
		goto err_destroy;
	}

	/* prepare buffer for memory mapping */
	memset(&mreq, 0, sizeof(mreq));
	mreq.handle = buf->handle;
	ret = drmIoctl(fd, DRM_IOCTL_MODE_MAP_DUMB, &mreq);
	if (ret) {
		fprintf(stderr, "cannot map dumb buffer (%d): %m\n",
			errno);
		ret = -errno;
		/*lint -e{801} suppress Use of goto is deprecated*/
		goto err_fb;
	}

	/*lint -e{747} */
	buf->map = (uint8_t*)mmap(0, buf->size, PROT_READ | PROT_WRITE, MAP_SHARED,
				fd, (long)mreq.offset);
	if (buf->map == MAP_FAILED) {
		fprintf(stderr, "cannot mmap dumb buffer (%d): %m\n",
			errno);
		ret = -errno;
		/*lint -e{801} suppress Use of goto is deprecated*/
		goto err_fb;
	}

	/* clear the framebuffer to 0 */
	memset(buf->map, 0, buf->size);

	return 0;

err_fb:
	drmModeRmFB(fd, buf->fb);
err_destroy:
	memset(&dreq, 0, sizeof(dreq));
	dreq.handle = buf->handle;
	drmIoctl(fd, DRM_IOCTL_MODE_DESTROY_DUMB, &dreq);
	return ret;
}

/************************************************************************
*FUNCTION			: DestroyFrameBuff
*DESCRIPTION		: DestroyFrameBuff
*PARAMETER			: int (I)
*PARAMETER				FileDescriptor.
*PARAMETER			  struct(I)
*PARAMETER			  	modeset_buf
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
void drmrender::DestroyFrameBuff(int fd, struct modeset_buf *buf)
{
	ETG_TRACE_USR4(("DestroyFrameBuff"));
	struct drm_mode_destroy_dumb dreq;

	/* unmap buffer */
	munmap(buf->map, buf->size);

	/* delete framebuffer */
	drmModeRmFB(fd, buf->fb);

	/* delete dumb buffer */
	memset(&dreq, 0, sizeof(dreq));
	dreq.handle = buf->handle;
	drmIoctl(fd, DRM_IOCTL_MODE_DESTROY_DUMB, &dreq);
}


/************************************************************************
*FUNCTION			: DeInitializeDRMRender
*DESCRIPTION		: DeInitializeDRMRender
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
void drmrender::DeInitializeDRMRender()
{
	ETG_TRACE_USR4(("DeInitializeDRMRender"));
	struct modeset_dev *iter;
	drmEventContext ev;
	int ret;

	/* init variables */
	memset(&ev, 0, sizeof(ev));
	ev.version = DRM_EVENT_CONTEXT_VERSION;
	ev.page_flip_handler = modeset_page_flip_event;

	while (modeset_list) {
		/* remove from global list */
		iter = modeset_list;
		modeset_list = iter->next;

		/* if a pageflip is pending, wait for it to complete */
		iter->cleanup = true;
		fprintf(stderr, "wait for pending page-flip to complete...\n");
		while (iter->pflip_pending) {
			ret = drmHandleEvent(m_FrameBufferfd, &ev);
			if (ret)
				break;
		}

		/* restore saved CRTC configuration */
		if (!iter->pflip_pending)
			drmModeSetCrtc(m_FrameBufferfd,
					   iter->saved_crtc->crtc_id,
					   iter->saved_crtc->buffer_id,
					   iter->saved_crtc->x,
					   iter->saved_crtc->y,
					   &iter->conn,
					   1,
					   &iter->saved_crtc->mode);
		drmModeFreeCrtc(iter->saved_crtc);

		/* destroy framebuffers */
		DestroyFrameBuff(m_FrameBufferfd, &iter->bufs[1]);
		DestroyFrameBuff(m_FrameBufferfd, &iter->bufs[0]);

		/* free allocated memory */
		free(iter);
	}
}

/************************************************************************
*FUNCTION			: modeset_page_flip_event
*DESCRIPTION		: Flip the Page.
*PARAMETER			: int (I)
*PARAMETER				File Descriptor
*PARAMETER			  uint (I)
*PARAMETER			    Frame
*PARAMETER			  uint (I)
*PARAMETER			    sec
*PARAMETER			  pointer (I)
*PARAMETER			    data
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
void drmrender::modeset_page_flip_event(int fd, unsigned int frame,
				    unsigned int sec, unsigned int usec,
				    void *data)
{
// for gen3 and gen4 compatibality
//	(void*)sec;
//	(void*)fd;
//	(void*)frame;
//	(void*)usec;

	ETG_TRACE_USR4(("modeset_page_flip_event"));
	struct modeset_dev *dev = (modeset_dev*)data;

	dev->pflip_pending = false;
//    if (!dev->cleanup)
//        modeset_draw_dev(fd, dev);
}

/*********************************************************************************************
*FUNCTION			: SetDrawEnv
*DESCRIPTION		: SetDrawEnv
*PARAMETER			: None
*RETURNVALUE		: bool
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
13-07-2017	| 0.2		| ALA5COB			| CMG3G-14492-Improvements in the recovery hmi
*********************************************************************************************/
bool drmrender::SetDrawEnv()
{
	ETG_TRACE_USR4(("SetDrawEnv"));

	bool breturn = true;

	m_pDRMDevice 			= modeset_list;
	if(m_pDRMDevice !=NULL)
	{

		m_pDRMDevice->r 		=static_cast<uint8_t> (rand() % 0xff);
		m_pDRMDevice->g 		=static_cast<uint8_t> ( rand() % 0xff);
		m_pDRMDevice->b 		=static_cast<uint8_t> ( rand() % 0xff);
		m_pDRMDevice->r_up 		= m_pDRMDevice->g_up = m_pDRMDevice->b_up = true;

		//fprintf(stderr, "Buffer:%u\n", m_pDRMDevice->front_buf ^ 1);

		m_pDRMBuff 				= &m_pDRMDevice->bufs[m_pDRMDevice->front_buf ^ 1];
		m_pDRMDevice->front_buf = m_pDRMDevice->front_buf ^ 1;


		ETG_TRACE_USR4(("Pitch:%u", m_pDRMBuff->stride));
	}
	else
	{
		fprintf(stderr,"drmrender : modeset_list is NULL,So recovery hmi can not be up..... \n");
		breturn = false;
	}
	return breturn;
}

/************************************************************************
*FUNCTION			: FlipThePage
*DESCRIPTION		: FlipThePage
*PARAMETER			: None
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
void drmrender::FlipThePage()
{
	int ret = 0;

	if(m_pDRMDevice != NULL && m_pDRMBuff != NULL){
		ret = drmModeSetCrtc(m_FrameBufferfd, m_pDRMDevice->crtc, m_pDRMBuff->fb, 0, 0, &m_pDRMDevice->conn, 1, &m_pDRMDevice->mode);
//		ret = drmModePageFlip(m_FrameBufferfd, m_pDRMDevice->crtc, m_pDRMBuff->fb,
//				      DRM_MODE_PAGE_FLIP_EVENT, m_pDRMDevice);
	}

	if (ret)
	{
		if(m_pDRMDevice != NULL){
			ETG_TRACE_USR4(("cannot flip CRTC for connector %u (%d):", m_pDRMDevice->conn, errno));
		}
	}
	else
	{
		ETG_TRACE_USR4(("Flip Page is Successful"));
	}

}


/************************************************************************
*FUNCTION			: MarkDirtyRegion
*DESCRIPTION		: MarkDirtyRegion
*PARAMETER			: u16(I)
*PARAMETER				xPos
*PARAMETER			  u16(I)
*PARAMETER			    YPos
*PARAMETER			  u16(I)
*PARAMETER			  	width
*PARAMETER			  u16(I)
*PARAMETER			  	Height
*RETURNVALUE		: void
*HISTORY:
*Date 		| Rev. 		| Author 			| Modification
----------------------------------------------------------------------------
20-02-2016	| 0.1		| VRI7COB			| Initial
************************************************************************/
void drmrender::MarkDirtyRegion(tU16 u16XPos, tU16 u16YPos, tU16 u16Width, tU16 u16Height)
{
	ETG_TRACE_USR4(("MarkDirtyRegion"));

	drmModeClipPtr pClipPosition = new drm_clip_rect;
	if(pClipPosition != NULL) 
	{
		pClipPosition->x1 = u16XPos;
		pClipPosition->y1 = u16YPos;
		pClipPosition->x2 = u16Width;
		pClipPosition->y2 = u16Height;
	}

	if(m_pDRMDevice != NULL)
	{
		if(drmModeDirtyFB(m_FrameBufferfd, m_pDRMDevice->crtc, pClipPosition, 1) == -EINVAL)
			printf("Dirty is not Set \n");
		else
			printf("Dirty is Set \n");
	}


	if(pClipPosition != NULL) {
		delete pClipPosition;
		pClipPosition = NULL;
	}
}

uint8_t drmrender::next_color(bool *up, uint8_t cur, unsigned int mod)
{
	uint8_t next;

	/*lint -e{573}*/
	//Coverity fix for 22062.As coverity tool suggested to use a compliant random number generator, such as /dev/random
	next =static_cast<uint8_t> ( cur + (*up ? 1 : -1) * ( rand() % mod) );
	if ((*up && next < cur) || (!*up && next > cur)) {
		*up = !*up;
		next = cur;
	}

	return next;
}


void drmrender::modeset_draw(void)
{
	ETG_TRACE_USR4(("modeset_draw"));

	uint8_t r, g, b;
	bool r_up, g_up, b_up;
	unsigned int i, j, k, off;
	struct modeset_dev *iter;

	srand((static_cast<unsigned int> (time(NULL))));
	//Coverity fix for 22063.As coverity tool suggested to use a compliant random number generator, such as /dev/random
	r =static_cast<uint8_t> ( rand() % 0xff);
	g =static_cast<uint8_t> ( rand() % 0xff);
	b =static_cast<uint8_t> ( rand() % 0xff);
	r_up = g_up = b_up = true;

	for (i = 0; i < 50; ++i)
	{
		r = next_color(&r_up, r, 20);
		g = next_color(&g_up, g, 10);
		b = next_color(&b_up, b, 5);

		for (iter = modeset_list; iter; iter = iter->next)
		{
			ETG_TRACE_USR4(("Iter List"));
			for (j = 0; j < iter->bufs[0].height; ++j)
			{
				for (k = 0; k < iter->bufs[0].width; ++k)
				{
					off = iter->bufs[0].stride * j + k * 4;
					/*lint -e{826} suppress Use of goto is deprecated*/
					*(uint32_t*)&iter->bufs[0].map[off] = (r << 16) | (g << 8) | b;
				}
			}
		}

		usleep(100000);
	}
}

void drmrender::modeset_drawdoubleBuffer() {

	uint8_t r, g, b;
	bool r_up, g_up, b_up;
	unsigned int i, j, k, off;
	struct modeset_dev *iter;
	struct modeset_buf *buf;
	int ret;

	srand((static_cast<unsigned int> (time(NULL))) );
	//Coverity fix for 22210.As coverity tool suggested to use a compliant random number generator, such as /dev/random
	r =static_cast<uint8_t> ( rand() % 0xff);
	g =static_cast<uint8_t> ( rand() % 0xff);
	b =static_cast<uint8_t> ( rand() % 0xff);
	r_up = g_up = b_up = true;

		for (i = 0; i < 50; ++i) {
			r = next_color(&r_up, r, 20);
			g = next_color(&g_up, g, 10);
			b = next_color(&b_up, b, 5);

			for (iter = modeset_list; iter; iter = iter->next) {
				buf = &iter->bufs[iter->front_buf ^ 1];
				for (j = 0; j < buf->height; ++j) {
					for (k = 0; k < buf->width; ++k) {
						off = buf->stride * j + k * 4;
						/*lint -e{826} */
						*(uint32_t*)&buf->map[off] =
							     (r << 16) | (g << 8) | b;
					}
				}

				ret = drmModeSetCrtc(m_FrameBufferfd, iter->crtc, buf->fb, 0, 0,
						     &iter->conn, 1, &iter->mode);
				if (ret)
					fprintf(stderr, "cannot flip CRTC for connector %u (%d): %m\n",
						iter->conn, errno);
				else {
					iter->front_buf ^= 1;
				}
			}

			usleep(100000);
		}
}

