/*************************************************************************
* FILE:         SWUpdateControlMain.cpp
* PROJECT:      Gen3 Platform
* SW-COMPONENT: SW Update
*----------------------------------------------------------------------
*
* DESCRIPTION:  Application to restart the system in the recovery download mode when a valid USB device is detected.
*
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim
**************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include <errno.h>
#include <string>
#include <cstdio>
#include <map>
#include <unistd.h>
#include <time.h>
#include <getopt.h>
#include <fcntl.h>
#include "include/SWUpdateControlMain.h"
#include "include/SWUpdateControl.h"
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include <wordexp.h>

#include "base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_I_TTFIS_CMD_PREFIX "SWUCTRL_"
  #define ETG_I_TRACE_CHANNEL     TR_TTFIS_SWUPDATE_CTRL
  #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_TEST
  #include "trcGenProj/Header/SWUpdateControlMain.cpp.trc.h"
#endif



using namespace std;
using namespace ai_sw_update::common;


// constants for command line option arguments
const string MODE_RECOVERY("recovery");
const string MODE_NORMAL("normal");
const string RESET_PROCESSOR_SYSTEM("system");
const string RESET_PROCESSOR_APP("app");
const string RESET_PROCESSOR_SCC("scc");
const string SHUTDOWN_NORMAL("normal");
const string SHUTDOWN_ACKNOWLEDGE("ack");
const string BOOT_CHAIN_ONE("1");
const string BOOT_CHAIN_TWO("2");

const int   GETV850MODE=254;
const int   SETV850MODE=253;
const int   V850CHECKUPD=252;
const int   GETRSTCNT=251;
const int   SETRSTCNT=250;
const int   GETFRCDNL=249;
const int   SETFRCDNL=248;
const int   NORMAGIC=247;
const int   WRITENORVALUE=246;
const int   KDSKEY=245;
const int   KDSOFF=244;
const int   READKDSLEN=243;
const int   WRITEKDSVAL=242;
const int   INCPORT=241;
const int   INCHOST=240;
const int   INCSENDVAL=239;
const int   INCSENDRECVVAL=238;
const int   BCKG_AS_USER=237;
const int   EXEC_AS_USER=236;
const int   GETSECMODE=235;
const int   GETSRKREVO=234;
const int   PERFSRVR=233;
const int   PERFSTART=232;
const int   PERFSTOP=231;
const int   PERFSTOPALL=230;
const int   PERFDUMP=229;
const int   PROGRAM_VERSION=228;
const int   CPLD_VERSION=227;
const int   SETFCID=226;
const int   REGISTRY_KEY=225;
const int   REGISTRY_VALUE=224;
const int   GETEMMCINFO=223;

#define OSAL_EXIT_WITH_CODE(n) 	{								\
									OSAL_vSetProcessExitCode(n);\
									OSAL_vProcessExit();		\
								}

// This is a hack because for some unknown reason it is not possible to redirect stdout to a file in dlinitramfs when used directly
// Using a duplicate of stdout's file descriptor works, which is even more strange
void print2stdout(const char *Format, ...)
{
  int     fd=dup(STDOUT_FILENO);
  char    msg[1024];
  va_list arglst;

  if(fd<0)
    return;

  va_start(arglst, Format);
  vsnprintf(msg, sizeof(msg), Format, arglst);
  va_end(arglst);
  write(fd, msg, strlen(msg));
  close(fd);
}

void print2errmem(const char *Format, ...)
{
  int     fd=open("/dev/errmem", O_WRONLY);
  char    msg[1024];
  va_list arglst;

  if(fd<0)
    return;

  va_start(arglst, Format);
  vsnprintf(msg, sizeof(msg), Format, arglst);
  va_end(arglst);
  write(fd, msg, strlen(msg));
  close(fd);
}

void hexdump(const unsigned char *Buff, const int Len)
{
  int i, j;

  for(i=0; i<Len; i+=16)
  {
    print2stdout("%04i:", i);
    for(j=0; j<8 && i+j<Len; j++)
      print2stdout(" %02X", Buff[i+j]);
    if(i+8<Len)
      print2stdout(" -");
    else
      print2stdout("  ");
    for(j=8; j<16 && i+j<Len; j++)
      print2stdout(" %02X", Buff[i+j]);

    for(; j<16; j++)
      print2stdout("   ");
    print2stdout("    ");

    for(j=0; j<16 && i+j<Len; j++)
      if(Buff[i+j]>=32)
        print2stdout("%c", (Buff[i+j]>=32)?Buff[i+j]:'#');

    print2stdout("\n");
  }
}

int ExecCommand(const char *FileName, char *Args[], const bool Wait, const char *UsrName, const char *GrpName)
{
  char * const  env[] = { strdup("HOME=/"), strdup("PATH=/bin:/usr/bin:/sbin:/usr/sbin:/opt/bosch/base/bin"), NULL };
  int           retval=-1;
  int           pid=fork();

  if(!pid)
  {
    struct passwd  *pw=NULL;
    struct group   *gr=NULL;

    if(GrpName && !(gr=getgrnam(GrpName)))
    {
      print2stdout("ExecCommand: Unknown group %s\n", GrpName);
      return -1;
    }
    if(gr && gr->gr_gid!=getgid())
    {
      if(setgid(gr->gr_gid))
      {
        print2stdout("ExecCommand: failed to set group %s:%i\n", GrpName, gr->gr_gid);
        return -1;
      }
    }

    if(UsrName && !(pw=getpwnam(UsrName)))
    {
      print2stdout("ExecCommand: Unknown user %s\n", UsrName);
      return -1;
    }
    if(pw)
    {
      if(!gr && pw->pw_gid!=getgid())
      {
        if(setgid(pw->pw_gid))
        {
          print2stdout("ExecCommand: failed to set group %i\n", pw->pw_gid);
          return -1;
        }
      }
      if(pw->pw_uid!=getuid())
      {
        if(setuid(pw->pw_uid))
        {
          print2stdout("ExecCommand: failed to set user %s:%i\n", UsrName, pw->pw_uid);
          return -1;
        }
      }
    }

    (void)execvpe(FileName, Args, env);
    // execvpe() will not return if it succeeds. If this code is reached, it failed.
    print2stdout("ExecCommand: failed to execute %s\n", FileName);
    // This is still a new (but now useless) process, created by fork(), so we will terminate it.
    _Exit(127);
  }
  else if(pid==-1)
    print2stdout("ExecCommand: Failed to execute %s\n", FileName);
  else
  {
    if(!Wait)
      retval=0;
    else
    {
      int status;
      while(waitpid(pid, &status, 0)==-1)
      {
        if(errno!=EINTR)
        {
          print2stdout("ExecCommand: waitpid failed\n");
          break;
        }
      }

      if(WIFEXITED(status))
        retval=WEXITSTATUS(status);
    }
  }

  for(int i=0; env[i]; i++)
    free(env[i]);

  return retval;
}

int ExecCommandAsUser(const char *CmdLine, const bool Wait)
{
   int         i, argc;
   char        **argv=NULL, *usr=strdup(CmdLine), *grp=NULL, *cmd=NULL;
   wordexp_t   exp;
   int         retval=-1;

   if(!(cmd=strchr(usr, ':')))
      goto EXIT2;
   *cmd++=0;

   if((grp=strchr(usr, ',')))
      *grp++=0;

   if(wordexp(cmd, &exp, WRDE_NOCMD))
      goto EXIT2;

   argc=static_cast<int> (exp.we_wordc);

   if(!(argv=(char **)malloc(sizeof(char *)*(argc+1))))
      goto EXIT;

   for(i=0; i<argc; i++)
      if(!(argv[i]=strdup(exp.we_wordv[i])))
         goto EXIT;
   argv[argc]=NULL;

#if 0
print2stdout("Args (%i):\n", argc);
for(i=0; i<=argc; i++)
 print2stdout("%s\n", argv[i]?argv[i]:"NULL");
print2stdout("User: %s\n", usr?usr:"NULL");
print2stdout("Group: %s\n", grp?grp:"NULL");
#endif
   retval=ExecCommand(argv[0], argv, Wait, usr, grp);

EXIT:
   wordfree(&exp);
EXIT2:
   if(argv)
   {
      for(i=0; i<argc; i++)
         if(argv[i])
            free(argv[i]);

      free(argv);
   }

   if(usr)
      free(usr);

   return retval;
}
#if 0 // Old implementation without wordexp
{
   int         retval;
   char        *usr, *cmd, *tmp;
   const char  *quot1, *quot2;

   if(!(quot1=strchr(CmdLine, '\"')))
      usr=strdup(CmdLine);
   else
   {
      if(!(quot2=strrchr(CmdLine, '\"')) || quot1==quot2)
      {
         print2stdout("ExecCommandAsUser: missing \"\n");
         return false;
      }
      usr=strdup(quot1);
      *strrchr(usr, '\"')=0;
   }

   cmd=strchr(usr, ':');

   if(!cmd)
   {
      print2stdout("ExecCommandAsUser: \':\' not found\n");
      free(usr);
      return false;
   }
   *cmd++=0;

   char  *grp=strchr(usr, ',');
   if(grp)
      *grp++=0;

   char **args=NULL;
   int  cnt=0, n, l=strlen(cmd), i;

   for(n=0, i=0; i<l; i++)
      if(cmd[i]==' ')
      {
         if(i>n+1)
         {
            args=(char **)realloc(args, (cnt+1)*sizeof(char *));
            args[cnt]=(char *)malloc(i-n+1);
            memcpy(args[cnt], cmd+n, i-n);
            args[cnt][i-n]=0;
            cnt++;
         }
         n=i+1;
      }

   if(i>n+1)
   {
      args=(char **)realloc(args, (cnt+1)*sizeof(char *));
      args[cnt]=(char *)malloc(i-n+1);
      memcpy(args[cnt], cmd+n, i-n);
      args[cnt][i-n]=0;
      cnt++;
   }

   args=(char **)realloc(args, (cnt+1)*sizeof(char *));
   args[cnt]=NULL;

#ifdef DEBUG_EXEC
print2stdout("Args (%i):\n", cnt);
for(i=0; i<=cnt; i++)
 print2stdout("%s\n", args[i]?args[i]:"NULL");
print2stdout("User: %s\n", usr?usr:"NULL");
print2stdout("Group: %s\n", grp?grp:"NULL");
#endif //DEBUG_EXEC

   retval=ExecCommand(args[0], args, Wait, usr, grp);

   free(usr);
   return retval;
}
#endif



void helpOptionArgument()
{
    print2stdout("Usage: swupdatecontrol_out.out [OPTIONS] \n");
    print2stdout("DESCRIPTION: \n");
    print2stdout("swupdatecontrol_out.out command line tool is used to detect the force.dnl file and update the DL Magics \n");
    print2stdout("In addition, tool can be used for switching the boot chain and resetting the SCC and APP processors \n");
    print2stdout("OPTIONS: \n");
    print2stdout("\t -d/--dump : Dump the Boot Chain Configuration to stdout \n");
    print2stdout("\t -m/--mode MODE: Set the Download System mode. MODE: \"normal\" or \"recovery\" \n");
    print2stdout("\t -r/--reset RESET_TYPE: Reset processor. RESET_TYPE: \"system\" or \"scc\" or \"app\" \n");
    print2stdout("\t -s/--shutdown SHUTDOWN_TYPE: Shutdown the system. SHUTDOWN_TYPE: \"normal\" or \"ack\" \n");
    print2stdout("\t -b/--bootchain BOOT_CHAIN: Set the active bootchain. BOOT_CHAIN: \"1\" or \"2\" \n");
    print2stdout("\t -l/--backlight STATE : Turn back light on or of. STATE \"on\" or \"off\" \n");
    print2stdout("\t -a/--acknowledgewakeup : Acknowledge all wakeup resaons \n");
    print2stdout("\t -i/--indicatestartup : Indicate iMX startup finished to V850 \n");
    print2stdout("\t    --getv850mode : Print the v850's current mode which is bootloader or application \n");
    print2stdout("\t    --setv850mode MODE: Set the v850's mode. MODE \"bl\" or \"app\" \n");
//    print2stdout("\t -v/--v850version (MODE): Display version information for bootloader and applications. Optional MODE \"bl\" or \"app\" \n"); // Temporarily changed to required due to problems with get_opt and optional arguments
    print2stdout("\t -v/--v850version MODE: Display version information for bootloader and applications. MODE \"bl\" or \"app\" or \"all\"\n");
    print2stdout("\t    --v850checkupd: Check if the v850 needs an update \n");
    print2stdout("\t -f/--fcid : print the fcid of the device [set new fcid if provided]\n");
    print2stdout("\t    --setfcid \"fcid\": set the new fcid, provide in form of hex [e.g 0x0012] \n");
    print2stdout("\t -p/--pjid : print the pjid of the device [set new pjid if provided] \n");
    print2stdout("\t   /--getresetcounter : get the current value of the cyclic reset counter in PRAM\n");
    print2stdout("\t   /--setresetcounter VALUE : set the cyclic reset counter in PRAM to the given value \n");
    print2stdout("\t   /--getforcednlmagic : get the current value of the force dnl magic in PRAM\n");
    print2stdout("\t   /--setforcednlmagic VALUE : set the force dnl magic in PRAM to the given value \n");
    print2stdout("\t -c/--checkrstcntr RESETCNTR_VALUE : Check the reset counter value and set recovery magic \n");
    print2stdout("\t -z/--resetcntr RESETCNTR_VALUE: Set the reset counter to given value \n");
    print2stdout("\t   /--normagic MAGIC --writenorvalue VALUE: write VALUE into MAGIC in NOR \n");
    print2stdout("\t   /--kdskey KEY --kdsoffset OFFSET --readkdslen VALUE: read LEN bytes from KDS at OFFSET \n");
    print2stdout("\t   /--kdskey KEY --kdsoffset OFFSET --writekdsvalue VALUE: \n");
    print2stdout("\t              write VALUE into KDS at OFFSET where VALUE is a byte seqeunce like \"128/0xff/0777/-1\" \n");
    print2stdout("\t   /--incport PORT --inchost HOST --incsendvalue VALUE: \n");
    print2stdout("\t              send VALUE to inc port PORT of HOST (defaults to scc) where VALUE is a byte seqeunce like \"128/0xff/0777/-1\" \n");
    print2stdout("\t   /--incport PORT --inchost HOST --incsendrecvvalue VALUE: \n");
    print2stdout("\t              send VALUE to inc port PORT of HOST (defaults to scc) and receive a response \n");
    print2stdout("\t   /--bckg_cmd_as_user USER[,GROUP]:COMMAND : run COMMAND in background as user USER and GROUP. Same restrictions as in exec_cmd apply \n");
    print2stdout("\t   /--exec_cmd_as_user USER[,GROUP]:COMMAND : execute COMMAND as user USER and GROUP and wait for result. Note that quotes like \"USER[,GROUP]:COMMAND\" are fine but quoted arguments inside are currently not supported! \n");
    print2stdout("\t              --exec_cmd_as_user \"dummyuser,dummygroup:strace ls\" will work while \n");
    print2stdout("\t              --exec_cmd_as_user \"dummyuser,dummygroup:strace \\\"ls -la \\\"\" will NOT work \n");
    print2stdout("\t   /--getsecuremode : check if secure boot mode is enabled \n");
    print2stdout("\t   /--getsrkrevocation : check which SRKs are revoced \n");
    print2stdout("\t   /--perfserver : start as server for performance measurement \n");
    print2stdout("\t   /--perfstart NAME : start performance measurement for channel NAME \n");
    print2stdout("\t   /--perfstop NAME : stop performance measurement for channel NAME \n");
    print2stdout("\t   /--perfstopall : stop performance measurement for all channels \n");
    print2stdout("\t   /--perfdump : dump all collected performance measurements \n");
    print2stdout("\t   /--version : display and return version number (%i.%02i) \n", PROGRAM_VERSION_MAJOR, PROGRAM_VERSION_MINOR);
    print2stdout("\t   /--registrykey KEY --registryvalue VALUE \n");
    print2stdout("\t -e/--noexit : Do not exit swupdatecontrol - A temporary Hack, required to work around a bug in /dev/wup \n");
    print2stdout("\t   /--incport PORT --inchost HOST --cpldversion --incsendvalue VALUE: \n");
    print2stdout("\t   /--emmcinfo : print emmc chip make (samsung/micron) and their version \n");
    print2stdout("\t -h/--help : Display help \n \n");
}

void failUnsupportedOptionArgument()
{
  helpOptionArgument();
  print2stdout("unsupported option argument\n");
  OSAL_EXIT_WITH_CODE(SWUPDCTRL_EXIT_FAILURE);
}

bool setActiveBootChain(string bootSelection, SWUpdateControl &swuc)
{
  bool retValue = false;
  if (BOOT_CHAIN_ONE == bootSelection)
     retValue = swuc.switchApplicationBootChainOne();
  else if(BOOT_CHAIN_TWO == bootSelection)
     retValue = swuc.switchApplicationBootChainTwo();
  else
     failUnsupportedOptionArgument();
  return retValue;
}



int main(int argc, char *argv[])
{
   vInitPlatformEtg();
   ETG_I_REGISTER_CHN(0);
   
  SWUpdateControl     swuc;
  bool                noexit=false;
  int                 exitcode=0;
  OSAL_tIODescriptor  rGpioIODescriptor=OSAL_ERROR;

  if (1 < argc) {
    // evaluate command line options

    int         c, index;
    extern char *optarg;
    extern int  optopt;
    int         normagic=0, norvalue=0;
    bool        normagic_valid=false;
    int         kdskey=-1, kdsoff=-1, kdslen=0;
    uint16_t    incport=-1;
    string      inchost("scc");
    string      registrykey;
    char*       registryvalue;

    static struct option long_options[]=
    {
      {"dump",              no_argument,        0,  'd'},
      {"mode",              required_argument,  0,  'm'},
      {"reset",             required_argument,  0,  'r'},
      {"shutdown",          required_argument,  0,  's'},
      {"bootchain",         required_argument,  0,  'b'},
      {"backlight",         required_argument,  0,  'l'},
      {"acknowledgewakeup", no_argument,        0,  'a'},
      {"indicatestartup",   no_argument,        0,  'i'},
      {"getv850mode",       no_argument,        0,  GETV850MODE},
      {"setv850mode",       required_argument,  0,  SETV850MODE},
//      {"v850version",       optional_argument,  0,  'v'}, // Temporarily changed to required due to problems with get_opt and optional arguments
      {"v850version",       required_argument,  0,  'v'},
      {"v850checkupd",      no_argument,        0,  V850CHECKUPD},
      {"fcid",              no_argument,        0,  'f'},
      {"setfcid",           required_argument,  0,  SETFCID},
      {"pjid",              no_argument,        0,  'p'},
      {"getresetcounter",   no_argument,        0,  GETRSTCNT},
      {"setresetcounter",   required_argument,  0,  SETRSTCNT},
      {"getforcednlmagic",  no_argument,        0,  GETFRCDNL},
      {"setforcednlmagic",  required_argument,  0,  SETFRCDNL},
      {"checkrstcntr",      required_argument,  0,  'c'},
      {"resetcntr",         required_argument,  0,  'z'},
      {"normagic",          required_argument,  0,  NORMAGIC},
      {"writenorvalue",     required_argument,  0,  WRITENORVALUE},
      {"kdskey",            required_argument,  0,  KDSKEY},
      {"kdsoffset",         required_argument,  0,  KDSOFF},
      {"readkdslen",        required_argument,  0,  READKDSLEN},
      {"writekdsvalue",     required_argument,  0,  WRITEKDSVAL},
      {"incport",           required_argument,  0,  INCPORT},
      {"inchost",           required_argument,  0,  INCHOST},
      {"incsendvalue",      required_argument,  0,  INCSENDVAL},
      {"incsendrecvvalue",  required_argument,  0,  INCSENDRECVVAL},
      {"bckg_cmd_as_user",  required_argument,  0,  BCKG_AS_USER},
      {"exec_cmd_as_user",  required_argument,  0,  EXEC_AS_USER},
      {"getsecuremode",     no_argument,        0,  GETSECMODE},
      {"getsrkrevocation",  no_argument,        0,  GETSRKREVO},
      {"perfserver",        no_argument,        0,  PERFSRVR},
      {"perfstart",         required_argument,  0,  PERFSTART},
      {"perfstop",          required_argument,  0,  PERFSTOP},
      {"perfstopall",       no_argument,        0,  PERFSTOPALL},
      {"perfdump",          no_argument,        0,  PERFDUMP},
      {"version",           no_argument,        0,  PROGRAM_VERSION},
	  {"cpldversion",       no_argument,        0,  CPLD_VERSION},
      {"registrykey",       required_argument,  0,  REGISTRY_KEY},
      {"registryvalue",     required_argument,  0,  REGISTRY_VALUE},
      {"noexit",            no_argument,        0,  'e'},   // Temporary Hack, required to work around a bug in /dev/wup
      {"emmcinfo",          no_argument,        0,  GETEMMCINFO},
      {"help",              no_argument,        0,  'h'},
      {0,                   0,                  0,   0 }
    };

//    while((c = getopt_long_only(argc, argv, "dm:s:r:b:aiv::f::p::ehc:z:", long_options, &index)) != -1)   // Temporarily removed due to problems with get_opt and optional arguments
    while((c = getopt_long_only(argc, argv, "dm:s:r:b:l:aiv:fpehc:z:", long_options, &index)) != -1)
    {
      switch(c) {
        case 'd': // dump
          swuc.dumpBootChainConfig();
          break;

        case 'm': // mode
        {
          string mode_arg(optarg);
          if (MODE_RECOVERY == mode_arg) {
            print2stdout("activate recovery mode");
            if(!swuc.activateRecoveryBootChain())
              exitcode=1;
            sync();
          }
          else if (MODE_NORMAL == mode_arg) {
            print2stdout("activate normal mode");
            if(!swuc.activateApplicationBootChain())
              exitcode=1;
            sync();
          }
          else {
            failUnsupportedOptionArgument();
          }
        }
        break;

        case 'r': // reset
        {
          string reset_arg(optarg);
          if (RESET_PROCESSOR_SYSTEM == reset_arg) {
            print2stdout("reset processor system");
            if(!swuc.resetProcessorSystem())
              exitcode=SWUPDCTRL_EXIT_FAILURE;
          }
          else if (RESET_PROCESSOR_SCC == reset_arg)
          {
            print2stdout("reset processor scc");
            if(!swuc.resetProcessorSCC())
              exitcode=SWUPDCTRL_EXIT_FAILURE;
          }
          else if (RESET_PROCESSOR_APP == reset_arg)
          {
            print2stdout("reset processor application");
            if(!swuc.resetProcessorApp())
              exitcode=SWUPDCTRL_EXIT_FAILURE;
          }
          else
          {
            failUnsupportedOptionArgument();
          }
        }
        break;

        case 's': // shutdown
        {
          string shutdown_arg(optarg);
          if (SHUTDOWN_NORMAL == shutdown_arg)
          {
            print2stdout("shutdown system\n");
            if(!swuc.shutdownSystem())
              exitcode=1;
          }
          else if(shutdown_arg==SHUTDOWN_ACKNOWLEDGE)
          {
            print2stdout("acknowledging wakeup reasons\n");
            if(!swuc.acknowledgeWUPReasons())
              exitcode=1;
            else
            {
              print2stdout("shutdown system\n");
              if(!swuc.shutdownSystem())
                exitcode=1;
            }
          }
          else {
            failUnsupportedOptionArgument();
          }
        }
        break;

        case 'b': //boot chain selection
        {
          string bootSelection(optarg);
          if(!setActiveBootChain(bootSelection, swuc))
            exitcode=1;
        }
        break;

        case 'l': // back light on / off
        {
          string state(optarg);
          if(state=="on")
            swuc.switchBacklight(true);
          else if(state=="off")
            swuc.switchBacklight(false);
          else
            failUnsupportedOptionArgument();
        }
        break;

        case 'a': // acknowledge wakeup
          print2stdout("acknowledging wakeup reasons\n");
          if(!swuc.acknowledgeWUPReasons())
            exitcode=1;
          break;

        case 'i': // indicate startup finished
          if(!swuc.indicateStartupFinished())
            exitcode=1;
          break;

        case GETV850MODE:
        {
          SWUpdateControl::V850_MODE  mode=swuc.getV850Mode();
          if(mode==SWUpdateControl::V850_MODE_BOOTLOADER)
            print2stdout("The V850 is in bootloader mode\n");
          else if(mode==SWUpdateControl::V850_MODE_APPLICATION)
            print2stdout("The V850 is in application mode\n");
          else
            print2stdout("The V850 is in an unknown state\n");
          break;
        }

        case SETV850MODE:
        {
          SWUpdateControl::V850_MODE  mode=SWUpdateControl::V850_MODE_UNDEFINED;
          string  shutdown_arg(optarg);

          if(shutdown_arg=="bl")
            mode=SWUpdateControl::V850_MODE_BOOTLOADER;
          else if(shutdown_arg=="app")
            mode=SWUpdateControl::V850_MODE_APPLICATION;
          else
            failUnsupportedOptionArgument();

          if(swuc.setV850Mode(mode)==mode)
            print2stdout("V850's mode changed\n");
          break;
        }

        case 'v':
        {
          if(optarg)
          {
            string which_arg(optarg);
            if(which_arg=="bl")
              swuc.dumpV850Version(SWUpdateControl::V850_MODE_BOOTLOADER);
            else if(which_arg=="app")
              swuc.dumpV850Version(SWUpdateControl::V850_MODE_APPLICATION);
            else if(which_arg=="all")
              swuc.dumpV850Version(SWUpdateControl::V850_MODE_UNDEFINED);
            else
              failUnsupportedOptionArgument();
          }
          else
            swuc.dumpV850Version(SWUpdateControl::V850_MODE_UNDEFINED);
          break;
        }

        case V850CHECKUPD:
        {
          swuc.checkV850Update();
          print2stdout("swuc.checkV850Update() done\n");
          break;
        }

        case 'f':
        {
          unsigned int fcid;

          if(!swuc.GetFcId(&fcid))
            exitcode=1;
          else
            print2stdout("0x%04X\n", fcid);
          break;
        }
        case SETFCID:
        {
           if(optarg) {
              string value(optarg);
              if(!swuc.setFcId(value)) {
                 print2stdout("fcid set failed\n");
                 exitcode=1;              
              } else {
                 print2stdout("fcid set:%s\n", value.c_str());
              }
           }
           break;
        }

        case 'p':
        {
          unsigned int pjid;

          if(!swuc.GetPjId(&pjid))
            exitcode=1;
          else
            print2stdout("0x%04X\n", pjid);
          break;
        }

        case GETRSTCNT:
        {
          int cnt=swuc.getResetCounter();
          if(cnt>=0)
            print2stdout("Cyclic reset counter in PRAM is set to %i\n", cnt);
          else
          {
            print2stdout("Failed to read cyclic reset counter in PRAM!\n");
            exitcode=1;
          }
          break;
        }

        case SETRSTCNT:
        case 'z' :
        {
          string        cnt_arg(optarg);
          unsigned char cnt=(unsigned char)atoi(cnt_arg.c_str());
          if(swuc.setResetCounter(cnt))
            print2stdout("Set cyclic reset counter in PRAM to %i\n", cnt);
          else
          {
            print2stdout("Failed to set cyclic reset counter in PRAM!\n");
            exitcode=1;
          }
          break;
        }

        case GETFRCDNL:
        {
          int cnt=swuc.getForceDnlMagic();
          if(cnt>=0)
            print2stdout("Force dnl magic in PRAM is set to %i\n", cnt);
          else
          {
            print2stdout("Failed to read force dnl magic from PRAM!\n");
            exitcode=1;
          }
          break;
        }

        case SETFRCDNL:
        {
          string        cnt_arg(optarg);
          unsigned char cnt=(unsigned char)atoi(cnt_arg.c_str());
          if(swuc.setForceDnlMagic(cnt))
            print2stdout("Set force dnl magic in PRAM to %i\n", cnt);
          else
          {
            print2stdout("Failed to set force dnl magic in PRAM!\n");
            exitcode=1;
          }
          break;
        }

        case 'c' :
        {
        	string strResetCntrValue(optarg);
        	tU8 u8rstCntrVal =static_cast<tU8> (atoi (strResetCntrValue.c_str()) );
        	tU8 u8ReturnVal = swuc.checkRstCtrAndSetRecvFlag(u8rstCntrVal);
        	OSAL_EXIT_WITH_CODE (u8ReturnVal);
        	break;
        }

        case NORMAGIC:
          normagic=static_cast<int> (strtoul(optarg, NULL, 0) );
          normagic_valid=true;
          break;

        case WRITENORVALUE:
          norvalue=static_cast<int> (strtoul(optarg, NULL, 0) );

          if(normagic_valid && !swuc.WriteNORMagic(normagic, norvalue))
            exitcode=1;
          break;

        case KDSKEY:
          kdskey=static_cast<int> (strtol(optarg, NULL, 0) );
          break;

        case KDSOFF:
          kdsoff=static_cast<int> (strtol(optarg, NULL, 0) );
          break;

        case READKDSLEN:
          kdslen=static_cast<int> (strtol(optarg, NULL, 0) );

          if(kdskey>=0 && kdsoff>=0 && !swuc.ReadKDSValue(kdskey, kdsoff, kdslen))
            exitcode=1;
          break;

        case WRITEKDSVAL:
          if(kdskey>=0 && kdsoff>=0 && !swuc.WriteKDSValue(kdskey, kdsoff, optarg))
            exitcode=1;
          break;

        case INCPORT:
          incport=static_cast<uint16_t> (strtol(optarg, NULL, 0) );
          break;

        case INCHOST:
          inchost=optarg;
          break;

        case INCSENDVAL:
          if(incport>=0 && !swuc.IncSendValue(incport, inchost, optarg))
            exitcode=1;
          break;

        case INCSENDRECVVAL:
          if(incport>=0 && !swuc.IncSendRecvValue(incport, inchost, optarg))
            exitcode=1;
          break;

        case EXEC_AS_USER:
          ExecCommandAsUser(optarg, true);
          break;

        case BCKG_AS_USER:
          ExecCommandAsUser(optarg, false);
          break;

        case GETSECMODE:
        {
          int  secmode=swuc.GetSecureBootModeEnabled();
          if(secmode<0)
          {
            print2stdout("Failed to get the status of secure boot mode\n");
            exitcode=1;
          }
          else if(secmode==0)
            print2stdout("Secure boot mode is disabled\n");
          else
            print2stdout("Secure boot mode is enabled\n");
          break;
        }

        case GETSRKREVO:
        {
          int  srkrevo=swuc.GetSRKRevocation();
          if(srkrevo<0)
          {
            print2stdout("Failed to get the status of SRK revocation\n");
            exitcode=1;
          }
          else
            print2stdout("Revocation bits are set to %i\n", srkrevo);
          break;
        }

        case PERFSRVR:
          swuc.StartPerfServer();
          break;

        case PERFSTART:
        {
          string  message("start ");
          message+=optarg;
          swuc.SendPerfMsg(message.c_str());
          break;
        }

        case PERFSTOP:
        {
          string  message("stop ");
          message+=optarg;
          swuc.SendPerfMsg(message.c_str());
          break;
        }

        case PERFSTOPALL:
          swuc.SendPerfMsg("stop_all");
          break;

        case PERFDUMP:
          swuc.SendPerfMsg("dump");
          break;

        case 'e':   // Temporary Hack, required to work around a bug in /dev/wup
          print2stdout("/dev/gpio will be kept open\n");
          rGpioIODescriptor=OSAL_IOOpen("/dev/gpio", OSAL_EN_READWRITE);
          noexit=true;
          break;

        case PROGRAM_VERSION:
          print2stdout("Version %i.%02i\n", PROGRAM_VERSION_MAJOR, PROGRAM_VERSION_MINOR);
          exitcode=PROGRAM_VERSION_MAJOR*100+PROGRAM_VERSION_MINOR;
          break;

        case CPLD_VERSION:
        	swuc.cpldFlag=1;
          break;

        case REGISTRY_KEY:
        	registrykey = optarg;
          break;
        case REGISTRY_VALUE:
        	registryvalue = optarg;
        	if((!registrykey.empty()) && (NULL !=registryvalue) && (registryvalue[0] != '\0'))
        		swuc.WriteToRegistry(registrykey, registryvalue);
        	else
        		failUnsupportedOptionArgument();
          break;

        case GETEMMCINFO:
           swuc.dumpEmmcInfo();
           break;

        case 'h' :
          helpOptionArgument();
          break;

        case ':':       // -m, -s or -r without operand
          print2stdout("Option -%c requires an operand\n", optopt);
          OSAL_EXIT_WITH_CODE(SWUPDCTRL_EXIT_FAILURE);
          break;

        case '?':
          print2stdout("Unrecognized option: -%c\n", optopt);
          OSAL_EXIT_WITH_CODE(SWUPDCTRL_EXIT_FAILURE);
          break;

        default:
          print2stdout("getopt_long_only() returned the unsupported code %i\n", c);
          OSAL_EXIT_WITH_CODE(SWUPDCTRL_EXIT_FAILURE);
          break;
      }
    }
  }
  else {
    // no command line options supplied, continue with default behaviour

    // Stautz: wait until dl_RemoteExecOperations::initRessources () has initialized currentStdOutBuffer
    // Otherwise we get a reset in OSAL_pvMemorySet (currentStdOutBuffer, 0, sizeof (currentStdOutBuffer));
    // Ticket 14944
    // TODO: Check whether this is still required.


    timespec sleepTime1sec;
    sleepTime1sec.tv_sec = 1;
    sleepTime1sec.tv_nsec = 0;

    nanosleep(&sleepTime1sec, 0);

    // USB stick detection is done by udev. In case a USB stick with the file force.dnl is detected,
    // this application is started.
    swuc.dumpBootChainConfig();
    print2stdout("SWUpdateControlMain: ------------------------------------\n");
    swuc.activateRecoveryBootChain();
    print2stdout("SWUpdateControlMain: ------------------------------------\n");
    swuc.dumpBootChainConfig();

    // sync file system
    sync();
    print2stdout("SWUpdateControlMain: synced file system\n");

    // wait 5 secs before shutdown
/*    print2stdout("SWUpdateControlMain: wait 5 secs before triggering system shutdown\n");

    timespec sleepTime5sec;
    sleepTime5sec.tv_sec = 5;
    sleepTime5sec.tv_nsec = 0;

    nanosleep(&sleepTime5sec, 0);*/

//    swuc.shutdownSystem();
    if(!swuc.resetProcessorSystem())
      exitcode=1;
  }

  if(noexit)
    sleep(100000);

  if(rGpioIODescriptor!=OSAL_ERROR)
    OSAL_s32IOClose(rGpioIODescriptor);

  ETG_I_UNREGISTER_CHN ();
  OSAL_EXIT_WITH_CODE(exitcode);
}
