/*******************************************************************************
*
* FILE:          SWUpdateControl.cpp
*
* SW-COMPONENT:  SW Update Core Control
*
* PROJECT:
*
* DESCRIPTION:   Component is not intended for production usage. It is a workaround for triggering a system reset to recovery download mode in case a download medium is detected.
*                The implementation is based on the Gen3 GM MY16 FC_Download component.
*
* AUTHOR:
*
* COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/





#include <errno.h>
#include <string>
#include <cstdio>
#include <map>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>
#include <inc/inc.h>
#include "inc/inc_ports.h"
#include "inc/dgram_service.h"
#include <sys/time.h>
#include <pthread.h>
#include <fstream>
#include <sstream>

#include "include/SWUpdateControl.h"
#include "include/SWUpdateControlMain.h"

#include "base/imp/BootChainMgr.h"
#include "platform/imp/WUP_Access.hpp"
#include "base/imp/RegistryAccess.h"


#include "util/swu_util.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_execCommand.h"
#include "util/swu_kds.h"
#include "util/swu_bootChainAccess.hpp"

#define DEVICE_NAME "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS"

#include "base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_I_TTFIS_CMD_PREFIX "SWUCTRL_"
  #define ETG_I_TRACE_CHANNEL     TR_TTFIS_SWUPDATE_CTRL
  #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_TEST
  #include "trcGenProj/Header/SWUpdateControl.cpp.trc.h"
#endif


namespace ai_sw_update {
namespace common {



static const char  *PRAM_RESET_COUNTER="/dev/pram/reset_counter";
static const char  *PRAM_FORCEDNL_MAGIC="/dev/pram/dnl_magic";



static int StringToByte(unsigned char **Buff, const char *Str)
{
  int           bufflen=0;
  unsigned char c;
  unsigned char *buff=*Buff;
  char          *endval=NULL;

  for(const char *nextval=Str; nextval && *nextval; nextval=(*endval)?endval+1:NULL)
  {
    endval=NULL;
    c=(unsigned char)strtoul(nextval, &endval, 0);
    if(endval==nextval || (*endval && *endval!='/'))
    {
      print2stdout("\nSWUpdateControl::WriteKDSValue(): Syntax error in value\n");
      free(buff);
      return 0;
    }

    buff=(unsigned char *)realloc(buff, bufflen+1);
    buff[bufflen++]=c;
  }

  if(!bufflen)
  {
      free(buff);
      return 0;
  }

  *Buff=buff;
  return bufflen;
}



static double GetTimeInMS(void)
{
  struct timeval tp;

  gettimeofday(&tp, NULL);
  return (double)tp.tv_sec+(double)tp.tv_usec/1000000.0;
}



class KdsDriver {
  public:
    bool read(tsKDSEntry &kdsEntry);
    bool write(tsKDSEntry &kdsEntry);

  private:
    bool open();
    void close();

    OSAL_tIODescriptor _IOKdsHandle;
};



bool KdsDriver::read(tsKDSEntry &kdsEntry)
{
  bool res=true;

  if(!open())
  {
    return false;
  }

  tU16 entry=kdsEntry.u16Entry;
  tS32 osalRet=OSAL_s32IORead(_IOKdsHandle, (tPS8)&kdsEntry, sizeof(tsKDSEntry));
  kdsEntry.u16Entry=entry;
  if(osalRet==OSAL_ERROR)
  {
    print2stdout("KdsDriver::read(key=0x%04x, len=%u flags=0x%08x) failed\n", kdsEntry.u16Entry, kdsEntry.u16EntryLength, kdsEntry.u16EntryFlags);
    res=false;
  }
  close();

  return res;
}

bool KdsDriver::write(tsKDSEntry &kdsEntry)
{
  if(!open())
  {
    return false;
  }

  tS32 osalEnable=(tS32)TRUE;
  tS32 osalRes=OSAL_s32IOControl(_IOKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE, osalEnable);
  if(osalRes!=OSAL_OK)
  {
    print2stdout("KdsDriver::write: Failed to enable write mode for KDS! osalRes=0x%x\n", osalRes);
    close();
    return false;
  }

  osalRes=OSAL_s32IOControl(_IOKdsHandle, OSAL_C_S32_IOCTRL_KDS_INVALIDATE_ENTRY, kdsEntry.u16Entry);
  if(osalRes!=OSAL_OK)
  {
    print2stdout("KdsDriver::write: Failed to invalidate entry! osalRes=0x%x\n", osalRes);
    close();
    return false;
  }

#if 0
  print2stdout("KdsDriver::write: Writing...\n");
  hexdump(kdsEntry.au8EntryData, kdsEntry.u16EntryLength);
  print2stdout("...to KDS\n");
#endif

  tU16 entry=kdsEntry.u16Entry;
  osalRes=OSAL_s32IOWrite(_IOKdsHandle, (tPS8)&kdsEntry, sizeof(tsKDSEntry));
  kdsEntry.u16Entry=entry;
  if(osalRes==OSAL_ERROR)
  {
    print2stdout("KdsDriver::write(key=0x%04x, len=%u flags=0x%08x) failed! osalRes=0x%x\n", kdsEntry.u16Entry, kdsEntry.u16EntryLength, kdsEntry.u16EntryFlags, osalRes);
    close();
    return false;
  }
  usleep(100000);

  osalRes=OSAL_s32IOControl(_IOKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_BACK, 0);
  if(osalRes!=OSAL_OK)
  {
    print2stdout("KdsDriver::write: Failed to write entry back! osalRes=0x%x\n", osalRes);
    close();
    return false;
  }
  usleep(100000);

  osalEnable=(tS32)FALSE;
  osalRes=OSAL_s32IOControl(_IOKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE, osalEnable);
  if(osalRes!=OSAL_OK)
  {
    print2stdout("KdsDriver::write: Failed to disable write mode for KDS! osalRes=0x%x\n", osalRes);
    close();
    return false;
  }

  close();
  sync();
  return true;
}

bool KdsDriver::open()
{
  _IOKdsHandle=OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE);
  if(OSAL_ERROR==_IOKdsHandle)
  {
    print2stdout(" KdsDriver::open(): -> KDS open operation failed, OSAL_u32ErrorCode=0x%08X\n", OSAL_u32ErrorCode());
    return false;
  }

  return true;
}

void KdsDriver::close()
{
  if(_IOKdsHandle!=OSAL_ERROR)
  {
    OSAL_s32IOClose(_IOKdsHandle);
    _IOKdsHandle=OSAL_ERROR;
  }
}



class KdsTable
{
  public:
    KdsTable(tU16 key, tU16 len);
    const tsKDSEntry  *getKDSEntry(void) const;
    bool read();
    bool write();
    tU32 getLen();
    bool getBytes(tU8 *data, tU16 offset, tU16 len);
    bool getU8(tU8 *data, tU16 offset);
    bool getU16(tU16 *data, tU16 offset);
    bool setBytes(tU8 *data, tU16 offset, tU16 len);

  private:
    tsKDSEntry _kdsEntry;
};

KdsTable::KdsTable(tU16 key, tU16 len)
{
  _kdsEntry.u16Entry=key;
  _kdsEntry.u16EntryLength=len;
  _kdsEntry.u16EntryFlags=M_KDS_ENTRY_FLAG_NONE;
//  ETG_TRACE_USR4(("KdsTable CTOR:key=0x%04x, len=%u", key, len));
}

const tsKDSEntry *KdsTable::getKDSEntry(void) const
{
  return &_kdsEntry;
}

bool KdsTable::read()
{
  bool res=true;

  KdsDriver drv;
  if(!drv.read(_kdsEntry))
  {
    memset(_kdsEntry.au8EntryData, 0, _kdsEntry.u16EntryLength);
    res=false;
  }
  return res;
}

bool KdsTable::write()
{
  KdsDriver drv;
  return drv.write(_kdsEntry);
}

tU32 KdsTable::getLen()
{
  return _kdsEntry.u16EntryLength;
}

bool KdsTable::getBytes(tU8 *data, tU16 offset, tU16 len)
{
  if(getLen()>=len+offset)
  {
    memcpy(data, _kdsEntry.au8EntryData+offset, len);
    return true;
  }
  return false;
}

bool KdsTable::getU8(tU8 *data, tU16 offset)
{
   if(getLen() >= static_cast<tU32> (offset+1)) {
      *data=_kdsEntry.au8EntryData[offset];
      return true;
   }
   return false;
}

bool KdsTable::getU16(tU16 *data, tU16 offset)
{
   if(getLen() >= static_cast<tU32> (offset+2)) {
      *data=(unsigned short)(((unsigned int)_kdsEntry.au8EntryData[offset]<<8)|(unsigned int)_kdsEntry.au8EntryData[offset+1]);
      return true;
   }
   return false;
}

bool KdsTable::setBytes(tU8 *data, tU16 offset, tU16 len)
{
   if(getLen() < len+offset) {
      print2stdout("getLen:%u dataLength:%u\n",getLen(), len+offset); 
      return false;
   }

  memcpy(_kdsEntry.au8EntryData+offset, data, len);

#if 0
  print2stdout("KdsTable::setBytes: Writing...\n");
  hexdump(kdsEntry.au8EntryData, kdsEntry.u16EntryLength);
  print2stdout("...to KDS\n");
#endif
  return true;
}



class INCConnection
{
public:
  int       sockfd;
  sk_dgram  *dgram;

  INCConnection();
  ~INCConnection();

  bool  open(const uint16_t Port, const std::string Host);
  bool  open(const uint16_t Port);
  void  close();

  bool  send(const void *Msg, const int MsgLen);
  int   recv(void *Msg, const int MsgLen);
  bool  recv(void *Msg, const int MsgLen, const int RecvLen);
private:
  INCConnection ( const INCConnection & );  //Coverity fix for 48207
  INCConnection& operator = ( const INCConnection & ); //Coverity fix for 47746
};

INCConnection::INCConnection() : sockfd(-1), dgram(NULL)
{
}

INCConnection::~INCConnection()
{
  close();
}

bool INCConnection::open(const uint16_t Port, const std::string Host)
{
  sockfd=socket(AF_BOSCH_INC_AUTOSAR, SOCK_STREAM, 0);
  if(sockfd<0)
  {
    print2stdout("INCConnection::open(): Failed to open socket, errno=%i\n", errno);
    return false;
  }

  // Init dgram
  dgram=dgram_init(sockfd, DGRAM_MAX, NULL);
  if(!dgram)
  {
    print2stdout("INCConnection::open(): dgram_init() failed, errno=%i\n", errno);
    close();
    return false;
  }

  // get local adress
  // local host address, iMX
  struct hostent  *hostentLocal=gethostbyname(std::string(Host+"-local").c_str());
  if(!hostentLocal)
  {
    print2stdout("INCConnection::open(): ERROR no such local host\n");
    close();
    return false;
  }
  struct sockaddr_in  sockaddrLocal;
  sockaddrLocal.sin_family=AF_INET;
  memcpy(&sockaddrLocal.sin_addr.s_addr, hostentLocal->h_addr, hostentLocal->h_length);
  sockaddrLocal.sin_port=htons(Port);
  bzero(sockaddrLocal.sin_zero, sizeof(sockaddrLocal.sin_zero));

  //bind to local host
  if(bind(sockfd, (struct sockaddr *)&sockaddrLocal, sizeof(sockaddrLocal))<0)
  {
    print2stdout("INCConnection::open(): ERROR on binding, errno=%i\n", errno);
    close();
    return false;
   }

  // get remote addres
  struct hostent  *hostentRemote=gethostbyname(Host.c_str());
  if(!hostentRemote)
  {
    print2stdout("INCConnection::open(): gethostbyname failed\n");
    close();
    return false;
  }

  struct sockaddr_in  sockaddrRemote;
  sockaddrRemote.sin_family=AF_INET;
  memcpy(&sockaddrRemote.sin_addr.s_addr, hostentRemote->h_addr, hostentRemote->h_length );
  sockaddrRemote.sin_port=htons(Port);
  bzero(sockaddrRemote.sin_zero, sizeof(sockaddrRemote.sin_zero));

  //Connect to remote host
  if(connect(sockfd, (struct sockaddr *)&sockaddrRemote, sizeof(sockaddrRemote))<0)
  {
    print2stdout("INCConnection::open(): ERROR connecting, errno=%i\n", errno);
    close();
  }

  fcntl(sockfd, F_SETFL, O_NONBLOCK);

  return true;
}



bool INCConnection::open(const uint16_t Port)
{
  return open(Port, "scc");
}



void INCConnection::close()
{
  if(sockfd!=-1)
  {
    ::close(sockfd);
    sockfd=-1;
  }
  if(dgram)
  {
    dgram_exit(dgram);
    dgram=NULL;
  }
}



bool INCConnection::send(const void *Msg, const int MsgLen)
{
  int retval;

  if(sockfd<0 || !dgram)
  {
    return false;
  }

  retval=static_cast<int> (dgram_send(dgram, (void *)Msg, MsgLen) );
  if(retval<0)
  {
    print2stdout("INCConnection::send(): dgram_send() failed with code %i, errno=%i\n", retval, errno);
    return false;
  }
  return true;
}



int INCConnection::recv(void *Msg, const int MsgLen)
{
  int retval;

  for(int i=9; i>=0; i--)
  {
    retval=static_cast<int> (dgram_recv(dgram, Msg, MsgLen) );

    if(retval>=0)
      break;
    else
    {
      if(errno==EAGAIN || errno==EWOULDBLOCK)
        usleep(50000);
      else
        break;
    }
  }

  if(retval<0)
  {
    print2stdout("INCConnection::recv(): dgram_recv() failed with code %i, errno=%i\n", retval, errno);
  }

  return retval;
}



bool INCConnection::recv(void *Msg, const int MsgLen, const int RecvLen)
{
  int retval=recv(Msg, MsgLen);

  if(retval!=RecvLen)
  {
    print2stdout("INCConnection::recv(): Expected %i bytes but received %i\n", RecvLen, retval);
  }

  return retval==RecvLen;
}



SWUpdateControl::SWUpdateControl():
  AckThreadRunning(false),
  EventNameValid(false),
  EventName(0),
  cpldFlag(0)
{
}


SWUpdateControl::~SWUpdateControl()
{
}


bool SWUpdateControl::activateRecoveryBootChain()
{
  print2stdout("SWUpdateControl::activateRecoveryBootChain()\n");
  print2errmem("SWUpdateControl activates recovery mode\n");
#ifndef VARIANT_S_FTR_ENABLE_G4G
  return ai_sw_update::common::BootChainMgr::activateRecoveryMode();
#else
  return swu::BootChainAccess<swu::rcarBootChain>::activateRecoveryMode(); 
#endif
}


bool SWUpdateControl::activateApplicationBootChain()
{
  print2stdout("SWUpdateControl::activateApplicationBootChain()\n");
  print2errmem("SWUpdateControl activates application mode\n");
#ifndef VARIANT_S_FTR_ENABLE_G4G
  return ai_sw_update::common::BootChainMgr::activateApplicationMode();
#else
  return swu::BootChainAccess<swu::rcarBootChain>::activateApplicationMode(); 
#endif
}

bool SWUpdateControl::switchApplicationBootChainOne()
{
  print2stdout("SWUpdateControl::switchApplicationBootChainOne()\n");
  print2errmem("SWUpdateControl switches to bootchain #1\n");
#ifndef VARIANT_S_FTR_ENABLE_G4G
  return ai_sw_update::common::BootChainMgr::switchBootChainOne();
#else
  return swu::BootChainAccess<swu::rcarBootChain>::switchBootChainOne(); 
#endif
}

bool SWUpdateControl::switchApplicationBootChainTwo()
{
  print2stdout("SWUpdateControl::switchApplicationBootChainTwo()\n");
  print2errmem("SWUpdateControl switches to bootchain #2\n");
#ifndef VARIANT_S_FTR_ENABLE_G4G
  return ai_sw_update::common::BootChainMgr::switchBootChainTwo();
#else
  return swu::BootChainAccess<swu::rcarBootChain>::switchBootChainTwo(); 
#endif
}

bool SWUpdateControl::WriteNORMagic(size_t Addr, unsigned int Value)
{
  print2stdout("SWUpdateControl::WriteNORMagic: Addr %X, Value %X\n", Addr, Value);

  switch(Addr)
  {
  case DL_MAGIC_SECTOR_VALID_FLAG:
  case DL_MAGIC_SECTOR_COUNTER_FLAG:
  case DL_UPDATE_STATUS_FLAG:
  case DL_ETHERNET_FLAG:
  case DL_ROOTFS_FLAG:

  case DL_MAGIC_SECTOR_VALID_FLAG_2:
  case DL_MAGIC_SECTOR_COUNTER_FLAG_2:
  case DL_UPDATE_STATUS_FLAG_2:
  case DL_ETHERNET_FLAG_2:
  case DL_ROOTFS_FLAG_2:
    ai_sw_update::common::bWriteMagicToRawFlash(static_cast<unsigned int> (Addr), Value) ;
    print2stdout("%X written to %X\n", Value, Addr);
    return true;
  }

  print2stdout("Invalid NOR address %X\n", Addr);
  return false;
}

void SWUpdateControl::dumpBootChainConfig()
{
  print2stdout("SWUpdateControl::dumpBootChainConfig()\n");

  tU32 u32Magic[10]={0};


  print2stdout("Sector 1 (0x160000)\n");

  ai_sw_update::common::bReadFlagFromRawFlash(DL_MAGIC_SECTOR_VALID_FLAG, &u32Magic[0]);
  print2stdout("read DL_MAGIC_SECTOR_VALID_FLAG (0x%x) = (0x%x)\n", DL_MAGIC_SECTOR_VALID_FLAG, u32Magic[0]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_MAGIC_SECTOR_COUNTER_FLAG, &u32Magic[1]);
  print2stdout("read DL_MAGIC_SECTOR_COUNTER_FLAG (0x%x) = (0x%x)\n", DL_MAGIC_SECTOR_COUNTER_FLAG, u32Magic[1]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_UPDATE_STATUS_FLAG, &u32Magic[2]);
  print2stdout("read DL_UPDATE_STATUS_FLAG (0x%x) = (0x%x)\n", DL_UPDATE_STATUS_FLAG, u32Magic[2]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_ETHERNET_FLAG, &u32Magic[3]);
  print2stdout("read DL_ETHERNET_FLAG (0x%x) = (0x%x)\n", DL_ETHERNET_FLAG, u32Magic[3]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_ROOTFS_FLAG, &u32Magic[4]);
  print2stdout("read DL_ROOTFS_FLAG (0x%x) = (0x%x)\n", DL_ROOTFS_FLAG, u32Magic[4]);

	//-----------------------------------------------------------------------------------------

  print2stdout("Sector 2 (0x180000)\n");

  ai_sw_update::common::bReadFlagFromRawFlash(DL_MAGIC_SECTOR_VALID_FLAG_2, &u32Magic[5]);
  print2stdout("read DL_MAGIC_SECTOR_VALID_FLAG_2 (0x%x) = (0x%x)\n", DL_MAGIC_SECTOR_VALID_FLAG_2, u32Magic[5]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_MAGIC_SECTOR_COUNTER_FLAG_2, &u32Magic[6]);
  print2stdout("read DL_MAGIC_SECTOR_COUNTER_FLAG_2 (0x%x) = (0x%x)\n", DL_MAGIC_SECTOR_COUNTER_FLAG_2, u32Magic[6]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_UPDATE_STATUS_FLAG_2, &u32Magic[7]);
  print2stdout("read DL_UPDATE_STATUS_FLAG_2 (0x%x) = (0x%x)\n", DL_UPDATE_STATUS_FLAG_2, u32Magic[7]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_ETHERNET_FLAG_2, &u32Magic[8]);
  print2stdout("read DL_ETHERNET_FLAG_2 (0x%x) = (0x%x)\n", DL_ETHERNET_FLAG_2, u32Magic[8]);

  ai_sw_update::common::bReadFlagFromRawFlash(DL_ROOTFS_FLAG_2, &u32Magic[9]);
  print2stdout("read DL_ROOTFS_FLAG_2 (0x%x) = (0x%x)\n", DL_ROOTFS_FLAG_2, u32Magic[9]);

  print2stdout("=============================\n");
  if(u32Magic[0]==0xDEADBEEF)
  {
    print2stdout("Sector 1 is active\n");
    if(u32Magic[2]==0x2342ABCD)
      print2stdout("Recovery Mode is active\n");
    else
      print2stdout("Full Operation Mode is active\n");
    if(u32Magic[4]==0x5AA501CB)
      print2stdout("Bootchain 2 is active\n");
    else
      print2stdout("Bootchain 1 is active\n");
  }
  else if(u32Magic[5]==0xDEADBEEF)
  {
    print2stdout("Sector 2 is active\n");
    if(u32Magic[7]==0x2342ABCD)
      print2stdout("Recovery Mode is active\n");
    else
      print2stdout("Full Operation Mode is active\n");
    if(u32Magic[9]==0x5AA501CB)
      print2stdout("Bootchain 2 is active\n");
    else
      print2stdout("Bootchain 1 is active\n");
  }
  else
  {
    print2stdout("Both sectors are inactive!\n");
    print2stdout("Recovery Mode is active\n");
    print2stdout("Bootchain 1 is active\n");
  }
}


const bool SWUpdateControl::resetProcessorSystem()
{
  WUP_Access wup;

  sync();
  if(setV850Mode(V850_MODE_APPLICATION)!=V850_MODE_APPLICATION)
  {
    print2stdout("Can't switch V850 to application mode\n");
  }

  DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
  {
    .u8Processor = DEV_WUP_C_U8_ENTIRE_SYSTEM,
    .u8ResetMode = DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
    .u16ResetReason = DEV_WUP_C_U16_RESET_REASON_SW_DOWNLOAD,
    .u16ResetDurationMs = 10
  };

  print2errmem("SWUpdateControl calls /dev/wup to reset the system\n");
  return wup.ioctl(OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR,reinterpret_cast<intptr_t>(&rResetProcessorInfo) );
}



const bool SWUpdateControl::resetProcessorApp()
{
  WUP_Access wup;

  sync();
  DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
  {
    .u8Processor = DEV_WUP_C_U8_APPLICATION_PROCESSOR,
    .u8ResetMode = DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
    .u16ResetReason = DEV_WUP_C_U16_RESET_REASON_SW_DOWNLOAD,
    .u16ResetDurationMs = 10
  };

  print2errmem("SWUpdateControl calls /dev/wup to reset the iMX only\n");
  return wup.ioctl(OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR,reinterpret_cast<intptr_t>(&rResetProcessorInfo) );
}


const bool SWUpdateControl::resetProcessorSCC()
{
  WUP_Access wup;

  sync();
  if(setV850Mode(V850_MODE_APPLICATION)!=V850_MODE_APPLICATION)
  {
    print2stdout("Can't switch V850 to application mode\n");
  }

  DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
  {
    .u8Processor = DEV_WUP_C_U8_SYSTEM_COMMUNICATION_CONTROLLER,
    .u8ResetMode = DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
    .u16ResetReason = DEV_WUP_C_U16_RESET_REASON_SW_DOWNLOAD,
    .u16ResetDurationMs = 10
  };

  print2errmem("SWUpdateControl calls /dev/wup to reset the V850 only\n");
  return wup.ioctl(OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR,reinterpret_cast<intptr_t>(&rResetProcessorInfo) );
}


bool SWUpdateControl::switchBacklight(bool state)
{
  static const char MSG_ON[][2]=
  {
    { 0x20, 0x01 },
    { 0x30, 0x01 },
    { 0x32, 0x01 },
  };
  static const char MSG_OFF[][2]=
  {
    { 0x20, 0x01 },
    { 0x30, 0x00 },
    { 0x32, 0x00 },
  };
  static const char **MSG=(state)?(const char **)MSG_ON:(const char **)MSG_OFF;
  static const int  MSG_LEN=2;
  INCConnection     connection;
  char              msg[256];
  bool              retval;

  if(!connection.open(50971))
  {
    print2stdout("SWUpdateControl::switchBacklight() connection.open() failed\n");
    return false;
  }

  if(!connection.send(MSG[0], MSG_LEN))
  {
    print2stdout("SWUpdateControl::switchBacklight() connection.send(MSG[0]) failed\n");
    return false;
  }
  // FIXME: Response is thrown away for now. It should be checked!
  retval=connection.recv((void *)msg, sizeof(msg));

  if(!connection.send(MSG[1], MSG_LEN))
  {
    print2stdout("SWUpdateControl::switchBacklight() connection.send(MSG[1]) failed\n");
    return false;
  }
  // FIXME: Response is thrown away for now. It should be checked!
  retval=connection.recv((void *)msg, sizeof(msg));

  if(!connection.send(MSG[2], MSG_LEN))
  {
    print2stdout("SWUpdateControl::switchBacklight() connection.send(MSG[2]) failed\n");
    return false;
  }
  // FIXME: Response is thrown away for now. It should be checked!
  retval=connection.recv((void *)msg, sizeof(msg));

  return true;
}


bool SWUpdateControl::acknowledgeWUPReasons()
{
  pthread_t         ackThread;
  OSAL_tEventHandle hEventHandle;
  bool              retval=true;

  AckThreadRunning=true;
  EventNameValid=false;
  EventName=NULL;
  pthread_create(&ackThread, NULL, acknowledgeWUPReasonsThread, this);

  while(AckThreadRunning && !EventNameValid)
    usleep(50000);
  if(!EventNameValid)
    retval=false;
  else
  {
    if(OSAL_s32EventOpen((char *)EventName, &hEventHandle)!=OSAL_OK)
    {
      print2stdout("SWUpdateControl::acknowledgeWUPReasons: OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT failed\n");
      retval=false;
    }
    else
    {
      sleep(1);
      if(OSAL_s32EventPost(hEventHandle, EVENT_MASK_STOP, OSAL_EN_EVENTMASK_OR)!=OSAL_OK)
      {
        print2stdout("SWUpdateControl::acknowledgeWUPReasons: OSAL_s32EventPost failed\n");
        retval=false;
      }
      OSAL_s32EventClose(hEventHandle);
    }
  }

  while(retval && AckThreadRunning)
    usleep(50000);

  return retval;
}


void *SWUpdateControl::acknowledgeWUPReasonsThread(void *Arg)
{
  static const int    EVENT_MASK_ALL=DEV_WUP_C_U32_EVENT_MASK_ONOFF_EVENT_CHANGED_NOTIFY|DEV_WUP_C_U32_EVENT_MASK_ONOFF_STATE_CHANGED_NOTIFY|EVENT_MASK_STOP;

  SWUpdateControl     *swuc=(SWUpdateControl *)Arg;
  OSAL_tIODescriptor  hDeviceWup=OSAL_IOOpen(OSAL_C_STRING_DEVICE_WUP, OSAL_EN_READWRITE);
  OSAL_tEventHandle   hEventHandle;
  OSAL_tEventMask     rEventMaskResult=0;

  DEV_WUP_trClientRegistration             rClientRegistration;
  DEV_WUP_trOnOffReasonChangedRegistration rOnOffReasonChangedRegistration;

  if(hDeviceWup==OSAL_ERROR)
    print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_IOOpen failed\n");
  else
  {
    if(OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT, (intptr_t)&rClientRegistration)!=OSAL_OK)
      print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT failed\n");
    else
    {
      swuc->EventName=strdup(rClientRegistration.szNotificationEventName);
      swuc->EventNameValid=true;

      if(OSAL_s32EventOpen(rClientRegistration.szNotificationEventName, &hEventHandle)!=OSAL_OK)
        print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT failed\n");
      else
      {
        rOnOffReasonChangedRegistration.u32ClientId       =rClientRegistration.u32ClientId;
        rOnOffReasonChangedRegistration.u8NotificationMode=DEV_WUP_C_U8_NOTIFY_STATES_AND_EVENTS_WITH_PAST_ONES;
        rOnOffReasonChangedRegistration.bIsSystemMaster   =TRUE;

        if(OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_REGISTER_ONOFF_REASON_CHANGED_NOTIFICATION, (intptr_t)&rOnOffReasonChangedRegistration)!=OSAL_OK)
          print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT failed\n");
        else
        {
          while(OSAL_s32EventWait(hEventHandle, EVENT_MASK_ALL, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &rEventMaskResult)==OSAL_OK)
          {
            if(OSAL_s32EventPost(hEventHandle, ~rEventMaskResult, OSAL_EN_EVENTMASK_AND)!=OSAL_OK)
            {
              print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_s32EventPost failed\n");
              break;
            }

            if(rEventMaskResult&EVENT_MASK_STOP)
            {
              print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: Thread stopped\n");
              break;
            }
            if(rEventMaskResult&DEV_WUP_C_U32_EVENT_MASK_ONOFF_EVENT_CHANGED_NOTIFY)
            {
              print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: DEV_WUP_C_U32_EVENT_MASK_ONOFF_EVENT_CHANGED_NOTIFY!\n");

              DEV_WUP_trOnOffEventHistory rOnOffEventHistory;

              rOnOffEventHistory.u32ClientId=rClientRegistration.u32ClientId;
              if(OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_EVENTS, (intptr_t)&rOnOffEventHistory)!=OSAL_OK)
                print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_EVENTS failed\n");
              else
              {
                DEV_WUP_trOnOffEventAcknowledge rOnOffEventAcknowledge;
                tU8                             u8Index;
                tU8                             u8Event;
                tU16                            u16MsgHandle;

                rOnOffEventAcknowledge.u32ClientId = rClientRegistration.u32ClientId;

                for(u8Index=0; u8Index<rOnOffEventHistory.u8NumberOfOnOffEvents; u8Index++)
                {
                  u8Event     =rOnOffEventHistory.arOnOffEvent[u8Index].u8Event;
                  u16MsgHandle=rOnOffEventHistory.arOnOffEvent[u8Index].u16MsgHandle;

                  // Only clients with bIsSystemMaster set to TRUE are allowed to perform the below on/off event acknowledge
                  rOnOffEventAcknowledge.u16OnOffEventMsgHandle = u16MsgHandle;

                  if(OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_EVENT, (intptr_t)&rOnOffEventAcknowledge)==OSAL_ERROR)
                    print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_EVENT failed\n");
                }
              }
            }
            if(rEventMaskResult&DEV_WUP_C_U32_EVENT_MASK_ONOFF_STATE_CHANGED_NOTIFY)
            {
              print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: DEV_WUP_C_U32_EVENT_MASK_ONOFF_STATE_CHANGED_NOTIFY!\n");

              DEV_WUP_trOnOffStates rOnOffStates;

              if(OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_STATES, (intptr_t)&rOnOffStates)!=OSAL_OK)
                print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_STATES failed\n");
              else
              {
                DEV_WUP_trOnOffStateAcknowledge rOnOffStateAcknowledge;

                // Only clients with bIsSystemMaster set to TRUE are allowed to perform the below on/off state acknowledge
                rOnOffStateAcknowledge.u32ClientId = rClientRegistration.u32ClientId;
                rOnOffStateAcknowledge.u16OnOffStateMsgHandle = rOnOffStates.u16MsgHandle;

                if(OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_STATE, (intptr_t)&rOnOffStateAcknowledge)==OSAL_ERROR)
                  print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_STATE failed\n");
              }
            }
            else
            {
              print2stdout("SWUpdateControl::acknowledgeWUPReasonsThread: Unknown Event!\n");
              break;
            }
          }
          OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_UNREGISTER_ONOFF_REASON_CHANGED_NOTIFICATION, (intptr_t)rOnOffReasonChangedRegistration.u32ClientId);
        }
        OSAL_s32EventClose(hEventHandle);
      }
      OSAL_s32IOControl(hDeviceWup, OSAL_C_S32_IOCTRL_WUP_UNREGISTER_CLIENT, (intptr_t)rClientRegistration.u32ClientId);
    }
    OSAL_s32IOClose(hDeviceWup);
  }

  swuc->AckThreadRunning=false;
  return NULL;
}


bool SWUpdateControl::indicateStartupFinished()
{
  bool  retval=true;

  print2stdout("SWUpdateControl::indicateStartupFinished() indicate startup finished to V850 via /dev/wup\n");

  OSAL_tIODescriptor _hDeviceWup=OSAL_IOOpen(OSAL_C_STRING_DEVICE_WUP, OSAL_EN_READWRITE);

  if(_hDeviceWup==OSAL_ERROR)
  {
    print2stdout("SWUpdateControl::indicateStartupFinished() failed to open /dev/wup\n");
    return false;
  }

  if(OSAL_s32IOControl(_hDeviceWup, OSAL_C_S32_IOCTRL_WUP_INDICATE_STARTUP_FINISHED, (intptr_t)0)!=OSAL_OK)
  {
    print2stdout("SWUpdateControl::indicateStartupFinished() failed to indicate startup finished\n");
    retval=false;
  }

  if(OSAL_s32IOClose(_hDeviceWup)!=OSAL_OK)
  {
    print2stdout("SWUpdateControl::indicateStartupFinished() failed to close /dev/wup\n");
  }

  return retval;
}


bool SWUpdateControl::shutdownSystem()
{
  bool  retval=true;

  // Execute system shutdown via /dev/wup
  print2stdout("SWUpdateControl::shutdownSystem() trigger shutdown via /dev/wup\n");

  if(getV850Mode()!=V850_MODE_APPLICATION)
  {
    setV850Mode(V850_MODE_APPLICATION);
    sleep(6);   // @todo: This is a hack to work around a bug in V850 software
  }

  tU32 u32ShutdownMethod = DEV_WUP_SHUTDOWN_NORMAL;
  OSAL_tIODescriptor _hDeviceWup = (OSAL_tIODescriptor)OSAL_ERROR;
  tCString DEV_WUP = "/dev/wup";

	_hDeviceWup = OSAL_IOOpen(DEV_WUP, OSAL_EN_READWRITE);

	if (OSAL_ERROR == _hDeviceWup)
	{
		print2stdout("SWUpdateControl::shutdownSystem() /dev/wup could not be opened\n");
		retval=false;
	}
	else
	{
    print2errmem("SWUpdateControl calls /dev/wup to shut down the device\n");
		if (OSAL_s32IOControl(_hDeviceWup, OSAL_C_S32_IOCTRL_WUP_SHUTDOWN,  (tS32)u32ShutdownMethod ) == OSAL_OK)
		{
			// If SHUTDOWN works, we should never come here
			print2stdout("SWUpdateControl::shutdownSystem() shutdown triggered\n");
		}
		else
		{
			// IOCTRL error! Check returned error code and/or dev_wup filehandle!
			print2stdout("SWUpdateControl::shutdownSystem() IOCTRL error! Check returned error code and/or dev_wup filehandle!\n");
			retval=false;
		}

		if (OSAL_s32IOClose(_hDeviceWup) != OSAL_OK)
		{
			// ERROR!
      print2stdout("SWUpdateControl::shutdownSystem() Error on closing /dev/wup\n");
      retval=false;
		}
  }

  return retval;
}



SWUpdateControl::V850_MODE SWUpdateControl::getV850Mode()
{
  static const unsigned char MSG[]={0x22, 0xf1, 0x00};
  static const int  MSG_LEN=3;
  static const int  RESPONSE_LEN=7;
  INCConnection     connection;
  char              msg[256];
  bool              retval;

  if(!connection.open(50955))
  {
    print2stdout("SWUpdateControl::getV850Mode() connection.open() failed\n");
    return V850_MODE_UNDEFINED;
  }

  if(!connection.send(MSG, MSG_LEN))
  {
    print2stdout("SWUpdateControl::getV850Mode() connection.send() failed\n");
    return V850_MODE_UNDEFINED;
  }

  retval=connection.recv((void *)msg, sizeof(msg), RESPONSE_LEN);
  if(!retval)
  {
    print2stdout("SWUpdateControl::getV850Mode() connection.recv() failed\n");
    return V850_MODE_UNDEFINED;
  }

  if(msg[4]==0x00)
    return V850_MODE_BOOTLOADER;
  else if(msg[4]==0x61)
    return V850_MODE_APPLICATION;

  return V850_MODE_UNDEFINED;
}



SWUpdateControl::V850_MODE SWUpdateControl::setV850Mode(V850_MODE Mode)
{
  static const char BL_MSG[]={0x10, 0x60};
  static const char APP_MSG[]={0x11, 0x01};
  INCConnection     connection;
  const char        *msg;
  int               msg_len;
  V850_MODE         currmode;

  currmode=getV850Mode();
  if(currmode==Mode)
  {
    print2stdout("SWUpdateControl::setV850Mode() The V850 is already in that mode\n");
    return currmode;
  }

  if(Mode==V850_MODE_BOOTLOADER)
  {
    msg=BL_MSG;
    msg_len=sizeof(BL_MSG);//BL_MSG_LEN;
  }
  else if(Mode==V850_MODE_APPLICATION)
  {
    msg=APP_MSG;
    msg_len=sizeof(APP_MSG);//APP_MSG_LEN;
  }
  else
  {
    print2stdout("SWUpdateControl::setV850Mode() unknown mode\n");
    return V850_MODE_UNDEFINED;
  }

  if(!connection.open(50955))
  {
    print2stdout("SWUpdateControl::setV850Mode() connection.open() failed\n");
    return V850_MODE_UNDEFINED;
  }

  if(!connection.send(msg, msg_len))
  {
    print2stdout("SWUpdateControl::setV850Mode() connection.send() failed\n");
    return V850_MODE_UNDEFINED;
  }

  connection.close();

  usleep(500000);
  currmode=getV850Mode();
  if(currmode!=Mode)
  {
    print2stdout("SWUpdateControl::setV850Mode() V850 did not change mode\n");
    return currmode;
  }

  return Mode;
}



void SWUpdateControl::dumpV850Version(V850_MODE Mode)
{
  static const char BL_MSG[]={0x22, 0x25, 0x22};
  static const char APP_MSG[]={0x22, 0x25, 0x32};
  static const int  RESPONSE_LEN=43;
  INCConnection     connection;
  char              msg[256];

  if(!connection.open(50955))
  {
    print2stdout("SWUpdateControl::setV850Mode() connection.open() failed\n");
    return;
  }

  if(Mode!=V850_MODE_APPLICATION)
  {
    if(!connection.send(BL_MSG, sizeof(BL_MSG)))
    {
      print2stdout("SWUpdateControl::setV850Mode() connection.send() failed\n");
      return;
    }

    if(!connection.recv((void *)msg, sizeof(msg), RESPONSE_LEN))
    {
      print2stdout("SWUpdateControl::getV850Mode() connection.recv() failed\n");
      return;
    }

    if(Mode==V850_MODE_BOOTLOADER)
      print2stdout("%s\n", msg+3);
    else
      print2stdout("V850 bootloader version: %s\n", msg+3);
  }

  if(Mode!=V850_MODE_BOOTLOADER)
  {
    if(!connection.send(APP_MSG, sizeof(APP_MSG)))
    {
      print2stdout("SWUpdateControl::setV850Mode() connection.send() failed\n");
      return;
    }

    if(!connection.recv((void *)msg, sizeof(msg), RESPONSE_LEN))
    {
      print2stdout("SWUpdateControl::getV850Mode() connection.recv() failed\n");
      return;
    }
    if(Mode==V850_MODE_APPLICATION)
      print2stdout("%s\n", msg+3);
    else
      print2stdout("V850 applications version: %s\n", msg+3);
  }
}



void SWUpdateControl::checkV850Update()
{
#if 1
# define CHECK_DEBUG_PRINT(...)    print2stdout(__VA_ARGS__)
#else
# define CHECK_DEBUG_PRINT(...)
#endif

  static const char *SWU_V850_UPDATER_1="/usr/bin/swu_common_v850_app_out.out";
  static const char *SWU_V850_UPDATER_2="/opt/bosch/base/bin/swu_common_v850_app_out.out";

  static const char *SWITCH_MODE_COUNT_FILE="/var/opt/bosch/persistent/swupdate/switch_v850_mode_try_count.txt";
  static const char *FINALIZE_COUNT_FILE="/var/opt/bosch/persistent/swupdate/finalize_v850_try_count.txt";
  static const char *FINALIZE_SCRIPT="/var/opt/bosch/persistent/swupdate/finalize_v850.sh";
  static const char *DATA_FILE="/var/opt/bosch/persistent/swupdate/v850.dnl";
  int               mode;
  int               switch_mode_count=-1, finalize_count=-1;
  FILE              *fd_switch_mode_count;
  FILE              *fd_finalize_count;
  bool              has_data_file=!access(DATA_FILE, F_OK);
  bool              has_flash_script;

  fd_switch_mode_count=fopen(SWITCH_MODE_COUNT_FILE, "r");
  if(fd_switch_mode_count)
  {
    fscanf(fd_switch_mode_count, "%d", &switch_mode_count);
    fclose(fd_switch_mode_count);
    CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: read switch_mode_count as %i\n", switch_mode_count);
    if(switch_mode_count>0)
    {
      // If there is a marker file, update it immediately, just in case any attempt to do anything with the v850 would cause a crash
      fd_switch_mode_count=fopen(SWITCH_MODE_COUNT_FILE, "w");
      if(fd_switch_mode_count)
      {
        fprintf(fd_switch_mode_count, "%d", switch_mode_count-1);
        fclose(fd_switch_mode_count);
        sync();
        CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: updated %s with a count of %i\n", SWITCH_MODE_COUNT_FILE, switch_mode_count-1);
      }
    }
  }

  fd_finalize_count=fopen(FINALIZE_COUNT_FILE, "r");
  if(fd_finalize_count)
  {
    fscanf(fd_finalize_count, "%d", &finalize_count);
    fclose(fd_finalize_count);
    CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: read finalize_count as %i\n", finalize_count);
    if(finalize_count>0)
    {
      // If there is a marker file, update it immediately, just in case any attempt to do anything with the v850 would cause a crash
      fd_finalize_count=fopen(FINALIZE_COUNT_FILE, "w");
      if(fd_finalize_count)
      {
        fprintf(fd_finalize_count, "%d", finalize_count-1);
        fclose(fd_finalize_count);
        sync();
        CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: updated %s with a count of %i\n", FINALIZE_COUNT_FILE, finalize_count-1);
      }
    }
  }

  if(finalize_count<=0)
  {
    // There is no pending update, so bl mode would be unintentional
    mode=getV850Mode();
    if(mode!=SWUpdateControl::V850_MODE_APPLICATION)
    {
      print2stdout("SWUpdateControl::checkV850Update: V850 is not in application mode\n");
      if(switch_mode_count<0)
      {
        // create a marker file before attempting to switch the mode, just in case that causes a crash
        fd_switch_mode_count=fopen(SWITCH_MODE_COUNT_FILE, "w");
        if(fd_switch_mode_count)
        {
          switch_mode_count=3;    // This is intentional!
          // If we would set the count when we are unable to write the marker file, we could get stuck in an endless update loop.
          // This way there will be a regular update and there is at least a chance to create some sort of rescue stick.
          fprintf(fd_switch_mode_count, "%d", switch_mode_count);
          fclose(fd_switch_mode_count);
          sync();
          CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: Created marker file with try count for mode switches\n");
        }
      }
      if(switch_mode_count>0)
      {
        // Not in Application mode, try to switch
        if(setV850Mode(SWUpdateControl::V850_MODE_APPLICATION)==SWUpdateControl::V850_MODE_APPLICATION)
        {
          // Successfully switched to application mode
          // Clear counter, obsolete marker file will be removed later
          print2stdout("SWUpdateControl::checkV850Update: Switched V850 to application mode\n");
          switch_mode_count=-1;
        }
        else
        {
          print2stdout("SWUpdateControl::checkV850Update: Failed to switch V850 to application mode\n");
          // Failed to change the mode, so probably no applications in V850 available
          if(switch_mode_count==-1)
          {
            // No failed tries so far
            CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: No previous tries to switch mode\n");
            fd_switch_mode_count=fopen(SWITCH_MODE_COUNT_FILE, "w");
            if(fd_switch_mode_count)
            {
              switch_mode_count=3;    // This is intentional!
              // If we would set the count when we are unable to write the marker file, we could get stuck in an endless update loop.
              // This way there will be a regular update and there is at least a chance to create some sort of rescue stick.
              fprintf(fd_switch_mode_count, "%i", switch_mode_count);
              fclose(fd_switch_mode_count);
              sync();
              CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: Created marker file with try count for mode switches\n");
            }
          }
        }
      }
    }
    else if(switch_mode_count>=0)
    {
      switch_mode_count=-1;
    }
  }

  CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: switch_mode_count is %i\n", switch_mode_count);
  CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: finalize_count is %i\n", finalize_count);

  int count;

  if(!has_data_file)
    count=-1;
  else if(switch_mode_count==-1)
    count=finalize_count;
  else if(finalize_count==-1)
    count=switch_mode_count;
  else if(finalize_count<=switch_mode_count)
    count=finalize_count;
  else
    count=switch_mode_count;

  CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: count is %i\n", count);

  if(count>0)
  {
    int flash_retval=-1;

    // Update the marker files
    count--;
    if(switch_mode_count!=-1)
    {
      fd_switch_mode_count=fopen(SWITCH_MODE_COUNT_FILE, "w");
      if(fd_switch_mode_count)
      {
        fprintf(fd_switch_mode_count, "%d", count);
        fclose(fd_switch_mode_count);
        CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: updated %s with a count of %i\n", SWITCH_MODE_COUNT_FILE, count);
      }
    }
    if(finalize_count!=-1)
    {
      fd_switch_mode_count=fopen(FINALIZE_COUNT_FILE, "w");
      if(fd_switch_mode_count)
      {
        fprintf(fd_switch_mode_count, "%d", count);
        fclose(fd_switch_mode_count);
        CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: updated %s with a count of %i\n", FINALIZE_COUNT_FILE, count);
      }
    }
    sync();

    has_flash_script=!access(FINALIZE_SCRIPT, X_OK);
    if(!has_flash_script)
    {
      // No update script available, so create a default file
      CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: %s not found, creating it\n", FINALIZE_SCRIPT);
      FILE *fd_script=fopen(FINALIZE_SCRIPT, "w");
      if(fd_script)
      {
        struct stat buffer;
        fprintf(fd_script, "#!/bin/sh\n");
#if 1
        if(!stat(SWU_V850_UPDATER_1, &buffer))
          fprintf(fd_script, "%s -dnl /var/opt/bosch/persistent/swupdate/v850.dnl -bl -nocrc\n", SWU_V850_UPDATER_1);
        else
          fprintf(fd_script, "%s -dnl /var/opt/bosch/persistent/swupdate/v850.dnl -bl -nocrc\n", SWU_V850_UPDATER_2);
#else
        struct stat buffer;
        if(!stat(SWU_V850_UPDATER_1, &buffer))
          fprintf(fd_script, "%s -dnl /var/opt/bosch/persistent/swupdate/v850.dnl -bl -noreset -nocrc\n", SWU_V850_UPDATER_1);
        else
          fprintf(fd_script, "%s -dnl /var/opt/bosch/persistent/swupdate/v850.dnl -bl -noreset -nocrc\n", SWU_V850_UPDATER_2);
        fprintf(fd_script, "echo \"*flash* *flash* *flash*\"\n");                               // DEBUG!
        fprintf(fd_script, "%s/swupdatecontrol_out.out -setv850mode app\n", SWU_TARGET_PATH);   // DEBUG!
        fprintf(fd_script, "%s/swupdatecontrol_out.out -getv850mode\n", SWU_TARGET_PATH);       // DEBUG!
#endif
        fclose(fd_script);
        chmod(FINALIZE_SCRIPT, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IWGRP|S_IXGRP|S_IROTH|S_IWOTH|S_IXOTH);
        sync();
        has_flash_script=!access(FINALIZE_SCRIPT, X_OK);
      }
    }

    if(has_data_file)     CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: V850 data file is available\n");
    else                  CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: V850 data file is NOT available\n");
    if(has_flash_script)  CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: V850 flash script is available\n");
    else                  CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: V850 flash script is NOT available\n");

    if(has_data_file && has_flash_script)
    {
      char **args=(char **)malloc(sizeof(char *)*2);
      args[0]=strdup(FINALIZE_SCRIPT);
      args[1]=NULL;

      flash_retval=ExecCommand(args[0], args, true);
      free(args[0]);
      free(args);
      CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: Script returned %i\n", flash_retval);
    }

    if(!flash_retval)
    {
      CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: count is %i\n", count);
      // The update was successful - the v850 should be in application mode
      mode=getV850Mode();
      if(mode!=SWUpdateControl::V850_MODE_APPLICATION)
        // It is not, reset and try again
        flash_retval=-1;
    }

    if(flash_retval)
    {
      // Failed to flash the V850
      print2stdout("SWUpdateControl::checkV850Update: Update failed - rebooting\n");
      resetProcessorSystem(); // This will probably not work while the V850 is in BL mode
      sleep(15);
      resetProcessorApp();
      sleep(15);
    } 
    print2stdout("SWUpdateControl::checkV850Update: Update succeeded\n");
  }
  else if(count==0)
  {
    if(getV850Mode()!=SWUpdateControl::V850_MODE_APPLICATION)
    {
      print2stdout("The V850 is still not in application mode but three tries to update it failed.\n");
      print2stdout("No more tries will be performed so a special rescue stick can be used!\n");
      // Leave the marker files alone so no more updates will be performed
      return;
    }
  }

  // The update was successful and v850 is in application mode
  // Therefore the marker files are not needed anymore
  if(!access(SWITCH_MODE_COUNT_FILE, F_OK))
  {
    CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: Removing obsolete file %s\n", SWITCH_MODE_COUNT_FILE);
    unlink(SWITCH_MODE_COUNT_FILE);
    sync();
  }
  if(!access(FINALIZE_COUNT_FILE, F_OK))
  {
    CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: Removing obsolete file %s\n", FINALIZE_COUNT_FILE);
    unlink(FINALIZE_COUNT_FILE);
    sync();
  }
  if(!access(FINALIZE_SCRIPT, F_OK))
  {
    CHECK_DEBUG_PRINT("SWUpdateControl::checkV850Update: Removing obsolete file %s\n", FINALIZE_SCRIPT);
    unlink(FINALIZE_SCRIPT);
    sync();

    /*
      this is required because ADR is failing in the same power cycle only when debug board is connected.
      need power cycle for ADR to work as expected, 
      this is needed only for compactability index increase otherwise it is not needed.
      this hack to be removed after the ADR fix
    */
    //print2stdout("SWUpdateControl::checkV850Update: reset the system to fix ADR issue when debug connected\n");
    //resetProcessorSystem();

  }  

}



bool SWUpdateControl::GetFcId(unsigned int *FcId)
{
  unsigned int    pjid;
  unsigned short  fcid;
  KdsTable        table(0xA070, 16);

  if(!GetPjId(&pjid))
  {
    print2stdout("SWUpdateControl::GetFcId(): SWUpdateControl::GetPjId() failed\n");
    return false;
  }

  if(!table.read())
  {
    print2stdout("SWUpdateControl::GetFcId(): table.read() failed\n");
    return false;
  }

  if(pjid==0x09 || pjid==0x0A)  // GM project: FCID is in Bytes 1/2
  {
    if(!table.getU16(&fcid, 1))
    {
      print2stdout("SWUpdateControl::GetFcId(): table.getU16() failed\n");
      return false;
    }
  }
  else  // All projects except GM: FCID is in bytes 3/4
  {
    if(!table.getU16(&fcid, 3))
    {
      print2stdout("SWUpdateControl::GetFcId(): table.getU16() failed\n");
      return false;
    }
  }

  *FcId=(unsigned int)fcid;

  return true;
}

bool SWUpdateControl::setFcId(std::string value) {

   tU32 u32Val;
   std::stringstream ss;
   ss << std::hex << value;
   ss >> u32Val;
   ETG_TRACE_USR2(("int value for fcid:%u", u32Val));
   
   swu::KdsItemDescription<tU32> kds (swu::KdsItemDescription<tU32>(0xA070, 16, 3, 2));
   return kds.write(u32Val);
   
}



bool SWUpdateControl::GetPjId(unsigned int *PjId)
{
  unsigned char   pjid;
  KdsTable        table(0xA070, 16);

  if(!table.read())
  {
    print2stdout("SWUpdateControl::GetPjId(): table.read() failed\n");
    return false;
  }
  if(!table.getU8(&pjid, 15))
  {
    print2stdout("SWUpdateControl::GetPjId(): table.getU16() failed\n");
    return false;
  }

  *PjId=(unsigned int)pjid;

  return true;
}



bool SWUpdateControl::ReadKDSValue(unsigned int Key, unsigned int Off, unsigned int Len)
{
  KdsTable      table(static_cast<tU16> (Key),static_cast<tU16> (Off+Len));
  unsigned char *data=NULL;

  if(!table.read())
  {
    print2stdout("SWUpdateControl::ReadKDSValue(): table.read() failed\n");
    return false;
  }
  data=(unsigned char *)malloc(Off+Len);
  if(!table.getBytes(data,static_cast<tU16> (Off),static_cast<tU16> (Len)))
  {
    print2stdout("SWUpdateControl::ReadKDSValue(): table.getBytes() failed\n");
    free(data);
    return false;
  }

  print2stdout("Read from KDS:\n");
  for(uint32_t i=0; i<Len; i+=16)
  {
    print2stdout("%04i:", Off+i);
    for(uint32_t j=0; j<8 && i+j<Len; j++)
      print2stdout(" %02X", data[i+j]);
    if(i+8<Len)
      print2stdout(" -");
    for(uint32_t j=8; j<16 && i+j<Len; j++)
      print2stdout(" %02X", data[i+j]);
    print2stdout("\n");
  }

  free(data);
  return true;
}

bool SWUpdateControl::WriteKDSValue(unsigned int Key, unsigned int Off, const char *Val)
{
  tU16           vallen=0;
  unsigned char *val=NULL;

  if(!(vallen=static_cast<tU16> (StringToByte(&val, Val))))
  {
      print2stdout("\nSWUpdateControl::WriteKDSValue(): No values to write\n");
      return false;
  }
  print2stdout("XXX:vallen:%u\n", vallen);

  KdsTable          table(static_cast<tU16>(Key),static_cast<tU16> (Off+vallen) );
  const tsKDSEntry  *entry=table.getKDSEntry();
  if(!table.read())
  {
    print2stdout("SWUpdateControl::WriteKDSValue(): table.read() failed\n");
    free(val);
    return false;
  }
  print2stdout("XX:vallen:%u\n", vallen);
  if(!table.setBytes(val,static_cast<tU16> (Off),vallen))
  {
    print2stdout("SWUpdateControl::WriteKDSValue(): table.setBytes() failed\n");
    free(val);
    return false;
  }

  print2stdout("Write to KDS:\n");
  hexdump(entry->au8EntryData, entry->u16EntryLength);

  if(!table.write())
  {
    print2stdout("SWUpdateControl::ReadKDSValue(): table.write() failed\n");
    free(val);
    return false;
  }

  free(val);
  return true;
}

void SWUpdateControl::WriteToRegistry(std::string key, char* value)
{
	RegistryAccess registry;
	registry.u32RegWriteValue(DEVICE_NAME, key.c_str(), value);
}

void SWUpdateControl::dumpEmmcInfo() {

   std::string outputFile = "/tmp/emmcInfo.txt";
   std::string cmd = "cat /sys/class/mmc_host/mmc1/mmc1:0001/cid > " + outputFile;
   unlink(outputFile.c_str());

   std::string info;

   do {

      if(swu::execCommand(cmd.c_str())) {
         break;
      }

      std::ifstream infile(outputFile.c_str());
      if(!infile.is_open()) {
         break;
      }

      if(!std::getline(infile, info)) {
         break;
      }

      if(info.compare(0, 2, "13") == 0) {
         print2stdout("emmcChipType:Micron \n");

         unlink(outputFile.c_str());
         cmd = "cat /sys/kernel/debug/mmc1/mmc1\:0001/ext_csd > " + outputFile;
         if(!swu::execCommand(cmd.c_str())) {
            std::string versionInfo;

            if(swu::loadFile(outputFile, versionInfo)) {
               print2stdout("emmcChipVersion:\n %s", versionInfo.c_str());
               print2stdout("\n");
               
               //[261-258]
               std::string upVersion;
               for(int index = 522; index >= 516; index = index - 2) { 
                  upVersion = upVersion + versionInfo.substr(index, 2);
               }
               print2stdout("upper version[261-258]:0x%s", upVersion.c_str());
               print2stdout("\n");

               //[257-254]
               std::string lowerVersion;
               for(int index = 514; index >= 508; index = index - 2) { 
                  lowerVersion = lowerVersion + versionInfo.substr(index, 2);
               }
               print2stdout("lower version[257-254]:0x%s", lowerVersion.c_str());
               print2stdout("\n");

            } 
         }

      } else if (info.compare(0, 2, "15") == 0) {
         print2stdout("emmcChipType:Samsung \n");         
      } else {
         print2stdout("emmcChipType:unknown \n");
      }
      break;

   }while(0);
}

bool SWUpdateControl::IncSendValue(uint16_t Port, const std::string Host, const char *Val)
{
  int               vallen=0;
  unsigned char     *val=NULL;
  INCConnection     connection;

  if(!(vallen=StringToByte(&val, Val)))
  {
      print2stdout("\nSWUpdateControl::IncSendValue(): No values to send\n");
      return false;
  }

  if(!connection.open(Port, Host))
  {
    print2stdout("SWUpdateControl::IncSendValue() connection.open() failed for %s:%X\n", Host.c_str(), Port);
    free(val);
    return false;
  }

  print2stdout("Sending to %s:%X\n", Host.c_str(), Port);
  hexdump(val, vallen);
  if(!connection.send(val, vallen))
  {
    print2stdout("SWUpdateControl::IncSendValue() connection.send() failed\n");
    free(val);
    return false;
  }
  free(val);

  return true;
}

bool SWUpdateControl::IncSendRecvValue(uint16_t Port, const std::string Host, const char *Val)
{
  int               vallen=0;
  unsigned char     *val=NULL;
  INCConnection     connection;
  unsigned char     msg[256];
  int               retval;

  if(!(vallen=StringToByte(&val, Val)))
  {
      print2stdout("\nSWUpdateControl::IncSendRecvValue(): No values to send\n");
      return false;
  }

  if(!connection.open(Port, Host))
  {
    print2stdout("SWUpdateControl::IncSendRecvValue() connection.open() failed for %s:%X\n", Host.c_str(), Port);
    free(val);
    return false;
  }

  print2stdout("Sending to %s:%X\n", Host.c_str(), Port);
  hexdump(val, vallen);
  if(!connection.send(val, vallen))
  {
    print2stdout("SWUpdateControl::IncSendRecvValue() connection.send() failed\n");
    free(val);
    return false;
  }
  free(val);

  if((retval=connection.recv((void *)msg, sizeof(msg)))<=0)
  {
    print2stdout("SWUpdateControl::IncSendRecvValue() connection.recv() failed\n");
  }

  print2stdout("Received %i bytes:\n", retval);

  if(cpldFlag)
  {

 	  getCpldVersion((unsigned char *)msg);
 	  cpldFlag = 0;
  }

  hexdump(msg, retval);

  return true;
}

void SWUpdateControl::getCpldVersion(unsigned char *msg)
{
	print2stdout("Inside getCpldVersion\n");
	std::string version = " " ;
	if( NULL != msg )
	{
	   print2stdout("msg[0]= 0x%x\n",msg[0]); //Response code
	   print2stdout("msg[1]= 0x%x\n",msg[1]); //size of msg
	   print2stdout("msg[2]= 0x%x\n",msg[2]); //will have all information,need to break down to get version
/*
	   switch( msg[0] )
	   {	               // response from V850
	      case DL_RESPONSE_RECEIVED:
	      {
	                  //work on the server response
	          bRetVal = bV850DownloadProcess( &msg[2] , msg[1]);
	          break;
	      }
	      default:
	      {
	         break;
	      }
	    }
	*/

		if( version.compare("") )
		{
			print2stdout("INFO: Cpld version : %s\n",version.c_str());
		}
		std::ofstream cpldVersion("/tmp/CpldVersion.txt");
		cpldVersion << version;
		cpldVersion.close();
	}
}

#if 0
# define PRAM_DEBUG_PRINT(...)    print2stdout(__VA_ARGS__)
#else
# define PRAM_DEBUG_PRINT(...)
#endif
int SWUpdateControl::getPRAMCounter(const char *Name)
{
  int retval=-1;

  OSAL_tIODescriptor hDevicePRAM=OSAL_IOOpen(Name, OSAL_EN_READWRITE);
  if(hDevicePRAM==OSAL_ERROR)
    print2stdout("SWUpdateControl::getPRAMCounter(): OSAL_IOOpen failed for %s\n", Name);
  else
  {
    PRAM_DEBUG_PRINT("SWUpdateControl::getPRAMCounter(): opened %s\n", Name);
    int ret=OSAL_s32IOControl(hDevicePRAM, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, PRAM_COUNTER_INDEX);
    if(ret==OSAL_ERROR)
      print2stdout("SWUpdateControl::getPRAMCounter(): PRAM_SEEK failed for %s\n", Name);
    else
    {
      unsigned char count;

      ret=OSAL_s32IORead(hDevicePRAM, (signed char *)&count, sizeof(count));
      if(ret==OSAL_ERROR || ret!=sizeof(count))
        print2stdout("SWUpdateControl::getPRAMCounter(): PRAM_READ ERROR for %s! Error=0x%08x\n", Name, ret);
      else
      {
        PRAM_DEBUG_PRINT("SWUpdateControl::getPRAMCounter(): read %i from %s\n", count, Name);
        // read the check counter
        ret=OSAL_s32IOControl(hDevicePRAM, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, PRAM_CHECK_INDEX);
        if(ret==OSAL_ERROR)
        {
          print2stdout("SWUpdateControl::getPRAMCounter(): PRAM_SEEK failed for %s\n", Name);
        }
        else
        {
          unsigned char check;

          ret=OSAL_s32IORead(hDevicePRAM, (signed char *)&check, sizeof(check));
          if(ret==OSAL_ERROR || ret!=sizeof(check))
          {
            print2stdout("SWUpdateControl::getPRAMCounter(): PRAM_READ ERROR for %s! Error=0x%08x\n", Name, ret);
          }
          else
          {
            if(check==0xFF-count)
            {
              retval=(int)count;
              PRAM_DEBUG_PRINT("SWUpdateControl::getPRAMCounter(): read Check %i from %s\n", check, Name);
            }
            else
            {
              // If the counter is not initialized, it is considered to have a value of 0
              retval=0;
              PRAM_DEBUG_PRINT("SWUpdateControl::getPRAMCounter(): Check is %i but should be %i for %s\n", check, 0xFF-count, Name);
            }
          }
        }
      }
    }
    OSAL_s32IOClose(hDevicePRAM);
  }

  return retval;
}



bool SWUpdateControl::setPRAMCounter(const char *Name, unsigned char Count)
{
  bool retval=false;

  OSAL_tIODescriptor hDevicePRAM=OSAL_IOOpen(Name, OSAL_EN_READWRITE);
  if(hDevicePRAM==OSAL_ERROR)
    print2stdout("SWUpdateControl::setPRAMCounter(): OSAL_IOOpen failed for %s\n", Name);
  else
  {
    PRAM_DEBUG_PRINT("SWUpdateControl::setPRAMCounter(): opened %s\n", Name);
    // after the device has been opened use IOControl to hand over callback pointer.
    int ret=OSAL_s32IOControl(hDevicePRAM, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, PRAM_COUNTER_INDEX);
    if(ret==OSAL_ERROR)
      print2stdout("SWUpdateControl::setPRAMCounter(): PRAM_SEEK failed for %s\n", Name);
    else
    {
      unsigned char count=Count;

      ret=OSAL_s32IOWrite(hDevicePRAM, (signed char *)&count, sizeof(count));
      if(ret==OSAL_ERROR || ret!=sizeof(count))
        print2stdout("SWUpdateControl::setPRAMCounter(): PRAM_WRITE ERROR for %s! Error=0x%08x\n", Name, ret);
      else
      {
        PRAM_DEBUG_PRINT("SWUpdateControl::getPRAMCounter(): wrote %i to %s\n", count, Name);
        // write the check counter
        ret=OSAL_s32IOControl(hDevicePRAM, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, PRAM_CHECK_INDEX);
        if(ret==OSAL_ERROR)
        {
          print2stdout("SWUpdateControl::setPRAMCounter(): PRAM_SEEK failed for %s\n", Name);
        }
        else
        {
          unsigned char check=static_cast<unsigned char> (0xFF-Count);

          ret=OSAL_s32IOWrite(hDevicePRAM, (signed char *)&check, sizeof(check));
          if(ret==OSAL_ERROR || ret!=sizeof(check))
          {
            print2stdout("SWUpdateControl::setPRAMCounter(): PRAM_WRITE ERROR for %s! Error=0x%08x\n", Name, ret);
          }
          else
          {
            retval=true;
            PRAM_DEBUG_PRINT("SWUpdateControl::getPRAMCounter(): wrote Check %i to %s\n", check, Name);
          }
        }
      }
    }
    OSAL_s32IOClose(hDevicePRAM);
  }

  return retval;
}



int SWUpdateControl::getResetCounter(void)
{
  return getPRAMCounter(PRAM_RESET_COUNTER);
}

bool SWUpdateControl::setResetCounter(unsigned char Count)
{
  return setPRAMCounter(PRAM_RESET_COUNTER, Count);
}

int SWUpdateControl::getForceDnlMagic(void)
{
  return getPRAMCounter(PRAM_FORCEDNL_MAGIC);
}

bool SWUpdateControl::setForceDnlMagic(unsigned char Count)
{
  return setPRAMCounter(PRAM_FORCEDNL_MAGIC, Count);
}



tU8 SWUpdateControl::checkRstCtrAndSetRecvFlag (tU8 u8ResetCntrVal)
{
  int RstCntrValRead;
  tU8 u8Return = SWU_RETURN_FAILURE;

  //read reset counter
  RstCntrValRead=getPRAMCounter(PRAM_RESET_COUNTER);
  if(RstCntrValRead<0)
  {
    //failed to read reset counter
    print2stdout( ( "Could not read reset counter from DEV_PRAM\n" ) );
    return u8Return;
  }

  //successfully read the reset counter
  if (u8ResetCntrVal == (tU8)RstCntrValRead) //check the value of reset counter with default value passed by script
  {
    print2stdout("Cyclic reset handling to be done!! Setting recovery magic..\n");
    activateRecoveryBootChain();        //set the recovery magic
    u8Return = SWU_RETURN_SUCCESS;
  }
  else
  {
    //no reset handling required
    print2stdout("No Cyclic reset handling required. Continue...\n");
    u8Return  = SWU_RETURN_FAILURE_RSTCNTR_CMP;
    return u8Return;
  }

  PRAM_DEBUG_PRINT("SWUpdateControl::checkRstCtrAndSetRecvFlag left with code %d\n", u8Return);   //to be removed*/

  return u8Return;
}



bool SWUpdateControl::ParseHexChar(unsigned int *Value, const unsigned char Char)
{
  if(Char>='0' && Char<='9')
    *Value=(*Value<<4)+Char-'0';
  else if(Char>='a' && Char<='f')
    *Value=(*Value<<4)+10+Char-'a';
  else if(Char>='A' && Char<='F')
    *Value=(*Value<<4)+10+Char-'A';
  else
    return false;
  return true;
}

bool SWUpdateControl::AddToOTPMap(const unsigned char *Buff, const int BuffLen)
{
  unsigned int key=0, value=0;

  if(BuffLen<FUSE_FILE_LINE_LENGTH-1)
    return false;

  for(int i=0; i<FUSE_FILE_ID_LENGTH; i++)
    if(!ParseHexChar(&key, Buff[i]))
      return false;

  if(Buff[FUSE_FILE_ID_LENGTH]!=':' || Buff[FUSE_FILE_ID_LENGTH+1]!=' ')
    return false;

  for(int i=0; i<FUSE_FILE_VALUE_LENGTH; i+=2)
  {
    if(!ParseHexChar(&value, Buff[FUSE_FILE_VALUE_OFFSET+i]))
      return false;
    if(!ParseHexChar(&value, Buff[FUSE_FILE_VALUE_OFFSET+i+1]))
      return false;
  }

  DataMap[key]=value;

  return true;
}

bool SWUpdateControl::ReadOTPArea(void)
{
  FILE            *otp_file;
  unsigned char   buff[FUSE_FILE_LINE_LENGTH];
  int             bufflen=0;
  bool            result=false;

  DataMap.clear();
  if(!(otp_file=fopen(FUSE_FILE_NAME, "r")))
  {
      print2stdout("SWUpdateControl::ReadOTPArea(): Failed to open otp file");
  }
  else
  {
    memset(buff, 0, sizeof(buff));
    for(int c=fgetc(otp_file); c!=EOF; c=fgetc(otp_file))
    {
      if(bufflen >= static_cast<int> (sizeof(buff)))  
      {
        print2stdout("SWUpdateControl::ReadOTPArea() buffsize exceeded");
        goto FAIL;
      }
      if(c!='\n')
        buff[bufflen++]=(unsigned char)c;
      else
      {
        if(!AddToOTPMap(buff, bufflen))
        {
          print2stdout("SWUpdateControl::ReadOTPArea() add failed");
          goto FAIL;
        }
        memset(buff, 0, sizeof(buff));
        bufflen=0;
      }
    }
    if(bufflen && !AddToOTPMap(buff, bufflen))
    {
      print2stdout("SWUpdateControl::ReadOTPArea() add failed");
      goto FAIL;
    }
    result=true;
  }
  goto DONE;

FAIL:
  DataMap.clear();

DONE:
  if(otp_file)
    fclose(otp_file);

  return result;
}

bool SWUpdateControl::GetOTPData(unsigned int *Value, unsigned int FuseID)
{
  OTPDataMap::const_iterator  iter;

  if(!DataMap.size() && !ReadOTPArea())
    return false;

  if((iter=DataMap.find(FuseID))==DataMap.end())
    return false;

  *Value=(*iter).second;
  return true;
}

int SWUpdateControl::GetSecureBootModeEnabled(void)
{
  unsigned int  fuse;
  if(GetOTPData(&fuse, SECURE_BOOT_MODE_ENABLED_FUSE))
    return (fuse&SECURE_BOOT_MODE_ENABLED_BIT)?1:0;
  return -1; 
}

int SWUpdateControl::GetSRKRevocation(void)
{
  unsigned int  fuse;
  if(GetOTPData(&fuse, SECURE_BOOT_REVOCATION_FUSE))
    return fuse&SECURE_BOOT_REVOCATION_BITS;
  return -1; 
}



SWUpdateControl::PerfChannelInfo *SWUpdateControl::PerfChannelInfoBase=NULL, *SWUpdateControl::PerfChannelInfoCurrent=NULL;

#if 0
# define PERF_DEBUG_PRINT(...)    print2stdout(__VA_ARGS__)
#else
# define PERF_DEBUG_PRINT(...)
#endif

void SWUpdateControl::StartPerfChn(const char *Name)
{
  PerfChannelInfo *info, *child, *prev;

  if(!PerfChannelInfoBase)
  {
    PERF_DEBUG_PRINT("Creating Base Info\n");
    PerfChannelInfoBase=new PerfChannelInfo();
    PerfChannelInfoBase->Name=strdup("Base");
    PerfChannelInfoBase->StartTime=GetTimeInMS();
    PerfChannelInfoBase->EndTime=-1.0;
    PerfChannelInfoCurrent=PerfChannelInfoBase;
  }

  for(child=PerfChannelInfoCurrent->Child; child; child=child->Next)
    if(!strcmp(child->Name, Name))
    {
      PERF_DEBUG_PRINT("Continue with Channel %s\n", child->Name);
      PerfChannelInfoCurrent=child;
      PerfChannelInfoCurrent->StartTime=GetTimeInMS();
      PerfChannelInfoCurrent->EndTime=-1.0;
      print2stdout("\033[44m-=[TIME]=-\033[0m\tStarted channel \"%s\" at second %.3f\n", Name, PerfChannelInfoCurrent->StartTime-PerfChannelInfoBase->StartTime);
      return;
    }

  PERF_DEBUG_PRINT("Adding Channel %s as Child to %s\n", Name, PerfChannelInfoCurrent->Name);
  info=new PerfChannelInfo();
  info->Name=strdup(Name);
  info->Parent=PerfChannelInfoCurrent;
  prev=PerfChannelInfoCurrent->Child;
  if(!prev)
    PerfChannelInfoCurrent->Child=info;
  else
  {
    while(prev->Next)
      prev=prev->Next;
    prev->Next=info;
  }
  PERF_DEBUG_PRINT("New: %s, Parent: %s, Prev: %s\n", info->Name, info->Parent->Name, prev?prev->Name:"None");

  PerfChannelInfoCurrent=info;
  PerfChannelInfoCurrent->StartTime=GetTimeInMS();
  PerfChannelInfoCurrent->EndTime=-1.0;

  print2stdout("\033[44m-=[TIME]=-\033[0m\tStarted channel \"%s\" at second %.3f\n", Name, PerfChannelInfoCurrent->StartTime-PerfChannelInfoBase->StartTime);
}



void SWUpdateControl::StopPerfChn(const char *Name)
{
  PerfChannelInfo *info;
  double          endsec;

  if(!strcmp(PerfChannelInfoCurrent->Name, Name))
  {
    PERF_DEBUG_PRINT("Stopping active Channel %s\n", Name);
    if(PerfChannelInfoCurrent->EndTime>=0.0)
    {
      print2stdout("\033[44m-=[TIME]=-\033[0m\tChannel \"%s\" was stopped already\n", Name);
      return;
    }
    PerfChannelInfoCurrent->EndTime=GetTimeInMS();
    endsec=PerfChannelInfoCurrent->EndTime-PerfChannelInfoCurrent->StartTime;
    PerfChannelInfoCurrent->DeltaTime+=endsec;
    print2stdout("\033[44m-=[TIME]=-\033[0m\tStop channel \"%s\" at second %.3f (was running for %.3f sec)\n", Name, PerfChannelInfoCurrent->EndTime-PerfChannelInfoBase->StartTime, endsec);
    PerfChannelInfoCurrent=PerfChannelInfoCurrent->Parent;
    return;
  }

  PERF_DEBUG_PRINT("Channel %s is not the active one!\n", Name);
  for(info=PerfChannelInfoCurrent->Parent; info && info!=PerfChannelInfoBase; info=info->Parent)
    if(!strcmp(info->Name, Name))
    {
      if(info->EndTime>=0.0)
      {
        print2stdout("\033[44m-=[TIME]=-\033[0m\tChannel \"%s\" was stopped already\n", Name);
        return;
      }
      PERF_DEBUG_PRINT("Stopping Channel %s and all children\n", info->Name);
      info->EndTime=GetTimeInMS();
      endsec=info->EndTime-info->StartTime;
      info->DeltaTime+=endsec;
      //print2stdout("\033[44m-=[TIME]=-\033[0m\tStop channel \"%s\" after % 9.3f seconds\n", info->Name, endsec);
      print2stdout("\033[44m-=[TIME]=-\033[0m\tStop channel \"%s\" at second %.3f (was running for %.3f sec)\n", info->Name, info->EndTime-PerfChannelInfoBase->StartTime, endsec);
      StopPerfChn(info->Child);
      PerfChannelInfoCurrent=info;
      return;
    }
  PERF_DEBUG_PRINT("%s is not an active channel!\n", Name);
}



void SWUpdateControl::StopPerfChn(PerfChannelInfo *Info)
{
  double  endsec;

  if(Info->EndTime<0.0)
  {
    Info->EndTime=GetTimeInMS();
    endsec=Info->EndTime-Info->StartTime;
    Info->DeltaTime+=endsec;
    //print2stdout("\033[44m-=[TIME]=-\033[0m\tStop channel \"%s\" after % 9.3f seconds\n", Info->Name, endsec);
    print2stdout("\033[44m-=[TIME]=-\033[0m\tStop channel \"%s\" at second %.3f (was running for %.3f sec)\n", Info->Name, Info->EndTime-PerfChannelInfoBase->StartTime, endsec);
  }
else
 PERF_DEBUG_PRINT("%s already stopped\n", Info->Name);

  if(Info->Child)
    StopPerfChn(Info->Child);
  if(Info->Next)
    StopPerfChn(Info->Next);
}



void SWUpdateControl::DumpPerf(PerfChannelInfo *Info, int Depth)
{
  int i;

  print2stdout("\033[44m-=[TIME]=-\033[0m % 9.3f sec", Info->DeltaTime);
  for(i=0; i<Depth; i++)
    print2stdout("..");
  print2stdout(" %s\n", Info->Name);

  if(Info->Child)
    DumpPerf(Info->Child, Depth+1);
  if(Info->Next)
    DumpPerf(Info->Next, Depth);
}



void SWUpdateControl::StartPerfServer(void)
{
  int                 sockfd=-1;
  struct sockaddr_in  sockaddrLocal;
  struct sockaddr_in  sockaddrRemote;
  unsigned int        addrLen=sizeof(sockaddrRemote);
  char                recvBuff[512];
  int                 recvLen=sizeof(sockaddrRemote);

  sockfd=socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if(sockfd<0)
  {
    print2stdout("SWUpdateControl::StartPerfServer(): socket() failed with errno=%i\n", errno);
    goto EXIT;
  }

  memset(&sockaddrLocal, 0, sizeof(sockaddrLocal));
  sockaddrLocal.sin_family=AF_INET;
  sockaddrLocal.sin_addr.s_addr=htonl(INADDR_ANY);
  sockaddrLocal.sin_port=htons(666);

  if(bind(sockfd, (struct sockaddr *)&sockaddrLocal, sizeof(sockaddrLocal))<0)
  {
    print2stdout("SWUpdateControl::StartPerfServer(): bind() failed with errno=%i\n", errno);
    goto EXIT;
   }

  for(; ; )
  {
    recvLen=static_cast<int> (recvfrom(sockfd, recvBuff, sizeof(recvBuff), 0, (struct sockaddr *)&sockaddrRemote, &addrLen) );
    if(recvLen>0)
    {
      recvBuff[recvLen]=0;
      PERF_DEBUG_PRINT("received message: \"%s\"\n", recvBuff);
      PERF_DEBUG_PRINT("Base: %s\n", PerfChannelInfoBase?PerfChannelInfoBase->Name:"None");
      PERF_DEBUG_PRINT("Current: %s\n", PerfChannelInfoCurrent?PerfChannelInfoCurrent->Name:"None");
      if(!strncmp(recvBuff, "start ", 6))
        StartPerfChn(recvBuff+6);
      else if(!strncmp(recvBuff, "stop ", 5))
        StopPerfChn(recvBuff+5);
      else if(!strcmp(recvBuff, "stop_all"))
      {
        if(PerfChannelInfoBase && PerfChannelInfoBase->Child)
        {
          PerfChannelInfoCurrent=PerfChannelInfoBase;
          StopPerfChn(PerfChannelInfoBase->Child);
        }
      }
      else if(!strcmp(recvBuff, "dump"))
      {
        if(!PerfChannelInfoBase)
          print2stdout("SWUpdateControl::StartPerfServer(): No entries to dump\n");
        else
        {
          PerfChannelInfoBase->EndTime=GetTimeInMS();
          PerfChannelInfoBase->DeltaTime=PerfChannelInfoBase->EndTime-PerfChannelInfoBase->StartTime;
          DumpPerf(PerfChannelInfoBase, 0);
        }
      }
      PERF_DEBUG_PRINT("Base: %s\n", PerfChannelInfoBase?PerfChannelInfoBase->Name:"None");
      PERF_DEBUG_PRINT("Current: %s\n", PerfChannelInfoCurrent?PerfChannelInfoCurrent->Name:"None");
    }
  }

EXIT:
  if(sockfd!=-1)
  {
    close(sockfd);
    sockfd=-1;
  }
}



void SWUpdateControl::SendPerfMsg(const char *Msg)
{
  int                 sockfd=-1;
  struct hostent      *hostentRemote;
  struct sockaddr_in  sockaddrRemote;

  sockfd=socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if(sockfd<0)
  {
    print2stdout("SWUpdateControl::StartPerfChn(): socket() failed with errno=%i\n", errno);
    return;
  }

  hostentRemote=gethostbyname("localhost");
  if(!hostentRemote)
  {
    print2stdout("SWUpdateControl::StartPerfChn(): gethostbyname() failed with errno=%i\n", errno);
    close(sockfd);
    return;
  }

  memset(&sockaddrRemote, 0, sizeof(sockaddrRemote));
  sockaddrRemote.sin_family=AF_INET;
  memcpy((void *)&sockaddrRemote.sin_addr, hostentRemote->h_addr_list[0], hostentRemote->h_length);
  sockaddrRemote.sin_port=htons(666);

  if(sendto(sockfd, Msg, strlen(Msg), 0, (struct sockaddr *)&sockaddrRemote, sizeof(sockaddrRemote))<0)
  {
    print2stdout("SWUpdateControl::StartPerfChn(): sendto() failed with errno=%i\n", errno);
  }

  close(sockfd);
  return;
}

} // namespace ai_sw_update {
} // namespace common {

