/**
 * @addtogroup ai_sw_update
 * @{
 *
 * @file       DigestStat.h
 * @author     Philipp Blanke (blp4hi) <philipp.blanke@de.bosch.com>
 * @date       Thu 14 Apr 2016, 13:26
 * @copyright  Robert Bosch Car Multimedia GmbH
 * @brief      Classes to encapsulate Digests and Stats for files.
 */

#ifndef DIGESTSTAT_H
#define DIGESTSTAT_H
#include <sys/types.h>
#include <openssl/md5.h> // MD5_DIGEST_LENGTH

namespace fingerprint
{
const int DIGEST_LEN = MD5_DIGEST_LENGTH;   // 16
const int UID_LEN  = sizeof(uid_t);         // 1
const int GID_LEN  = sizeof(gid_t);         // 1
const int MODE_LEN = sizeof(mode_t);        // 4
const int STAT_LEN = UID_LEN + GID_LEN + MODE_LEN;
const int DIGEST_STR_LEN = 2* DIGEST_LEN + 1;
const int STAT_STR_LEN = 2 * STAT_LEN + 1;

struct Digest {
   unsigned char x[DIGEST_LEN];

   Digest();
   Digest(const Digest& other);

   Digest& operator=(const Digest& other);
   bool operator== (const Digest& other) const;
   bool operator!= (const Digest& other) const;
   void print(char* hex, int len) const;
   bool read(const char* hex, int len);
};

/** Aggregated digest and stat information for a file. */
class DigestStat
{
public:

   DigestStat();

   void printDigest (char* hex, int len) const;
   bool readDigest (const char* hex, int len);

   void printCapDigest (char* hex, int len) const;
   bool readCapDigest (const char* hex, int len);

   void printStat (char* hex, int len) const;
   bool readStat (const char* hex, int len);

   void copyDigest (const DigestStat& other);

   bool isRegularFile () const;
   bool isSymlink () const;
   bool isDirectory () const;

   uid_t uid() const;
   void setUid (uid_t uid);
   uid_t gid() const;
   void setGid(gid_t gid);
   mode_t mode() const;
   void setMode(mode_t mode);

   bool isLinkToReg () const; 
   void setLinkToReg (bool b);

   const Digest& digest () const;
   void setDigest(const Digest& digest);

   const Digest& capDigest () const;
   void setCapDigest(const Digest& capDigest);

   bool isDigestSet () const;
   bool isCapDigestSet() const;

private:
   Digest _digest;
   Digest _capDigest;
   bool _is_digest_set;
   bool _is_cap_digest_set;
   uid_t _uid;
   gid_t _gid;
   mode_t _mode;
   bool _link_to_reg; // if this is a link and it links to a regular file
};

} // ns fingerprint
#endif
