/*
 * @file       DigestStat.cpp
 * @author     Philipp Blanke (blp4hi) <philipp.blanke@de.bosch.com>
 * @date       Fri 25 Nov 2016, 09:34
 * @copyright  Robert Bosch Car Multimedia GmbH
 */
#define _XOPEN_SOURCE 500  // Use X/Open 5, incorporating POSIX 1995
#include <cstdio>
#include <cassert>
#include <sys/stat.h>
#include "DigestStat.h"

#include "my_etg.h"

using namespace fingerprint;

/** Helper to convert an unsigned char to an ascii character representing a 
 * hexadecimal. */
const char hexmap[] = {'0', '1', '2', '3', '4', '5', '6', '7',
                       '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

/** Convert ascii character that represents a hexadecimal digit to its 
 *  value as an unsigned char. */
unsigned char hex2uc(char c)
{
   if (c >= '0' && c <= '9') {
      return static_cast<unsigned char>(c - '0');
   }
   else if (c >= 'a' && c <= 'f') {
      return static_cast<unsigned char>((c - 'a') +  10);
   }
   else if (c >= 'A' && c <= 'F') {
      return static_cast<unsigned char>((c - 'A') +  10);
   }
   else {
      ETG_TRACE_FATAL(("Not a hex number. %c", c));
   }
   return 0;
}

// --------------------------------------------------------------- class Digest
Digest::Digest()
{
   for (int i = 0; i < DIGEST_LEN; i++) { x[i] = 0; }
}

Digest::Digest(const Digest& other)
{
   for (int i = 0; i < DIGEST_LEN; i++) { x[i] = other.x[i]; }
}

Digest& Digest::operator=(const Digest& other) {
   for (int i = 0; i < DIGEST_LEN; i++) { x[i] = other.x[i]; }
   return *this;
}

bool Digest::operator== (const Digest& other) const
{
   for (int i = 0; i < DIGEST_LEN; ++i) {
      if (x[i] != other.x[i]) {
         return false;
      }
   }
   return true;
}

bool Digest::operator!= (const Digest& other) const
{ return ! operator==(other); }

void Digest::print(char* hex, int len) const
{
   assert(len >= DIGEST_STR_LEN);
   for (int i = 0; i < DIGEST_LEN; ++i) {
      hex[2*i]   = hexmap[(x[i] & 0xF0) >> 4];  
      hex[2*i+1] = hexmap[ x[i] & 0x0F ];  
   }
   hex[2*DIGEST_LEN] = '\0';
}

bool Digest::read(const char* hex, int len)
{
   if (len != 2 * DIGEST_LEN) {
      ETG_TRACE_FATAL(("Provided string has false length."));
      return false;
   }

   for (int i = 0; i < DIGEST_LEN; i++) {
      x[i] =static_cast<unsigned char> ((unsigned char)(hex2uc(hex[2*i]) << 4) + hex2uc(hex[2*i+1]) );
   }
   return true;
}

// ----------------------------------------------------------- class DigestStat

DigestStat::DigestStat ()
   : _digest(), _is_digest_set(false), _is_cap_digest_set(false), _uid(0), _gid(0), _mode(0), _link_to_reg(false)
{}

void DigestStat::printDigest(char* hex, int len) const
{ _digest.print(hex, len); }

bool DigestStat::readDigest(const char* hex, int len)
{
   _is_digest_set = _digest.read(hex, len);
   return _is_digest_set;
}

void DigestStat::printCapDigest(char* hex, int len) const
{ _capDigest.print(hex, len); }

bool DigestStat::readCapDigest(const char* hex, int len)
{
   _is_cap_digest_set = _capDigest.read(hex, len);
   return _is_cap_digest_set;
}

void DigestStat::printStat (char* hex, int len) const
{
   assert(len >= STAT_STR_LEN);
   sprintf(hex, "%08x%08x%08x", _uid, _gid, _mode);
}

bool DigestStat::readStat (const char* hex, int len)
{
   if (len != 2 * STAT_LEN) {
      ETG_TRACE_FATAL(("Provided string has false length."));
      return false;
   }
   _uid = 0;
   _gid = 0;
   _mode = 0;

   int i = 0;
   for (; i < 2*UID_LEN; i++) {
      _uid <<= 4;
      _uid += hex2uc(hex[i]);
   }
   for (; i < 2*(UID_LEN + GID_LEN); i++) {
      _gid <<= 4;
      _gid += hex2uc(hex[i]);
   }
   for (; i < 2*STAT_LEN; i++) {
      _mode <<= 4;
      _mode += hex2uc(hex[i]);
   }
   return true;
}

void DigestStat::copyDigest (const DigestStat& other)
{
   for (int i = 0; i < DIGEST_LEN; ++i) {
      _digest.x[i] = other._digest.x[i];
   }
}

bool DigestStat::isRegularFile () const
{ return S_ISREG(_mode); }

bool DigestStat::isSymlink () const
{ return S_ISLNK(_mode); }

bool DigestStat::isDirectory () const
{ return S_ISDIR(_mode); }

uid_t DigestStat::uid () const
{ return _uid; }

void DigestStat::setUid (uid_t uid_)
{ _uid = uid_; }

gid_t DigestStat::gid () const
{ return _gid; }

void DigestStat::setGid(gid_t gid_)
{ _gid = gid_; }

mode_t DigestStat::mode () const
{ return _mode; }

void DigestStat::setMode(mode_t mode_)
{ _mode = mode_; }

bool DigestStat::isLinkToReg () const
{ return _link_to_reg; }

void DigestStat::setLinkToReg (bool b)
{ _link_to_reg = b; }

const Digest& DigestStat::digest () const
{ return _digest; }

void DigestStat::setDigest(const Digest& digest)
{ _digest = digest; _is_digest_set = true; }

const Digest& DigestStat::capDigest() const
{ return _capDigest; }

void DigestStat::setCapDigest(const Digest& capDigest)
{ _capDigest = capDigest; _is_cap_digest_set = true; }

bool DigestStat::isDigestSet () const
{ return _is_digest_set; }

bool DigestStat::isCapDigestSet()const
{ return _is_cap_digest_set; }
