#ifndef FCSWUPD_BXML_H
#define FCSWUPD_BXML_H

#include <vector>
#include <string>
#include <map>
#include <stddef.h>
#include "tinyxml/tinyxml.h"
#include "util/fcswupd_types.hpp"



class TiXmlElement;

namespace fcswupdate {

class XmlItemAccessIter;




TiXmlElement *getBxmlRoot(TiXmlDocument *bxmlDoc);

TiXmlElement *getOverallSection(TiXmlDocument *bxmlDoc);

TiXmlElement *getReleaseSection(TiXmlDocument *bxmlDoc);

TiXmlElement *getBoschXmlElem(TiXmlElement *parent, char const *path);
TiXmlElement *getBoschXmlElem(TiXmlElement *parent, std::string const path);
TiXmlElement *getSubModuleXmlByIndex(TiXmlElement *release, tUInt dev, tUInt mod, tUInt sub);
char const *getNodeName(TiXmlElement *parent, tU32 nodeLevel);

std::string getMediaPathFromOverallSection(TiXmlElement *overallSection);
tU16 getFcIdFromReleaseSection(TiXmlElement const *releaseSection, char const *xmlElemName="FCID");
std::string getParentName(std::string childName);

void fetchExtraArgs(TiXmlElement const *fromElem, std::map< std::string, std::string > &toMap);
std::string getExtraArgsVal(TiXmlElement const *fromElem, std::string const
 &key, std::string defaultVal);

tU32 getExtraArgsValAsUInt(TiXmlElement const *fromElem, std::string const &key, tU32 defaultVal=0);

char const *getTextFromChildRecursive(TiXmlElement *bxmlElem, std::string name, char const *defaultRes=0, 
                                      bool isReleaseXml = false);
tUInt getUIntFromChildRecursive(TiXmlElement *bxmlElem, std::string name, tU32 defaultRes = 0, bool 
                                isReleaseXml = false);
bool isTextExistsInSubModule(TiXmlElement *releaseElem, std::string refText);

bool stringToBool(std::string val);

class XmlItemAccess {
private:
XmlItemAccess():
   _enType(enType_Invalid),
   _xmlItem(0) {
}
public:
   XmlItemAccess(TiXmlElement *xmlItem);
   XmlItemAccess(XmlItemAccessIter &iter);
   XmlItemAccess(TiXmlElement *root, std::string const &longName);
   
   TiXmlElement const *getXml() const {
      return _xmlItem;
   }
   TiXmlElement *getXml() {
      return _xmlItem;
   }

   TiXmlElement *copyChildFrom(XmlItemAccess const &from, char const *name);
   TiXmlElement *copyChildFrom(TiXmlElement const *from, char const *name);

   TiXmlElement *moveChild(TiXmlElement *to);
   TiXmlElement *moveChildFrom(XmlItemAccess &from, char const *name);
   void moveChildRecur(XmlItemAccess &from, char const *name);
   void changeTextFrom(XmlItemAccess &from, char const *name, char const *to);
   void removeChildRecur();

   XmlItemAccess getRelease() {
      XmlItemAccess cur=*this;
      while (!cur.isRelease()) {
         cur=cur.getParent() ;
      }
      return cur;
   }

   typedef enum {
      enType_Root,
      enType_Release,
      enType_Device,
      enType_Module,
      enType_Submodule,
      enType_FlashItems,
      enType_FlashItem,
      enType_Last=enType_FlashItem,
      enType_Invalid
   } tenType;

   struct ItemProps {
      tenType enType;
      char const *nodeTypeTag;
      char const *nodeNameTag;
   };

   tenType enGetType() {
      return _enType;
   }

   bool isRelease() {
      return _enType==enType_Release;
   }

   bool isDevice() {
      return _enType==enType_Device;
   }

   bool isModule() {
      return _enType==enType_Module;
   }

   bool isSubModule() {
      return _enType==enType_Submodule;
   }


   const char *getTypeTag() {
      return itemPropsAr[_enType].nodeTypeTag;
   }
   const char *getChildTypeTag() {
      return itemPropsAr[_enType+1].nodeTypeTag;
   }

   const char *getNameTag() {
      return itemPropsAr[_enType].nodeNameTag;
   }
   
   char const *getName();
   std::string getLongName();
   std::string getLongName(XmlItemAccess::tenType level) {
      if ((tU32)level==(tU32)_enType) {
         return getLongName();
      }
      else if ((tU32)level<(tU32)_enType) {
         return getParent().getLongName(level);
      } 
      // error-case
      return "";
   }

   std::string getNameDisplay();
   bool isBootloader() {
      return isSubModule() && !strcmp(getName(), "BOOTLOADER");
   };
   char const * getVersion();
   
   char const * getFromVersion();
   
   tUInt getUpdateTime();

   bool isGood() {
       return _xmlItem && enGetType()!=XmlItemAccess::enType_Invalid;
   }
   bool isRoot() {
       return _xmlItem->Parent() ? false : true;
   }

   bool hasParent() {
      return _xmlItem->Parent() && !isRelease();
   }

   TiXmlElement *getParentXml() {
      return (TiXmlElement *)_xmlItem->Parent();
   }

   TiXmlElement *getSiblingXml() {
      return _xmlItem->NextSiblingElement(getTypeTag());
   }

   XmlItemAccess getNextSibling() {
      TiXmlElement *sibling=_xmlItem->NextSiblingElement(getTypeTag());
      return sibling ? XmlItemAccess(sibling) : XmlItemAccess();
   }

   XmlItemAccess getParent() {
      return XmlItemAccess(getParentXml());
   }

   std::string getParentLongName() {
      return getParent().getLongName();
   }

   XmlItemAccess getFirstChild() {
      TiXmlElement *child=getFirstChildXml();
      return child ? XmlItemAccess(child) : XmlItemAccess();
   }
   TiXmlElement * getFirstChildXml() {
      if (_enType>=enType_Last) {
         return 0;
      }
      char const *childNodeTypeTag=itemPropsAr[_enType + 1].nodeTypeTag;

      return _xmlItem->FirstChildElement(childNodeTypeTag);
   }

   TiXmlElement * getNextChildXml(TiXmlElement *prev) {
      if (_enType>=enType_Last) {
         return 0;
      }
      char const *childNodeTypeTag=itemPropsAr[XmlItemAccess(prev)._enType].nodeTypeTag;

      return prev->NextSiblingElement(childNodeTypeTag);
   }

   std::list<TiXmlElement *> getChildren() {
      std::list<TiXmlElement *> children;
      if (_enType>=enType_Last) {
         return children;
      }
      char const *childNodeTypeTag=itemPropsAr[_enType + 1].nodeTypeTag;

      for (TiXmlElement *child=_xmlItem->FirstChildElement(childNodeTypeTag);
           child;
           child=child->NextSiblingElement(childNodeTypeTag)) {
         children.push_back(child);
      }
      return children;
   }

   
   std::string getExtraArgsVal(std::string key, std::string defaultVal="0");
   bool getExtraArgsValAsBool(std::string key);
   void setExtraArg(std::string key, std::string val);
   void addExtraArgs(std::string key, std::string val);

private:
   void evalItemType();
   TiXmlElement *_xmlItem;
   tenType _enType;
   static const ItemProps itemPropsAr[];
};


   class BoschXmlNode: public XmlItemAccess {
   public:
      BoschXmlNode(TiXmlElement *boschXmlElement):
         XmlItemAccess(boschXmlElement) {
         SWU_ASSERT_RETURN(init());
      }
      BoschXmlNode(XmlItemAccess &access):
         XmlItemAccess(access.getXml()){
         SWU_ASSERT_RETURN(init());
      }
      BoschXmlNode(XmlItemAccessIter &iter):
         XmlItemAccess(iter){
         SWU_ASSERT_RETURN(init());
      }
      BoschXmlNode(TiXmlElement *root, std::string const &longName):
         XmlItemAccess(root, longName),
	 _itemsItem(0) {      //Coverity fix for 30071
         SWU_ASSERT_RETURN(init());
      }


      bool init() {
         _infoItem=getXml()->FirstChildElement("INFO");
         if (!_infoItem) {
            _infoItem=getXml()->LinkEndChild(new TiXmlElement("INFO"))->ToElement();

         }
         SWU_ASSERT_RETURN_FALSE(_infoItem);
         _itemsItem=_infoItem->FirstChildElement("ITEMS");
         if (!_itemsItem) {
            _itemsItem=_infoItem->LinkEndChild(new TiXmlElement("ITEMS"))->ToElement();
         }
         SWU_ASSERT_RETURN_FALSE(_itemsItem);
         return true;
      }

      bool clear() {
         _infoItem->RemoveChild(_itemsItem);
         getXml()->RemoveChild(_infoItem);
         _itemsItem=0;
         _infoItem=0;
         return init();
      }

      XmlItemAccess &getAccess() { return *this;}
      TiXmlElement *getInfoItem() { return _infoItem;}
      TiXmlElement *getItemsItem() { return _itemsItem;}
   
   private:
      TiXmlElement *_infoItem;
      TiXmlElement *_itemsItem;
   };



class XmlItemAccessIter {
public:
   typedef enum {
      enEnter,
      enLeave,
   } tenDirection;

   typedef enum {
      enOnEnter,
      enOnLeave,
      enOnBoth,
   } tenMode;
   XmlItemAccessIter(TiXmlElement *root, tenMode enMode, TiXmlElement *startNode=0, bool startOnEnter=true);
#if 0
   XmlItemAccessIter(TiXmlElement *serialized);
   TiXmlElement *serialize();
#endif

   virtual XmlItemAccess next();
   
   XmlItemAccess getAccess() {
      return XmlItemAccess(_current);
   }

   
   TiXmlElement *getXml() {
      return _current;
   }
   TiXmlElement *getXml() const {
      return _current;
   }
   
   bool isDone() const {
      return _done;
   }
   
   tenDirection getDirection() const {
      return _enDirection;
   }
   private:
   
   TiXmlElement *_root;
   TiXmlElement *_current;
   TiXmlElement *_next;
   tenDirection _enDirection; // direction we are taking on current node
   tenDirection _enNextDirection; // direction we are taking for next node
   bool _bNotifyOnEnter;
   bool _bNotifyOnLeave;
   bool _done;
};

   struct Scope {
      typedef enum  {
         enScope_Yes,
         enScope_No,
         enScope_NotApplicable,
         enScope_Some,
         enScope_Incompatible,
         enScope_Fcs,
         enScope_Downgrade,
         enScope_None,
         enScope_NOK
      } tenScope;

      Scope(std::string itemName_, tenScope enScope_=enScope_NotApplicable):
         itemName(itemName_), 
         enScope(enScope_), 
         hasNotApplicableChild(false) 
      {};



      tenScope enScope;
      bool hasNotApplicableChild;
      std::string itemName;
   };


template<class SCOPE>
class ScopeList {
public:
   void clear() {
      _scopes.clear();
   }
   void insert(SCOPE &scope) {
      typename std::map<std::string, SCOPE>::iterator iter=_scopes.find(scope.itemName);
      if (iter==_scopes.end()) {
         iter=_scopes.insert(std::pair< std::string, SCOPE >(scope.itemName, scope)).first;
      }
      else {
         iter->second=scope;
      }

      SCOPE *pScope=&scope;
      for (std::string parentName=getParentName(scope.itemName); parentName != ""; parentName=getParentName(parentName)) {
         SCOPE &parentScope=get(parentName);
         parentScope.addChild(*pScope);
         pScope=&parentScope;
      }
   }
   SCOPE &get(std::string name) {
      typename std::map<std::string, SCOPE>::iterator iter=_scopes.find(name);
      if (iter==_scopes.end()) {
         // create new scope:
         iter = _scopes.insert(std::pair< std::string, SCOPE >(name, SCOPE(name))).first;
      }
      return iter->second;
   }
private:
   std::map<std::string, SCOPE> _scopes;
      
};

class BXmlAccess {
private:
   BXmlAccess(): _releaseElem(0){
   }
public:
   BXmlAccess(TiXmlDocument *doc);
   BXmlAccess(TiXmlElement *elem);
   BXmlAccess(const TiXmlElement *elem);

   TiXmlElement *getModuleSection();
   TiXmlElement *getDeviceSection();
   const char *getTextFromModule(char const *childName, bool assert = true, char const *defaultRes=0);
   const char *getTextFromDevice(char const *childName, bool assert = true, char const *defaultRes=0);   

private:
   TiXmlElement *_releaseElem;

};



TiXmlElement * copyBxmlSkeleton(TiXmlElement const *fromRoot, XmlItemAccess::tenType leafLevel=XmlItemAccess::enType_Submodule);
TiXmlElement *getUXmlSection(TiXmlDocument* bxmlDoc, XmlItemAccess::tenType enType);


}

#endif
