#include "util/swu_util.hpp"
#include "util/fcswupd_types.hpp"
#include "util/fcswupd_bxml.h"
#include "util/fcswupd_trace.hpp"
#include "util/swu_xmlDoc.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_BXML
#include "trcGenProj/Header/fcswupd_bxml.cpp.trc.h"
#endif

using namespace std;

namespace fcswupdate {

TiXmlElement *getBxmlRoot(TiXmlDocument *bxmlDoc) {
   if (!bxmlDoc) {
      return 0;
   }
   return bxmlDoc->FirstChildElement();
}

TiXmlElement *getOverallSection(TiXmlDocument *bxmlDoc) {
   TiXmlElement *root=getBxmlRoot(bxmlDoc);
   if (!root) {
      return 0;
   }
   return root->FirstChildElement("OVERALL");
}


TiXmlElement *getReleaseSection(TiXmlDocument *bxmlDoc) {
   TiXmlElement *root=getBxmlRoot(bxmlDoc);
   if (!root) {
      return 0;
   }
   return root->FirstChildElement("RELEASE");
}

TiXmlElement *getUXmlSection(TiXmlDocument* bxmlDoc, XmlItemAccess::tenType enType) {
   TiXmlElement *root=getBxmlRoot(bxmlDoc);
   SWU_ASSERT_RETURN_VAL(root, 0);
   TiXmlElement *release = root->FirstChildElement("RELEASE");
   SWU_ASSERT_RETURN_VAL(release, 0);
   TiXmlElement *device = release->FirstChildElement("DEVICE");
   SWU_ASSERT_RETURN_VAL(device, 0);
   TiXmlElement *module = device->FirstChildElement("MODULE");
   SWU_ASSERT_RETURN_VAL(module, 0);

   switch(enType) {
      case XmlItemAccess::enType_Root:
         return root;
      case XmlItemAccess::enType_Release:
         return release;
      case XmlItemAccess::enType_Device:
         return device;
      case XmlItemAccess::enType_Module:
         return module;        
      default:
         return 0;
   }
}



TiXmlElement * copyBxmlSkeleton(TiXmlElement const *srcRoot, XmlItemAccess::tenType leafLevel) {
   ETG_TRACE_ERR(("copyBxmlSkeleton START, leafLevel=%u", leafLevel));
   TiXmlElement *dstRoot=0;
   TiXmlElement *srcRoot_=const_cast<TiXmlElement *>(srcRoot);
   for (XmlItemAccessIter srcIter(srcRoot_, XmlItemAccessIter::enOnEnter);!srcIter.isDone();srcIter.next()) {
      ETG_TRACE_ERR(("next iteration:"));
      XmlItemAccess srcAccess(srcIter);
      if (srcAccess.enGetType()> leafLevel) {
         continue;
      }
      ETG_TRACE_ERR(("copyBxmlSkeleton create new node typeTag=%30s nameTag=%30s name=%s", srcAccess.getTypeTag(), srcAccess.getNameTag(), srcAccess.getName()));
      TiXmlElement *newNode=new TiXmlElement(srcAccess.getTypeTag());
      swu::addTextChild(newNode, srcAccess.getNameTag(), srcAccess.getName());
      if (!dstRoot) {
         ETG_TRACE_ERR(("Create root-node"));
         dstRoot=newNode;
      }
      else {
         string longName=srcAccess.getParent().getLongName();
         ETG_TRACE_ERR(("add root-node to %s", longName.c_str()));
         XmlItemAccess dstAccess(dstRoot, longName);
         dstAccess.getXml()->LinkEndChild(newNode);
      }
   }
   ETG_TRACE_ERR(("copyBxmlSkeleton END"));

   return dstRoot;
}

TiXmlElement *getBoschXmlElem(TiXmlElement *parent, std::string const path) {
   return getBoschXmlElem(parent, path.c_str());
};


static const char *nodeTypes[]={"RELEASE", "DEVICE", "MODULE", "SUBMODULE", "FLASH_ITEMS", "FLASH_ITEM" };
static const char *nodeNameKeys[]={"", "NAME", "NAME", "TYPE", "FLASH_ITEMS", "FILENAME" };


char const *getNodeName(TiXmlElement *parent, tU32 nodeLevel) {
   const char *nodeNameKey=nodeNameKeys[nodeLevel];
   if (!strcmp(nodeNameKey, "FLASH_ITEMS")) {
      return "FLASH_ITEMS";
   }
   return swu::getTextFromChild(parent, nodeNameKeys[nodeLevel]);
}



TiXmlElement *getBoschXmlElem(TiXmlElement *parent, char const *path) {
   ETG_TRACE_USR1(("getBoschXmlElem:path=%30s, parent->Value()=%s", path, parent->Value())); 
   tUInt nodeLevel=0;
   for (;;) 
   {
      ETG_TRACE_USR2(("getBoschXmlElem:nodeType=%s ", nodeTypes[nodeLevel])); 

      if (!strcmp(parent->Value(), nodeTypes[nodeLevel])) {
         break;
      }
      ++nodeLevel;
      if (nodeLevel==4) {
         ETG_TRACE_USR1(("getBoschXmlElem:failed to find nodeLevel of parent")); 

         return 0;
      }
   }
   ETG_TRACE_USR4(("getBoschXmlElem: parents nodeLevel=%u, ", nodeLevel)); 
   nodeLevel++;
   
   char *section=(char *)alloca(strlen(path)+2);
   char *c=0;
   strcpy(section, path);
   section[strlen(path)+1]=0;
   for(c=section; *c; c++)
   {
      if(*c=='/')
      {
         *c=0;
      }
   }

   bool found = false;
   /*lint -e{690} suppress 690 for next statement/block*/
   for(c=section; parent && *c; c+=strlen(c)+1)
   {
#if 0
      ETG_TRACE_USR1(("iter nodeLevel=%u c=%10s (type=%10s name=%10s)",
                       nodeLevel, c, nodeTypes[nodeLevel], nodeNameKeys[nodeLevel]));
#endif
      found = false;
      for (parent=parent->FirstChildElement(nodeTypes[nodeLevel]); 
           parent;
           parent=parent->NextSiblingElement(nodeTypes[nodeLevel])) {
         ETG_TRACE_USR1(("check:%s", swu::getTextFromChild(parent, nodeNameKeys[nodeLevel])));
         if(!strcmp(getNodeName(parent, nodeLevel), c)) {
            found=true;
            break;
         }
      }
      if (!found) {
         ETG_TRACE_USR1(("getBoschXmlElem:path(%s) not found !!!", path));
         return 0;
      }
      nodeLevel++;
   }

   return parent;
}

TiXmlElement *getSubModuleXmlByIndex(TiXmlElement *release, tUInt dev, tUInt mod, tUInt sub) {
   ETG_TRACE_USR4(("getSubModuleXmlByIndex(%u:%u:%u)", dev, mod, sub));
   TiXmlElement *node=release;
   tUInt indexAr[]={dev, mod, sub};
   for (tUInt nodeLevel=1; nodeLevel<4;++nodeLevel) {
      ETG_TRACE_USR4(("getSubModuleXmlByIndex:nodeLevel=%u type=%20s name=%s", 
                      nodeLevel, nodeTypes[nodeLevel],
                      swu::getTextFromChildOrEmpty(node, nodeNameKeys[nodeLevel])));
      node=node->FirstChildElement(nodeTypes[nodeLevel]);
      if (!node) {
         ETG_TRACE_USR1(("getSubModuleXmlByIndex:first not found, nodeLevel=%u", nodeLevel));
         return 0;
      };
      tUInt imax = indexAr[nodeLevel-1]; 
      for (tUInt i=0;i<imax;++i) {
         ETG_TRACE_USR4(("getSubModuleXmlByIndex:next"));
         node=node->NextSiblingElement(nodeTypes[nodeLevel]);
         if (!node) {
            ETG_TRACE_USR1(("getSubModuleXmlByIndex:next not found nodeLevel=%u", nodeLevel));
            return 0;
         } else {
            ETG_TRACE_USR4(("getSubModuleXmlByIndex: next found: nodeLevel=%u type=%20s name=%s", 
                            nodeLevel, nodeTypes[nodeLevel],
                            getNodeName(node, nodeLevel)));

         }
      }
   }
   return node;
}
  
std::string getMediaPathFromOverallSection(TiXmlElement *overallSection) {
   string mediaPath;
   SWU_ASSERT_RETURN_VAL(overallSection, mediaPath);
   const char* media_path = swu::getTextFromChild(overallSection, "MEDIUM_DRIVE", false);
   SWU_ASSERT_RETURN_VAL(media_path, media_path);
   mediaPath = media_path;
   SWU_ASSERT_RETURN_VAL(mediaPath.size(), mediaPath);
   ETG_TRACE_USR1(("Found Media Drive : %s", mediaPath.c_str()));
   if (mediaPath[mediaPath.length() - 1] != '/') {
      mediaPath += "/";
   }

   const char* base_path = swu::getTextFromChild(overallSection, "BASEPATH", false);
   SWU_ASSERT_RETURN_VAL(base_path, mediaPath);
   ETG_TRACE_USR1(("Found Base Path : %s", base_path));
   std::string basepath = base_path;
   if ((basepath.size() > 0) && (basepath != ".")) {
      mediaPath += basepath;
      if (mediaPath[mediaPath.length() - 1] != '/') {
         mediaPath += "/";
      }
   }

   ETG_TRACE_USR1(("Mediapath: %s", mediaPath.c_str()));
   return mediaPath;
}

/* return fcid from release section, 0 is regarded as invalid */
tU16 getFcIdFromReleaseSection(TiXmlElement const *releaseSection, char const *xmlElemName) {

   BXmlAccess access(releaseSection);
   const char *fcidHex = access.getTextFromModule(xmlElemName, false, "0");   

   ETG_TRACE_USR1(("%15s(Hex)=%s", xmlElemName, fcidHex));
   tU32 newFcId = 0;
   if (1 == sscanf(fcidHex, "%x", &newFcId)) {
      ETG_TRACE_USR1(("newFcId=%u", newFcId));
   }
   return (static_cast<tU16> (newFcId));
}


string getParentName(string childName) {
   if (childName.find("/")==string::npos) {
      return "";
   }
   return childName.substr(0,childName.find_last_of("/"));
}

void fetchExtraArgs(TiXmlElement const *fromElem, std::map< string, string > &toMap)
{
   for (TiXmlElement const *curElem = fromElem->FirstChildElement("EXTRA_ARG"); curElem != 0; 
        curElem = curElem->NextSiblingElement("EXTRA_ARG"))
   {
      string key = swu::getTextFromChild(curElem, "KEY", false, "");
      string value = swu::getTextFromChild(curElem, "VALUE", false, "");
      ETG_TRACE_USR1(("EXTRA_ARG[%30s]=%s", key.c_str(), value.c_str()));
      toMap[key] = value;
   }
}

std::string  getExtraArgsVal(TiXmlElement const *fromElem, std::string const &key, std::string defaultVal)
{
   bool found = false;
   std::string val = "";

   for (TiXmlElement const *curElem = fromElem->FirstChildElement("EXTRA_ARG"); curElem != 0; 
        curElem = curElem->NextSiblingElement("EXTRA_ARG"))
   {
      if (swu::getTextFromChild(curElem, "KEY", false, "") == key)
      {
         val = swu::getTextFromChild(curElem, "VALUE", false, "");
         found = true;
         break;
      }
   }

   ETG_TRACE_USR1(("getExtraArgsVal found=%u key=%30s val=%s", found, key.c_str(), val.c_str()));
   return found ? val : defaultVal;
}

tU32 getExtraArgsValAsUInt(TiXmlElement const *fromElem, std::string const &key, tU32 defaultVal)
{
   bool found = false;
   tU32 val = 0;

   for (TiXmlElement const *curElem = fromElem->FirstChildElement("EXTRA_ARG"); curElem != 0; 
        curElem = curElem->NextSiblingElement("EXTRA_ARG"))
   {
      if (swu::getTextFromChild(curElem, "KEY", false, "") == key)
      {
         val = (tU32)atoi(swu::getTextFromChild(curElem, "VALUE", false, "0"));
         found = true;
         break;
      }
   }

   ETG_TRACE_USR1(("getExtraArgsVal found=%u key=%30s val=%u", found, key.c_str(), val));
   return found ? val : defaultVal;
}

char const *getTextFromChildRecursive(TiXmlElement *bxmlElem, std::string name, char const *defaultRes, bool isReleaseXml) {
     
   if (!bxmlElem ) {      
      return defaultRes;
   }
   
   if (swu::getTextFromChildAsBool(bxmlElem, name.c_str())) {
      return swu::getTextFromChild(bxmlElem, name.c_str());
   }   

   XmlItemAccess access = XmlItemAccess(bxmlElem); 
   if(access.isGood()) {
      ETG_TRACE_FATAL(("enType:%d", access.enGetType()));
      if (access.enGetType() > XmlItemAccess::enType_Release) {
         return getTextFromChildRecursive(bxmlElem->Parent()->ToElement(), name, defaultRes, isReleaseXml);
      }
      if (access.enGetType() == XmlItemAccess::enType_Release && !isReleaseXml) { 
         TiXmlElement *root =  bxmlElem->Parent()->ToElement();
         if(root) {
            return getTextFromChildRecursive(root->FirstChildElement("OVERALL"), name, defaultRes, isReleaseXml);
         }      
      }    
   }

   return defaultRes;

}

tUInt getUIntFromChildRecursive(TiXmlElement *bxmlElem, std::string name, tU32 defaultRes, bool isReleaseXml) { 
   
   const char* txt = getTextFromChildRecursive(bxmlElem, name, 0, isReleaseXml);
   if (!txt) {      
      return defaultRes;
   }      
   tUInt res=atoi(txt);
   return res;
}

bool isTextExistsInSubModule(TiXmlElement *releaseElem, std::string refText) {
   if(!releaseElem) {
      return false;
   }

   for (XmlItemAccessIter iter(releaseElem, XmlItemAccessIter::enOnLeave); !iter.isDone();iter.next()) {
      BoschXmlNode access(iter);
      if(access.enGetType()==XmlItemAccess::enType_Submodule) {
         std::string text = access.getName();
         if(text == refText) {
            return true;
         }
      }
   }
   return false;
}


   const XmlItemAccess::ItemProps XmlItemAccess::itemPropsAr[]= {
      {XmlItemAccess::enType_Root,       "UPDATEXML",    ""},
      {XmlItemAccess::enType_Release,    "RELEASE",     "NAME"},
      {XmlItemAccess::enType_Device,     "DEVICE",      "NAME"},
      {XmlItemAccess::enType_Module,     "MODULE",      "NAME"},
      {XmlItemAccess::enType_Submodule,  "SUBMODULE",   "TYPE"},
      {XmlItemAccess::enType_FlashItems, "FLASH_ITEMS", "FLASH_ITEMS"},
      {XmlItemAccess::enType_FlashItem,  "ITEM",  "FILENAME"},
      {XmlItemAccess::enType_Invalid,    "INVALID",     ""}
   };

void XmlItemAccess::evalItemType() {
   _enType=enType_Invalid;
    for (int i=0; i<enType_Invalid; ++i) {
      if (!strcmp(_xmlItem->Value(), itemPropsAr[i].nodeTypeTag)) {
           _enType=(tenType)i;
          break;
      }
   }

}
XmlItemAccess::XmlItemAccess(TiXmlElement *root, std::string const &longName) {
    _xmlItem=getBoschXmlElem(root, longName);
    evalItemType();
}

XmlItemAccess::XmlItemAccess(XmlItemAccessIter &iter) {
   _xmlItem=iter.getXml();
    evalItemType();
}

XmlItemAccess::XmlItemAccess(TiXmlElement *xmlItem):
      _xmlItem(xmlItem)
{
   ETG_TRACE_USR4(("XmlItemAccess(xmlItem=%p) start", _xmlItem));
#if 0
      ETG_TRACE_USR4(("XmlItemAccess: Value=%20s text=%s", 
                       _xmlItem->Value(), _xmlItem->GetText()));
#endif      
      evalItemType();      
#if 1
      ETG_TRACE_USR4(("XmlItemAccess: Value=%20s-> type=%u", 
                       _xmlItem->Value(), _enType));
#endif
   ETG_TRACE_USR4(("XmlItemAccess(xmlItem) end"));
      // eval type
   }

TiXmlElement *XmlItemAccess::copyChildFrom(XmlItemAccess const &from, char const *name) {
   SWU_ASSERT_RETURN_VAL(name, 0);
   TiXmlElement const *fromChild=from.getXml()->FirstChildElement(name);
   if (!fromChild) {
      return 0;
   }
   return (getXml()->InsertEndChild(*fromChild))->ToElement();
}

TiXmlElement *XmlItemAccess::copyChildFrom(TiXmlElement const *from, char const *name) {
   SWU_ASSERT_RETURN_VAL(from, 0);
   SWU_ASSERT_RETURN_VAL(name, 0);
   TiXmlElement const *fromChild=from->FirstChildElement(name);
   if (!fromChild) {
      return 0;
   }
   return (getXml()->InsertEndChild(*fromChild))->ToElement();
}

TiXmlElement *XmlItemAccess::moveChild(TiXmlElement *to) {
   return swu::moveElem(getXml(), to);
}

TiXmlElement *XmlItemAccess::moveChildFrom(XmlItemAccess &from, char const *name) {
   TiXmlElement *fromChild = from.getXml()->FirstChildElement(name);
   if (!fromChild) {
      return 0;
   }
   return swu::moveElem(fromChild, getXml());
}

void XmlItemAccess::moveChildRecur(XmlItemAccess &from, char const *name) {
   TiXmlElement *fromChild = from.getXml()->FirstChildElement(name);
   if (!fromChild) {
      return;
   }	
   TiXmlElement* elem = getXml()->NextSiblingElement(getTypeTag());
   if (!elem){
      swu::moveElem(fromChild, getXml());
      return;
   }
   else {
      getXml()->InsertEndChild(*fromChild);
   }

   XmlItemAccess access(elem);
   access.moveChildRecur(from, name);
}

void XmlItemAccess::changeTextFrom(XmlItemAccess &from, char const *name, char const *to){
   TiXmlElement *fromChild = from.getXml()->FirstChildElement(name);
   if (!fromChild) {
      return;
   }
   swu::setTextChild(getXml(), to, fromChild->GetText());
}

void XmlItemAccess::removeChildRecur() {
   char const *childNodeTypeTag = itemPropsAr[_enType + 1].nodeTypeTag;
   bool isFirstChild = true;
	
   for (TiXmlElement *moduleXml = getXml()->FirstChildElement(childNodeTypeTag); moduleXml;) {		
      TiXmlElement *tmp = moduleXml;
      moduleXml = moduleXml->NextSiblingElement(childNodeTypeTag);
      if (tmp && !isFirstChild) {
         getXml()->RemoveChild(tmp);
      }
      else {
         isFirstChild = false;
      }
   }
}


char const *XmlItemAccess::getName() {
   char const *name=swu::getTextFromChild(_xmlItem, getNameTag(), false);
   if (!name) {
      name = "";
   }
   ETG_TRACE_USR4(("XmlItemAccess::getName:%s", name));
   return name;
};

string XmlItemAccess::getLongName() {
   string longName;
   if (_enType == enType_Root) {
      longName = "";
   }
   else if (_enType == enType_Release) {
      longName = "";
   }
   else if (_enType == enType_Device) {
      longName = getName();
   }
   else {
      longName = getParent().getLongName() + "/" + getName();
   }
   //ETG_TRACE_USR4(("XmlItemAccess::getLongName:%s", longName.c_str()));
   return longName;
};

string XmlItemAccess::getNameDisplay() {
   char const *name=swu::getTextFromChild(_xmlItem, "NAME_DISPLAY", false);
   if (!name || name[0]==0 || !strcmp(name, "unknown")) {
      name=getName();
   }
   if (!name || name[0]==0) {
      name="unknown";
   }
   ETG_TRACE_USR4(("XmlItemAccess::getNameDisplay:%s", name));
   return name;
};

   char const * XmlItemAccess::getVersion() {
      //char const *version=getTextFromChild(_xmlItem, "VERSION", false, "0");
      char const *version=swu::getTextFromChild(_xmlItem, "VERSION", false, "0");
      ETG_TRACE_USR4(("XmlItemAccess::getVersion:%s", version));

      return version;
   };
   
    tUInt XmlItemAccess::getUpdateTime() {
      tUInt updateTime=swu::getUIntFromChild(_xmlItem, "UPDATE_TIME");
      ETG_TRACE_USR4(("XmlItemAccess::getUpdateTime:%u", updateTime));
      return updateTime;
   };

   char const * XmlItemAccess::getFromVersion() {
     char const *version=swu::getTextFromChild(_xmlItem, "FROMVERSION", false, "0");
     ETG_TRACE_USR4(("XmlItemAccess::getVersion:%s", version));
     return version;
   };



std::string XmlItemAccess::getExtraArgsVal(std::string key, std::string defaultVal) {
   // take default from caller
   std::string val=defaultVal;
   // get val from parent
   if (_enType >= enType_Release && _enType <= enType_Submodule) {
      val=getParent().getExtraArgsVal(key, val);
   }
   // special-case release: check for overall-section
   if (_enType == enType_Release) {
      TiXmlElement *parentXml=getParentXml();
      if (parentXml) {
         TiXmlElement *overallXml=parentXml->FirstChildElement("OVERALL");
         if (overallXml) {
            val=fcswupdate::getExtraArgsVal(overallXml, key, val);
         }
      }
   }
   // add own extra-args
   val=fcswupdate::getExtraArgsVal(getXml(), key, val);
   return val;
}

bool XmlItemAccess::getExtraArgsValAsBool(std::string key) {
   return swu::stringToBool(getExtraArgsVal(key));
}

void XmlItemAccess::setExtraArg(std::string key, std::string val) {
   TiXmlElement *me=getXml();
   for (TiXmlElement *curElem = me->FirstChildElement("EXTRA_ARG"); curElem != 0; )
   {
      TiXmlElement *tmp=curElem;
      curElem = curElem->NextSiblingElement("EXTRA_ARG");
      if (swu::getTextFromChild(curElem, "KEY", false, "") == key)
      {
         me->RemoveChild(tmp);
      }
   }

   TiXmlElement *extraArg = new TiXmlElement("EXTRA_ARG");
   swu::addTextChild(extraArg, "KEY", key);
   swu::addTextChild(extraArg, "VALUE", val);
   me->LinkEndChild(extraArg);
}

void XmlItemAccess::addExtraArgs(std::string key, std::string val) {

   bool isnewnode = false;
   TiXmlElement *me=getXml();
   TiXmlElement *extraArgs = me->FirstChildElement("EXTRA_ARGS");
   if(extraArgs == 0) {
      extraArgs = new TiXmlElement("EXTRA_ARGS");
      isnewnode = true;
   }

   //todo:need to customize.
   TiXmlElement *extraArg = new TiXmlElement("EXTRA_ARG");
   swu::addTextChild(extraArg, "KEY", key);
   swu::addTextChild(extraArg, "VALUE", val);
   extraArgs->LinkEndChild(extraArg);

   if(isnewnode) {
      me->LinkEndChild(extraArgs);
   }
}



XmlItemAccessIter::XmlItemAccessIter(TiXmlElement *root, tenMode enMode, TiXmlElement *startNode, bool startOnEnter):
   _root(root),
   _current(root),
   _next(root),
   _enDirection(enEnter),
   _enNextDirection(enEnter),
   _bNotifyOnEnter(true),
   _bNotifyOnLeave(true),
   _done(false) {
   _next=XmlItemAccess(_current).getFirstChildXml();
   if (enMode==enOnEnter) {
      _bNotifyOnLeave=false;
   }  else if (enMode==enOnLeave) {
      _bNotifyOnEnter=false;
   }
   
   if (!_next) {
      // no child-nodes
      _next=_root;
      _enNextDirection=enLeave;
   }

   if (startNode) {
      tUInt numHits=1;
      if (enMode==enOnBoth) {
         if (!startOnEnter) {
            numHits=2;
         }
      }
      while (!_done) {
          if (_current==startNode) {
             --numHits;
          }
          if (!numHits) {
             break;
          }
          next();
       }
     
   }

   if (!_bNotifyOnEnter) {
      while (_enDirection == enEnter && !_done) {
         next();
      }
      if (_done) {
         // special case, no children
         _done=false;
      }
   }
 
   XmlItemAccess curAcc(_current);
   if(curAcc.isGood()) {
      string strName = curAcc.getLongName(); //Coverity fix for 209536
      ETG_TRACE_USR4(("XmlItemAccessIter::CTOR  _enDirection=%u _bNotifyOnEnter=%u _bNotifyOnLeave=%u"
                      "curAcc.getLongName()=%10s curAcc.getTypeTag()=%10s curAcc.enGetType=%u", 
                      _enDirection, _bNotifyOnEnter, _bNotifyOnLeave, strName.c_str(),
                      curAcc.getTypeTag(), curAcc.enGetType()));
   }
}


XmlItemAccess XmlItemAccessIter::next() {
   XmlItemAccess curAcc(_current);
   if(curAcc.isGood()) {
      string strName = curAcc.getLongName();     //Coverity fix for 209501
      ETG_TRACE_USR4(("XmlItemAccessIter::next start  _enDirection=%u _bNotifyOnEnter=%u _bNotifyOnLeave=%u"
                      "curAcc.getLongName()=%10s curAcc.getTypeTag()=%10s curAcc.enGetType=%u", 
                      _enDirection, _bNotifyOnEnter, _bNotifyOnLeave, strName.c_str(),
                      curAcc.getTypeTag(), curAcc.enGetType()));
   }

   ETG_TRACE_USR4(("XmlItemAccessIter::next(): _current=%p _root=%p _next=%p", _current, _root, _next));
   _enDirection=_enNextDirection;
   if (_current == _root && _enDirection==enLeave) {
      ETG_TRACE_USR4(("XmlItemAccessIter::next(): setting done"));
         _done=true;
         _next=_root;
         return XmlItemAccess(_root);
   } else {
      _current=_next;
   }
   if (_done) {
      // end-state
      _enDirection=enLeave;
      return XmlItemAccess(_root);
   }

   switch (_enDirection) {
      case enEnter:
      {
         // find the next node
         TiXmlElement *firstChild=XmlItemAccess(_current).getFirstChildXml();
         if (firstChild) {
            /* next step will be enter this child */
            _next=firstChild;
            _enNextDirection=enEnter;
         }
         else {
            // next step will be to leave this leaf-node
            _next=_current;
            _enNextDirection=enLeave;
         }

         // decide if we have to return this node ...
         if (_bNotifyOnEnter) {
            ETG_TRACE_USR4(("XmlItemAccessIter::next: found enEnter"));

            return _current;
         }
         // .. or continue with the next one
         ETG_TRACE_USR4(("XmlItemAccessIter::next: recurse enEnter"));
         return next(); 
      }
      case enLeave: 
      {

         // find the next node
         TiXmlElement *parent=XmlItemAccess(_current).getParentXml();
         if (!parent) {
            _next=_root;
            _enNextDirection=enLeave;
         }
         else {
            TiXmlElement *sibling=XmlItemAccess(parent).getNextChildXml(_current);
            if (sibling) {
               /* next step will be enter our sibling */
               _next=sibling;
               _enNextDirection=enEnter;
            }
            else {
               // next step will be to leave this leaf-node
               _next=parent;
               _enNextDirection=enLeave;
            }
         }
         
         // decide if we have to return this node ...
         if (_bNotifyOnLeave) {
            ETG_TRACE_USR4(("XmlItemAccessIter::next: found enLeave"));
            return _current;
         }
         // .. or continue with the next one
         ETG_TRACE_USR4(("XmlItemAccessIter::next: recurse enLeave"));
         return next(); 
      }
      default:
      {
         ETG_TRACE_ERR(("XmlItemAccessIter::next: recurse default"));
         return _current; 
      }
   }
}

BXmlAccess::BXmlAccess(TiXmlElement *elem){
   _releaseElem = elem;
}

BXmlAccess::BXmlAccess(const TiXmlElement *elem){
   _releaseElem = (TiXmlElement*)elem;
}


BXmlAccess::BXmlAccess(TiXmlDocument *doc){
   TiXmlElement* root = doc->FirstChildElement();
   _releaseElem = root->FirstChildElement("RELEASE");
}


const char* BXmlAccess::getTextFromModule(char const *childName, bool assert, char const *defaultRes) {

   if(!defaultRes) {
      defaultRes = "";
   }
   TiXmlElement *device = _releaseElem->FirstChildElement("DEVICE");
   if(assert) {
      SWU_ASSERT_RETURN_VAL(device, defaultRes);
   }
   TiXmlElement *module = device->FirstChildElement("MODULE");
   if(assert) {
      SWU_ASSERT_RETURN_VAL(module, defaultRes);
   }

   const char *text = swu::getTextFromChild(module, childName, assert, defaultRes);
   if(text) {
      return text;
   } else {
      return defaultRes;
   }
}

const char* BXmlAccess::getTextFromDevice(char const *childName, bool assert, char const *defaultRes) {

   if(!defaultRes) {
      defaultRes = "";
   }
   TiXmlElement *device = _releaseElem->FirstChildElement("DEVICE");
   if(assert) {
      SWU_ASSERT_RETURN_VAL(device, defaultRes);
   }
 
   const char *text = swu::getTextFromChild(device, childName, assert, defaultRes);
   if(text) {
      return text;
   } else {
      return defaultRes;
   }
}

TiXmlElement *BXmlAccess::getModuleSection() {
   TiXmlElement *device = _releaseElem->FirstChildElement("DEVICE");
   SWU_ASSERT_RETURN_VAL(device, 0);
   TiXmlElement *module = device->FirstChildElement("MODULE");
   SWU_ASSERT_RETURN_VAL(module, 0);
   return module;
}

TiXmlElement *BXmlAccess::getDeviceSection() {
   TiXmlElement *device = _releaseElem->FirstChildElement("DEVICE");
   SWU_ASSERT_RETURN_VAL(device, 0);   
   return device;
}



}

