/*
 * fcswupd_parserCMS.h
 *
 *  Created on: Jul 15, 2015
 *      Author: sof4hi
 */

#ifndef FCSWUPD_PARSERCMS_H_
#define FCSWUPD_PARSERCMS_H_

#include "main/fcswupd_parserIf.h"
#include "util/swu_caCert.h"
#include "util/swu_targetKey.h"
#include "util/swu_uin.hpp"

class TiXmlDocument;

namespace fcswupdate {

/**
 * @brief A parser for Cryptographic Message Syntax files used in the updating process.
 *
 * Which parser is chosen is selected in fcswup_main.cpp FcSwUpdCore::getReleaseList
 */
class CMSParser : public ParserIf {

private:
   CMSParser & operator=(const CMSParser &);
   CMSParser(const CMSParser &);
public:
	/**
	 * @brief Default C'tor.
	 *
	 * Tries to get a valid certificate from NOR memory, see swu::NORCaCert::instance().
	 * If the request for a certificate fails, an error code is set which can be read with CMSParser::getParserError().
	 *
	 * @todo There should also be a constructor which gets a certificate from a Secure Data Container.
	 *
	 */
	CMSParser ();

   virtual ~CMSParser (); 

   /** @brief Looks for CMS files in the given path, parses them, extracts and analyzes the payload and compiles a list
    * of <overallSections> from bosch.xml files which contain the meta-data for the current update.
    *
    * The CMS files which are found are verified against the target's local certificate found in NOR memory. bosch.xml
    * files are verified against their hashes.
    *
    * @todo Currently, only the first bosch.xml that is found is returned.
    *
    * @param sourceInfo   Info including Path in which CMS files are looked for.
    * @param errorCode  Error state of the CMSParser.
    * @return           A list of bosch.xml
    */
   virtual std::list<TiXmlElement> getReleaseList(trSourceInfo const &sourceInfo, tenSwUpdateError *errorCode);

   /** @brief Set the active meta-info for the update. */
   virtual bool setOverallSection(TiXmlElement const &overallSection);

   /** @brief Returns the meta-info for release with given id.
    *
    * @param id
    * @return
    */
   virtual TiXmlDocument *getReleaseXml(size_t id=0);

   /** @brief Returns the current error state of the parser. */
   virtual tenSwUpdateError getParserError(void) const;

   virtual bool configure(const ParserConfig& parserConfig);

private:

   TiXmlDocument*      _document;
   tenSwUpdateError    _errorCode;
   swu::CACertIf*      _caCert;
   swu::CTargetKeyIf*  _targetKey;
   swu::UinIf*         _uin;
   
   /** @brief Helper method - parses a CMS file.
    *
    * The method will load the file in the given path, verify its signature and extract its payload. The payload
    * contains location and digest of the bosch.xml file for this update. The bosch.xml file is then also verified and
    * parsed. Finally a copy of the bosch.xml file is created, to which additional information is added, i.e.,
    * location of the CMS file, encryption parameters and CKSUM nodes. The content of the copied bosch.xml is also
    * contained in the given TiXmlDocument* parameter.
    *
    * @param path Path to the CMS file.
    * @param file Name of the CMS file to be parsed.
    * @param doc  Contains a copy of the bosch.xml described in the CMS file with additional nodes.
    * @return True, if CMS file and bosch.xml are parsed successfully, False otherwise.
    */
   tenSwUpdateError parseCMSFile(trSourceInfo const &sourceInfo, TiXmlDocument* doc);
   
   /** @brief Helper method - returns the suffix of a CMS file. */
   static const ::std::string CMS_SUFFIX() { return "cms"; }
};

}



#endif /* FCSWUPD_PARSERCMS_H_ */
