/**
 * @file      fcswupd_parserCMS.cpp
 * @author    Philipp Blanke (blp4hi) <philipp.blanke@de.bosch.com>
 * @date      Tue 09 Aug 2016, 13:37
 * @copyright Robert Bosch Car Multimedia GmbH
 */
#include <sys/stat.h>
#include <dirent.h>
#include <openssl/err.h>

#include "config/fcswupd_config.hpp"
#include "tinyxml/tinyxml.h"
#include "util/swu_certificate.h"
#include "util/swu_cms.h"
#include "util/swu_cryptoFactory.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_securityEngine.h"
#include "util/swu_types.h"
#include "util/swu_filesystem.h"
#include "util/swu_util.hpp"
#include "util/swu_execCommand.h"
#include "util/swu_globallog.h"
#include "util/fcswupd_parserUtil.h"
#include "main/fcswupd_propDevMgr.h"
#include "main/fcswupd_releaseFilterIf.h"
#include "main/fcswupd_xmlFilterIf.h"

#include "fcswupd_parserCMS.h"

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_PARSER
#include "trcGenProj/Header/fcswupd_parserCMS.cpp.trc.h"
#endif

using namespace std;

namespace fcswupdate {


CMSParser::CMSParser()
   : _document(0), _errorCode(tenSwUpdateError_ECU_TO_BE_UPDATED), _caCert(0), _targetKey(0), _uin(0)
{
   // Initialize OpenSSL 
   OpenSSL_add_all_algorithms();
   ERR_load_BIO_strings();
   ERR_load_crypto_strings();
}

bool CMSParser::configure(const ParserConfig& parserConfig)
{
   _config=parserConfig;

   Config* cfg = Config::instance();
   swu::CryptoFactory cf(static_cast< swu::tenCertificateAndKeySource >(
                                                                        cfg->cfg_CertificateAndKeySource.get()));

   _caCert = cf.createCaCert();
   if( not _caCert->load()) {
      if (_caCert->isEmpty()) {
         // if there is an empty cert and strict checking is not wanted,
         // CMS will skip the check, see CMS::validate()
         ETG_TRACE_ERR(("Empty CA certificate in target."));
      }
      else {
         ETG_TRACE_ERR(("Corrupted CA certificate in target, can't load it."));
         _errorCode = tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
      }
   }

   if (_config.useEncryption) {
      // if the private key and UIN can actually be loaded is checked 
      // when they are used (lazy loading), see parseCMSFile().
      _targetKey = cf.createTargetKey();
      _uin = cf.createUin();
   }
   return true;
}

CMSParser::~CMSParser()
{
   EVP_cleanup();
   if (_document) {
      delete _document;
      _document = 0;
   }
   _caCert=0;
   _uin=0;
}

#define SET_PARSER_ERROR(ERR)                            \
   if (errorCode) { *errorCode=ERR; }                     \
   _errorCode=ERR;                                        \

#define SET_PARSER_ERROR_RETURN(ERR)                           \
   SET_PARSER_ERROR(ERR)                                       \
   _errorCode=ERR;                                              \   
   if (_errorCode != tenSwUpdateError_OK) {                     \
      swu::LOG_FAILURE("error:%u", _errorCode);                  \
      return result;                                           \
   }                                                          \

#define SET_PARSER_ERROR_CONT(ERR)                      \
   SET_PARSER_ERROR(ERR)                                \
   _errorCode=ERR;                                      \   
   if (_errorCode != tenSwUpdateError_OK) {            \
      swu::LOG_FAILURE("found no appropriate release:skiperror:%u", _errorCode);    \
      continue;                                         \
   }                                                    \


std::list<TiXmlElement> CMSParser::getReleaseList(trSourceInfo const &sourceInfo, tenSwUpdateError *errorCode)
{
   std::string relDir=sourceInfo.path;
   std::string infoFile=sourceInfo.infoFile;
   tenSwUpdateError errCode;
   std::list<TiXmlElement> result;
   ETG_TRACE_USR1(("CMSParser::getReleaseList %s", relDir.c_str()));

   SET_PARSER_ERROR_RETURN(tenSwUpdateError_OK);

   // The CA Cert has already been loaded in the constructor, but we had no
   // chance to return there with an error. So we check now if an error occured.
   if(tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED == _errorCode) {
      ETG_TRACE_ERR(("CMSParser::getReleaseList: Corrupted CA certificate, no way to verify cms files"));
      SET_PARSER_ERROR_RETURN(tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED);
   }

   if (! relDir.c_str()) {
      ETG_TRACE_ERR(("No path given. Stop parsing."));
      SET_PARSER_ERROR_RETURN(tenSwUpdateError_ERROR_METAINFO_NOT_FOUND);
   }

   else if (!swu::isDirectory(relDir)) {
      ETG_TRACE_ERR(("Not a directory:%s", relDir.c_str()));
      SET_PARSER_ERROR_RETURN(tenSwUpdateError_ERROR_METAINFO_NOT_FOUND);
   }
   else if (!infoFile.empty() && !swu::isFile(relDir + "/" + infoFile)) {
      ETG_TRACE_ERR(("File %30s does not exist in dir:%s", infoFile.c_str(), relDir.c_str()));
      SET_PARSER_ERROR_RETURN(tenSwUpdateError_ERROR_METAINFO_NOT_FOUND);
   }

   ETG_TRACE_USR1(("Parsing path '%s'", relDir.c_str()));
   if (!infoFile.empty()) {
      ETG_TRACE_USR1(("looking only for file '%s'", infoFile.c_str()));
      
   }


   bool isCMSFileFound = false;
   struct dirent *dent=0;
   DIR *dir=opendir(relDir.c_str());
   if (dir) {
      while (0 != (dent = readdir(dir))) {
         char* filename = dent->d_name;
         
         // look for files with suffix "cms"
         if( ! swu::hasSuffix(filename, CMS_SUFFIX()) ) {
            continue;
         }
         if (!infoFile.empty() && !swu::isSameFile(relDir + "/" + infoFile, relDir + "/" + filename)) {
            continue;
         }
         
         string fileNameAndPath = relDir + "/" + filename;
         if(!swu::isFile(fileNameAndPath)) {
            ETG_TRACE_ERR(("ERR stat S_ISREG fileNameAndPath=%s ", fileNameAndPath.c_str()));
            SET_PARSER_ERROR_CONT(tenSwUpdateError_ERROR_METAINFO_READ);
         }
         isCMSFileFound = true;
         swu::ScopedPointer<TiXmlDocument> scopedDoc(new TiXmlDocument);
         TiXmlDocument *doc = scopedDoc.getPtr();
         trSourceInfo sourceInfo1=sourceInfo;
         sourceInfo1.infoFile=filename;
         errCode = parseCMSFile(sourceInfo1,  doc);
         SET_PARSER_ERROR_CONT(errCode);
         
         ETG_TRACE_USR3(("CMSParser::getReleaseList: Parsed %s cms File", fileNameAndPath.c_str()));
         TiXmlElement *root = doc->FirstChildElement();
         if (!root) {
            ETG_TRACE_ERR(("ERR No root element."));
            SET_PARSER_ERROR_CONT(tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
         }
         TiXmlElement *overall=root->FirstChildElement("OVERALL");
         if(!overall) {
            ETG_TRACE_ERR(("ERR No OVERALL element."));
            SET_PARSER_ERROR_CONT(tenSwUpdateError_ERROR_METAINFO_SECTION_NOT_FOUND);
         }

         ReleaseFilterContext context;
         context.sourceInfo=sourceInfo1;
         errCode =_postFilters->apply(doc, context);
         
         if (errCode != tenSwUpdateError_OK) {
            ETG_TRACE_ERR(("ERR !postFilters, error-code=%u", _errorCode ));
            SET_PARSER_ERROR_CONT(errCode);
         }

         result.push_back(*((TiXmlElement *)overall->Clone()));
      }
      closedir (dir);
   }
   return result;
}


tenSwUpdateError CMSParser::parseCMSFile(trSourceInfo const &sourceInfo, TiXmlDocument *doc)
{
   std::string const path=sourceInfo.path;
   std::string const infoFile=sourceInfo.infoFile;
   tenSourceType sourceType=sourceInfo.enSourceType;
   tenSwUpdateError errorCode = tenSwUpdateError_OK;
   std::string cmsData;
   std::string cmsFileAndName = sourceInfo.getInfoFilePathAndName();
   ETG_TRACE_USR4(("Parsing file %s", cmsFileAndName.c_str()));
   if( not swu::loadFile(cmsFileAndName, cmsData)) {
      ETG_TRACE_ERR(("CMSParser::parseCMSFile: can't read cms-file %s", cmsFileAndName.c_str()));
      return tenSwUpdateError_ERROR_METAINFO_READ;
   }

   Config* cfg = Config::instance();
   // load configuration for certificate verification
   bool allowEmptyCert = (bool)(cfg->cfg_CmsParserAllowEmptyCACert.get());   
   // load last authentic time
   std::string lastAuthenticTimeFormat = (std::string)(cfg->cfg_LastAuthenticTimeFormat.get());
   std::string lastAuthenticTime = (std::string)(cfg->cfg_LastAuthenticTime.get());
   time_t authenticTime = swu::parseTimeString(lastAuthenticTime, lastAuthenticTimeFormat);
   ETG_TRACE_USR1(("Last authentic time: %s", ctime(&authenticTime) ));

   swu::CMS cms(_caCert, authenticTime);

   if( not cms.parse(cmsFileAndName, allowEmptyCert)) {
      ETG_TRACE_ERR(("Corrupted CMS file %s", cmsFileAndName.c_str() ));
      return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
   }
   authenticTime = cms.getLastAuthenticTime();
   cfg->cfg_LastAuthenticTime.set(
                                  swu::writeTimeString(authenticTime, lastAuthenticTimeFormat));
   ETG_TRACE_USR1(("New authentic time: %s", ctime(&authenticTime) ));

   swu::SecurityEngine engine;
   std::string xmlFile, xmlDigest;
   // Read name of XML manifest and its digest. This has to exist, since
   // we can't build a chain of trust otherwise.
   if ( not cms.extractXmlNameAndDigest(xmlFile, xmlDigest, engine._enDigestType)) {
      ETG_TRACE_ERR(("Corrupted cms, can't find bosch xml file name or its digest"));
      return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
   }

   // Read encryption parameters and save to SecurityEngine.
   if (_config.useEncryption && cms.isUpdateEncrypted()) {
      std::string encParam;
      if ( (_uin == 0) or (not _uin->load()) ) {
         ETG_TRACE_ERR(("Could not load target's UIN."));
         if (_uin == 0) ETG_TRACE_ERR(("_uin is empty."));
         return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
      }

      if ( (_targetKey == 0) or (not _targetKey->load()) ) {
         ETG_TRACE_ERR(("Could not load target's private key."));
         return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
      }

      if ( not cms.extractEncryptParam(encParam, _uin->getUin(), allowEmptyCert) ) {
         ETG_TRACE_ERR(("Can't find encryption parameter in CMS file."));
         return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
      }

      if (cms.isMIME(encParam)) { // payload is S/MIME formatted
         ETG_TRACE_USR3(("Trying to decrypt MIME formatted license."));
         if( ! _targetKey->decryptCMS(encParam, engine._encryptParam)) {
            ETG_TRACE_ERR(("Can't decrypt symmmetric key & initial vector."));
            return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
         }
      }
      else { // payload is unformatted base64-encoded data
         std::vector<swu::SWU_BYTE> binEncParam = swu::decodeBase64(encParam);
         if ( ! binEncParam.size() ) {
            ETG_TRACE_ERR(("Failed to base64-decode symmetric encryption parameters.\n%s", encParam.c_str()));
            return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
         }

         if( ! _targetKey->decrypt(binEncParam, engine._encryptParam)) {
            ETG_TRACE_ERR(("Can't decrypt symmmetric key & initial vector."));
            return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
         }
      }
   }

   // std::vector< swu::SWU_BYTE > binXmlDigest = swu::decodeHex(xmlDigest);
   std::vector< swu::SWU_BYTE > binXmlDigest = swu::decodeDigest(xmlDigest, engine._enDigestType);
   if(0 == binXmlDigest.size()) {
      ETG_TRACE_ERR(("Failed to convert to binary: %s", xmlFile.c_str()));
      return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
   }

   engine._digests.push_back(binXmlDigest);
   if ( not engine.processAndCatenate(path + "/" + xmlFile, "/tmp/" + xmlFile)) {
      ETG_TRACE_ERR(("Can't verify or decrypt %s",  xmlFile.c_str()));
      return tenSwUpdateError_ERROR_SIGNATURE_CHECK_FAILED;
   }

   if(_srcFilters && Config::instance()->cfg_ApplyXMLFilterChain.readAsBool() && sourceType != tenSourceType_SCOMO_INDEX) {
      std::string boschXmlPath = "/tmp/" + xmlFile;
         
      ETG_TRACE_USR4(("CMSParser::parseCMSFile:before filter srcPath:%s", boschXmlPath.c_str()));
      ReleaseFilterContext context;
      
      errorCode=_srcFilters->apply(boschXmlPath, context);
      if(errorCode != tenSwUpdateError_OK) {
         return errorCode;
      }
      
      xmlFile = context.sourceInfo.infoFile;
      ETG_TRACE_USR4(("CMSParser::parseCMSFile:after filter srcPath:%s", xmlFile.c_str()));
   }
   
   if ( not doc->LoadFile(("/tmp/" + xmlFile).c_str()))
   {
      ETG_TRACE_ERR(("Could not load %s", ("/tmp/" + xmlFile).c_str()));
      ETG_TRACE_ERR(("Reason: %s", doc->ErrorDesc()));
      ETG_TRACE_ERR(("Error at row: %3d col:: %3d", doc->ErrorRow(), doc->ErrorCol()));
      return tenSwUpdateError_ERROR_METAINFO_READ;
   }
   ReleaseFilterContext context;
   context.sourceInfo=sourceInfo;
   errorCode =_preFilters->apply(doc, context);
   if (errorCode != tenSwUpdateError_OK) {
      return errorCode;
   }

   TiXmlElement *root = doc->FirstChildElement();
   if(!root)
   {
      ETG_TRACE_ERR(("Root node of boschxml missing in %s", xmlFile.c_str()));
      return tenSwUpdateError_ERROR_METAINFO_PARSE;
   }
   TiXmlElement *overall = root->FirstChildElement("OVERALL");
   if (!overall) {
      ETG_TRACE_ERR(("parseCMSFile: %s OVERALL section missing", xmlFile.c_str()));
      return tenSwUpdateError_ERROR_METAINFO_PARSE;
   }
   swu::setTextChild(overall, "MEDIUM_DRIVE", path);
   swu::setTextChild(overall, "INFO_FILE", infoFile);
   swu::setUIntChild(overall, "SOURCE_TYPE", (tU32)sourceInfo.enSourceType);

   swu::setTextChild(overall, "BXML_FILE", xmlFile);

   std::string hash;
   std::string filePathName = "/tmp/" + xmlFile;

   if(swu::exists(filePathName)) {
      if (swu::calculateSHA256HashFromFile(filePathName, hash)) {
         swu::setTextChild(overall, "BXML_CHECKSUM", hash);
         ETG_TRACE_USR1(("Added node BXML_CHECKSUM [%s]", hash.c_str()));
      }
   }

   TiXmlElement encryptElement("ENCRYPT_PARAM");
   encryptElement.InsertEndChild(TiXmlText(engine._encryptParam.c_str()));
   return swu::addNodeToAll(root, "CKSUM", encryptElement) ? tenSwUpdateError_OK : tenSwUpdateError_ERROR_METAINFO_PARSE;
}

bool CMSParser::setOverallSection(TiXmlElement const &overallSection)
{
   ETG_TRACE_USR1(("CMSParser::setOverallSection"));
   std::string mediumPathP= swu::getTextFromChildOrEmpty(&overallSection, "MEDIUM_DRIVE");
   if (mediumPathP.empty())
   {
      ETG_TRACE_ERR(("CMSParser::setOverallSection: missing MEDIUM_DRIVE"));
      _errorCode = tenSwUpdateError_ERROR_METAINFO_NOT_FOUND;
      return false;
   }

   trSourceInfo sourceInfo=PropDevMgr::instance()->getSourceInfo(mediumPathP);
   sourceInfo.infoFile = swu::getTextFromChildOrEmpty(&overallSection, "INFO_FILE");
   if (sourceInfo.infoFile.empty())
   {
      ETG_TRACE_ERR(("CMSParser::setOverallSection: missing INFO_FILE"));
      _errorCode = tenSwUpdateError_ERROR_METAINFO_NOT_FOUND;
      return false;
   }
   ETG_TRACE_USR1(("Found infoFile %30s/%30s in overallSection", sourceInfo.path.c_str(), sourceInfo.infoFile.c_str()));
   if (_document) delete _document;
   _document = new TiXmlDocument;
   if (!_document) {
      return false;
   }

   if (tenSwUpdateError_OK != parseCMSFile(sourceInfo,
                                           _document))
   {
      ETG_TRACE_USR3(("CMSParser::setOverallSection: Failed %s CMS File", sourceInfo.infoFile.c_str()));
      printf("CMSParser::setOverallSection: Failed %s CMS File\n", sourceInfo.infoFile.c_str());
      _errorCode = tenSwUpdateError_ERROR_METAINFO_NOT_FOUND;
      return false;
   }
   ETG_TRACE_USR3(("CMSParser::setOverallSection: Parsed %s CMS File", sourceInfo.infoFile.c_str()));

   if (_postFilters) {
      ReleaseFilterContext context;
      context.sourceInfo=sourceInfo;
      _errorCode = _postFilters->apply(_document, context);
   }

   return true;
}

TiXmlDocument *CMSParser::getReleaseXml(size_t id)
{
   TiXmlDocument *doc=_document;
   if (_document) _document=0;
   return doc;
}

tenSwUpdateError CMSParser::getParserError(void) const
{
   return _errorCode;
}

}
