#include "main/fcswupd_component.h"
#include "tinyxml/tinyxml.h"
#include "util/swu_util.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_execCommand.h"
#include "util/swu_constants.hpp"
#include "util/fcswupd_types.hpp"
#include "config/fcswupd_config.hpp"
#include "main/fcswupd_history.h"
#include "fcswupd_systemData.h"

#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <string.h>
#include <fstream>

#include "util/fcswupd_trace.hpp"

#define ETG_I_FILE_PREFIX fcswupdate::SystemData::instance()->

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "FCSWUPD_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#include "trcGenProj/Header/fcswupd_systemData.cpp.trc.h"
#endif 


namespace fcswupdate {

/*
 members must be initialized in order they are declared,
 otherwise lint will complain.
 please one initializer by line, otherwise merging is painfull ..
 */

std::string SXMHardwareType::_HWType="unknown";

SystemData::SystemData() :
      _enState(tenState_Initial),
            _trainVersion(),
            _defaultUpdateKey(),
            _defaultUpdateKeyInitialized(false),
            _hasValidTrain(false),
            _releaseXml("empty") {
}

SystemData::~SystemData() {

   for (std::map< std::string, ModVersionIf * >::iterator it = _modVersionIfs.begin();
         it != _modVersionIfs.end();
         ++it) {
      if (it->second->isOrphan()) {
         delete it->second;
      }
   }
   _modVersionIfs.clear();
}

void SystemData::traceState() {
   ETG_TRACE_COMP(("  _enState=%u", ETG_CENUM(SystemData::tenState, _enState)));
   ETG_TRACE_COMP(("  _trainVersion=%s", _trainVersion.c_str()));
   ETG_TRACE_COMP(("  _defaultUpdateKey=%s", _defaultUpdateKey.c_str()));
   ETG_TRACE_COMP(("  _defaultUpdateKeyInitialized=%u", _defaultUpdateKeyInitialized));
   ETG_TRACE_COMP(("  Current versions:"));
   for (std::map< std::string, ModVersionIf * >::iterator iter = _modVersionIfs.begin();
         iter != _modVersionIfs.end();
         ++iter) {
      ETG_TRACE_COMP(("    MOD=%s:", iter->first.c_str()));
      ETG_TRACE_COMP(("      VERSION=%s:", iter->second->getVersion().c_str()));
      ETG_TRACE_COMP(("      BOSCH-VERSION=%s:", iter->second->getBoschVersion().c_str()));

   }

   // todo: iterate over devices, modules, submodules

}

tVoid SystemData::vInit() {
   ETG_TRACE_USR1(("SystemData::vInit() START"));
   ETG_I_REGISTER_FILE();

   if (_enState == tenState_Initial) {
      _enState = tenState_Initialized;
   }

#ifndef VARIANT_S_FTR_ENABLE_G4G
   // register modversion-ifs for all known subprocessors
   SystemData::instance()->registerModVersionIf(new ImxAppVersionIf);
   SystemData::instance()->registerModVersionIf(new ImxBlVersionIf);
   SystemData::instance()->registerModVersionIf(new V850AppVersionIf);
   SystemData::instance()->registerModVersionIf(new V850BlVersionIf);
   SystemData::instance()->registerModVersionIf(new AdrVersionIf);
   SystemData::instance()->registerModVersionIf(new TeseoAppVersionIf);
   SystemData::instance()->registerModVersionIf(new SxmAppVersionIf);
   SystemData::instance()->registerModVersionIf(new MISAppVersionIf);
   SystemData::instance()->registerModVersionIf(new MCPAppVersionIf);
   //SystemData::instance()->registerModVersionIf(new FpgaVersionIf);
   //SystemData::instance()->registerModVersionIf(new CPLDVersionIf);
   //SystemData::instance()->registerModVersionIf(new ExtFpgaVersionIf);
#endif
   ETG_TRACE_USR1(("SystemData::vInit() END"));
}

tVoid SystemData::vDeInit() {
   ETG_I_UNREGISTER_FILE();
}

bool SystemData::registerInstalledRelease(const TiXmlElement *release,
      const TiXmlElement *releaseProgress, const std::string &type) {
   SWU_ASSERT_RETURN_FALSE(_enState == tenState_Initialized);
   ETG_TRACE_USR1(("SystemData::registerInstalledRelease deprecated"));
   return true;
}


bool SystemData::registerModVersionIf(ModVersionIf *modVersionIf) {
   SWU_ASSERT_RETURN_FALSE(_enState == tenState_Initialized);
   SWU_ASSERT_RETURN_FALSE(modVersionIf);
   ETG_TRACE_USR1(("SystemData::registerModVersionIf(%s)",
                   modVersionIf->getModName().c_str()));
   for (std::map< std::string, ModVersionIf * >::iterator it = _modVersionIfs.begin();
        it != _modVersionIfs.end();
        ++it) {
      ETG_TRACE_USR1(("   SystemData::registerModVersionIf have: %s",
                      it->second->getModName().c_str()));

   }

   std::map< std::string, ModVersionIf * >::iterator it = _modVersionIfs.find(modVersionIf->getModName());
   if (it == _modVersionIfs.end()) {
      _modVersionIfs[modVersionIf->getModName()] = modVersionIf;
   }
   else {
      ETG_TRACE_COMP(("SystemData::registerModVersionIf(%s): already registered",
            modVersionIf->getModName().c_str()));
      it->second = modVersionIf;
   }

   return true;
}

void SystemData::deregisterModVersionIf(ModVersionIf *modVersionIf) {
   SWU_ASSERT_RETURN(_enState == tenState_Initialized);
   SWU_ASSERT_RETURN(modVersionIf);
   ETG_TRACE_USR1(("SystemData::deregisterModVersionIf(%s)",
         modVersionIf->getModName().c_str()));

   for (std::map< std::string, ModVersionIf * >::iterator it = _modVersionIfs.begin();
         it != _modVersionIfs.end();
         ++it) {
      if (it->second == modVersionIf) {
         if (it->second->isOrphan()) {
            delete it->second;
         }
         _modVersionIfs.erase(it);
         return;
      }
   }
   ETG_TRACE_ERR(("SystemData::deregisterModVersionIf(%s): not registered",
         modVersionIf->getModName().c_str()));

}



ModVersionIf *SystemData::getModVersionIf(std::string name) {
   ModVersionIf *modIf=0;

   std::map< std::string, ModVersionIf * >::iterator it = _modVersionIfs.find(name);
   if (it != _modVersionIfs.end()) {
      modIf = it->second;
      ETG_TRACE_USR1(("   SystemData::getModVersionIf found: %s", name.c_str()));
   }
   else {
      ETG_TRACE_ERR(("SystemData::getModVersionIf(%s): not registered, create default",
            name.c_str()));
      modIf = ModVersionDefaultIf::create(name);
      SWU_ASSERT_RETURN_VAL(modIf, modIf);
      registerModVersionIf(modIf);
   }
   return modIf;
}


ETG_I_CMD_DEFINE((getCurrentModVersion, "getCurrentModVersion %80s", ETG_I_STRING))
std::string SystemData::getCurrentModVersion(std::string name) {
   SWU_ASSERT_RETURN_VAL(_enState == tenState_Initialized, "");
   std::string version = "";
   ModVersionIf *modIf=getModVersionIf(name);
   SWU_ASSERT_RETURN_VAL(modIf, "");
   version=modIf->getVersion();
   swu::purifyString(version);
   ETG_TRACE_USR1(("   SystemData::getCurrentModVersion(%50s): %s",
                   name.c_str(), version.c_str()));
   return version;
}

ETG_I_CMD_DEFINE((onModVersionChanged, "onModVersionChanged %80s", ETG_I_STRING))
void SystemData::onModVersionChanged(std::string name) {
   SWU_ASSERT_RETURN(_enState == tenState_Initialized);
   ETG_TRACE_USR1(("SystemData::onModVersionChanged: %s", name.c_str()));
   ModVersionIf *modIf=getModVersionIf(name);
   SWU_ASSERT_RETURN(modIf);
   modIf->invalidate();
}


std::string AdrVersionIf::readBoschVersion() {
   if (Config::instance()->cfg_EnterFullOperation.get()) {
      return Config::instance()->cfg_AdrSwVersion.get();
   } else {
      std::string version="unknown";
      std::string tmpFileName="/tmp/adr3_version.txt";
      unlink(tmpFileName.c_str());

      do {
         std::string command = "/opt/bosch/base/bin/swu_common_adr3_app_out.out -v -p -x /opt/bosch/base/bin/xl_flasher662.dnl -o " + tmpFileName;
         if (swu::execCommand(command.c_str())) {
            ETG_TRACE_ERR(("AdrVersionIf: cmd failed: %s", command.c_str()));
            break;
         }
         std::ifstream infile(tmpFileName.c_str());
         if (!infile.is_open()) {
            ETG_TRACE_ERR(("AdrVersionIf: could not open result-file: %s", tmpFileName.c_str()));
            break;
         }
         if (!std::getline(infile, version)) {
            ETG_TRACE_ERR(("AdrVersionIf: could not read from result-file: %s", tmpFileName.c_str()));
            break;
         }
         swu::trimString(version);
         if (swu::purifyString(version)) {
            version="unknown";
         }
         ETG_TRACE_USR1(("read adr-version: %s", version.c_str()));
      }
      while (0);
      //      unlink(tmpFileName.c_str());
      return version;
      
   }
}


std::string ImxAppVersionIf::readBoschVersion() {
  return Config::instance()->cfg_RunningSwLabel.get();
}


std::string ImxBlVersionIf::readBoschVersion() {
  return Config::instance()->cfg_RunningSwLabel.get();
}



std::string SxmAppVersionIf::readBoschVersion() {
   if (Config::instance()->cfg_EnterFullOperation.get()) {
      return Config::instance()->cfg_SxmVersion.get();
   } 

   std::string version="unknown";
   std::string tmpFileName="/tmp/sxm_version.txt";
   unlink(tmpFileName.c_str());
   
   do {
      std::string command = "/opt/bosch/base/bin/swu_common_sxm_app_out.out -v > " + tmpFileName;

      // fixme: what a strange resultcode (20), and in an enum ...
      if (swu::execCommand(command.c_str())) {
         ETG_TRACE_ERR(("SxmVersionIf: cmd failed: %s", command.c_str()));
         break;
      }
      std::ifstream infile(tmpFileName.c_str());
      if (!infile.is_open()) {
         ETG_TRACE_ERR(("SxmVersionIf: could not open result-file: %s", tmpFileName.c_str()));
         break;
      }
      while (std::getline(infile, version)) {
         ETG_TRACE_USR2(("SxmVersionIf: got line: %s", version.c_str()));
         swu::trimString(version);
         if (!strstr(version.c_str(), "SXM_SW_VER")) {
            ETG_TRACE_USR2(("SxmVersionIf: line without SXM_SW_VER"));
            continue;
         }
         version.erase(0, version.find("=") + 1);
         ETG_TRACE_USR2(("SxmVersionIf: line after erase:%s", version.c_str()));
         swu::trimString(version);
         if (!version.empty()) {
            ETG_TRACE_USR2(("SxmVersionIf: found version:%s", version.c_str()));
            break;
         }
         version="unknown";

      }
   } while (0);
   //   unlink(tmpFileName.c_str());
   
   ETG_TRACE_USR1(("read sxm-version: %s", version.c_str()));
   return version;
}

std::string CPLDVersionIf::readBoschVersion() {

   std::string version="unknown";   
   return version;

   /*
     reading the version through v850, still the implementation is pending from AUTOSAR side.
     contact: Alexander Neufeld & Andre Bernde
     reference:
     https://hi-dms.de.bosch.com/docushare/dsweb/Services/Document-921238
     https://hi-dms.de.bosch.com/docushare/dsweb/Services/Document-941449
    */

   std::string tmpFileName="/tmp/cpld_version.txt";
   unlink(tmpFileName.c_str());
   do {
      std::string command = "i2cget -f -y 0x10 0xfe > " + tmpFileName;
      if (swu::execCommand(command.c_str())) {
         ETG_TRACE_ERR(("CPLDVersionIf: cmd failed: %s", command.c_str()));
         version="unknown";
         break;
      }

      std::ifstream infile(tmpFileName.c_str());
      if (!infile.is_open()) {
         ETG_TRACE_ERR(("CpldVersionIf: could not open result-file: %s", tmpFileName.c_str()));
         break;
      }
      if (!std::getline(infile, version)) {
         ETG_TRACE_ERR(("CpldVersionIf: could not read from result-file: %s", tmpFileName.c_str()));
         break;
      }
      swu::trimString(version);
      if (!version.empty()) {
         swu::replaceSubString(version, "0x", "");
         ETG_TRACE_USR2(("CpldVersionIf: found version:%s", version.c_str()));
         break;
      }
      version="unknown";
   }
   while (0);
      ETG_TRACE_USR1(("read cpld-version: %s", version.c_str()));
   return version;
}

std::string FpgaVersionIf::readBoschVersion() {

   /*
     FPGA spec: https://hi-dms.de.bosch.com/docushare/dsweb/Services/Document-837044
     0xFC - DeviceId (i2cget -f -y 1 0x10 0xfc) - d"deviceId"v
     0xFF - HWversion (i2cget -f -y 1 0x10 0xff) - d"deviceId"v"HWversion"
     
   */
   std::string tmpFileName="/tmp/fpga_version.txt";
   std::string version = "d";
   std::string addr = "0xfc";
   
   do {
      std::string fpgaVersion;
      unlink(tmpFileName.c_str());      
      std::string command = "i2cget -f -y 1 0x10 " + addr + " > " + tmpFileName;
      if (swu::execCommand(command.c_str())) {
         ETG_TRACE_ERR(("FpgaVersionIf: cmd failed: %s", command.c_str()));
         version="unknown";
         break;
      }

      /*
      todo:
      below code is not used, no definite mechanism to read the version.
      when the definite mechanism is found we can switch to normal flow.
      till then, jamplayer verify code action command is used to find whether the firmware version is different or same.
       */
      std::ifstream infile(tmpFileName.c_str());
      if (!infile.is_open()) {
         ETG_TRACE_ERR(("FpgaVersionIf: could not open result-file: %s", tmpFileName.c_str()));
         break;
      }
      if (!std::getline(infile, fpgaVersion)) {
         ETG_TRACE_ERR(("FpgaVersionIf: could not read from result-file: %s", tmpFileName.c_str()));
         break;
      }
      swu::trimString(fpgaVersion);
      if (!fpgaVersion.empty()) {
         swu::replaceSubString(fpgaVersion, "0x", "");
         version = version + fpgaVersion;
         ETG_TRACE_USR2(("FpgaVersionIf: found version:%s", version.c_str()));

         if(addr == "0xfc") {
            version = version + "v";
            addr = "0xff";
            continue;
         } else {
            break;
         }         
      }
      version="unknown";
   }while (1);
   ETG_TRACE_USR1(("read fpga-version: %s", version.c_str()));
   return version;
}

std::string ExtFpgaVersionIf::readBoschVersion() {
   FpgaVersionIf fpgaIf;
   std::string version = fpgaIf.readBoschVersion();
   ETG_TRACE_USR1(("read Extfpga-version: %s", version.c_str()));
   return version;
}


std::string V850AppVersionIf::readBoschVersion() {
   if (Config::instance()->cfg_EnterFullOperation.get()) {
      return Config::instance()->cfg_V850AppVersion.get();
   }
 
   std::string version="unknown";
   std::string tmpFileName="/tmp/v850_version.txt";
   unlink(tmpFileName.c_str());
   
   do {
      std::string command = "/opt/bosch/base/bin/swupdatecontrol_out.out -v app > " + tmpFileName;
      if (swu::execCommand(command.c_str())) {
         ETG_TRACE_ERR(("V850AppVersionIf: cmd failed: %s", command.c_str()));
         break;
      }
      std::ifstream infile(tmpFileName.c_str());
      if (!infile.is_open()) {
         ETG_TRACE_ERR(("V850AppVersionIf: could not open result-file: %s", tmpFileName.c_str()));
         break;
      }
      if (!std::getline(infile, version)) {
         ETG_TRACE_ERR(("V850AppVersionIf: could not read from result-file: %s", tmpFileName.c_str()));
         break;
      }
      swu::trimString(version);
      ETG_TRACE_USR1(("read v850 app version: %s", version.c_str()));
   }
   while (0);
   //   unlink(tmpFileName.c_str());
   return version;
   
}


std::string V850BlVersionIf::readBoschVersion() {
   return Config::instance()->cfg_V850BlVersion.get();
  // only checking Fw version
}


std::string TeseoAppVersionIf::readBoschVersion() {
   if (Config::instance()->cfg_EnterFullOperation.get()) {
      // read from registry
      return Config::instance()->cfg_TeseoVersion.get();
   } 
   
   std::string version="unknown";
   std::string tmpFileName="/tmp/teseo_version.txt";
   unlink(tmpFileName.c_str());

   do {
      std::string command = "/opt/bosch/base/bin/swu_common_teseo_app_out.out -r -o " + tmpFileName;
      tU32 teseoRes=swu::execCommand(command.c_str());

      // fixme: what a strange resultcode (20), and in an enum ...
      if (teseoRes && (teseoRes != 20)) {
         ETG_TRACE_ERR(("TeseoVersionIf: cmd failed: %s", command.c_str()));
         break;
      }
      std::ifstream infile(tmpFileName.c_str());
      if (!infile.is_open()) {
         ETG_TRACE_ERR(("TeseoVersionIf: could not open result-file: %s", tmpFileName.c_str()));
         break;
      }
      while (std::getline(infile, version)) {
         ETG_TRACE_USR2(("TeseoVersionIf: got line: %s", version.c_str()));
         swu::trimString(version);
         if (!version.empty()) {
            break;
         }
      }
      if (version.empty()) {
         ETG_TRACE_ERR(("TeseoVersionIf: could not read from result-file: %s", tmpFileName.c_str()));
         version="unknown";
      }
   } while (0);
   //   unlink(tmpFileName.c_str());
   
   ETG_TRACE_USR1(("read teseo-version: %s", version.c_str()));
   return version;
}

std::string PdConfigAppVersionIf::readBoschVersion() {
   std::string version=Config::instance()->cfg_PdConfigVersion.readAsString();
   ETG_TRACE_USR1(("PdConfigAppVersionIf::readBoschVersion: %s", version.c_str()));
   return version;
}

std::string CdDefConfigAppVersionIf::readBoschVersion() {
   std::string version=Config::instance()->cfg_CdDefConfigVersion.readAsString();
   ETG_TRACE_USR1(("CdDefConfigAppVersionIf::readBoschVersion: %s", version.c_str()));
   return version;
}

std::string CdConfigAppVersionIf::readBoschVersion() {
   std::string version=Config::instance()->cfg_CdConfigVersion.readAsString();
   ETG_TRACE_USR1(("CdConfigAppVersionIf::readBoschVersion: %s", version.c_str()));
   return version;
}


std::string SXMHardwareType::get() {
  if(_HWType != "unknown")
    return _HWType;

  if (Config::instance()->cfg_EnterFullOperation.get()) {
    _HWType = Config::instance()->cfg_SxmHwType.get();
    return _HWType;
  }

  std::string version="unknown";
  std::string tmpFileName="/tmp/sxm_version.txt";
  unlink(tmpFileName.c_str());

  do {
    std::string command = "/opt/bosch/base/bin/swu_common_sxm_app_out.out -v > " + tmpFileName;

    // fixme: what a strange resultcode (20), and in an enum ...
    if (swu::execCommand(command.c_str())) {
      ETG_TRACE_ERR(("SXMHardwareType: cmd failed: %s", command.c_str()));
      break;
    }
    std::ifstream infile(tmpFileName.c_str());
    if (!infile.is_open()) {
      ETG_TRACE_ERR(("SXMHardwareType: could not open result-file: %s", tmpFileName.c_str()));
      break;
    }
    while (std::getline(infile, version)) {
      ETG_TRACE_USR2(("SXMHardwareType: got line: %s", version.c_str()));
      swu::trimString(version);
      if (!strstr(version.c_str(), "SXM_HW_TYPE")) {
        ETG_TRACE_USR2(("SXMHardwareType: line without SXM_HW_TYPE"));
        continue;
      }
      version.erase(0, version.find("=") + 1);
      ETG_TRACE_USR2(("SXMHardwareType: line after erase:%s", version.c_str()));
      swu::trimString(version);
      if (!version.empty()) {
        ETG_TRACE_USR2(("SXMHardwareType: found version:%s", version.c_str()));
        break;
      }
      version="unknown";
    }
  } while (0);
  _HWType = version;

  ETG_TRACE_USR1(("read SXM_TYPE: %s", version.c_str()));
  return _HWType;

}

std::string DtvVersionIf::readBoschVersion() {
   
   if (Config::instance()->cfg_EnterFullOperation.get()) {
      return Config::instance()->cfg_DtvVersion.get();
   } 
 
   std::string version="unknown";
   std::string tmpFileName="/tmp/dtv_version.txt";
   unlink(tmpFileName.c_str());

   do {
      std::string command = "/opt/bosch/base/bin/swu_common_dtv_app_out.out -v > " + tmpFileName;
      if (swu::execCommand(command.c_str())) {
         ETG_TRACE_ERR(("DtvVersionIf: cmd failed: %s", command.c_str()));
         break;
      }
      std::ifstream infile(tmpFileName.c_str());
      if (!infile.is_open()) {
         ETG_TRACE_ERR(("DtvVersionIf: could not open result-file: %s", tmpFileName.c_str()));
         break;
      }
      while (std::getline(infile, version)) {
         ETG_TRACE_USR2(("DTV-Version: got line: %s", version.c_str()));
         swu::trimString(version);
         if (!strstr(version.c_str(), "DTV_SW_VER")) {
            ETG_TRACE_USR2(("DTV-Version: line without DTV_SW_VER"));
            continue;
         }
         version.erase(0, version.find("=") + 1);
         ETG_TRACE_USR2(("DTV-Version: line after erase:%s", version.c_str()));
         swu::trimString(version);
         if (!version.empty()) {
            ETG_TRACE_USR2(("DTV-Version: found version:%s", version.c_str()));
            break;
         }
         version="unknown";
         
      }
   } while (0);
   return version;
}


std::string MCPAppVersionIf::readBoschVersion() {

   std::string version="unknown";
 /*  [TO DO :  Need to read MCP version and test on target]
   std::string tmpFileName="/tmp/mcp_version.txt";
   if (Config::instance()->cfg_EnterFullOperation.get()) {

       unlink(tmpFileName.c_str());
   
       do {
               std::string command = "/opt/bosch/base/bin/swu_common_mcp_app_out.out -v > " + tmpFileName;

               // fixme: what a strange resultcode (20), and in an enum ...
               if (swu::execCommand(command.c_str())) {
                       ETG_TRACE_ERR(("MCPAppVersionIf: cmd failed: %s", command.c_str()));
                       break;
               }
               std::ifstream infile(tmpFileName.c_str());
               if (!infile.is_open()) {
                       ETG_TRACE_ERR(("MCPAppVersionIf: could not open result-file: %s", tmpFileName.c_str()));
                       break;
               }
               while (std::getline(infile, version)) {
                       ETG_TRACE_USR2(("MCPAppVersionIf: got line: %s", version.c_str()));
                       swu::trimString(version);
                       if (!strstr(version.c_str(), "MCP_SW_VERSION")) {
                               ETG_TRACE_USR2(("MCPAppVersionIf: line without MCP_SW_VERSION"));
                               continue;
                       }
                       version.erase(0, version.find(":") + 1);
                       ETG_TRACE_USR2(("MCPAppVersionIf: line after erase:%s", version.c_str()));
                       swu::trimString(version);
                       if (!version.empty()) {
                               ETG_TRACE_USR2(("MCPAppVersionIf: found version:%s", version.c_str()));
                               break;
                       }
                       version="unknown";
       
               }
       } while (0);
   
   ETG_TRACE_USR1(("read MCP_SW_VERSION: %s", version.c_str()));
   } */

  // Recovery Mode No need to get the version because this will only run in normal mode.
  return version;
}

std::string MISAppVersionIf::readBoschVersion() {
   //MIS version check will be taken care by python scripts, so returning unknown 
   std::string version="unknown";
   return version;
}


bool SystemData::getFCID(tU16 & fcid)
{
   fcid=0;
   tU32 u32Fcid;
   bool res= Config::instance()->cfg_FcId.get(u32Fcid);
   fcid=(tU16)u32Fcid;
   return res;
}

tU64 u64GetBuildTime() {
   std::string buildTime;;
   tU64 u64BuildTime=0;

   bool res= Config::instance()->cfg_RunningBuildTime.get(buildTime);
   if (res) {
   // cast- for gen3 and gen4 compatibality
      if (1 != sscanf(buildTime.c_str(), "%llu", (long long unsigned int*)&u64BuildTime)) {
         ETG_TRACE_ERR(("u64GetBuildTime: Can't sscanf timestamp")); 
      }
   }
   return u64BuildTime;
}









}
