#include <fstream>
#include "util/swu_execCommand.h"
#include "util/swu_util.hpp"
#include "util/swu_trace.h"
#include "util/swu_filesystem.h"
#include "util/swu_wupAccess.h"
#include "util/swu_incIf.hpp"
#include "util/swu_bootChain.h"
#include "util/swu_globallog.h"

#include "config/fcswupd_config.hpp"
#include "util/fcswupd_types.hpp"
#include "main/fcswupd_component.h"
#include "main/fcswupd_main.h"
#include "main/fcswupd_spmIf.h"
#include "main/fcswupd_hmisettings.h"
#include "main/fcswupd_resourcemanager.h"



#define ETG_I_FILE_PREFIX fcswupdate::SpmIf::instance()->

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "FCSWUPD_SPMIF_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#include "trcGenProj/Header/fcswupd_spmIf.cpp.trc.h"
#endif


namespace fcswupdate {


static const tS32 RESET_COUNTER_PRAM_INDEX      =1;
static const tS32 RESET_COUNTER_PRAM_CHECK_INDEX=2;




void SpmIf::sendSytemModeRequestToSpm(tenSystemMode enSystemMode) {
   FCSWUPD_NS_SPM_T::T_e8_StateModes fiMode= FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeDownloadExit;
   switch (enSystemMode) {
      case enSystemMode_Normal:{
         if (_modeSentToSpm==enSystemMode_Update) {
            fiMode = FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeDownloadExit;
         }
         else {
            fiMode = FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeCustomerDownloadExit;
         }
      }
         break;
      case enSystemMode_Update:{
         fiMode = FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeDownload;
      }
         break;
      case enSystemMode_CustomerUpdate:{
         fiMode = FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeCustomerDownloadStart;         
      }
         break;  
      case enSystemMode_Lock:
         fiMode = FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeMapDownloadLock;
         break;
      case enSystemMode_UnLock:
         fiMode = FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeMapDownloadUnlock;
         break;
      default:
         break;
   }
   ETG_TRACE_USR1(("SpmIf::sendSytemModeRequestToSpm:enSystemMode=%u --> fiMode=%u",
                   ETG_CENUM(tenSystemMode, enSystemMode), ETG_CENUM(FCSWUPD_NS_SPM_T::T_e8_StateModes, fiMode)));
   swu::LOG_INTERFACE("sendSystemModeStart:fiMode:%u", fiMode);
   SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSystemModeStart, fiMode);

   _modeSentToSpm = enSystemMode;


}

tVoid Msg_SpmModeReached::vTrace() {
   ETG_TRACE_USR1((" Msg_SpmModeReached(enSystemMode=%u)",
                   ETG_CENUM(tenSystemMode, enSystemMode)));
}


SpmIf::SpmIf():
   _modeSentToSpm(enSystemMode_Normal),
   _modeReceivedFromSpm(enSystemMode_Normal),
   _modeRequestedByUser(enSystemMode_Normal),
   _mapDownloadLockOwner(0xFFFF),
   _bFirst(true){
   ETG_I_REGISTER_FILE();
}

SpmIf::~SpmIf() {
   ETG_I_UNREGISTER_FILE();
}

void SpmIf::vInit() {   
   Msg_SubStatesStatus::vSubscribe(this);
   Msg_SystemStateStatus::vSubscribe(this);
   Msg_CvmEventStatus::vSubscribe(this);
   Msg_SystemModeResult::vSubscribe(this);
   //Msg_MapDownloadLockStateStatus::vSubscribe(this);
   Msg_UpdateLockResult::vSubscribe(this);
   Msg_UpdateLockStatesStatus::vSubscribe(this);

   swu::Msg_ProxyAvailability<FCSWUPD_NS_SPM::SPM_CORE_FIProxy, FcSwUpdRoot>::vSubscribe(this);

#ifndef __SW_UPDATE_UNIT_TESTING__
   _spmProxy.init("spmFiPort");
#endif
}

void SpmIf::vDeInit() {   
}

void SpmIf::traceState() {
   ETG_TRACE_COMP(("  _modeSentToSpm=%u",ETG_CENUM(tenSystemMode, _modeSentToSpm)));
   ETG_TRACE_COMP(("  _modeReceivedFromSpm=%u", ETG_CENUM(tenSystemMode, _modeReceivedFromSpm)));
   ETG_TRACE_COMP(("  _modeRequestedByUser=%u",ETG_CENUM(tenSystemMode, _modeRequestedByUser)));
   ETG_TRACE_COMP(("  _bFirst=%u",_bFirst));
   ETG_TRACE_COMP(("  isInStandAloneMode=%u",isInStandAloneMode()));

#ifndef __SW_UPDATE_UNIT_TESTING__
   ETG_TRACE_COMP(("  timerRunning=%u", _waitSpmTimer.isRunning()));
#endif
}

void SpmIf::sendSpmModeReachedToClients() {
   ETG_TRACE_USR1(("SpmIf::sendSpmModeReachedToClients(%u)",
                   ETG_CENUM(tenSystemMode, _modeReceivedFromSpm)));
   Msg_SpmModeReached msg(_modeReceivedFromSpm);
   notify(msg);
}

void SpmIf::vProcess(Msg_SubStatesStatus *pMsg) {
   // todo: currently the messages from SPM are not populated and sent as expected
   tU32 subStates = pMsg->payload->getSubState();
   //bool isDlMode =  (subStates & (tU32)FCSWUPD_NS_SPM_T::T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_DOWNLOAD);
   ETG_TRACE_COMP(("SpmIf::vProcess(Msg_SubStatesStatus):bDlModeSentToSpm=%u (_neodeReceivedFromSpm=%u) subStates=0x%08x",
                   ETG_CENUM(tenSystemMode, _modeSentToSpm), ETG_CENUM(tenSystemMode, _modeReceivedFromSpm), subStates));
}

void SpmIf::onSystemModeAck() {
   ETG_TRACE_COMP(("SpmIf::onSystemModeAck:_modeSentToSpm=%u _modeReceivedFromSpm=%u",
                   ETG_CENUM(tenSystemMode, _modeSentToSpm), ETG_CENUM(tenSystemMode, _modeReceivedFromSpm)));

   if (_modeSentToSpm != _modeReceivedFromSpm) {
      _modeReceivedFromSpm=_modeSentToSpm;
#ifndef __SW_UPDATE_UNIT_TESTING__
      _waitSpmTimer.stop();
#endif
      if(_modeRequestedByUser == enSystemMode_Lock && _mapDownloadLockOwner == SPM_APPID_SWUPDATE) {
         ETG_TRACE_USR3(("Lock is provided by SPM"));
      } else {
         ETG_TRACE_USR3(("Lock is not provided by SPM _mapDownloadLockOwner:%u !!!", _mapDownloadLockOwner));
      }
      //Whether Lock is provided or not,FcSwUpdate has to continue otherwise it goes to hang state. 
      sendSpmModeReachedToClients();
   }

}

void SpmIf::vProcess(Msg_SystemStateStatus *pMsg) {
   FCSWUPD_NS_SPM_T::T_SPM_e32_SYSTEM_STATES systemState=pMsg->payload->getSystemState();
   ETG_TRACE_COMP(("SpmIf::vProcess(Msg_SystemStateStatus):_modeSentToSpm=%u _modeReceivedFromSpm=%u systemModeStatus=%u",
                   ETG_CENUM(tenSystemMode, _modeSentToSpm),
                   ETG_CENUM(tenSystemMode, _modeReceivedFromSpm),
                   systemState));

   if (systemState==FCSWUPD_NS_SPM_T::T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_ON_SWDL && _modeSentToSpm != enSystemMode_Update) {
      ETG_TRACE_ERR(("unexpected system-state from SPM:SPM_SYSTEM_MMI_ON_SWDL ,try to set to normal"));
      SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSystemModeStart, FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeDownloadExit);
   }
   if (systemState==FCSWUPD_NS_SPM_T::T_SPM_e32_SYSTEM_STATES__SPM_SYSTEM_MMI_STANDBY_CUSTOMER_SWDL && _modeSentToSpm != enSystemMode_CustomerUpdate) {
      ETG_TRACE_ERR(("unexpected system-state from SPM:SPM_SYSTEM_MMI_STANDBY_CUSTOMER_SWDL ,try to set to normal"));
      SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSystemModeStart, FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeCustomerDownloadExit);
   }


}


void SpmIf::vProcess(Msg_SystemModeResult *pMsg) {  
   tS32 u32Return = 0;
   u32Return = pMsg->payload->getS32ReturnVal();
   ETG_TRACE_USR1(("Msg_SystemModeResult:return %d", u32Return));
   ETG_TRACE_COMP(("SpmIf::vProcess(Msg_SystemModeResult):_modeSentToSpm=%u _modeReceivedFromSpm=%u",
                   ETG_CENUM(tenSystemMode, _modeSentToSpm), ETG_CENUM(tenSystemMode, _modeReceivedFromSpm)));

   // todo: currently message from SPM does not contain the reached system-mode, so we gues it is the one we last requested from SPM, with big risk of race-conditions
   onSystemModeAck();
}

void SpmIf::vProcess(Msg_MapDownloadLockStateStatus *pMsg) {
tU16 owner = 0;
bool status = false;
   status = pMsg->payload->hasU16AppIdLockOwner();
   owner = pMsg->payload->getU16AppIdLockOwner();
   ETG_TRACE_COMP(("SpmIf::vProcess(Msg_MapDownloadLockStateStatus): status=%u owner=%u", (int)status, owner));  
   if(status) {
      _mapDownloadLockOwner = owner;         
   }
}

void SpmIf::vProcess(Msg_UpdateLockResult *pMsg) {
   tS32 s32ReturnVal = 0;
   s32ReturnVal = pMsg->payload->getS32ReturnVal();
   ETG_TRACE_USR1(("SpmIf::Msg_UpdateLockResult returnVal:%d", s32ReturnVal));
   swu::LOG_INTERFACE("UpdateLockResult: val:%u", s32ReturnVal);
}

void SpmIf::vProcess(Msg_UpdateLockStatesStatus *pMsg) {
   ETG_TRACE_USR1(("SpmIf::Msg_UpdateLockStatesStatus")); 
   swu::LOG_INTERFACE("Msg_UpdateLockStatesStatus");
   ::boost::ptr_vector<FCSWUPD_NS_SPM_T::T_SPM_UpdateLockState> boost_spmLocksInfo =  pMsg->payload->getLLocks();
   ::std::vector <FCSWUPD_NS_SPM_T::T_SPM_UpdateLockState> spmLocksInfo = 
                 swu::convertBoostToStlVector<FCSWUPD_NS_SPM_T::T_SPM_UpdateLockState>(boost_spmLocksInfo);
   ResourceManager::instance()->handleSpmResouces(spmLocksInfo);   
}


void SpmIf::vProcess(swu::Msg_ProxyAvailability<FCSWUPD_NS_SPM::SPM_CORE_FIProxy, FcSwUpdRoot> *pMsg) {
   ETG_TRACE_COMP(("SpmIf::vProcess(Msg_ProxyAvailability):available=%u", pMsg->available));
   swu::LOG_INTERFACE("Msg_ProxyAvailability:available=%u", pMsg->available);
#ifndef __SW_UPDATE_UNIT_TESTING__
   if (_waitSpmTimer.isRunning()) {
      _waitSpmTimer.stop();
      requestSystemMode(_modeRequestedByUser);
   }
#endif  
}

tVoid SpmIf::vProcess(SpmIf::Msg_WaitSpmTimer *pMsg) {
   (void)pMsg; 
   ETG_TRACE_COMP(("SpmIf::vProcess(Msg_WaitSpmTimer)"));
   // supervision: when we receive no answer from spm, just request again.
   if (!_bFirst) {
      // missing result from spm, just fake the expected result
      onSystemModeAck();
   }
   _bFirst = false;
   requestSystemMode(_modeRequestedByUser);
}

/*
  todo: problem: spm currently does not tell in which mode he is.
  so we have to request the correct mode on each startup.
  If progress-section is present, ctrl will take care, otherwise we send
 */
void SpmIf::requestSystemMode(tenSystemMode enMode) {
   ETG_TRACE_COMP(("SpmIf::requestSystemMode(%u) START, state follows",
                   ETG_CENUM(tenSystemMode, enMode)));
   traceState();
   // store
   _modeRequestedByUser=enMode;
   if (isInStandAloneMode() && _bFirst) {
      // wait-timer to check if spm is really not available
#ifndef __SW_UPDATE_UNIT_TESTING__
      _waitSpmTimer.start(this, Config::getSpmWaitAvailTimeMs());
#endif
      return;
   }
   else if (isInStandAloneMode()) {
      _modeSentToSpm = _modeRequestedByUser;
      _modeReceivedFromSpm = _modeSentToSpm;
      sendSpmModeReachedToClients();
   }
   else if (_modeRequestedByUser== _modeSentToSpm) {
      if (_modeSentToSpm==_modeReceivedFromSpm) {
         // we are already in dl-mode, simulate correct answer
         sendSpmModeReachedToClients();
      } else {
         // mode was already requested, do nothing, wait for ack
      }
   }
   else {
      sendSytemModeRequestToSpm(_modeRequestedByUser);

         ETG_TRACE_COMP(("SpmIf::requestDlMode:request SystemModeStart(%u)",
                         ETG_CENUM(tenSystemMode, _modeSentToSpm)));
#ifndef __SW_UPDATE_UNIT_TESTING__
         _waitSpmTimer.start(this, Config::getSpmWaitTimeMs());
#endif
   }
   _bFirst=false;

   ETG_TRACE_COMP(("SpmIf::requestDlMode END, state follows:"));
   traceState();
}

void SpmIf::checkMapDownloadLockState()
{
   tU16 owner = 0;
#ifndef __SW_UPDATE_UNIT_TESTING__
   bool status = getProxy()->hasMapDownloadLockState();
   if (status)
      owner = getProxy()->getMapDownloadLockState().getU16AppIdLockOwner();
#endif
   ETG_TRACE_USR1(("SpmIf::checkMapDownloadLockState(%u):%u", (int)status, owner));

   if (owner != 0 && owner != (tU16)-1)
      SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSystemModeStart, FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeMapDownloadMasterUnlock);
}

bool SpmIf::isMapLockAvailableForSWU() {
   bool bReturn = false;
   if(_mapDownloadLockOwner == SPM_APPID_INVALID || _mapDownloadLockOwner == SPM_APPID_SWUPDATE || _mapDownloadLockOwner == 0) {
      bReturn = true;
   } 
   ETG_TRACE_USR3(("Appid:%u, isMapLockAvailableForSWU:%u",_mapDownloadLockOwner, bReturn));
   return bReturn;
}

bool SpmIf::isLockHoldBySwUpdate() {
   bool bReturn = false;
   if(_mapDownloadLockOwner == SPM_APPID_SWUPDATE) {
      bReturn = true;
   } 
   ETG_TRACE_USR3(("isLockHoldBySwUpdate:%u", bReturn));
   return bReturn;
}
 

bool SpmIf::isInStandAloneMode() {
#ifndef __SW_UPDATE_UNIT_TESTING__
   bool res =  !(getProxy()->isAvailable());
#else
   bool res = true;
#endif
   ETG_TRACE_USR1(("bIsInStandAloneMode=%u", res));
   return res;
}

ETG_I_CMD_DEFINE((reboot, "reboot"))
void SpmIf::reboot() {
   ETG_TRACE_COMP(("reboot isInStandAloneMode=%u", isInStandAloneMode()));
   ::sync();
   bool allOk=true;

   hmisettings::instance()->vStoreHMISettings();

#ifndef VARIANT_S_FTR_ENABLE_G4G
   if(!isInStandAloneMode())
   {
      ETG_TRACE_USR1(("SpmIf::reboot(): using SPM RESTART_DOWNLOAD_FINISHED"));  
      swu::LOG_INTERFACE("sendRestartSystemStart: SPM RESTART_DOWNLOAD_FINISHED");    
      SEND_REQ_TO_PROXY(SpmIf, getProxy, sendRestartSystemStart, FCSWUPD_NS_SPM_T::T_SPM_e32_RESTART_TYPE__SPM_U32_RESTART_DOWNLOAD_FINISHED);

      sleep(30);
      // todo: currently system hangs when unmounting fs ,so we do a hard reset after 30 seconds

      ETG_TRACE_FATAL(("SpmIf::reboot(): LCM did not reset the system within 30 seconds"));
   }

   // DEV_WUP_C_U8_ENTIRE_SYSTEM is no longer possible if the V850 is in bootloader mode 
   if(setV850Mode(V850_MODE_APPLICATION)!=V850_MODE_APPLICATION)
   {
      ETG_TRACE_FATAL(("SpmIf::reboot(): Can't switch V850 to application mode"));
   }

   DEV_WUP_trResetProcessorInfo rResetProcessorInfo;
   rResetProcessorInfo.u8Processor = DEV_WUP_C_U8_ENTIRE_SYSTEM;
   rResetProcessorInfo.u8ResetMode = DEV_WUP_C_U8_RESET_MODE_UNLOGGED;
   rResetProcessorInfo.u16ResetReason = DEV_WUP_C_U16_RESET_REASON_SW_DOWNLOAD;
   rResetProcessorInfo.u16ResetDurationMs = 10;

   swu::print2errmem("SpmIf::reboot calls /dev/wup to reset the system\n");
   swu::WUP_Access wupAccess;
   swu::print2errmem("SpmIf::reboot calls /dev/wup to reset the system\n");
   allOk=wupAccess.ioctl(OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR,(static_cast<tS32> ((intptr_t)&rResetProcessorInfo )) );

   // todo: if all else fails, use /proc/sysrq-trigger to reset the system but this code should never be executed
   sleep(5);
   ETG_TRACE_FATAL(("SpmIf::reboot(): hard reset needed"));
   sleep(1);
   swu::print2errmem("SpmIf::reboot uses /proc/sysrq-trigger to reboot the iMX\n");
   std::ofstream ofs("/proc/sysrq-trigger", std::ofstream::out);
   ofs << "b";
   ofs.close();
#endif
   return;
}

void SpmIf::requestLockResource(FCSWUPD_NS_SPM_T::T_e8_UpdateLockCommand reqCmd, std::string resName) {

   ETG_TRACE_COMP(("resName:%80s cmd:%u", resName.c_str(), ETG_CENUM(FCSWUPD_NS_SPM_T::T_e8_UpdateLockCommand, reqCmd))); 
   swu::LOG_INTERFACE("sendUpdateLockStart:resName:%s cmd:%u", resName.c_str(), reqCmd); 
   SEND_REQ_TO_PROXY(SpmIf, getProxy, sendUpdateLockStart, resName, reqCmd);
}

ETG_I_CMD_DEFINE((rebootToRecoveryMode, "requestRecoveryMode"))
void SpmIf::rebootToRecoveryMode() {
 
   ETG_TRACE_COMP(("SpmIf::rebootToRecoveryMode()"));

   persistData();

   swu::execCommand("/opt/bosch/base/bin/swupdatecontrol_out.out --setforcednlmagic 1");
   swu::execCommand("/opt/bosch/base/bin/swupdatecontrol_out.out -r system");
   sleep(30);
   ETG_TRACE_FATAL(("SpmIf::rebootToRecoveryMode() FAILED !!!"));
   SWU_ASSERT_RETURN_ALWAYS();
}

void SpmIf::persistConfigData() {

   if(Config::instance()->cfg_EnterFullOperation.get()) {
      Config::instance()->cfg_UpdateStartDistanceTotalizer.set(FcSwUpdCore::getIfOrDefault<DistTotalizerIf>()->getDistTotal());
      swu::LocalTime lc;
      FcSwUpdCore::getIfOrDefault<LocalTimeIf>()->getLocalTime(lc);
      Config::instance()->cfg_UpdateStartTime.set(lc.toString());
   }
}

//todo: to be checked whether it is required for requestRecoveryMode(true)
void SpmIf::persistData() {

 //Store the Data of HMI Settings before setting the recovery mode magic flag
  hmisettings::instance()->vStoreHMISettings();
  persistConfigData();
 
}

ETG_I_CMD_DEFINE((requestRecoveryMode, "requestRecoveryMode %u", tU8))
void SpmIf::requestRecoveryMode(bool recovery) {
   ETG_TRACE_COMP(("SpmIf::requestRecoveryMode(%u)", recovery));
   
   if (swu::BootChain::isRecoveryMagicSet()==recovery) {
      return;
   }
   std::string cmd="/opt/bosch/base/bin/swupdatecontrol_out.out -m ";
   cmd += recovery ? "recovery" : "normal";
   swu::execCommand(cmd);
}


ETG_I_CMD_DEFINE((requestSystemModeDl, "requestSystemModeDl %u", tU8))
void SpmIf::requestSystemModeDl(bool on) {
   ETG_TRACE_COMP(("SpmIf::requestSystemModeDl(%u)", on));
   swu::LOG_INTERFACE("requestSystemModeDl(%u)", on);
   if (on) {
#if 0
      SEND_REQ_TO_PROXY(SpmIf, getSpmProxy, sendSubStatesSet,
                        FCSWUPD_NS_SPM_T::T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_DOWNLOAD, true);
#endif
      SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSystemModeStart,
                        FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeDownload);
   }
   else {
#if 0
      SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSubStatesSet,
                        FCSWUPD_NS_SPM_T::T_SPM_e32_SubStateType__SPM_U32_SUBSTATE_DOWNLOAD, false);
#endif
      SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSystemModeStart,
                        FCSWUPD_NS_SPM_T::T_e8_StateModes__SPM_e8_SysModeDownloadExit);

   }
}


ETG_I_CMD_DEFINE((simSpmRequestSystemMode, "simSpmRequestSystemMode %u", tU8))
void SpmIf::simSpmRequestSystemMode(tU8 mode) {
   ETG_TRACE_COMP(("SpmIf::simSpmRequestSystemMode(%u)", mode));
   tenSystemMode enMode = (tenSystemMode) mode;
   requestSystemMode(enMode);
   //SEND_REQ_TO_PROXY(SpmIf, getProxy, sendSystemModeStart, (FCSWUPD_NS_SPM_T::T_e8_StateModes)mode);
}


SpmIf::V850MODE SpmIf::getV850Mode(void)
{
   static const tU8 MSG[]={0x22, 0xf1, 0x00};
   static const int  RESPONSE_LEN=7;

   swu::IncIf incIfDl(DOWNLOAD_PORT);
   if(!incIfDl.bOpen())
   {
      ETG_TRACE_ERR(("SpmIf::getV850Mode(): could not open incIfDl"));
      return V850_MODE_UNDEFINED;
   }

   std::vector<tU8> msg;
   msg.push_back(MSG[0]);
   msg.push_back(MSG[1]);
   msg.push_back(MSG[2]);
   if(!incIfDl.bSend(msg))
   {
      ETG_TRACE_ERR(("SpmIf::getV850Mode(): ERROR writing SCC_C_GET_V850_MODE_MSGID to INC socket"));
      return V850_MODE_UNDEFINED;
   };

   if(!incIfDl.bRecv(msg))
   {
      ETG_TRACE_ERR(("SpmIf::getV850Mode(): ERROR reading from INC socket"));
      return V850_MODE_UNDEFINED;
   }

   if(msg.size()!=RESPONSE_LEN)
   {
      ETG_TRACE_ERR(("SpmIf::setV850Mode(): ERROR GET_V850_MODE_MSGID invalid len=%u", msg.size()));
      return V850_MODE_UNDEFINED;
   }

   if(msg[4]==0x00)
      return V850_MODE_BOOTLOADER;
   else if(msg[4]==0x61)
      return V850_MODE_APPLICATION;
   return V850_MODE_UNDEFINED;
}



SpmIf::V850MODE SpmIf::setV850Mode(SpmIf::V850MODE Mode)
{
   static const char BL_MSG[]={0x10, 0x60};
   static const char APP_MSG[]={0x11, 0x01};

   V850MODE currmode=getV850Mode();
   if(Mode==currmode)
      return currmode;

   swu::IncIf incIfDl(DOWNLOAD_PORT);
   if(!incIfDl.bOpen())
   {
      ETG_TRACE_ERR(("SpmIf::setV850Mode(): could not open incIfDl"));
      return V850_MODE_UNDEFINED;
   }

   std::vector<tU8> msg;
   if(Mode==V850_MODE_BOOTLOADER)
   {
      msg.push_back(BL_MSG[0]);
      msg.push_back(BL_MSG[1]);
   }
   else if(Mode==V850_MODE_APPLICATION)
   {
      msg.push_back(APP_MSG[0]);
      msg.push_back(APP_MSG[1]);
   }
   else
   {
      ETG_TRACE_ERR(("SpmIf::setV850Mode(): invalid mode requested"));
      return V850_MODE_UNDEFINED;
   }

   if(!incIfDl.bSend(msg))
   {
      ETG_TRACE_ERR(("SpmIf::setV850Mode(): ERROR writing SCC_C_GET_V850_MODE_MSGID to INC socket"));
      return V850_MODE_UNDEFINED;
   };

   incIfDl.vClose();

   usleep(100000);
   currmode=getV850Mode();
   if(currmode!=Mode)
   {
      ETG_TRACE_ERR(("SpmIf::setV850Mode() V850 did not change mode\n"));
      return V850_MODE_UNDEFINED;
   }

   return currmode;
}



ETG_I_CMD_DEFINE((shutdown, "shutdown"))
// interface to shut down the system
void SpmIf::shutdown()
{
   bool     allOk=false;

   ETG_TRACE_USR4(("SpmIf::shutdown() START"));
#ifndef VARIANT_S_FTR_ENABLE_G4G
   if(setV850Mode(V850_MODE_APPLICATION)==V850_MODE_APPLICATION)
   {
      swu::print2errmem("SpmIf::shutdown calls /dev/wup to shut down the device\n");
      ETG_TRACE_USR4(("SpmIf::shutdown() V850 is in application mode - shutdown"));
#ifndef __SW_UPDATE_UNIT_TESTING__
      sleep(6);     // @todo: Currently required because of bug in V850 software. Remove when there is no longer a delay in V850 SW between notification and real shutdown.
#endif
      swu::WUP_Access wupAccess;
      // In production: Switch target off for force.dnl with option SHUTDOWN=1
      allOk=wupAccess.ioctl(OSAL_C_S32_IOCTRL_WUP_SHUTDOWN, static_cast<tS32>(DEV_WUP_SHUTDOWN_NORMAL));
   }
   else
   {
      ETG_TRACE_USR4(("SpmIf::shutdown() V850 is not in application mode - reset"));
   }

#ifndef __SW_UPDATE_UNIT_TESTING__
   sleep(60);
#endif
   ETG_TRACE_USR4(("SpmIf::shutdown() END, allOk=%u", allOk));
   if (!allOk)
   {
      reboot();
   }
#endif
}



// interface to shut down the system
ETG_I_CMD_DEFINE((resetV850IfNeeded, "resetV850IfNeeded"))
void SpmIf::resetV850IfNeeded() {
   if (Config::instance()->cfg_V850RebootNeeded.readAsBool()) {
      Config::instance()->cfg_V850RebootNeeded.setFromBool(false);
      ETG_TRACE_COMP(("reboot resetting v850 first"));
      resetV850();
      sleep(1);
   }
}



bool SpmIf::clearResetCounter(void)
{
   bool retval=false;

   ETG_TRACE_USR4(("SpmIf::clearResetCounter() START"));

   OSAL_tIODescriptor   hDevicePramResetCounter=OSAL_IOOpen("/dev/pram/reset_counter", OSAL_EN_READWRITE);
   if(hDevicePramResetCounter==OSAL_ERROR)
   {
      ETG_TRACE_ERR(("SpmIf::clearResetCounter() OSAL_IOOpen failed for /dev/pram/reset_counter"));
   }
   else
   {
      // after the device has been opened use IOControl to hand over callback pointer.
      // write the real counter
      tS32  ret=OSAL_s32IOControl(hDevicePramResetCounter, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, RESET_COUNTER_PRAM_INDEX);
      if(ret==OSAL_ERROR)
      {
         ETG_TRACE_ERR(("SpmIf::clearResetCounter() PRAM_SEEK failed for Reset Counter."));
      }
      else
      {
         tS8   count=0;

         ret=OSAL_s32IOWrite(hDevicePramResetCounter, &count, sizeof(count));
         if(ret==OSAL_ERROR || ret!=sizeof(count))
         {
            ETG_TRACE_ERR(("SpmIf::clearResetCounter() PRAM_WRITE ERROR! Error=0x%08x", ret));
         }
         else
         {
            // write the check counter
            ret=OSAL_s32IOControl(hDevicePramResetCounter, OSAL_C_S32_IOCTRL_DEV_PRAM_SEEK, RESET_COUNTER_PRAM_CHECK_INDEX);
            if(ret==OSAL_ERROR)
            {
               ETG_TRACE_ERR(("SpmIf::clearResetCounter() PRAM_SEEK failed for Reset Check Counter."));
            }
            else
            {
               tS8   check=(static_cast<tS8> (0xFF) );

               ret=OSAL_s32IOWrite(hDevicePramResetCounter, &check, sizeof(check));
               if(ret==OSAL_ERROR || ret!=sizeof(check))
               {
                  ETG_TRACE_ERR(("SpmIf::clearResetCounter() PRAM_WRITE ERROR! Error=0x%08x", ret));
               }
               else
                  retval=true;
            }
         }
      }
      OSAL_s32IOClose(hDevicePramResetCounter);
   }

   ETG_TRACE_USR4(("SpmIf::clearResetCounter() END"));
   return retval;
}



ETG_I_CMD_DEFINE((resetV850, "resetV850"))
void SpmIf::resetV850() {
   //tCString DEV_WUP = "/dev/wup";
   //tCString DEV_WUP = OSAL_C_STRING_DEVICE_WUP;
   //tU32 u32ShutdownMethod = DEV_WUP_SHUTDOWN_NORMAL ;
   ETG_TRACE_USR4(("SpmIf::resetV850()  START"));

   OSAL_s32ThreadWait(1000);
   bool allOk=true;
#if 1


   swu::execCommand("/opt/bosch/base/bin/swu_common_v850_app_out.out -reset");


#else
   DEV_WUP_trResetProcessorInfo rResetProcessorInfo =
      {
         DEV_WUP_C_U8_SYSTEM_COMMUNICATION_CONTROLLER,
         DEV_WUP_C_U8_RESET_MODE_UNLOGGED,
         DEV_WUP_C_U16_RESET_REASON_UNSPECIFIED
      };

   swu::WUP_Access wupAccess;
   allOk=wupAccess.ioctl(OSAL_C_S32_IOCTRL_WUP_RESET_PROCESSOR, (tS32)&rResetProcessorInfo );
#endif
   OSAL_s32ThreadWait(2000);


   ETG_TRACE_USR4(("SpmIf::resetV850() END, allOk=%u", allOk));

}


ETG_I_CMD_DEFINE((wupGetStartType, "wupGetStartType"))
void SpmIf::wupGetStartType() {
   ETG_TRACE_USR4(("SpmIf::wupGetStartType()  START"));

   OSAL_s32ThreadWait(1000);
   tU8 u8StartType = 0;
   swu::WUP_Access wupAccess;
   bool allOk=wupAccess.ioctl(OSAL_C_S32_IOCTRL_WUP_GET_STARTTYPE,(static_cast<tS32> ( (intptr_t)&u8StartType )));
   OSAL_s32ThreadWait(2000);


   ETG_TRACE_USR4(("SpmIf::wupGetStartType() END, allOk=%u u8StartType=%u", allOk, u8StartType));

}

ETG_I_CMD_DEFINE((simSpmSystemModeResult, "simSpmSystemModeResult"))
void SpmIf::simSpmSystemModeResult() {
   ETG_TRACE_USR4(("simSpmSystemModeResult"));
   const ::boost::shared_ptr< FCSWUPD_NS_SPM::SystemModeResult > payload(
         new FCSWUPD_NS_SPM::SystemModeResult());
#ifndef __SW_UPDATE_UNIT_TESTING__
   onSystemModeResult(getProxy(), payload);
#endif
}

ETG_I_CMD_DEFINE((simCheckForMapLock, "simCheckForMapLock"))
void SpmIf::simCheckForMapLock() {
   ETG_TRACE_USR4(("simCheckForMapLock"));
   isMapLockAvailableForSWU();
}


}
