#ifndef FCSWUPD_PARSER_IF_H
#define FCSWUPD_PARSER_IF_H
#include "util/fcswupd_types.hpp"
#include "tinyxml/tinyxml.h"

namespace fcswupdate {

class ReleaseFilterIf;
class ReleaseFilterChain;
class XMLFilterChain;

class ParserConfig
{
 public:
   ParserConfig():useEncryption(false){};
   bool useEncryption;
   
};
/** @brief Interface for parsers used in the update process.
 *
 * Implementations of this interface only have to allow two sequences of operation. The order of the calls
 * will not be changed. Only one sequence will be conducted on a ParserIf object before is is destroyed.
 *
 * 1. First sequence
 *		- Constructor
 *		- getReleaseList()
 *		- Destructor
 *
 * 2. Second sequence
 *		- Constructor
 *		- setOverallSection(overallSection), where overallSection is an element of the list returned by getReleaseList()
 *		- getReleaseXml()
 *		- getParserError()
 *		- Destructor
 */
class ParserIf {
public:

   virtual ~ParserIf() {};

   /** @brief Looks for CMS files in the given path, parses them, extracts and analyzes the payload and compiles a list
    * of <overallSections> from bosch.xml files which contain the meta-data for the current update.
    *
    * @param sourceInfo
    * @param errorCode
    * @return Each element of returned list is an <overallSection>
    */
   virtual std::list<TiXmlElement>
   getReleaseList(trSourceInfo const &sourceInfo, tenSwUpdateError *errorCode)=0;

   /** @brief  Set the current overallSection to the given one and parse it. The result can be extracted by a call
    * to ParserIf::getReleaseXml().
    *
    * @return  False if an error occurred while parsing, True otherwise.
    */
   virtual bool setOverallSection(TiXmlElement const &overallSection)=0;

   /** @brief Return a pointer to the current meta-info.
    *
    * Parser must not destroy the returned document later, custody goes to caller.
    */
   virtual TiXmlDocument *getReleaseXml(size_t id=0)=0;

   virtual tenSwUpdateError getParserError(void) const=0;

   void setSrcFilters(XMLFilterChain *srcFilters) {
      _srcFilters = srcFilters;
   }
   void setPreFilters(ReleaseFilterChain *preFilters) {
      _preFilters=preFilters;
   }
   void setPostFilters(ReleaseFilterChain *postFilters) {
      _postFilters= postFilters;
   }

   virtual bool configure(ParserConfig const &parserConfig) {
      _config=parserConfig;
      return true;
   }

protected:
   XMLFilterChain     *_srcFilters;
   ReleaseFilterChain *_preFilters;
   ReleaseFilterChain *_postFilters;
   ParserConfig _config;

   tenSwUpdateError applyPreFilters();
   tenSwUpdateError applyPostFilters();
   

};
}

#endif
