/**
 * @file fcswupd_bxmlCheck.cpp
 * @author hcg1hi
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup ai_sw_update
 * @{
 */

#include <sstream>
#include <iomanip>
#include "fcswupd_bxmlCheck.h"
#include "tinyxml/tinyxml.h"
#include "main/fcswupd_propDevMgr.h"
#include "util/fcswupd_trace.hpp"
#include "util/swu_preInit.h"
#include "util/swu_filesystem.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "FCSWUPD_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_MAIN
#include "trcGenProj/Header/fcswupd_bxmlCheck.cpp.trc.h"
#endif

namespace fcswupdate {


bool CurrentBoschXml::_simStickRemoved=false;

/**
 * Helper class for adding this file to the TTFis command list chain.
 */
static class PreInitBxmlCheck: public swu::PreInit < FcSwUpdRoot >
{
public:
   // method will be called during initialization of fcswupd
   virtual void execute()
   {
      ETG_I_REGISTER_FILE();
   }
} _preBxmlCheck;

/**
 * Trace info when the message is triggered.
 */
tVoid Msg_CurrentBoschXmlChanged::vTrace()
{
   ETG_TRACE_USR1(("Msg_CurrentBoschXmlChanged"));
}
;


/**
 * Constructor.
 */
CurrentBoschXml::CurrentBoschXml() :
_version(""),
_checksum(""),
_root(0),
_hasOtherRelease(false),
_errorCode(0)
{
   ETG_TRACE_USR1(("CurrentBoschXml CTOR"));
};

/**
 * Destructor.
 */
CurrentBoschXml::~CurrentBoschXml()
{
   ETG_TRACE_USR1(("CurrentBoschXml DTOR"));

   Msg_NotifyDevManagerChanged::vUnSubscribe(this);
   // ETG_I_UNREGISTER_FILE();
   delete _root;
   _root = 0;
};


tBool CurrentBoschXml::isValid() const {
   if (_simStickRemoved) {
      return false;
   } else {
      return _root != 0;
   }
};

tBool CurrentBoschXml::hasOtherRelease() const {
   return _hasOtherRelease;
};

int CurrentBoschXml::getErrorCode() const {
   return _errorCode;
}

/**
 * Sets the version and file checksum of the current Bosch XML and checks
 * if a matching bosch.xml is already available.
 */
tBool CurrentBoschXml::setConfig(const std::string &version, const std::string &checksum)
{
   ETG_TRACE_USR1(("CurrentBoschXml::setConfig() version=%30s checksum=%30s",
                   version.c_str(), checksum.c_str()));
   _version = version;
   _checksum = checksum;
   vProcess((Msg_NotifyDevManagerChanged *)0);
   Msg_NotifyDevManagerChanged::vSubscribe(this);
   return isValid();
}
;

/**
 * Gets the version and file checksum of the Bosch XML that shall be monitored.
 */
tBool CurrentBoschXml::getConfig(std::string &version, std::string &checksum) const
{
   version = _version;
   checksum = _checksum;
   ETG_TRACE_USR1(("CurrentBoschXml::getConfig() version=%30s checksum=%30s valid=%u",
                   version.c_str(), checksum.c_str(), isValid()));
   return isValid();
};


/**
 * Helper that looks for a bosch.xml with the specified version and checksum in all sources
 * and releases and if found, it makes a copy of the subtree of the OVERALL section.
 * must be called on change of configuration or on change of sourcesList
 */
tVoid CurrentBoschXml::lookupBoschXml(std::map < std::string, trSourceInfo > sourcesList)
{
   if (_root) {
      delete _root;
   }
   _root = 0;
   ETG_TRACE_USR1(("CurrentBoschXml::lookupBoschXml() START"));

   // iterate over all sources
   for (std::map < std::string, trSourceInfo >::iterator iterSrc = sourcesList.begin();
            iterSrc != sourcesList.end(); ++iterSrc) {
      trSourceInfo source = iterSrc->second;
      ETG_TRACE_USR1(("CurrentBoschXml::lookupBoschXml() check source name=%30s path=%s", 
                      source.name.c_str(), source.path.c_str()));
      std::list < TiXmlElement > xmlRelList;
      // get release-list of current source
      tenSwUpdateError errorCode;
      // resetting error code for next checks
      _errorCode = 0;
      FcSwUpdCore::instance()->getReleaseList(xmlRelList, source, &errorCode);

      // iterate over all releases of current source
      _hasOtherRelease=false;
      for (std::list < TiXmlElement >::iterator iterRel = xmlRelList.begin();
               iterRel != xmlRelList.end(); ++iterRel) {
         TiXmlElement &overallSection = *iterRel;
         TiXmlDocument* boschdoc = FcSwUpdCore::instance()->getReleaseXml(overallSection, errorCode);
         if(boschdoc) {
            BXmlAccess relAccess(boschdoc);
            std::string ver = relAccess.getTextFromModule("FINALNAME");
            std::string chksum = swu::getTextFromChild(&overallSection, "BXML_CHECKSUM", false, "");
            ETG_TRACE_USR1(("CurrentBoschXml::lookupBoschXml():check release ver=%30s checksum=%s",
                            ver.c_str(), chksum.c_str()));
         // check version (optionally checksum)
            if (ver == _version && (_checksum.empty() || chksum == _checksum)) {
               _root = (TiXmlElement *) overallSection.Clone();
               ETG_TRACE_USR1(("CurrentBoschXml::lookupBoschXml(): Matching release found"));
               break;
            }
            else {
               _hasOtherRelease=true;
            }
         }else {
            _hasOtherRelease=true;
         }
      }
      _errorCode = errorCode;
   }
}

/**
 * Callback triggered by PropDevMgr when update sources have changed.
 */
tVoid CurrentBoschXml::vProcess(Msg_NotifyDevManagerChanged const *pMsg) {
   //  get list of all sources
   lookupBoschXml(PropDevMgr::instance()->get()._mapSources);
   Msg_CurrentBoschXmlChanged msg(this);
   notify(msg);
}

/**
 * Only used for testing.
 * FINALNAME change is not needed since it i doing nothing.
 */
tVoid CurrentBoschXml::vProcess(Msg_CurrentBoschXmlChanged const *pMsg)
{
   ETG_TRACE_USR1(("Message %p received",pMsg));
   if (pMsg) {
      CurrentBoschXml * pCurrentBoschXml = pMsg->_currentBoschXml;
      TiXmlElement * pOverallSection = pCurrentBoschXml->getOverallXml();
      if (pCurrentBoschXml->isValid()) {
         std::string version = swu::getTextFromChild(pOverallSection, "FINALNAME", false, "");
         if (version.empty()) {
            ETG_TRACE_USR1(("Bosch.xml version not found"));
         } else {
            ETG_TRACE_USR1(("Bosch.xml version %s", version.c_str()));
         }
         std::string checksum = swu::getTextFromChild(pOverallSection, "BXML_CHECKSUM", false, "");
         if (checksum.empty()) {
            ETG_TRACE_USR1(("Bosch.xml SHA256 checksum not found"));
         } else {
            ETG_TRACE_USR1(("Bosch.xml SHA256 checksum %s", checksum.c_str()));
         }
      } else {
         ETG_TRACE_USR1(("Bosch.xml not found"));
      }
   }
};

/**
 * An instance of CurrentBoschXml is created and registers itself for change notifications.
 * This calls the vProcess method as soon as setConfig is invoked.
 */
ETG_I_CMD_DEFINE((fcswupdate::CurrentBoschXml::testGetBoschXml, "testGetBoschXml %s", ETG_I_STRING))
tVoid CurrentBoschXml::testGetBoschXml(const char *version)
{
   CurrentBoschXml *cbxml=CurrentBoschXml::instance();
   ETG_TRACE_USR1(("testGetBoschXml with %s called",version));
   Msg_CurrentBoschXmlChanged::vSubscribe(cbxml);
   cbxml->setConfig(version, "");
   Msg_CurrentBoschXmlChanged::vUnSubscribe(cbxml);
};

/**
 * Calls the function calculateSHA256Hash and converts the return hash string
 * to hexadecimal digits.
 */
ETG_I_CMD_DEFINE((fcswupdate::CurrentBoschXml::testSHA256Hash, "testSHA256Hash %s", ETG_I_STRING))
tVoid CurrentBoschXml::testSHA256Hash(const char *text)
{
   std::string hash;
   ETG_TRACE_USR1(("testSHA256Hash with %s called",text));
   std::stringstream buffer;
   buffer << text;
   std::string s = buffer.str();
   swu::calculateSHA256Hash(s, hash);
   // convert hash to hexadecimal string
   std::stringstream ss;
   for (uint32_t i = 0; i < hash.size(); i++) {    // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      ss << std::setfill('0') << std::hex << std::setw(2) << static_cast < int >(hash[i]);
   }
   ETG_TRACE_USR1(("swu::calculateSHA256Hash returns [%s]",ss.str().c_str()));
};

ETG_I_CMD_DEFINE((fcswupdate::CurrentBoschXml::simStickRemoved, "simStickRemoved %u", tU8))
void CurrentBoschXml::simStickRemoved(uint8 removed) {
   ETG_TRACE_COMP(("CurrentBoschXml::simStickRemoved(removed%u)", removed));
   _simStickRemoved=removed;
   Msg_CurrentBoschXmlChanged *msg=new Msg_CurrentBoschXmlChanged(CurrentBoschXml::instance());
   msg->bNotifyLater();
}

}
