#include <cstdio>
#include <cstring>
#include <ctime>
#include <fstream>
#include <sstream>
#include <sys/stat.h>
#include <sys/types.h>

#include "config/fcswupd_config.hpp"
#include "ctrl/fcswupd_ctrl.h"
#include "ctrl/fcswupd_ctrlEngineering.hpp"
#include "ctrl/fcswupd_ctrlProgress.h"
#include "fcswupdatesrv/FcSwUpdateSrvJson.h"
#include "main/fcswupd_bxmlCheck.h"
#include "main/fcswupd_component.h"
#include "main/fcswupd_dataStore.h"
#include "main/fcswupd_history.h"
#include "main/fcswupd_mainMessages.hpp"
#include "main/fcswupd_parserIf.h"
#include "main/fcswupd_propProgress.h"
#include "main/fcswupd_spmIf.h"
#include "main/fcswupd_spmIf.h"
#include "main/fcswupd_systemData.h"
#include "main/fcswupd_specialRelease.h"
#include "main/fcswupd_resourcemanager.h"
#include "tinyxml/tinyxml.h"

#include "util/fcswupd_types.hpp"
#include "util/fcswupd_globalLog.h"
#include "util/swu_asfUtil.hpp"
#include "util/swu_boardId.h"
#include "util/swu_bootChain.h"
#include "util/swu_constants.hpp"
#include "util/swu_filesystem.h"
#include "util/swu_execCommand.h"
#include "util/swu_member.hpp"
#include "util/swu_memberList.hpp"
#include "util/swu_preInit.h"
#include "util/swu_singleton.hpp"
#include "util/swu_util.hpp"
#include "util/swu_xmlDoc.h"
#include "util/swu_globallog.h"

#define ETG_I_FILE_PREFIX fcswupdate::CampaignManager::instance()->getCtrlInstance()->

#include "util/fcswupd_trace.hpp"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX "FCSWUPD_CTRL_"
#define ETG_I_TRACE_CHANNEL    TR_TTFIS_FCSWUPDATE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FCSWUPDATE_CTRL

#include "trcGenProj/Header/fcswupd_ctrl.cpp.trc.h"
#endif

#define FCSWUPD_TESTING_CONTROLL

#define FCSWUPD_SYSTEM_ERROR_RETURN(a)  if (!(a)) {swu::traceAssert(__FILE__, __LINE__); traceState(); vEnterStateSystemError(); return;}
#define FCSWUPD_SYSTEM_ERROR_RETURN_ALWAYS() swu::traceAssert(__FILE__, __LINE__); traceState(); vEnterStateSystemError(); return
#define FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(a)  if (!(a)) {swu::traceAssert(__FILE__, __LINE__); traceState(); vEnterStateSystemError(); return false;}
#define FCSWUPD_SYSTEM_ERROR_RETURN_VAL(a, val)  if (!(a)) {swu::traceAssert(__FILE__, __LINE__); traceState(); vEnterStateSystemError(); return val;}
#define FCSWUPD_SYSTEM_ERROR_RETURN_FALSE_ALWAYS() swu::traceAssert(__FILE__, __LINE__); traceState(); vEnterStateSystemError(); return false

#define FCSWUPD_SPOOFING_RETURN_VAL(val) if(Ctrl::_isttifsSpoofing) { return val; }
#define FCSWUPD_SPOOFING_RETURN() if(Ctrl::_isttifsSpoofing) { return; }

namespace fcswupdate {

static  bool xmlCksumToAsf(TiXmlElement *xmlCkSum, FCSWUPD_NS_SWUCORE_T::trCkSum *asfCksum);


static class PreInitCtrl: public swu::PreInit<FcSwUpdRoot> {
public:
   // method will be called during initialization of fcswupd, before creating ctrl.
   virtual void execute() {
      ETG_I_REGISTER_FILE();
   }
} _preInitCtrl;



const std::string Ctrl::k_UPDATE_STATE_RUNNING = "running";
const std::string Ctrl::k_UPDATE_STATE_NEXT_RELEASE = "release_done";
const std::string Ctrl::k_UPDATE_STATE_RESULT = "result";

tVoid Msg_NotifyStartModInstall::vTrace() {
   ETG_TRACE_USR1((" Msg_NotifyStartModInstall()"));
};

void ProgressInformation::traceState() {
   ETG_TRACE_COMP(("  ProgressInformation.deviceName=%s", deviceName.c_str()));
   ETG_TRACE_COMP(("  ProgressInformation.moduleName=%s", moduleName.c_str()));
   ETG_TRACE_COMP(("  ProgressInformation.submoduleName=%s", subModuleName.c_str()));
   ETG_TRACE_COMP(("  ProgressInformation.retries=%u", retries));
   ETG_TRACE_COMP(("  ProgressInformation.curRetries=%u", curRetries));
   ETG_TRACE_COMP(("  ProgressInformation.numAll=%u", numAll));
   ETG_TRACE_COMP(("  ProgressInformation.numComplete=%u", numComplete));
   ETG_TRACE_COMP(("  ProgressInformation.numNotApplicable=%u", numNotApplicable));
   ETG_TRACE_COMP(("  ProgressInformation.numFailed=%u", numFailed));
   ETG_TRACE_COMP(("  ProgressInformation.numRemaining=%u", numRemaining));
   ETG_TRACE_COMP(("  ProgressInformation.percentRelease=%u", percentRelease));
   ETG_TRACE_COMP(("  ProgressInformation.percentReleaseForPhase=%u", percentReleaseForPhase));
   ETG_TRACE_COMP(("  ProgressInformation.percent=%u", percent));
}

CtrlAdapter::CtrlAdapter(Ctrl *pCtrl) :
   _pCtrl(pCtrl) {
   SWU_ASSERT_RETURN(_pCtrl);
   ETG_TRACE_USR1(("CtrlAdapter CTOR"));
}

CtrlAdapter::~CtrlAdapter() {
   ETG_TRACE_USR1(("CtrlAdapter DTOR"));
   _pCtrl = 0;
}

update_decision_setting_type Ctrl::_defaultDecision = update_decision_setting_type__invalid;
bool Ctrl::_sendAutoInstallResponse = false;
bool Ctrl::_isttifsSpoofing = false;

static const std::string misPendingMarkerFile="/var/opt/bosch/persistent/fcswupdate/misUpdatePending";
static const std::string misDoneMarkerFile="/var/opt/bosch/persistent/fcswupdate/misUpdateDone";
static const std::string mcpUpdateResult="/var/opt/bosch/persistent/fcswupdate/mcpupdateresult.xml";

Ctrl::Ctrl(CtrlProgressSection* ctrlProgress, tenSwUpdateMode enMode, TiXmlDocument *xmlDoc) :
   _enState(tenState_Initial),
   _enInState(tenInState_Initial),
   _enRunState(tenRunState_Initial),
   _enOutState(tenOutState_Initial),
   _enMode(enMode), 
   _xmlDoc(xmlDoc), 
   _pAdapter(NULL),
   _lastProgSectionState(CtrlProgressSection::enState_Initial), 
   _skip(false),
   _abort(false),
   _needWaitRecoveryAllowed(false),
   _numCurFailed(0),
   _enStickState(tenStickState_initial),
   _firstResult(false),
   _coreProxyAvailable(false),
   _coreSessionStarted(false),
   _startInstallSendTimeStampSec(0),
   _progSection(ctrlProgress),
   _cfg(Config::instance()),
   _updateStartTime(0)
{
   ETG_TRACE_USR4(("Ctrl:CTOR"));
   SWU_ASSERT_RETURN(_progSection);
   setState(tenState_Initial);
}

Ctrl::~Ctrl() {
   ETG_TRACE_USR1(("Ctrl DTOR START"));
   //swu::deleteAndZero(_xmlDoc);
   _memberList.deleteMember(_pAdapter);
   _progSection = 0;
   ETG_TRACE_USR1(("Ctrl DTOR END"));
}

void Ctrl::traceState() {
   ETG_TRACE_COMP(("  _enState=%u", ETG_CENUM(Ctrl::tenState, _enState)));
   ETG_TRACE_COMP(("  _enMode=%u", ETG_CENUM(Ctrl::tenSwUpdateMode, _enMode)));
   ETG_TRACE_COMP(("  mediaPath=%s", _progSection->getMediaPath().c_str()));
   ETG_TRACE_COMP(("  _skip=%u", _skip));
   ETG_TRACE_COMP(("  _abort=%u", _abort));
   ETG_TRACE_COMP(("  _curRefKey=%s", _curRefKey.c_str()));
   ETG_TRACE_COMP(("  _sendAutoInstallResponse=%u", _sendAutoInstallResponse));
   accessProgressInfo().traceState();
   _memberList.traceState(getName());
   traceCtrlStates();
}

void Ctrl::traceCtrlStates() {
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "traceCtrlStates:State:%u", ETG_CENUM(Ctrl::tenState, _enState)));
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "traceCtrlStates:InState:%u", ETG_CENUM(Ctrl::tenInState, _enInState)));
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "traceCtrlStates:RunState:%u", ETG_CENUM(Ctrl::tenRunState, _enRunState)));
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "traceCtrlStates:OutState:%u", ETG_CENUM(Ctrl::tenOutState, _enOutState)));
}




bool Ctrl::isOtaUpdate() {
   bool res= _progSection->isOtaUpdate();
   ETG_TRACE_COMP(("Ctrl::isOtaUpdate=%u", res));
   return res;
}


tVoid Ctrl::vInit() {
   ETG_TRACE_COMP(("Ctrl::vInit(_enState=%d) START", ETG_CENUM(Ctrl::tenState, _enState)));

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Initial);
   vEnterStateIn();
}

void Ctrl::vEnterStateIn() {
   SWU_ASSERT_RETURN(_progSection);
#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "vEnterStateIn"));
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Initial);
   setState(tenState_In);
   setInState(tenInState_WaitRunCommand);
   ETG_TRACE_COMP(("Ctrl::vEnterStateIn START"));

   Msg_EnterUpdateSessionResponse::vSubscribe(this);
   Msg_ExitUpdateSessionResponse::vSubscribe(this);
   Msg_ProgressUpdate::vSubscribe(this);
   Msg_StartInstallResponse::vSubscribe(this);
   Msg_AbortInstallResponse::vSubscribe(this);
   Msg_CtrlHandleError::vSubscribe(this);
   Msg_StartInstallResponseSignal::vSubscribe(this);
   Msg_NextRequest::vSubscribe(this);
   
   swu::Msg_ProxyAvailability < FCSWUPD_NS_SWUCORE::SWUCoreServiceProxy, FcSwUpdRoot >::vSubscribe(this);
   _coreProxyAvailable = FcSwUpdateComponent::instance()->getSwUpdCoreProxy()->isAvailable();
   _memberList.vInit();
   ETG_TRACE_COMP(("Ctrl::vInit: create instance of progSection START"));
   CtrlProgressSection::tenState progressSectionState=_progSection->getState();

   if (!_cfg->cfg_EnterFullOperation.get()) {
      _cfg->cfg_AllowRebootDuringInstall.set(1);
   }

   ETG_TRACE_COMP(("Ctrl::vInit: _enMode:%u", _enMode));

   if (_enMode == tenSwUpdateMode_Invalid) {
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:Resume update"));
      Msg_CurrentBoschXmlChanged::vSubscribe(this);
      ETG_TRACE_COMP(("Ctrl::vInit: Resume update"));

      // Initialize the Update Adapter
      if (_cfg->cfg_EnterEngineering.get()) {
         _cfg->cfg_CtrlRetryOnRemovedStick.set(0);
         _enMode= tenSwUpdateMode_Engineering;
         _pAdapter = new CtrlEngineering(this);
      }
      else if (_cfg->cfg_EnterRecovery.get()) {
         _enMode= tenSwUpdateMode_Emergency;
         _pAdapter = poGetProjectSpecific()->pCreateCtrlAdapter(this, _enMode);
      }
      else {
         _enMode=tenSwUpdateMode_Hmi;
         _pAdapter = poGetProjectSpecific()->pCreateCtrlAdapter(this, _enMode);
      }
      ETG_TRACE_USR4(("CTRL::call setCtrl(%p) adapter=%p", this, _pAdapter));
      FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
      _memberList.addMember(_pAdapter);

      _pAdapter->modifyConfig(_staticConfig);

      // todo: add state restoring to ctrl
      setInState(tenInState_WaitRunCommand);

      _progSection->restore(false, &_persistentConfig, &_staticConfig);
#if 0 
      // todo: what is this?, do we want to change the mode on the fly?
      if(_cfg->cfg_EnterFullOperation.get()) {
         _enMode=_progSection->getInitialMode();
      }
#endif
   } else {
#ifdef FCSWUPD_TESTING_CONTROLL      
      _xmlDoc->SaveFile("/tmp/input_control_normal_start.xml");      
#endif
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:Start installation"));
      ETG_TRACE_COMP(("Ctrl::vInit: new update"));

      if (_enMode == tenSwUpdateMode_Engineering) {
         _pAdapter = new CtrlEngineering(this);
      }
      else {
         _pAdapter = poGetProjectSpecific()->pCreateCtrlAdapter(this, _enMode);
      }
      _persistentConfig.enInitialMode.set(_enMode);

      FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
      _memberList.addMember(_pAdapter);
      ETG_TRACE_COMP(("Ctrl::vInit: call modifyConfig _persistentConfig"));
      _pAdapter->modifyConfig(_persistentConfig);
      ETG_TRACE_COMP(("Ctrl::vInit: call modifyConfig _staticConfig"));
      _pAdapter->modifyConfig(_staticConfig);
      ETG_TRACE_COMP(("Ctrl::vInit: call generateProgressSection"));
      generateProgressSection();
   }

   _cfg->cfg_CurrentMediaType.set(_progSection->getSourceType());


   ETG_TRACE_USR1(("Controll.vInit: END _coreProxyAvailable=%u", _coreProxyAvailable));
}

tVoid Ctrl::vDeInit() {
   ETG_TRACE_COMP(("Controll.vDeInit was called!"));
#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 
   _memberList.vDeInit();
   _enState = tenState_Done;
   setOutState(tenOutState_Done);
   Msg_NextRequest::vUnSubscribe(this);
}

void Ctrl::updateProgress() {

   CtrlProgressSection::Statistics::Counters const &counters= _progSection->getStatistics();
   ProgressInformation &progress=accessProgressInfo();
   progress.numAll=counters.numAll;
   progress.numComplete=counters.numOk;   // todo: check if this includes numFailed
   progress.numNotApplicable=0;
   progress.numFailed=counters.numFailed;
   progress.numRunning=counters.numRunning;
   progress.numRemaining=counters.numPending + counters.numRunning;
   progress.curRetries=_progSection->getCurRetries();
   progress.percentRelease=static_cast<uint8_t> (counters.getFraction(CtrlProgressSection::Statistics::enAll) * 100);
   progress.percentReleaseForPhase=static_cast<uint8_t> (counters.getFraction(CtrlProgressSection::Statistics::enPhase) * 100);
   progress.percent=static_cast<uint8_t> (counters.getFraction(CtrlProgressSection::Statistics::enCurrent) * 100);
   progress.estimatedUpdateTime = counters.getTimes(CtrlProgressSection::Statistics::enAll).tAll; 
   progress.estimatedUpdateTimeForPhase = counters.getTimes(CtrlProgressSection::Statistics::enPhase).tAll;
   progress.estimatedCurrentModuleUpdateTime = counters.getTimes(CtrlProgressSection::Statistics::enCurrent).tAll;
   if (_curProgItem.isValid()) {
      progress.deviceName = _curProgItem.getNameOfLevel(XmlItemAccess::enType_Device);
      progress.moduleName = _curProgItem.getNameOfLevel(XmlItemAccess::enType_Module);
      progress.subModuleName = _curProgItem.getNameOfLevel(XmlItemAccess::enType_Submodule);
      progress.refKey = _curProgItem.getRefKey();
   }
   progress.train = _progSection->getFinalNameCustomer();
   progress.source = _progSection->getMediaPath();

  if(!Config::instance()->cfg_SilentUpdate.get() ){
   notifyProgressProp();
   }
}

tVoid Ctrl::setInState(tenInState enNewInState) {
   if (_enInState != enNewInState) {
      _enInState = enNewInState;
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:setInState:%u",
                          ETG_CENUM(Ctrl::tenInState, enNewInState)));
      ETG_TRACE_COMP(("Control:setInState:%u",
                      ETG_CENUM(Ctrl::tenInState, enNewInState)));
   }
}

tVoid Ctrl::setRunState(tenRunState enNewRunState) {
   if (_enRunState != enNewRunState) {
      _enRunState = enNewRunState;
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:setRunState:%u",
                          ETG_CENUM(Ctrl::tenRunState, enNewRunState)));
      ETG_TRACE_COMP(("Control:setRunState:%u",
                      ETG_CENUM(Ctrl::tenRunState, enNewRunState)));
   }
}

tVoid Ctrl::setOutState(tenOutState enNewOutState) {
   if (_enOutState != enNewOutState) {
      _enOutState = enNewOutState;
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:setOutState:%u",
                          ETG_CENUM(Ctrl::tenOutState, enNewOutState)));
      ETG_TRACE_COMP(("Control:setOutState:%u",
                      ETG_CENUM(Ctrl::tenOutState, enNewOutState)));
   }
}


tVoid Ctrl::setState(tenState enNewState) {
   if (_enState != enNewState) {
      _enState = enNewState;
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:setState:%u",
                          ETG_CENUM(Ctrl::tenState, enNewState)));
      ETG_TRACE_COMP(("Control:setState:%u",
                      ETG_CENUM(Ctrl::tenState, enNewState)));
   }
}


tVoid Ctrl::vRun() {
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_In);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enInState == tenInState_WaitRunCommand);
   FCSWUPD_SYSTEM_ERROR_RETURN(OSAL_NULL!=_pAdapter);
   ETG_TRACE_USR1(("Controll.vRun was called!"));
   updateProgressText("", "", "", true);
   vEnterInStateWaitResources();
}



tVoid Ctrl::vProcess(Msg_ProgressSectionChanged *pMsg) {
   ETG_TRACE_USR1(("Ctrl:vProcess(Msg_ProgressSectionChanged) START"));

   CtrlProgressSection::tenState progressSectionState=_progSection->getState();
   ETG_TRACE_USR1(("Ctrl:vProcess(Msg_ProgressSectionChanged) ctrl-state=%u prog-state=%u last prog-state=%u",
                   ETG_CENUM(Ctrl::tenState, _enState), ETG_CENUM(CtrlProgressSection::tenState, progressSectionState), 
                   ETG_CENUM(CtrlProgressSection::tenState, _lastProgSectionState)));
   updateProgress();
   if (_lastProgSectionState == progressSectionState) {
      return;
   }
   _lastProgSectionState = progressSectionState;

   switch (progressSectionState) {
      case CtrlProgressSection::enState_Initial:
         // todo error: no progress running
         break;
      case CtrlProgressSection::enState_WaitStart:
      {
         FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_In);
         FCSWUPD_SYSTEM_ERROR_RETURN(_enInState == tenInState_WaitProgressSection);
         vEnterStateRun();
         
         break;
      }
      case CtrlProgressSection::enState_Running:
      {
         if (_enRunState==tenRunState_FetchItem) {
            vEnterRunStateFetchItem();
         }
         break;
      }
      case CtrlProgressSection::enState_ScomoPending:
      {
         setRunState(tenRunState_Done);
         vEnterStateOut();
         break;
      }
      case CtrlProgressSection::enState_Result:
      {         
         vEnterStateResult();
         break;
      }
      case CtrlProgressSection::enState_Reboot:
      {
         setRunState(tenRunState_Done);
         vEnterStateOut();
      }
      break;

      default:
         break;
   }

}

tVoid Msg_CtrlReady::vTrace() {
   ETG_TRACE_USR1(("Msg_CtrlReady: readyMode=%u readyAction=%u enInterval=%u"
                   "u32NumIntervals=%u line1=%50s line2=%s",
                   ETG_CENUM(tenCtrlReadyMode, _readyAction.enMode),
                   ETG_CENUM(tenCtrlReadyAction, _readyAction.enAction),
                   ETG_CENUM(tenCtrlReadyInterval, _readyAction.enInterval),
                   _readyAction.u32NumIntervals,
                   _readyAction.line1.c_str(),
                   _readyAction.line2.c_str()))
      }

tVoid Ctrl::vEnterInStateWaitProgressSection() {

#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "vEnterInStateWaitProgressSection"));
   traceCtrlStates();
#endif 

   ETG_TRACE_USR1(("Ctrl::vEnterInStateWaitProgressSection()"));
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_In);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enInState == tenInState_WaitSwUpdCoreSessionStart);
   setInState(tenInState_WaitProgressSection);

   /* now subsribe to changes of progress-section,
      earlier subsrciption lead to unwanted triggers
   */
   Msg_ProgressSectionChanged::vSubscribe(this);
   /* force check of progress-section,
      we could to this async, but then also all other subscribers would be notified
   */
   Msg_ProgressSectionChanged msg;
   vProcess(&msg);
}

tVoid Ctrl::vProcess(Msg_EnterUpdateSessionResponse *pMsg) {
   (void)pMsg;
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_In);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enInState == tenInState_WaitSwUpdCoreSessionStart);
   ETG_TRACE_COMP(("Ctrl::vProcess(Msg_EnterUpdateSessionResponse(), state %u", _enState));
   _coreSessionStarted=1;
   vEnterInStateWaitProgressSection();
}

tVoid Ctrl::vProcess(Msg_ExitUpdateSessionResponse *pMsg) {
   (void)pMsg;
   ETG_TRACE_COMP(("Ctrl::vProcess(Msg_ExitUpdateSessionResponse()"));
#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Out);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enOutState == tenOutState_WaitSwUpdCoreSessionEnd);
   vEnterOutStateWaitReleaseResources();
}

tVoid Ctrl::vProcess(Msg_ProgressUpdate *pMsg) {
   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   updateProgress();
}

tVoid Ctrl::vProcess(Msg_DisplayTextUpdate *pMsg) {
   FCSWUPD_NS_SWUCORE_T::trDisplayText const &displayText=SWU_ASF_GET_DATA(pMsg->payload, DisplayText);
   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   updateProgressText(displayText.getLine1(), displayText.getLine2(), displayText.getLuaCmd(), false);
}

//Fix to avoid dbus reply timeout error (rtc story -439901)
tVoid Ctrl::vProcess(Msg_StartInstallResponseSignal *pMsg) {

#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "StartInstallResponseSignal"));
   traceCtrlStates();
#endif
  
   //GlobalLog::logResourcesInfo("RESINFO:StartInstallResponseSignal");

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Run);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enRunState == tenRunState_WaitStartInstallResponse);

   _installTimer.stop();
   printModuleDuration();
   _curRefKey="";
   enum FCSWUPD_NS_SWUCORE_T::tenInstallResult result = pMsg->payload->getEnResult();

   ETG_TRACE_COMP(("Ctrl::vProcess(Msg_StartInstallResponseSignal(result=%u)", result));
   swu::LOG_INTERFACE("Msg_StartInstallResponseSignal(result=%u)", result);

   ProgressInformation &progress=accessProgressInfo();

   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "got StartInstallResponse from Core: result %u", result));

   if (result == FCSWUPD_NS_SWUCORE_T::tenInstallResult__OK) {
      ETG_TRACE_COMP(("Ctrl::vProcess(Msg_StartInstallResponseSignal result=OK"));
      SystemData::instance()->onModVersionChanged(_curProgItem.getName());
      updateProgress();
      onUpdateSuceed(true);
   }
   //todo: this error code is not given by SWUCore.
   //todo: whether this implementation is needed in SWUCore / removal of this in FcSwUpdate.
   else if (result == FCSWUPD_NS_SWUCORE_T::tenInstallResult__ABORD_ON_REQUEST) {
      ETG_TRACE_COMP(("Ctrl::vProcess(Msg_StartInstallResponseSignal result=ABORD_ON_REQUEST"));
      updateProgress();
      onUpdateSuceed(false);
   }
   else if (result == FCSWUPD_NS_SWUCORE_T::tenInstallResult__RETRY_RECOVERY) {
      ETG_TRACE_COMP(("Ctrl::vProcess(Msg_StartInstallResponseSignal result=RETRY_RECOVERY"));
      updateProgress();
      _curProgItem.scheduleRetry();
      _progSection->setRecoveryNeeded();
      vEnterRunStateFetchItem();
   }
   else if (result == FCSWUPD_NS_SWUCORE_T::tenInstallResult__RETRY_ALL_RECOVERY) {
      ETG_TRACE_COMP(("Ctrl::vProcess(Msg_StartInstallResponseSignal result=RETRY_ALL_RECOVERY"));
      updateProgress();
      _curProgItem.scheduleRetry();
      setRunState(tenRunState_FetchItem);
      _progSection->requestFullUpdate();
      // reinitiate progress-section for full update in recovery-mode
   }
   else {
      ETG_TRACE_COMP(("Ctrl::vProcess(Msg_StartInstallResponseSignal default handling"));
      swu::LOG_FAILURE("Msg_StartInstallResponseSignal Failed(result:%u)", result);
      /*
        add some delay here to cover the case, that the error has
        been caused by a removed stick, but have not yet been notified.
      */
      Msg_InstallFailedTimer msg;
      msg.u32Error=(uint32_t) result;
      _installFailedTimer.start(this, 5000, &msg);
   }
}

//we do not process anything here as it will be returned immediatelyafter swucore receives startInstallRequest.
//The actual processing of result is done in vProcess(Msg_StartInstallResponseSignal)
tVoid Ctrl::vProcess(Msg_StartInstallResponse *pMsg) {

   enum FCSWUPD_NS_SWUCORE_T::tenInstallResult result = pMsg->payload->getEnResult();

   ETG_TRACE_COMP(("Ctrl::vProcess(Msg_StartInstallResponse(result=%u)", result));
   swu::LOG_INTERFACE("Msg_StartInstallResponse(result=%u)", result);

}

tVoid Ctrl::vProcess(Msg_AbortInstallResponse *pMsg) {
   _abortInstallTimer.stop();
   bool abort_ok = pMsg->payload->getBOk();
   const std::string &module_id = pMsg->payload->getOutModId();
   if (abort_ok) {
      ETG_TRACE_COMP(("Ctrl::vProcess(Msg_AbortInstallResponse(kOk=true, moduleID=%s)", module_id.c_str()));
   } else {
      ETG_TRACE_COMP(("Ctrl::vProcess(Msg_AbortInstallResponse(kOk=false, moduleID=%s)", module_id.c_str()));
   }
   // Response of Abort Install Message can be ignored.
   // If Ctrl gets a response "not ok" it means the download was done
   // successfuly before Request got into Core or it can not abort.
   // True means it is aborted, but this is also reported by StartInstallResult
   // Therefore the interesting result comes with the startInstallResponse:
   // - ok
   // - abort on User Request (enum Value not added yet)
   // - error
}


tVoid Ctrl::vProcess(Ctrl::Msg_TerminateTimer *pMsg) {
   (void)pMsg;
   ETG_TRACE_COMP(("Ctrl::Msg_TerminateTimer START"));

#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Out);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enOutState==tenOutState_Terminate);


   ProgressInformation &progress=accessProgressInfo();
   bool updateSuccess = progress.numFailed ? false : true;
   /*
     todo: only change ctrlReadyAction, if it is still invalid
    */
   if (_ctrlReadyAction.enAction != enCtrlReadyAction_invalid) {
      ETG_TRACE_COMP(("Ctrl::Msg_TerminateTimer: ctrlReadyAction already set:%u", _ctrlReadyAction.enAction));
      
   } else if(_cfg->cfg_EnterFullOperation.get() && _progSection->isScomoMode()) {      

      ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: SCOMO update, action:Idle. Hanling is up to campaign-manaer"));
      _ctrlReadyAction.enMode=enCtrlReadyMode_intermediate;
      _ctrlReadyAction.enAction=enCtrlReadyAction_idle;

   } 
   else if(_cfg->cfg_EnterFullOperation.get() && _progSection->needsSwitchToRecovery()){ //condition for usb update in normal mode
      ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: USB update triggered in normal mode"));
      if(_progSection->isCompatUpdate() || (_progSection->getSpecialMode() == "ENTER_FULL_UPDATE")){
		 ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: isCompatUpdate=%u special mode=%s ", _progSection->isCompatUpdate(), _progSection->getSpecialMode().c_str()));
	    _ctrlReadyAction.enMode = enCtrlReadyMode_success; 
        _ctrlReadyAction.enAction = enCtrlReadyAction_reboot;
      }
      else{
	       ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: intermediate, reboot!"));
           _ctrlReadyAction.enMode = enCtrlReadyMode_intermediate;
           _ctrlReadyAction.enAction = enCtrlReadyAction_reboot;
      }
   }
   else if (isSpecialUpdate()) {

      ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: success, reboot!"));
      _ctrlReadyAction.enMode = enCtrlReadyMode_success;
      _ctrlReadyAction.enAction = enCtrlReadyAction_reboot;
      
   } else {
      /* 
         here the update is really over
      */
      ETG_TRACE_FATAL(("Controll:Msg_TerminateTimer: update over, but no ready-action is set!"));

      bool forceShutdown = _persistentConfig.forceShutdown.get();
      // ask adapter, parameters: success/failed, forcedShutdownRequested
      if (forceShutdown) {
         _ctrlReadyAction.enMode = updateSuccess ? enCtrlReadyMode_success : enCtrlReadyMode_failed;
         _ctrlReadyAction.enAction=enCtrlReadyAction_shutdown;
      } else  {
         _ctrlReadyAction.enMode = updateSuccess ? enCtrlReadyMode_success : enCtrlReadyMode_failed;
      }
      ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: updateSuccess=%u!", updateSuccess));
         
   }

   ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: enter DeInit"));
   setOutState(tenOutState_DeInit);
   Msg_CtrlReady *message = OSAL_NEW Msg_CtrlReady(_ctrlReadyAction);
   FCSWUPD_SYSTEM_ERROR_RETURN(message);
   message->bNotifyLater();
   message = 0;
   ETG_TRACE_COMP(("Controll:Msg_TerminateTimer: END"));

}

bool Ctrl::isSpecialUpdate() {
   bool bReturn = false; 
   do {
      if(_progSection->isCompatUpdate()) {
         bReturn = true;
         break;
      }
      if(_progSection->getSpecialMode() == "ENTER_FULL_UPDATE") {
         bReturn = true;
         break;
      }
      //todo: to check whether this is correct check.
      if(_progSection->exists()) {
         bReturn = true;
         break;
      }

   }while(0);
   return bReturn;
}


tVoid Ctrl::vProcess(Msg_InstallFailedTimer *pMsg) {
   FCSWUPD_SYSTEM_ERROR_RETURN(pMsg);
   tU32 u32Error=pMsg->u32Error;
   ETG_TRACE_SYS(("Ctrl::vProcess(Msg_InstallFailedTimer) u32Error=%u", u32Error));

   if (_curProgItem.isValid()) {
      _curProgItem.finalize(false);
   }
   if (_abort) {
      ETG_TRACE_COMP(("Ctrl::Msg_InstallFailedTimer() calling progSection->abort"));
      setRunState(tenRunState_FetchItem);
      _progSection->abort();
   } else {
      if (_curProgItem.isValid() && _cfg->cfg_CtrlRetryOnRemovedStick.get() && wasStickRemoved()) {
         ETG_TRACE_COMP(("Ctrl::Msg_InstallFailedTimer() checking for retrying"));
         //todo: check this is required.
         if (_enStickState==tenStickState_recovered) {            
            _progSection->setRecoveryNeeded();            
         } 
         _curProgItem.scheduleRetry();
         vEnterRunStateFetchItem();
      } 
      else {
         ETG_TRACE_COMP(("Ctrl::Msg_InstallFailedTimer() error state"));
         std::list< uint32_t > errors;
         errors.push_back( (uint32_t) u32Error);
         vEnterRunStateError(errors);
      }
   }
}

tVoid Ctrl::vProcess(Msg_FullOpResultSupervisionTimer *pMsg) {
   (void)pMsg;
   /*
     we started in state result in full-op-mode.
     We wait for a while to receive updateEndRequest from HMI.
     There where some problems in the HMI that cause that we never receive the UpdateEndRequest.
     To make sure, that we don't get stuck here forever, we spoof the UpdateEndRequest on timeout.
   */
   ETG_TRACE_SYS(("Ctrl::vProcess(Msg_FullOpResultSupervisionTimer)"));
   vUpdateEndRequest();
}

tVoid Ctrl::printModuleDuration() {
   if (_startInstallSendTimeStampSec && !_curRefKey.empty()) {
      tU32 duration=(OSAL_ClockGetElapsedTime()/1000) - _startInstallSendTimeStampSec;
      GlobalLog::print(std::string("**** Updating ") + _curRefKey + " took " + swu::intToString(duration) + " seconds ****");
   }
   _startInstallSendTimeStampSec=0;
}

tVoid Ctrl::vProcess(Msg_TimeOutInstall *pMsg) {
   (void)pMsg;
   ETG_TRACE_SYS(("Ctrl::vProcess(Msg_TimeOutInstall)"));
   GlobalLog::print("**** "+_curRefKey + " took more than " +
                    swu::intToString(Config::instance()->getInstallTimeoutMs()) + " milliseconds *****");
   printModuleDuration();
   bAbortAllRequest();
}

tVoid Ctrl::vProcess(Msg_TimeOutAbortInstall *pMsg) {
   (void)pMsg;
   ETG_TRACE_SYS(("Ctrl::vProcess(Msg_TimeOutAbortInstall)"));
   printModuleDuration();
   setRunState(tenRunState_Done);
   vEnterStateOut();
}

void Ctrl::vHandleError(tenErrorHandling enMode) {
   Msg_CtrlHandleError *msg = new Msg_CtrlHandleError(enMode) ;
   msg->bNotifyLater();
   msg=0;
}

tVoid Ctrl::vProcess(Msg_CtrlHandleError *msg) {

   tenErrorHandling errorHandling=msg->errorHandling;
   ETG_TRACE_USR3(("Msg_CtrlHandleError, errorHandling=%u", ETG_CENUM(tenErrorHandling, errorHandling)));

#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Run);
        
   switch(errorHandling) {
      case tenErrorHandling_skip: {
         ETG_TRACE_USR4(("Msg_CtrlHandleError mode skipping"));
         _progSection->skip(XmlItemAccess::enType_Submodule);
         _pAdapter->vOnCtrlRunning();
         vEnterRunStateFetchItem();
      }
         break;
      case tenErrorHandling_retry: {
         ETG_TRACE_USR4(("Msg_CtrlHandleError mode retrying"));
         FCSWUPD_SYSTEM_ERROR_RETURN(_curProgItem.isValid());
         _curProgItem.scheduleRetry();
         _pAdapter->vOnCtrlRunning();
         vEnterRunStateFetchItem();
      }
         break; 
      case  tenErrorHandling_abort: {
         // State Abort
         /*
           we will end up in state Result(failed).
           Only in state result the updated can be terminated.
         */
         ETG_TRACE_USR4(("Msg_CtrlHandleError mode aborting"));
         _progSection->abort();
         _abort=true;
         vEnterStateResult();
      }
       break;
      default:
         break;
   }
   // todo: let Progress be handled by progSection that will notify Ctrl via Msg_ProgSectionProgress(result)

}



tVoid Ctrl::vEnterRunStateWaitStick() {
   ETG_TRACE_COMP(("Ctrl::vEnterStateWaitStick()"));

#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Run);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enRunState==tenRunState_WaitResources);
   setRunState(tenRunState_WaitStick);
   
   Msg_CurrentBoschXmlChanged::vSubscribe(this);
   Msg_CurrentBoschXmlChanged *msg=new Msg_CurrentBoschXmlChanged(CurrentBoschXml::instance());
   msg->bNotifyLater();
}


tVoid Ctrl::vEnterOutStateWaitStickRemoved() {
   ETG_TRACE_COMP(("Ctrl::vEnterOutStateWaitStickRemoved()"));
#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Out);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enOutState==tenOutState_WaitRecoveryAllowed);
   setOutState(tenOutState_WaitStickRemoved);
   if (_persistentConfig.waitForStickRemovedOnSuccess.get() && !_cfg->cfg_EnterFullOperation.get() && 
       !_persistentConfig.needsResultOkInHmiMode.get() && !_persistentConfig.forceShutdown.get() &&
       !_progSection->isCompatUpdate()) {
      updateProgressText(_persistentConfig.waitForStickRemovedLine1.get(), 
                         _persistentConfig.waitForStickRemovedLine2.get(), "", true);

      Msg_CurrentBoschXmlChanged::vSubscribe(this);
      // create our own notification, since the stick is already gone
      Msg_CurrentBoschXmlChanged *msg=new Msg_CurrentBoschXmlChanged(CurrentBoschXml::instance());
      msg->bNotifyLater();
   }
   else {
      // skip this state and go to next one
      vEnterOutStateTerminate();
   }
   
}

bool Ctrl::wasStickRemoved() {

   FCSWUPD_SPOOFING_RETURN_VAL(false);

   if(!_cfg->cfg_EnterEngineering.get()) {
      return _enStickState==tenStickState_missing ||
         _enStickState==tenStickState_recovered;
   } else {
      return false;
   }
}

bool Ctrl::bAbortAllRequest() {
   // direction: out
   ETG_TRACE_COMP(("Ctrl::bAbortAllRequest()"));
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(_pAdapter);
   if (_enState != tenState_Run && _enOutState != tenOutState_WaitRecoveryAllowed) {
      ETG_TRACE_ERR(("vAboardRequest only valid in state running/waitAllowRecovery, we are in state %u", _enState));
      return false;
   }
   _skip = true;
   _abort = true;
   _pAdapter->vOnCtrlAborting();

   if (_enRunState==tenRunState_WaitStartInstallResponse) {

      // trying with marker file to abort Lua
      Config::instance()->cfg_SwuAbortLuaMarker.setFromBool(true);

      if (isMISRunning()){
         _progSection->abort();
         _curProgItem.finalize(true);
         setState(tenState_Out);
         setOutState(tenOutState_Terminate);
         _terminateTimer.start(this, _cfg->getRebootWaitTimeMs());
         Msg_ProgressSectionChanged::vUnSubscribe(this); //to avoid unwanted triggers
      }
      else{
      //ASF is delayed for aborting the Lua
      //SEND_REQ_TO_PROXY(FcSwUpdateComponent, getSwUpdCoreProxy, sendAbortInstallRequest, _curRefKey);
        Msg_AbortInstallRequest *pMsg=new Msg_AbortInstallRequest(_curRefKey);
        pMsg->bNotifyLater();
        pMsg = nullptr;
      _abortInstallTimer.start(this, Config::getAbortInstallTimeoutMs());
      }
   } else {
      if (_enOutState == tenOutState_WaitRecoveryAllowed) {
         _needWaitRecoveryAllowed=false;
         _progSection->abort();
         updateProgress();
         Config::instance()->cfg_AllowRebootDuringInstall.set(true);
         Msg_ConfigChanged msg;
         notify(msg);
      }
     else if (_enRunState==tenRunState_WaitStartInstallResponse) {
         //SEND_REQ_TO_PROXY(FcSwUpdateComponent, getSwUpdCoreProxy, sendAbortInstallRequest, _curRefKey);
            Msg_AbortInstallRequest *pMsg=new Msg_AbortInstallRequest(_curProgItem.getRefKey());
            pMsg->bNotifyLater();
            pMsg = nullptr;
         _abortInstallTimer.start(this, Config::getAbortInstallTimeoutMs());
      }
      else {
         _progSection->abort();
         setRunState(tenRunState_FetchItem);
      }
   }

   cleanupKdsConfigFile();
   return true;
}

bool Ctrl::vUpdateEndRequest() {
   // direction: out

   ETG_TRACE_COMP(("Ctrl::vUpdateEndRequest()"));
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(_pAdapter);
   if (_enState != tenState_Result) {
      ETG_TRACE_ERR(("vUpdateEndRequest only valid in state result, we are in state %u", _enState));
      return false;
   }
   bool success=_progSection->getStatistics().numFailed ? false : true;

   // todo: send async trigger to ourself to do further handling.
   // todo: tell resource-manager to release all locks.
   // check interface via marker-file that all locks shall be released on next startup.
   // -- this works also in recovery-mode, where we have no spm
   // -- no locks will be released in current power-cycle, 
   // so nobody can accidently use the locks in this power-cycly 
   // (we have to wait for activation, whick happens on next startup) 
   // if(_cfg->cfg_EnterFullOperation.get() && _cfg->cfg_HandleMapLock.get()) {
   //    if(SpmIf::instance()->isLockHoldBySwUpdate()) {
   //       ETG_TRACE_USR2(("Lock hold by FcSwUpdate"));
   //       SystemModes::instance()->releaseLockAfterUpdate();
   //    }
   // }

   // todo: wait for stick removed if required
   _fullOpResultSupervisionTimer.stop();
  
   // some system have to wait for stick-removal, otherwise on next reboot the update will start again.
   vEnterStateOut();

   return true;
}

tVoid Ctrl::vEnterOutStateWaitRecoveryAllowed() {
   // direction: out
   ETG_TRACE_USR1(("Ctrl::vEnterOutStateWaitRecoveryAllowed()"));
#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 
   SWU_ASSERT_RETURN(_enState == tenState_Out);
   SWU_ASSERT_RETURN(_enOutState == tenOutState_WaitReleaseResources);
   setOutState(tenOutState_WaitRecoveryAllowed);
   if (!Config::instance()->cfg_AllowRebootDuringInstall.get() && _needWaitRecoveryAllowed) {
      _pAdapter->vOnCtrlWaitRecoveryAllowed();
      Msg_ConfigChanged::vSubscribe(this);
   }
   else {
      _needWaitRecoveryAllowed=false;
      vEnterOutStateWaitStickRemoved();
   }
}

tVoid Ctrl::vProcess(Msg_ConfigChanged *pMsg) {
   if (_enState==tenState_Out &&
       _enOutState==tenOutState_WaitRecoveryAllowed) {
      if (Config::instance()->cfg_AllowRebootDuringInstall.get()) {
         vEnterOutStateWaitStickRemoved();
      }
   }
}

tVoid Ctrl::vProcess(Msg_NextRequest *pMsg){
   ETG_TRACE_COMP(("Ctrl::nextRequested()"));
   nextRequested();
}

tVoid Ctrl::vEnterOutStateTerminate() {
   // direction: out
#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Out);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enOutState==tenOutState_WaitStickRemoved);
   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   ETG_TRACE_USR1(("Ctrl::vEnterOutStateTerminate()"));
   setOutState(tenOutState_Terminate);
   GlobalLog::stopLoggingFile();

   _pAdapter->vOnCtrlRebooting();
   if (_progSection->needsSwitchToRecovery()) {
      /* in case of scomo-update progress-section will only request recovery-mode 
         if it has reached the corresponding phase */
      SpmIf::instance()->requestRecoveryMode(true);
   }
   _terminateTimer.start(this, _cfg->getRebootWaitTimeMs());
}


tVoid Ctrl::vEnterRunStateError(std::list < uint32_t > const &errors) {
   ETG_TRACE_USR1(("Controll.vEnterRunStateError was called!"));

   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Run);

   //Error can be handled in ctrl separtely, don't depend on progresssection change.
   Msg_ProgressSectionChanged::vUnSubscribe(this);
   Msg_StartInstallResponse::vUnSubscribe(this); 
  
   if ((_enState == tenState_Run) && 
       (_enRunState == tenRunState_Error)) {
      ETG_TRACE_USR1(("Controll.vEnterRunStateError: already in state error, do nothing"));
      return;
   }
   /* If update stops due to removed stick before state changes to running, 
      then update is stuck at 0%, because we are not posting it
      FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Run);
   */
   setRunState(tenRunState_Error);
   ETG_TRACE_ERR(("Controll.vEnterRunStateError going into error state! Previous state: %u", _enState));
   
   _errors = errors;
   if (wasStickRemoved()) {
      _errors.clear();
      _errors.push_back((uint32_t)FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_MEDIUM_REMOVED);
   }
   Msg_NotifyCtrlError msg(_errors);
   notify(msg);

   _numCurFailed++;
   if (_numCurFailed < _persistentConfig.maxModTries.get()) {
      // automatically retry.
      vHandleError(tenErrorHandling_retry);
   } else {
      tenErrorHandling errorHandling=_pAdapter->enOnCtrlError(errors);
      if (errorHandling != tenErrorHandling_interact) {
         vHandleError(errorHandling);
      }      
   }
}

/*
when system enters into system error, it is a non-recoverable state.
need a power cycle to resume the update.
*/

tVoid Ctrl::vEnterStateSystemError() {
   ETG_TRACE_USR1(("Controll.vEnterStateSystemError was called!"));
   if (_enState != tenState_SystemError) {
      ETG_TRACE_FATAL(("Controll.vEnterStateSystemError going into (unrecoverable) system error state! Previous state: %u", _enState));
      setState(tenState_SystemError);
      _errors.clear();
      _errors.push_back((uint32_t)FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_UNKNOWN);
      Msg_NotifyCtrlError msg(_errors);
      notify(msg);

   }
}


/*
  todo: check if we can leave a marker-file for the spm to release the resources
  on next startup.
  This intelligence could be placed in the ResourceManager.
*/
tVoid Ctrl::vEnterOutStateWaitReleaseResources() {
   ETG_TRACE_USR1(("Ctrl::vEnterOutStateWaitReleaseResources()"));
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Out);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enOutState == tenOutState_WaitSwUpdCoreSessionEnd);
   setOutState(tenOutState_WaitReleaseResources);
   Msg_ResourceManagerChanged::vSubscribe(this);
   ResourceManager *resMan=ResourceManager::instance();
   // todo: configure  ResourceManager
   std::string resNameDummy="resNameDummy";
   
   //resMan->configure(ResourceManager::enReqMode_add, resNameDummy);
   resMan->applyConfig();

}

tVoid Ctrl::vProcess(swu::Msg_ProxyAvailability < FCSWUPD_NS_SWUCORE::SWUCoreServiceProxy, FcSwUpdRoot > *pMsg) {
   ETG_TRACE_USR1(("Ctrl::vProcess(Msg_ProxyAvailability< FCSWUPD_NS_SWUCORE::SWUCoreServiceProxy >* )"));
   swu::LOG_INTERFACE("SWUCoreServiceProxy:Msg_ProxyAvailability(avail=%u)", pMsg->available);

   _coreProxyAvailable = pMsg->available;

   if (_coreProxyAvailable) {
      ETG_TRACE_USR1(("Ctrl::vProcess(Msg_ProxyAvailability< FCSWUPD_NS_SWUCORE::SWUCoreServiceProxy >* ) switched to available"));
      if (_enState == tenState_In  &&
          _enInState == tenInState_WaitSwUpdCoreSessionStart) {
         SEND_REQ_TO_PROXY(FcSwUpdateComponent, getSwUpdCoreProxy, sendEnterUpdateSessionRequest);
      }
   } else {
      ETG_TRACE_USR1(("Ctrl::vProcess(Msg_ProxyAvailability< FCSWUPD_NS_SWUCORE::SWUCoreServiceProxy >* ) switched to not available"));
   }
}

tVoid Ctrl::vEnterInStateWaitResources() {
#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:vEnterInStateWaitResources"));
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_In);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enInState == tenInState_WaitRunCommand);
   setInState(tenInState_WaitResources);

   Msg_ResourceManagerChanged::vSubscribe(this);  
   ResourceManager *resMan=ResourceManager::instance(); 
   tenReqMode enResMode;
   _pAdapter->getNeededResourceOnIn(enResMode);   
   resMan->configure(enResMode, CENTRAL_LOCK_TEXT);
   if(enResMode == _enReqMode_invalid) {
      resMan->applyConfig();
   }
}

tVoid Ctrl::vEnterInStateWaitSwUpdCoreSessionStart() {

#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:vEnterInStateWaitSwUpdCoreSessionStart"));
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_In);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enInState == tenInState_WaitResources);
   setInState(tenInState_WaitSwUpdCoreSessionStart);
   // request update-session from swUpdCore
   ETG_TRACE_USR1(("Ctrl::vEnterInStateWaitSwUpdCoreSessionStart()"));
   if (_coreProxyAvailable) {
      SEND_REQ_TO_PROXY(FcSwUpdateComponent, getSwUpdCoreProxy, sendEnterUpdateSessionRequest);
   } else {
      ETG_TRACE_USR1(("Ctrl::vEnterInStateWaitSwUpdCoreSessionStart() will not send Message yet, Core Proxy is not available. Will wait until it is available."));
   }
}

tVoid Ctrl::vEnterStateRun() {
   ETG_TRACE_USR1(("Ctrl::vEnterStateRun()"));
   Msg_CurrentBoschXmlChanged::vSubscribe(this);

#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 

   /*
     todo: check handling of first 
   */

   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   bool first;
   if ((_enState == tenState_In) && (_enInState == tenInState_WaitProgressSection)) {
      FCSWUPD_SYSTEM_ERROR_RETURN(_progSection->getState()==CtrlProgressSection::enState_WaitStart);
      first=true;
      _enStickState=CurrentBoschXml::instance()->isValid()? tenStickState_ok: tenStickState_missing;
      _updateStartTime = time(0);
      // Write the from and to versions to the ErrMem
      GlobalLog::marker();
      GlobalLog::print(std::string("Updating from ") + _cfg->cfg_RunningSwLabel.get() + " to " + _progSection->getToLabel(), GlobalLog::logErrmem);
      GlobalLog::print(std::string("TRAIN: ") + _cfg->cfg_RunningSwTrain.get() + " -> " + _progSection->getToVersion(), GlobalLog::logErrmem);
      GlobalLog::print(std::string("CUSTOMER_VERSION: ") + _cfg->cfg_RunningCustomerVersion.get() + " -> " + _progSection->getToVersionCustomer(), GlobalLog::logErrmem);
      GlobalLog::marker();
   }
   else if ((_enState == tenState_Result) && (_enRunState == tenRunState_Done))  {
      first=false;
      _progSection->reOpen(); // progress-secdtion will go to state reboot
   } else {
      FCSWUPD_SYSTEM_ERROR_RETURN_ALWAYS();
   }

   
   setState(tenState_Run);
   setRunState(tenRunState_Initial);
   _pAdapter->vOnCtrlRunning();
   if (!_cfg->cfg_EnterFullOperation.get() &&  _cfg->cfg_ClearOutdatedOtaUpdate.readAsBool() &&
       !isOtaUpdate()) {
      ETG_TRACE_USR1(("ClearOutdatedOtaUpdate:"));
      swu::removeDirectoryRecursive("/var/opt/bosch/ota/", true);
      swu::removeDirectoryRecursive("/var/opt/bosch/persistent/fcota");
   }

   if (!_cfg->cfg_EnterFullOperation.get()) {
      // in case we got here without setting of recovery-magics, do it here:
      SpmIf::instance()->requestRecoveryMode(true);
   }

   std::string mediaPath=_progSection->getMediaPath();
   if (_cfg->cfg_WriteFcIdBeforeUpdate.readAsBool()) {
      ETG_TRACE_USR1(("Check needed update FCID before update"));
      setNewFcid();
   }

   if (_cfg->cfg_EnterFullOperation.get() && Config::instance()->cfg_CtrlForceRecovery.get() && Config::instance()->cfg_AllowRebootDuringInstall.get()) {
      ETG_TRACE_USR1(("Ctrl::vOnCtrlRunning(): force Recovery-mode, rebooting ..."));
      SpmIf::instance()->rebootToRecoveryMode();
   }

   Msg_DisplayTextUpdate::vSubscribe(this);

   _progSection->startUpdate();
   vEnterRunStateFetchItem();
   // progress-section will notify us, when everything is ready
   // download started, we are done...
}


void Ctrl::cleanupConfigFiles() {
   const char  dlt_config[]="/var/opt/bosch/persistent/dlt_configs/dlt-runtime.cfg";

   TiXmlElement const *overallSection=_progSection->getOverallSection(false);
   // If there is no overall section, something went wrong. Leave trace config alone
   if(overallSection) {
      if(!getExtraArgsValAsUInt(overallSection, "keep_trace_config"))
      {
         swu::removeFile("/var/opt/bosch/persistent/tr_dev_cnfg.cfg");
         swu::removeFile("/var/opt/bosch/persistent/tr_class_cnfg.cfg");
      }
   }
   swu::removeFile("/var/opt/bosch/persistent/.readahead");
   swu::removeFile("/var/opt/bosch/persistent/.readahead.done");
   
   if(!access(dlt_config, F_OK)) {
      // If the dlt configuration file exists, use grep to check if it contains a line with "BlockMode = X" where X is a number != 0
      if(!swu::execCommandSimple("grep", ".*BlockMode.*=.*[1-9].*", dlt_config, NULL)) {
         // Use sed to replace the line with "BlockMode = X" by "BlockMode = 0"
         swu::execCommandSimple("sed", "-i", "/.*BlockMode.*=.*[1-9].*/c\\BlockMode = 0", dlt_config, NULL);
      }
   }
}

tVoid Ctrl::vEnterStateResult() {
   ETG_TRACE_COMP(("Ctrl::vEnterStateResult() oldState=%u",
                   ETG_CENUM(Ctrl::tenState, _enState)));
#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 
    //Coverity fix for 121728
   if (_enState==tenState_Result || NULL == _progSection ) {
      return;
   }
   tU32 fullOpResultSupervisionSec=_cfg->cfg_CtrlFullOpResultSupervisionSec.get();

   FCSWUPD_SYSTEM_ERROR_RETURN((_enState==tenState_Run &&  
                                 (_enRunState==tenRunState_FetchItem || _enRunState == tenRunState_Error)) || 
                           _enState==tenState_In && _enInState==tenInState_WaitProgressSection);

   /* usecase 1: system gets stuck, if HMI never sends update-end-request.
         Start timer and spoof updateEndRequest on timeout
   */
   if ((_enState == tenState_In && _enInState == tenInState_WaitProgressSection) && 
       _cfg->cfg_EnterFullOperation.get()  &&  fullOpResultSupervisionSec) {      
      _fullOpResultSupervisionTimer.start(this, fullOpResultSupervisionSec * 1000);
   }

   bool success=_progSection->getStatistics().numFailed ? false : true;
   notifyCp(Msg_NotifyFlashingDone(success, SystemData::instance()->getTrain()));
   _pAdapter->vOnUpdateInstalled(SystemData::instance()->hasValidTrain(), SystemData::instance()->getTrain());


   Msg_CurrentBoschXmlChanged::vUnSubscribe(this);
   Msg_DisplayTextUpdate::vUnSubscribe(this);
   updateProgressText("", "", "", true);

   setState(tenState_Result);
   _firstResult=false;


   //store this value, after UpdateEndRequest app will not get the correct value
   bool isPostImageMode = _progSection->isPostImageMode();
   bool isScomoMode = _progSection->isScomoMode();

   if (!_progSection->isStoredToHistory()) {
      // we enter state result the first time, store it ...
      _firstResult=true;
      ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Ctrl::vEnterStateResult(): reached state Result: success=%u", success));
      if (!isOtaUpdate() && _progSection->isFirmwareUpdate()) {
         _cfg->cfg_MarkerUsbUpdateDone.set(1);
      }
      //_progSection->setIsStoredToHistory();
      _progSection->finalizeRelease();
      SpmIf::instance()->persistConfigData();  

      //todo: need to check for oem updaters, currently let it goes to exist history.
      // if(isScomoModeNormal()) {
      //    std::string historyName = _cfg->getPersitentDirName();
      //    historyName = historyName + "oem_history.xml";
      //    h->setHistoryFileName(historyName);         

      // } 
      //update of history & others are applicable only for firmware update.
       if(_progSection->isFirmwareUpdate()) {
          swu::ScopedPointer<History> h(History::instance());
          h->vInit();
          h->addRelease(_progSection);
          
          cleanupConfigFiles();
          if (success && _cfg->cfg_WriteFcIdAfterUpdate.readAsBool()) {
             ETG_TRACE_USR1(("Ctrl::vEnterStateResult(): Check needed update FCID after update"));
             setNewFcid();
          }
          //for postImage don't update customer version it will be done by diagnosis.
          if(!_progSection->isPostImageMode()) {
          updateCustomerVersionOnTarget(success);
          }
       }

       if(success)
       {
          updateOverallCISSWVersiononTarget(success);
       }

       if(Config::instance()->cfg_AllowMisCmcRelease.get()){
          updateMisResult(success);
       }

       _progSection->setIsStoredToHistory();
       _progSection->storeProgressFile();

   }
 
   //if it's compact update, don't leave the recovery mode.
   if (success && !_progSection->isCompatUpdate()) {
      SpmIf::instance()->requestRecoveryMode(false);
	  if(!_progSection->isUpdateDone()){
	  _progSection->setUpdateDone(1);
      }
   }
   //if it's compact update, remove the persistent file which is no longer required.
   if(_progSection->isCompatUpdate()) {
      _progSection->remove();
   }
   time_t sec = time(0) - _updateStartTime;
   std::ostringstream duration;
   if( (sec/60) < 0  || (sec/60) > 180)
   {
      duration << "(Duration unknown)";
   }
   else
   {
   duration << "(Duration " << sec/60 << "m " << sec%60 << "s)";
   }

   tU8 logOptions = GlobalLog::logErrmem | GlobalLog::logMarker;
   if(!Config::instance()->cfg_EnterFullOperation.readAsBool()) {
      logOptions = logOptions | GlobalLog::logstdOut;
   }
   
   // Write the from and to versions to the ErrMem
   GlobalLog::print(std::string("Update to ")
                    + _progSection->getToLabel()
                    + (success ? " SUCCESS" : " FAILED")
                    + duration.str(), logOptions);


   GlobalLog::print(std::string("Update ")
                    + (success ? " SUCCEEDED" : " FAILED"), GlobalLog::logErrmem);

   if(success) {
      //dump the boot chain for verification
      std::string cmd="/opt/bosch/base/bin/swupdatecontrol_out.out -d";
      swu::execCommand(cmd);
   }
 
   GlobalLog::exportLoggingFile(_progSection->getMediaPath());

   //to delete all locks on next power cycle.
   if(_cfg->cfg_HandleCentralLock.get()) {
      _cfg->cfg_SpmMarkerForLock.set(1);
   }

   // progress-section has to store if result shall be displayed in HMI-mode.
   /*
     since we have reache the result, we can decide here, what is the ctrl-ready-action
    */
   _ctrlReadyAction=CtrlReadyAction();
   _ctrlReadyAction.enMode=(success ? enCtrlReadyMode_success : enCtrlReadyMode_failed);
   _ctrlReadyAction.enAction=enCtrlReadyAction_reboot;

   bool forceShutdown=_persistentConfig.forceShutdown.get();
   if (forceShutdown) {
      // shutdown has priority, regardless of result
      ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): force shutdown is set."));
      _pAdapter->modifyCtrlResultAction(success, &_ctrlReadyAction);
      _pAdapter->vOnCtrlResult(success);
      _ctrlReadyAction.enAction=enCtrlReadyAction_shutdown;
   }
   else  if (_cfg->cfg_EnterFullOperation.get()) {
      ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): running in full-op."));
      /*
        we reach state result in full-op mode
      */
      if(isPostImageMode ){
         ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): have post-image-mode"));
         // no call to adapter, post-image mode is handled internal
         _ctrlReadyAction.enAction=enCtrlReadyAction_idle;
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));
         _staticConfig.interactOnResult = true;
      }
      else if (_progSection->isScomoModeNormal()) {
         /*
           just go to idle, campaign-manager will decide when progress-secition has to be deleted.
         */
         ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): SCOMO update, action:Idle"));
         _ctrlReadyAction.enMode=enCtrlReadyMode_intermediate;
         _ctrlReadyAction.enAction=enCtrlReadyAction_idle;  
         //result is needed, scomoIf
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));       
      }
      else if (_firstResult && _persistentConfig.needsResultAfterActivation.get() && success) {
         ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): first Result and needsResultAfterActivation"));
         //stay in ctrl, don't accept the new SW-version.
         _staticConfig.interactOnResult = true;
         /*
           we have to show a pre-result, after next power-cycle we will show the final-result only if success
           This hanlding comes from the old ota-update.
           pre-result should no longer be needed.  ota-update is now always scomo-update.
           here the result is reported before final reset to fcota. Hmi will be triggered by fcota as needed. 
           enCtrlReadyMode_intermediate - mode will not delete the progress section.          
         */
         if (_persistentConfig.needsPreResult.get()) {
            ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): needs pre-result"));
            _ctrlReadyAction.enMode=enCtrlReadyMode_intermediate;
            _ctrlReadyAction.enAction=enCtrlReadyAction_enterPreResult;
         }
         GlobalLog::print("UPDATE DONE, waiting for next power-cycle to display result", 
                          GlobalLog::logErrmem | GlobalLog::logMarker);
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));
         /*
           we have to find out, if we just go to idle or we wait next power-cycle or if we
           wait for a command to force the power-cycle.
           This should be handled by campaign-manager.
         */
      }       
      else if (_progSection->isScomoModeFirmware() && success) {
         /*
           goto intermediate state if it's SCOMO firmware.
           control will reached here after activation in recovery mode & comes in normal mode.
          */
         ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): SCOMO update Firmware, action:Idle"));
         _ctrlReadyAction.enMode=enCtrlReadyMode_intermediate;
         _ctrlReadyAction.enAction=enCtrlReadyAction_idle;  
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));
      }
      else {
         // update is relly over here
         // the update is over here, so we ask the adapter how to leave the update:         
         _pAdapter->modifyCtrlResultAction(success, &_ctrlReadyAction);
         ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): default: ask adapter"));
         _pAdapter->vOnCtrlResult(success);
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));
      }
   }
   else {
      ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): not full op"));
      if(isPostImageMode ){
         ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): have post-image-mode"));
         // no call to adapter, post-image mode is handled internal
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));
      } else if (success && 
                 _persistentConfig.needsResultOkInHmiMode.get()) {
         // we have to reset into full-op mode to display the result
		 _pAdapter->vOnCtrlResult(success);
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));
         GlobalLog::print(std::string("Resetting to Full-Operation to display result"), 
                          GlobalLog::logErrmem | GlobalLog::logMarker);
         _ctrlReadyAction.enMode=enCtrlReadyMode_intermediate;         
      } else {
         ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): default: ask adapter"));
         _pAdapter->vOnCtrlResult(success);
         notifyCp(Msg_NotifyCtrlResult(_progSection, success));
         // the update is over here, so we ask the adapter how to leave the update:
         _pAdapter->modifyCtrlResultAction(success, &_ctrlReadyAction);
         // Now we wait for UpdateEndRequest from HMI
      }
   }
   if (!_staticConfig.interactOnResult) {
      setRunState(tenRunState_Done);
      vEnterStateOut();
   }
   ETG_TRACE_COMP(("Ctrl::vEnterStateResult(): DONE"));
}


tVoid Ctrl::vEnterStateOut() {
   ETG_TRACE_USR1(("Ctrl::vEnterStateOut()"));

#ifdef FCSWUPD_TESTING_CONTROLL
   traceCtrlStates();
#endif 

   Msg_ProgressSectionChanged::vUnSubscribe(this);
    Msg_StartInstallResponse::vUnSubscribe(this);
   
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Result || 
                               (_enState==tenState_Run && _enRunState==tenRunState_Done));
   FCSWUPD_SYSTEM_ERROR_RETURN(_enOutState==tenOutState_Initial)
   setState(tenState_Out);
   setOutState(tenOutState_WaitSwUpdCoreSessionEnd);

   if (_coreSessionStarted) {
      _coreSessionStarted=false;
      SEND_REQ_TO_PROXY(FcSwUpdateComponent, getSwUpdCoreProxy, sendExitUpdateSessionRequest);
   } else {
      // fake the message to get always the same flow
      const ::boost::shared_ptr< FCSWUPD_NS_SWUCORE::ExitUpdateSessionResponse > 
         payload(new FCSWUPD_NS_SWUCORE::ExitUpdateSessionResponse());
      Msg_ExitUpdateSessionResponse *msg=
         new Msg_ExitUpdateSessionResponse(FcSwUpdateComponent::instance()->getSwUpdCoreProxy(), payload);
      msg->bNotifyLater();
      msg=0;
   }
}

bool Ctrl::getPredefinedKvPairs(std::map< std::string, std::string > &kvPairs) {

   tU16 fcidStick=0;
   char hexstr[4];

   fcidStick=getFcIdFromReleaseSection(_progSection->getReleaseSection());
   ETG_TRACE_USR4(("Ctrl::getPredefinedKvPairs(): FcidStick %.4X", fcidStick));
   if (fcidStick) {
      sprintf(hexstr,"%.4X",fcidStick);
      std::string fcidStickStr(hexstr);
      kvPairs["FcidStick"]= fcidStickStr;
      ETG_TRACE_USR4(("Ctrl::getPredefinedKvPairs(): FcidStick 0x%s", fcidStickStr.c_str()));
   }

   tU16 fcidTarget=0;
   SystemData::instance()->getFCID(fcidTarget);
   ETG_TRACE_USR4(("Ctrl::getPredefinedKvPairs(): FcidTarget  %.4X", fcidTarget));
   if(fcidTarget){
      sprintf(hexstr,"%.4X",fcidTarget);
      std::string fcidOnTargetStr(hexstr);
      kvPairs["FcidTarget"]= fcidOnTargetStr;
      ETG_TRACE_USR4(("Ctrl::getPredefinedKvPairs(): FcidTarget 0x%s", fcidOnTargetStr.c_str()));
   }

   kvPairs["installedSwuCompatIndex"]= swu::uIntToString(_cfg->cfg_SwuCompatIndex.get());

   char bid[5];
   sprintf(bid, "%04x", swu::BoardId().get());
   kvPairs["dnlid"]= bid;

   std::set<swu::ConfigItemBase *> configItems=_cfg->getItems();
   for (std::set<swu::ConfigItemBase *>::iterator iter=configItems.begin();
        iter!=configItems.end();
        ++iter) {
      swu::ConfigItemBase *item=*iter;
      //ETG_TRACE_COMP(("Ctrl:check cfg-item %s",item->getName().c_str())); 
      if (item->getExport()) {
         ETG_TRACE_COMP(("Ctrl:item exports:%s",item->readAsString().c_str()));
         kvPairs[item->getName()]= item->readAsString();
      }
   }

   return true;
}

ProgressInformation &Ctrl::accessProgressInfo() {
   return PropProgress::instance()->access().progress;
}

void Ctrl::notifyProgressProp() {
   PropProgress::instance()->iNotify(this);
}


bool xmlCksumToAsf(TiXmlElement *xmlCkSum, FCSWUPD_NS_SWUCORE_T::trCkSum *asfCksum) {
   ETG_TRACE_USR4(("CTRL: xmlCksumToAsf START"));
   swu::CCkSumElement cksum;
   if (!cksum.init(xmlCkSum)) {
      ETG_TRACE_USR4(("CTRL: xmlCksumToAsf cksum.init() failed"));

      return false;
   }



   asfCksum->setEncryptParam(cksum.GetEncryptParam());
   FCSWUPD_NS_SWUCORE_T::tenCkSumType asfCksumType=FCSWUPD_NS_SWUCORE_T::tenCkSumType__NONE;



   switch (cksum.GetType()) {
      case swu::tenDigestTypeNone:
         asfCksumType = FCSWUPD_NS_SWUCORE_T::tenCkSumType__NONE;
         break;
      case swu::tenDigestTypeMD5:
         asfCksumType = FCSWUPD_NS_SWUCORE_T::tenCkSumType__MD5;
         break;
      case swu::tenDigestTypeSHA1:
         asfCksumType = FCSWUPD_NS_SWUCORE_T::tenCkSumType__SHA1;
         break;
      case swu::tenDigestTypeSHA256:
         asfCksumType = FCSWUPD_NS_SWUCORE_T::tenCkSumType__SHA256;
         break;
      default:
         ETG_TRACE_USR4(("CTRL: xmlCksumToAsf failed, unknown type:%u", cksum.GetType()));
         return false;
   }
   asfCksum->setEnType(asfCksumType);
   asfCksum->setU64Len(cksum.GetSize());
   asfCksum->setU32SegSize(cksum.GetSeglen());
   asfCksum->setChkSumChunks(cksum.GetDigests());
   ETG_TRACE_USR4(("CTRL: xmlCksumToAsf END, cksum OK!"));
   return true;
}


bool Ctrl::sendStartInstallRequestToCore() {
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(_curProgItem.isValid());

   ETG_TRACE_USR4(("CTRL: call sendStartInstallRequestToCore for module %s",
                   _curProgItem.getScriptName().c_str()));
   ETG_TRACE_USR4(("CTRL: call sendStartInstallRequestToCore with toolspath %s",
                   _curProgItem.getToolsPath().c_str()));
   ETG_TRACE_USR4(("CTRL: call sendStartInstallRequestToCore with datapath %s",
                   _curProgItem.getDataPath().c_str()));
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "sendStartInstallRequestToCore: module %s",
                       _curProgItem.getScriptName().c_str()));


   FCSWUPD_NS_SWUCORE_T::trCkSum toolsCksum;
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(xmlCksumToAsf(_curProgItem.getToolsChecksum(), &toolsCksum));

   FCSWUPD_NS_SWUCORE_T::trCkSum dataFileCksum;
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(xmlCksumToAsf(_curProgItem.getDataChecksum(), &dataFileCksum));

   std::map<std::string, std::string> kvPairs;
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(getPredefinedKvPairs(kvPairs));
   _curProgItem.getExtraArgs(kvPairs);

   ::std::vector< FCSWUPD_NS_SWUCORE_T::trKeyValuePair > kv_vector;
   for (std::map< std::string, std::string >::const_iterator it = kvPairs.begin(); it != kvPairs.end(); ++it) {
      const std::string key = it->first;
      std::string value = it->second;
      ETG_TRACE_USR4(("Extra Arg Key  : %s", key.c_str()));
      swu::purifyString(value);
      ETG_TRACE_USR4(("Extra Arg Value: %s", value.c_str()));
      kv_vector.push_back( FCSWUPD_NS_SWUCORE_T::trKeyValuePair(key, value));
   }
   ETG_TRACE_USR4(("CTRL: call sendStartInstallRequestToCore: pass to ASF START"));
   swu::LOG_INTERFACE("sendStartInstallRequest module:%s", _curProgItem.getScriptName().c_str());

   SEND_REQ_TO_PROXY(FcSwUpdateComponent,
                     getSwUpdCoreProxy,
                     sendStartInstallRequest,
                     _curProgItem.getScriptName(),
                     _curProgItem.getToolsPath(),
                     "",   // FIXME: Change prototype since _logPath is no longer needed
                     toolsCksum,
                     _curProgItem.getDataPath(),
                     dataFileCksum,
                     kv_vector);
   ETG_TRACE_USR4(("CTRL: call sendStartInstallRequestToCore: pass to ASF DONE"));

   _installTimer.start(this, Config::getInstallTimeoutMs());
   return true;

   /*

     Handling of SwuCompatIndex:
     the installedSwuCompatIndex is hardcoded in fcSwUpdate.
     This has the advantage that no assumptions of the filesystem are to be taken,
     it works for full operation and recovery mode.
     The value is forwarded as kv-pair "installedSwuCompatIndex" to the lua-scripting.
     For old versions of fcswupdate that do not provide this value, the
     tooling will default to installedSwuCompatIndex=0

     The neededSwuCompatIndex is received as EXTRA_ARG "neededSwuCompatIndex" from the bosch.xml
     So this value is forwarded to the lua-scripting without any changes in the c-code

     Tooling:
     The new submodule imx_main_bootloader has always to be installed first. (defined by position in bosch.xml)
     The new updater imx_main_bootloader.lua updates the bootchain.
     Afterwards it checks if neededSwuCompatIndex > installedSwuCompatIndex.
     In this case, the bootchain and the rest of the modules are proceed. Most of them do nothing
     but the v850 copies the new firmware into persistent and brings v850 into bootloader mode.
     Then the target restarts. In recovery the v850 is update by the firmware in the persistent partition and
     restarts the whole update.

     Hint:
     This solution works in ctrl while we alway delete all progress-information after restart.
     If this is no longer done, ctrl would have to store in progress-section that a reset due to wrong swuCompatIndex
     is expected.

   */
}

void Ctrl::vEnterRunStateWaitStartInstallResponse() {

#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:vEnterRunStateWaitStartInstallResponse"));
   traceCtrlStates();
#endif 
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Run);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enRunState == tenRunState_WaitStick);
   
   if (sendStartInstallRequest()) {
      setRunState(tenRunState_WaitStartInstallResponse);
   } else {
      ETG_TRACE_FATAL(("Ctrl::startNextUpdate() sendStartInstallRequest failed"));
      vEnterStateSystemError();
   }
}


bool Ctrl::sendStartInstallRequest() {

   FCSWUPD_SPOOFING_RETURN_VAL(true);

   ETG_TRACE_USR4(("Ctrl::sendStartInstallRequest() START"));
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(_pAdapter);

   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(_enState == tenState_Run);
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(_curProgItem.isValid());

   bool res=false;
   do {
      if (_curRefKey != _curProgItem.getRefKey()) {
         _curRefKey=_curProgItem.getRefKey();
         _numCurFailed=0;
      }
      _startInstallSendTimeStampSec=OSAL_ClockGetElapsedTime()/1000;
      GlobalLog::print(std::string("**** Start Updating ") + _curRefKey + "****");
      
      
      if (CtrlProgressSection::enState_Result == _progSection->getState()) {
         ETG_TRACE_USR4(("Ctrl::sendStartInstallRequest(): found State Result, finalizing!"));
         break;
      }

      if (_enStickState==tenStickState_recovered) {
         ETG_TRACE_USR4(("Ctrl::sendStartInstallRequest() stick recovered, set stick-state to OK"));
         _enStickState=tenStickState_ok;
      }      
      
      bool sendToCore=true;
      uint32_t result=0;
      // adapter may notify to Msg_NotifyStartModInstall()
      if (!_sendAutoInstallResponse) {
         notifyCp(Msg_NotifyStartModInstall(_curProgItem, sendToCore, result));
      }
      
      if (sendToCore && !_sendAutoInstallResponse) {
         ETG_TRACE_USR4(("Ctrl::sendStartInstallRequest(): sendToCore=true _sendAutoInstallResponse=%u", _sendAutoInstallResponse));
         
         GlobalLog::startLoggingFile();
         sendStartInstallRequestToCore();
      } else {
         ETG_TRACE_USR4(("Ctrl::module has been handeled by Adapter: %s", _curProgItem.getName().c_str()));
         /* registering of new version in normally done by installer, 
            since we dont call the scripting we have to do it here
         */
         if (!result && _curProgItem.getLevel()==XmlItemAccess::enType_Submodule) {
            InstalledModInfo installedModInfo(_curProgItem.getName(), false);
            installedModInfo.setVal("version", _curProgItem.getVersion());
            installedModInfo.write();
         }
         const ::boost::shared_ptr< FCSWUPD_NS_SWUCORE::StartInstallResponseSignal > payload(new FCSWUPD_NS_SWUCORE::StartInstallResponseSignal());
         payload->setEnResult((FCSWUPD_NS_SWUCORE_T::tenInstallResult)result);
         Msg_StartInstallResponseSignal *pMsg = new Msg_StartInstallResponseSignal(FcSwUpdateComponent::instance()->getSwUpdCoreProxy(), payload);
         FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(pMsg);
         ETG_TRACE_COMP(("Controll:simulated StartInstallResponse was called!"));
         pMsg->bNotifyLater();
         pMsg = 0;
      }
      res=true;
   } while(0);
   
   

   ETG_TRACE_USR4(("Ctrl::sendStartInstallRequest() END"));

   return true;
}


tVoid Ctrl::generateProgressSection() {

   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   update_decision_setting_type defaultSetting = _defaultDecision;
   if (_defaultDecision == update_decision_setting_type__invalid) {
      defaultSetting = _staticConfig.defaultDecision;
   }
   //   _progSection->create(_enMode, _xmlDoc, defaultSetting, _cfg->cfg_CtrlCreateWithPersistentProgress.get());
   _progSection->create(_xmlDoc, &_persistentConfig, &_staticConfig);

   ETG_TRACE_USR1(("Ctrl::generateProgressSection() END"));
}


tVoid Ctrl::onUpdateSuceed(bool succeeded, bool stopUpdate) {
   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Run);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enRunState == tenRunState_WaitStartInstallResponse);
   //FCSWUPD_SYSTEM_ERROR_RETURN(_enState == tenState_Run);
   FCSWUPD_SYSTEM_ERROR_RETURN(_curProgItem.isValid());
   ETG_TRACE_USR1(("Ctrl::onUpdateSuceed(succeeded=%u stopUpdate=%u): module %s",
                   succeeded, stopUpdate, _curProgItem.getName().c_str()));
   if (_progSection->isLastOf(XmlItemAccess::enType_Device)) {
      ETG_TRACE_USR1(("Ctrl::onUpdateSuceed(): module %s was last of device", _curProgItem.getName().c_str()));
      _pAdapter->vOnDeviceSucceed(_curProgItem.getLongNameOfLevel(XmlItemAccess::enType_Device), succeeded);
   }
   _curProgItem.finalize(succeeded);

   if (_enStickState==tenStickState_missing && !_cfg->cfg_CtrlRetryOnRemovedStick.get()) {
      std::list < uint32_t > errors_;
      errors_.push_back((uint32_t)FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_MEDIUM_REMOVED);
      vEnterRunStateError(_errors);
   }


   if (!stopUpdate) {
      vEnterRunStateFetchItem();
  }
}



void Ctrl::vProcess(Msg_ResourceManagerChanged *msg) {
   ETG_TRACE_USR4(("Ctrl:Msg_ResourceManagerChanged"));

#ifdef FCSWUPD_TESTING_CONTROLL   
   traceCtrlStates();
#endif 

   if (ResourceManager::instance()->isReady()) {
      Msg_ResourceManagerChanged::vUnSubscribe(this);
      if(_enState == tenState_In && (_enInState == tenInState_WaitResources)) {
         vEnterInStateWaitSwUpdCoreSessionStart();
      } else if (_enState==tenState_Run && (_enRunState==tenRunState_WaitResources)) {
         vEnterRunStateWaitStick();
      } else if (_enState==tenState_Out && (_enOutState==tenOutState_WaitReleaseResources)) {
         vEnterOutStateWaitRecoveryAllowed();
      }
   }
}



void Ctrl::vEnterRunStateWaitResources() {
#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:vEnterRunStateWaitResources"));
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Run);
   FCSWUPD_SYSTEM_ERROR_RETURN(_enRunState==tenRunState_FetchItem);
   FCSWUPD_SYSTEM_ERROR_RETURN(_pAdapter);
   setRunState(tenRunState_WaitResources);

   Msg_ResourceManagerChanged::vSubscribe(this);   
   ResourceManager *resMan=ResourceManager::instance();
   std::set<std::string> res;
   tenReqMode enResMode;

   _pAdapter->getNeededResourcesOnRun(enResMode, res);   
   resMan->configure(enResMode, res);
   resMan->applyConfig();
}

void Ctrl::vEnterRunStateFetchItem() {

#ifdef FCSWUPD_TESTING_CONTROLL
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "Control:vEnterRunStateFetchItem"));
   traceCtrlStates();
#endif 

   FCSWUPD_SYSTEM_ERROR_RETURN(_enState==tenState_Run);   
   setRunState(tenRunState_FetchItem);
   if (startNextUpdate()) {
      vEnterRunStateWaitResources();
   }
}


bool Ctrl::startNextUpdate() {
   ETG_TRACE_USR1(("Ctrl::startNextUpdate() START: _enState=%u", _enState));
   bool res=false;
   do {
      if (_enState != tenState_Run) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate() ctrl not in state tenState_Run, state=%u", _enState));
         FCSWUPD_SYSTEM_ERROR_RETURN_FALSE_ALWAYS();
      }
      if (_enRunState != tenRunState_FetchItem) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate() ctrl not in run-state tenRunState_FetchItem, state=%u", _enRunState));
         FCSWUPD_SYSTEM_ERROR_RETURN_FALSE_ALWAYS();
      }
      _curProgItem = _progSection->currentItem();

      if (!_curProgItem.isValid()) {
         ETG_TRACE_FATAL(("Ctrl::startNextUpdate() _curProgItem not valid, nothing sent!"));
         break;
      }

      if (_curProgItem.getState() == "running") {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate() _curProgItem allready in state running, nothing sent!"));
         FCSWUPD_SYSTEM_ERROR_RETURN_FALSE_ALWAYS();
      }

      ETG_TRACE_USR1(("Ctrl::startNextUpdate_() START _abort=%u _skip=%u", _abort, _skip));
      if (_abort) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate_() aborting, nothing sent"));
         _progSection->abort();
         break;
      }
      if (_skip) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate_() skipping, nothing sent"));
         _progSection->skip(XmlItemAccess::enType_Submodule);
         break;
      }
      std::list < uint32_t > no_errors;
      Msg_NotifyCtrlError msg(no_errors);
      notify(msg);

      _curProgItem =_progSection->setNextItem();
      if (!_curProgItem.isValid()) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate() START curItem invalid"));
         break;
      }

      bool needsPostInstall=_curProgItem.needsPostInstall();
      bool isInPostInstallMode=_progSection->isInPostInstallMode();
      ETG_TRACE_USR4(("needsPostInstall=%u", needsPostInstall));
      ETG_TRACE_USR4(("isInPostInstallMode=%u", isInPostInstallMode));
      if (_curProgItem.needsPostInstall() && !_progSection->isInPostInstallMode()) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate() enter post-install phase, nothing sent"));
         // todo: leave instructions
         enterPostInstallPhase();
         // we will get notified from PropDevMgr to continue
         break;
      }


      if (_progSection->getState()==CtrlProgressSection::enState_ScomoPending) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate() progress-section in state ScomoPending, nothing sent"));
         // progress-section will notify async with new state ScomoPending
         break;
      }

      if (_progSection->needsSwitchToRecovery()) {
         ETG_TRACE_USR1(("Ctrl::startNextUpdate() found needsSwitchToRecovery()"));
         if (_progSection->isLastScomoPhase() && _progSection->isScomoModeFirmware()) {
            ETG_TRACE_USR1(("Ctrl::startNextUpdate() rebooting in recovery mode for scomo-ota update"));
            Config::instance()->cfg_AllowRebootDuringInstall.set(1);
            SpmIf::instance()->rebootToRecoveryMode();
         }
         else if (!_cfg->cfg_AllowRebootDuringInstall.get()) {
            ETG_TRACE_USR1(("Ctrl::startNextUpdate() setting _needWaitRecoveryAllowed"));
            _needWaitRecoveryAllowed=true;
         } 
         // todo: leave instructions
         setRunState(tenRunState_Done);
         vEnterStateOut();
         break;
      }



      // now we have a prog-item that we want to send.
      res=true;
      _curProgItem.start();
      ETG_TRACE_USR1(("Ctrl::startNextUpdate() calling sendStartInstallRequest"));
   } while (0);
   
   ETG_TRACE_USR1(("Ctrl::startNextUpdate() END! res=%u", res));
   return res;
}

void Ctrl::enterPostInstallPhase() {
   // copy
   ETG_TRACE_USR4(("enterPostInstallMode"));

   TiXmlElement const *overallSection=_progSection->getOverallSection();
   std::string mediaPath=_progSection->getMediaPath();
   mediaPath=swu::realPath(mediaPath);
      
   std::string dstDir=FCSWUPD_PERSISTENT_ROOT "/postInstall";
   swu::removeDirectoryRecursive(dstDir);
   swu::makeDirectoryRecursive(dstDir, 0700);
   dstDir=swu::realPath(dstDir);

   std::string cmsFile=swu::getTextFromChildOrEmpty(overallSection, "INFO_FILE");
   //make coverity happy
   (void)swu::copyFile(mediaPath + "/" + cmsFile, dstDir + "/" + cmsFile);
   std::string bxmlFile=swu::getTextFromChildOrEmpty(overallSection, "BXML_FILE");
   //make coverity happy
   (void)swu::copyFile(mediaPath + "/" + bxmlFile, dstDir + "/" + bxmlFile);
   std::string toolsContainer=swu::realPath(_curProgItem.getToolsPath());
   ETG_TRACE_USR4(("toolsContainer=%s", toolsContainer.c_str()));
   ETG_TRACE_USR4(("mediaPath=%s", mediaPath.c_str()));
   ETG_TRACE_USR4(("dstDir=%s", dstDir.c_str()));
   size_t foundPos=toolsContainer.find(mediaPath);
      
   std::string toolsContainerDst=dstDir;
   if (foundPos != std::string::npos) {
      ETG_TRACE_USR4(("foundPos=%u", foundPos));
      std::string subPath=toolsContainer.substr(foundPos + mediaPath.size());
      ETG_TRACE_USR4(("subPath=%s", subPath.c_str()));
      toolsContainerDst += "/" + subPath;
   }
   ETG_TRACE_USR4(("toolsContainerDst=%s", toolsContainerDst.c_str()));
   // create dst-dir
   swu::makeDirectoryRecursive(toolsContainerDst, 0700);
   // we created additional dir with the file-name, remove it again ...
   swu::removeDirectoryRecursive(toolsContainerDst);
   //make coverity happy
   (void)swu::copyFile(toolsContainer, toolsContainerDst);
      
   // simulate lost stick
   _enStickState=tenStickState_missing;
   _progSection->setPostInstallMode();
   SpecialRelease::instance()->set(dstDir + "/" + cmsFile, true);
}





bool Ctrl::vRetryFailedModules() {
   FCSWUPD_SYSTEM_ERROR_RETURN_FALSE(_pAdapter);
   if (_enState != tenState_Result) {
      return false;
   }
   vEnterStateRun();
   return true;

}


bool Ctrl::setNewFcid() {
   // gen3armmake, gen3x86make, gen4rcar, gen4lsim: conversion to 'tU16 {aka short unsigned int}' from 'unsigned int'
   // So the type of oldFcId, newFcId is changed from tU16 to uint_least32_t
   uint_least32_t oldFcId = _cfg->cfg_FcId.get();
   TiXmlElement const *releaseSection=_progSection->getReleaseSection();
   FCSWUPD_SYSTEM_ERROR_RETURN_VAL(releaseSection, false);

   uint_least32_t newFcId = getFcIdFromReleaseSection(releaseSection, "NEXT_FCID");
   if (!newFcId) {
      newFcId = getFcIdFromReleaseSection(releaseSection);
   }
   if (newFcId && (newFcId != oldFcId)) {
      _cfg->cfg_FcId.set(newFcId);
      ETG_TRACE_USR1(("Ctrl:: Updated fcid: oldFcId=%04x newFcId=%04x", oldFcId, newFcId));
      return true;
   }
   return false;
}

void Ctrl::vProcess(Msg_CurrentBoschXmlChanged *msg) {
   SWU_ASSERT_RETURN(msg->_currentBoschXml);

   bool stickValid=msg->_currentBoschXml->isValid();
   bool hasOtherRelease=msg->_currentBoschXml->hasOtherRelease();
   int errorCode = msg->_currentBoschXml->getErrorCode();
   tenStickState oldStickState=_enStickState;
   ETG_TRACE_USR1(("Ctrl::vProcess(Msg_CurrentBoschXmlChanged oldStickState=%u stickValid=%u",
                   ETG_CENUM(Ctrl::tenStickState, oldStickState), stickValid));

   // handling of initial stick-state
   if (oldStickState==tenStickState_initial) {
      if  (stickValid) {
         ETG_TRACE_USR4(("Ctrl::Msg_CurrentBoschXmlChanged(): set stick-state from initial to OK"));
         _enStickState=tenStickState_ok;
         GlobalLog::startLoggingFile();
      } else {
         ETG_TRACE_USR4(("Ctrl::Msg_CurrentBoschXmlChanged(): set stick-state from initial to missing"));
         _enStickState=tenStickState_missing;
      }
      return;
   }

   // special case, we are waiting for stick-removal to continue Out
   if (_enState == tenState_Out &&
       _enOutState==tenOutState_WaitStickRemoved && !stickValid) {
      vEnterOutStateTerminate();
      return;
   }

   // special case, our stick is missing, but we found another release.
   if  (oldStickState==tenStickState_missing && !stickValid && hasOtherRelease) {
      // forget everything about the current release and try to use the other release after reboot
      _progSection->remove();
      SpmIf::instance()->reboot();
      return;
   }

   // we cannat count on trigger  stickLost or stickRecovered because we are spoofing the msg as trigger to ourself
   if (_enState==tenState_Run && _enRunState == tenRunState_WaitStick && stickValid) {
      _pAdapter->vOnCtrlRunning();
      vEnterRunStateWaitStartInstallResponse();
   }


   bool stickLost=false;
   bool stickRecovered=false;
   // distinguish the relevant events  stickLost and stickRecovered
   if ( (oldStickState==tenStickState_ok || oldStickState==tenStickState_recovered) && !stickValid) {
      stickLost=true;
   }
   else if (oldStickState==tenStickState_missing && stickValid){
      stickRecovered=true;
   }


   if (stickLost) {
      // stick has been removed
      GlobalLog::stopLoggingFile();
      _progSection->stopProgressTimer();
      ETG_TRACE_USR4(("Ctrl::Msg_CurrentBoschXmlChanged(): set stick-state from OK to missing"));

      _enStickState=tenStickState_missing;
      
      if (_enState==tenState_Run) {
         
         // let hmi know that the stick has been remove and wait for result of current module
         std::list < uint32_t > errors;
         errors.push_back((uint32_t)FCSWUPD_NS_SWUCORE_T::tenInstallResult__ERR_MEDIUM_REMOVED);
         Msg_NotifyCtrlError msg(errors);
         notify(msg);
         
         tenErrorHandling errorHandling=_pAdapter->enOnCtrlError(errors);        
         if (errorHandling == tenErrorHandling_abort) {
            vEnterRunStateError(errors);
         }                  
      }
      return;
   }
   else if (stickRecovered) {
      GlobalLog::startLoggingFile();
      _progSection->startProgressTimer(10);
      ETG_TRACE_USR4(("Ctrl::Msg_CurrentBoschXmlChanged(): set stick-state from missing to recovered,"));
      _enStickState=tenStickState_recovered;    
      if (_enState == tenState_Run && _enRunState == tenRunState_WaitStick){
         vHandleError(tenErrorHandling_retry);
      }
      return;
   }

   if(errorCode != tenSwUpdateError_OK) {
      ETG_TRACE_USR4(("Ctrl::Msg_CurrentBoschXmlChanged(): stick reconnected with invalid content"));
      if (_enState == tenState_Run && _enRunState == tenRunState_WaitStick){
         _pAdapter->vOnCtrlReconnectInvalidStick();
      }
   }
}


void Ctrl::updateProgressTextItem(std::string const &newVal, std::string &val, bool force, bool &changed) {
   if ((force || !newVal.empty()) && (newVal != val)) {
      ETG_TRACE_COMP(("Ctrl::updateProgressTextItem() newVal='%50s' val='%s'",newVal.c_str(), val.c_str()));
      val = newVal;
      changed=true;
   }
}
void Ctrl::updateProgressText(std::string line1, std::string line2, std::string luaCmd, bool force)
{
   bool bChanged=false;
   ProgressInformation &progress=PropProgress::instance()->access().progress;
   updateProgressTextItem(line1, progress.line1, force, bChanged);
   updateProgressTextItem(line2, progress.line2, force, bChanged);
   updateProgressTextItem(luaCmd, progress.luaCmd, force, bChanged);
   if (bChanged) {
      ETG_TRACE_USR1(("Ctrl::updateProgressText: force=%u", force));
      PropProgress::instance()->iNotify(this);
   }
}


ETG_I_CMD_DEFINE((setSendAutoInstallResponse, "setSendAutoInstallResponse %u", tU8))
void Ctrl::setSendAutoInstallResponse(bool val) {
   ETG_TRACE_COMP(("Ctrl::setSendAutoInstallResponse(%d)", val));
   _sendAutoInstallResponse = val;
   CtrlProgressSection::handleRecoveryMode(!val);
}

ETG_I_CMD_DEFINE((allowRebootDuringInstall, "allowRebootDuringInstall %u", tU8))
void Ctrl::allowRebootDuringInstall(bool val) {
   ETG_TRACE_COMP(("Ctrl::allowRebootDuringInstall(%d)", val));
   if (val) {
      Config::instance()->cfg_AllowRebootDuringInstall.set(1);
   } else {
      Config::instance()->cfg_AllowRebootDuringInstall.set(0);
   }
}

ETG_I_CMD_DEFINE((setDefaultDecision, "setDefaultDecision %u", tU8))
void Ctrl::setDefaultDecision(uint8 u8Decision) {
   if (u8Decision >= (uint8) update_decision_setting_type__invalid) {
      _defaultDecision = update_decision_setting_type__invalid;
   } else {
      _defaultDecision = (update_decision_setting_type) u8Decision;
   }
   ETG_TRACE_COMP(("Ctrl::setDefaultDecision(%u)", ETG_CENUM(update_decision_setting_type, _defaultDecision)));
}

void Ctrl::updateCustomerVersionOnTarget(bool success)
{
   using namespace swu::Constants;

   ETG_TRACE_USR4(("ENTER updateCustomerVersionOnTarget" ));

   if (success) {
      // Write out customer version. The information is read from bosch.xml's
      // overall section.

      // remove old version file from SPM
      if (swu::exists(Registry::PATH_CUSTOMER_VERSION_REG)) {
         //make coverity happy
         (void)::remove(Registry::PATH_CUSTOMER_VERSION_REG.c_str());
      }
      if (swu::exists(Registry::PATH_CUSTOMER_VERSION_CFG)) {
         //make coverity happy
         (void)::remove(Registry::PATH_CUSTOMER_VERSION_CFG.c_str());
      }
      
      // make sure the registry directory exists.
      if (! swu::exists(Registry::PATH_BASE) ) {
         if (swu::makeDirectoryRecursive(Registry::PATH_BASE, Registry::MODE_BASE)) {
            ::chown(Registry::PATH_BASE.c_str(), User::root, Group::eco_osal);
         }
         else {
            ETG_TRACE_FATAL(( "Registry directory does not exist and I can't create it. %s",
                              Registry::PATH_BASE.c_str() ));
         }
      }      

      TiXmlElement const *release = _progSection->getReleaseSection();      
      if ( !release ) {
         ETG_TRACE_FATAL(("No current Bosch XML available."));
         return;
      }
      BXmlAccess releaseAccess(release);

      const std::string& custVerCfg = Registry::PATH_CUSTOMER_VERSION_CFG;
      std::string finalName = releaseAccess.getTextFromModule(XML::TAG_FINALNAME, false);
      std::string buildLabel = releaseAccess.getTextFromModule(XML::TAG_BUILD_LABEL, false);

      if (   ("" == finalName) || ("" == buildLabel) )
      {
         ETG_TRACE_FATAL(("Customer version string is incomplete. finalName='%16s' buildLabel='%s'",
                          finalName.c_str(), buildLabel.c_str()));
         return;
      }

      std::string finalNameCustomer = releaseAccess.getTextFromModule(XML::TAG_FINALNAME_CUSTOMER, false);      
      if ("" == finalNameCustomer) {
         ETG_TRACE_USR1(("No customer-version given"));
         return;
      }
      ETG_TRACE_COMP(("set new customer-version:%s", finalNameCustomer.c_str()));

      std::ofstream ofs(custVerCfg.c_str());
      if ( ! ofs ) {
         ETG_TRACE_FATAL(("Can't open file for writing: %s", custVerCfg.c_str()));
         return;
      }
      ofs << finalNameCustomer.c_str() << "_" << finalName.c_str() << "_" << buildLabel.c_str();
      ofs.close();
      ::chown(custVerCfg.c_str(), User::eco_lcm, Group::eco_osal);
      ETG_TRACE_USR4(("EXIT updateCustomerVersionOnTarget" ));
   }
}

ETG_I_CMD_DEFINE((simSpoofingStatus, "startSpoofing", ETG_I_CONST_ARG(1)))
ETG_I_CMD_DEFINE((simSpoofingStatus, "stopSpoofing", ETG_I_CONST_ARG(0)))
void Ctrl::simSpoofingStatus(tUInt bStatus) {
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "simSpoofingStatus:%u", bStatus));
   _isttifsSpoofing = bStatus;
}

ETG_I_CMD_DEFINE((simRunStateTransistion, "simRunStateTransistion_Initial", ETG_I_CONST_ARG(1)))
ETG_I_CMD_DEFINE((simRunStateTransistion, "simRunStateTransistion_FetchItem", ETG_I_CONST_ARG(2)))
ETG_I_CMD_DEFINE((simRunStateTransistion, "simRunStateTransistion_WaitResources", ETG_I_CONST_ARG(3)))
ETG_I_CMD_DEFINE((simRunStateTransistion, "simRunStateTransistion_WaitStick", ETG_I_CONST_ARG(4)))
ETG_I_CMD_DEFINE((simRunStateTransistion, "simRunStateTransistion_WaitStartInstallResponse", ETG_I_CONST_ARG(5)))
ETG_I_CMD_DEFINE((simRunStateTransistion, "simRunStateTransistion_Done", ETG_I_CONST_ARG(6)))
ETG_I_CMD_DEFINE((simRunStateTransistion, "simRunStateTransistion_Error", ETG_I_CONST_ARG(7)))
void Ctrl::simRunStateTransistion(tUInt state) {
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "simRunStateTransistion - START"));
   
   tenRunState enState = (tenRunState) state;
   switch(state) {
      case tenRunState_Initial:
         break;
      case tenRunState_FetchItem:
         vEnterRunStateFetchItem();
         break;
      case tenRunState_WaitResources:
         vEnterRunStateWaitResources();
         break;
      case tenRunState_WaitStick:
         vEnterRunStateWaitStick();
         break;
      case tenRunState_WaitStartInstallResponse:
         vEnterRunStateWaitStartInstallResponse();
         break;
      case tenRunState_Done:
         vEnterStateResult();
         break;
      case tenRunState_Error: {
         std::list< uint32_t > errors;
         vEnterRunStateError(errors);
      }
         break;
      default:
         break;
   }   
   ETG_TRACE_COMP_CLS((TR_CLASS_FCSWUPDATE_CUST, "simRunStateTransistion - END"));
}

void Ctrl::updateMisResult(bool success)
{
	bool isPostImageMode = _progSection->isPostImageMode();
	bool isEngineeringMode = Config::instance()->cfg_EnterEngineering.readAsBool();
	bool isAllowMisCmcRelease = checkAllowMisCmcRelease();
	
	ETG_TRACE_USR1(("Ctrl::updateMisResult success=%u isAllowMisCmcRelease=%u isPostImageMode=%u isEngineeringMode=%u",
                   success, isAllowMisCmcRelease, isPostImageMode, isEngineeringMode ));
    
	if( isAllowMisCmcRelease )
	{
		if( success )
		{
			if( (!isPostImageMode) && (!isEngineeringMode) )
			{
				bool misDoneMarkerFilePresent = swu::exists(misDoneMarkerFile);
				ETG_TRACE_USR1(("Ctrl::updateMisResult misDoneMarkerFilePresent=%u", misDoneMarkerFilePresent));
				if( misDoneMarkerFilePresent )
				{
					(void)::remove(misDoneMarkerFile.c_str());
					ETG_TRACE_USR1(("Ctrl::updateMisResult: misDoneMarkerFile is removed"));
				}
				else
				{
					createMisPendingMarkerFile(misPendingMarkerFile);
				}
			}
			else if( (isPostImageMode) || (isEngineeringMode)  )
			{
				(void)::remove(misDoneMarkerFile.c_str());
				(void)::remove(misPendingMarkerFile.c_str());
				ETG_TRACE_USR1(("Ctrl::updateMisResult:: misDoneMarkerFile is removed"));
				ETG_TRACE_USR1(("Ctrl::updateMisResult:: misPendingMarkerFile is removed"));
			}
		}
		else if( !success )
		{
			(void)::remove(misDoneMarkerFile.c_str());
			ETG_TRACE_USR1(("Ctrl::updateMisResult:: misDoneMarkerFile is removed"));
		}
	}
}

bool Ctrl::checkAllowMisCmcRelease()
{
   bool hasMisCmcRelease = false;
   TiXmlElement const *overallSection=_progSection->getOverallSection();
   ETG_TRACE_USR1(("START Ctrl::checkAllowMisCmcRelease"));
   if(overallSection) {
      ETG_TRACE_USR1(("Ctrl::checkAllowMisCmcRelease overall section is present")); 
      if(getExtraArgsValAsUInt(overallSection, "allow_mis_cmc_release"))
      {
           hasMisCmcRelease = true;
           ETG_TRACE_USR1(("Ctrl::checkAllowMisCmcRelease allow_mis_cmc_release is present"));
      }
    }
   ETG_TRACE_USR1(("END Ctrl::checkAllowMisCmcRelease"));
   return hasMisCmcRelease;
}

void Ctrl::createMisPendingMarkerFile(const std::string misPendingMarkerFile) {
   if (swu::writeFile(" ",misPendingMarkerFile)) {
	 ETG_TRACE_USR1(("ctrl::createMisPendingMarkerFile(): misPendingMarkerFile created"));
   }else {
	 ETG_TRACE_ERR(("ctrl::createMisPendingMarkerFile(): misPendingMarkerFile creation failed"));
   }      
}

bool Ctrl::getMisUpdateResult()
{
   bool misUpdateStatus = false;
   if( swu::exists(misDoneMarkerFile) ){
         ETG_TRACE_USR1(("Ctrl::getMisUpdateResult MIS update is success"));
         misUpdateStatus = true;
   }
   else
   {
         ETG_TRACE_USR1(("Ctrl::getMisUpdateResult MIS update is failed"));
         misUpdateStatus = false;
   }
   return misUpdateStatus;
}

bool Ctrl::getMcpUpdateResult()
{
   bool mcpUpdateStatus = false;
   if(!swu::exists(mcpUpdateResult))
   {
         ETG_TRACE_USR4(("Ctrl::getMcpUpdateResult mcpUpdateResult file not exists"));
         return mcpUpdateStatus;
   }
   
   std::string line;
   ifstream inFile;
   inFile.open(mcpUpdateResult.c_str());

   if(!(inFile.is_open())) {
         ETG_TRACE_USR4(("Ctrl::getMcpUpdateResult read Unable to open %s", mcpUpdateResult.c_str()));
         return mcpUpdateStatus;
   }

   while(getline(inFile, line))
   {
         if (line.find("success") != std::string::npos) {
                mcpUpdateStatus = true;
                ETG_TRACE_USR4(("Ctrl::getMcpUpdateResult mcpUpdateStatus : %d", mcpUpdateStatus));
                ETG_TRACE_USR4(("Ctrl::getMcpUpdateResult MCP update is success "));
                break;
         }
         else if (line.find("failed") != std::string::npos) {
                
                mcpUpdateStatus = false;
                ETG_TRACE_USR4(("Ctrl::getMcpUpdateResult mcpUpdateStatus : %d", mcpUpdateStatus));
                ETG_TRACE_USR4(("Ctrl::getMcpUpdateResult MCP update is failed "));
                break;
         }
   }
   inFile.close();
   return mcpUpdateStatus;
}

void Ctrl::updateOverallCISSWVersiononTarget(bool success){

      using namespace swu::Constants;
      static const std::string progessFilePersistent = "/var/opt/bosch/persistent/fcswupdate/fcswupd_running.xml";

      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget START" ));
      bool isEngineeringMode = Config::instance()->cfg_EnterEngineering.readAsBool();
      bool isPostImageMode = _progSection->isPostImageMode();
      bool isHMITriggerUpdate = swu::exists(progessFilePersistent);
      bool isRecoveryBasedUpdate = (!isHMITriggerUpdate);
      bool isCompactUpdate = _progSection->isCompatUpdate();


      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget isEngineeringMode : %d", isEngineeringMode));
      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget isPostImageMode : %d", isPostImageMode));
      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget isRecoveryBasedUpdate : %d", isRecoveryBasedUpdate));
      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget isCompactUpdate : %d", isCompactUpdate));
      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget isHMITriggerUpdate : %d", isHMITriggerUpdate));
      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget getMisUpdateResult : %d", getMisUpdateResult()));
      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget getMcpUpdateResult : %d", getMcpUpdateResult()));

      if(success)
      {
          std::string overallCISSWVerName = "";
          const std::string& overallCISSWVerReg = Registry::PATH_OVERALL_CIS_SW_VERSION_REG;

          TiXmlElement const *release = _progSection->getReleaseSection();      
          if ( !release ) {
               ETG_TRACE_FATAL(("No current Bosch XML available."));
               return;
          }

          BXmlAccess releaseAccess(release);
          overallCISSWVerName = releaseAccess.getTextFromModule(XML::TAG_CISSWVERSIONNAME, false);

          if( isPostImageMode )
          {
               ETG_TRACE_USR1(("Ctrl::updateOverallCISSWVersiononTarget PostImageMode"));
               if (  "" == overallCISSWVerName )
               {
                     ETG_TRACE_USR1(("Overall CIS SW version string is empty so setting overallCISSWVerName to 0000"));
                     overallCISSWVerName.assign("0000");
               }
           }

           else if( isEngineeringMode || isCompactUpdate || isRecoveryBasedUpdate )
           {

               if (  "" == overallCISSWVerName )
               {
                     ETG_TRACE_USR1(("Overall CIS SW version string is incomplete"));
                     return;
               }
           }

           else if ( isHMITriggerUpdate )
           {
               std::string subModulelist = _progSection->getSubModuleListConfiguredToUpdate();
               ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget subModulelist :  %s", subModulelist.c_str()));
               if ( subModulelist.find("MIS") != string::npos ) {
                       ETG_TRACE_USR2(("Ctrl::updateOverallCISSWVersiononTarget: MIS submodule is found"));
                       if( !getMisUpdateResult() ){
                              ETG_TRACE_USR2(("Ctrl::updateOverallCISSWVersiononTarget: MIS submodule update is failed"));
                              overallCISSWVerName.assign("Invalid");
                       }
               }

               if ( subModulelist.find("MCP") != string::npos ) {
                       ETG_TRACE_USR2(("Ctrl::updateOverallCISSWVersiononTarget: MCP submodule is found"));
                       if( !getMcpUpdateResult() ){
                              ETG_TRACE_USR2(("Ctrl::updateOverallCISSWVersiononTarget: MCP submodule update is failed"));
                              overallCISSWVerName.assign("Invalid");
                       }
               }

           }
           else 
           {
               ETG_TRACE_USR1(("Overall CIS SW version registry file not created"));
               return;
           }

           ETG_TRACE_COMP(("set new Overall CIS SW version:%s", overallCISSWVerName.c_str()));

          // remove old version file
          if (swu::exists(Registry::PATH_OVERALL_CIS_SW_VERSION_REG)) {
               //make coverity happy
               (void)::remove(Registry::PATH_OVERALL_CIS_SW_VERSION_REG.c_str());
          }

          // make sure the registry directory exists.
          if (! swu::exists(Registry::PATH_BASE) ) {
               if (swu::makeDirectoryRecursive(Registry::PATH_BASE, Registry::MODE_BASE)) {
                      (void)::chown(Registry::PATH_BASE.c_str(), User::root, Group::eco_osal);
               }
               else {
                      ETG_TRACE_FATAL(( "Registry directory does not exist and I can't create it. %s",
                      Registry::PATH_BASE.c_str() ));
               }
          }

           std::ofstream ofs(overallCISSWVerReg.c_str());
           if ( ! ofs ) {
                 ETG_TRACE_FATAL(("Can't open file for writing: %s", overallCISSWVerReg.c_str()));
                 return;
           }

           ofs << "OVERALL_CIS_SW_VERSION=" << overallCISSWVerName.c_str() ;
           ofs.close();
           (void)::chown(overallCISSWVerName.c_str(), User::root, Group::eco_osal);
      }

      ETG_TRACE_USR4(("Ctrl::updateOverallCISSWVersiononTarget END" ));
}
// deleting the KDS config file when abort triggered
void Ctrl::cleanupKdsConfigFile(){
   const char* odxefilepath="/var/opt/bosch/persistent/diagnosis/swu/pdconfig/config.pdx";
   ETG_TRACE_USR1(("Ctrl::Abort: deleting the KDS Configuration file(ODxe File)"));
     
     if (swu::exists(odxefilepath)) {
        swu::removeFile(odxefilepath);
     }

   ETG_TRACE_USR4(("Ctrl::Abort: deleting the KDS Configuration file(ODxe File)  END" ));
  }

void Ctrl::nextRequested()
  {
   ETG_TRACE_COMP(("Ctrl::nextRequested()"));

   if (isMISRunning())
   {
        killMISUpdater();
        _curProgItem.finalize(true);
   }
  }

void Ctrl::killMISUpdater()
  {
   ETG_TRACE_COMP(("Ctrl::killMISUpdater : Killing doipcmcupdate.py script"));
   ETG_TRACE_ERRMEM(("Ctrl::killMISUpdater : Killing doipcmcupdate.py script"));
   swu::execCommand("pkill -f  doipcmcupdate.py");
  }

bool Ctrl::isMISRunning()
  {
   ETG_TRACE_COMP(("Ctrl::isMISRunning()"));
   bool res = false;
   if (_curProgItem.isValid())
   {
        std::string myName = _curProgItem.getName().c_str();
        std::string myState = _curProgItem.getState().c_str();
        if (myName == "IVI/FIRMWARE/MIS" && myState == "running")
        {
                 ETG_TRACE_USR1(("MIS is in Running State"));
                 ETG_TRACE_ERRMEM(("MIS is in Running State"));
                 res = true;
        }
        return res;
   }
  }
}

