/**
 *\file     SWULCMComponent.cpp
 *\brief    SW Update LCM ASF Component
 *
 *\par Copyright:
 * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
 * This software is property of Robert Bosch GmbH. Unauthorized
 * duplication and disclosure to third parties is prohibited.
 */


#include <pthread.h>
#include "SWULCMComponent.h"

#include "base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_CORE
  #include "trcGenProj/Header/SWULCMComponent.cpp.trc.h"
#endif

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "platform/imp/WUP_Access.hpp"
#include "asf/core/Timer.h"


namespace ai_sw_update {
namespace common {
namespace core {
namespace imp {

DEFINE_CLASS_LOGGER_AND_LEVEL ("swupdate/swupdatecore", SWULCMComponent, Info);


static tUInt u32GetEnvVarVal(std::string key) {
   tUInt foundVal=0;
   char *envValue = getenv(key.c_str());
   if (envValue) {
      foundVal = (tUInt)atoi(envValue);
   }
   ETG_TRACE_USR2(("u32GetEnvVarVal(key=%30s)=%u", key.c_str(), foundVal));
   return foundVal;
}

SWULCMComponent::SWULCMComponent()
: ::asf::core::BaseComponent()
, _WdgCtrl()
, _TimerWUPExtendPowerOff()
, AckThreadRunning(false)
, EventNameValid(false)
, EventName(NULL)
{
  ETG_TRACE_USR2(("SWULCMComponent: Ctor"));

  // Ad-hoc implementation for INC watchdog and wake-up handling
  // @todo: refactor evaluation of environment variable to mechanism of fcswupd
  const std::string ENV_RECOVERY_MODE("SW_UPDATE_RECOVERY_MODE");
  const std::string ENV_ENGINEERING_MODE("SW_UPDATE_ENGINEERING_MODE");

  if (u32GetEnvVarVal(ENV_ENGINEERING_MODE)==1 || u32GetEnvVarVal(ENV_RECOVERY_MODE)==1) {
     ETG_TRACE_USR2(("SWULCMComponent: Activate V850 watchdog-handling"));


      // Enable watchdog triggering
      _WdgCtrl.startTriggering();

      // Directly send 'Startup finished' message to SCC after object is constructed.
      // Assumption: When this object is instantiated the system startup is finished.
      sendMsgWUPIndicateStartupFinished();

      // Enable cyclic sending of WUP message 'Extended Power Off'. This extends the time the SCC
      // waits before shutting down the application processor forcefully. The message is send cyclically because the
      // timeout extension may be reset on SCC side to the smaller default value in case new wake-up reasons are detected.
      // Details: http://hi0vm019.de.bosch.com/wiki/index.php?title=/dev/wup#OSAL_C_S32_IOCTRL_WUP_EXTEND_POWER_OFF_TIMEOUT
      sendMsgWUPExtendPowerOffTimeout();
      _TimerWUPExtendPowerOff.start(*this, DEFAULT_WUP_EXTENDED_POWER_OFF_TRIGGER_INTERVAL_MS, ::asf::core::Timer::Periodic);


     ETG_TRACE_USR2(("SWULCMComponent: Start acknowledging wakeup reasons via /dev/wup"));
     pthread_t         ackThread;

     AckThreadRunning=true;
     EventNameValid=false;
     EventName=NULL;
     pthread_create(&ackThread, NULL, acknowledgeWUPReasonsThread, this);
  }
}


SWULCMComponent::~SWULCMComponent()
{
  OSAL_tEventHandle hEventHandle;

  ETG_TRACE_USR2(("SWULCMComponent: Dtor"));

  if(AckThreadRunning)
  {
    while(!EventNameValid)
      usleep(50000);

    if(OSAL_s32EventOpen((char *)EventName, &hEventHandle)!=OSAL_OK)
    {
      ETG_TRACE_ERR(("SWULCMComponent: Failed to open event"));
    }
    else
    {
      sleep(1);
      if(OSAL_s32EventPost(hEventHandle, EVENT_MASK_STOP, OSAL_EN_EVENTMASK_OR)!=OSAL_OK)
      {
        ETG_TRACE_ERR(("SWULCMComponent: failed to post event"));
      }
      OSAL_s32EventClose(hEventHandle);
    }
    free((void *)EventName);
  }
}


void SWULCMComponent::onExpired(::asf::core::Timer& /*timer*/, boost::shared_ptr< ::asf::core::TimerPayload > /*data*/)
{
  // Important: In case multiple timers are used in this class, check to which timer object the 'onExpired' callback belongs.

  sendMsgWUPExtendPowerOffTimeout();
}


void SWULCMComponent::sendMsgWUPExtendPowerOffTimeout()
{
  WUP_Access wa;
  tU16 u16TimeoutS = DEFAULT_WUP_EXTENDED_POWER_OFF_TIMEOUT_SEC;

  // u16TimeoutS will be used first as the new timeout and updated with the current timeout afterwards
  wa.ioctl(OSAL_C_S32_IOCTRL_WUP_EXTEND_POWER_OFF_TIMEOUT, reinterpret_cast<intptr_t> (&u16TimeoutS));

  ETG_TRACE_USR2(("SWULCMComponent: WUP_EXTEND_POWER_OFF_TIMEOUT = %u sec", u16TimeoutS));
}


void SWULCMComponent::sendMsgWUPIndicateStartupFinished()
{
  WUP_Access wa;
  tU16 u16TimeoutS = 0;
  wa.ioctl(OSAL_C_S32_IOCTRL_WUP_INDICATE_STARTUP_FINISHED, reinterpret_cast<intptr_t> (&u16TimeoutS));
}


void *SWULCMComponent::acknowledgeWUPReasonsThread(void *Arg)
{
  static const int    EVENT_MASK_ALL=DEV_WUP_C_U32_EVENT_MASK_ONOFF_EVENT_CHANGED_NOTIFY|DEV_WUP_C_U32_EVENT_MASK_ONOFF_STATE_CHANGED_NOTIFY|EVENT_MASK_STOP;

  WUP_Access wa;

  SWULCMComponent     *swuc=(SWULCMComponent *)Arg;
  OSAL_tEventHandle   hEventHandle;
  OSAL_tEventMask     rEventMaskResult=0;

  DEV_WUP_trClientRegistration             rClientRegistration;
  DEV_WUP_trOnOffReasonChangedRegistration rOnOffReasonChangedRegistration;

  if(!wa.ioctl(OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT, reinterpret_cast<intptr_t> (&rClientRegistration)))
  {
    ETG_TRACE_ERR(("SWULCMComponent: OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT failed"));
  }
  else
  {
    swuc->EventName=strdup(rClientRegistration.szNotificationEventName);
    swuc->EventNameValid=true;

    if(OSAL_s32EventOpen(rClientRegistration.szNotificationEventName, &hEventHandle)!=OSAL_OK)
    {
      ETG_TRACE_ERR(("SWULCMComponent: OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT failed"));
    }
    else
    {
      rOnOffReasonChangedRegistration.u32ClientId       =rClientRegistration.u32ClientId;
      rOnOffReasonChangedRegistration.u8NotificationMode=DEV_WUP_C_U8_NOTIFY_STATES_AND_EVENTS_WITH_PAST_ONES;
      rOnOffReasonChangedRegistration.bIsSystemMaster   =TRUE;

      if(!wa.ioctl(OSAL_C_S32_IOCTRL_WUP_REGISTER_ONOFF_REASON_CHANGED_NOTIFICATION, reinterpret_cast<intptr_t> (&rOnOffReasonChangedRegistration)))
      {
        ETG_TRACE_ERR(("SWULCMComponent: OSAL_C_S32_IOCTRL_WUP_REGISTER_CLIENT failed - there is probably already an active WUP master"));
      }
      else
      {
        while(OSAL_s32EventWait(hEventHandle, EVENT_MASK_ALL, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &rEventMaskResult)==OSAL_OK)
        {
          if(OSAL_s32EventPost(hEventHandle, ~rEventMaskResult, OSAL_EN_EVENTMASK_AND)!=OSAL_OK)
          {
            ETG_TRACE_ERR(("SWULCMComponent: OSAL_s32EventPost failed"));
            break;
          }

          if(rEventMaskResult&DEV_WUP_C_U32_EVENT_MASK_ONOFF_EVENT_CHANGED_NOTIFY)
          {
            DEV_WUP_trOnOffEventHistory rOnOffEventHistory;

            rOnOffEventHistory.u32ClientId=rClientRegistration.u32ClientId;
            if(!wa.ioctl(OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_EVENTS, reinterpret_cast<intptr_t> (&rOnOffEventHistory)))
            {
              ETG_TRACE_ERR(("SWULCMComponent: OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_EVENTS failed"));
            }
            else
            {
              DEV_WUP_trOnOffEventAcknowledge rOnOffEventAcknowledge;
              tU8                             u8Index;
              tU8                             u8Event;
              tU16                            u16MsgHandle;

              rOnOffEventAcknowledge.u32ClientId = rClientRegistration.u32ClientId;

              for(u8Index=0; u8Index<rOnOffEventHistory.u8NumberOfOnOffEvents; u8Index++)
              {
                u8Event     =rOnOffEventHistory.arOnOffEvent[u8Index].u8Event;
                u16MsgHandle=rOnOffEventHistory.arOnOffEvent[u8Index].u16MsgHandle;

                // Only clients with bIsSystemMaster set to TRUE are allowed to perform the below on/off event acknowledge
                rOnOffEventAcknowledge.u16OnOffEventMsgHandle = u16MsgHandle;

                if(!wa.ioctl(OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_EVENT, reinterpret_cast<intptr_t> (&rOnOffEventAcknowledge)))
                {
                  ETG_TRACE_ERR(("SWULCMComponent: OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_EVENT failed"));
                }
              }
            }
          }
          if(rEventMaskResult&DEV_WUP_C_U32_EVENT_MASK_ONOFF_STATE_CHANGED_NOTIFY)
          {
            DEV_WUP_trOnOffStates rOnOffStates = DEV_WUP_trOnOffStates();

            if(!wa.ioctl(OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_STATES, reinterpret_cast<intptr_t> (&rOnOffStates)))
            {
              ETG_TRACE_ERR(("SWULCMComponent: OSAL_C_S32_IOCTRL_WUP_GET_ONOFF_STATES failed"));
            }
            else
            {
              DEV_WUP_trOnOffStateAcknowledge rOnOffStateAcknowledge;

              // Only clients with bIsSystemMaster set to TRUE are allowed to perform the below on/off state acknowledge
              rOnOffStateAcknowledge.u32ClientId = rClientRegistration.u32ClientId;
              rOnOffStateAcknowledge.u16OnOffStateMsgHandle = rOnOffStates.u16MsgHandle;

              if(!wa.ioctl(OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_STATE, reinterpret_cast<intptr_t> (&rOnOffStateAcknowledge)))
              {
                ETG_TRACE_ERR(("SWULCMComponent: OSAL_C_S32_IOCTRL_WUP_ACKNOWLEDGE_ONOFF_STATE failed"));
              }
            }
          }
          if(rEventMaskResult&EVENT_MASK_STOP)
            break;
        }
        wa.ioctl(OSAL_C_S32_IOCTRL_WUP_UNREGISTER_ONOFF_REASON_CHANGED_NOTIFICATION, reinterpret_cast<intptr_t> (&rOnOffReasonChangedRegistration.u32ClientId));
      }
      OSAL_s32EventClose(hEventHandle);
    }
  }

  return NULL;
}


} // namespace imp
} // namespace core
} // namespace common
} // namespace ai_sw_update


