/**
 * @file LuaInterpreter.h
 * @author Stefan Krause (CM-AI/EAR-S)
 * @copyright (c) 2013 Robert Bosch Car Multimedia GmbH
 * @addtogroup ai_sw_update/common
 * @{
 */


#ifndef LUAINTERPRETER_H
#define LUAINTERPRETER_H

//#define LUA_RESTART //klb2hi - 13.05.2015: It's already defined in ai_SW_update_prod.xml - don't redefined it again!!!

#include <map>
#include <string>
#include <iostream>
#include <execinfo.h>
#include "base/imp/ErrorHandler.h"

typedef struct lua_State lua_State;
typedef struct lua_Debug lua_Debug;
typedef void (*lua_Hook) (lua_State *L, lua_Debug *ar);
typedef int   (*luaFunction) (lua_State *state);

namespace ai_sw_update {
namespace common {

/**
 * Encapsulation of a Lua interpreter (version 5.2.3).
 */
class LuaInterpreter
{
public:
  LuaInterpreter();
  ~LuaInterpreter();


 void LuaInitFramework();

  /**
   * Executes a lua script at the defined path.
   * @param inFilePath Path of lua script file.
   * @return true on successful script execution, false otherwise
   */
  const bool execScript(const std::string & inFilePath);


  /**
   * Set a global variable in the lua environment.
   * @param inName Name of the global variable in lua environment.
   * @param inValue Value of the global variable.
   */
  void setGlobalVariable(const std::string & inName, const std::string & inValue);
  void addToSwuEnvTable(const std::string & inName, const std::string & inValue);


  void luaInitEnv(const char*);
  void luaSaveField(const char*, const char*);
  void luaSaveEnv();

private:

  static const double METHOD_SUCCESS; // must match LUA_NUMBER, file luaconf.h
  static const double METHOD_FAILURE; // must match LUA_NUMBER, file luaconf.h
  static       int iRegistered_functions;

  lua_State *_LState;

  std::string envVar;
  std::map<std::string, std::string> env_map;

  static const bool getLuaStackValue(lua_State* inLState, const int inIdx, std::string& outValue);
  static const bool getLuaStackValue(lua_State* inLState, int inIdx, unsigned int* outValue);
  static const std::string& luaThreadStatusToString(const int inThreadStatus);
  static const std::string& luaHookEventToString(const int inHookEvent);

  static int writeAdr3(lua_State* ioLState);
  static int fileAvailable(lua_State* ioLState);
  static int writeMagic(lua_State *ioLState);
  static int writeMagicToActive(lua_State *ioLState);
  static int readMagic(lua_State *ioLState);
  static int writeMagicBootChain(lua_State *ioLState);
  static int writeMagicBootChainFR(lua_State *ioLState);
  static int printLog(lua_State *ioLState);
  static int printDisplay(lua_State *ioLState);
  static int waitTime(lua_State *ioLState);
  static int remoteExecFR(lua_State *ioLState);
  static int remoteExec(lua_State *ioLState);
  static int writeRawNor(lua_State *ioLState);
  static int enableDebug(lua_State *ioLState);
  static int disableDebug(lua_State *ioLState);
  static int vinDelete(lua_State *ioLState);
  static int setError(lua_State *ioLState);
  static int swu_setError(lua_State *ioLState);
  static int setErrorFR(lua_State *ioLState);
  static int swu_setErrorFR(lua_State *ioLState);
  static void abortDownload (lua_State *lState, char const *command,enum SwUpdateErrorType luaErrorcode);
  static int traceback_handler(lua_State *lua);
  static int bgimageUpdate (lua_State* ioLState);


  /**
   * Lua debug hook for providing access to internal lua interpreter information.
   * For details refer to: http://www.lua.org/manual/5.2/manual.html#lua_Debug
   * @param ioLState
   * @param ioLDebug
   */
  static void luaHook(lua_State *ioLState, lua_Debug *ioLDebug);
};


} // namespace ai_sw_update
} // namespace common

#endif // LUAINTERPRETER_H
