/*************************************************************************
 * FILE:         gen2_FileOperations.cpp
 * PROJECT:      Gen3
 * SW-COMPONENT: SW Update
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Download Flash Operations ported from Gen2 GM project.
 *
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 **************************************************************************/


// @todo:
// - improve tracing
// - remove OSAL
// - refactor all implementations here in a second step
// - use filesystem functions from fcswupdate, they need to be relocated from the VAG MIB to the common part


#include "gen2_FileOperations.h"
#include "ErrorHandler.h"

#include <errno.h>
#include <stdio.h>

#include "swupd_trace.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_BASE
  #include "trcGenProj/Header/gen2_FileOperations.cpp.trc.h"
#endif


namespace ai_sw_update {
namespace common {

/*************************************************************************/
/*
 * FUNCTION:   tBool dl_tclFlashOperations::bCopyFileToRAM
 *
 * DESCRIPTION:  Copy a file from the source to the RAM
 *
 *               Attention: If the FileBuffer is no more needed, it must be freed by the calling function
 *              e.g. if (pvTemporaryBuffer != NULL) OSAL_vMemoryFree(*pvFileBuffer);
 *
 *
 * PARAMETER INPUT:  tCString CStrSrcFileName  e.g. "/dev/uda1/maindir/file.dat"
 *
 * PARAMETER OUTPUT: tPVoid   *pvFileBuffer      Buffer in RAM where the file data is stored
 *         tPU32    pu32FileSize   Number of bytes in the buffer
 *
 * RETURNVALUE:      TRUE   in case of success
 *                   FALSE  in case of error
 */
/*************************************************************************/

//@todo:
// changes from GM Gen2 DL-Core:
// - removed usage of OSAL device OSAL_C_STRING_DEVICE_USB_DNL for access of USB function
//   -> USB function not used in Gen3
// - remove check for admin block presence
// - remove CRC check when an admin block is available
//   -> affected components (list may not be complete): ADR3
// - remove special handling on detection of a special downloaderrorsimulation file
// - remove closing of USB function device on detection of file /dev/usb_dnl/download.off
// - remove io control OSAL_C_S32_IOCTRL_USB_FLASH_ERROR on usb function device on detection of file /dev/usb_dnl/download.err
// todo for this function
// - refactor implementation



#define GRANULATION                   ((tU8)           5) // bargraph step size
#define BARGRAPH_MAXIMUM              ((tU16)        100) // the max value we can send to HMI


tBool bCopyFileToRAM (
    tCString CStrSrcFileName,
    tPVoid *pvFileBuffer,
    tPU32 pu32FileSize,
    tPBool /*stopRequested*/)
{
  tBool bRetVal       = FALSE;
  tU32  u32SizeOfFile = 0;
  tU32  u32BytesRead  = 0;
  tS32  s32Success    = 0;
  OSAL_tIODescriptor fdSourceFile  = OSAL_ERROR;

  tPVoid pvTemporaryBuffer = NULL;

  ETG_TRACE_USR4(("DNL: bCopyFileToRam"));
  ETG_TRACE_USR4(("%s", CStrSrcFileName));

  *pu32FileSize = 0;
  *pvFileBuffer = NULL;

  tCString method = "bCopyFileToRAM";
  ETG_TRACE_USR4 (("%s was entered", method));


  //
  // open file ------------------------------------------------------------------------------
  //
  fdSourceFile = OSAL_IOOpen( CStrSrcFileName, OSAL_EN_READONLY);

  // check for errors
  if(fdSourceFile == OSAL_ERROR)
  {
    //@todo: change to TTFis tracing
    fprintf(stdout, "%s: Error opening file \"%s\"\n", __func__, CStrSrcFileName);
    fprintf(stdout, "Error no is : %d\n", errno);
    fprintf(stdout, "Error description is : %s\n",strerror(errno));
    iSetSwUpdateError(SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, SW_UPDATE_ERROR_TYPE_READ, "file open", NULL);

    ETG_TRACE_USR4 (("DL_ERROR_CAN_NOT_OPEN_FILE_EXCLUSIVE"));
    ETG_TRACE_USR4 (("bCopyFileToRAM was left"));
    return FALSE;
  }
  else
  {
    ETG_TRACE_USR4(("DNL: File open OK"));
  }

  //
  // get size of file -----------------------------------------------------------------------
  //
  s32Success = OSAL_s32IOControl
      (
          fdSourceFile,
          OSAL_C_S32_IOCTRL_FIONREAD,
          (intptr_t)&u32SizeOfFile
      );

  ETG_TRACE_USR4 (("DNL: u32SizeOfFile %d", u32SizeOfFile));
  if(s32Success == OSAL_ERROR)
  {
    ETG_TRACE_ERR (("OSAL ERROR"));
    iSetSwUpdateError(SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, SW_UPDATE_ERROR_TYPE_READ, "file size", NULL);
    OSAL_s32IOClose(fdSourceFile);
    ETG_TRACE_USR4 (("bCopyFileToRAM was left"));
    return FALSE;
  }

  //
  // allocate buffer
  //
  ETG_TRACE_USR4 (("DNL: Try to allocate file buffer %d", u32SizeOfFile));

  //first time buffer is allocated
  //depending on the size of file

  pvTemporaryBuffer = OSAL_pvMemoryAllocate (u32SizeOfFile);
  if (pvTemporaryBuffer != OSAL_NULL)
  {
    ETG_TRACE_USR4 (("DNL: buffer alloc OK %d", u32SizeOfFile));
  }
  else
  {
    ETG_TRACE_USR4(("DNL: buffer alloc ERROR"));
    OSAL_s32IOClose(fdSourceFile);
    ETG_TRACE_USR4 (("bCopyFileToRAM: DNL: buffer alloc ERROR"));
    iSetSwUpdateError(SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, SW_UPDATE_ERROR_TYPE_MEMORY_SIZE, "buffer alloc", NULL);
    return FALSE;
  }

  //
  // copy file to ram -----------------------------------------------------------------------
  //

  // Read from usb host stick
  //read in blocks ( for generation of bargraphs )
  tU32 u32Blocksize = (u32SizeOfFile * GRANULATION) / BARGRAPH_MAXIMUM;
  tU32 u32BytesReadOfBlock = 0;
  //TRACE_DOWNLOAD_INFO3(30);
  // Handling of small files with less than 20 bytes:
  if (u32Blocksize == 0) u32Blocksize = u32SizeOfFile;
  do
  {
    if (u32Blocksize > (u32SizeOfFile - u32BytesRead))
    {
      u32Blocksize = u32SizeOfFile - u32BytesRead;
    }
    //ETG_TRACE_USR4((9,u32Blocksize,"DNL: read u32Blocksize"));
    u32BytesReadOfBlock = (tU32)OSAL_s32IORead
        (
            fdSourceFile,
            (tPS8)((intptr_t)pvTemporaryBuffer + u32BytesRead),
            u32Blocksize
        );
    if((tS32)u32BytesReadOfBlock == OSAL_ERROR)
    {
      u32BytesRead = 0;
      //TRACE_DOWNLOAD_INFO2(117,u32SizeOfFile);
      //TRACE_ERROR2(DL_ERROR_CAN_NOT_READ_BLOCK_FROM_DVD ,u32BytesReadOfBlock);
      //TRACE_OSAL_ERROR();
      ETG_TRACE_ERR (("DL_ERROR_CAN_NOT_READ_BLOCK_FROM_DVD %d", u32BytesReadOfBlock));
      ETG_TRACE_ERR (("OSAL_ERROR"));
      iSetSwUpdateError(SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, SW_UPDATE_ERROR_TYPE_READ, "file read", NULL);
      break;
    }
    u32BytesRead += u32BytesReadOfBlock;
    //ETG_TRACE_USR4((9,u32BytesRead,"DNL: u32BytesRead"));
    //}while(u32BytesReadOfBlock > 0);
  }while(u32BytesRead < u32SizeOfFile);


  //close file
  s32Success = OSAL_s32IOClose(fdSourceFile);
  if(s32Success == OSAL_ERROR)
  {
    ETG_TRACE_ERR (("OSAL ERROR"));
    iSetSwUpdateError(SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, SW_UPDATE_ERROR_TYPE_READ, "file close", NULL);
    //@todo: improve error handling here
  }

  // check for errors
  if( u32BytesRead != u32SizeOfFile)
  {
    iSetSwUpdateError(SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, SW_UPDATE_ERROR_TYPE_READ, "NUMBER_OF_READ_FILE_IS_NOT_EQUAL_TO_FILE_SIZE", NULL);
    ETG_TRACE_ERR(("DL_ERROR_NUMBER_OF_READ_FILE_IS_NOT_EQUAL_TO_FILE_SIZE %d", u32BytesRead));
    OSAL_vMemoryFree(pvTemporaryBuffer); // Free buffer
    ETG_TRACE_USR4 (("bCopyFileToRAM was left"));
    return FALSE;
  }
  else
  {
    *pu32FileSize = u32SizeOfFile;
    *pvFileBuffer = pvTemporaryBuffer;
    //ETG_TRACE_USR4((9,u32SizeOfFile,"DNL: File read OK"));
    ETG_TRACE_USR4(("DNL: File read OK %d",u32BytesRead));
    bRetVal = TRUE;
  }


  //
  // close file -----------------------------------------------------------------------------
  //

  OSAL_s32IOClose(fdSourceFile);
  ETG_TRACE_USR4(("DNL: CloseFile"));


  ETG_TRACE_USR4 (("bCopyFileToRam was left"));
  return bRetVal;
}


//@todo use implementation from fcswupdate after added to ai_sw_update/common
bool fileAvailable(const char* fileName)
{
  bool retValue = false;

  ETG_TRACE_USR4 (("FileOperations::fileAvailable: check for %s", fileName));

  FILE * pFile = fopen(fileName, "r");

  if (0 != pFile)
  {
    // file available
    fclose(pFile);
    retValue = true;
    ETG_TRACE_USR4 (("FileOperations::fileAvailable: true"));
  }
  else
  {
    // file unavailable
    retValue = false;
    ETG_TRACE_USR4 (("FileOperations::fileAvailable: false"));
  }

  return retValue;
}

} //namespace ai_sw_update {
} //namespace common {

