/**
 * @file SystemCall.cpp
 * @author kut2hi
 * @copyright (c) 2013 Robert Bosch Car Multimedia GmbH
 * @addtogroup ai_sw_update/common
 * @{
 */

#include "SystemCall.h"
#include <cstdlib>
#include <cstdio>
#include <cerrno>
#include <cstring>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include "DownloadPipe.h"

#include "swupd_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_BASE
#include "trcGenProj/Header/SystemCall.cpp.trc.h"
#endif

#define SYSTEM_CALL_MAX_BUFFER 256

namespace ai_sw_update {
namespace common {

const char *logSystemRetVal = ";echo $? > ";
const char *logSystemRetFile = "/tmp/dl_ShellRetVal";
const char *logSystemErrRetFile = "/tmp/dl_ShellErrRetVal";

std::string usbLogPath;

static bool exists(std::string pathname, bool &existing) {
   struct stat path_stat;
   memset(&path_stat, 0, sizeof(path_stat));
   int r = stat(pathname.c_str(), &path_stat);
   if (r == 0) {
      existing = true;
      return true;
   } else if (errno == ENOENT) {
      existing = false;
      return true;
   }
   ETG_TRACE_ERR(("error trying to get stat (errno %8d)of file %s", errno, pathname.c_str()));
   return false;
}

int32_t SystemCall::execCommand(const std::string & inSystemCall) {
   ETG_TRACE_COMP(("execScript was entered for command %s", inSystemCall.c_str()));

   pid_t ret, pid;
   int status;
   int log = 0, fd = -1;

   char * const env[] = { strdup("HOME=/"), strdup("PATH=/bin:/usr/bin:/sbin:/usr/sbin:/opt/bosch/base/bin"), NULL };
   char log_file[SYSTEM_CALL_MAX_BUFFER] = {0};

   char *str1, *token, *saveptr;
   char * args[SYSTEM_CALL_MAX_BUFFER] = {0};
   int j = 0;

   str1 = const_cast < char* >(inSystemCall.c_str());
   for (;; j++, str1 = NULL) {
      token = strtok_r(str1, " ", &saveptr);
      if (token == NULL)
         break;

      if (log) {
         ETG_TRACE_COMP(("log file name = %s", token));
         strncpy(log_file, token, SYSTEM_CALL_MAX_BUFFER);
         log_file[SYSTEM_CALL_MAX_BUFFER - 1] = 0;
      }

      if (!log)
         if (strcmp(token, ">>") == 0)
            log = 1;
      if (!log) {
         args[j] = (char *) malloc(sizeof(char) * (strlen(token) + 1));
         strcpy(args[j], token);
      }
      if (j == SYSTEM_CALL_MAX_BUFFER - 1) {
         ETG_TRACE_ERR(("Too much params: %s", inSystemCall.c_str()));
         for (int k = 0; env[k] != NULL; k++) {
            free(env[k]);
         }
         for (int k = 0; args[k] != NULL; k++) {
            free (args[k]);
         }
         return 32000;
      }
   }

   args[j] = NULL;
   int i = 0;

   ETG_TRACE_COMP(("Parameters:"));
   while (args[i] != NULL) {
      ETG_TRACE_COMP(("  %s ", args[i]));
      i++;
   }

   if (log)
      fd = open(log_file, O_CREAT | O_WRONLY | O_APPEND, 0666);

   pid = fork();
   if (pid == -1) {
      ETG_TRACE_ERR(("fork error"));
      for (int k = 0; env[k] != NULL; k++) {
         free(env[k]);
      }
      for (int k = 0; args[k] != NULL; k++) {
         free (args[k]);
      }
      if (fd >= 0) {
         close(fd);
      }
      return 32001;
   } else if (pid != 0) {
      ETG_TRACE_ERR(("execCommand started pid %d", pid));
      while ((ret = waitpid(pid, &status, 0)) == -1) {
         if (errno != EINTR) {
            /* Handle error */
            ETG_TRACE_ERR(("Error in not EINTR, it is %s", strerror(errno)));
            ETG_TRACE_ERR(("  as int: %d", errno));
            break;
         }
      }
      for (int k = 0; env[k] != NULL; k++) {
         free(env[k]);
      }
      for (int k = 0; args[k] != NULL; k++) {
         free (args[k]);
      }

      ETG_TRACE_COMP(("execCommand: RET=%d", ret));
      ETG_TRACE_COMP(("execCommand: status=%d", status));

      if (WIFEXITED(status)) {
         ETG_TRACE_ERR(("execCommand: returned %d", WEXITSTATUS(status)));
         if (fd >= 0) {
            close(fd);
         }
         return WEXITSTATUS(status);
      }

      if (fd >= 0) {
         close(fd);
      }
      return 0;
   } else {
      if (log) {
         close(1);
         if (fd >= 0) {
            dup2(fd, 1);
         }
      }
      if (execvpe(args[0], args, env) == -1) {
         ETG_TRACE_ERR(("execCommand: could not start command %s", inSystemCall.c_str()));
         _Exit(127);
      }

      if (fd >= 0) {
         close(fd);
      }

      // Here are dragons!
      ETG_TRACE_FATAL(("execCommand: dragons found..."));
   }

   // Here are dragons!
   ETG_TRACE_FATAL(("execCommand: was left in the area with the dragons..."));
   return 32002;
}

//@todo: Check whether system() an be relaced by a more performant and secure version
// Check http://stackoverflow.com/questions/900666/system-calls-in-c-and-their-roles-in-programming
//       http://www.cplusplus.com/forum/articles/11153/
// -----------------------------------------------------------------------------------
const bool SystemCall::exec(const std::string & inSystemCall, SystemCallExecResult *result, bool global_fail)
// -----------------------------------------------------------------------------------
{
   int exitVal = 1;
   bool bRetVal = false;
   ETG_TRACE_USR4(("SystemCall::exec was entered for command %s", inSystemCall.c_str()));
   
   std::string commandBuffer = inSystemCall;
   // copy logs to USB medium, if standard downloadPiple exisits
   std::string downloadPipeStr = "downloadPipe";
 
   commandBuffer = commandBuffer + logSystemRetVal + logSystemRetFile;
   ETG_TRACE_USR4(("SystemCall::exec will exec %s", commandBuffer.c_str()));

   system(commandBuffer.c_str());

   //read the exit value from the logfile
   FILE* retFile = fopen(logSystemRetFile, "r");

   if (NULL != retFile) {
      if (!feof(retFile)) {
         (void) fscanf(retFile, "%d", &exitVal);
         if (!exitVal) {
            bRetVal = true;
         } else {
            ETG_TRACE_ERR(("SystemCall::exec during execution %s", commandBuffer.c_str()));
            ETG_TRACE_ERR(("SystemCall::exec found exit code %d", exitVal));
         }
      }
      fclose(retFile);
   }

   if (result != NULL) {
      result->retValue = exitVal;
   }

   if (global_fail && (!bRetVal)) {
      bool existing;
      if (!exists(logSystemErrRetFile, existing)) {
         ETG_TRACE_USR4(("SystemCall::exec could not check if error file exists, will override it"));
         existing = false;
      }
      if (!existing) {
         ETG_TRACE_USR4(("SystemCall::exec will set err out variable"));
         FILE* errFile = fopen(logSystemErrRetFile, "w");
         if(0 != errFile) {
            fprintf(errFile, "%d", exitVal);
            fclose(errFile);
         }
         else {
            ETG_TRACE_USR4(("SystemCall::error writing to file %s", logSystemErrRetFile));
         }
      }
   }
   ETG_TRACE_USR4(("SystemCall::exec exitVal = %d for command =%s", exitVal, inSystemCall.c_str()));
   ETG_TRACE_USR4(("SystemCall::exec was left"));
   return bRetVal;

#if 0 // old implementation
   int status = -1;     // 0: OK, else: Error
   bool result = false;
   int error = 0;
   int exitCode = -1;// @todo: chose correct default value

#if 0 // test for shell availability, only for debug
   // Test whether a shell is available. system() returns 0 which means no shell is available.
   // But the command is executed nevertheless. TBC with CF3.
   const char* null = 0;
   status = system(null);
   exitCode = WEXITSTATUS(status);
   error = errno;

   ETG_TRACE_USR4(("Null Test SystemCall::exec: status %i", status));
   ETG_TRACE_USR4(("Null Test SystemCall::exec: exit code %i", exitCode));
   ETG_TRACE_USR4(("Null Test SystemCall::exec: errno %i", error));
   ETG_TRACE_USR4(("Null Test SystemCall::exec: errstr %s", strerror(error)));
#endif

   status = system(inSystemCall.c_str());
   exitCode = WEXITSTATUS(status);
   error = errno;

   ETG_TRACE_USR4(("SystemCall::exec: %s", inSystemCall.c_str()));
   ETG_TRACE_USR4(("SystemCall::exec: status %i", status));
   ETG_TRACE_USR4(("SystemCall::exec: exit code %i", exitCode));
   ETG_TRACE_USR4(("SystemCall::exec: errno %i", error));
   ETG_TRACE_USR4(("SystemCall::exec: errstr %s", strerror(error)));

   result = (0 == exitCode); //@todo: check whether exit code 0 means each time success or whether there are also exceptions.

   //@todo: system() reports an error everytime. Therefore we cannot check the return value for errors and have to assume a successful operation. TBC with CF3.
   return true;
#endif
}

const bool SystemCall::clearError() {
   bool existing;
   if (!exists(logSystemErrRetFile, existing)) {
      ETG_TRACE_USR4(("SystemCall::clearError could not check if error file exists, will override it"));
      return false;
   }

   if (existing) {
      if (unlink(logSystemErrRetFile)) {
         ETG_TRACE_USR4(("SystemCall::clearError error during unlink, errno %10d (%20s)", errno, strerror(errno)))
         ;
         return false;
      }
   }
   return true;
}

const int32_t SystemCall::getErrorCode() {
   bool existing;
   int32_t exitVal = 0;
   if (!exists(logSystemErrRetFile, existing)) {
      ETG_TRACE_USR4(("SystemCall::getErrorCode could not check if error file exists, will override it"));
      return ENOENT;
   }

   if (!existing) {
      return 0;
   }

   //read the exit value from the logfile
   FILE* file = fopen(logSystemErrRetFile, "r");

   if (NULL != file) {
      if (!feof(file)) {
         (void) fscanf(file, "%d", &exitVal);
      }
      fclose(file);
   }
   return exitVal;
}

const bool SystemCall::hasError() {
   bool existing;
   if (!exists(logSystemErrRetFile, existing)) {
      ETG_TRACE_USR4(("SystemCall::hasError could not check if error file exists, will override it"));
      return true;
   }
   return existing;
}

} // namespace common
} // namespace ai_sw_update
