/*******************************************************************************************
* FILE:			RegistryAccess.cpp
*
* SW-COMPONENT: Software Update
*
* DESCRIPTION: Registry access class, which read and write the registry using OSAL call.
*				
*
* AUTHOR:		Aditya Kumar Jha
*
* COPYRIGHT:	(c) 2014  2015 Robert Bosch Engineering and Business Solutions Ltd, Bangalore.
*
*********************************************************************************************
* HISTORY:
*				18.02.2014	Aditya Kumar Jha
*				Initial Version.
********************************************************************************************/


#include "RegistryAccess.h"
#include "unistd.h"

#define DEBUG_LEVEL 0
#define DBG_ENABLED
#ifdef DBG_ENABLED
#define DEBUG_MSG(level, fmtstr, args...) \
	(level <= DEBUG_LEVEL) ? printf("\n%s: " fmtstr "\n",__func__ , ##args) : 0; 
#else
#define DEBUG_MSG(level, fmtstr, args...)
#endif

//Defined for error message
#define REG_ACCESS_SUCCESS	0
#define REG_OPEN_ERROR		1
#define REG_CREATE_ERROR	2
#define REG_READ_ERROR		3
#define REG_WRITE_ERROR		4
#define REG_CLOSE_ERROR		5
#define REG_MISMATCH_ERROR	6

/*******************************************************************************************/

#ifndef __SW_UPDATE_UNIT_TESTING__
// Implementation of the methods getInstanceOfRegistryAccess and releaseInstanceOfRegistryAccess
// In case of __SW_UPDATE_UNIT_TESTING__ they are implemented in mock!
SWU_IMPL_GET_AND_RELEASE_INSTANCE(RegistryAccess);
#endif

// Implementation of the singleton methods
SWU_IMPL_SINGLETON(RegistryAccess);

/*******************************************************************************************/


/*******************************************************************************************
* FUNCTION:		RegistryAccess
* DESCRIPTION:	Private Constructor
* PARAMETER:
*					NULL			
*
* RETURNVALUE:	Void
*
* HISTORY:
*					18.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
RegistryAccess::RegistryAccess()
{
}
      
/*******************************************************************************************
* FUNCTION:		~RegistryAccess
* DESCRIPTION:	Destructor
* PARAMETER:
*					NULL			
*
* RETURNVALUE:	Void
*
* HISTORY:
*					18.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
RegistryAccess::~RegistryAccess()
{
}

  
/*******************************************************************************************
* FUNCTION:		u32RegWriteValue
* DESCRIPTION:	This function create a write a value to the key.
* PARAMETER:
*					tCString cDeviceName //location path	
*					tCString cKeyName    //key 
*					tCString cKeyValue   //key value
*
* RETURNVALUE:	tU32
*
* HISTORY:
*					18.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 RegistryAccess::u32RegWriteValue(tCString cDeviceName, tCString cKeyName, tString cKeyValue ) const
{
	//tS32 s32DevDesc = 0;
	OSAL_trIOCtrlRegistry rReg;
	tS8   pBuffer[255];
	tU32 u32RetVal = REG_ACCESS_SUCCESS;

	/* Open the device in readwrite  mode */
	tS32 s32DevDesc =static_cast<tS32> (OSAL_IOOpen(cDeviceName, OSAL_EN_READWRITE) );
	if(s32DevDesc != OSAL_ERROR )
	{
		DEBUG_MSG(1, "Open success");
		/* Write the value */
		rReg.pcos8Name = (tCS8*)cKeyName;
		rReg.ps8Value  = (tU8*)cKeyValue;
		rReg.u32Size   = 256;
		rReg.s32Type = OSAL_C_S32_VALUE_STRING;
		if(OSAL_s32IOControl(s32DevDesc,OSAL_C_S32_IOCTRL_REGSETVALUE,(intptr_t)&rReg) != OSAL_ERROR)
		{
			DEBUG_MSG(1, "Write success");
			/* Read the value */
			if ( u32RegReadValue(cDeviceName, cKeyName, pBuffer) == REG_ACCESS_SUCCESS )
			{
				/* compare with key value */
				if(strncmp(cKeyValue, (const char*)pBuffer, strlen(cKeyValue)))
				{
					DEBUG_MSG(2, "Compare Failed");
					u32RetVal = REG_MISMATCH_ERROR;
				}
			}
			else
			{
				u32RetVal = REG_READ_ERROR;
			}
		}
		else
		{
			u32RetVal = REG_WRITE_ERROR;
		}

		/* Close the device */
		if (OSAL_ERROR == OSAL_s32IOClose(s32DevDesc))
		{
			DEBUG_MSG(2, "Close ERROR");
			u32RetVal = REG_CLOSE_ERROR;
		}

	}
	else
	{
		u32RetVal = REG_OPEN_ERROR;
	}

	return u32RetVal;
}

/*******************************************************************************************
* FUNCTION:		u32RegReadValue
* DESCRIPTION:	This function read key value.
* PARAMETER:
*					tCString cDeviceName //location path	
*					tCString cKeyName    //key 
*					tS8* pcKeyValue   //distination buffer of key value
*
* RETURNVALUE:	tU32
*
* HISTORY:
*					18.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 RegistryAccess::u32RegReadValue(tCString cDeviceName, tCString cKeyName, tS8*   pcKeyValue ) const
{
	//tS32 s32DevDesc = 0;
	OSAL_trIOCtrlRegistry rReg;
	tU32 u32RetVal = REG_ACCESS_SUCCESS;

	/* Open the device in readwrite  mode */
	tS32 s32DevDesc =static_cast<tS32> (OSAL_IOOpen(cDeviceName, OSAL_EN_READONLY) );

	if(s32DevDesc != OSAL_ERROR )
	{
		/* Write the value */
		rReg.pcos8Name = (tCS8*)cKeyName;
		rReg.ps8Value  = (tU8*)pcKeyValue;
		rReg.u32Size   = 256;
		rReg.s32Type = OSAL_C_S32_VALUE_STRING;
		//rReg.s32Type = OSAL_C_S32_VALUE_S32;
		if(OSAL_s32IOControl(s32DevDesc,OSAL_C_S32_IOCTRL_REGGETVALUE,(intptr_t)&rReg) != OSAL_ERROR)
		{
			DEBUG_MSG(1, "Read Success.. Value: %s", pcKeyValue);
		}
		else
		{
			u32RetVal = REG_READ_ERROR;
			DEBUG_MSG(1, "Read Error");
		}

		/* Close the device */
		if (OSAL_ERROR == OSAL_s32IOClose(s32DevDesc))
		{
			u32RetVal = REG_CLOSE_ERROR;
		}

	}
	else
	{
		u32RetVal = REG_OPEN_ERROR;
	}

	return u32RetVal;
}

/*******************************************************************************************
* FUNCTION:		u32RegCreateWriteValue
* DESCRIPTION:	This function create a registry key and set a value to the key.
* PARAMETER:
*					tCString cDeviceName //location path	
*					tCString cKeyName    //key 
*					tCString cKeyValue   //key value
*
* RETURNVALUE:	tU32
*
* HISTORY:
*					18.02.2014	Aditya Kumar Jha
*					Initial Version.
********************************************************************************************/
tU32 RegistryAccess::u32RegCreateWriteValue(tCString cDeviceName, tCString cKeyName, tCString cKeyValue ) const
{
	//tS32 s32DevDesc = 0;
	OSAL_tIODescriptor hRegKey = 0;
	OSAL_trIOCtrlRegistryValue sValue;
	tU32 u32RetVal = REG_ACCESS_SUCCESS;

	/* Open the device in readwrite  mode */
	tS32 s32DevDesc =static_cast<tS32> (OSAL_IOOpen(cDeviceName, OSAL_EN_READWRITE) );

	if(s32DevDesc != OSAL_ERROR )
	{
		/*Create the Key*/
		hRegKey = OSAL_IOCreate(cKeyName, OSAL_EN_READWRITE);
		if ( OSAL_ERROR == hRegKey)
		{
			u32RetVal = REG_CREATE_ERROR;
		}
		else
		{
			/* Write the value */
			OSAL_pvMemoryCopy(sValue.s8Name, (const void*)cKeyName, OSAL_u32StringLength(cKeyName)+1);
			OSAL_pvMemoryCopy(sValue.s8Value, (const void*)cKeyValue, OSAL_u32StringLength(cKeyValue)+1);
			sValue.s32Type   =  OSAL_C_S32_VALUE_STRING;

			if(OSAL_s32IOControl(hRegKey,OSAL_C_S32_IOCTRL_REGSETVALUE,(intptr_t)&sValue) == OSAL_ERROR)
			{
				u32RetVal = REG_WRITE_ERROR;
			}	

			/* Close the device */
			if (OSAL_ERROR == OSAL_s32IOClose(hRegKey))
			{
				u32RetVal = REG_CLOSE_ERROR;
			}
		}

		/* Close the device */
		if (OSAL_ERROR == OSAL_s32IOClose(s32DevDesc))
		{
			u32RetVal = REG_CLOSE_ERROR;
		}

	}
	else
	{
		u32RetVal = REG_OPEN_ERROR;
	}

	return u32RetVal;
}

