#include "MTD_IO.h"
#include "ErrorHandler.h"
#include "UpdateLog.h"
#include "DownloadPipe.h"

//@todo:
// Implementation is taken over from GM Gen2 DL-Core. It needs to be refactored:
// - refactor complete file / implementation
// - remove OSAL
// - refactor tracing
// - enable TTFis tracing
// - code cleanup

#include <mtd/mtd-abi.h>  // for MTD ioctl commands and structs.
#include <fcntl.h>        // for open() command and flags.
#include <sys/ioctl.h>    // for ioctl command.
#include <unistd.h>       // for pwrite32/pread64.
#include <map>
#include <cerrno>
#include <cstdio>
#include <sstream>


#include "swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_BASE
  #include "trcGenProj/Header/MTD_IO.cpp.trc.h"
#endif


namespace ai_sw_update {
namespace common {


using namespace std;


#define TRITON_FLASH_BASE_ADDR                  0x40000000
#define TRITON_FLASH_SECTOR_SIZE                0x00020000    // 128 kB
#define TRITON_FLASH_SECTOR_MAX_COUNT           512           // 512 * 128 KB = 64 MB Total
#define TRITON_FLASH_SECTOR_COUNT_OFFSET        0

#define BARGRAPH_MAXIMUM                        ((tU16)   100) // the max value we can send to HMI
#define U8_MAX_SIZE 255
#define MTD_PROC_FILE    "/proc/mtd"
#define MTD_PARTITION_NAME_SIZE  63


// Partition analysis
struct MtdPartitionInfo
{
    tU8 u8Index;
    off_t u32Size;
    tU32 u32EraseSize;
    off_t u32Offset;
};

tBool bIsPartitionRead = FALSE;
tU8 u8FlopPartitionCount;
std::map<tU8,MtdPartitionInfo> flopDevPartitions; // Map to handle partition number and partition info


/* **************************************************************************** */
/*
*  Class      : MTD_IO
*  Function   : bProgramDataToRawFlash
*  Description: Downloads the specified SW
*  Parameter  :
*               tPU8  pu8BufferPointer,
*         off_t  inDestinationOffset,
*       tU32  u32Size,
*       tBool bErase
*
*  Returns    : TRUE  -> no error
*               FALSE -> error
*/
/* *************************************************************************** */
tBool bProgramDataToRawFlash(
    const tU8 *pu8BufferPointer,
    off_t inDestinationOffset,
    tU32  u32Size,
    tBool bErase)
{
  ETG_TRACE_USR4 (("bProgramDataToRawFlash was entered"));

  tBool bRetVal = TRUE;
  tU32 u32FlashSectorNumber = 0;
  tU32 u32FlashSectorStart = 0;
  tU32 u32FlashSectorEnd = 0;
  tU32 u32NumBytesWritten = 0;
  tU32 u32BlockSize = 0;
  tU16 u16WriteProgress = 0;
  tU32 u32BlockStart = 0;
  tU32 u32BlockEnd = 0;
  tU32 u32BlockSecStart = 0;
  tU32 u32BlockSecNext = 0;
  tU32 u32VerifyCount = 0;

  tChar tmpBuffer [256] = {0};
  int fd = -1;
  int retcode = 0;
  std::string devname;

  off_t u32FlashDevOffset = 0;
  off_t destinationOffsetRelative = 0;

  // Set the raw flash device name for the linux driver and adjust the flash address to the flash device offset
  bSetRawFlashDevName(inDestinationOffset, devname, &u32FlashDevOffset);
  destinationOffsetRelative = inDestinationOffset - u32FlashDevOffset;

  // number of sectors to be erased:

  //erase procedure
  u32FlashSectorNumber = u32Size / (tU32)TRITON_FLASH_SECTOR_SIZE;

  if ( (u32Size % (tU32)TRITON_FLASH_SECTOR_SIZE) != 0)
    u32FlashSectorNumber+=1;

  if (u32FlashSectorNumber > TRITON_FLASH_SECTOR_MAX_COUNT)
  {
    ETG_TRACE_USR4(("DNL: Sector count exceeded, ERROR !!! %d", u32FlashSectorNumber));
    ETG_TRACE_FATAL(("DNL: RawNorSector count exceeded, ERROR !!!"));
        //dl_tclCoreUtilities::vWriteErrmemDNL(U16_M_ERRMEM_DNL_ERROR(DNL_U8_ERRMEM_TYPE_STRING),(tU8*)"DNL: RawFlash ERROR1", 20); //@todo: enable dl_tclCoreUtilities::vWriteErrmemDNL
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_ERASE, "RawNorSector count exceeded", NULL);

    return FALSE;
  }

  u32FlashSectorStart =static_cast<tU32> (TRITON_FLASH_SECTOR_COUNT_OFFSET + destinationOffsetRelative / TRITON_FLASH_SECTOR_SIZE );
  u32FlashSectorEnd   = u32FlashSectorStart + u32FlashSectorNumber - 1;

  ETG_TRACE_USR4(("DNL: first sector to be written %d", u32FlashSectorStart));
  ETG_TRACE_USR4(("DNL: last  sector to be written %d", u32FlashSectorEnd));

  if (u32FlashSectorEnd >= TRITON_FLASH_SECTOR_MAX_COUNT)
  {
    ETG_TRACE_USR4(("DNL: Last Sector exceeded, ERROR !!! %d", u32FlashSectorEnd));
    ETG_TRACE_FATAL(("DNL: RawNorLastSector exceeded, ERROR !!!"));
        //dl_tclCoreUtilities::vWriteErrmemDNL(U16_M_ERRMEM_DNL_ERROR(DNL_U8_ERRMEM_TYPE_STRING),(tU8*)"DNL: RawFlash ERROR2", 20); //@todo: dl_tclCoreUtilities::vWriteErrmemDNL
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_ERASE, "RawNorLastSector exceeded", NULL);

    return FALSE;
  }

  ETG_TRACE_USR4(("DNL: Attention, writing critical flash section %d", u32FlashSectorEnd));
  ETG_TRACE_USR4(("DNL: DO NOT POWER OFF TARGET"));

  // Open flash for write and read
  fd = open(devname.c_str(), O_RDWR,0x1A4);
  if(fd >= 0)
  {
    ETG_TRACE_USR4(("DNL: Flash open OK"));
    if (bErase)
    {
      // erase raw flash
      {
        erase_info_user einfo;
        memset(&einfo,0,sizeof(einfo));
        einfo.start  = u32FlashSectorStart  * (tU32)TRITON_FLASH_SECTOR_SIZE;
        einfo.length = u32FlashSectorNumber * (tU32)TRITON_FLASH_SECTOR_SIZE;
        sprintf(tmpBuffer, "erase flash adress=0x%8.8X size=0x%8.8X",(unsigned int) (u32FlashSectorStart * (tU32)TRITON_FLASH_SECTOR_SIZE), (unsigned int) (u32FlashSectorNumber * (tU32)TRITON_FLASH_SECTOR_SIZE));
        printToDownloadPipe(tmpBuffer);
        retcode = ioctl( fd , MEMERASE, &einfo );
        if(retcode)
        {
          ETG_TRACE_ERR (("DNL: Flash sector erase ERROR !!!"));
          ETG_TRACE_USR4(("DNL: Flash sector erase ERROR !!! %d", u32FlashSectorNumber));
          iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_ERASE, "ProgramDataToRawFlash SectorErase", NULL);
          bRetVal = FALSE;
          printToDownloadPipe("erase flash device ERROR");
        }
      }
    } //if (bErase)

    // Flash erase OK:
    if (bRetVal)
    {
            // Write blocks to flash by using OSAL routines
      ETG_TRACE_USR4(("DNL: Flash sector write begin"));
      do
      {
        // remaining bytes to be written
        u32BlockSize = u32Size - u32NumBytesWritten;

        // reduce block to max. sector size
        u32BlockSize = ( u32BlockSize > TRITON_FLASH_SECTOR_SIZE ) ? TRITON_FLASH_SECTOR_SIZE : u32BlockSize ;
        //check if we cross a sector border
        u32BlockStart     =static_cast<tU32> (destinationOffsetRelative + u32NumBytesWritten);
        u32BlockEnd       = u32BlockStart +    u32BlockSize;
        u32BlockSecStart  = u32BlockStart &   ~(TRITON_FLASH_SECTOR_SIZE-1);
        u32BlockSecNext   = u32BlockSecStart + TRITON_FLASH_SECTOR_SIZE  ;

        //reduce u32BlockSize to fit into this sector
        u32BlockSize = (u32BlockEnd > u32BlockSecNext) ? (u32BlockSecNext -  u32BlockStart) : u32BlockSize;

        // write to raw flash
        retcode =static_cast<int> (pwrite(fd, (const tS8*)pu8BufferPointer + u32NumBytesWritten, u32BlockSize, u32BlockStart) );

        if(retcode != static_cast<int> (u32BlockSize))  // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
        {
          ETG_TRACE_USR4 (("Raw flash write ERROR"));
                    //MTD_IO::vSendFlashErrorMessage (DL_FLASH_ERROR_WRITE | DL_FLASH_NOR_RAW); //@todo MTD_IO::vSendFlashErrorMessage
          iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "ProgramDataToRawFlash FlashWrite", NULL);
          bRetVal = FALSE;
          printToDownloadPipe("write flash device ERROR");
          break;
        }

        if(u32Size == 0)
        {
                    //TRACE_ERROR2(DL_ERROR_CAN_NOT_WRITE_FLASH_NO_DATA_TO_WRITE ,0);
          ETG_TRACE_USR4 (("DL_ERROR_CAN_NOT_WRITE_FLASH_NO_DATA_TO_WRITE"));
          //TRACE_DOWNLOAD_INFO2(157,rFlashData.ps8Buffer);
          //TRACE_DOWNLOAD_INFO2(158,rFlashData.u32FlashAddress);
          //TRACE_DOWNLOAD_INFO2(159,rFlashData.u32NumBytes);
          iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "CAN_NOT_WRITE_FLASH_NO_DATA_TO_WRITE", NULL);
          u32Size = 1;
        }

        u32NumBytesWritten += u32BlockSize;
        u16WriteProgress = (tU16)(BARGRAPH_MAXIMUM * u32NumBytesWritten / u32Size);
        ETG_TRACE_USR4(("DNL: Flash sector write progress %d", u32NumBytesWritten));
        ETG_TRACE_USR4(("DNL: Flash sector write progress %d", u16WriteProgress));
        //OSAL_s32ThreadWait(DL_TIME_TO_WAIT_TO_LET_OTHER_TASKS_WORK); //@todo:OSAL_s32ThreadWait(DL_TIME_TO_WAIT_TO_LET_OTHER_TASKS_WORK) not required under Linux
      }
      while((u32NumBytesWritten < u32Size));
      ETG_TRACE_USR4(("DNL: Flash sector write end"));
    } //if (bRetVal)


    tCU32 BLOCK_SIZE    = 1024;
    tS8 pvVerifyBuffer [BLOCK_SIZE] = {0};
    tU32  blocks        = u32Size / BLOCK_SIZE;
    tU32  lastChunkSize = u32Size % BLOCK_SIZE;
    tBool breakForLoop = FALSE;
    tU32 sizeToRead = 0;
    tU32 u32OffSet = 0;
    ETG_TRACE_USR4(("DNL: Flash read start"));

    //go through all the blocks
    for (tU32 i = 0; (!breakForLoop && i<=blocks); i++)
    {
      // Read flash back for verification
      if (i < blocks)
      {
        //deal with normal block size
        sizeToRead = BLOCK_SIZE;
      }
      else if (lastChunkSize > 0)
      {
        ETG_TRACE_USR4 (("All blocks were read correctly"));
        ETG_TRACE_USR4 (("Now read the last chunk"));
                //deal with the last chunk
                //then break the for loop
        breakForLoop = TRUE;
        sizeToRead = lastChunkSize;
      }
      else //lastChunkSize == 0
      {
        //do nothing
        break;
      }

      //compute the internal offset to read
      u32OffSet  = BLOCK_SIZE * i;

      //rFlashData.ps8Buffer       = pvVerifyBuffer;
      //rFlashData.u32FlashAddress = (tU32)pu8DestinationOffset + u32OffSet;   // Start of resident unit in flash
      //rFlashData.u32NumBytes     = sizeToRead;

      // read back from raw flash
      retcode =static_cast<int> (pread( fd, pvVerifyBuffer, sizeToRead, destinationOffsetRelative + u32OffSet) );
      if(retcode != static_cast<int> (sizeToRead))  // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      {
        ETG_TRACE_USR4 (("Raw flash read ERROR"));
		iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_READ, "ProgramDataToRawFlash FlashRead", NULL);
        bRetVal = FALSE;
        printToDownloadPipe("read back flash device ERROR");
        break;
      }

      if (0 != memcmp (pvVerifyBuffer,
          pu8BufferPointer + u32OffSet,
          sizeToRead))
      {
        tCU32 ERROR_BUFSIZE = 80;
        tChar szErrorString[ERROR_BUFSIZE]= {0};
        snprintf(szErrorString, ERROR_BUFSIZE, "0x%8.8X", (unsigned int)(TRITON_FLASH_BASE_ADDR + destinationOffsetRelative + u32VerifyCount));
        ETG_TRACE_USR4(("DNL: RawFlashVerifyError !!!"));
		iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_VERIFY, "bProgramDataToRawFlash FlashVerify", NULL);
        bRetVal = FALSE;
        break;
      }
    } // end for (tU32 i = 0; !breakForLoop; i++)

    ETG_TRACE_USR4(("DNL: Flash verify end %d", u32Size));
    if (bRetVal)
    {
            printToDownloadPipe("writing flash device done, OK");
    }
    else
    {
            printToDownloadPipe("writing flash device ERROR");
            iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "writing flash device", NULL);
    }
    if(fd>=0)
    {
      fsync(fd);	// write buffer from os to physical device
      close(fd);
    }
  }
  else  // flash open error:
  {
    ETG_TRACE_USR4(("DNL: Flash open ERROR"));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "Flash open", NULL);
    bRetVal = FALSE;
  }

  if (!bRetVal)
  {
    ETG_TRACE_USR4(("DNL: RawFlash ERROR !!!"));
    ETG_TRACE_FATAL(("DNL: RawFlash ERROR !!!"));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "RawFlash", NULL);
  }

  ETG_TRACE_USR4(("bProgramDataToRawFlash was left"));

  return bRetVal;
}


/* **************************************************************************** */
/*
 *  Class      : MTD_IO
 *  Function   : bProgramDataToRawFlash
 *  Description: Downloads the specified SW
 *  Parameter  :
 *               tCString CStrSrcFileName,
 *       off_t   inDestinationOffset,
 *       tBool    bErase
 *
 *  Returns    :  TRUE  -> no error
 *                FALSE -> error
 */
/* *************************************************************************** */
tBool bProgramDataToRawFlash(
    tCString inSrcFile,
    off_t inDestinationOffset,
    tBool bErase)
{
  tCString method = "MTD_IO::bProgramDataToRawFlash";
  ETG_TRACE_USR4(("%s was entered", method));

  tBool bRetVal = TRUE;
  tU32 u32FlashSectorNumber = 0;
  tU32 u32FlashSectorStart  = 0;
  tU32 u32FlashSectorEnd    = 0;
  tU32 u32VerifyCount       = 0;
  tU32 u32Size              = 0;
  tS32 s32Success           = 0;
  FILE* fdSourceFile = 0;

  tChar tmpBuffer[256] = { 0 };
  int fd = -1;
  int retcode;
  std::string devname;

  off_t u32FlashDevOffset = 0;
  off_t destinationOffsetRelative = 0;

  // Set the raw flash device name for the linux driver and adjust the flash address to the flash device offset
  bSetRawFlashDevName(inDestinationOffset, devname, &u32FlashDevOffset);
  destinationOffsetRelative = inDestinationOffset - u32FlashDevOffset;

  tCU32 BLOCK_SIZE   = 4096;
  tU32 blocks        = 0;
  tU32 lastChunkSize = 0;
  tBool breakForLoop = FALSE;
  tU32 sizeToRead    = 0;
  tU32 u32OffSet     = 0;

  void *writeIntoRawNorBuffer = 0; //this buffer is filled by the content of file
  void *readFromRawNorBuffer = 0; //this buffer is filled from OSAL_IOControl

  fdSourceFile = fopen(inSrcFile, "r");
  if (0 == fdSourceFile)
  {
    fprintf(stdout, "%s: Error opening file \"%s\"\n", __func__, inSrcFile);
    fprintf(stdout, "Error no is : %d\n", errno);
    fprintf(stdout, "Error description is : %s\n", strerror(errno));
    ETG_TRACE_ERR(("File %s could not be opened", inSrcFile));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "File open", NULL);
    NORMAL_M_ASSERT_ALWAYS ();

    return FALSE;
  }

  //
  // get size of file -----------------------------------------------------------------------
  //
  // seek to end of file
  s32Success = fseek(fdSourceFile, 0, SEEK_END);
  if (0 == s32Success)
    s32Success = OSAL_OK;
  // get current file pointer
  u32Size =static_cast<tU32> (ftell(fdSourceFile) );
  fseek(fdSourceFile, 0, SEEK_SET); // seek back to beginning of file

  ETG_TRACE_USR4(("DNL: u32SizeOfFile %d", u32Size));
  if (s32Success == OSAL_ERROR || !u32Size)
  {
    ETG_TRACE_ERR(("OSAL ERROR"));
    fclose(fdSourceFile);
    ETG_TRACE_USR4(("%s was left", method));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "File size", NULL);

    return FALSE;
  }

  ETG_TRACE_USR4(("size of file to be written is %d", u32Size));

  //how many blocks must be written
  blocks = u32Size / BLOCK_SIZE;
  ETG_TRACE_USR4(("blocks to be written are %d", blocks));

  // the size of last chunk
  lastChunkSize = u32Size % BLOCK_SIZE;
  ETG_TRACE_USR4(("last chunk size is %d", lastChunkSize));

  //erase procedure
  u32FlashSectorNumber = u32Size / (tU32) TRITON_FLASH_SECTOR_SIZE;

  if ((u32Size % (tU32) TRITON_FLASH_SECTOR_SIZE) != 0)
    u32FlashSectorNumber += 1;

  if (u32FlashSectorNumber > TRITON_FLASH_SECTOR_MAX_COUNT)
  {
    ETG_TRACE_USR4(
        ("DNL: Sector count exceeded, ERROR !!! %d", u32FlashSectorNumber));
    ETG_TRACE_FATAL(("DNL: RawNorSector count exceeded, ERROR !!!"));
//    dl_tclCoreUtilities::vWriteErrmemDNL(
//        U16_M_ERRMEM_DNL_ERROR(DNL_U8_ERRMEM_TYPE_STRING),
//        (tU8*) "DNL: RawFlash ERROR1", 20);
    fclose(fdSourceFile);
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_ERASE, "RawNorSector count exceeded", NULL);

    return FALSE;
  }

  u32FlashSectorStart =static_cast<tU32> (TRITON_FLASH_SECTOR_COUNT_OFFSET
      + destinationOffsetRelative / TRITON_FLASH_SECTOR_SIZE );

  u32FlashSectorEnd = u32FlashSectorStart + u32FlashSectorNumber - 1;
  ETG_TRACE_USR4(
      ("DNL: first sector to be written %d", u32FlashSectorStart));
  ETG_TRACE_USR4(("DNL: last  sector to be written %d", u32FlashSectorEnd));
  if (u32FlashSectorEnd >= TRITON_FLASH_SECTOR_MAX_COUNT)
  {
    ETG_TRACE_USR4(
        ("DNL: Last Sector exceeded, ERROR !!! %d", u32FlashSectorEnd));
    ETG_TRACE_FATAL(("DNL: RawNorLastSector exceeded, ERROR !!!"));
//    dl_tclCoreUtilities::vWriteErrmemDNL(
//        U16_M_ERRMEM_DNL_ERROR(DNL_U8_ERRMEM_TYPE_STRING),
//        (tU8*) "DNL: RawFlash ERROR2", 20);
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_ERASE, "Last Sector exceeded", NULL);
    fclose(fdSourceFile);

    return FALSE;
  }
  ETG_TRACE_USR4(("DNL: Attention, writing critical flash section %d", u32FlashSectorEnd));
  ETG_TRACE_USR4(("DNL: DO NOT POWER OFF TARGET"));

  //allocate memory for both buffers
  if (0 == (writeIntoRawNorBuffer = OSAL_pvMemoryAllocate(BLOCK_SIZE)))
  {
    ETG_TRACE_ERR(("can not allocate memory"));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "can not allocate memory", NULL);
    FATAL_M_ASSERT_ALWAYS ();
    fclose(fdSourceFile);
    return FALSE;
  }

  if (0 == (readFromRawNorBuffer = OSAL_pvMemoryAllocate(BLOCK_SIZE)))
  {
    ETG_TRACE_ERR(("can not allocate memory"));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "can not allocate memory", NULL);
    OSAL_vMemoryFree(writeIntoRawNorBuffer);
    FATAL_M_ASSERT_ALWAYS ();
    fclose(fdSourceFile);
    return FALSE;
  }

  // Open flash for write and read
  ETG_TRACE_USR4(("%s: Device Name", devname.c_str()));
  fd = open(devname.c_str(), O_RDWR, 0x1A4);
  if (fd >= 0)
  {
    ETG_TRACE_USR4(("DNL: Flash open OK"));
    if (bErase)
    {
      // erase raw flash
      {
        erase_info_user einfo;
        memset(&einfo, 0, sizeof(einfo));
        einfo.start = u32FlashSectorStart * (tU32) TRITON_FLASH_SECTOR_SIZE;
        einfo.length = u32FlashSectorNumber * (tU32) TRITON_FLASH_SECTOR_SIZE;

        sprintf(tmpBuffer,
            "erase flash adress      =0x%8.8X size=0x%8.8X",
            (int)(u32FlashSectorStart * (tU32)TRITON_FLASH_SECTOR_SIZE),
            (int)(u32FlashSectorNumber * (tU32)TRITON_FLASH_SECTOR_SIZE));
        printToDownloadPipe(tmpBuffer);

        retcode = ioctl(fd, MEMERASE, &einfo);

        if (retcode)
        {
          ETG_TRACE_ERR(("DNL: Flash sector erase ERROR !!!"));
            ETG_TRACE_USR4(("DNL: Flash sector erase ERROR !!! %d", u32FlashSectorNumber));
//          MTD_IO::vSendFlashErrorMessage(
//              DL_FLASH_ERROR_ERASE | DL_FLASH_NOR_RAW);
    	    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_ERASE, "sector erase", NULL);
            bRetVal = FALSE;
            printToDownloadPipe("erase flash device ERROR");
        }
      }

    } //if (bErase)

    // Flash erase OK:
    ETG_TRACE_USR4(("destinationOffsetRelative %d = 0x%8.8X", destinationOffsetRelative, destinationOffsetRelative));
    ETG_TRACE_USR4(("DNL: Flash binary %s write start", inSrcFile));

    //go through all the blocks
    for (tU32 i = 0; (!breakForLoop && i<=blocks); i++)
    {
      if (i < blocks)
      {
        //deal with normal block size
        sizeToRead = BLOCK_SIZE;
      }
      else if (lastChunkSize > 0)
      {
        ETG_TRACE_USR4(("All blocks were read / written correctly"));
        ETG_TRACE_USR4(("Now read / write the last chunk"));
        //deal with the last chunk, then break the for loop
        breakForLoop = TRUE;
        sizeToRead = lastChunkSize;
      }
      else //lastChunkSize == 0
      {
        //do nothing
        break;
      }

      tU32 readBytes = 0;
      readBytes =static_cast<tU32> (fread(writeIntoRawNorBuffer, 1, sizeToRead, fdSourceFile) );

      //now read from the file
      if (readBytes != sizeToRead)
      {
        ETG_TRACE_ERR(("not all bytes could be read"));
        bRetVal = FALSE;
        iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, (int)SW_UPDATE_ERROR_TYPE_READ, "not all bytes could be read", NULL);
        NORMAL_M_ASSERT_ALWAYS ();
        break;
      }

      //compute the internal offset to read / write
      u32OffSet = BLOCK_SIZE * i;

      ETG_TRACE_USR4(("Offset is equal to %d", u32OffSet));
      ETG_TRACE_USR4(("Size to read / write is %d", sizeToRead));

      ETG_TRACE_USR4(("Flash address to read / write is %d or hex 0x%x",
          destinationOffsetRelative + u32OffSet,
          destinationOffsetRelative + u32OffSet));

      // write to raw flash
      retcode =static_cast<int> (pwrite(fd, writeIntoRawNorBuffer, sizeToRead, destinationOffsetRelative + u32OffSet) );

      if (retcode != static_cast<int> (sizeToRead)) // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      {
        ETG_TRACE_USR4(("Raw flash write ERROR"));
//        MTD_IO::vSendFlashErrorMessage(
//            DL_FLASH_ERROR_WRITE | DL_FLASH_NOR_RAW);
        iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "raw flash write", NULL);
        bRetVal = FALSE;
        printToDownloadPipe("write flash device ERROR");
        break;
      }

      // read back from raw flash
      retcode =static_cast<int> (pread(fd, readFromRawNorBuffer, sizeToRead, destinationOffsetRelative + u32OffSet) );

      if (retcode != static_cast<int> (sizeToRead)) // gen3armake, gen3x86make, gen4lsim, gen4rcar: comparison between signed and unsigned integer expressions
      {
        ETG_TRACE_USR4(("Raw flash read ERROR"));
        iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_READ, "raw flash read", NULL);
        bRetVal = FALSE;
        printToDownloadPipe("read back flash device ERROR");
        break;
      }

      //now compare the result from RAW NOR Flash
      //and the read one from buffer
      if (0 != memcmp(writeIntoRawNorBuffer, readFromRawNorBuffer, sizeToRead))
      {
        tCU32 ERROR_BUFSIZE = 80;
        tChar szErrorString[ERROR_BUFSIZE] =  { 0 };

        snprintf(szErrorString, ERROR_BUFSIZE, "0x%8.8X",
            (unsigned int) (TRITON_FLASH_BASE_ADDR + destinationOffsetRelative + u32VerifyCount));

        ETG_TRACE_USR4(("DNL: RawFlashVerifyError !!!"));
        iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_VERIFY, "raw flash verify", NULL);

        printToDownloadPipe("compare flash device ERROR");


        bRetVal = FALSE;
        break;
      }
    }

    ETG_TRACE_USR4(("DNL: Flash binary %s write end", inSrcFile));
    if (bRetVal)
    {
      printToDownloadPipe("writing flash device done, OK");
    }
    else
    {
      iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "raw flash write", NULL);
      printToDownloadPipe("writing flash device ERROR");
    }
    if (fd >= 0)
    {
      fsync(fd);	// write buffer from os to physical device
      close(fd);
    }
  }
  else // flash open error:
  {
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "raw flash open", NULL);
    ETG_TRACE_USR4(("DNL: Flash open ERROR"));
    bRetVal = FALSE;
  }

  if (!bRetVal)
  {
    ETG_TRACE_FATAL(("DNL: RawFlash ERROR !!!"));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "raw flash write", NULL);
  }

  //release allocated memory
  OSAL_vMemoryFree(writeIntoRawNorBuffer);  // allocated as function is returned in case of allocation not possible
  OSAL_vMemoryFree(readFromRawNorBuffer); // allocated as function is returned in case of allocation not possible

  fclose(fdSourceFile);

  ETG_TRACE_USR4(("%s was left", method));
  ETG_TRACE_USR4(
      ("method was left with return value %s", bRetVal ? "TRUE" : "FALSE"));

  return bRetVal;
}

/*******************************************************************************/
/*
 *  Function   : bSetRawFlashDevName
 *  Description: Sets the raw nor flash device name for the linux driver
 *               and sets the raw nor flash address offset for this raw flash device
 *
 *  Parameter  : tU32     u32FlashAdr
  *
 *  Returns    : tCString *pCStrRawFlashDevName
 *             : tPU32    pu32FlashDevOffset
 *             : tBool    bRet
 *               TRUE  -> no error
 *               FALSE -> error
 *
 */
/*******************************************************************************/

tBool bSetRawFlashDevName(
    off_t inFlashAdr,
    std::string& outFlashDevName,
    off_t* outFlashDevOffset)
{
  tCString method = "MTD_IO::bSetRawFlashDevName";
  tBool    bRet   = TRUE;

  if (bIsPartitionRead == FALSE){
    u8FlopPartitionCount = 0;
    /**Get MTD partition info*/
    flopDevPartitions.clear();
    tU8 u8Ret = u8GetMTDPartitions();

    ETG_TRACE_USR4 (("%d :sRet", u8Ret));

    bIsPartitionRead = TRUE;
  }

  ETG_TRACE_USR4 (("%s was entered", method));

  tU8 u8Partition = U8_MAX_SIZE;
  for(tU8 u8Index=0; u8Index < u8FlopPartitionCount; u8Index++)
  {
    ETG_TRACE_USR4 (("%x offset", flopDevPartitions[u8Index].u32Offset));
    if(flopDevPartitions[u8Index].u32Offset <= inFlashAdr &&
        inFlashAdr < (flopDevPartitions[u8Index].u32Offset + flopDevPartitions[u8Index].u32Size))
    {
      *outFlashDevOffset = flopDevPartitions[u8Index].u32Offset;
      u8Partition = u8Index;
      ETG_TRACE_USR4 (("u8Index %d", u8Index));
      break;
    }
  }

  ETG_TRACE_USR4 (("%d:mtd",inFlashAdr));
  if (U8_MAX_SIZE == u8Partition)
  {
    ETG_TRACE_USR4(("Could not resolve the address\n"));
    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW, (int)SW_UPDATE_ERROR_TYPE_WRITE, "could not resolve the address", NULL);
    bRet = FALSE;
    return bRet;
  }

  // build flash device string
  std::stringstream sstr;
  const int partNumber = static_cast< int >(u8Partition);
  sstr << "/dev/mtd" << partNumber;   //@todo: replace hardcoded device path for better maintainability

  outFlashDevName = sstr.str();

  ETG_TRACE_USR4 (("%s:mtd",outFlashDevName.c_str()));
  ETG_TRACE_USR4 (("%d:mtd",flopDevPartitions[u8Partition].u32Offset));
  ETG_TRACE_USR4 (("bRet=%d", bRet));
  ETG_TRACE_USR4 (("%s was left", method));

  return bRet;
}


/*
* Function    : s32GetMTDPartitions
* Description   : To get Partition Layout of the NOR Flash Chip.
* Return value    : integer
*/
tU8 u8GetMTDPartitions(tVoid)
{
    FILE* pcFileProcMtdFd = NULL;
    size_t tlineLen = 0;
    off_t u32Offset = 0;
    tChar *cBuf = NULL;
    struct MtdPartitionInfo rMtdInfo;

    u8FlopPartitionCount = 0;

    pcFileProcMtdFd = fopen(MTD_PROC_FILE,"r"); // Open the /proc/mtd file
    if(0 != pcFileProcMtdFd)
    {
        while (-1 != getline(&cBuf,&tlineLen,pcFileProcMtdFd))
        {
            tU32 u32Size = 0;
            tU32 u32EraseSize = 0;
            tS32 s32Matches = 0;
            tU8 u8Num = U8_MAX_SIZE;
            tChar cName[MTD_PARTITION_NAME_SIZE];
            cName[0] = '\0';

                if(cBuf != NULL)
              s32Matches = sscanf(cBuf, "mtd%d: %x %x \"%63[^\"]",
                  (int*)&u8Num, (unsigned int*)&u32Size,(unsigned int*)&u32EraseSize, cName);

            ETG_TRACE_USR4 (("%d:mtd",u8Num));
            ETG_TRACE_USR4 (("%x:mtd size",u32Size));

            if (4 == s32Matches)
            {
              rMtdInfo.u8Index = u8Num;
              rMtdInfo.u32Size = u32Size;
              rMtdInfo.u32EraseSize = u32EraseSize;
              rMtdInfo.u32Offset = u32Offset;

              flopDevPartitions[u8FlopPartitionCount] = rMtdInfo;

              u8FlopPartitionCount++;
              u32Offset += u32Size;
            }
        }

	  fclose(pcFileProcMtdFd);
    }
    else {
      ETG_TRACE_USR4 (("u8GetMTDPartitions: FileOpenError"));
    }

    ETG_TRACE_USR4 ((" u8FlopPartitionCount %d",u8FlopPartitionCount));
    return u8FlopPartitionCount;
}

} // namespace ai_sw_update {
} // namespace common {


