/**********************************************************************************************
* FILE:           dl_tclDABPlusMain.cpp
* PROJECT:        Nissan
* SW-COMPONENT:   Download
*-----------------------------------------------------------------------------------------------
* DESCRIPTION:  DAB Plus Firmware Main Class
*
* This file is the class dl_tclDABPlusMain defined. An object of this class is created by the
* dl_Copy class. This class is a singletone class.
*------------------------------------------------------------------------------------------------
* COPYRIGHT: (c) 2011 Robert Bosch Car Multimedia GmbH, Hildesheim
* HISTORY:
* Date       |Author								  |Modification
* 21.02.11   |Ghatikar Nageetha (CM-AI/PJ-VW36 RBEI)  |Initial
*************************************************************************************************/

//----------------------------------------------------------------------------------------
// includes
//----------------------------------------------------------------------------------------
#include <stdio.h>

#ifdef DL_ADR3_GEN2_CORE
#include "../core/dl_MessageCodes.h"
#include "../core/dl_FlashOperations.h"
#include "../core/dl_ThreadLib.h"
#endif
#include "dl_tclDABPlusMain.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
/* Needed for Trace */
#define ETG_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etg_if.h"
#endif

/* Needed for ETG trace */
#ifdef DL_ADR3_GEN2_CORE
#include "dl_Adr3TraceMacros.h"
#else
#include "ai_sw_update/common/base/imp/swupd_trace.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL TR_TTFIS_SWUPDATE_CTRL
#define ETG_I_TTFIS_CMD_PREFIX "DNL_"
#define ETG_I_FILE_PREFIX dl_tclDABPlusMain::
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_ADR3
#include "trcGenProj/Header/dl_tclDABPlusMain.cpp.trc.h"
#endif


#ifndef DL_ADR3_GEN2_CORE
#include "base/imp/gen2_AdminData.h"
#include "base/imp/gen2_FileOperations.h"
#endif

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#include "dldab_ProcessCmd.h"
#include "dl_tclHelp.h"

/*******************************************************************************************/
#ifndef __SW_UPDATE_UNIT_TESTING__
// Implementation of the methods getInstanceOfdl_tclDABPlusMain and releaseInstanceOfdl_tclDABPlusMain
// In case of __SW_UPDATE_UNIT_TESTING__ they are implemented in mock!
SWU_IMPL_GET_AND_RELEASE_INSTANCE(dl_tclDABPlusMain);
#endif

// Implementation of the singleton methods
SWU_IMPL_SINGLETON(dl_tclDABPlusMain);
/*******************************************************************************************/

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::dl_tclDABPlusMain()
*
* DESCRIPTION:  Constructor for dl_tclDABPlusMain class
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
dl_tclDABPlusMain::dl_tclDABPlusMain():m_poProcessCmd(OSAL_NULL),
                                       myXloaderFileName(),
                                       myAdr3FileName()
{
	ETG_TRACE_USR3(("dl_tclDABPlusMain::Constructor"));
	ETG_I_REGISTER_FILE();

	m_poProcessCmd = getInstanceOfdldab_tclProcessCmd();
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::~dl_tclDABPlusMain()
*
* DESCRIPTION:  Destructor for dl_tclDABPlusMain class
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
dl_tclDABPlusMain::~dl_tclDABPlusMain()
{
	ETG_TRACE_USR3(("dl_tclDABPlusMain::Destructor"));

	if(m_poProcessCmd != OSAL_NULL)
	{
		releaseInstanceOfdldab_tclProcessCmd();
		m_poProcessCmd = OSAL_NULL;
	}
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::u32GetSizeOfAdmin()
* 
* DESCRIPTION:  calculate the size of the admin block (with crc etc.) at the end of the file
*
* PARAMETER:	pu8Buffer, u32SizeOfFile
*
* RETURN VALUE: size of the admin block
*
******************************************************************************************/
tU32 dl_tclDABPlusMain::u32GetSizeOfAdmin(tPU8 pu8Buffer, tU32 u32SizeOfFile ) const
{
	tCString method = "dl_tclDABPlusMain::u32GetSizeOfAdmin";
	ETG_TRACE_USR3 (("%s was entered", method));
    tU32  u32SizeOfAdmin;
    tBool bEndOfAdminFound  = FALSE;
    tPU8  pu8LastByteOfFile = pu8Buffer + u32SizeOfFile - 1;
    
    for( u32SizeOfAdmin = 0 ; !bEndOfAdminFound && (u32SizeOfAdmin < ( u32SizeOfFile - 8 )) ; u32SizeOfAdmin++)
    {
       bEndOfAdminFound = 
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 7)) == 'A') &&
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 6)) == 'D') &&
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 5)) == 'M') &&
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 4)) == 'I') &&
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 3)) == 'N') &&
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 2)) == 'E') &&
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 1)) == 'N') &&
          (*(pu8LastByteOfFile - (u32SizeOfAdmin + 0)) == 'D') ; 
       
       if(bEndOfAdminFound) break;
    }
	ETG_TRACE_USR3 (("%s was left", method));
    return (u32SizeOfAdmin + 8 );
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunSequence()
* 
* DESCRIPTION:  This is the main function to update adr3. The content is shifted to
*               bRunSequenceInternal. This function implements a workaround for a HW error
*               This HW-Error leads to an error during the download of the XLOADER.
*               If this error has been detected in the lower level, this function 
*               performs up to 3 retries!
*
* PARAMETER:	xloaderFileName, adr3FileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunSequence(tCString xloaderFileName, tCString adr3FileName) const
{
	tCString method = "dl_tclDABPlusMain::bRunSequence";
	ETG_TRACE_USR3 (("%s was entered", method));
	tBool bRetVal = FALSE;
    tU32 u32Trials = 0;
	tU32 u32LastStatus = 0;

	dl_tclHelp::vStartTrace();
	dl_tclHelp::vTrace("Start ADR3 download in bRunSequence");
    do
	{
	    u32Trials++;
	    bRetVal = getInstanceOfdl_tclDABPlusMain()->bRunSequenceXloader(xloaderFileName);
		if(bRetVal==TRUE) // Proceed only if xloader is downloaded properly
		{
			bRetVal = getInstanceOfdl_tclDABPlusMain()->bRunSequenceWriteFile(adr3FileName);
		}
		releaseInstanceOfdl_tclDABPlusMain();
		u32LastStatus = u32GetStatusADR3();
	} while ((u32LastStatus == DL_ADR3_NO_XLOADER_RESPONSE) && (u32Trials < 3));

	if(bRetVal==TRUE) // Last Message before end of ADR3 download...
	{
		ETG_TRACE_USR3(("Download Finished Main activated !!!"));
		dl_tclHelp::vTrace("Finished ADR3 download with SUCCESS");
	}
	else
	{
		ETG_TRACE_FATAL(("Flashing of ADR3 failed!!!"));
		dl_tclHelp::vTrace("Finished ADR3 download with ERROR");
	}
	
	dl_tclHelp::vStopTrace();
	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunReadSequence()
*
* DESCRIPTION:  This is the main function to read out the content of the adr3.
*               This function also implements a workaround for a HW error
*               This HW-Error leads to an error during the download of the XLOADER.
*               If this error has been detected in the lower level, this function
*               performs up to 3 retries!
*
* PARAMETER:	xloaderFileName, adr3FileName, outFileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunReadSequence(tCString xloaderFileName, tCString adr3FileName, tCString outFileName) const
{
	tCString method = "dl_tclDABPlusMain::bRunReadSequence";
	ETG_TRACE_USR3 (("%s was entered", method));
	tBool bRetVal = FALSE;
    tU32 u32Trials = 0;
	tU32 u32LastStatus = 0;

	dl_tclHelp::vStartTrace();
	dl_tclHelp::vTrace("Start to read out from ADR3 in bRunReadSequence");
    do
	{
	    u32Trials++;
	    bRetVal = getInstanceOfdl_tclDABPlusMain()->bRunSequenceXloader(xloaderFileName);
		if(bRetVal==TRUE)
		{
			bRetVal = getInstanceOfdl_tclDABPlusMain()->bRunSequenceReadFile(adr3FileName, outFileName);
		}
		releaseInstanceOfdl_tclDABPlusMain();
		u32LastStatus = u32GetStatusADR3();
	} while ((u32LastStatus == DL_ADR3_RESPONSE_TIMEOUT) && (u32Trials < 3));
	// NOTE: In case of READ DL_ADR3_RESPONSE_TIMEOUT is reported instead of DL_ADR3_NO_XLOADER_RESPONSE

	if(bRetVal==TRUE)
	{
		ETG_TRACE_USR3(("Read out of ADR3 succeeded"));
		dl_tclHelp::vTrace("Read out of ADR3 succeeded");
	}
	else
	{
		ETG_TRACE_FATAL(("Read out of ADR3 failed"));
		dl_tclHelp::vTrace("Read out of ADR3 failed");
	}

	dl_tclHelp::vStopTrace();
	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunReadVersionSequence()
*
* DESCRIPTION:  This is the main function to read out the version of the adr3.
*               This function also implements a workaround for a HW error
*               This HW-Error leads to an error during the download of the XLOADER.
*               If this error has been detected in the lower level, this function
*               performs up to 3 retries!
*
* PARAMETER:	xloaderFileName, adr3FileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunReadVersionSequence(tCString xloaderFileName, tU32 adr3AddressOfVersion, tCString versionFileName) const
{
	tCString method = "dl_tclDABPlusMain::bRunReadVersionSequence";
	ETG_TRACE_USR3 (("%s was entered", method));
	tBool bRetVal = FALSE;
    tU32 u32Trials = 0;
	tU32 u32LastStatus = 0;
	tU8 au8RamBuffer[MAX_DAB_DATABYTES_TOBEREAD];

	// Try to read from registry first
	bRetVal = bGetVersionFromRegistry(au8RamBuffer, MAX_DAB_DATABYTES_TOBEREAD);
	if(bRetVal==TRUE)
	{
		bRetVal = bWriteVersionToFile(au8RamBuffer, versionFileName);
		// Immediately return because there is no need to load xloader or trace anything...
		return bRetVal;
	}

	dl_tclHelp::vStartTrace();
	dl_tclHelp::vTrace("Start to read out version from ADR3 in bRunReadVersionSequence");

    do
	{
	    u32Trials++;
	    bRetVal = getInstanceOfdl_tclDABPlusMain()->bRunSequenceXloader(xloaderFileName);
		if(bRetVal==TRUE)
		{
			bRetVal = getInstanceOfdl_tclDABPlusMain()->bRunSequenceReadVersion(adr3AddressOfVersion, versionFileName);
		}
		releaseInstanceOfdl_tclDABPlusMain();
		u32LastStatus = u32GetStatusADR3();
	} while ((u32LastStatus == DL_ADR3_RESPONSE_TIMEOUT) && (u32Trials < 3));
	// NOTE: In case of READ DL_ADR3_RESPONSE_TIMEOUT is reported instead of DL_ADR3_NO_XLOADER_RESPONSE

	if(bRetVal==TRUE)
	{
		ETG_TRACE_USR3(("Read Version of ADR3 via XLOADER succeeded"));
		dl_tclHelp::vTrace("Read Version of ADR3 via XLOADER succeeded");
		printf("Read Version of ADR3 via XLOADER succeeded\n");
	}
	else
	{
		ETG_TRACE_FATAL(("Read Version of ADR3 failed"));
		dl_tclHelp::vTrace("Read Version of ADR3 failed");
	}

	dl_tclHelp::vStopTrace();
	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunSequenceXloader()
* 
* DESCRIPTION:  Write XLoader
*
* PARAMETER:	xloaderFileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunSequenceXloader(tCString xloaderFileName) const
{
	tCString method = "dl_tclDABPlusMain::bRunSequenceXloader";
	ETG_TRACE_USR3 (("%s was entered", method));

	tBool bRetVal = FALSE;
	tU32 u32SizeOfxLoaderFile = 0;
	tU32 u32AdminSize = 0;
	OSAL_tIODescriptor fdSourceFile = OSAL_ERROR;
	tBool stopRequested = FALSE;
	tPU8 pu8RamBuffer = 0;


	if(m_poProcessCmd == OSAL_NULL)
	{
	    NORMAL_M_ASSERT_ALWAYS();
		return FALSE;
	}


	// Open file to check if its available
	if ((fdSourceFile = OSAL_IOOpen (xloaderFileName, OSAL_EN_READONLY)) == OSAL_ERROR)
	{
		ETG_TRACE_FATAL (("File %s could not be opened", xloaderFileName));
		NORMAL_M_ASSERT_ALWAYS ();
		char buf[256];
		snprintf (buf, 255, "File %s could not be opened", xloaderFileName);
		dl_tclHelp::vTrace(buf);
		return FALSE;
	}
	OSAL_s32IOClose(fdSourceFile);


    // Read xloader file to RAM buffer
#ifdef DL_ADR3_GEN2_CORE
    bRetVal = dl_tclFlashOperations::bCopyFileToRAM (xloaderFileName, (tPVoid *)&pu8RamBuffer, &u32SizeOfxLoaderFile, &stopRequested);
#else
	bRetVal = ai_sw_update::common::bCopyFileToRAM (xloaderFileName, (tPVoid *)&pu8RamBuffer, &u32SizeOfxLoaderFile, &stopRequested);
#endif
	if(bRetVal==TRUE)
	{
		ETG_TRACE_USR3 (("Size of xloader to be written is %d", u32SizeOfxLoaderFile));
	}
	else
	{
		ETG_TRACE_FATAL(("bCopyFileToRAM failed -- file corrupted?!"));
		dl_tclHelp::vTrace("bCopyFileToRAM failed -- file corrupted?!");
	}

	if ((pu8RamBuffer == NULL) || (u32SizeOfxLoaderFile == 0))
	{
		ETG_TRACE_FATAL (("(pu8RamBuffer == NULL) || (u32SizeOfxLoaderFile == 0)"));
		dl_tclHelp::vTrace("(pu8RamBuffer == NULL) || (u32SizeOfxLoaderFile == 0)");
        NORMAL_M_ASSERT_ALWAYS();
		bRetVal = FALSE;
	}

	if(bRetVal==TRUE) // Proceed only if xloader is copied to RAM correctly
	{
		// Ignore admin data at the end
#ifdef DL_ADR3_GEN2_CORE
        u32AdminSize = u32GetSizeOfAdmin(pu8RamBuffer,u32SizeOfxLoaderFile); 
#else
        u32AdminSize = ai_sw_update::common::u32GetSizeOfAdmin(pu8RamBuffer,u32SizeOfxLoaderFile);
#endif
        if (u32SizeOfxLoaderFile <= u32AdminSize)
	    {
			ETG_TRACE_FATAL (("(u32SizeOfxLoaderFile <= u32AdminSize)"));
			dl_tclHelp::vTrace("(u32SizeOfxLoaderFile <= u32AdminSize)");
	        NORMAL_M_ASSERT_ALWAYS();
            bRetVal = FALSE;
	    }
	}

	if(bRetVal==TRUE) // Proceed only if size of admin is plausible
	{
	    u32SizeOfxLoaderFile -= u32AdminSize;
		bRetVal = m_poProcessCmd->WriteXLoaderCmd(pu8RamBuffer,u32SizeOfxLoaderFile);
        if(bRetVal==TRUE)
		{
			//Stores the buffer pointer and the size of the file
			bRetVal = m_poProcessCmd->bHandleInitilization(); //Start uploading xloader
		}
		else
		{
			ETG_TRACE_FATAL(("m_poProcessCmd->WriteXLoaderCmd failed!!!"));
			dl_tclHelp::vTrace("m_poProcessCmd->WriteXLoaderCmd failed!!!");
		}
	}

	// Free buffer after downloading XLoader
	if (pu8RamBuffer != NULL)
	{
		OSAL_vMemoryFree(pu8RamBuffer);
		pu8RamBuffer = 0;
	}

	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunSequenceWriteFile()
* 
* DESCRIPTION:  read ADR
*
* PARAMETER:	adr3FileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunSequenceWriteFile(tCString adr3FileName) const
{
	tCString method = "dl_tclDABPlusMain::bRunSequenceWriteFile";
	ETG_TRACE_USR3 (("%s was entered", method));

	tBool bRetVal = FALSE;
	tU32 u32AdminSize = 0;
	OSAL_tIODescriptor fdSourceFile = OSAL_ERROR;
	tU32 u32SizeOfFlashFile = 0;
	tBool stopRequested = FALSE;
	tPU8 pu8RamBuffer = 0;


	if(m_poProcessCmd == OSAL_NULL)
	{
	    NORMAL_M_ASSERT_ALWAYS();
		return FALSE;
	}

		ETG_TRACE_USR3(("Try to open %s",adr3FileName));
		// Code to upload the Flash data file
		// Open file to check if its available
		if ((fdSourceFile = OSAL_IOOpen (adr3FileName, OSAL_EN_READONLY)) == OSAL_ERROR)
		{
			ETG_TRACE_FATAL (("File %s could not be opened", adr3FileName));
			NORMAL_M_ASSERT_ALWAYS ();
			char buf[256];
			snprintf (buf, 255, "File %s could not be opened", adr3FileName);
			dl_tclHelp::vTrace(buf);
			return FALSE;
		}
		OSAL_s32IOClose(fdSourceFile);

		// Read source file to RAM buffer
#ifdef DL_ADR3_GEN2_CORE
		bRetVal = dl_tclFlashOperations::bCopyFileToRAM (adr3FileName, (tPVoid *)&pu8RamBuffer, &u32SizeOfFlashFile, &stopRequested);
#else
		bRetVal = ai_sw_update::common::bCopyFileToRAM (adr3FileName, (tPVoid *)&pu8RamBuffer, &u32SizeOfFlashFile, &stopRequested);
#endif
		if(bRetVal==TRUE)
		{
			ETG_TRACE_USR3 (("Size of flash file to be written is %d", u32SizeOfFlashFile));
		}
		else
		{
			ETG_TRACE_FATAL(("bCopyFileToRAM failed -- file corrupted?!"));
			dl_tclHelp::vTrace("bCopyFileToRAM failed -- file corrupted?!");
		}

		if ((pu8RamBuffer == NULL) || (u32SizeOfFlashFile == 0))
		{
			ETG_TRACE_FATAL (("(pu8RamBuffer == NULL) || (u32SizeOfFlashFile == 0)"));
			NORMAL_M_ASSERT_ALWAYS();
			dl_tclHelp::vTrace("(pu8RamBuffer == NULL) || (u32SizeOfFlashFile == 0)");
			bRetVal = FALSE;
		}

		if(bRetVal==TRUE)
		{
			//Ignore admin data at the end
#ifdef DL_ADR3_GEN2_CORE
		    u32AdminSize = u32GetSizeOfAdmin(pu8RamBuffer,u32SizeOfFlashFile); 
#else
		    u32AdminSize = ai_sw_update::common::u32GetSizeOfAdmin(pu8RamBuffer,u32SizeOfFlashFile);
#endif
            //Wrong Admin-block detected
            if (u32SizeOfFlashFile <= u32AdminSize)
		    {
				ETG_TRACE_FATAL (("(u32SizeOfFlashFile <= u32AdminSize)"));
		        NORMAL_M_ASSERT_ALWAYS();
				dl_tclHelp::vTrace("(u32SizeOfFlashFile <= u32AdminSize)");
                bRetVal = FALSE;
			}
		}

		if(bRetVal==TRUE) // Proceed only if size of admin is plausible
		{
            u32SizeOfFlashFile -= u32AdminSize;
			// Calculating the Checksum for complete flash file
			ETG_TRACE_USR3(("Calculating the Checksum for complete flash file before doing anything else!!!"));
			m_poProcessCmd->u32ComputeBufCRC2007(pu8RamBuffer, u32SizeOfFlashFile);
			//Stores the buffer pointer and the size of the file
			bRetVal = m_poProcessCmd->bHandleErase(pu8RamBuffer, u32SizeOfFlashFile); // Start flash erase
			if(bRetVal==TRUE)
			{
				ETG_TRACE_USR3(("XLoader has been flashed and erase is done!!!"));
				// Function to Start Data File downloading
				bRetVal = m_poProcessCmd->bHandleDataTransfer(pu8RamBuffer,u32SizeOfFlashFile);
			}
			else
			{
				ETG_TRACE_FATAL(("Erase failed"));
				dl_tclHelp::vTrace("Erase failed");
			}
		}

	// Free buffer after downloading File
	if (pu8RamBuffer != NULL)
	{
		OSAL_vMemoryFree(pu8RamBuffer);
		pu8RamBuffer = 0;
	}


	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunSequenceReadFile()
* 
* DESCRIPTION:  read ADR
*
* PARAMETER:	adr3FileName, outFileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunSequenceReadFile(tCString adr3FileName, tCString outFileName) const
{
	tCString method = "dl_tclDABPlusMain::bRunSequenceReadFile";
	ETG_TRACE_USR3 (("%s was entered", method));

	tBool bRetVal = FALSE;
	tU32 u32AdminSize = 0;
	OSAL_tIODescriptor fdSourceFile = OSAL_ERROR;
	FILE * fdOutFile = NULL;
	tU32 u32SizeOfFlashFile = 0;
	tBool stopRequested = FALSE;
	tPU8 pu8RamBuffer = 0;

	if(m_poProcessCmd == OSAL_NULL)
	{
	    NORMAL_M_ASSERT_ALWAYS();
		return FALSE;
	}

	ETG_TRACE_USR3(("Try to open %s",adr3FileName));
	// Open file to check if its available
	if ((fdSourceFile = OSAL_IOOpen (adr3FileName, OSAL_EN_READONLY)) == OSAL_ERROR)
	{
		ETG_TRACE_FATAL (("File %s could not be opened", adr3FileName));
		NORMAL_M_ASSERT_ALWAYS ();
		char buf[256];
		snprintf (buf, 255, "File %s could not be opened", adr3FileName);
		dl_tclHelp::vTrace(buf);
		return FALSE;
	}
	OSAL_s32IOClose(fdSourceFile);

	// Read source file to RAM buffer
#ifdef DL_ADR3_GEN2_CORE
	bRetVal = dl_tclFlashOperations::bCopyFileToRAM (adr3FileName, (tPVoid *)&pu8RamBuffer, &u32SizeOfFlashFile, &stopRequested);
#else
	bRetVal = ai_sw_update::common::bCopyFileToRAM (adr3FileName, (tPVoid *)&pu8RamBuffer, &u32SizeOfFlashFile, &stopRequested);
#endif
	if(bRetVal==TRUE)
	{
		ETG_TRACE_USR3 (("Size of flash file to be read is %d", u32SizeOfFlashFile));
	}
	else
	{
		ETG_TRACE_FATAL(("bCopyFileToRAM failed -- file corrupted?!"));
		dl_tclHelp::vTrace("bCopyFileToRAM failed -- file corrupted?!");
	}

	if ((pu8RamBuffer == NULL) || (u32SizeOfFlashFile == 0))
	{
		ETG_TRACE_FATAL (("(pu8RamBuffer == NULL) || (u32SizeOfFlashFile == 0)"));
		NORMAL_M_ASSERT_ALWAYS();
		dl_tclHelp::vTrace("(pu8RamBuffer == NULL) || (u32SizeOfFlashFile == 0)");
		bRetVal = FALSE;
	}

	if(bRetVal==TRUE)
	{
		//Ignore admin data at the end
#ifdef DL_ADR3_GEN2_CORE
	    u32AdminSize = u32GetSizeOfAdmin(pu8RamBuffer,u32SizeOfFlashFile);
#else
	    u32AdminSize = ai_sw_update::common::u32GetSizeOfAdmin(pu8RamBuffer,u32SizeOfFlashFile);
#endif
	    //Wrong Admin-block detected
	    if (u32SizeOfFlashFile <= u32AdminSize)
	    {
			ETG_TRACE_FATAL (("(u32SizeOfFlashFile <= u32AdminSize)"));
	        NORMAL_M_ASSERT_ALWAYS();
			dl_tclHelp::vTrace("(u32SizeOfFlashFile <= u32AdminSize)");
			bRetVal = FALSE;
		}
	}

	if(bRetVal==TRUE) // Proceed only if size of admin is plausible
	{
		u32SizeOfFlashFile -= u32AdminSize;
		// Read out from flash
		bRetVal = m_poProcessCmd->bHandleRead(pu8RamBuffer, u32SizeOfFlashFile);
		if(bRetVal==TRUE)
		{
			ETG_TRACE_USR3(("READ is done!!!"));
		}
		else
		{
			ETG_TRACE_FATAL(("READ failed"));
			dl_tclHelp::vTrace("READ failed");
		}
	}

	if((bRetVal==TRUE) && (pu8RamBuffer != NULL)) // (pu8RamBuffer != NULL) to satisfy LINT
	{
		if ((fdOutFile = fopen (outFileName, "w")) == NULL)
		{
			ETG_TRACE_FATAL (("File %s could not be opened", outFileName));
			NORMAL_M_ASSERT_ALWAYS ();
			char buf[256];
			snprintf (buf, 255, "File %s could not be opened", outFileName);
			dl_tclHelp::vTrace(buf);
			bRetVal = FALSE;
		}
		else
		{
			tU32 u32Blocksize = u32SizeOfFlashFile;
			tU32 u32BytesOfBlockWritten = 0;
			tU32 u32BytesWritten = 0;

			do
			{
				if (u32Blocksize > (u32SizeOfFlashFile - u32BytesWritten))
				{
					u32Blocksize = u32SizeOfFlashFile - u32BytesWritten;
				}
				u32BytesOfBlockWritten = (tU32)fwrite((void *)(pu8RamBuffer + u32BytesWritten), 1, u32Blocksize, fdOutFile);
				if((tS32)u32BytesOfBlockWritten == 0)
				{
					u32BytesWritten = 0;
					ETG_TRACE_ERR (("bRunSequenceReadFile - Cannot Write Block"));
					bRetVal = FALSE;
					break;
				}
				u32BytesWritten += u32BytesOfBlockWritten;
			} while(u32BytesWritten < u32SizeOfFlashFile);
			fclose(fdOutFile);
		}
	}
	else
	{
		ETG_TRACE_FATAL(("READ of Flash failed (no file written)!!!"));
		dl_tclHelp::vTrace("READ of Flash failed (no file written)!!!");
	}

	// Free buffer after downloading File
	if (pu8RamBuffer != NULL)
	{
		OSAL_vMemoryFree(pu8RamBuffer);
		pu8RamBuffer = 0;
	}

	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunSequenceReadVersion()
* 
* DESCRIPTION:  read ADR Version
*
* PARAMETER:	adr3AddressOfVersion, versionFileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunSequenceReadVersion(tU32 adr3AddressOfVersion, tCString versionFileName) const
{
	tCString method = "dl_tclDABPlusMain::bRunSequenceReadVersion";
	ETG_TRACE_USR3 (("%s was entered", method));

	tBool bRetVal = FALSE;
	tU8 au8RamBuffer[MAX_DAB_DATABYTES_TOBEREAD];

	
	if(m_poProcessCmd == OSAL_NULL)
	{
	    NORMAL_M_ASSERT_ALWAYS();
		return FALSE;
	}

	// Read out from flash
	bRetVal = m_poProcessCmd->bHandleReadVersion(au8RamBuffer, adr3AddressOfVersion, MAX_DAB_DATABYTES_TOBEREAD);

	if(bRetVal==TRUE)
	{
		ETG_TRACE_USR3(("READ Version is done!!!"));
		bRetVal = bWriteVersionToFile(au8RamBuffer, versionFileName);
	}
	else
	{
		ETG_TRACE_FATAL(("READ of Version failed (no file written)!!!"));
		dl_tclHelp::vTrace("READ of Version failed (no file written)!!!");
	}

	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bGetVersionFromRegistry()
*
* DESCRIPTION:  return the ADR3 status
*
* PARAMETER:	au8RamBuffer, u32Length
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bGetVersionFromRegistry (tPU8 au8RamBuffer, tU32 u32Length) const
{
#if (defined DL_ADR3_GEN2_CORE) || (defined __SW_UPDATE_UNIT_TESTING__)
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(au8RamBuffer);
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32Length);
	return FALSE;
#else
    const tChar* regKeyName  = "ADR_SW_Version";
    const tChar* regPathName = "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/VERSIONS";

    reg_tclRegKey oReg;
    if ( oReg.bOpen(regPathName) )
    {
        // Reading Data from Registry
        if (oReg.bQueryString(regKeyName, (tChar*) au8RamBuffer, u32Length))
		{
            printf("%s read from Registry\n",regKeyName);
        }
		else
		{
            printf("%s could not be read from Registry\n",regKeyName);
			return FALSE;
		}
        oReg.vClose();
		return TRUE;
    }
    else
    {
        printf("%s could not be opened\n",regPathName);
		return FALSE;
    }
#endif
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bWriteVersionToFile(tPU8 versionName, tCString versionFileName)
* 
* DESCRIPTION:  Write the Version to a specific file
*
* PARAMETER:	versionName, versionFileName
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bWriteVersionToFile(tPU8 versionName, tCString versionFileName) const
{
	tCString method = "dl_tclDABPlusMain::bWriteVersionToFile";
	ETG_TRACE_USR3 (("%s was entered", method));

	tBool bRetVal = FALSE;
	FILE * fdOutFile = NULL;

	tU32 u32Length =static_cast<tU32> (OSAL_u32StringLength(versionName) );
	if (u32Length >= MAX_DAB_DATABYTES_TOBEREAD)
	{
		ETG_TRACE_ERR (("bWriteVersionToFile - Incorrect version-string"));
		printf("bWriteVersionToFile - Incorrect version-string");
	}
	else
	{
		if ((fdOutFile = fopen (versionFileName, "w")) == NULL)
		{
			ETG_TRACE_FATAL (("File %s could not be opened", versionFileName));
			NORMAL_M_ASSERT_ALWAYS ();
			printf ("File %s could not be opened", versionFileName);
		}
		else
		{
			tS32 s32BytesOfBlockWritten = 0;
			s32BytesOfBlockWritten =static_cast<tU32> (fwrite((tPCS8)versionName, 1, u32Length, fdOutFile) );
			if(s32BytesOfBlockWritten != u32Length)
			{
				ETG_TRACE_ERR (("bWriteVersionToFile - Cannot Write Block"));
				printf("bWriteVersionToFile - Cannot Write Block");
			}
			else
			{
				bRetVal = TRUE;
			}
			fclose(fdOutFile);
		}
	}

	ETG_TRACE_USR3 (("%s was left", method));
	return bRetVal;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::bRunADR3Thread()
*
* DESCRIPTION:  update ADR (calls bRunSequence from a different thread!)
*
* PARAMETER:	xloaderFileName, adr3FileName
*
* RETURN VALUE: tBool
*
******************************************************************************************/
tBool dl_tclDABPlusMain::bRunADR3Thread(tString xloaderFileName, tString adr3FileName) const
{
	tBool returnValue = FALSE;
	ETG_TRACE_FATAL(("NOT YET IMPLEMENTED: STILL ONLY BLOCKING MODE AVAILABLE"));
	dl_tclHelp::vTrace("NOT YET IMPLEMENTED: STILL ONLY BLOCKING MODE AVAILABLE");

	returnValue = getInstanceOfdl_tclDABPlusMain()->bRunSequence(xloaderFileName, adr3FileName);
	releaseInstanceOfdl_tclDABPlusMain();
	return returnValue;
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::vBreakADR3Thread()
*
* DESCRIPTION:  break ADR3 thread
*
* PARAMETER:	Void
*
* RETURN VALUE: Void
*
******************************************************************************************/
tVoid dl_tclDABPlusMain::vBreakADR3Thread()
{
	ETG_TRACE_USR4(("vBreakADR3Thread entered\n"));

	if(m_poProcessCmd == OSAL_NULL)
	{
	    NORMAL_M_ASSERT_ALWAYS();
		return;
	}

	m_poProcessCmd->vBreakADR3Download();

	ETG_TRACE_USR4(("vBreakADR3Thread left\n"));
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::vADR3Thread()
*
* DESCRIPTION:  Thread from where bRunSequence is called to start ADR3 download
*
* PARAMETER:	pArg - contains the pointer to the structure with required info
*
* RETURN VALUE: tVoid
*
******************************************************************************************/
tVoid dl_tclDABPlusMain::vADR3Thread (tPVoid pArg)
{
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	tBool returnValue = FALSE;

	ETG_TRACE_USR4(("vADR3Thread entered\n"));

	ETG_TRACE_FATAL(("myXloaderFileName: %s\n", getInstanceOfdl_tclDABPlusMain()->myXloaderFileName));
	ETG_TRACE_FATAL(("myAdr3FileName: %s\n", getInstanceOfdl_tclDABPlusMain()->myAdr3FileName));

	returnValue = getInstanceOfdl_tclDABPlusMain()->bRunSequence((tString)getInstanceOfdl_tclDABPlusMain()->myXloaderFileName, (tString)getInstanceOfdl_tclDABPlusMain()->myAdr3FileName);
	releaseInstanceOfdl_tclDABPlusMain();

	ETG_TRACE_USR4(("vADR3Thread left\n"));
}

/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::vResetADR3()
*
* DESCRIPTION:  Reset the ADR3
*
* PARAMETER:	Void
*
* RETURN VALUE: Void
*
******************************************************************************************/
tVoid dl_tclDABPlusMain::vResetADR3 ()
{
	ETG_TRACE_FATAL(("dl_tclDABPlusMain::vResetADR3"));
	dl_tclHelp::vTrace("dl_tclDABPlusMain::vResetADR3");
	if(m_poProcessCmd == OSAL_NULL)
	{
	    NORMAL_M_ASSERT_ALWAYS();
		return;
	}
	m_poProcessCmd->vResetADR3();

}  // end of vResetADR3()


/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::u32GetStatusADR3()
*
* DESCRIPTION:  return the ADR3 status
*
* PARAMETER:	Void
*
* RETURN VALUE: tU32
*
******************************************************************************************/
tU32 dl_tclDABPlusMain::u32GetStatusADR3 ()
{
	tU32 returnCode = 0;
	ETG_TRACE_USR4(("dl_tclDABPlusMain::u32GetStatusADR3 entered"));

	if (mpInstance == OSAL_NULL)
	{
		returnCode = dldab_tclProcessCmd::u8GetLastError();
	}
	else
	{
		ETG_TRACE_FATAL(("Cound not get the ADR3 status - still running"));
		dl_tclHelp::vTrace("Cound not get the ADR3 status - still running");
	}

	ETG_TRACE_USR4(("dl_tclDABPlusMain::u32GetStatusADR3 left"));

	return returnCode;
}  // end of u32GetStatusADR3()


ETG_I_CMD_DEFINE((vBreakADR3Dnl, "vBreakADR3Dnl"))
/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::vBreakADR3Dnl()
*
* DESCRIPTION:  break the running ADR3 download
*
* PARAMETER:	Void
*
* RETURN VALUE: Void
*
******************************************************************************************/
tVoid dl_tclDABPlusMain::vBreakADR3Dnl () // TTFIS ETG_I
{
	ETG_TRACE_USR4(("dl_tclDABPlusMain::vBreakADR3Dnl entered"));

	if (mpInstance == OSAL_NULL)
	{
		ETG_TRACE_FATAL(("No ADR3 Download Thread running"));
		dl_tclHelp::vTrace("Unsuccessful trial to break download via ETG_I");
	}
	else
	{
		ETG_TRACE_FATAL(("Break ADR3 Download Thread"));
		dl_tclHelp::vTrace("Try to break ADR3 Download Thread");
		getInstanceOfdl_tclDABPlusMain()->vBreakADR3Thread();
	}

	ETG_TRACE_USR4(("dl_tclDABPlusMain::vBreakADR3Dnl left"));

}  // end of vBreakADR3Dnl()

ETG_I_CMD_DEFINE((vGetStatusADR3Dnl, "vGetStatusADR3Dnl"))
/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::vGetStatusADR3Dnl()
*
* DESCRIPTION:  get the ADR3 status
*
* PARAMETER:	Void
*
* RETURN VALUE: Void
*
******************************************************************************************/
tVoid dl_tclDABPlusMain::vGetStatusADR3Dnl () // TTFIS ETG_I
{
	ETG_TRACE_USR4(("dl_tclDABPlusMain::vGetStatusADR3Dnl entered"));
	tU32 returnData = 0;

	returnData = dl_tclDABPlusMain::u32GetStatusADR3();
	ETG_TRACE_FATAL(("ADR3 Status = 0x%x", returnData));
	dl_tclHelp::vTrace("Try to get status via ETG_I");

	ETG_TRACE_USR4(("dl_tclDABPlusMain::vGetStatusADR3Dnl left"));
}  // end of vGetStatusADR3Dnl()

ETG_I_CMD_DEFINE((vResetADR3Dnl, "vResetADR3Dnl"))
/******************************************************************************************
*
* FUNCTION:		dl_tclDABPlusMain::vResetADR3Dnl()
*
* DESCRIPTION:  Resets the ADR3
*
* PARAMETER:	Void
*
* RETURN VALUE: Void
*
******************************************************************************************/
tVoid dl_tclDABPlusMain::vResetADR3Dnl () // TTFIS ETG_I
{
	ETG_TRACE_USR4(("dl_tclDABPlusMain::vResetADR3Dnl entered"));

	if (mpInstance == OSAL_NULL)
	{
		ETG_TRACE_FATAL(("Reset ADR3 not possible: No ADR3 Download Thread running"));
		dl_tclHelp::vTrace("Reset ADR3 via ETG_I not possible: No ADR3 Download Thread running");
	}
	else
	{
		ETG_TRACE_FATAL(("Reset the ADR3"));
		dl_tclHelp::vTrace("Reset the ADR3 via ETG_I");
		getInstanceOfdl_tclDABPlusMain()->vResetADR3();
	}

	ETG_TRACE_USR4(("dl_tclDABPlusMain::vResetADR3Dnl left"));

}  // end of vResetADR3Dnl()



//EOF
