/**********************************************************************************************
* FILE:           dldab_DnlFsmActions.cpp
* PROJECT:        Nissan
* SW-COMPONENT:   Download
*----------------------------------------------------------------------------------------------
* DESCRIPTION:  
*    Declares all the actions to be executed in State Machine
*----------------------------------------------------------------------------------------------
* COPYRIGHT: (c) 2011 Robert Bosch Car Multimedia GmbH, Hildesheim
* HISTORY:
* Date       |Author								  |Modification
* 23.02.11   |Ghatikar Nageetha (CM-AI/PJ-VW36 RBEI)  |Initial
**********************************************************************************************/

//--------------------------------------------------------------------------------------------
// includes
//--------------------------------------------------------------------------------------------
#include "dldab_DnlFsmActions.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
/* Needed for Trace */
#define ETG_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etg_if.h"
#endif

/* Needed for ETG trace */
#ifdef DL_ADR3_GEN2_CORE
#include "../dl_Adr3TraceMacros.h"
#else
#include "ai_sw_update/common/base/imp/swupd_trace.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL TR_TTFIS_SWUPDATE_CTRL
#define ETG_I_TTFIS_CMD_PREFIX "DNL_"
#define ETG_I_FILE_PREFIX dldab_DnlFsmActions::

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_ADR3
#include "trcGenProj/Header/dldab_DnlFsmActions.cpp.trc.h"
#endif

#include "../dl_tclHelp.h"

/*******************************************************************************************/

#define MAX_NO_OF_RECEIVED_ERRORS   3
#define MAX_NO_OF_TIMEOUTS          100
#define MAX_NO_OF_XLOADER_TIMEOUTS  10

/*******************************************************************************************/

#ifndef __SW_UPDATE_UNIT_TESTING__
// Implementation of the methods getInstanceOfdldab_DnlFsmActions and releaseInstanceOfdldab_DnlFsmActions
// In case of __SW_UPDATE_UNIT_TESTING__ they are implemented in mock!
SWU_IMPL_GET_AND_RELEASE_INSTANCE(dldab_DnlFsmActions);
#endif

// Implementation of the singleton methods
SWU_IMPL_SINGLETON(dldab_DnlFsmActions);

/*******************************************************************************************/

//--------------------------------------------------------------------------------------------
// static initialization
//--------------------------------------------------------------------------------------------
bool	dldab_DnlFsmActions::bInsertOneWrongByte = FALSE; // TTFIS ETG_I Used for testing failures
bool	dldab_DnlFsmActions::bBreakDownload = FALSE;      // can be set to TRUE before object is created!
bool	dldab_DnlFsmActions::bXloaderError = FALSE;       // TTFIS ETG_I Used for testing xloader error

/**********************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::dldab_DnlFsmActions()
* 
* DESCRIPTION:  Constructor for dldab_DnlFsmActions class
*
* PARAMETER: void
*
* RETURN VALUE: none
*
**********************************************************************************************/
dldab_DnlFsmActions::dldab_DnlFsmActions():m_poSpiADR3(OSAL_NULL),
                                           m_poDnlFsmActions(OSAL_NULL),
										   m_poProcessCmd(OSAL_NULL),
										   bSPIMode(FALSE),
										   bVerificationSucess(FALSE),
										   bWriteRes(FALSE),
										   bSendChecksumCmd(FALSE),
										   bSendDataCmd(FALSE),
										   bValidRspCheck(FALSE),
										   bErrorReceived(FALSE),
										   bxloaderRdRspSuccess(FALSE),
										   s32FileSize(0),
										   pDataBuffer(OSAL_NULL),
										   u16CmdLength(0),
										   u32FrameNumber(0),
										   u32StartAddress(0),
										   u8MessageStatus(0),
										   u8NumberOfRetries(0),
										   u8NumberOfReceivedErrors(0),
										   u32MsgFrameSize(0),
										   u32NoOfBytesRead(0),
										   m_dwCurrentAddress(0x00000000),
										   bxLoaderFrm(FALSE),
										   bNextFrameToBeSent(FALSE),
										   bReadCmdSent(FALSE),
										   bLastFrameSent(FALSE),
										   u8errorCode(0),
										   bxLoaderDnlSuccess(FALSE),
										   bSecEraseSuccess(FALSE),
										   bChecksumCalcDone(FALSE),
										   bReadFlashSuccess(FALSE),
										   bPortOpened(FALSE)
{		
	ETG_TRACE_USR3(("dldab_DnlFsmActions::Constructor"));
	
	ETG_I_REGISTER_FILE();

	m_poDnlFsmActions = this;	
	
	// Create the state machine object
	mpFSM = OSAL_NEW DABPlusDNLFsm::Fsm(this);
	NORMAL_M_ASSERT(mpFSM);
	
	m_poSpiADR3 = (tclSpiADR3*)getInstanceOftclSpiADR3();
	
	if(m_poDnlFsmActions != OSAL_NULL)
		m_poDnlFsmActions->vInit();	//For Initializing the State Machine
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::vInit()
* 
* DESCRIPTION:  Function to initialize the State Machine
*
* PARAMETER:	Void
*
* RETURN VALUE: None
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::vInit ( tVoid )
{
    if ( mpFSM != OSAL_NULL){
       mpFSM->init();      
    }
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::~dldab_DnlFsmActions()
* 
* DESCRIPTION:  Destructor for dldab_DnlFsmActions class
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
dldab_DnlFsmActions::~dldab_DnlFsmActions()
{		
	ETG_TRACE_USR3(("dldab_DnlFsmActions Destructor !!"));
	
	if(mpFSM != OSAL_NULL){
		delete mpFSM;
		mpFSM = OSAL_NULL;	
	}	
	pDataBuffer = OSAL_NULL;
    releaseInstanceOftclSpiADR3();
	m_poSpiADR3 = OSAL_NULL;
	m_poProcessCmd = OSAL_NULL;
	m_poDnlFsmActions = OSAL_NULL;	
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::OpenPort( void* pArg)
* 
* DESCRIPTION:  Opens SPI Port
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::OpenPort (tVoid* pArg )
{		
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);

	if(m_poSpiADR3->bOpen() == TRUE)
	{		
		ETG_TRACE_USR3(("Port Opened Successful !!"));
		bPortOpened = TRUE;
	}	
	else
	{		
		ETG_TRACE_FATAL(("Port Opened Failed !!"));
		dl_tclHelp::vTrace("Port Opened Failed !!");
		bPortOpened = FALSE;
	}			
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::AcceptEvent ( FsmEvent event, tVoid* pArg )
* 
* DESCRIPTION:  Calls acceptEvent from FSM
*
* PARAMETER:    FSM-event, void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::vAcceptEvent ( DABPlusDNLFsm::FsmEvent event, tVoid* pArg )
{
	if(mpFSM != OSAL_NULL)	
		mpFSM->acceptEvent(event, pArg);	
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::ReadAck( void* pArg)
* 
* DESCRIPTION:  Reads the response and stores in the Response Buffer
*
* PARAMETER:	void
*
* RETURN VALUE: None
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::ReadAck (tVoid* pArg )
{
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);		
	
	ETG_TRACE_USR4(("dldab_DnlFsmActions::ReadAck"));

	if(m_poSpiADR3 == OSAL_NULL)
		return;	
		
	if(bPortOpened == TRUE)// Read not needed..after opening port
	{
		ETG_TRACE_USR3(("bPortOpened == TRUE"));
		bPortOpened = FALSE;	
		if(mpFSM != OSAL_NULL)	
			mpFSM->acceptEvent(DABPlusDNLFsm::evVerify,0);	
			
		//After Loading Xloader Read is skipped.. 
		//Currently Status 0x42 is not supported
		//Making bReadCmdSent TRUE skips the read		
		bReadCmdSent = TRUE;
	}
	else//Reads response of other cmds
	{
		ETG_TRACE_USR4(("bPortOpened != TRUE"));
		if(bReadCmdSent!=TRUE)
		{						
			ETG_TRACE_USR4(("bReadCmdSent!=TRUE"));
			//Timeout to Exit in case of No Response
			m_poSpiADR3->bSetReadTimeout();
			
			//Size of the response buffer
			tU32 u32SizeofRspBuf = DAB_READ_RESPONSE_BUFFER_SIZE;
			tS32 s32NoOfbytsRead = 0;//No of Bytes Read

			tU8 u8Cnt=0; 			
			tS8 s8ReadBuff[DAB_READ_RESPONSE_BUFFER_SIZE];
			 	
			s32NoOfbytsRead = m_poSpiADR3->s32Read(
										&s8ReadBuff[u8Cnt],
										u32SizeofRspBuf); 			

			ETG_TRACE_USR4(("No Of Bytes Read: %d",s32NoOfbytsRead));	
			if((s32NoOfbytsRead != OSAL_ERROR) && (s32NoOfbytsRead > 0))
			{
				//Will copy all the bytes read into u8ResponseBuffer 
				//which is used for validation later
				memcpy(u8ResponseBuffer,s8ReadBuff, s32NoOfbytsRead );	
				ETG_TRACE_USR4(("Response Received: 0x%x 0x%x",u8ResponseBuffer, s32NoOfbytsRead));
			}			
			else
			{
				//Will clear the u8ResponseBuffer
				//which is used for validation later (in case of timeout!)
				memset(u8ResponseBuffer, 0x00, DAB_READ_RESPONSE_BUFFER_SIZE );	
				ETG_TRACE_USR4(("No Response Received -> Clear Buffer"));
			}			

		}
		bReadCmdSent = FALSE;//To Ensure read is called next time.
		bWriteRes = FALSE;
	}
	
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::VerifyData( void* pArg)
* 
* DESCRIPTION:  Verifies the binary data to be flashed
*
* PARAMETER:	void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::VerifyData (tVoid* pArg )
{	
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	
	if(mpFSM == OSAL_NULL)
		return;
		
	if(bSPIMode==TRUE)
	{
		bSPIMode = FALSE;
		mpFSM->acceptEvent(DABPlusDNLFsm::evVerified,0);
	}	
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::CreateMsgFrame( void* pArg)
* 
* DESCRIPTION:  Creates Message commands
*
* PARAMETER:	void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::CreateMsgFrame (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::CreateMsgFrame"));
	
	if((mpFSM == OSAL_NULL) || (m_poProcessCmd == OSAL_NULL))
		return;
	
	if(bVerificationSucess)//Writing xloader directly no need to send as Frame
	{
		bVerificationSucess = FALSE;
		s32FileSize = m_poProcessCmd->s32FileSize;
		pDataBuffer = m_poProcessCmd->pDataBuffer;
		
		ETG_TRACE_USR3(("Xloader Buffer Size: 0x%x",s32FileSize));	
		ETG_TRACE_USR3(("Xloader Buffer Pointer: 0x%x",pDataBuffer));	

		if((s32FileSize>0) && (pDataBuffer != OSAL_NULL))
		{
			bxLoaderFrm = TRUE;			
			mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);
		}
	}
	if(bxloaderRdRspSuccess)// Now sending Sector Erase Cmd		
	{		
		bxloaderRdRspSuccess = FALSE;
		u8MessageStatus = DAB_SEC_ERASE_CMD;//Erase Command Status	0x29	
		if(bSendFBUSPacket(u8MessageStatus))
		{			
			mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);
		}
	}
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::CreateReadCmd()
* 
* DESCRIPTION:  Creates Read Command
*
* PARAMETER:	None
*
* RETURN VALUE: None
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::CreateReadCmd(tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::CreateReadCmd call"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	
	if(mpFSM == OSAL_NULL)
		return;
					
	if(bxloaderRdRspSuccess == FALSE)
	{
		u8MessageStatus = DAB_READ_CMD;//Read Command Status
		u16CmdLength = READ_CMD_BUFFER_SIZE;
		// Update the buffer with Read command and Call OSAL_s32IOWrite func
		// 0x00, 0x0D, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x6E
		// Byte1   Length Hi	
		// Byte2   Length Lo
		// Byte3   Message - 0x48
		// Byte4.. Byte 7	Start Address 0x00000000
		// Byte8.. Byte 11	Number of bytes to be read 0x00000A (Size of XLOADER)
		// Byte12..Checksum	Length_Hi + Length_Lo + Status + Data1 + ... + Data m)%256				
		
		tPU8 pCmdBuffer = OSAL_NEW tU8 [u16CmdLength];	//Used to create Read command
		if(OSAL_NULL == pCmdBuffer)			return;
		
		pCmdBuffer[0] = 0x00; // Message LENGHT HI
		pCmdBuffer[1] = 0x0c; // Message LENGHT LO
		pCmdBuffer[2] = DAB_READ_CMD; // Message ID(SWITCH_MODE)
		pCmdBuffer[3] = 0x00; // START ADDRESS
		pCmdBuffer[4] = 0x00; // "
		pCmdBuffer[5] = 0x00; // "
		pCmdBuffer[6] = 0x00; // "
		pCmdBuffer[7] = 0x00; // No of Bytes to be read(Size of XLOADER)
		pCmdBuffer[8] = 0x00; // Later if we want to read completed xloader
		pCmdBuffer[9] = 0x00; // "
		pCmdBuffer[10] = 0x10;// "

		tU32 dwCheckSum = 0;
		dwCheckSum = u8CheckSumModCalc(&pCmdBuffer[0],static_cast<tU16>(u16CmdLength-1)) ;
	
		pCmdBuffer[u16CmdLength - 1] = (tU8)dwCheckSum; // CHECKSUM %256

		memcpy(s8ReadCmdBuffer,pCmdBuffer,u16CmdLength);
		delete [] pCmdBuffer;
		
        ETG_TRACE_USR3(("Read Command Sent: %02x ", ETG_LIST_LEN(READ_CMD_BUFFER_SIZE), ETG_LIST_PTR_T8(s8ReadCmdBuffer)));
		mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);
	}
	else
	{	//Event to Trigger Erase Command	
		mpFSM->acceptEvent(DABPlusDNLFsm::evErase,0);	
	}
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::ErrorReport( void* pArg)
* 
* DESCRIPTION:  Reports Error to the upper layers
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::ErrorReport (tVoid* pArg )
{
	if((m_poProcessCmd == OSAL_NULL) || (m_poSpiADR3 == OSAL_NULL))
		return;
		
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	
	ETG_TRACE_USR3(("dldab_DnlFsmActions::ErrorReport call"));
	
	if(m_poSpiADR3->bClose()==TRUE)
	{
		ETG_TRACE_USR3(("Port is Closed Successfully !!"));
	}
	
	m_poProcessCmd->SetErrorInfo();

}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::SendMsg( void* pArg)
* 
* DESCRIPTION:  Sends Coomands to the ADR3 Chip
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::SendMsg (tVoid* pArg )
{
	tS32 BytesWritten = 0; 	
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	
	ETG_TRACE_USR4(("dldab_DnlFsmActions::SendMsg"));

	if((m_poSpiADR3 == OSAL_NULL) ||(mpFSM == OSAL_NULL) || (m_poProcessCmd == OSAL_NULL))
		return;
		
	if(bxLoaderFrm)
	{	//Write the xloader completely	
		bxLoaderFrm = FALSE; // xLoader is Written
		BytesWritten = m_poSpiADR3->s32Write((tPS8)pDataBuffer, (m_poProcessCmd->s32FileSize));	
		ETG_TRACE_USR3(("BytesWritten=0x%x  m_poProcessCmd->s32FileSize=0x%x", BytesWritten, m_poProcessCmd->s32FileSize));
		// Workaround for the problem that the XLOADER needs some time before he is able to receive the 1st READ message!
		ETG_TRACE_USR3(("XXXXXXXXXXXXXXXXXX Wait 100ms XXXXXXXXXXXXXXXXXXXXXX"));
		OSAL_s32ThreadWait(100);
        if(BytesWritten==(m_poProcessCmd->s32FileSize))
		{			
		  bWriteRes = TRUE;		
		  mpFSM->acceptEvent(DABPlusDNLFsm::evRead,0);
	    }
		else
		{
			ETG_TRACE_FATAL(("XLOADER: BytesWritten!=s32FileSize"));
			dl_tclHelp::vTrace("XLOADER: BytesWritten!=s32FileSize");
			mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
	    }
	}	
	else//for Other Commands like Erase, Checksum etc.,
	{
		ETG_TRACE_USR4(("u8MessageStatus=0x%x", u8MessageStatus));
		if(u8MessageStatus == DAB_READ_CMD)	
			BytesWritten = m_poSpiADR3->s32Write(s8ReadCmdBuffer, u16CmdLength);								
		if(u8MessageStatus == DAB_SEC_ERASE_CMD)
			BytesWritten = m_poSpiADR3->s32Write(s8EraseCmdBuffer, u16CmdLength);						
		if(u8MessageStatus == DAB_WRITE_CMD)
			BytesWritten = m_poSpiADR3->s32Write(s8DataCmdBuffer, u16CmdLength);	
		if(u8MessageStatus == DAB_CRC_CALC_CMD)	
			BytesWritten = m_poSpiADR3->s32Write(s8ChecksumCmdBuffer, u16CmdLength);	

        if(BytesWritten==u16CmdLength)
		{			
			bWriteRes = TRUE;		
			mpFSM->acceptEvent(DABPlusDNLFsm::evRead,0);
	    }
		else
		{
			ETG_TRACE_FATAL(("BytesWritten!=u16CmdLength"));
			dl_tclHelp::vTrace("BytesWritten!=u16CmdLength");
			mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
		}
	}
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::VerifyRsp( void* pArg)
* 
* DESCRIPTION:   Verifies response
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::VerifyRsp (tVoid* pArg )
{
/*	Byte 1		Byte 2		Byte 3	Byte 4	Byte 5	Byte n-1	Byte n
	Length hi	Length lo	Data 1	Data 2	  ...	Data m-1	Data m
						   (Status)							   (Checksum)
							
	*** Length(hi+lo) = m = n - 2 = NumberOfReceivedData - 2	***
	***				content of RxBuffer							***
	***	NumberOfReceivedData = n = m + 2 = Length(hi+lo) + 2	***
*//*	Status 	Description
value
0x42	Xloader  is ready to use (will be send after Xloader image is transferred) 
0x10	A new command is received and will be handled 
0x20	Sector erase is finished with fault
0x21	Sector erase is finished without fault
0x30	Writing data to flash is finished with fault
0x31	Writing data to flash is finished without fault 
0x40	Reading data is not possible
0x41	Reading data was fine. The data bytes include the read back bytes from flash
0x50	CRC calculation not possible
0x51	The CRC calculation was fine and 
		the CRC value is stored in the followed 4 data bytes (starting with the highest byte)*/
	
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	

	ETG_TRACE_USR4(("dldab_DnlFsmActions::VerifyRsp"));

	if((m_poProcessCmd == OSAL_NULL) || (mpFSM == OSAL_NULL))
		return;	
		
	bValidRspCheck 	= FALSE;

	//Validation for XLoader 
	if(bxLoaderDnlSuccess == FALSE)
	{
		bxLoaderDnlSuccess = TRUE;
		bValidRspCheck 	= TRUE;
		ETG_TRACE_USR3(("Xloader Check: Response received"));
	}

	// This is used to break the ADR3 download
    if (bBreakDownload == TRUE)
	{
		ETG_TRACE_FATAL(("Break the download!"));
		dl_tclHelp::vTrace("Break the download!");
		u8errorCode = DAB_ABORTED_BY_BREAK;
		u8NumberOfRetries = 0;
		bBreakDownload = FALSE;
		mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
		return;
	}

	// This is used to simulate XLOADER error (no response for READ)
    if (bXloaderError == TRUE)
	{
        if ((u8MessageStatus == DAB_READ_CMD) && (u8ResponseBuffer[2] == DAB_READ_SUCCESS))
        {
		    ETG_TRACE_FATAL(("Simulate XLOADER error (no response for READ)!"));
		    dl_tclHelp::vTrace("Simulate XLOADER error (no response for READ)!");
            u8ResponseBuffer[2] = 0;
            bXloaderError = FALSE;
        }
	}

	//Validation for Read Response Cmd
	if(u8MessageStatus == DAB_READ_CMD)	
	{
		if(u8ResponseBuffer[2] == DAB_READ_SUCCESS)
		{
			ETG_TRACE_USR3(("VerifyRsp DAB_READ_SUCCESS"));
			bValidRspCheck 	= TRUE;
			tU32 BytesRead = ((tU32)(u8ResponseBuffer[0]))*0x100 + ((tU32)(u8ResponseBuffer[1])) - 4;
			if (bxloaderRdRspSuccess == FALSE)
			{
				bxloaderRdRspSuccess = TRUE;
			}
			else
			{
				// Set bNextFrameToBeSent here because this is not the response to XLOADER-Test. Instead it is the response for READ_FLASH
				bNextFrameToBeSent = TRUE;
				// Copy back to read buffer (don't do this for WRITE, because it destroys the WRITE buffer)
				memcpy((m_poProcessCmd->pDataBuffer) + m_dwCurrentAddress, &u8ResponseBuffer[3] , BytesRead);
			}
		}
		else if(u8ResponseBuffer[2] == DAB_READ_CMD_ERROR)	{
			ETG_TRACE_USR3(("VerifyRsp DAB_READ_CMD_ERROR"));
			u8errorCode = u8ResponseBuffer[2];
		    bErrorReceived = TRUE;
			mpFSM->acceptEvent(DABPlusDNLFsm::evError,0);
			return;
		}
		ETG_TRACE_USR3(("VerifyRsp DAB_READ_CMD"));
	}

	//Validation for Erase Cmd
	if(u8MessageStatus == DAB_SEC_ERASE_CMD)
	{
		if(u8ResponseBuffer[2] == DAB_ERASE_SUCCESS)
		{
			ETG_TRACE_USR3(("VerifyRsp DAB_ERASE_SUCCESS"));
			bSecEraseSuccess = TRUE;			
			bValidRspCheck 	= TRUE;
		}
		else if(u8ResponseBuffer[2] == DAB_ERASE_ERROR)	{
			ETG_TRACE_USR3(("VerifyRsp DAB_ERASE_ERROR"));
			u8errorCode = u8ResponseBuffer[2];
		    bErrorReceived = TRUE;
			mpFSM->acceptEvent(DABPlusDNLFsm::evError,0);
			return;
		}
		ETG_TRACE_USR3(("VerifyRsp DAB_SEC_ERASE_CMD"));
	}

	//Validation for Write Frame Cmd	
	if(u8MessageStatus == DAB_WRITE_CMD)	
	{
		if(u8ResponseBuffer[2] == DAB_WRITE_SUCCESS)
		{
			ETG_TRACE_USR4(("VerifyRsp DAB_WRITE_SUCCESS"));
			bNextFrameToBeSent = TRUE;	
			bValidRspCheck 	= TRUE;
		}
		else if(u8ResponseBuffer[2] == DAB_WRITE_DATA_ERROR)	{
			ETG_TRACE_USR3(("VerifyRsp DAB_WRITE_DATA_ERROR"));
			u8errorCode = u8ResponseBuffer[2];
		    bErrorReceived = TRUE;
			mpFSM->acceptEvent(DABPlusDNLFsm::evError,0);
			return;
		}		
		ETG_TRACE_USR4(("VerifyRsp DAB_WRITE_CMD"));
	}

	//Validation for CRC Calculation Cmd
	if(u8MessageStatus == DAB_CRC_CALC_CMD)	
	{
		if(u8ResponseBuffer[2] == DAB_CRC_CALC_SUCCESS)
		{
			// Compare the File checksum calculated and received 
			tU32 m_dwCheckSum = ((tU32)(u8ResponseBuffer[3]))*0x1000000 + 
								((tU32)(u8ResponseBuffer[4]))*0x10000 +
								((tU32)(u8ResponseBuffer[5]))*0x100 + 
								((tU32)(u8ResponseBuffer[6]));
								
			ETG_TRACE_USR3(("CheckSum Received: 0x%2x",m_dwCheckSum));
			ETG_TRACE_USR3(("CheckSum Expected: 0x%2x",m_poProcessCmd->u32FileChecksum));
			char buf[256];
			snprintf (buf, 255, "CheckSum Received: 0x%2x",(unsigned int)m_dwCheckSum);
			dl_tclHelp::vTrace(buf);
			snprintf (buf, 255, "CheckSum Expected: 0x%2x",(unsigned int)m_poProcessCmd->u32FileChecksum);
			dl_tclHelp::vTrace(buf);

			if(m_poProcessCmd->u32FileChecksum == m_dwCheckSum){
				bChecksumCalcDone = TRUE;	
				bValidRspCheck 	= TRUE;	
			}
			else
			{
				// Wrong CRC received - no retry required, just send evFailure instead of evError
				ETG_TRACE_USR3(("VerifyRsp DAB_CRC_CALC_ERROR"));
				u8errorCode = DAB_CRC_CALC_ERROR;
				u8NumberOfRetries = 0;
				mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
				return;
			}
		}
		else if(u8ResponseBuffer[2] == DAB_CRC_CALC_NOT_SUPORTED)	{
			// No retry required, just send evFailure instead of evError
			u8errorCode = u8ResponseBuffer[2];
			u8NumberOfRetries = 0;
			mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
			return;
		}
		ETG_TRACE_USR3(("VerifyRsp DAB_CRC_CALC_CMD"));
	}

	/* Still if the response is not received then the Failure event will be triggered */
	if(bValidRspCheck==TRUE)
	{
		u8NumberOfRetries = 0;
		u8NumberOfReceivedErrors = 0;
		memset(u8ResponseBuffer, 0x00, DAB_READ_RESPONSE_BUFFER_SIZE );	
		mpFSM->acceptEvent(DABPlusDNLFsm::evSuccess,0);
	} 
	else
	{
		if((u8MessageStatus == DAB_READ_CMD) && (bxloaderRdRspSuccess == FALSE))
		{
            ETG_TRACE_USR3(("evSuccess Not Triggered -> retry (DAB_READ_CMD)"));
		    // retry 10 times (wait for response max. 10s)
		    if(u8NumberOfRetries < MAX_NO_OF_XLOADER_TIMEOUTS)
		    {
			    u8NumberOfRetries++;
			    bErrorReceived = TRUE;
			    mpFSM->acceptEvent(DABPlusDNLFsm::evRead,0);
		    }
		    else
		    {
                ETG_TRACE_FATAL(("RESET and Retry download of XLOADER"));
				dl_tclHelp::vTrace("RESET and Retry download of XLOADER");
			    u8NumberOfRetries = 0;
                u8errorCode = DAB_NO_XLOADER_RESPONSE;
			    mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
			}
		}
		else
		{
            ETG_TRACE_USR4(("evSuccess Not Triggered -> retry"));
		    if(u8NumberOfRetries < MAX_NO_OF_TIMEOUTS)
		    {
			    u8NumberOfRetries++;
			    bErrorReceived = TRUE;
			    mpFSM->acceptEvent(DABPlusDNLFsm::evRead,0);
		    }
		    else
		    {
			    ETG_TRACE_USR4(("evFailure"));
			    u8NumberOfRetries = 0;
                u8errorCode = DAB_RESPONSE_TIMEOUT;
			    mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
			}
		}
	}
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::HandleError( void* pArg)
* 
* DESCRIPTION:   Handles error when invalid response obtained
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::HandleError (tVoid* pArg )
{
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	
	ETG_TRACE_USR3(("dldab_DnlFsmActions::HandleError()"));
	if(mpFSM == OSAL_NULL)
		return;
	bErrorReceived = FALSE;
	u8NumberOfReceivedErrors++;
	if (u8NumberOfReceivedErrors <= MAX_NO_OF_RECEIVED_ERRORS)
	{
		mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);//Resending the command
	}
	else
	{
		mpFSM->acceptEvent(DABPlusDNLFsm::evFailure,0);
	}

}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::CreateDataMsgFrames( void* pArg)
* 
* DESCRIPTION:  Creates Data Message Frames each 512 bytes
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::CreateDataMsgFrames (tVoid* pArg )
{
	if((mpFSM == OSAL_NULL) ||(m_poProcessCmd == OSAL_NULL))
		return;
	
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	
	
	ETG_TRACE_USR4(("Creating Data Msg Frames !!!"));
	
	
	if(bLastFrameSent == FALSE)
	{
	bSecEraseSuccess = FALSE;
	//Trigger will always come to this function after successfully sending the data to the 
	//RAM . This should be repeated till the data to be transferred is completely sent

	//1. Load to the buffer 
	//2. Take a buffer and write the Header (Length + Message Status)
	//3. Upload 512 bytes to the same buffer
	//4. Add the Footer - Checksum
	//5. Trigger the evSend Event... This will write u32DataCmdBuffer buffer to ADR3 Chip

	//0x38	Write Flash. The first 4 byte include the Flash start address to write to 
	//the next 4 bytes (tU32) will include the number of bytes to be programmed and 
	//than all data bytes to be programmed will follow.
		
		tU8 DataRead[MAX_DAB_DATABYTES_TOWRITE] = {0};//MAX_DAB_DATABYTES_TOWRITE - 512	
		
		tS32 ActualBytesRead = MAX_DAB_DATABYTES_TOWRITE;
		if ((tS32)u32NoOfBytesRead + ActualBytesRead >= m_poProcessCmd->s32FileSize)
		{
			ActualBytesRead = m_poProcessCmd->s32FileSize - u32NoOfBytesRead;
		}
		memcpy(DataRead, (m_poProcessCmd->pDataBuffer) + u32NoOfBytesRead, (tU32)ActualBytesRead);

		//lenght - 2 , Messge Status - 1, Checksum - 1, Address - 4 , No of Bytes - 4			
		tU16 length =static_cast<tU16> (12 + ActualBytesRead);
		
		//MAX_DAB_DATAFRAME_CMD_SIZE - 524 (lenght + MAX_DAB_DATABYTES_TOWRITE)		
		tU8 outBuff[MAX_DAB_DATAFRAME_CMD_SIZE] = {0};

		tU16 nToWrite = length;	
		u8MessageStatus = DAB_WRITE_CMD;

		if(bNextFrameToBeSent){	
			bNextFrameToBeSent = FALSE;			
			u32StartAddress += ActualBytesRead;			
			u32FrameNumber++;				
		}
		//Copying the Start address for next Data Transfer cmd
		m_dwCurrentAddress = u32StartAddress;	
		u32NoOfBytesRead += ActualBytesRead;	
		
		outBuff[0] = 0x02; //524 0x20C - Length HI
		outBuff[1] = 0x0c;//Length - Lo
		outBuff[2] = u8MessageStatus;// Message Frame Status - 0x38
		outBuff[3] = (tU8)((m_dwCurrentAddress & 0xff000000) >> 24); //Start Address
		outBuff[4] = (tU8)((m_dwCurrentAddress & 0x00ff0000) >> 16); 
		outBuff[5] = (tU8)((m_dwCurrentAddress & 0x0000ff00) >> 8); 
		outBuff[6] = (tU8)(m_dwCurrentAddress & 0x000000ff); 
		outBuff[7] = (tU8)((ActualBytesRead & 0xff000000) >> 24); //No Of Bytes 
		outBuff[8] = (tU8)((ActualBytesRead & 0x00ff0000) >> 16); //to be programmed
		outBuff[9] = (tU8)((ActualBytesRead & 0x0000ff00) >> 8); 
		outBuff[10] = (tU8)(ActualBytesRead & 0x000000ff);
		memcpy(&outBuff[11], &DataRead[0], ActualBytesRead);//Copying data bytes read 
	    
		// This is used to simulate an error via TTFIS ETG_I
	    if (bInsertOneWrongByte == TRUE)
		{
			outBuff[11] =static_cast<tU8> (outBuff[11] + 1);
			bInsertOneWrongByte = FALSE;
			ETG_TRACE_FATAL(("Insert one wrong Byte now!"));
			dl_tclHelp::vTrace("Insert one wrong Byte now! (ETG_I)");
		}

		tU32 dwCheckSum = 0;
		dwCheckSum = u8CheckSumModCalc(&outBuff[0], length);
		outBuff[nToWrite - 1] = (tU8)dwCheckSum;	
		//Copying data bytes to u32DataCmdBuffer Buffer
		memcpy(&s8DataCmdBuffer[0], &outBuff[0], nToWrite);
		
		u32MsgFrameSize = nToWrite;	
		u16CmdLength = nToWrite;	
		mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);
			
		if((ActualBytesRead != MAX_DAB_DATABYTES_TOWRITE)||
				(u32NoOfBytesRead == m_poProcessCmd->s32FileSize))
		{
			ETG_TRACE_USR3(("Current Address: 0x%2x",m_dwCurrentAddress));
			ETG_TRACE_USR3(("Actual Bytes: 0x%2x",ActualBytesRead));
			bLastFrameSent = TRUE;		
		}
	}
	else//(bLastFrameSent)
	{// Once all bytes/ complete data file is sent... Checksum Event needs to be triggered
		bLastFrameSent = FALSE;
		ETG_TRACE_USR3(("evChecksum Trigger"));
		//Event to Trigger the Checksum Command
		mpFSM->acceptEvent(DABPlusDNLFsm::evChecksum,0);
	}
	
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::CreateReadFlashCmd( void* pArg)
* 
* DESCRIPTION:  Creates Read Data Message Frames each 512 bytes
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::CreateReadFlashCmd (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::CreateReadFlashCmd call"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	
	
	if((mpFSM == OSAL_NULL) ||(m_poProcessCmd == OSAL_NULL))
		return;

	// Read-sequence does not require XLOADER-Test, so it can be set to true here
	bxloaderRdRspSuccess = TRUE;

	if(bLastFrameSent == FALSE)
	{
		u16CmdLength = READ_CMD_BUFFER_SIZE;
		u8MessageStatus = DAB_READ_CMD;

		tS32 ActualBytesRead = MAX_DAB_DATABYTES_TOBEREAD;
		if ((tS32)u32NoOfBytesRead + ActualBytesRead >= m_poProcessCmd->s32FileSize)
		{
			ActualBytesRead = m_poProcessCmd->s32FileSize - u32NoOfBytesRead;
		}

		tU8 outBuff[READ_CMD_BUFFER_SIZE] = {0};

		if(bNextFrameToBeSent){	
			bNextFrameToBeSent = FALSE;			
			u32StartAddress += ActualBytesRead;			
			u32FrameNumber++;				
		}
		//Copying the Start address for next Data Transfer cmd
		m_dwCurrentAddress = u32StartAddress;	
		u32NoOfBytesRead += ActualBytesRead;	

		outBuff[0] = 0x00; //Message LENGHT HI
		outBuff[1] = 0x0c; //Message LENGHT LO
		outBuff[2] = u8MessageStatus;// Message ID
		outBuff[3] = (tU8)((m_dwCurrentAddress & 0xff000000) >> 24); //Start Address
		outBuff[4] = (tU8)((m_dwCurrentAddress & 0x00ff0000) >> 16); 
		outBuff[5] = (tU8)((m_dwCurrentAddress & 0x0000ff00) >> 8); 
		outBuff[6] = (tU8)(m_dwCurrentAddress & 0x000000ff); 
		outBuff[7] = (tU8)((ActualBytesRead & 0xff000000) >> 24); //No Of Bytes 
		outBuff[8] = (tU8)((ActualBytesRead & 0x00ff0000) >> 16); //to be read
		outBuff[9] = (tU8)((ActualBytesRead & 0x0000ff00) >> 8); 
		outBuff[10] = (tU8)(ActualBytesRead & 0x000000ff);

		tU32 dwCheckSum = 0;
		dwCheckSum = u8CheckSumModCalc(&outBuff[0],static_cast<tU16> (u16CmdLength-1));
		outBuff[u16CmdLength - 1] = (tU8)dwCheckSum; // CHECKSUM %256

		memcpy(s8ReadCmdBuffer,outBuff,u16CmdLength);

		mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);

		char buf[256];
		snprintf (buf, 255, "Current Address: 0x%2x",(unsigned int)m_dwCurrentAddress);
		dl_tclHelp::vTrace(buf);
		snprintf (buf, 255, "Actual Bytes: 0x%2x",ActualBytesRead);
		dl_tclHelp::vTrace(buf);
		snprintf (buf, 255, "u32NoOfBytesRead: 0x%2x",(unsigned int)u32NoOfBytesRead);
		dl_tclHelp::vTrace(buf);
		snprintf (buf, 255, "m_poProcessCmd->s32FileSize: 0x%2x",m_poProcessCmd->s32FileSize);
		dl_tclHelp::vTrace(buf);
		
		if((ActualBytesRead != MAX_DAB_DATABYTES_TOBEREAD)||
				(u32NoOfBytesRead == m_poProcessCmd->s32FileSize))
		{
			ETG_TRACE_USR3(("Current Address: 0x%2x",m_dwCurrentAddress));
			ETG_TRACE_USR3(("Actual Bytes: 0x%2x",ActualBytesRead));
			bLastFrameSent = TRUE;		
			dl_tclHelp::vTrace("bLastFrameSent");
		}
	}
	else//(bLastFrameSent)
	{// Once all bytes/ complete data is read... evNormalMode needs to be triggered
		bLastFrameSent = FALSE;
		ETG_TRACE_USR3(("evNormalMode Trigger"));
		//Trigger the evNormalMode
		mpFSM->acceptEvent(DABPlusDNLFsm::evNormalMode,0);
		bReadFlashSuccess = TRUE;
	}
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::CreateReadVersionCmd( void* pArg)
* 
* DESCRIPTION:  Creates Read Data Message Frame to read out the version
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::CreateReadVersionCmd (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::CreateReadVersionCmd call"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	

	if((mpFSM == OSAL_NULL) ||(m_poProcessCmd == OSAL_NULL))
		return;

	// Read-sequence does not require XLOADER-Test, so it can be set to true here
	bxloaderRdRspSuccess = TRUE;

	if(bLastFrameSent == FALSE)
	{
		u16CmdLength = READ_CMD_BUFFER_SIZE;
		u8MessageStatus = DAB_READ_CMD;

		tU32 StartAddress = m_poProcessCmd->u32VersionAddr;
		tU32 SizeOfVersion = m_poProcessCmd->u32VersionSize;
		tU8 outBuff[READ_CMD_BUFFER_SIZE] = {0};

		if(bNextFrameToBeSent){	
			bNextFrameToBeSent = FALSE;			
		}

		//Note: m_dwCurrentAddress is used to calculate the address inside the buffer...
		m_dwCurrentAddress = 0;	

		outBuff[0] = 0x00; //Message LENGHT HI
		outBuff[1] = 0x0c; //Message LENGHT LO
		outBuff[2] = u8MessageStatus;// Message ID
		outBuff[3] = (tU8)((StartAddress & 0xff000000) >> 24); //Start Address
		outBuff[4] = (tU8)((StartAddress & 0x00ff0000) >> 16); 
		outBuff[5] = (tU8)((StartAddress & 0x0000ff00) >> 8); 
		outBuff[6] = (tU8)(StartAddress & 0x000000ff); 
		outBuff[7] = (tU8)((SizeOfVersion & 0xff000000) >> 24); //No Of Bytes 
		outBuff[8] = (tU8)((SizeOfVersion & 0x00ff0000) >> 16); //to be read
		outBuff[9] = (tU8)((SizeOfVersion & 0x0000ff00) >> 8); 
		outBuff[10] = (tU8)(SizeOfVersion & 0x000000ff);

		tU32 dwCheckSum = 0;
		dwCheckSum = u8CheckSumModCalc(&outBuff[0],static_cast<tU16> (u16CmdLength-1));
		outBuff[u16CmdLength - 1] = (tU8)dwCheckSum; // CHECKSUM %256

		memcpy(s8ReadCmdBuffer,outBuff,u16CmdLength);

		mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);

		char buf[256];
		snprintf (buf, 255, "StartAddress: 0x%2x",(unsigned int)StartAddress);
		dl_tclHelp::vTrace(buf);
		snprintf (buf, 255, "SizeOfVersion: 0x%2x",(unsigned int)SizeOfVersion);
		dl_tclHelp::vTrace(buf);
		
		bLastFrameSent = TRUE;		
	}
	else//(bLastFrameSent)
	{// Once all bytes/ complete data is read... evNormalMode needs to be triggered
		bLastFrameSent = FALSE;
		ETG_TRACE_USR3(("evNormalMode Trigger"));
		//Trigger the evNormalMode
		mpFSM->acceptEvent(DABPlusDNLFsm::evNormalMode,0);
		bReadFlashSuccess = TRUE;
	}
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::CreateChecksumCmd( void* pArg)
* 
* DESCRIPTION:  Creates Checksum Commands
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::CreateChecksumCmd (tVoid* pArg )
{
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	ETG_TRACE_USR3(("dldab_DnlFsmActions::CreateChecksumCmd()"));
	if((mpFSM == OSAL_NULL) ||(m_poProcessCmd == OSAL_NULL))
				return;
	if((bSendChecksumCmd==TRUE) && (bChecksumCalcDone==FALSE))//Data was Flashed
	{	// Now send the Checksum Cmd
		bSendChecksumCmd = FALSE;		
		u8MessageStatus = DAB_CRC_CALC_CMD;//Checksum Command Status
		
	// Update the buffer with Checksum command and Call OSAL_s32IOWrite func
	// 0x00, 0x0D, 0x58, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x01, 0x53
	// Byte1   Length Hi	
	// Byte2   Length Lo
	// Byte3   Message - 0x58
	// Byte4.. Byte 7	Start Address 0x00000000
	// Byte8.. Byte 11	Number of bytes to be used 0x200000 (2MB.. 2097152 Bytes)
	// Byte12..Mode for Checksum Calc - 0x00 -> TrojanCRC, 0x01 -> CRC_2007.
	// Byte13  Checksum	Length_Hi +Length_Lo + Status + Data1 + ... + Data m) modulo 256
		
		u16CmdLength = CHECKSUM_CMD_BUFFER_SIZE;
		
		tPU8 pCmdBuffer = OSAL_NEW tU8 [u16CmdLength];//Used to create Checksum command
		s32FileSize = m_poProcessCmd->s32FileSize;//Size of the Data File loaded
		
		if(OSAL_NULL != pCmdBuffer)
		{
			pCmdBuffer[0] = 0x00; // Message LENGHT HI
			pCmdBuffer[1] = 0x0d; // Message LENGHT LO
			pCmdBuffer[2] = DAB_CRC_CALC_CMD; //Checksum Command Status
			pCmdBuffer[3] = 0x00; // START ADDRESS
			pCmdBuffer[4] = 0x00; // "
			pCmdBuffer[5] = 0x00; // "
			pCmdBuffer[6] = 0x00; // "					
			pCmdBuffer[7] = (tU8)((s32FileSize & 0xff000000) >> 24); 
			pCmdBuffer[8] = (tU8)((s32FileSize & 0x00ff0000) >> 16); 
			pCmdBuffer[9] = (tU8)((s32FileSize & 0x0000ff00) >> 8); 
			pCmdBuffer[10] = (tU8)(s32FileSize & 0x000000ff); 
			pCmdBuffer[11] = DAB_CRC_2007_MODE; //MODE - CRC_2007 0x01
			
			tU32 dwCheckSum = 0;
			dwCheckSum = u8CheckSumModCalc(&pCmdBuffer[0],static_cast<tU16> (u16CmdLength-1));			
			pCmdBuffer[u16CmdLength - 1] = (tU8)dwCheckSum;	//CHECKSUM %256			
			memcpy(s8ChecksumCmdBuffer,pCmdBuffer,u16CmdLength);
		}
		
		delete [] pCmdBuffer;
		
        ETG_TRACE_USR3(("Checksum Command Sent: %02x ", ETG_LIST_LEN(CHECKSUM_CMD_BUFFER_SIZE), ETG_LIST_PTR_T8(s8ChecksumCmdBuffer)));
		mpFSM->acceptEvent(DABPlusDNLFsm::evSend,0);
	}
	if(bChecksumCalcDone==TRUE)//After Success trigger this func will be invoked again
	{
		ETG_TRACE_USR3(("evNormalMode Trigger"));
		mpFSM->acceptEvent(DABPlusDNLFsm::evNormalMode,0);//This Event will be queued	
	}
}	

/*******************************************************************************
 *FUNCTION:       dldab_DnlFsmActions::vSendFBUSPacket
 *
 *DESCRIPTION:    Function to create the commands sucs as Erase , Checksum etc.,
 *
 *PARAMETER:      u16MessageStatus : Pointer to Wrapper service
 *
 *RETURNVALUE:    tBool
 *
 *HISTORY:
 *
 *******************************************************************************/
tBool dldab_DnlFsmActions::bSendFBUSPacket(tU16 u16MessageStatus)  
{
	if(u16MessageStatus == DAB_SEC_ERASE_CMD)//Erase Command
	  {
	  // Update the buffer with Sector Erase command and Call OSAL_s32IOWrite func
	  // 0x00, 0x0C, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x55
	  // Byte1	Length Hi	
	  // Byte2	Length Lo
	  // Byte3	Message
	  // Byte4 to Byte 7	Start Address 0x00000000
	  // Byte8 to Byte 11	Number of bytes to be erased
	  // Byte12 Checksum	Length_Hi +Length_Lo + Status + Data1 + ... + Data m) modulo 256
        tU8 pCmdBuffer [ERASE_CMD_BUFFER_SIZE] = {0x00,0x0C,DAB_SEC_ERASE_CMD,0x00,0x00,0x00,0x00,0x00,0x20,0x00,0x00,0xFF};

        if(m_poProcessCmd == OSAL_NULL)
		{
			return FALSE;
		}
		s32FileSize = m_poProcessCmd->s32FileSize; //Size of the Data File loaded
		// overwrite the fix 2MB size with actual filesize
		pCmdBuffer[7] = (tU8)((s32FileSize & 0xff000000) >> 24); 
		pCmdBuffer[8] = (tU8)((s32FileSize & 0x00ff0000) >> 16); 
		pCmdBuffer[9] = (tU8)((s32FileSize & 0x0000ff00) >> 8); 
		pCmdBuffer[10] = (tU8)(s32FileSize & 0x000000ff);
		// overwrite checksum
        pCmdBuffer[ERASE_CMD_BUFFER_SIZE-1] = u8CheckSumModCalc(&pCmdBuffer[0], (ERASE_CMD_BUFFER_SIZE-1));
	  
        memcpy(s8EraseCmdBuffer,pCmdBuffer,ERASE_CMD_BUFFER_SIZE);
        ETG_TRACE_USR3(("Erase Command Sent: %02x ", ETG_LIST_LEN(ERASE_CMD_BUFFER_SIZE), ETG_LIST_PTR_T8(s8EraseCmdBuffer)));
	 }
	return TRUE;//(u16CmdLength > 0);
}
  
/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::u8CheckSumModCalc()
* 
* DESCRIPTION:  Function to Calculate checksum %256
*
* PARAMETER:	const tU8  *pcu8Buffer, tU16 u16Len
*
* RETURN VALUE: tU8
*
******************************************************************************************/
tU8 dldab_DnlFsmActions::u8CheckSumModCalc(const tU8  *pcu8Buffer, tU16 u16Len) const
{
	if ( u16Len < 1 )	return 0;

	tU32 u32CheckSum = 0;
	tS32 s32Count;

	for( s32Count = 0; s32Count < u16Len; s32Count++)
	{
		u32CheckSum += pcu8Buffer[ s32Count];
	}

	return (tU8)u32CheckSum % 256;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::NotifyDNLFinished( void* pArg)
* 
* DESCRIPTION:  Notifies Download finished in the normal mode 
*
* PARAMETER:	void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::NotifyDNLFinished (tVoid* pArg )
{
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	
	
	if(m_poSpiADR3 == OSAL_NULL)
		return;
		
	if(m_poSpiADR3->bClose()==TRUE)
	{
		ETG_TRACE_USR3(("Port is Closed Successfully !!"));
	}
	ETG_TRACE_USR3(("Download or Read Finished !!!"));
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::PortOpened( void* pArg)
* 
* DESCRIPTION:  Will be set once SPI Port is opened
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::PortOpened (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::PortOpened()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	return bPortOpened;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::SPIMode( void* pArg)
* 
* DESCRIPTION:  Will be set once SPI Mode is activated
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::SPIMode (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::SPIMode()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	bSPIMode = TRUE;
	return bSPIMode;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::VerificationSucess( void* pArg)
* 
* DESCRIPTION:  Will be set once data file is verified
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::VerificationSucess (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::VerificationSucess()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	bVerificationSucess = TRUE;
	return bVerificationSucess;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::xloaderDnlSuccess( void* pArg)
* 
* DESCRIPTION:   Will be set if xloader was dnl successfully
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::xloaderDnlSuccess (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::xloaderDnlSuccess()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	return bxLoaderDnlSuccess;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::xloaderRspSuccess( void* pArg)
* 
* DESCRIPTION:   Will be set if xloader was dnl successfully
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::xloaderRspSuccess (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::xloaderRspSuccess()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);		
	return bxloaderRdRspSuccess;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::ValidRsp( void* pArg)
* e
* DESCRIPTION:   Will be set valid response is received
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::ValidRsp (tVoid* pArg )
{
	ETG_TRACE_USR4(("dldab_DnlFsmActions::ValidRsp()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	return bValidRspCheck;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::AckReceived( void* pArg)
* 
* DESCRIPTION:   Will be set once response is received for the command set
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::AckReceived (tVoid* pArg )
{
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	return bWriteRes;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::ErrorResponse( void* pArg)
* 
* DESCRIPTION:  Will be set if invalid response is received
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::ErrorResponse (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::ErrorResponse()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);	
	return bErrorReceived;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::EraseSuccess( void* pArg)
* 
* DESCRIPTION:  Will be set if the Erase command was set successfully
*
* PARAMETER: void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::EraseSuccess (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::EraseSuccess()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	bSendDataCmd = TRUE;
	return bSecEraseSuccess;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::DataSent( void* pArg)
* 
* DESCRIPTION:  Will be set after Data is being sent
*
* PARAMETER:	void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::DataSent (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::DataSent()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	bSendChecksumCmd = TRUE;
	return bSendChecksumCmd;
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::DnlFinished( void* pArg)
* 
* DESCRIPTION:  Will be set after the Dnl is finished
*
* PARAMETER:	void
*
* RETURN VALUE: tBool
*
******************************************************************************************/
bool dldab_DnlFsmActions::DnlFinished (tVoid* pArg )
{
	ETG_TRACE_USR3(("dldab_DnlFsmActions::DnlFinished()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
	return bChecksumCalcDone;
}

/*******************************************************************************
 *FUNCTION:       dldab_DnlFsmActions::vStoreReference
 *
 *DESCRIPTION:    Stores reference required for dlbt_tclProcessCmd
 *
 *PARAMETER:      m_poProcessCmd : Pointer to Wrapper service
 *
 *RETURNVALUE:    None
 *
 *HISTORY:
 *
 *******************************************************************************/
tVoid dldab_DnlFsmActions::vStoreReference(dldab_tclProcessCmd*  poProcessCmd)
{
    if ( OSAL_NULL  == poProcessCmd  )
    {
        NORMAL_M_ASSERT(FALSE);
    }
    else
    {       
       m_poProcessCmd = poProcessCmd;
    }
}

/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::vResetADR3()
* 
* DESCRIPTION:  Reset ADR3
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::vResetADR3 ()
{
	if(m_poSpiADR3 == OSAL_NULL)
		return;
	
	if(m_poSpiADR3->bReset() == FALSE)
	{
		ETG_TRACE_FATAL(("dldab_DnlFsmActions::vResetADR3 not possible"));
		dl_tclHelp::vTrace("dldab_DnlFsmActions::vResetADR3 not possible");
	}
}


/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::vBreakADR3Download()
* 
* DESCRIPTION:  Function break a running ADR3 download  
*
* PARAMETER:	tVoid
*
* RETURN VALUE: tVoid
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::vBreakADR3Download() 
{
	ETG_TRACE_FATAL(("dldab_DnlFsmActions::vBreakADR3Download"));
	dl_tclHelp::vTrace("dldab_DnlFsmActions::vBreakADR3Download");
	bBreakDownload = TRUE;
}


ETG_I_CMD_DEFINE((vInsertOneWrongByte, "vInsertOneWrongByte"))
/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::vInsertOneWrongByte()
* 
* DESCRIPTION:  Insert one wrong byte into static data of ADR3 (to simulate error)
*
* PARAMETER:	Void
*
* RETURN VALUE: Void
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::vInsertOneWrongByte () // TTFIS ETG_I
{
	ETG_TRACE_FATAL(("dldab_DnlFsmActions::vInsertOneWrongByte -> error in next CRC calculation"));
	dl_tclHelp::vTrace("dldab_DnlFsmActions::vInsertOneWrongByte -> error in next CRC calculation (ETG_I)");
	bInsertOneWrongByte = TRUE;
}  // end of vInsertOneWrongByte()


ETG_I_CMD_DEFINE((vXloaderError, "vXloaderError"))
/******************************************************************************************
*
* FUNCTION:		dldab_DnlFsmActions::vXloaderError()
* 
* DESCRIPTION:  Simulate xloader error
*
* PARAMETER:	Void
*
* RETURN VALUE: Void
*
******************************************************************************************/
tVoid dldab_DnlFsmActions::vXloaderError () // TTFIS ETG_I
{
	ETG_TRACE_FATAL(("dldab_DnlFsmActions::vXloaderError -> error in next download cycle"));
	dl_tclHelp::vTrace("dldab_DnlFsmActions::vXloaderError -> error in next download cycle (ETG_I)");
	bXloaderError = TRUE;
}  // end of vXloaderError()

//EOF
