#ifndef _SHAREDSINGLETON_H
#define _SHAREDSINGLETON_H

/***********************************************************************/
/*!
* \file  SharedSingleton.h
* \brief Singleton Utility
*************************************************************************
\verbatim

PROJECT:         Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Singleton class implementation using shared pointer
AUTHOR:         rur1kor
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
04.05.2017  | Ramya Murthy          | Initial Version

\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <cstddef>
#include <memory>
#include "Lock.h"

/****************************************************************************/
/*!
* \class SharedSingleton
* \brief Singleton class implementation using shared pointer
****************************************************************************/
template <typename T>
class SharedSingleton
{

public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  std::shared_ptr<T> SharedSingleton::getInstance()
   ***************************************************************************/
   /*!
   * \fn      static std::shared_ptr<T> getInstance()
   * \brief   Method to get the instance of the singleton class
   * \param   None
   * \sa
   ***************************************************************************/
   static std::shared_ptr<T> getInstance()
   {
      if (!m_instance)
      {
         //Two threads can call getInstance() method at the same time.
         //In such cases, to prevent creating object twice, make it thread safe.

         //Additional NULL check was kept above, to avoid entering critical section every time,
         //whenever clients does a getInstance(). With this approach we avoid entering into critical section always
         //and avoid delay in acquiring the object. 
         m_sLock.s16Lock();
         if (!m_instance)
         {
            m_instance = std::shared_ptr<T>(new T);
         }//if (!m_instance)
         m_sLock.vUnlock();

      }//if (!m_instance)
      return m_instance;
   }

   /***************************************************************************
   ** FUNCTION:  template <class C> static std::shared_ptr<T> getInstance(C* val)
   ***************************************************************************/
   /*!
   * \fn      template <class C> static std::shared_ptr<T> getInstance(C* val)
   * \brief   Method to get the instance of the singleton class which takes
   *		  one parameter as an argument
   * \param   Template class object
   * \sa
   ***************************************************************************/
   template <class C> 
   static std::shared_ptr<T> getInstance(C* val)
   {
      if (!m_instance)
      {
         //Two threads can call getInstance() method at the same time.
         //In such cases, to prevent creating object twice, make it thread safe.

         //Additional NULL check was kept above, to avoid entering critical section every time,
         //whenever clients does a getInstance(). With this approach we avoid entering into critical section always
         //and avoid delay in acquiring the object. 
         m_sLock.s16Lock();
         if (!m_instance)
         {
            m_instance = std::shared_ptr<T>(new T(val));
         }//if (!m_instance)
         m_sLock.vUnlock();

      }//if (!m_instance)
      return m_instance;
   }

   /***************************************************************************
   ** FUNCTION:  template <class C, class M> static std::shared_ptr<T> getInstance(C* val1, M* val2)
   ***************************************************************************/
   /*!
   * \fn      template <class C, class M> static std::shared_ptr<T> getInstance(C* val1, M* val2)
   * \brief   Method to get the instance of the singleton class which takes
   *		  two parameters as arguments.
   * \param   Template class object
   * \sa
   ***************************************************************************/
   template <class C, class M> 
   static std::shared_ptr<T> getInstance(C* val1, M* val2)
   {
      if (!m_instance)
      {
         //Two threads can call getInstance() method at the same time.
         //In such cases, to prevent creating object twice, make it thread safe.

         //Additional NULL check was kept above, to avoid entering critical section every time,
         //whenever clients does a getInstance(). With this approach we avoid entering into critical section always
         //and avoid delay in acquiring the object. 
         m_sLock.s16Lock();
         if (!m_instance)
         {
            m_instance = std::shared_ptr<T>(new T(val1, val2));
         }//if (!m_instance)
         m_sLock.vUnlock();

      }//if (!m_instance)
      return m_instance;
   }
   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  SharedSingleton::SharedSingleton()
   ***************************************************************************/
   /*!
   * \fn      SharedSingleton()
   * \brief   Constructor
   * \param   tVoid
   * \sa      ~SharedSingleton()
   ***************************************************************************/
   SharedSingleton()
   {
   }

   /***************************************************************************
   ** FUNCTION: virtual SharedSingleton::~SharedSingleton()
   ***************************************************************************/
   /*!
   * \fn      virtual ~SharedSingleton()
   * \brief   Destructor
   * \param   tVoid
   * \sa      SharedSingleton()
   ***************************************************************************/
   virtual ~SharedSingleton()
   {

   }

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   //! Static instance of the class
   static std::shared_ptr<T> m_instance;

   static Lock m_sLock;

   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

}; //class SharedSingleton

template <class T> std::shared_ptr<T> SharedSingleton<T>::m_instance = nullptr;
template <class T> Lock SharedSingleton<T>::m_sLock;


#endif // _SHAREDSINGLETON_H

///////////////////////////////////////////////////////////////////////////////
// <EOF>
