/***********************************************************************/
/*!
* \file  DynamicLibHandler.h
* \brief Wrapper class for shared library usage
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Wrapper class for shared library usage
AUTHOR:         Unmukt Jain
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
27.02.2018  | Unmukt Jain           | Initial Version

\endverbatim
*************************************************************************/

#ifndef _DYNAMIC_LIB_HANDLER_H_
#define _DYNAMIC_LIB_HANDLER_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "BaseTypes.h"

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/****************************************************************************/
/*!
* \class DynamicLibHandler
* \brief
* It provides methods to interface with shared libraries.
****************************************************************************/
class DynamicLibHandler
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  DynamicLibHandler::DynamicLibHandler()
   ***************************************************************************/
   /*!
   * \fn      DynamicLibHandler()
   * \brief   Default Constructor
   * \sa      ~DynamicLibHandler()
   ***************************************************************************/
    explicit DynamicLibHandler(const t_String &szLibName);

   /***************************************************************************
   ** FUNCTION: DynamicLibHandler::~DynamicLibHandler()
   ***************************************************************************/
   /*!
   * \fn      ~DynamicLibHandler()
   * \brief   Destructor
   * \sa      DynamicLibHandler()
   ***************************************************************************/
   virtual ~DynamicLibHandler();

   /***************************************************************************
   ** FUNCTION:  t_U32 DynamicLibHandler::u32GetSymbol(const t_String &szSymbolName,
   **                                                  t_Void *vpoSymbol)
   ***************************************************************************/
   /*!
   * \fn      u32GetSymbol
   * \brief   Populates the address of the szSymbolName into vpoSymbol.
   *          Please typecast vpoSymbol into proper type before using.
   *          If szSymbolName does not exist, then vpoSymbol is made NULL.
   * \param   szSymbolName [IN] : Name of the symbol to be searched
   * \param   vpoSymbol [OUT] : Address of szSymbolName
   * \retval  t_U32: value > 0 if szSymbolName is found
   *                 0 otherwise
   ***************************************************************************/
   t_U32 u32GetSymbol(const t_String &szSymbolName, t_Void **vpoSymbol);

   /***************************************************************************
   ** FUNCTION:  t_U32 DynamicLibHandler::u32CloseLib()
   ***************************************************************************/
   /*!
   * \fn      u32CloseLib
   * \brief   Unloads the shared Library that was loaded when object was constructed.
   *          Please use this function only when it is safe to unload the shared
   *          library.
   * \param   None
   * \retval  t_U32: value > 0 if szSymbolName is found
   *                 0 otherwise
   ***************************************************************************/
   t_U32 u32CloseLib();

private:

   /***************************************************************************
    ** FUNCTION: DynamicLibHandler(const DynamicLibHandler &rhs)
    ***************************************************************************/
   /*!
    * \fn      DynamicLibHandler(const DynamicLibHandler &rhs)
    * \brief   Copy constructor not implemented hence made private
    **************************************************************************/
   DynamicLibHandler(const DynamicLibHandler &rhs);

   /***************************************************************************
    ** FUNCTION: const DynamicLibHandler & operator=(const DynamicLibHandler &rhs);
    ***************************************************************************/
   /*!
    * \fn      const DynamicLibHandler & operator=(const DynamicLibHandler &rhs);
    * \brief   assignment operator not implemented hence made private
    **************************************************************************/
   DynamicLibHandler& operator=(const DynamicLibHandler &rhs);

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   //! handle for Dynamic Library
   t_Void *m_vHandle;

   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

};

#endif  // _DYNAMIC_LIB_HANDLER_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
