/***********************************************************************/
/*!
* \file  spi_tclStaticConfiguration.cpp
* \brief Class to get the Variant specific Info for INF4CV Project
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Class to get the Variant specific Info for INF4CV Project
AUTHOR:         Chaitra Srinivasa
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
23.04.2015  | Chaitra Srinivasa     | Initial Version. Based on AIVI configuration.

\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include<bitset>
#include "Trace.h"
#include "spi_tclStaticConfiguration.h"
#include "AAPTypes.h"
#include "DiPOTypes.h"


#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_APPLICATION
#include "trcGenProj/Header/spi_tclStaticConfiguration.cpp.trc.h"
#endif
#endif

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

static const t_U16 cou16VARIANTCONFIG_KEY = 0xA110;
static const t_U8 cou8VARIANTCONFIG_SIZE = 0x40;
static const t_U8 cou8SPI_SUPPORTED_BYTE = 32;

static const t_U8 couCARPLAY_SUPPORTED_BIT = 8; //the bit poistion 4 indicate the CarPlay enabled flag. So the corresponding digit is 8.
static const t_U8 couAAP_SUPPORTED_BIT = 16; // the bit position 5 indicates the AAP enabled flag. So the corresponding digit is 16.
static const t_U8 couMSPIN_SUPPORTED_BIT = 32; // the bit position 6 indicates the MySPIN enabled flag. So the corresponding digit is 32.
static const t_U8 couML_SUPPORTED_BIT = 64;  // the bit position 7 indicates the ML enabled flag. So the corresponding digit is 64.

static const t_U16 cou16DRIVESIDECONFIG_KEY = 0xA112;
static const t_U8 cou8DRIVESIDECONFIG_SIZE = 0x40;

static const t_U8 cou8REGIONCONFIG_BYTE = 0;

//Default Client Friendly Name
static const t_String sczClientFriendlyName = "" ;
//Client Manufacturer Name
static const t_String sczHeadUnitManfacturerName = "Bosch";
//Vehicle Model Name
static const t_String sczHeadUnitModelName = "Media Center";

//Changing the Vehicle Manfacturer Name from "Bosch" to "Daimler" according to the request from Project team
//Vehicle Manufacturer Name
static t_String sczVehicleManfacturerName = "Daimler";
//vehicle Model Year
static t_String sczVehicleModelYear = "MY2019";
//Changing the Vehicle model Name from "Bosch" to "BR453" according to the request from Project team
//Vehicle Model Name
static t_String sczVehicleModelName = "BR453";

//new smartphoneIntegeration table is created by the KDS
static const t_U16 scou16SPIKDS_KEY = 0xA0C4;
static const t_U16 scou16SPIKDSKNOBSUPPORTED_KEY = 0xA114;
static const t_U16 scou16SPIKDSKNOBSUPPORTED_SIZE = 32;
static const t_U16 scou16SPIKDSKEY_SIZE = 224;
static const t_U16 scou16MAXNUMBEROFBYTESTOBEREAD = 50;

static const t_U16 scou16OEMIconNameStartByte = 4;
static const t_U16 scou16ManufacturerNameStartBye = 54;
static const t_U16 scou16ManufacturerModelNameStartByte = 104;
static const t_U16 scou16ManufacturerModelYearStartByte = 154;
static const t_U16 scou16KnobKeySupoortByte = 6;

static const t_U8 scou8OEMIconEnabledBit = 1;
static const t_U8 scou8NavigationInfoStartBit = 2;
static const t_U8 scou8NavigationInfoEndBit = 3;
static const t_U8 scou8DriverSideInfoStartBit = 4;
static const t_U8 scou8DriverSideInfoEndBit = 5;
static const t_U16 scou16KnobKeySupoortBit = 6;

static const t_U8 scouRegionInfoStartBit = 3;
static const t_U8 scouRegionInfoEndBit = 7;

static const t_String scszOEMIconPath = "/var/opt/bosch/static/spi/CarPlay/icons/smart/";

static const t_U8 scouSpiConfigDataByte = 3;

static const t_U8 cou8OEMBRAND_SIZE = 0x05;

static const t_U8 scou8OEMBrandInfoByte = 4;

static const t_U16 scou16NAVVARIANTCONFIG_KEY = 0xA114;
static const t_U8 scou8NAVVARIANTCONFIG_SIZE = 0x40;
static const t_U8 scou8NAVVARIANTCONFIG_BYTE = 0;

static const t_U16 scou16CMVARIANTCONFIG_KEY = 0x0DF4;
static const t_U8 scou8CMVARIANTCONFIG_SIZE = 0x40;
static const t_U8 scou8CMVARIANTCONFIG_BYTE = 14;

//#define OEM_ICON_NAME ""
//#define NISAAN_ICON_PATH ""
//#define RENAULT_ICON_PATH "/var/opt/bosch/static/spi/CarPlay/icons/Nissan/Renault.png"
#define ICON_INFO_LENGTH 150

/***************************************************************************
** FUNCTION:  spi_tclStaticConfiguration::spi_tclStaticConfiguration()
***************************************************************************/
spi_tclStaticConfiguration::spi_tclStaticConfiguration(spi_tclCalibrationReaderBase *poCalibrationReader):
spi_tclStaticConfigurationBase(poCalibrationReader),
m_enGeoCoordinateSystemType(e8GEO_COORDINATE_SYSTEM_WGS),
m_u8NavigationDataEnabled(0x00),
m_u8DriveSideInfo(e8RIGHT_HAND_DRIVE),
m_bOEMIconEnabled(false)
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration() entered "));
   //vGetSPIConfigurationData();
}

/***************************************************************************
** FUNCTION:  spi_tclStaticConfiguration::~spi_tclStaticConfiguration()
***************************************************************************/
spi_tclStaticConfiguration::~spi_tclStaticConfiguration()
{
   ETG_TRACE_USR1(("~spi_tclStaticConfiguration() entered "));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfiguration::vGetSpiFeatureSupport(...)
***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
{

   t_U8 u8DataBuf[cou8VARIANTCONFIG_SIZE];
   t_Bool bIsDipoSupp = false, bIsAAPSupp = false, bIsMLSupp = false, bIsMySPINSupp = false;
   t_Bool bReadStatus = false;
   if(NULL != m_poCalibrationReader)
   {
      bReadStatus = m_poCalibrationReader->bReadData(cou16VARIANTCONFIG_KEY, cou8VARIANTCONFIG_SIZE, u8DataBuf);
   }
   if (true == bReadStatus)
   {
	   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetSpiFeatureSupport: CarPlay KDS Data byte value = %d", u8DataBuf[cou8SPI_SUPPORTED_BYTE]));
       ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetSpiFeatureSupport: AAP KDS Data byte value = %d", u8DataBuf[cou8SPI_SUPPORTED_BYTE]));

      bIsDipoSupp = (t_Bool) (u8DataBuf[cou8SPI_SUPPORTED_BYTE] & couCARPLAY_SUPPORTED_BIT);
      bIsAAPSupp  = (t_Bool) (u8DataBuf[cou8SPI_SUPPORTED_BYTE] & couAAP_SUPPORTED_BIT);
      bIsMLSupp = (t_Bool)(u8DataBuf[cou8SPI_SUPPORTED_BYTE] & couML_SUPPORTED_BIT);
	  bIsMySPINSupp = (t_Bool)(u8DataBuf[cou8SPI_SUPPORTED_BYTE] & couMSPIN_SUPPORTED_BIT);
   }

   //! Set supported SPI features
   rfrSpiFeatureSupport.vSetMirrorLinkSupport(bIsMLSupp);
   rfrSpiFeatureSupport.vSetDipoSupport(bIsDipoSupp);
   rfrSpiFeatureSupport.vSetAndroidAutoSupport(bIsAAPSupp);
   rfrSpiFeatureSupport.vSetmySPINSupport(bIsMySPINSupp);
   rfrSpiFeatureSupport.vSetOnCarSupport(false);

   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetSpiFeatureSupport: MIRRORLINK supported = %d ",
         ETG_ENUM(BOOL, bIsMLSupp)));
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetSpiFeatureSupport: DIPO supported = %d ",
         ETG_ENUM(BOOL, bIsDipoSupp)));
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetSpiFeatureSupport: ANDROID AUTO supported = %d ",
         ETG_ENUM(BOOL, bIsAAPSupp)));
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetSpiFeatureSupport: MYSPIN supported = %d",
         ETG_ENUM(BOOL,bIsMySPINSupp)));
}

/***************************************************************************
** FUNCTION   :t_Void spi_tclStaticConfiguration::vGetOemIconData(
              trVehicleBrandInfo& rfrVehicleBrandInfo)
***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo)
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() entered "));
   memset(rfrVehicleBrandInfo.szOemIconPath, 0, ICON_INFO_LENGTH);
   memset(rfrVehicleBrandInfo.szOemName, 0, ICON_INFO_LENGTH);
   memset(rfrVehicleBrandInfo.szManufacturer, 0, STR_LENGTH);
   memset(rfrVehicleBrandInfo.szModel, 0, STR_LENGTH);
   memset(rfrVehicleBrandInfo.szOemIconsPath, 0, ICON_INFO_LENGTH);


    rfrVehicleBrandInfo.bIsOEMIconEnable = m_bOEMIconEnabled;
    ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData  OEMIconEnabled = %d",ETG_ENUM(BOOL,rfrVehicleBrandInfo.bIsOEMIconEnable)));

    strncpy(rfrVehicleBrandInfo.szManufacturer, m_szVehicleManufacturerName.c_str(), m_szVehicleManufacturerName.length());
    strncpy(rfrVehicleBrandInfo.szModel , m_szVehicleModelName.c_str(), m_szVehicleModelName.length());

    if(rfrVehicleBrandInfo.bIsOEMIconEnable)
    {
        strncpy(rfrVehicleBrandInfo.szOemIconsPath , scszOEMIconPath.c_str(), scszOEMIconPath.length());
        //only If OEM Icon is Enabled
        if(false == m_szOEMIconName.empty())
        {
            strncpy(rfrVehicleBrandInfo.szOemName , m_szOEMIconName.c_str(), m_szOEMIconName.length());
            strncpy(rfrVehicleBrandInfo.szOemIconPath , szGetOEMIconPath().c_str(),szGetOEMIconPath().length());
            strncat(rfrVehicleBrandInfo.szOemIconsPath, m_szOEMIconName.c_str(), m_szOEMIconName.length());
        }
        else
        {
            ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() - OEMIconName  is Empty/Contains only whitespaces "));
        }

    }
    else
    {
        ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() - OEMIcon is Disabled "));
    }

    ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() - OEMIcon Name is %s ",rfrVehicleBrandInfo.szOemName));
    ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() - OEMIcon Path is %s ",rfrVehicleBrandInfo.szOemIconPath));
    ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() - Vehicle Manufacturer Name is %s ",rfrVehicleBrandInfo.szManufacturer));
    ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() - Vehicle Model Name Path is %s ",rfrVehicleBrandInfo.szModel));
    ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetOemIconData() - OEMIcons Path is %s ",rfrVehicleBrandInfo.szOemIconsPath));
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetClientID()
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetClientID() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::szGetClientID() left with ClientID = %s",
      cszClientProfileId.c_str()));

   return cszClientProfileId.c_str();
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetClientFriendlyName()
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetClientFriendlyName() const
{

   ETG_TRACE_USR1(("spi_tclStaticConfiguration::szGetClientFriendlyName() left with FriendlyName = %s",
      sczClientFriendlyName.c_str()));

   return sczClientFriendlyName.c_str();
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetClientManufacturerName()
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetClientManufacturerName() const
{
   ETG_TRACE_USR2(("Vehicle Manufacturer Name = %s",m_szVehicleManufacturerName.c_str()));
   return m_szVehicleManufacturerName;
}
/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetModelYear()
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetModelYear()
{
	if((false == bGetParsedString(m_szVehicleModelYear))||(true == m_szVehicleModelYear.empty()))
	{
		m_szVehicleModelYear = sczVehicleModelYear;
	}

	ETG_TRACE_USR2(("spi_tclStaticConfiguration::szGetModelYear left with Vehicle Manufacturer Name = %s",m_szVehicleModelYear.c_str()));
	return m_szVehicleModelYear;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetHeadUnitManufacturerName()
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetHeadUnitManufacturerName()
{
	ETG_TRACE_USR1(("spi_tclStaticConfiguration::szGetHeadUnitManufacturerName() left with Head Unit Manufacturer name = %s",
      sczHeadUnitManfacturerName.c_str()));
   return sczHeadUnitManfacturerName;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szReadVehicleManufacturerName(..
***************************************************************************/
t_String spi_tclStaticConfiguration::szReadVehicleManufacturerName(const t_String coszVehicleManufacturerName)
{
	ETG_TRACE_USR1(("spi_tclStaticConfiguration::szReadVehicleManufacturerName Entered "));
	t_String szVehicleManufacturerName = coszVehicleManufacturerName;
	//If the value read from the KDS is empty, then populating it with the Default value
	if((false == bGetParsedString(szVehicleManufacturerName))||(true == szVehicleManufacturerName.empty()))
	{
		szVehicleManufacturerName = sczVehicleManfacturerName;
	}
	ETG_TRACE_USR2(("spi_tclStaticConfiguration::szGetVehcileManufacturerName left with Vehicle Manufacturer Name = %s",szVehicleManufacturerName.c_str()));
	return szVehicleManufacturerName;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetHeadUnitModelName()
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetHeadUnitModelName()
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::szGetHeadUnitModelName() left with Head unit model name = %s",
      sczHeadUnitModelName.c_str()));
   return sczHeadUnitModelName;
}


/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetHeadUnitModelName(const t_String coszVehicleModelName)
***************************************************************************/
t_String spi_tclStaticConfiguration::szReadVehicleModelName(const t_String coszVehicleModelName)
{
	ETG_TRACE_USR1(("spi_tclStaticConfiguration::szReadVehicleModelName Entered"));
	t_String szVehicleModelName = coszVehicleModelName;
	//If the value read from the KDS is empty/containing only whitespaces, then populating it with the Default value
	if((false == bGetParsedString(szVehicleModelName))||(true == szVehicleModelName.empty()))
	{
		szVehicleModelName = sczVehicleModelName;
	}
	ETG_TRACE_USR1(("spi_tclStaticConfiguration::szGetVehicleModelName() left with Vehicle model name = %s",
			szVehicleModelName.c_str()));
	return szVehicleModelName;
}

/***************************************************************************
** FUNCTION:  tenRegion spi_tclStaticConfiguration::enGetRegion()
***************************************************************************/
tenRegion spi_tclStaticConfiguration::enGetRegion() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::enGetRegion() "));
   //return default value
   return e8_WORLD;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfiguration::bGetRotaryCtrlSupport()
***************************************************************************/
t_Bool spi_tclStaticConfiguration::bGetRotaryCtrlSupport() 
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::bGetRotaryCtrlSupport Entered"));

   t_U8 u8DataBuff[scou16SPIKDSKNOBSUPPORTED_SIZE];
   // By default Rotary Controller is enabled
   t_Bool bIsRotCtrlSupported = true;
   t_Bool bReadStatus = false;

   if(NULL != m_poCalibrationReader)
   {
      bReadStatus = m_poCalibrationReader->bReadData(scou16SPIKDSKNOBSUPPORTED_KEY, scou16SPIKDSKNOBSUPPORTED_SIZE , u8DataBuff);
   }
   if(bReadStatus)
   {
	   //reading 6th Bit from 6th Byte of 0xA114
	   t_U8 u8KnobNotSupported = u8GetNBitsFromByte(u8DataBuff[scou16KnobKeySupoortByte] , scou16KnobKeySupoortBit, scou16KnobKeySupoortBit);
	   bIsRotCtrlSupported = (u8KnobNotSupported)? false : true;
       ETG_TRACE_USR1(("spi_tclStaticConfiguration::bGetRotaryCtrlSupport u8KnobNotSupported= %d",u8KnobNotSupported));
   }
   else
   {
	   ETG_TRACE_ERR(("[ERR]:Error in reading from KDS "));
   }

   ETG_TRACE_USR1(("spi_tclStaticConfiguration::bGetRotaryCtrlSupport bIsRotCtrlSupported= %d",ETG_ENUM(BOOL,bIsRotCtrlSupported)));
   return bIsRotCtrlSupported;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::szGetHardwareVersion()
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetHardwareVersion()
{
    tU8  u8DataBuf[KDS_MAX_ENTRY_LENGTH];
    tU8  u8KDSValueSize = 3;
    tU16 u16KDSAddress = 0x010E;
    t_String szHwVersion;

    if((NULL != m_poCalibrationReader) && (false != m_poCalibrationReader->bReadData(u16KDSAddress, u8KDSValueSize, u8DataBuf)))
    {
       u8DataBuf[u8KDSValueSize] = 0; //string termination
       ETG_TRACE_USR4(("[PARAM]spi_tclStaticConfiguration::szGetHardwareVersion - Hardware version: %s", u8DataBuf));
       szHwVersion = t_String((const char*)u8DataBuf);
    }
    else
    {
       ETG_TRACE_ERR(("[ERR]::szGetHardwareVersion: failed to read Hardware version from KDS"));
    }
    return szHwVersion;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::vGetDisplayInputParam()
***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetDisplayInputParam(t_U8 &rfu8DisplayInput)
{
   t_U8 u8AllianceRegionType = 0;
   vGetRegionInfo(u8AllianceRegionType);
   
   if(u8AllianceRegionType == 0x19) //Japan Region
   {
      //Low fidelity for Japan Region
      rfu8DisplayInput = (true == bGetRotaryCtrlSupport()) ? (LOW_FIDELITY_TOUCH_WITH_KNOB) : (LOW_FIDELITY_TOUCH);
   }
   else
   {
      //High Fidelity for other Regions
      rfu8DisplayInput = (true == bGetRotaryCtrlSupport()) ? (HIGH_FIDELITY_TOUCH_WITH_KNOB) : (HIGH_FIDELITY_TOUCH);
   }

    ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetDisplayInputParam() left with rfu8DisplayInput = %d ",
        rfu8DisplayInput));
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfiguration::vGetVehicleInfoDataAAP(trVehicleInfo& rfrVehicleInfo)
***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetVehicleInfoDataAAP(trVehicleInfo& rfrVehicleInfo)
{

   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetVehicleInfoDataAAP() Entered"));
   rfrVehicleInfo.szModel = m_szVehicleModelName;
   rfrVehicleInfo.szManufacturer = m_szVehicleManufacturerName;
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetVehicleInfoDataAAP() Left"));

}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclStaticConfiguration::vGetDefaultProjUsageSettings
 ***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetDefaultProjUsageSettings(tenDeviceCategory enSPIType, tenEnabledInfo &enEnabledInfo)
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetDefaultProjUsageSettings entered"));
   SPI_INTENTIONALLY_UNUSED(enSPIType);
   enEnabledInfo = e8USAGE_CONF_REQD;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfiguration::bGetNativeNavigationEnabled
 ***************************************************************************/
t_Bool spi_tclStaticConfiguration::bGetNativeNavigationEnabled()
{
   t_U8 u8DataBuf[scou8NAVVARIANTCONFIG_SIZE];
   memset(u8DataBuf, 0, sizeof(u8DataBuf));
   t_Bool bReadStatus = false;

   if (NULL != m_poCalibrationReader)
   {
      bReadStatus = m_poCalibrationReader->bReadData(scou16NAVVARIANTCONFIG_KEY, scou8NAVVARIANTCONFIG_SIZE, u8DataBuf);
   }

   if (true == bReadStatus)
   {
      t_U8 u8NativeNavigationByte = u8DataBuf[scou8NAVVARIANTCONFIG_BYTE];
      t_U8 u8NativeNavigationType = 0xFF;
      t_U8 u8NativeNavigationStartBit = 0;
      t_U8 u8NativeNavigationStopBit = 4;

      u8NativeNavigationType = u8GetNBitsFromByte(u8NativeNavigationByte,
               u8NativeNavigationStartBit,
               u8NativeNavigationStopBit);
      ETG_TRACE_USR1(("spi_tclConfigReader::bGetNativeNavigationEnabled():u8NativeNavigationType = %d ", u8NativeNavigationType));

      //0x00,0x01 = Navigation variant; 0x02 = Non Navigation DA variant
      bReadStatus = (u8NativeNavigationType >= 0x02) ? false : true;
   }
   ETG_TRACE_USR1(("spi_tclConfigReader::bGetNativeNavigationEnabled():left with result = %d", ETG_ENUM(BOOL,
            bReadStatus)));
   return bReadStatus;
}

/***************************************************************************
** FUNCTION:  t_U8 spi_tclStaticConfiguration::u8GetNBitsFromByte(...)
***************************************************************************/
t_U8 spi_tclStaticConfiguration::u8GetNBitsFromByte(t_U8 u8Byte, t_U8 u8LowerBit, t_U8 u8HigherBit)
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::u8GetNBitsFromByte entered"));
   t_U8 u8mask = 0;
   for (t_U8 i = u8LowerBit; i <= u8HigherBit; i++)
       u8mask |= 1 << i;
   return ((u8Byte & u8mask) >> u8LowerBit);
}
/***************************************************************************
** FUNCTION:  t_U8 spi_tclStaticConfiguration::vGetRegionInfo(...)
***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetRegionInfo(t_U8 &rfu8AllianceRegionType)
{
   t_U8 u8DataBuf[cou8DRIVESIDECONFIG_SIZE];
   t_Bool bReadStatus = false;
 
   if(NULL != m_poCalibrationReader)
   {
      bReadStatus = m_poCalibrationReader->bReadData(cou16DRIVESIDECONFIG_KEY, cou8DRIVESIDECONFIG_SIZE, u8DataBuf);
   }
   if(true == bReadStatus)
   {
      //reading 0th byte
      t_U8 u8RegionByte = u8DataBuf[cou8REGIONCONFIG_BYTE];
      rfu8AllianceRegionType = u8GetNBitsFromByte(u8RegionByte, scouRegionInfoStartBit, scouRegionInfoEndBit);
   }
   
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::u8GetRegionInfo() left with u8AllianceRegionType = %d ",
        rfu8AllianceRegionType));
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfiguration::bIsLocDataEnabled
***************************************************************************/
t_Bool spi_tclStaticConfiguration::bIsLocDataEnabled()
{
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bIsLocDataEnabled entered"));
   t_Bool bResult=bGetNativeNavigationEnabled() || bIsGNSSAvailable();
   return bResult;
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfiguration::enGetGeoCoordinateSystem()
***************************************************************************/
tenGeoCoordinateSystemType spi_tclStaticConfiguration::enGetGeoCoordinateSystem()
{
   t_U8 u8AllianceRegionType = 0;
   vGetRegionInfo(u8AllianceRegionType);

   if((0x07 == u8AllianceRegionType) || (0x09 == u8AllianceRegionType)) //PRC Region = 0x07, HKG/MACAU = 0x09
   {
      m_enGeoCoordinateSystemType = e8GEO_COORDINATE_SYSTEM_GCJ;
   }
   else
   {
      // World excluding PRC region
      m_enGeoCoordinateSystemType = e8GEO_COORDINATE_SYSTEM_WGS;
   }
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::enGetGeoCoordinateSystem() left with Geo Coordinate System = %d",m_enGeoCoordinateSystemType));
   return m_enGeoCoordinateSystemType;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfiguration::szExtractStr(const t_U8* pu8Data,t_U8 u8FromPos,t_U8 u8Len)
***************************************************************************/
t_String spi_tclStaticConfiguration::szExtractStr(const t_U8* pu8Data,t_U8 u8FromPos,t_U8 u8Len)
{
   //logic is changed as a fix for NCG3D-87090
   t_String szOutput((const char*)(pu8Data + u8FromPos), u8Len);
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::szExtractStr-%s",szOutput.c_str()));
   return szOutput;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfiguration::vGetSPIConfigurationData()
***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetSPIConfigurationData()
{
	ETG_TRACE_USR1((" spi_tclStaticConfiguration::vGetSPIConfigurationData() Entered"));
	t_U8 au8DataBuff[scou16SPIKDSKEY_SIZE];
	memset(au8DataBuff, 0, sizeof(au8DataBuff));
	t_Bool bReadStatus = false;

	//reading the complete 224 bytes of data from KDS
   if(NULL != m_poCalibrationReader)
   {
      bReadStatus = m_poCalibrationReader->bReadData(scou16SPIKDS_KEY, scou16SPIKDSKEY_SIZE, au8DataBuff);
   }

	if(bReadStatus)
	{
		//reading vehicleManufacturerName
		m_szVehicleManufacturerName = szReadVehicleManufacturerName(szExtractStr(au8DataBuff, scou16ManufacturerNameStartBye,scou16MAXNUMBEROFBYTESTOBEREAD).c_str());

		//reading VehicleModelName
		m_szVehicleModelName = szReadVehicleModelName(szExtractStr(au8DataBuff, scou16ManufacturerModelNameStartByte,scou16MAXNUMBEROFBYTESTOBEREAD).c_str());

		//reading VehicleModelYear
		m_szVehicleModelYear = szExtractStr(au8DataBuff, scou16ManufacturerModelYearStartByte,scou16MAXNUMBEROFBYTESTOBEREAD).c_str();

		//Navigation Enabled
		m_u8NavigationDataEnabled = u8GetNBitsFromByte(au8DataBuff[scouSpiConfigDataByte],scou8NavigationInfoStartBit,scou8NavigationInfoEndBit);

		//DriverSideInfo
		m_u8DriveSideInfo= (tenDriveSideInfo)u8GetNBitsFromByte(au8DataBuff[scouSpiConfigDataByte],scou8DriverSideInfoStartBit,scou8DriverSideInfoEndBit);

		//OEMIcon Enabled
		m_bOEMIconEnabled = u8GetNBitsFromByte(au8DataBuff[scouSpiConfigDataByte],scou8OEMIconEnabledBit,scou8OEMIconEnabledBit);
		if(m_bOEMIconEnabled)
		{
			//reading OEMIcon Name
			m_szOEMIconName = szGetOEMIconName(szExtractStr(au8DataBuff, scou16OEMIconNameStartByte,scou16MAXNUMBEROFBYTESTOBEREAD).c_str());
		}
		else
		{
         ETG_TRACE_USR1((" spi_tclStaticConfiguration::vGetSPIConfigurationData() OEMIcon is disabled"));
		}
	}
	else
	{
	   ETG_TRACE_ERR(("[ERR]:spi_tclStaticConfiguration::vGetSPIConfigurationData() Error in reading from KDS "));
	}
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfiguration::szGetOEMIconName
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetOEMIconName(const t_String coszOEMICONName)
{
	ETG_TRACE_USR2(("spi_tclStaticConfiguration::szGetOEMIconName entered"));
	t_String szOEMIconName = coszOEMICONName;

	if(((false == bGetParsedString(szOEMIconName))||(true == szOEMIconName.empty())))
	{
		ETG_TRACE_ERR(("[ERR]:Error in reading from KDS/May be the String is Empty "));
		szOEMIconName.clear();
	}

	ETG_TRACE_USR2(("spi_tclStaticConfiguration::szGetOEMIconName left with OEMIcon Name is = %s",szOEMIconName.c_str()));
	return szOEMIconName;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfiguration::szGetOEMIconPath
***************************************************************************/
t_String spi_tclStaticConfiguration::szGetOEMIconPath()
{
	ETG_TRACE_USR2(("spi_tclStaticConfiguration::szGetOEMIconPath entered"));
	t_String szOEMIconPath;
	t_String szOEMIconPathNameExtension = ".png";
	if(m_bOEMIconEnabled)
	{
		szOEMIconPath = scszOEMIconPath + m_szOEMIconName + szOEMIconPathNameExtension;
	}
	ETG_TRACE_USR2(("spi_tclStaticConfiguration::szGetOEMIconPath left with OEMIcon Path is = %s",szOEMIconPath.c_str()));
	return szOEMIconPath;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfiguration::bGetParsedString
***************************************************************************/
t_Bool spi_tclStaticConfiguration::bGetParsedString(t_String &rszStringToBeParsed)
{
	//string that contains all the whitspaces
	t_String szwhitespaces (" \t\f\v\n\r");
	t_Bool bResult = false;
	ETG_TRACE_USR2(("spi_tclStaticConfiguration::bGetParsedString entered "));

	auto s8found = rszStringToBeParsed.find_last_not_of(szwhitespaces);

	if (s8found!=std::string::npos)
	{
		//removing all whitespaces from the end
		rszStringToBeParsed.erase(s8found+1);
		bResult = true;
	}
	else
	{
		ETG_TRACE_USR2(("spi_tclStaticConfiguration::bGetParsedString string contains whitespaces/Empty"));
	}

	ETG_TRACE_USR2(("spi_tclStaticConfiguration::bGetParsedString left with the string %s",rszStringToBeParsed.c_str()));
	ETG_TRACE_USR2(("spi_tclStaticConfiguration::bGetParsedString left with the bResult %d",ETG_ENUM(BOOL,bResult)));

	return bResult;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfiguration::bGetOOBTEnabled
***************************************************************************/
t_Bool spi_tclStaticConfiguration::bSetUpAPonOOBT()
{
	ETG_TRACE_USR2(("spi_tclStaticConfiguration::bSetUpAPonOOBT() Entered"));
	//As of Now, its Enabled by default - In-future, If multiple projects request for OOBT feature, then we need to have KDS Configuration
	return true;
}

/***************************************************************************
** FUNCTION:  tenDAPAttestationInfo spi_tclStaticConfiguration::enGetDAPAttestationInfo()
***************************************************************************/
tenDAPAttestationInfo spi_tclStaticConfiguration::enGetDAPAttestationInfo() const
{
    tenDAPAttestationInfo enDAPAttestationInfo = e8_DAP_ATTESTATION_NOT_REQ;
    ETG_TRACE_USR1(("spi_tclStaticConfiguration::enGetDAPAttestationInfo() left with enDAPAttestationInfo = %d ",
         ETG_ENUM(DAP_ATTEST_INFO,enDAPAttestationInfo)));
    return enDAPAttestationInfo;
}


/***************************************************************************
** FUNCTION:  tenDriveSideInfo spi_tclStaticConfiguration::enGetDriveSideInfofromKDS(...)
***************************************************************************/
tenDriveSideInfo spi_tclStaticConfiguration::enGetDriveSideInfofromKDS()
{
	ETG_TRACE_USR1(("spi_tclStaticConfiguration::enGetDriveSideInfofromKDS() DriverSideInfo = %x",m_u8DriveSideInfo));
	if(m_u8DriveSideInfo == e8UNKNOWN_DRIVE_SIDE)
	{
		ETG_TRACE_USR1(("spi_tclStaticConfiguration::enGetDriveSideInfofromKDS() DriverSideInfo is UNKNOWN"));
		//populating it with the default value
		m_u8DriveSideInfo = e8RIGHT_HAND_DRIVE;
	}
    ETG_TRACE_USR1(("spi_tclStaticConfiguration::enGetDriveSideInfofromKDS() left with enDriveSideInfo = %d ",
			ETG_ENUM(DRIVE_SIDE_TYPE, m_u8DriveSideInfo)));
	return m_u8DriveSideInfo;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfiguration::vSetVehicleManufacturerInformationtoKDS()
***************************************************************************/
t_Void spi_tclStaticConfiguration::vSetVehicleManufacturerInformationtoKDS(t_Bool bResult, trVehicleManufacturerInformation rVehicleManufacturerInfoAttr)
{
   ETG_TRACE_USR1(("spi_tclDynamicConfiguration::vSetVehicleManufacturerInformationtoKDS() entered"));

   /*
   //values read from the Datapool
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleMake = %s",rVehicleManufacturerInfoAttr.szVehicleMake.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleModel = %s",rVehicleManufacturerInfoAttr.szVehicleModel.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleModelYear = %s",rVehicleManufacturerInfoAttr.szVehicleModelYear.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleHUSwVersion = %s",rVehicleManufacturerInfoAttr.szVehicleHUSwVersion.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleHUHwVersion = %s",rVehicleManufacturerInfoAttr.szVehicleHUHwVersion.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleHUMake = %s",rVehicleManufacturerInfoAttr.szVehicleHUMake.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleHUModel = %s",rVehicleManufacturerInfoAttr.szVehicleHUModel.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() szVehicleHUSwBuild = %s",rVehicleManufacturerInfoAttr.szVehicleHUSwBuild.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation: enDriverPositionInfo  - %d ", ETG_ENUM(DRIVER_SIDE_INFO,rVehicleManufacturerInfoAttr.enDriverSideInfo)));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation: bNativeNavigationEnabled  - %d ", ETG_ENUM(BOOL, rVehicleManufacturerInfoAttr.bIsNativeNavSupported)));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation: bRotaryCtrlSupported  - %d ", ETG_ENUM(BOOL, rVehicleManufacturerInfoAttr.bIsRotaryCtrlSupported)));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation: SPI Supported Technologies  - %d ", rVehicleManufacturerInfoAttr.u16SupportedSPITechnologiesInfo));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() Vehicle OEM Name = %s",rVehicleManufacturerInfoAttr.rOEMIconDetails.szOEMName.c_str()));
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() Vehicle OEM Path = %s",rVehicleManufacturerInfoAttr.rOEMIconDetails.szOEMPath.c_str()));
   ETG_TRACE_USR4(("spi_tclStaticConfiguration::bReadVehicleManufacturerInformation() Fidelity  type = %u",ETG_ENUM(FIDELITY_TYPE,rVehicleManufacturerInfoAttr.enFidelity)));

   if((bResult==true)&&(!rVehicleManufacturerInfoAttr.szVehicleHUSwVersion.empty()))
   {
      m_rVehicleManufacturerInformationtoKDS = rVehicleManufacturerInfoAttr;
   }
   else
   {
      //For the First Time when no values will be in the datapool
      ETG_TRACE_USR1(("spi_tclStaticConfiguration::vSetVehicleManufacturerInformationtoKDS() Default Values"));
      m_rVehicleManufacturerInformationtoKDS.szVehicleMake = sczVehicleManfacturerName;
      m_rVehicleManufacturerInformationtoKDS.szVehicleModel = sczVehicleModelName;
      m_rVehicleManufacturerInformationtoKDS.szVehicleModelYear = sczVehicleModelYear;
      m_rVehicleManufacturerInformationtoKDS.szVehicleHUMake =sczHeadUnitManfacturerName ;
      m_rVehicleManufacturerInformationtoKDS.szVehicleHUModel = sczHeadUnitModelName;
      m_rVehicleManufacturerInformationtoKDS.szVehicleHUSwBuild = sczVehicleHUSwVersion;
      m_rVehicleManufacturerInformationtoKDS.szVehicleHUSwVersion = sczVehicleHUSwVersion;
      m_rVehicleManufacturerInformationtoKDS.szVehicleHUHwVersion = sczVehicleHUSwVersion;
      m_rVehicleManufacturerInformationtoKDS.enDriverSideInfo = e8LEFT_HAND_DRIVE;
      m_rVehicleManufacturerInformationtoKDS.u16SupportedSPITechnologiesInfo = couSPI_TECHNOLOGIES_SUPPORTED ;
      m_rVehicleManufacturerInformationtoKDS.bIsRotaryCtrlSupported = true ;
      m_rVehicleManufacturerInformationtoKDS.bIsNativeNavSupported = false ;
      m_rVehicleManufacturerInformationtoKDS.enFidelity = e8HIGH_FIDELITY;
   }
   */
   ETG_TRACE_USR4(("spi_tclStaticConfiguration::vSetVehicleManufacturerInformationtoKDS() left"));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclStaticConfiguration::vGetDiPOExtendedFeaturesSupport(...)
 ***************************************************************************/
t_Void spi_tclStaticConfiguration::vGetDiPOExtendedFeaturesSupport(trExtendedFeaturesSupport& rfrExtendedFeaturesSupport)
{
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::vGetDiPOExtendedFeaturesSupport() entered"));

   rfrExtendedFeaturesSupport.bEnhancedRequestUISupport = false;
   rfrExtendedFeaturesSupport.bVocoderInfoSupport = false;

   ETG_TRACE_USR4(("[PARAM]: vGetDiPOExtendedFeaturesSupport - Enhanced Request UI Support = %d",
     ETG_ENUM(BOOL, rfrExtendedFeaturesSupport.bEnhancedRequestUISupport)));
   ETG_TRACE_USR4(("[PARAM]: vGetDiPOExtendedFeaturesSupport - vocoder info Support = %d",
     ETG_ENUM(BOOL, rfrExtendedFeaturesSupport.bVocoderInfoSupport)));

}

/************************************************************************************************
 ** FUNCTION:  t_U8 spi_tclStaticConfiguration:: u8ReadOEMBrandFromKDS()
 ************************************************************************************************/
t_U8 spi_tclStaticConfiguration::u8ReadOEMBrandFromKDS()
{
   t_U8 u8DataBuf[cou8OEMBRAND_SIZE];
   memset(u8DataBuf, 0, sizeof(u8DataBuf));
   t_Bool bReadStatus = false;
   if (NULL != m_poCalibrationReader)
   {
      bReadStatus = m_poCalibrationReader->bReadData(cou16VARIANTCONFIG_KEY, cou8OEMBRAND_SIZE, u8DataBuf);
   }
   t_U8 u8OEMBrand = 0x05; //By Default return Diamler
   if (true == bReadStatus)
   {
      u8OEMBrand = u8DataBuf[scou8OEMBrandInfoByte] & 0x07;
   }
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::vReadOEMBrandFromKDS :Value of OEM brand Type from data buffer = %0x", u8OEMBrand));
   return u8OEMBrand;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclStaticConfiguration::bisGNSSAvailabe
 ***************************************************************************/
t_Bool spi_tclStaticConfiguration::bIsGNSSAvailable()
{
   ETG_TRACE_USR2(("spi_tclStaticConfiguration::bIsGNSSAvailable entered"));

   t_U8 u8DataBuf[scou8CMVARIANTCONFIG_SIZE];
   memset(u8DataBuf, 0, sizeof(u8DataBuf));
   t_Bool bReadStatus = false;

   if (NULL != m_poCalibrationReader)
   {
      bReadStatus = m_poCalibrationReader->bReadData(scou16CMVARIANTCONFIG_KEY, scou8CMVARIANTCONFIG_SIZE, u8DataBuf);
   }

   if (true == bReadStatus)
   {
      t_U8 u8GNSSByte = u8DataBuf[scou8CMVARIANTCONFIG_BYTE];
      t_U8 u8GNSSType = 0xFF;
      t_U8 u8GNSSStartBit = 2;
      t_U8 u8GNSSStopBit = 3;

      u8GNSSType = u8GetNBitsFromByte(u8GNSSByte, u8GNSSStartBit, u8GNSSStopBit);
      ETG_TRACE_USR1(("spi_tclStaticConfiguration::bIsGNSSAvailable():u8GNSSType = %d ", u8GNSSType));

      //0x00 = Info not Available; 0x01 = GNSS Mounted; 0x02 = GNSS Not Mounted
      bReadStatus = (u8GNSSType == 0x01) ? true : false;
   }
   ETG_TRACE_USR1(("spi_tclStaticConfiguration::bIsGNSSAvailable():left with result = %d", ETG_ENUM(BOOL, bReadStatus)));
   return bReadStatus;
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>
