/***********************************************************************/
/*!
* \file  spi_tclStaticConfiguration.h
* \brief Class to get the Variant specific Info for INF4CV Project
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Class to get the Variant specific Info for INF4CV Project
AUTHOR:         Chaitra Srinivasa
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
23.04.2015  | Chaitra Srinivasa     | Initial Version. Based on AIVI configuration.

\endverbatim
*************************************************************************/

#ifndef _spi_tclStaticConfiguration_H_
#define _spi_tclStaticConfiguration_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <sstream>
#include <string>
#include "Lock.h"
#include "SPITypes.h"
#include "spi_tclStaticConfigurationBase.h"
#include "spi_tclCalibrationReaderBase.h"

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/****************************************************************************/
/*!
* \class spi_tclStaticConfiguration
* \brief Class to get the Variant specific Info for Gen3 Projects
****************************************************************************/

class spi_tclStaticConfiguration: 
public spi_tclStaticConfigurationBase
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclStaticConfiguration::spi_tclStaticConfiguration()
   ***************************************************************************/
   /*!
   * \fn      spi_tclStaticConfiguration()
   * \brief   Default Constructor
   * \sa      ~spi_tclStaticConfiguration()
   **************************************************************************/
   spi_tclStaticConfiguration(spi_tclCalibrationReaderBase *poCalibrationReader);

   /***************************************************************************
   ** FUNCTION:  spi_tclStaticConfiguration::~spi_tclStaticConfiguration()
   ***************************************************************************/
   /*!
   * \fn      virtual ~spi_tclStaticConfiguration()
   * \brief   Destructor
   * \sa      spi_tclStaticConfiguration()
   **************************************************************************/
   virtual ~spi_tclStaticConfiguration();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclStaticConfiguration::vGetSpiFeatureSupport(...)
   ***************************************************************************/
   /*!
   * \fn      vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
   * \brief   Method to read supported SPI features info.
   * \param   rfrSpiFeatureSupport: [OUT] Provides supported SPI features
   * \retval  NONE
   **************************************************************************/
   t_Void vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport);


   /***************************************************************************
   ** FUNCTION: t_Void spi_tclStaticConfiguration::vGetOemIconData(
                     trVehicleBrandInfo& rfrVehicleBrandInfo)
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo)
   * \brief   Method to read the EOL value and get the Brand configuration data 
              from the look up table
   * \param   rfrVideoConfig: [IN]Vehicle Brand Data
   * \retval  NONE
   **************************************************************************/
   t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo);

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szGetClientID()
   ***************************************************************************/
   /*!
   * \fn      t_String szGetClientID()
   * \brief   Method to get the client ID.
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szGetClientID() const;

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szGetClientFriendlyName()
   ***************************************************************************/
   /*!
   * \fn      t_String szGetClientFriendlyName()
   * \brief   Method to get the client friendly name.
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szGetClientFriendlyName() const;

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szGetClientManufacturerName()
   ***************************************************************************/
   /*!
   * \fn      t_String szGetClientManufacturerName()
   * \brief   Method to get the client manufacturer name.
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szGetClientManufacturerName() const;


   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szGetModelYear()
   ***************************************************************************/
   /*!
   * \fn      t_String szGetModelYear()
   * \brief   Method to get the Model year of the vehicle
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szGetModelYear();

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szGetHeadUnitManufacturerName()
   ***************************************************************************/
   /*!
   * \fn      t_String szGetHeadUnitManufacturerName()
   * \brief   Method to get the Headunit manufacturer name.
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szGetHeadUnitManufacturerName();

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szGetHeadUnitModelName()
   ***************************************************************************/
   /*!
   * \fn      t_String szGetHeadUnitModelName()
   * \brief   Method to get the headunit model name.
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szGetHeadUnitModelName();

   /***************************************************************************
   ** FUNCTION:  tenRegion spi_tclStaticConfiguration::enGetRegion()
   ***************************************************************************/
   /*!
   * \fn      tenRegion enGetRegion()
   * \brief   Method to get the current vehicle sale region
   * \param   NONE
   * \retval  tenRegion
   **************************************************************************/
   tenRegion enGetRegion() const;


   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclStaticConfiguration::bGetRotaryCtrlSupport()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetRotaryCtrlSupport()
   * \brief   Method to know if Rotary Controller is supported
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_Bool bGetRotaryCtrlSupport();

   /***************************************************************************
    ** FUNCTION:  t_String spi_tclStaticConfiguration::szGetHardwareVersion()
    ***************************************************************************/
   /*!
    * \fn      t_String szGetHardwareVersion()
    * \brief   Method to get the accessory hardware version
    * \param   NONE
    * \retval  t_String
    **************************************************************************/
   t_String szGetHardwareVersion() ;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclStaticConfiguration::vGetDisplayInputParam()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetDisplayInputParam()
   * \brief   Method get the display input configuration.Input methods enabled.
   * \param   rfu8DisplayInput : [OUT] Display input parameter for DiPO
   * \retval  t_Void
   **************************************************************************/
   t_Void vGetDisplayInputParam(t_U8 &rfu8DisplayInput);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclStaticConfiguration::vGetVehicleInfoDataAAP(
                     trVehicleBrandInfo& rfrVehicleBrandInfo)
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetVehicleInfoDataAAP(trVehicleBrandInfo& rfrVehicleBrandInfo)
   * \brief   Method to read vehicle information (Model,manufacturer name)
   * \param   rfrVehicleInfo: [IN]Vehicle Info Data
   * \retval  NONE
   **************************************************************************/
   t_Void vGetVehicleInfoDataAAP(trVehicleInfo& rfrVehicleInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclStaticConfiguration::vGetDefaultProjUsageSettings
    ***************************************************************************/
   /*!
    * \fn     t_Void vGetDefaultProjUsageSettings()
    * \brief  Method to retrieve default setting for projection usage
    * \param [OUT] : returns the current value of device projection enable
    * \enSPIType  : indicates the type of SPI technology. e8DEV_TYPE_UNKNOWN indicates default value for any SPI technology
    * \retval t_Void
       **************************************************************************/
   t_Void vGetDefaultProjUsageSettings(tenDeviceCategory enSPIType, tenEnabledInfo &enEnabledInfo);

   /***************************************************************************
  ** FUNCTION: t_Void spi_tclStaticConfiguration::bGetNativeNavigationEnabled(
                    )
  ***************************************************************************/
  /*!
  * \fn      t_Void bGetNativeNavigationEnabled()
  * \brief   Method to read the KDS value for variant type
  * \param   none
  * \retval  NONE
  **************************************************************************/
   t_Bool bGetNativeNavigationEnabled() override;

  /***************************************************************************
   ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bIsLocDataEnabled()
   ***************************************************************************/
  /*!
   * \fn      t_Bool bIsLocDataEnabled()
   * \brief  Method to check if Locattion data is enabled.
   * \param [OUT] : none
   * \retval t_Bool
   **************************************************************************/
   t_Bool bIsLocDataEnabled() ;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclStaticConfiguration::enGetGeoCoordinateSystem()
   ***************************************************************************/
   /*!
   * \fn      t_Void enGetGeoCoordinateSystem()
   * \brief   Method to read the supported coordinate system
   * \retval  tenGeoCoordinateSystemType : To indicate type of Geo Coordinate System
   **************************************************************************/
   tenGeoCoordinateSystemType enGetGeoCoordinateSystem();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclStaticConfiguration::bSetUpAPonOOBT(const t_String coszVehicleModelName)
   ***************************************************************************/
   /*!
   * \fn      t_Bool bSetUpAPonOOBT(const t_String coszVehicleModelName)
   * \brief   Method to Read the OOBT Enabled.
   * \param   NONE
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bSetUpAPonOOBT();

   /***************************************************************************
    ** FUNCTION: tenDAPAttestationInfo spi_tclStaticConfiguration::enGetDAPAttestationInfo(...)
    ***************************************************************************/
   /*!
    * \fn      enGetDAPAttestationInfo()
    * \brief   Method to get DAP attestation information
    * \retval  tenDAPAttestationInfo
    **************************************************************************/
    tenDAPAttestationInfo enGetDAPAttestationInfo() const;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclStaticConfiguration::vSetVehicleManufacturerInformationtoKDS()
    ***************************************************************************/
   /*!
    * \fn     vSetVehicleManufacturerInformationtoKDS()
    * \brief  To set VehicleManufacturerInformation value to static configreader
    * \retval t_Void
    **************************************************************************/
    t_Void vSetVehicleManufacturerInformationtoKDS(t_Bool bResult, trVehicleManufacturerInformation rVehicleManufacturerInfoAttr) override;

    /***************************************************************************
     ** FUNCTION: t_Void spi_tclStaticConfiguration::vGetDiPOExtendedFeaturesSupport(...)
     ***************************************************************************/
     /*!
      * \fn      t_Void vGetDiPOExtendedFeaturesSupport(
      *           trExtendedFeaturesSupport& rfrExtendedFeaturesSupport)
      * \brief   Method to get the ExtendedFeatures support
      * \param   rfrExtendedFeaturesSupport: [OUT] Structure with
      *           extended features support
      * \retval  NONE
      **************************************************************************/
      t_Void vGetDiPOExtendedFeaturesSupport(trExtendedFeaturesSupport& rfrExtendedFeaturesSupport);

   /***************************************************************************
    ** FUNCTION: t_U8 spi_tclStaticConfiguration::u8ReadOEMBrandFromKDS()
    ***************************************************************************/
    /*!
     * \fn      t_U8 u8ReadOEMBrandFromKDS()
     * \brief   Method to read the KDS value and get the Brand type info
     * \retval  NONE
     **************************************************************************/
     t_U8 u8ReadOEMBrandFromKDS() override;

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/



   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclStaticConfiguration::vPrintVideoConfigData()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPrintVideoConfigData(trVideoConfigData& rfrVideoConfig)
    * \brief   Method to Print the video configuration data
    * \param   rfrVideoConfig: [IN] Video Config Data
    * \retval  NONE
    **************************************************************************/
   t_Void vPrintVideoConfigData(const trVideoConfigData& rfrVideoConfig);

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclStaticConfiguration::u8GetNBitsFromByte()
   ***************************************************************************/
   /*!
   * \fn      t_U8 u8GetNBitsFromByte()
   * \brief   Method to get the n specific bits from a given byte
   * \param   u8Byte : [IN] RegionByte
   * \param   u8LowerBit : [IN] Lower order bit
   * \param   u8HigherBit : [IN] Higher order bit
   * \retval  t_U8
   **************************************************************************/
   t_U8 u8GetNBitsFromByte(t_U8 u8Byte, t_U8 u8LowerBit, t_U8 u8HigherBit);
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclStaticConfiguration::vGetRegionInfo()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetRegionInfo()
   * \brief   Method to read Region Type
   * \param   rfu8AllianceRegionType : [OUT]  Returns the Region Type
   * \retval  t_Void
   **************************************************************************/
   t_Void vGetRegionInfo(t_U8 &rfu8AllianceRegionType);
   
   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szExtractStr(...)
   ***************************************************************************/
   /*!
   * \fn      t_String szExtractStr(const t_U8* pu8Data,t_U8 u8FromPos,t_U8 u8Len)
   * \brief   Method to extract the required string from the given array
   * \param   pu8Data : [IN] DataBuffer
   * \param   u8FromPos : [IN] Starting Byte Position
   * \param   u8Len     : [IN] total number of bytes to be read from the starting position
   * \retval  t_String
   **************************************************************************/
   t_String szExtractStr(const t_U8* pu8Data,t_U8 u8FromPos,t_U8 u8Len);

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclStaticConfiguration::vGetSPIConfigurationData()
   ***************************************************************************/
   /*!
   * \fn      t_Bool vGetSPIConfigurationData()
   * \brief   Method get the KDS data
   * \param   u8Data : [out]  DataBuffer
   * \retval  t_Bool
   **************************************************************************/
   t_Void vGetSPIConfigurationData();

   /***************************************************************************
   ** FUNCTION: t_String spi_tclStaticConfiguration::szGetOEMIconName(const t_String coszOEMICONName)
   ***************************************************************************/
   /*!
   * \fn      t_String szGetOEMIconName(const t_String coszOEMICONName)
   * \brief   Method get the OEMIcon Name
   * \retval  t_String
   **************************************************************************/
   t_String szGetOEMIconName(const t_String coszOEMICONName);

   /***************************************************************************
   ** FUNCTION: t_String spi_tclStaticConfiguration::szGetOEMIconPath()
   ***************************************************************************/
   /*!
   * \fn      t_String vGetOEMIconPath()
   * \brief   Method get the OEMIcon Path
   * \retval  t_String
   **************************************************************************/
   t_String szGetOEMIconPath();

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclStaticConfiguration::bGetParsedString(t_String &rszStringToBeParsed)
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetParsedString()
   * \brief   Method get the String Parsed after removal of the Whitespaces
   * Param    rszStringToBeParsed: [out] string after removing the whitespaces
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bGetParsedString(t_String &rszStringToBeParsed);

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szReadVehicleManufacturerName(const t_String coszVehicleManufacturerName)
   ***************************************************************************/
   /*!
   * \fn      t_String szReadVehicleManufacturerName(const t_String coszVehicleManufacturerName)
   * \brief   Method to Read the Vehicle manufacturer name.
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szReadVehicleManufacturerName(const t_String coszVehicleManufacturerName);

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclStaticConfiguration::szReadVehicleModelName(const t_String coszVehicleModelName)
   ***************************************************************************/
   /*!
   * \fn      t_String szReadVehicleModelName(const t_String coszVehicleModelName)
   * \brief   Method to Read the Vehicle model name.
   * \param   NONE
   * \retval  t_String
   **************************************************************************/
   t_String szReadVehicleModelName(const t_String coszVehicleModelName);

   /***************************************************************************
   ** FUNCTION:  tenDriveSideInfo spi_tclStaticConfiguration::enGetDriveSideInfofromKDS(...)
   ***************************************************************************/
   /*!
   * \fn      enGetDriveSideInfofromKDS()
   * \brief   Method to read Drive side info.
   * \param   NONE
   * \retval  Drive Side enum
   **************************************************************************/
   tenDriveSideInfo enGetDriveSideInfofromKDS() override;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclStaticConfiguration::bIsGNSSAvailable()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bIsGNSSAvailable()
   * \brief  Method to check if GNSS is available or not.
   * \param [OUT] : none
   * \retval t_Bool
   **************************************************************************/
   t_Bool bIsGNSSAvailable();

   t_Bool m_bIsBrandInfiniti;

   t_String m_szVehicleManufacturerName;
   t_String m_szVehicleModelName;
   t_String m_szVehicleModelYear;
   t_U8 m_u8NavigationDataEnabled;
   tenDriveSideInfo  m_u8DriveSideInfo;
   t_Bool m_bOEMIconEnabled;
   t_String m_szOEMIconName;

   //! Vehicle Manufacturer Information
   trVehicleManufacturerInformation m_rVehicleManufacturerInformationtoKDS;

   tenGeoCoordinateSystemType m_enGeoCoordinateSystemType;
   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

}; //spi_tclStaticConfiguration

#endif //_spi_tclStaticConfiguration_H_


///////////////////////////////////////////////////////////////////////////////
// <EOF>


