/***********************************************************************/
/*!
* \file  spi_tclSensorPVDataClientHandler.cpp
* \brief SENSOR PV Service Class
*************************************************************************
\verbatim

PROJECT        :   Gen3
SW-COMPONENT   :   Smart Phone Integration
DESCRIPTION    :
AUTHOR         :   SHITANSHU SHEKHAR (RBEI/ECO2:HSK5KOR)
COPYRIGHT      :   &copy; RBEI

HISTORY:
Date        | Author                         | Modification
28.12.2016  | SHITANSHU SHEKHAR (RBEI/ECO2)  | Initial Version

\endverbatim
*************************************************************************/


/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclSensorPVDataClientHandler.h"

#include "XFiObjHandler.h"
#include "FIMsgDispatch.h"
using namespace shl::msgHandler;

//Include common fi interface
#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

//Needed fi sensor_pv_fi related definitions.
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_PVFI_FUNCTIONIDS
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_PVFI_TYPES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_PVFI_ERRORCODES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_PVFI_SERVICEINFO
#include "sensor_fi_if.h"

#define MATH_S_IMPORT_INTERFACE_GENERIC
#include "math_if.h"

#include "spi_LoopbackTypes.h"
#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DATASERVICE
#include "trcGenProj/Header/spi_tclSensorPVDataClientHandler.cpp.trc.h"
#endif

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/*!
* \XFiObjHandler<sensor_pvfi_tclMsgdeadreckoning_infoStatus> spi_tDR_InfoStatus
* \brief  Property status, with auto extract & auto destroy feature.
*/

typedef XFiObjHandler<sensor_pvfi_tclMsgdeadreckoning_infoStatus>
   spi_tDR_InfoStatus;
/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
static const trLocationDataCallbacks rEmptyLocCallbacks;

//! Resolution of Lat/Long values as per SENSOR_PV_FI as per SENSOR_PV_FI properties
//! "DeadReckoning_Info"
//static const t_Double codLatLongResolution = 4294967296/360.0;

//! Resolution of Heading values as per SENSOR_PV_FI property "DeadReckoning_Info"
//static const t_Double codDRHeadingResolution = 100/1.0;

/* Forward declarations */

/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

BEGIN_MSG_MAP(spi_tclSensorPVDataClientHandler, ahl_tclBaseWork)
   // Add your ON_MESSAGE_SVCDATA() macros here to define which corresponding
   // method should be called on receiving a specific message.
   ON_MESSAGE_SVCDATA(SENSOR_PVFI_C_U16_DEADRECKONING_INFO,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDeadreckoningStatus)
END_MSG_MAP()


/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/***************************************************************************
*********************************PUBLIC*************************************
***************************************************************************/

/*******************************************************************************
* FUNCTION: spi_tclSensorPVDataClientHandler::
*             spi_tclSensorPVDataClientHandler(ahl_tclBaseOneThreadApp* poMainAppl)
*******************************************************************************/
spi_tclSensorPVDataClientHandler::spi_tclSensorPVDataClientHandler(ahl_tclBaseOneThreadApp* poMainApp)
   : ahl_tclBaseOneThreadClientHandler(
      poMainApp,                                          /* Application Pointer          */
      CCA_C_U16_SRV_PV,                                   /* ID of used Service           */
      SENSOR_PVFI_C_SERVICE_VERSION_1_1.u16MajorVersion,  /* MajorVersion of used Service */
      SENSOR_PVFI_C_SERVICE_VERSION_1_1.u16MinorVersion), /* MinorVersion of used Service */
     m_poMainApp(poMainApp)
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::spi_tclSensorPVDataClientHandler entered "
            "with Major Version = %d, Minor Version = %d",
            SENSOR_PVFI_C_SERVICE_VERSION_1_1.u16MajorVersion,
            SENSOR_PVFI_C_SERVICE_VERSION_1_1.u16MinorVersion));
}

/*******************************************************************************
* FUNCTION: spi_tclSensorPVDataClientHandler::
*             ~spi_tclSensorPVDataClientHandler(t_Void)
*******************************************************************************/
spi_tclSensorPVDataClientHandler::~spi_tclSensorPVDataClientHandler()
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::~spi_tclSensorPVDataClientHandler entered "));
   m_poMainApp = NULL;
}

/*******************************************************************************
* FUNCTION: t_Void spi_tclSensorPVDataClientHandler::vOnServiceAvailable()
*******************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vOnServiceAvailable()
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::vOnServiceAvailable entered "));
}

/*******************************************************************************
* FUNCTION: t_Void spi_tclSensorPVDataClientHandler::vOnServiceUnavailable(
*******************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vOnServiceUnavailable()
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::vOnServiceUnavailable entered "));
}

/**************************************************************************
** FUNCTION:  t_Void spi_tclSensorPVDataClientHandler::vRegisterForProperties()
**************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vRegisterForProperties(tenSensorDataType enDataType)
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::vRegisterForProperties:"
         " entered for sensor data type = %d \n",
         ETG_ENUM(SENSOR_DATA_TYPE, enDataType)));

   t_Bool bRegisterData = true;
   t_U16 u16RegisterFuncId = 0;
   switch (enDataType)
   {
      case e8GPS_DATA:
      case e8DEADRECKONING_DATA:
         {
            u16RegisterFuncId = SENSOR_PVFI_C_U16_DEADRECKONING_INFO;
         }//case e8DEADRECKONING_DATA:
      break;
      default:
         {
            bRegisterData = false;
            ETG_TRACE_ERR(("[ERR]:Invalid data type %d", ETG_ENUM(SENSOR_DATA_TYPE, enDataType)));
         }
      break;
   }//switch (enDataType)

   if (true == bRegisterData)
   {
      //!Always call the vAddAutoRegisterForProperty from entry thread as ahl is not thread safe
      vSendAutoRegisterPropertyRequest(u16RegisterFuncId);
      ETG_TRACE_USR2(("[DESC]:spi_tclSensorPVDataClientHandler:vRegisterForProperties: "
            "Registered FunctionID = 0x%x (for DataType = %d)",
            u16RegisterFuncId, ETG_ENUM(SENSOR_DATA_TYPE, enDataType)));
   }//if (true == bRegisterData)

}

/**************************************************************************
** FUNCTION:  t_Void spi_tclSensorPVDataClientHandler::vUnregisterForProperties()
**************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vUnregisterForProperties(
      tenSensorDataType enDataType)
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::vUnregisterForProperties:"
         " entered for sensor data type = %d \n",
         ETG_ENUM(SENSOR_DATA_TYPE, enDataType)));

   //! Unregister is triggered from connection manager context,
   //! which needs to be synchronized since there is a chance of
   //! simultaneous access.
   m_lock.s16Lock();
   t_Bool bUnregisterData = true;
   t_U16  u16RegisterFuncId = 0;
   switch (enDataType)
   {
      case e8GPS_DATA:
      case e8DEADRECKONING_DATA:
         {
            u16RegisterFuncId=SENSOR_PVFI_C_U16_DEADRECKONING_INFO;
         }//case e8DEADRECKONING_DATA:
      break;
      default:
         {
            bUnregisterData = false;
            ETG_TRACE_ERR(("[ERR]:Invalid data type %d", ETG_ENUM(SENSOR_DATA_TYPE, enDataType)));
         }
      break;
   }//switch (enDataType)

   ETG_TRACE_USR4(("spi_tclSensorPVDataClientHandler::vUnregisterForProperties:"
            " Unregistered FunctionID = 0x%x \n", u16RegisterFuncId));
   if (true == bUnregisterData)
      {
         //!Always call the vRemoveAutoRegisterForProperty from entry thread as ahl is not thread safe
         vSendAutoUnregisterPropertyRequest(u16RegisterFuncId);
         ETG_TRACE_USR2(("[DESC]:spi_tclSensorPVDataClientHandler:vUnregisterForProperties: "
               "Registered FunctionID = 0x%x (for DataType = %d)",
               u16RegisterFuncId, ETG_ENUM(SENSOR_DATA_TYPE, enDataType)));
      }//if (true == bRegisterData)
   m_lock.vUnlock();
}

/*******************************************************************************
** FUNCTION:   spi_tclSensorPVDataClientHandler::vOnDeadreckoningStatus()
*******************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vOnDeadreckoningStatus(amt_tclServiceData* poMessage)
{
   m_lock.s16Lock();
   spi_tDR_InfoStatus oDeadReckoning_InfoStatus(*poMessage, SENSOR_PVFI_C_SERVICE_VERSION_1_1.u16MajorVersion);

   if((true == oDeadReckoning_InfoStatus.bIsValid()) && (NULL != m_rLocDataCallbacks.fvOnGpsData))
   {
      ETG_TRACE_USR2(("[DESC]spi_tclSensorPVDataClientHandler::vOnDeadreckoningPositionStatus: Source Type = %d",
               oDeadReckoning_InfoStatus.drInfoSource.enType));

      //As per discussion with Kai, he mentioned that if the odometer and other sensors are not calibrated,
      //Info Source value is set to GNSS and the data quality is similar to the GPS data. Same was the behavior with Pos FI.
      //This is why FI_EN_SRC_DRPOS_DR is changed to FI_EN_SRC_DRPOS_INVALID in below condition
      if(sensor_fi_tcl_e8_deadreckoning_info_source::FI_EN_SRC_DRPOS_INVALID != oDeadReckoning_InfoStatus.drInfoSource.enType)
      {
         //! Create GPS data structure by extracting details from the FI data object
         trGPSData rGpsData;

         // When Odometer is calibrated and used for calculation of position, source is set to DR.
         // If Odometer is not calibrated or not available, but GNSS is available, source is set to GNSS.
         // Else it is set to INVALID.
         if(sensor_fi_tcl_e8_deadreckoning_info_source::FI_EN_SRC_DRPOS_DR == oDeadReckoning_InfoStatus.drInfoSource.enType)
         {
            rGpsData.enDRSourceType = e8DR_SRC_DR;
         }
         else
         {
            rGpsData.enDRSourceType = e8DR_SRC_GNSS;
         }
         rGpsData.bLatitudeAvailable = oDeadReckoning_InfoStatus.drInfoValidity.bPositionIsValid();
         rGpsData.bLongitudeAvailable = oDeadReckoning_InfoStatus.drInfoValidity.bPositionIsValid();
         rGpsData.bSpeedAvailable = oDeadReckoning_InfoStatus.drInfoValidity.bSpeedIsValid();
         rGpsData.bAltitudeAvailable = oDeadReckoning_InfoStatus.drInfoValidity.bAltitudeIsValid();

         ETG_TRACE_USR4(("[PARAM]spi_tclSensorPVDataClientHandler::vOnDeadreckoningPositionStatus: Position Availability = %d,"
                  "Heading Availability = %d, Speed Availability = %d, Altitude Availability = %d",
                  ETG_ENUM(BOOL, rGpsData.bLatitudeAvailable),ETG_ENUM(BOOL, rGpsData.bHeadingAvailable),
                  ETG_ENUM(BOOL, rGpsData.bSpeedAvailable),ETG_ENUM(BOOL, rGpsData.bAltitudeAvailable)));

         // Latitude Range {-90.0,90.0}
         rGpsData.d64Latitude = static_cast<t_Double> (oDeadReckoning_InfoStatus.drLatitude);
         // Longitude Range {-180.0,180.0}
         rGpsData.d64Longitude = static_cast<t_Double> (oDeadReckoning_InfoStatus.drLongitude);
         rGpsData.dLatLongResolution = 1;
         rGpsData.dHeadingResolution = 1;

         rGpsData.f32Speed = static_cast<t_Float> ((oDeadReckoning_InfoStatus.drSpeed)* 100);  // converted m/s -> cm/s
         rGpsData.f32Altitude = static_cast<t_Float> (oDeadReckoning_InfoStatus.drAltitude);   // unit -> meter
         //drTimestamp converted ms -> s
         rGpsData.PosixTime = static_cast<t_PosixTime> ((oDeadReckoning_InfoStatus.drTimestamp) / 1000);
         //ms part of drTimestamp populated as Exact time
         rGpsData.u16ExactTime = static_cast<t_U16> ((oDeadReckoning_InfoStatus.drTimestamp) % 1000);
         rGpsData.enSensorType = e8SENSOR_TYPE_COMBINED_LEFT_RIGHT_WHEEL;

         // The dead reckoning heading is Clockwise in [].
         // North is at 0, East is at 90, South is at 180, West is at 270.
         rGpsData.dHeading = static_cast<t_Double> (oDeadReckoning_InfoStatus.drHeading);
         rGpsData.bHeadingAvailable = oDeadReckoning_InfoStatus.drInfoValidity.bHeadingIsValid();
         rGpsData.enHeadingDir = e8CLOCKWISE_0_DEG_NORTH;

         //For TurnRate only use z-axis. The other values are not filled because they are not part of SENSOR_PV DR.
         //Same is signalized also via validity bit field
         //As per Z-axis Positive value means the front of the vehicle is turned to the left [/s].
         //degree is converted to centi degree(multiply by 100).
         if(oDeadReckoning_InfoStatus.drInfoValidity.bTurnrateZIsValid())
            rGpsData.f32TurnRate = static_cast<t_Float> ((oDeadReckoning_InfoStatus.drTurnrateZ) * 100);

#if 0
      //        Reason : Acceleration is not required by Carplay & ML, So, parameters are not populated
      //        Acceleration : sqrt(x2 + y2 + z2)
         if((oDeadReckoning_InfoStatus.drInfoValidity.bAccelerationXIsValid()) &&
                  (oDeadReckoning_InfoStatus.drInfoValidity.bAccelerationYIsValid()) &&
                  (oDeadReckoning_InfoStatus.drInfoValidity.bAccelerationZIsValid()))
         {
            rGpsData.s16Acceleration = static_cast<t_U16> (sqrt(pow(oDeadReckoning_InfoStatus.drAccelerationX,2)
                                       + pow(oDeadReckoning_InfoStatus.drAccelerationY,2)
                                       + pow(oDeadReckoning_InfoStatus.drAccelerationZ,2)));
         }
#endif

         //To avoid Gen4Compiler warnings converting double to float. Reason: ETG trace does not support for double values
         ETG_TRACE_USR4(("[PARAM]spi_tclSensorPVDataClientHandler::vOnDeadreckoningPositionStatus: Latitude = %f degree,"
                  " Longitude = %f degree, dHeading == %f degree, Speed = %f cm/s, Altitude = %f m, TurnRate = %f centidegree/s]",
                  static_cast<t_Float>(rGpsData.d64Latitude), static_cast<t_Float>(rGpsData.d64Longitude), static_cast<t_Float>(rGpsData.dHeading),
                  rGpsData.f32Speed, rGpsData.f32Altitude, rGpsData.f32TurnRate));

         ETG_TRACE_USR4(("[PARAM]spi_tclSensorPVDataClientHandler::vOnDeadreckoningPositionStatus: PosixTime = %s", std::to_string(rGpsData.PosixTime).c_str()));
         ETG_TRACE_USR4(("[PARAM]spi_tclSensorPVDataClientHandler::vOnDeadreckoningPositionStatus: u16ExactTime = %u", rGpsData.u16ExactTime));

         //! Forward data to subscriber
         m_rLocDataCallbacks.fvOnGpsData(rGpsData);
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclSensorPVDataClientHandler::vOnDeadreckoningPositionStatus: Message extraction failed! \n"));
   }
   m_lock.vUnlock();
}
/***************************************************************************************************************
** FUNCTION:   spi_tclSensorPVDataClientHandler::vRegisterCallbacks(trLocationDataCallbacks rLocDataCallbacks)
***************************************************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vRegisterCallbacks(trLocationDataCallbacks rLocDataCallbacks)
{
   m_rLocDataCallbacks = rLocDataCallbacks;
}


/***************************************************************************************************************
** FUNCTION:   spi_tclSensorPVDataClientHandler::vSendAutoRegisterPropertyRequest
***************************************************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vSendAutoRegisterPropertyRequest(t_U16 u16FunctionID)
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::vSendAutoRegisterPropertyRequest to entry thread u16FunctionID = %d", u16FunctionID));
   if (nullptr != m_poMainApp)
   {
      tLbAutoRegisterProperties oAutoRegisterProperties(m_poMainApp->u16GetAppId(), // Source AppID
               m_poMainApp->u16GetAppId(), // Target AppID
               e8_LOOPREGISTER_ID_SENSORPV_DATA, // RegisterID
               0, // CmdCounter
               CCA_C_U16_SRV_SMARTPHONEINTEGRATION,    // ServiceID
               SPI_C_U16_IFID_AUTOREGISTERPROPERTY, // Function ID
               AMT_C_U8_CCAMSG_OPCODE_STATUS //,      // Opcode
               );
      oAutoRegisterProperties.vSetWord(u16FunctionID);
      if (true == oAutoRegisterProperties.bIsValid())
      {
         if (AIL_EN_N_NO_ERROR != m_poMainApp->enPostMessage(&oAutoRegisterProperties, TRUE))
         {
            ETG_TRACE_ERR(("spi_tclSensorPVDataClientHandler::vSendAutoRegisterPropertyRequest() Loopback message posting failed! "));
         }
      } //if (true == oSPIStateMsg.bIsValid())
      else
      {
         ETG_TRACE_ERR(("spi_tclSensorPVDataClientHandler::vSendAutoRegisterPropertyRequest(): Loopback message creation failed! "));
      }
   }
}

/***************************************************************************************************************
** FUNCTION:   spi_tclSensorPVDataClientHandler::vSendAutoUnregisterPropertyRequest
***************************************************************************************************************/
t_Void spi_tclSensorPVDataClientHandler::vSendAutoUnregisterPropertyRequest(t_U16 u16FunctionID)
{
   ETG_TRACE_USR1(("spi_tclSensorPVDataClientHandler::vSendAutoUnregisterPropertyRequest to entry thread u16FunctionID = %d", u16FunctionID));
   if (nullptr != m_poMainApp)
   {
      tLbAutoUnregisterProperties oAutoUnregisterProperties(m_poMainApp->u16GetAppId(), // Source AppID
               m_poMainApp->u16GetAppId(), // Target AppID
               e8_LOOPREGISTER_ID_SENSORPV_DATA, // RegisterID
               0, // CmdCounter
               CCA_C_U16_SRV_SMARTPHONEINTEGRATION,    // ServiceID
               SPI_C_U16_IFID_AUTOUNREGISTERPROPERTY, // Function ID
               AMT_C_U8_CCAMSG_OPCODE_STATUS //,      // Opcode
               );
      oAutoUnregisterProperties.vSetWord(u16FunctionID);
      if (true == oAutoUnregisterProperties.bIsValid())
      {
         if (AIL_EN_N_NO_ERROR != m_poMainApp->enPostMessage(&oAutoUnregisterProperties, TRUE))
         {
            ETG_TRACE_ERR(("spi_tclSensorPVDataClientHandler::vSendAutoUnregisterPropertyRequest() Loopback message posting failed! "));
         }
      } //if (true == oSPIStateMsg.bIsValid())
      else
      {
         ETG_TRACE_ERR(("spi_tclSensorPVDataClientHandler::vSendAutoUnregisterPropertyRequest(): Loopback message creation failed! "));
      }
   }
}

// EOF
