/*!
 *******************************************************************************
 * \file              spi_tclMplayDeviceListHandler.cpp
 * \brief             Devicelist handling for apple devices reported by mediaplayer
 *******************************************************************************
 \verbatim
 PROJECT:        G3G
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPO Client handler class for MPlay service
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 20.09.2017 |  Pruthvi Thej Nagaraju      | Initial Version

 \endverbatim
 ******************************************************************************/


#include <algorithm>
#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclMplayDeviceListHandler.cpp.trc.h"
#endif

#include "ExtInterfacesTypes.h"
#include "spi_tclMplayDeviceListHandler.h"
#include "spi_tclDeviceIDDataIntf.h"
#include <regex>

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::spi_tclMplayDeviceListHandler
 ** ***************************************************************************/
spi_tclMplayDeviceListHandler::spi_tclMplayDeviceListHandler():m_fvDeviceListUpdate(NULL),m_bCPWEnabledStatus(false),m_bRoleSwitchTriggerStatus(false)
{
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::spi_tclMplayDeviceListHandler
 ** ***************************************************************************/
spi_tclMplayDeviceListHandler::~spi_tclMplayDeviceListHandler()
{
   m_bCPWEnabledStatus = false;
   vSetRoleSwitchTriggerStatus(false);
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::spi_tclMplayDeviceListHandler
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vRegisterCallbacks(const trDiscovererCbs& rfrDiscovererCbs)
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vRegisterCallbacks for discoverer \n"));
   m_rDiscovererCbs = rfrDiscovererCbs;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vRegisterCallbacks(const tfvDeviceListUpdate &rfrDeviceListUpdate)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vRegisterCallbacks for Devicelist update \n"));
   m_fvDeviceListUpdate = rfrDeviceListUpdate;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vUnregisterCallbacks()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vUnregisterCallbacks()
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vUnregisterCallbacks for discoverer \n"));
   m_rDiscovererCbs = trDiscovererCbs();
}
/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::u32GetMPDeviceTag
 ** ***************************************************************************/
t_U32 spi_tclMplayDeviceListHandler::u32GetMPDeviceTag(const t_U32 cou32DeviceID, tenDeviceConnectionType enConnType)
{
   t_U32 u32CarplayDeviceTag = 0;

   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(cou32DeviceID);
   if(m_rMplayDeviceList.end()!= itMap)
   {
      if(e8USB_CONNECTED == enConnType)
      {
         u32CarplayDeviceTag = itMap->second.USBDeviceTag;
      }
      else if(e8WIRELESS_CONNECTED == enConnType)
      {
         u32CarplayDeviceTag = itMap->second.BTDeviceTag;
      }
      //! @TODO WIFI connection in mediplayerlist not considered
   }
   m_oLockDeviceList.vUnlock();
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::u32GetMPDeviceTagfor DeviceID 0x%x is %d", cou32DeviceID, u32CarplayDeviceTag));

   return u32CarplayDeviceTag;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::u32GetSPIDeviceID
 ** ***************************************************************************/
t_U32 spi_tclMplayDeviceListHandler::u32GetSPIDeviceID(const t_U32 cou32DeviceTag, tenDeviceConnectionType enConnType)
{
   t_U32 u32SPIDeviceID = 0;
   m_oLockDeviceList.s16Lock();
   for(auto itMap = m_rMplayDeviceList.begin();itMap != m_rMplayDeviceList.end(); itMap++ )
   {
      if(e8USB_CONNECTED == enConnType)
      {
         if(cou32DeviceTag == itMap->second.USBDeviceTag)
         {
            u32SPIDeviceID = itMap->first;
            break;
         }
      }
      else if(e8WIRELESS_CONNECTED == enConnType)
      {
         if(cou32DeviceTag == itMap->second.BTDeviceTag)
         {
            u32SPIDeviceID = itMap->first;
            break;
         }
      }
   }
   m_oLockDeviceList.vUnlock();
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::u32GetSPIDeviceID for DeviceTag %d is 0x%x", cou32DeviceTag, u32SPIDeviceID));
   return u32SPIDeviceID;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bDeviceInfoChanged
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bDeviceInfoChanged(const t_U32 cou32DeviceID,
         const trDeviceInfo& corfrDeviceInfo)
{
   t_Bool bRetval = false;
   m_oLockDeviceList.s16Lock();
   auto itDeviceList = m_rMplayDeviceList.find(cou32DeviceID);
   if(itDeviceList != m_rMplayDeviceList.end())
   {
      ETG_TRACE_USR4(("itDeviceList->second.enDeviceConnectionStatus is %d", itDeviceList->second.enDeviceConnectionStatus));
      ETG_TRACE_USR4((" corfrDeviceInfo.enDeviceConnectionStatus is %d", corfrDeviceInfo.enDeviceConnectionStatus));

      ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enCarplaySupport is %d", itDeviceList->second.rProjectionCapability.enCarplaySupport));
      ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enCarplaySupport is %d", corfrDeviceInfo.rProjectionCapability.enCarplaySupport));

      ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enCarplayWirelessSupport is %d", itDeviceList->second.rProjectionCapability.enCarplayWirelessSupport));
      ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport is %d", corfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport));

      ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enMySPINSupport is %d", itDeviceList->second.rProjectionCapability.enMySPINSupport));
      ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enMySPINSupport is %d", corfrDeviceInfo.rProjectionCapability.enMySPINSupport));

      ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enCarlifeSupport is %d", itDeviceList->second.rProjectionCapability.enCarlifeSupport));
      ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enCarlifeSupport is %d", corfrDeviceInfo.rProjectionCapability.enCarlifeSupport));

      ETG_TRACE_USR4(("itDeviceList->second.enDeviceProfile is %d",itDeviceList->second.enDeviceProfile));
      ETG_TRACE_USR4(("corfrDeviceInfo.enDeviceProfile %d", corfrDeviceInfo.enDeviceProfile));
   }
   if(m_rMplayDeviceList.end() == itDeviceList)
   {
      bRetval = true;
   }
   else if((itDeviceList->second.enDeviceConnectionStatus != corfrDeviceInfo.enDeviceConnectionStatus)
            ||(itDeviceList->second.rProjectionCapability.enCarplaySupport != corfrDeviceInfo.rProjectionCapability.enCarplaySupport)
            ||(itDeviceList->second.rProjectionCapability.enCarplayWirelessSupport != corfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport)
            ||(itDeviceList->second.rProjectionCapability.enMySPINSupport != corfrDeviceInfo.rProjectionCapability.enMySPINSupport)
            ||(itDeviceList->second.rProjectionCapability.enCarlifeSupport != corfrDeviceInfo.rProjectionCapability.enCarlifeSupport)
            ||(itDeviceList->second.szDeviceName != corfrDeviceInfo.szDeviceName)
            ||(itDeviceList->second.szSerialNumber!= corfrDeviceInfo.szSerialNumber)
            ||(itDeviceList->second.enDeviceProfile!= corfrDeviceInfo.enDeviceProfile))
   {
      bRetval = true;
   }
   else
   {
      bRetval = false;
   }
   m_oLockDeviceList.vUnlock();

   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bDeviceInfoChanged :  %d", ETG_ENUM(BOOL,bRetval)));
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vEraseFromDevicelist
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vEraseFromDevicelist(const t_U32 cou32DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vEraseFromDevicelist: Erasing 0x%x", cou32DeviceHandle));
   m_oLockDeviceList.s16Lock();
   auto itMapDeviceList = m_rMplayDeviceList.find(cou32DeviceHandle);
   if (m_rMplayDeviceList.end() != itMapDeviceList)
   {
      m_rMplayDeviceList.erase(cou32DeviceHandle);
   }
   m_oLockDeviceList.vUnlock();
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vAddDeviceToList()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vAddDeviceToList(const trMPDeviceInfo &corfrDeviceInfo)
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vAddDeviceToList: Adding device 0x%x", corfrDeviceInfo.u32DeviceHandle));
   // Push back the connected device to the list
   m_oLockDeviceList.s16Lock();
   m_rMplayDeviceList[corfrDeviceInfo.u32DeviceHandle] = (corfrDeviceInfo);
   m_oLockDeviceList.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vHandleDeviceConnections
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vHandleDeviceConnections(const t_String& szDeviceName,
         const t_String& szBTMACAddress, const t_String& szSerialNumber, mplay_fi_tcl_DiPODeviceInfoItem& oDevInfo)
{
   trMPDeviceInfo rDeviceInfo;
   t_Bool bIsValidDisconnection = false;
   t_Bool bDeviceInfoInValid = false ;
   vPopulateDeviceDetails(oDevInfo, rDeviceInfo, szDeviceName, szBTMACAddress, szSerialNumber);
   
   //Check BT Media devices info
   if(bIsValidBTDevice(oDevInfo,szBTMACAddress))
   {
      //Send the device update with active media info
      vUpdateMediaDeviceInfo(rDeviceInfo.u32DeviceHandle,szBTMACAddress,oDevInfo.bDeviceActiveSource,oDevInfo.bDeviceConnected);
   }
   
   //If there is a Disconnection Update from MP, Need not Validate Roleswitch, Valid BT Device and DeviceInfo Invalidation
   if(true == bIsDisconnectionValid(oDevInfo))
   {
      ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vHandleDeviceConnections Valid DISCONNECTION, Send Disconnection Update"));
      bIsValidDisconnection = true;
   }
   else
   {
     rDeviceInfo.bIsWaitingForRoleSwitch = bIsDeviceWaitingForRoleswitch(oDevInfo);
     if(true == bIsDeviceWaitingForRoleswitch(oDevInfo))
     {
        rDeviceInfo.rConnectionStatus.enUSBConnectionStatus = e8DEV_CONNECTED;
        vAddDeviceToList(rDeviceInfo);
     }

     bDeviceInfoInValid = bIsDeviceInfoInvalid(oDevInfo, szBTMACAddress, szSerialNumber);
   }
   if ((false == bDeviceInfoInValid) || (true == bIsValidDisconnection))
   {
      t_Bool bDeviceConnectedOverUSB = bIsDeviceConnectedForUSB(oDevInfo);
      t_Bool bDeviceConnectediAP2OverBT = bIsDeviceIAP2BTConnected(oDevInfo);
      t_Bool bDeviceConnectedOverBTWithCPWFeasible = bIsDeviceConnectedOverBTWithCPWFeasible(oDevInfo);
      if ((true == bDeviceConnectedOverUSB) || (true == bDeviceConnectedOverBTWithCPWFeasible))
      {
         if(false == szDeviceName.empty())
         {
             rDeviceInfo.enDeviceConnectionStatus = e8DEV_CONNECTED;
             rDeviceInfo.rConnectionStatus.enUSBConnectionStatus =
                    (true == bDeviceConnectedOverUSB) ?
                             e8DEV_CONNECTED : enGetUSBConnectionStatus(rDeviceInfo.u32DeviceHandle);
             rDeviceInfo.rConnectionStatus.enBTConnectionStatus =
                    (true == bDeviceConnectediAP2OverBT) ?
                             e8DEV_CONNECTED : enGetBTConnectionStatus(rDeviceInfo.u32DeviceHandle);
             rDeviceInfo.rConnectionStatus.enWiFiConnectionStatus =
                    (true == bIsDeviceConnectedOverWiFi(oDevInfo)) ?
                             e8DEV_CONNECTED : enGetWiFiConnectionStatus(rDeviceInfo.u32DeviceHandle);
             t_Bool bInfoChanged = bDeviceInfoChanged(rDeviceInfo.u32DeviceHandle, rDeviceInfo);
             vAddDeviceToList(rDeviceInfo);
             if (true == bInfoChanged)
             {
                  vSendDeviceConnectionUpdate(rDeviceInfo);
             }
          }
      }
      else if ((true == bIsDeviceDisconnectedOverUSB(oDevInfo)) && (e8DEV_CONNECTED == enGetUSBConnectionStatus(rDeviceInfo.u32DeviceHandle)) && (true == bIsValidDisconnection))
      {
         ETG_TRACE_USR2(("[DESC] Device disconnected over USB 0x%x\n", rDeviceInfo.u32DeviceHandle));
         rDeviceInfo.enDeviceConnectionStatus = e8DEV_NOT_CONNECTED;
         vEraseFromDevicelist(rDeviceInfo.u32DeviceHandle);
         vSendDeviceDisconnectionUpdate(rDeviceInfo, e8_DISCOVERER_TYPE_USB, e8USB_CONNECTED);
      }
      else if(true == bIsDeviceDisconnectedOverWiFi(oDevInfo) && (true == bIsDeviceIAP2OverWiFiConnected(rDeviceInfo.u32DeviceHandle)))
      {
         ETG_TRACE_USR2(("[DESC] Device disconnected over WiFi 0x%x\n", rDeviceInfo.u32DeviceHandle));
         rDeviceInfo.enDeviceConnectionStatus = e8DEV_NOT_CONNECTED;
         vEraseFromDevicelist(rDeviceInfo.u32DeviceHandle);
         vSendDeviceDisconnectionUpdate(rDeviceInfo, e8_DISCOVERER_TYPE_WIFI, e8WIRELESS_CONNECTED);
      }
      else if ((true == bIsDeviceDisconnectedOverBT(oDevInfo)) && (e8DEV_CONNECTED == enGetBTConnectionStatus(rDeviceInfo.u32DeviceHandle)))
      {
         ETG_TRACE_USR2(("[DESC] Device disconnected over BT 0x%x\n", rDeviceInfo.u32DeviceHandle));
         vSetDeviceConnectionStatus(rDeviceInfo.u32DeviceHandle, e8_BLUETOOTH_CONNECTION, e8DEV_NOT_CONNECTED);
         t_U32 u32DeviceHandle = rDeviceInfo.u32DeviceHandle;
         if ( (false == bIsDeviceConnected(u32DeviceHandle) ) && (true != bIsWaitingForRoleSwitch(u32DeviceHandle)))
         {
            vEraseFromDevicelist(u32DeviceHandle);
            vSendDeviceDisconnectionUpdate(rDeviceInfo, e8_DISCOVERER_TYPE_BT, e8WIRELESS_CONNECTED);
         }
      }
      else if((true == bDeviceConnectediAP2OverBT) && (false == bDeviceConnectedOverBTWithCPWFeasible))
      {
         ETG_TRACE_USR1(("[DESC] Device connected over BT and CPW Not Feasible"));
         t_U32 u32DeviceHandle = rDeviceInfo.u32DeviceHandle;
         if ( (false == bIsDeviceConnected(u32DeviceHandle) ) && (true != bIsWaitingForRoleSwitch(u32DeviceHandle)))
         {
            vSendDeviceDisconnectionUpdate(rDeviceInfo, e8_DISCOVERER_TYPE_BT, e8WIRELESS_CONNECTED);
         }
      }
   }

   if(true == bDeviceInfoInValid)
   {
      //! this has to be done for Connection type BLUETOOTH and Device type iPhone
      if((false == bIsWaitingForRoleSwitch(rDeviceInfo.u32DeviceHandle)) && (mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_IPHONE == oDevInfo.e8DeviceType.enType)
                                                  &&
         (true == bIsDeviceIAP2BTConnected(oDevInfo) && (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI_NOT_FEASIBLE == oDevInfo.e8DiPOCaps.enType)))
      {
          tenDeviceConnectionStatus enDeviceConnectionStatus = e8DEV_NOT_CONNECTED;

          m_oLockDeviceList.s16Lock();
          auto itMap = m_rMplayDeviceList.find(rDeviceInfo.u32DeviceHandle);
          if (m_rMplayDeviceList.end() != itMap)
          {
             enDeviceConnectionStatus = (m_rMplayDeviceList[rDeviceInfo.u32DeviceHandle]).enDeviceConnectionStatus;
          }
          m_oLockDeviceList.vUnlock();

          if(e8DEV_NOT_CONNECTED == enDeviceConnectionStatus)
          {
             rDeviceInfo.enDeviceConnectionStatus = e8DEV_CONNECTED;
             m_oLockDeviceList.s16Lock();
             auto itMap = m_rMplayDeviceList.find(rDeviceInfo.u32DeviceHandle);
             if (m_rMplayDeviceList.end() != itMap)
             {
                (m_rMplayDeviceList[rDeviceInfo.u32DeviceHandle]).enDeviceConnectionStatus = rDeviceInfo.enDeviceConnectionStatus;
             }
             m_oLockDeviceList.vUnlock();
             vSendDeviceConnectionUpdate(rDeviceInfo);
          }
      }
   }
   if((mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_IPHONE == oDevInfo.e8DeviceType.enType))
   {
      vUpdateDeviceList(rDeviceInfo, bDeviceInfoInValid);
   }

}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vReceiveMplayDeviceList
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vOnMplayDeviceListUpdate(mplay_fi_tcl_DiPODeviceInfo& rfrMplayDeviceList)
{
   ETG_TRACE_USR2(("Number of Devices : %d", rfrMplayDeviceList.oItems.size()));
   for (size_t u8Index = 0; u8Index < rfrMplayDeviceList.oItems.size(); u8Index++)
   {
      mplay_fi_tcl_DiPODeviceInfoItem &oDevInfo = rfrMplayDeviceList.oItems[u8Index];
      ETG_TRACE_USR2(("Device Index = %d", u8Index));
      t_String szDeviceName, szSerialNumber, szBTMACAddress;
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oDevInfo.sDeviceName, mplay_fi_tclString::FI_EN_UTF8, szDeviceName);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oDevInfo.sDeviceUSBSerialNumber, mplay_fi_tclString::FI_EN_UTF8, szSerialNumber);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oDevInfo.sDeviceBTMacAddress, mplay_fi_tclString::FI_EN_UTF8, szBTMACAddress);
      std::transform(szBTMACAddress.begin(), szBTMACAddress.end(), szBTMACAddress.begin(), ::toupper);
      vDisplayDeviceInfoList(oDevInfo, szDeviceName, szBTMACAddress, szSerialNumber);
      vHandleDeviceConnections(szDeviceName, szBTMACAddress, szSerialNumber, oDevInfo);
      ETG_TRACE_USR4(("**************************************************"));
   }           // for loop end
   
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bValidateDeviceInformation
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceInfoInvalid(
         mplay_fi_tcl_DiPODeviceInfoItem& rfrMplayDeviceInfoItem,
         const t_String szDeviceBTAddress, const t_String coszSerialNo)
{
   t_Bool bDeviceInfoInValid = ((true == coszSerialNo.empty())
            || ("invalid" == coszSerialNo)
            || (mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_IPHONE != rfrMplayDeviceInfoItem.e8DeviceType.enType));

   if(false == bDeviceInfoInValid)
   {
      if((true == m_bCPWEnabledStatus) && (true == szDeviceBTAddress.empty()))
      {
         if((mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NATIVE_TRANSPORT == rfrMplayDeviceInfoItem.e8DiPOCaps.enType)||
                  (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARLIFE == rfrMplayDeviceInfoItem.e8DiPOCaps.enType)||
                  (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_FEASIBLE == rfrMplayDeviceInfoItem.e8DiPOCaps.enType) ||
                  (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_HOSTMODE_FEASIBLE == rfrMplayDeviceInfoItem.e8DiPOCaps.enType))

         {
            ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bDeviceInfoInValid MYSPIN/CARLIFE"));
            bDeviceInfoInValid = false;
         }
         else
         {
            bDeviceInfoInValid = true;
         }
      }
   }

   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bValidateDeviceInformation: Is Device info invalid :%d", ETG_ENUM(BOOL,
            bDeviceInfoInValid)));
   return bDeviceInfoInValid;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vSendDeviceConnectionUpdate
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vSendDeviceConnectionUpdate(const trMPDeviceInfo& corfrDeviceinfo)
{
   ETG_TRACE_USR1(("[FUNC]: spi_tclMplayDeviceListHandler::vSendDeviceConnectionUpdate: DeviceID 0x%x", corfrDeviceinfo.u32DeviceHandle));
   ETG_TRACE_USR1(("[FUNC]: spi_tclMplayDeviceListHandler::vSendDeviceConnectionUpdate: Device Profile %d", ETG_ENUM(USB_PROFILE,corfrDeviceinfo.enDeviceProfile)));

   if (NULL != m_rDiscovererCbs.fvDeviceConnectionCb)
   {
      (m_rDiscovererCbs.fvDeviceConnectionCb)(corfrDeviceinfo, e8_DISCOVERER_TYPE_USB);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vSendDeviceDisconnectionUpdate
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vSendDeviceDisconnectionUpdate(const trDeviceInfo& corfrDeviceinfo, tenDiscovererType enDiscovererType, tenDeviceConnectionType enConnType)
{
   ETG_TRACE_USR1(("[FUNC]: spi_tclMplayDeviceListHandler::vSendDeviceDisconnectionUpdate: DeviceID 0x%x Connection Type = %d", corfrDeviceinfo.u32DeviceHandle,
            ETG_ENUM(CONNECTION_TYPE,enConnType)));
   ETG_TRACE_USR2(("Device 0x%x disconnected over %d ", corfrDeviceinfo.u32DeviceHandle, ETG_ENUM(CONNECTION_TYPE,
            enConnType)));

   ETG_TRACE_USR4(("spi_tclMplayDeviceListHandler::vSendDeviceDisconnectionUpdate CarplayWirelessSupport is %d", ETG_ENUM(SPI_SUPPORT, corfrDeviceinfo.rProjectionCapability.enCarplayWirelessSupport)));

   if ((NULL != m_rDiscovererCbs.fvDeviceDisconnectionCb))
   {
      (m_rDiscovererCbs.fvDeviceDisconnectionCb)(corfrDeviceinfo.u32DeviceHandle, enDiscovererType, enConnType, corfrDeviceinfo.rProjectionCapability.enCarplayWirelessSupport);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vDisplayDeviceInfoList
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vDisplayDeviceInfoList(
         const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem, const t_String coszDeviceName,
         const t_String szDeviceBTAddress, const t_String coszSerialNo)
{
   ETG_TRACE_USR4(("**************************************************"));
   ETG_TRACE_USR2(("DeviceID : %d", corfrMplayDeviceInfoItem.u8DeviceTag));
   ETG_TRACE_USR2(("DeviceName : %s", coszDeviceName.c_str()));
   ETG_TRACE_USR2(("DeviceType : %d", ETG_ENUM(MPLAY_DEVICETYPE, corfrMplayDeviceInfoItem.e8DeviceType.enType)));
   ETG_TRACE_USR2(("Connected Flag : %d", corfrMplayDeviceInfoItem.bDeviceConnected));
   ETG_TRACE_USR2(("Connection type : %d", ETG_ENUM(MPLAY_CONNECTIONTYPE,
            corfrMplayDeviceInfoItem.e8ConnectionType.enType)));
   ETG_TRACE_USR2(("Device state : %d", ETG_ENUM(MPLAY_DEVICESTATE, corfrMplayDeviceInfoItem.e8DeviceState.enType)));
   ETG_TRACE_USR2(("Capability type : %d", ETG_ENUM(MPLAY_DIPOCAPS, corfrMplayDeviceInfoItem.e8DiPOCaps.enType)));
   ETG_TRACE_USR2(("Connection State : %d", ETG_ENUM(MPLAY_CONNECTIONSTATE,
            corfrMplayDeviceInfoItem.e8ConnectionState.enType)));
   ETG_TRACE_USR2(("Serial Number : %s", coszSerialNo.c_str()));
   ETG_TRACE_USR4(("BT MAC Address: %s", szDeviceBTAddress.c_str()));
   ETG_TRACE_USR4(("Is Active Audio source: %d", corfrMplayDeviceInfoItem.bDeviceActiveSource));
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceConnectedForUSB
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceConnectedForUSB(const mplay_fi_tcl_DiPODeviceInfoItem& corfrMplayDeviceInfoItem)
{

   /*
   1.For USB devices, DeviceState is set to INITIALIZING when the MediaPlayer is waiting for SPI to perform roleswitch. 
   e8DiPOCaps shall be set to NONE in this cae. Even bDeviceConnected  is ignored.
   2. DeviceState is set to CONNECTED and DiPOCaps changes to CP_FEASIBLE or CP, after Roleswitch response sent by SPI.
   3. DiPOCaps update not set to NONE is considered as valid USB connection update.-> This must not be done, if the roleswitch request is not sent to MP.
   */
   t_Bool bIsDeviceConnected = (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_USB
            == corfrMplayDeviceInfoItem.e8ConnectionType.enType)
            && ((true == corfrMplayDeviceInfoItem.bDeviceConnected)
                     || (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NONE != corfrMplayDeviceInfoItem.e8DiPOCaps.enType));

   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceConnectedForUSB : %d", ETG_ENUM(BOOL,
            bIsDeviceConnected)));

   return bIsDeviceConnected;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverBTWithCPWFeasible
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverBTWithCPWFeasible(const mplay_fi_tcl_DiPODeviceInfoItem& corfrMplayDeviceInfoItem)
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverBTWithCPWFeasible Entered"));

   //Initially Considering not Feasible
   t_Bool bIsCarplayWirelessFeasible = false;
   //checking for Wifi Feasiblity received from MP
   if(bIsDeviceIAP2BTConnected(corfrMplayDeviceInfoItem))
   {
      bIsCarplayWirelessFeasible = (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI_FEASIBLE
      == corfrMplayDeviceInfoItem.e8DiPOCaps.enType);
   }
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverBTWithCPWFeasible - CPW_WIF_Feasible %d",ETG_ENUM(BOOL, bIsCarplayWirelessFeasible)));
   return bIsCarplayWirelessFeasible;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceIAP2BTConnected
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceIAP2BTConnected(const
         mplay_fi_tcl_DiPODeviceInfoItem& corfrMplayDeviceInfoItem)
{
   t_Bool bIsiAP2BTConnected = (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_BLUETOOTH
            == corfrMplayDeviceInfoItem.e8ConnectionType.enType)
            && (mplay_fi_tcl_e8_DiPOConnectionState::FI_EN_E8CS_ATTACHED
                     == corfrMplayDeviceInfoItem.e8ConnectionState.enType);
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bDeviceConnectedForWirelessOverBT : %d", ETG_ENUM(BOOL,
     bIsiAP2BTConnected)));
   return bIsiAP2BTConnected;
}


/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverWiFi
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverWiFi(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem)
{
   t_Bool bDeviceConnectedOverWiFi = (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_WIFI
            == corfrMplayDeviceInfoItem.e8ConnectionType.enType)
                     && (mplay_fi_tcl_e8_DiPOConnectionState::FI_EN_E8CS_ATTACHED
                              == corfrMplayDeviceInfoItem.e8ConnectionState.enType);
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverWiFi : %d", ETG_ENUM(BOOL,
            bDeviceConnectedOverWiFi)));
   return bDeviceConnectedOverWiFi;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverUSB
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverUSB(
         const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem)
{
   //! indicates device connection status when role switch is in progress but authentication is pending.
   //! enDeviceState checks if the device is disconnected during role switch
   tenDeviceConnectionStatus enDeviceState =
            (mplay_fi_tcl_e8_DiPODeviceStatus::FI_EN_E8DS_INITIALIZING == corfrMplayDeviceInfoItem.e8DeviceState.enType) ?
                     e8DEV_CONNECTED : e8DEV_NOT_CONNECTED;
   t_Bool bDeviceDisconnected = ((false == bIsDeviceConnectedForUSB(corfrMplayDeviceInfoItem))
            && ((mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_USB
                     == corfrMplayDeviceInfoItem.e8ConnectionType.enType))&& (e8DEV_NOT_CONNECTED == enDeviceState));
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverUSB : %d", ETG_ENUM(BOOL,
            bDeviceDisconnected)));
   return bDeviceDisconnected;
}


/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverBT
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverBT(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem)
{
   //@NCG3D-85233
   /*iAP2BT connection may not be disconnected, if the BT component keeps the SPP connection when User says "No" to use CPW.
   In this case, DiPOCaps is always set to "DIPO_CAP_NONE" and subsequent USB connection of the same device will be treated as "Change in device list" internally
   and the auto selection is not started. Hence treat updates with DiPOCaps as DIPO_CAP_NONE as disconnected even if the connectedState is "ATTACHED".*/
   t_Bool bDisconnectedOverBT = (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_BLUETOOTH == corfrMplayDeviceInfoItem.e8ConnectionType.enType) && 
   ((mplay_fi_tcl_e8_DiPOConnectionState::FI_EN_E8CS_ATTACHED != corfrMplayDeviceInfoItem.e8ConnectionState.enType) ||
   (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NONE == corfrMplayDeviceInfoItem.e8DiPOCaps.enType));

   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverBT : %d", ETG_ENUM(BOOL,
            bDisconnectedOverBT)));
   return bDisconnectedOverBT;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverWiFi
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverWiFi(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem)
{
   t_Bool bDisconnectedOverWiFi = ((mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_WIFI
            == corfrMplayDeviceInfoItem.e8ConnectionType.enType)
            && (mplay_fi_tcl_e8_DiPOConnectionState::FI_EN_E8CS_ATTACHED
                     != corfrMplayDeviceInfoItem.e8ConnectionState.enType));
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverWiFi : %d", ETG_ENUM(BOOL,
            bDisconnectedOverWiFi)));
   return bDisconnectedOverWiFi;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vPopulateDeviceCapabilities
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vPopulateDeviceCapabilities(
         const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem, trDeviceInfo& rfrDeviceinfo)
{
   rfrDeviceinfo.rProjectionCapability.enDeviceType = e8_APPLE_DEVICE;
   rfrDeviceinfo.rProjectionCapability.enAndroidAutoSupport = e8SPI_NOTSUPPORTED;
   rfrDeviceinfo.rProjectionCapability.enMirrorlinkSupport = e8SPI_NOTSUPPORTED;
   rfrDeviceinfo.rProjectionCapability.enCarplaySupport =
            ((corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_FEASIBLE)
                     || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType
                              == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY)
                     || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType
                              == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_NATIVE_TRANSPORT) ||
             (corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI_FEASIBLE)
            || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType
                     == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI)) ?
                     e8SPI_SUPPORTED : e8SPI_SUPPORT_UNKNOWN;
   rfrDeviceinfo.rProjectionCapability.enCarplayWirelessSupport =
            ((corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI_FEASIBLE)
            || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType
                     == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI))?
                              e8SPI_SUPPORTED : e8SPI_SUPPORT_UNKNOWN;

   if((corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI_NOT_FEASIBLE)
            || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_HOSTMODE_FEASIBLE))
   {
      rfrDeviceinfo.rProjectionCapability.enCarplayWirelessSupport = e8SPI_NOTSUPPORTED;
      rfrDeviceinfo.rProjectionCapability.enCarplaySupport = e8SPI_NOTSUPPORTED;
   }

   ETG_TRACE_USR1(("  rfrDeviceinfo.rProjectionCapability.enCarplayWirelessSupport %d",   rfrDeviceinfo.rProjectionCapability.enCarplayWirelessSupport));
   rfrDeviceinfo.rProjectionCapability.enMySPINSupport =
            ((corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_HOSTMODE_FEASIBLE)
                     || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType
                              == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_NATIVE_TRANSPORT)
                     || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType
                              == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NATIVE_TRANSPORT)) ?
                     e8SPI_SUPPORTED : e8SPI_SUPPORT_UNKNOWN;

   rfrDeviceinfo.rProjectionCapability.enCarlifeSupport =((corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_HOSTMODE_FEASIBLE)
                                 || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType
                                          == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_NATIVE_TRANSPORT)
                                 || (corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_FEASIBLE)
                                 ||(corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARLIFE)) ?
                                 e8SPI_SUPPORTED : e8SPI_SUPPORT_UNKNOWN;

   rfrDeviceinfo.rProjectionCapability.enUSBPortType =
            (corfrMplayDeviceInfoItem.e8DiPOCaps.enType != mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NONE) ?
                     e8_PORT_TYPE_OTG : e8_PORT_TYPE_NON_OTG;

   ETG_TRACE_USR2(("vPopulateDeviceCapabilities USB Port Type : %d \n", ETG_ENUM(USB_PORT_TYPE,
            rfrDeviceinfo.rProjectionCapability.enUSBPortType)));

   if (corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY)
   {
      rfrDeviceinfo.enDeviceProfile = e8_PROFILE_HOSTMODE_CARPLAY;
   }
   else if (true == bIsDeviceIAP2BTConnected(corfrMplayDeviceInfoItem))
   {
     //checking for the feasibility and updating the Device profile
     if(false == bIsDeviceConnectedOverBTWithCPWFeasible(corfrMplayDeviceInfoItem))
        rfrDeviceinfo.enDeviceProfile = e8_PROFILE_IAP2BT_CPW_NOT_FEASIBLE;
      else
        rfrDeviceinfo.enDeviceProfile = e8_PROFILE_IAP2BT_CPW_FEASIBLE;
   }
   else if (corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NATIVE_TRANSPORT)
   {
      rfrDeviceinfo.enDeviceProfile = e8_PROFILE_NATIVETRANSPORT_MYSPIN;

   }else if(corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARLIFE)
   {
      rfrDeviceinfo.enDeviceProfile = e8_PROFILE_NATIVETRANSPORT_CARLIFE;
   }
   else
   {
      rfrDeviceinfo.enDeviceProfile = e8_PROFILE_DEFAULT;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vPopulateDeviceDetails
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vPopulateDeviceDetails(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem, trMPDeviceInfo &rfrDeviceinfo,
         const t_String coszDeviceName, const t_String szDeviceBTAddress, const t_String coszSerialNo)
{
   spi_tclDeviceIDDataIntf oDeviceDataIntf;

   //! Clean the Serial Number(if it contains unwanted character '-') Before it is being copied the structure that's used
   t_String szSerialNumber;
   std::regex rPattern("-");
   szSerialNumber = std::regex_replace(coszSerialNo, rPattern , "");

   if((true == coszSerialNo.empty()) || ("invalid" == coszSerialNo))
   {
      rfrDeviceinfo.u32DeviceHandle = oDeviceDataIntf.u32GetDeviceIDFromBTAddress(szDeviceBTAddress.c_str());
   }
   else
   {
      rfrDeviceinfo.u32DeviceHandle = oDeviceDataIntf.u32GetDeviceIDFromSerialNo(szSerialNumber.c_str());
   }

   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vPopulateDeviceDetails : Device handle is %d", rfrDeviceinfo.u32DeviceHandle));

   rfrDeviceinfo.szDeviceName = coszDeviceName;
   rfrDeviceinfo.szSerialNumber = szSerialNumber;
   rfrDeviceinfo.szBTAddress = szDeviceBTAddress;

   if(corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NATIVE_TRANSPORT)
   {
      rfrDeviceinfo.enDeviceCategory = e8DEV_TYPE_MYSPIN;

   }else if (corfrMplayDeviceInfoItem.e8DiPOCaps.enType == mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARLIFE)
   {
      rfrDeviceinfo.enDeviceCategory = e8DEV_TYPE_CARLIFE;

   }else
   {
      rfrDeviceinfo.enDeviceCategory = e8DEV_TYPE_UNKNOWN;
   }

   rfrDeviceinfo.enDeviceConnectionType = (true == bIsDeviceIAP2BTConnected(corfrMplayDeviceInfoItem)) ? e8WIRELESS_CONNECTED : e8USB_CONNECTED;

   vPopulateDeviceCapabilities(corfrMplayDeviceInfoItem, rfrDeviceinfo);
   rfrDeviceinfo.rVersionInfo.u32MajorVersion = 0;
   rfrDeviceinfo.rVersionInfo.u32MinorVersion = 0;
   rfrDeviceinfo.rVersionInfo.u32PatchVersion = 0;
   rfrDeviceinfo.bIsIAP2OverWiFiConnected = ((mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_CARPLAY_WIFI == corfrMplayDeviceInfoItem.e8DiPOCaps.enType)
            && (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_WIFI == corfrMplayDeviceInfoItem.e8ConnectionType.enType));

   if (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_USB == corfrMplayDeviceInfoItem.e8ConnectionType.enType)
   {
      rfrDeviceinfo.USBDeviceTag = corfrMplayDeviceInfoItem.u8DeviceTag;
   }
   else if (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_BLUETOOTH == corfrMplayDeviceInfoItem.e8ConnectionType.enType)
   {
      rfrDeviceinfo.BTDeviceTag = corfrMplayDeviceInfoItem.u8DeviceTag;

      if(mplay_fi_tcl_e8_DiPOConnectionState::FI_EN_E8CS_ATTACHED == corfrMplayDeviceInfoItem.e8ConnectionState.enType)
      {
         rfrDeviceinfo.rConnectionStatus.enBTConnectionStatus = e8DEV_CONNECTED;
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::enGetUSBConnectionStatus
 ** ***************************************************************************/
tenDeviceConnectionStatus spi_tclMplayDeviceListHandler::enGetUSBConnectionStatus(const t_U32 cou32DeviceHandle)
{
   tenDeviceConnectionStatus enDevConnection = e8DEV_NOT_CONNECTED;
   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(cou32DeviceHandle);
   if (m_rMplayDeviceList.end() != itMap)
   {
      enDevConnection = itMap->second.rConnectionStatus.enUSBConnectionStatus;
   }
   m_oLockDeviceList.vUnlock();
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::enGetUSBConnectionStatus : %d", ETG_ENUM(DEVICE_CONNECTION,
            enDevConnection)));
   return enDevConnection;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::enGetBTConnectionStatus
 ** ***************************************************************************/
tenDeviceConnectionStatus spi_tclMplayDeviceListHandler::enGetBTConnectionStatus(const t_U32 cou32DeviceHandle)
{
   tenDeviceConnectionStatus enDevConnection = e8DEV_NOT_CONNECTED;
   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(cou32DeviceHandle);
   if (m_rMplayDeviceList.end() != itMap)
   {
      enDevConnection = itMap->second.rConnectionStatus.enBTConnectionStatus;
   }
   m_oLockDeviceList.vUnlock();
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::enGetBTConnectionStatus : %d", ETG_ENUM(DEVICE_CONNECTION,
            enDevConnection)));
   return enDevConnection;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::enGetWiFiConnectionStatus
 ** ***************************************************************************/
tenDeviceConnectionStatus spi_tclMplayDeviceListHandler::enGetWiFiConnectionStatus(const t_U32 cou32DeviceHandle)
{
   tenDeviceConnectionStatus enDevConnection = e8DEV_NOT_CONNECTED;
   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(cou32DeviceHandle);
   if (m_rMplayDeviceList.end() != itMap)
   {
      enDevConnection = itMap->second.rConnectionStatus.enWiFiConnectionStatus;
   }
   m_oLockDeviceList.vUnlock();
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::enGetWiFiConnectionStatus : %d", ETG_ENUM(DEVICE_CONNECTION,
            enDevConnection)));
   return enDevConnection;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::bIsDeviceConnected
 ** ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceConnected(const t_U32 cou32DeviceHandle)
{
   t_Bool bIsDeviceConnected = false;
   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(cou32DeviceHandle);
   if (m_rMplayDeviceList.end() != itMap)
   {
      bIsDeviceConnected = (e8DEV_CONNECTED == itMap->second.rConnectionStatus.enUSBConnectionStatus) ||
               (e8DEV_CONNECTED == itMap->second.rConnectionStatus.enBTConnectionStatus) ||
               (e8DEV_CONNECTED == itMap->second.rConnectionStatus.enWiFiConnectionStatus);
   }
   m_oLockDeviceList.vUnlock();
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceConnected : %d", ETG_ENUM(BOOL,
            bIsDeviceConnected)));
   return bIsDeviceConnected;
}


/***************************************************************************
 ** FUNCTION:  spi_tclMplayDeviceListHandler::vUpdateDeviceList
 ** ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vUpdateDeviceList(trDeviceInfo &rfrDeviceInfo, t_Bool bDeviceInfoInValid)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::vUpdateDeviceList : for device 0x%x", rfrDeviceInfo.u32DeviceHandle));
   if(NULL != m_fvDeviceListUpdate)
   {
      ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler check"));
      (m_fvDeviceListUpdate)(rfrDeviceInfo.u32DeviceHandle, rfrDeviceInfo);
   }
   
         //! Updating the MPlayDeviceList only for the ProjectionCapability and Device Profile
      m_oLockDeviceList.s16Lock();
      auto itDeviceList = m_rMplayDeviceList.find(rfrDeviceInfo.u32DeviceHandle);
      if(itDeviceList != m_rMplayDeviceList.end())
      {
         ETG_TRACE_USR4(("BEFORE"));
         ETG_TRACE_USR4(("itDeviceList->second.enDeviceConnectionStatus is %d", itDeviceList->second.enDeviceConnectionStatus));
         ETG_TRACE_USR4((" corfrDeviceInfo.enDeviceConnectionStatus is %d", rfrDeviceInfo.enDeviceConnectionStatus));

         ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enCarplaySupport is %d", itDeviceList->second.rProjectionCapability.enCarplaySupport));
         ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enCarplaySupport is %d", rfrDeviceInfo.rProjectionCapability.enCarplaySupport));

         ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enCarplayWirelessSupport is %d", itDeviceList->second.rProjectionCapability.enCarplayWirelessSupport));
         ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport is %d", rfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport));

         ETG_TRACE_USR4(("itDeviceList->second.enDeviceProfile is %d",itDeviceList->second.enDeviceProfile));
         ETG_TRACE_USR4(("corfrDeviceInfo.enDeviceProfile %d", rfrDeviceInfo.enDeviceProfile));
         ETG_TRACE_USR4(("AFTER"));

         if(false == bDeviceInfoInValid)
         {
            (itDeviceList->second).rProjectionCapability.enCarplaySupport =  rfrDeviceInfo.rProjectionCapability.enCarplaySupport;
            (itDeviceList->second).rProjectionCapability.enCarplayWirelessSupport =  rfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport;
            (itDeviceList->second).rProjectionCapability.enMySPINSupport =  rfrDeviceInfo.rProjectionCapability.enMySPINSupport;
            (itDeviceList->second).rProjectionCapability.enCarlifeSupport =  rfrDeviceInfo.rProjectionCapability.enCarlifeSupport;
            (itDeviceList->second).enDeviceProfile =   rfrDeviceInfo.enDeviceProfile;
         }

         ETG_TRACE_USR4(("itDeviceList->second.enDeviceConnectionStatus is %d", itDeviceList->second.enDeviceConnectionStatus));
         ETG_TRACE_USR4((" corfrDeviceInfo.enDeviceConnectionStatus is %d", rfrDeviceInfo.enDeviceConnectionStatus));

         ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enCarplaySupport is %d", itDeviceList->second.rProjectionCapability.enCarplaySupport));
         ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enCarplaySupport is %d", rfrDeviceInfo.rProjectionCapability.enCarplaySupport));

         ETG_TRACE_USR4(("itDeviceList->second.rProjectionCapability.enCarplayWirelessSupport is %d", itDeviceList->second.rProjectionCapability.enCarplayWirelessSupport));
         ETG_TRACE_USR4((" corfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport is %d", rfrDeviceInfo.rProjectionCapability.enCarplayWirelessSupport));

         ETG_TRACE_USR4(("itDeviceList->second.enDeviceProfile is %d",itDeviceList->second.enDeviceProfile));
         ETG_TRACE_USR4(("corfrDeviceInfo.enDeviceProfile %d", rfrDeviceInfo.enDeviceProfile));
      }
      m_oLockDeviceList.vUnlock();
   
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bIsDeviceWaitingForRoleswitch()
***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceWaitingForRoleswitch(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem)
{
   t_Bool bWaitingForRoleSwitch= ((mplay_fi_tcl_e8_DiPODeviceStatus::FI_EN_E8DS_INITIALIZING == corfrMplayDeviceInfoItem.e8DeviceState.enType)
            && (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_USB == corfrMplayDeviceInfoItem.e8ConnectionType.enType));
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceWaitingForRoleswitch ? %d : for device with MplayDevice Tag 0x%x",
            ETG_ENUM(BOOL,bWaitingForRoleSwitch), corfrMplayDeviceInfoItem.u8DeviceTag));
   return bWaitingForRoleSwitch;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vSetDeviceConnectionStatus()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vSetDeviceConnectionStatus(const t_U32 cou32DeviceHandle, tenMPDeviceConnectionType enDevConntype, tenDeviceConnectionStatus enConnStatus)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::vSetDeviceConnectionStatus : for device 0x%x of type %d to connection status %d", cou32DeviceHandle,enDevConntype,
            ETG_ENUM(CONNECTION_STATUS,enConnStatus)));
   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(cou32DeviceHandle);
   if (m_rMplayDeviceList.end() != itMap)
   {
      switch(enDevConntype)
      {
         case e8_USB_CONNECTION:
         {
            itMap->second.rConnectionStatus.enUSBConnectionStatus = enConnStatus;
         }
         break;
         case e8_BLUETOOTH_CONNECTION:
         {
            itMap->second.rConnectionStatus.enBTConnectionStatus = enConnStatus;
         }
         break;
         case e8_WIFI_CONNECTION:
         {
            itMap->second.rConnectionStatus.enWiFiConnectionStatus = enConnStatus;
         }
         break;
         default:
         {
            ETG_TRACE_ERR(("[DESC]:spi_tclMplayDeviceListHandler::vSetDeviceConnectionStatus : Invlid connection type"));
         }
      }
   }
   m_oLockDeviceList.vUnlock();
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceIAP2OverWiFiConnected()
***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDeviceIAP2OverWiFiConnected(const t_U32 cou32DeviceHandle)
{
   t_Bool bIsAP2ConnectedOverWifi = false;
   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(cou32DeviceHandle);
   if (m_rMplayDeviceList.end() != itMap)
   {
      bIsAP2ConnectedOverWifi = itMap->second.bIsIAP2OverWiFiConnected;
   }
   m_oLockDeviceList.vUnlock();
   ETG_TRACE_USR1(("[DESC]:spi_tclMplayDeviceListHandler::bIsDeviceIAP2OverWiFiConnected Device %d is connected for IAP2overWifi = %d", cou32DeviceHandle,
            ETG_ENUM(BOOL, bIsAP2ConnectedOverWifi)));
   return bIsAP2ConnectedOverWifi;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsWaitingForRoleSwitch()
***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsWaitingForRoleSwitch(t_U32 u32DevID)
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bIsWaitingForRoleSwitch:DevID:0x%x",u32DevID));

   t_Bool bIsWaitingForRoleswitch = false;
   
   m_oLockDeviceList.s16Lock();
   auto itMap = m_rMplayDeviceList.find(u32DevID);
   if (m_rMplayDeviceList.end() != itMap)
   {
      bIsWaitingForRoleswitch = itMap->second.bIsWaitingForRoleSwitch;
   }
   m_oLockDeviceList.vUnlock();

   return bIsWaitingForRoleswitch;	
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bIsValidBTDevice()
***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsValidBTDevice(const mplay_fi_tcl_DiPODeviceInfoItem& corfrDevInfo,
                                                       t_String szBTMACAddress) const
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bIsValidBTDevice():ConnType:%d,ConnState:%d,BTAddress:%s",
                     corfrDevInfo.e8ConnectionType.enType,corfrDevInfo.e8ConnectionState.enType,szBTMACAddress.c_str()));

   t_Bool bRet = ( (mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_BLUETOOTH == corfrDevInfo.e8DeviceType.enType) &&
                     (!szBTMACAddress.empty()));
    //(mplay_fi_tcl_e8_DiPOConnectionState::FI_EN_E8CS_ATTACHED == corfrMplayDeviceInfoItem.e8ConnectionState.enType)

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vUpdateMediaDeviceInfo()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vUpdateMediaDeviceInfo(t_U32 u32DevID,
                                                             t_String szBTMACAddress,
                                                             t_Bool bIsMediaActive,
                                                             t_Bool bIsConnected)
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vUpdateMediaDeviceInfo():DeviceID:0x%x,IsMediaActive:%d,IsConnected:%d,BTAddress:%s",
                    u32DevID,ETG_ENUM(BOOL,bIsMediaActive),ETG_ENUM(BOOL,bIsConnected),szBTMACAddress.c_str()));

   //Post the message
   trMediaDeviceInfo rMediaDeviceInfo;
   rMediaDeviceInfo.u32DevID = u32DevID;
   rMediaDeviceInfo.bIsConnected = bIsConnected;
   rMediaDeviceInfo.bIsMediaPlaying = bIsMediaActive;
   rMediaDeviceInfo.szBTMACAddress = szBTMACAddress;

   if(nullptr != m_rMediaDeviceInfoCbs.fpvMediaDeviceInfo)
   {
      (m_rMediaDeviceInfoCbs.fpvMediaDeviceInfo)(rMediaDeviceInfo);
   }

}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vRegisterCallbacks(const trMediaDeviceInfoCbs& corMediaDeviceInfoCbs)
{
   m_rMediaDeviceInfoCbs = corMediaDeviceInfoCbs;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bIsDisconnectionValid()
***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bIsDisconnectionValid(const mplay_fi_tcl_DiPODeviceInfoItem& corfrMplayDeviceInfoItem)
{
   t_Bool bValidDisconnection = false;

   bValidDisconnection = ((mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_IPHONE == corfrMplayDeviceInfoItem.e8DeviceType.enType)&&
            (mplay_fi_tcl_e8_DiPOConnectionType::FI_EN_E8DCT_USB == corfrMplayDeviceInfoItem.e8ConnectionType.enType) &&
            (mplay_fi_tcl_e8_DiPODeviceStatus::FI_EN_E8DS_INITIALIZING != corfrMplayDeviceInfoItem.e8DeviceState.enType)&&
            (mplay_fi_tcl_e8_DiPOCaps::FI_EN_E8DIPO_CAP_NONE == corfrMplayDeviceInfoItem.e8DiPOCaps.enType) &&
            (false == bGetRoleSwitchTriggerStatus()));

   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bIsDisconnectionValid left with bValidDisconnection: %d", ETG_ENUM(BOOL, bValidDisconnection)));

     return bValidDisconnection;	
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vSetFeatureSupport()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vSetFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupport)
{
   m_bCPWEnabledStatus = corfrSpiFeatureSupport.bGetWirelessFeatureSupport(e8DEV_TYPE_DIPO);
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vSetFeatureSupport left with CPW KDS status as %d", ETG_ENUM(BOOL, m_bCPWEnabledStatus)));
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vRestoreSettings()
 ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vRestoreSettings()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclMplayDeviceListHandler::vRestoreSettings() entered "));

   m_oLockDeviceList.s16Lock();
   if (false == m_rMplayDeviceList.empty())
   {
      m_rMplayDeviceList.clear();
   }//if (nullptr != m_pSpiCmdIntf)
   m_oLockDeviceList.vUnlock();
}

/**************************************************************************
** FUNCTION   : tVoid spi_tclMplayDeviceListHandler:: vDeleteDevices()
***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vDeleteDevices(const std::vector<t_U32> &corfrvecDeletedDevices)
{

   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vDeleteDevices Entered"));

   for (auto itvec = corfrvecDeletedDevices.begin(); itvec != corfrvecDeletedDevices.end(); ++itvec)
   {
         ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vDeleteDevice - The device handle to be deleted from Discoverer is : %d", *itvec));
         m_oLockDeviceList.s16Lock();
         auto itMap = m_rMplayDeviceList.find(*itvec);
         if(itMap != m_rMplayDeviceList.end())
         {
            m_rMplayDeviceList.erase(itMap);
         }
         m_oLockDeviceList.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION:  t_void spi_tclMplayDeviceListHandler::vSetRoleSwitchTriggerStatus(...)
 ***************************************************************************/
t_Void spi_tclMplayDeviceListHandler::vSetRoleSwitchTriggerStatus(t_Bool bTriggerStatus)
{
   m_bRoleSwitchTriggerStatus = bTriggerStatus;
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::vSetRoleSwitchTriggerStatus :  %d", ETG_ENUM(BOOL,m_bRoleSwitchTriggerStatus)));
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bGetRoleSwitchTriggerStatus(...)
 ***************************************************************************/
t_Bool spi_tclMplayDeviceListHandler::bGetRoleSwitchTriggerStatus()
{
   ETG_TRACE_USR1(("spi_tclMplayDeviceListHandler::bGetRoleSwitchTriggerStatus :  %d", ETG_ENUM(BOOL,m_bRoleSwitchTriggerStatus)));
   return m_bRoleSwitchTriggerStatus;
}
