/*!
*******************************************************************************
* \file              spi_tclMplayDeviceListHandler.h
* \brief             Devicelist handling for apple devices reported by mediaplayer
*******************************************************************************
\verbatim
PROJECT:        G3G
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    DiPO Client handler class for MPlay service
COPYRIGHT:      &copy; RBEI

HISTORY:
 Date       |  Author                      | Modifications
 20.09.2017 |  Pruthvi Thej Nagaraju      | Initial Version

\endverbatim
******************************************************************************/

#ifndef COMPONENTS_PROJECTS_G3G_MPLAYAPPCONTROL_INC_SPI_TCLMPLAYDEVICELISTHANDLER_H_
#define COMPONENTS_PROJECTS_G3G_MPLAYAPPCONTROL_INC_SPI_TCLMPLAYDEVICELISTHANDLER_H_

#include "Lock.h"
#include "SPITypes.h"
#include "ExtInterfacesTypes.h"
#include "spi_tclMplayFItypes.h"
#include "spi_ConnMngrTypeDefines.h"
/****************************************************************************/
/*!
* \class spi_tclMplayDeviceListHandler
* \brief Devicelist handling for apple devices reported by mediaplayer
*
* Class MplayDeviceListHandler
*     Resposibility
* 1.  Responsible for processing device list received from mediaplayer
* 2.  Responsible for maintaining SPI and corresponding Mediaplayer Device IDs
* 3.  Responsible for determining carplay capability, OTG port etc
* 4.  Responsible for filtering invalid Device Information
* 5.  Responsible for checking if the device information has changed
* 6.  Responsible for generating DeviceID for Apple devices
* 7.  Responsible for detecting disconnections from devicelist update
* 8.  Responsible for sending Devicelist updates
*
*     Collaboration:
* 1. Uses CRC to generate DeviceID
* 2. Uses DeviceIDDataInterface to access device details
* 3. Used by MediaplayerClientHandler to process Devicelist
*
*
****************************************************************************/

class spi_tclMplayDeviceListHandler
{
   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclMplayDeviceListHandler::spi_tclMplayDeviceListHandler(..)
       ** ***************************************************************************/
      /*!
       * \fn     spi_tclMplayDeviceListHandler(ahl_tclBaseOneThreadApp* poMainApp)
       * \brief  Parameterized Constructor
       * \sa
       ***************************************************************************/
      spi_tclMplayDeviceListHandler();

       /***************************************************************************
       ** FUNCTION:  virtual spi_tclMplayDeviceListHandler::~spi_tclMplayDeviceListHandler()
       ***************************************************************************/
       /*!
       * \fn      ~spi_tclMplayDeviceListHandler()
       * \brief   Destructor
       * \sa
       **************************************************************************/
       ~spi_tclMplayDeviceListHandler();

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vRegisterCallbacks()
       ***************************************************************************/
       /*!
       * \brief  Registers callbacks for device connection and disconnection updates
       ***************************************************************************/
       t_Void vRegisterCallbacks(const trDiscovererCbs &rfrDiscovererCbs);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vRegisterCallbacks()
       ***************************************************************************/
       /*!
       * \brief  Registers callbacks for devicelist updates
       ***************************************************************************/
       t_Void vRegisterCallbacks(const tfvDeviceListUpdate &rfrDeviceListUpdate);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vUnregisterCallbacks()
       ***************************************************************************/
       /*!
       * \brief Unregisters callbacks for device connection and disconnection updates
       ***************************************************************************/
       t_Void vUnregisterCallbacks();

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::u32GetMPDeviceTag()
       ***************************************************************************/
       /*!
       * \fn      u32GetMPDeviceTag()
       * \brief   Provides mediaplayer device tag
       **************************************************************************/
       t_U32 u32GetMPDeviceTag(const t_U32 cou32DeviceID, tenDeviceConnectionType enConnType);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::u32GetSPIDeviceID()
       ***************************************************************************/
       /*!
       * \fn      u32GetSPIDeviceID()
       * \brief   Provides SPI device ID from mediplayer device ID
       **************************************************************************/
       t_U32 u32GetSPIDeviceID(const t_U32 cou32DeviceTag, tenDeviceConnectionType enConnType);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bDeviceInfoChanged()
       ***************************************************************************/
       /*!
       * \fn     bDeviceInfoChanged(const t_U32 u32DeviceHandle,const trDeviceInfo &corfrDeviceInfo);
       * \brief  Returns true if device information has changed
       * \retval t_Bool : True or false
       ***************************************************************************/
       t_Bool bDeviceInfoChanged(const t_U32 u32DeviceHandle,const trDeviceInfo &corfrDeviceInfo);


       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vReceiveMplayDeviceList()
       ***************************************************************************/
       /*!
       * \fn      vReceiveMplayDeviceList()
       * \brief   Devicelist update received from mediplayer clienthanlder
       * \param   rfrMplayDeviceList: reference to deviceinfo received from mediaplayer
       **************************************************************************/
       t_Void vOnMplayDeviceListUpdate(mplay_fi_tcl_DiPODeviceInfo& rfrMplayDeviceList);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vRegisterCallbacks()
       ***************************************************************************/
       /*!
       * \brief  Registers callbacks for Media Device Info
       * \param  corMediaDeviceInfoCbs: Media Device Info callbacks
       ***************************************************************************/
       t_Void vRegisterCallbacks(const trMediaDeviceInfoCbs& corMediaDeviceInfoCbs);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vSetFeatureSupport()
       ***************************************************************************/
       /*!
       * \fn     vSetFeatureSupport(t_Bool bCPWEnabledStatus)
       * \brief  Function to Set the KDS CPW Enabled Status
       * \param  corfrSpiFeatureSupport : [IN] Feature Support
       * \retval None
       **************************************************************************/
       t_Void vSetFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupport);
	   
	   
       /***************************************************************************
       ** FUNCTION: t_Void spi_tclMplayDeviceListHandler::vRestoreSettings()
       ***************************************************************************/
       /*!
       * \fn     vRestoreSettings()
       * \brief  Interface to restore settings in SPI.
       * \param  None
       * \sa
       **************************************************************************/
       t_Void vRestoreSettings();

       /***************************************************************************
        ** FUNCTION: t_Void spi_tclMplayDeviceListHandler::vDeleteDevices
        ***************************************************************************/
       /*!
        * \fn     vDeleteDevices
        * \brief  Deleting the Device
        * \param   corfrvecDeletedDevices : List of devices to be Deleted
        **************************************************************************/
       t_Void vDeleteDevices(const std::vector<t_U32> &corfrvecDeletedDevices);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vSetRoleSwitchTriggerStatus()
       ***************************************************************************/
       /*!
       * \fn     vSetRoleSwitchTriggerStatus()
       * \brief  Function to Set the Role Switch Trigger status.
       * \param  bTriggerStatus: [IN] True or false
       * \retval None
       **************************************************************************/
       t_Void vSetRoleSwitchTriggerStatus(t_Bool bTriggerStatus);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bGetRoleSwitchTriggerStatus()
       ***************************************************************************/
       /*!
       * \fn     bGetRoleSwitchTriggerStatus()
       * \brief  Function to return the role switch trigger status.
       * \retval t_Bool : True or false
       ***************************************************************************/
       t_Bool bGetRoleSwitchTriggerStatus();

   private:

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bValidateDeviceInformation()
      ***************************************************************************/
      /*!
      * \fn      bValidateDeviceInformation()
      * \brief   Check if the device information received is valid
      * \param   rfrMplayDeviceInfoItem: Device detials received from mediaplayer
      * \param   coszDeviceName: Device name
      * \param   szDeviceBTAddress: BT Address
      * \param   coszSerialNo: Serial number
      * \retval  returns true if the device information is valid
      **************************************************************************/
       t_Bool bIsDeviceInfoInvalid(mplay_fi_tcl_DiPODeviceInfoItem &rfrMplayDeviceInfoItem,
                 const t_String szDeviceBTAddress, const t_String coszSerialNo);


       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vSendDeviceConnectionUpdate()
       ***************************************************************************/
       /*!
       * \fn      vSendDeviceConnectionUpdate()
       * \brief    Send device connection update
       * \param   corfrDeviceinfo: Device info
       **************************************************************************/
       t_Void vSendDeviceConnectionUpdate(const trMPDeviceInfo &corfrDeviceinfo);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vSendDeviceDisconnectionUpdate()
       ***************************************************************************/
       /*!
       * \fn      vSendDeviceDisconnectionUpdate()
       * \brief    Send device disconnection update
       * \param   corfrDeviceinfo: Device info
       **************************************************************************/
       t_Void vSendDeviceDisconnectionUpdate(const trDeviceInfo &corfrDeviceinfo,tenDiscovererType enDiscovererType, tenDeviceConnectionType enConnType);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vDisplayDeviceInfoList()
       ***************************************************************************/
       /*!
       * \fn      vDisplayDeviceInfoList()
       * \brief   Display device list received by mediaplayer
       * \param   corfrMplayDeviceInfoItem: Device detials received from mediaplayer
       * \param   coszDeviceName: Device name
       * \param   szDeviceBTAddress: BT Address
       * \param   coszSerialNo: Serial number
       **************************************************************************/
       t_Void vDisplayDeviceInfoList(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem, const t_String coszDeviceName, const t_String szDeviceBTAddress, const t_String coszSerialNo);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceConnectedForUSB
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceConnectedForUSB()
       * \brief   Check if the apple device is connected over USB
       * \param   corfrMplayDeviceInfoItem: reference to deviceinfo received from mediaplayert
       **************************************************************************/
       t_Bool bIsDeviceConnectedForUSB(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceIAP2BTConnected
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceIAP2BTConnected()
       * \brief   Check if the apple device is connected over BT
       * \param   corfrMplayDeviceInfoItem: reference to deviceinfo received from mediaplayert
       **************************************************************************/
       t_Bool bIsDeviceIAP2BTConnected(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverWiFi
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceConnectedOverWiFi()
       * \brief   Check if the apple device is connected over Wi-Fi
       * \param   corfrMplayDeviceInfoItem: reference to deviceinfo received from mediaplayert
       **************************************************************************/
       t_Bool bIsDeviceConnectedOverWiFi(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverUSB
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceDisconnectedOverUSB()
       * \brief   Check if the apple device is disconnected over USB
       * \param   corfrMplayDeviceInfoItem: reference to deviceinfo received from mediaplayert
       **************************************************************************/
       t_Bool bIsDeviceDisconnectedOverUSB(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverBT
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceDisconnectedOverBT()
       * \brief   Check if the apple device is disconnected over BT
       * \param   corfrMplayDeviceInfoItem: reference to deviceinfo received from mediaplayer
       **************************************************************************/
       t_Bool bIsDeviceDisconnectedOverBT(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceDisconnectedOverWiFi
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceDisconnectedOverWiFi()
       * \brief   Check if the apple device is disconnected over WiFi
       * \param   corfrMplayDeviceInfoItem: reference to deviceinfo received from mediaplayer
       **************************************************************************/
       t_Bool bIsDeviceDisconnectedOverWiFi(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);


       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vPopulateDeviceCapabilities()
       ***************************************************************************/
       /*!
       * \fn      vPopulateDeviceCapabilities()
       * \brief   Populate device capabilities
       * \param   corfrMplayDeviceInfoItem: Device detials received from mediaplayer
       **************************************************************************/
       t_Void vPopulateDeviceCapabilities(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem, trDeviceInfo &rfrDeviceinfo);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vPopulateDeviceDetails()
       ***************************************************************************/
       /*!
       * \fn      vPopulateDeviceDetails()
       * \brief   populate device details from device info received from mediaplayer
       * \param   corfrMplayDeviceInfoItem: Device detials received from mediaplayer
       * \param   coszDeviceName: Device name
       * \param   szDeviceBTAddress: BT Address
       * \param   coszSerialNo: Serial number
       **************************************************************************/
       t_Void vPopulateDeviceDetails(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem, 
	                                 trMPDeviceInfo &rfrDeviceinfo, 
	                                 const t_String coszDeviceName, 
	                                 const t_String szDeviceBTAddress, 
	                                 const t_String coszSerialNo);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::enGetUSBConnectionStatus()
       ***************************************************************************/
       /*!
       * \fn      enGetUSBConnectionStatus()
       * \brief   Get USB connection status
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       tenDeviceConnectionStatus enGetUSBConnectionStatus(const t_U32 cou32DeviceHandle);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::enGetBTConnectionStatus()
       ***************************************************************************/
       /*!
       * \fn      enGetBTConnectionStatus()
       * \brief   Get BT connection status
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       tenDeviceConnectionStatus enGetBTConnectionStatus(const t_U32 cou32DeviceHandle);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::enGetWiFiConnectionStatus()
       ***************************************************************************/
       /*!
       * \fn      enGetWiFiConnectionStatus()
       * \brief   Get WiFi connection status
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       tenDeviceConnectionStatus enGetWiFiConnectionStatus(const t_U32 cou32DeviceHandle);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceConnected()
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceConnected()
       * \brief   Checks if the device is connected on atleast one of the transports
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       t_Bool bIsDeviceConnected(const t_U32 cou32DeviceHandle);


       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vHandleDeviceConnections()
       ***************************************************************************/
       /*!
       * \fn      vHandleDeviceConnections()
       * \brief   handles sending device connection and disconnection updates
       * \param   coszDeviceName: Device name
       * \param   szDeviceBTAddress: BT Address
       * \param   coszSerialNo: Serial number
       * \param   oDevInfo: reference to devicelist
       **************************************************************************/
       t_Void vHandleDeviceConnections(const t_String& szDeviceName, const t_String& szBTMACAddress,
               const t_String& szSerialNumber, mplay_fi_tcl_DiPODeviceInfoItem& oDevInfo);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vEraseFromDevicelist()
       ***************************************************************************/
       /*!
       * \fn      vEraseFromDevicelist()
       * \brief   Erase device from the list
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       t_Void vEraseFromDevicelist(const t_U32 cou32DeviceHandle);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vAddDeviceToList()
       ***************************************************************************/
       /*!
       * \fn      vAddDeviceToList()
       * \brief   Add device to the list
       * \param   corfrDeviceInfo: Device Info
       **************************************************************************/
       t_Void vAddDeviceToList(const trMPDeviceInfo &corfrDeviceInfo);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vUpdateDeviceList()
       ***************************************************************************/
       /*!
       * \fn      vUpdateDeviceList()
       * \brief   updates device list change info
       * \param   rfrDeviceInfo: reference to device info
	   * \param   bDeviceInfoInValid : Flag to say whether the DeviceList to be updated or not
       **************************************************************************/
       t_Void vUpdateDeviceList(trDeviceInfo &rfrDeviceInfo, t_Bool bDeviceInfoInValid);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bIsDeviceWaitingForRoleswitch()
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceWaitingForRoleswitch()
       * \brief   indictaes device connection when a device is waiting for useraction for early roleswitch
       * \param   corfrMplayDeviceInfoItem: reference to device info
       * \retval  true if device is waiting for role switch trigger
       **************************************************************************/
       t_Bool bIsDeviceWaitingForRoleswitch(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::vSetDeviceConnectionStatus()
       ***************************************************************************/
       /*!
       * \fn      vSetDeviceConnectionStatus()
       * \brief   Set device connection status for a particulat transport
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       t_Void vSetDeviceConnectionStatus(const t_U32 cou32DeviceHandle, tenMPDeviceConnectionType enDevConntype, tenDeviceConnectionStatus enConnStatus);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceIAP2OverWiFiConnected()
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceIAP2OverWiFiConnected()
       * \brief   Checks if the device is connected on iap2 over WiFi
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       t_Bool bIsDeviceIAP2OverWiFiConnected(const t_U32 cou32DeviceHandle);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDeviceConnectedOverBTWithCPWFeasible()
       ***************************************************************************/
       /*!
       * \fn      bIsDeviceConnectedOverBTWithCPWFeasible()
       * \brief   Checks if the device is connected on iap2BT and CPW_Feasible
       * \param   cou32DeviceHandle: Device ID
       **************************************************************************/
       t_Bool bIsDeviceConnectedOverBTWithCPWFeasible(const mplay_fi_tcl_DiPODeviceInfoItem& corfrMplayDeviceInfoItem);

	   
       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsWaitingForRoleSwitch()
       ***************************************************************************/
       /*!
       * \fn      bIsWaitingForRoleSwitch()
       * \brief   Checks if MPO is waiting for roleswitch of the device.
       * \param   cou32DeviceHandle: Device ID
	   * \retval  t_Bool
       **************************************************************************/	   
	   t_Bool bIsWaitingForRoleSwitch(t_U32 u32DevID);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::bIsValidBTDevice()
       ***************************************************************************/
       /*!
       * \fn      t_Bool bIsValidBTDevice(const mplay_fi_tcl_DiPODeviceInfoItem& corfrDevInfo,t_String szBTMACAddress)
       * \brief   To check whether it is a valid BT device
       * \param   corfrDevInfo: Device Info
       * \param   szBTMACAddress: BT MAC Address
       * \retval  t_Bool
       **************************************************************************/
       t_Bool bIsValidBTDevice(const mplay_fi_tcl_DiPODeviceInfoItem& corfrDevInfo,t_String szBTMACAddress) const;

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMplayDeviceListHandler::vUpdateMediaDeviceInfo()
       ***************************************************************************/
       /*!
       * \fn      t_Bool vUpdateMediaDeviceInfo(t_U32 u32DevID,t_String szBTMACAddress,t_Bool bIsMediaActive,t_Bool bIsConnected)
       * \brief   To update the active media BT devices info
       * \param   u32DevID: Device ID
       * \param   szBTMACAddress: BT MAC Address
       * \param   u32DevID: Device ID
       * \param   bIsConnected: Connected/Disconnected.
       * \retval  t_Bool
       **************************************************************************/
       t_Void vUpdateMediaDeviceInfo(t_U32 u32DevID,t_String szBTMACAddress,t_Bool bIsMediaActive,t_Bool bIsConnected);

	  /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMplayDeviceListHandler::bIsDisconnectionValid
       ***************************************************************************/
       /*!
       * \fn      bIsDisconnectionValid()
       * \brief   Check if the apple device is disconnected over USB for MYSIN/ CARLIFE
       * \param   corfrMplayDeviceInfoItem: reference to deviceinfo received from mediaplayert
       **************************************************************************/
       t_Bool bIsDisconnectionValid(const mplay_fi_tcl_DiPODeviceInfoItem &corfrMplayDeviceInfoItem);


       //! Hold the connected device list
       std::map<t_U32, trMPDeviceInfo> m_rMplayDeviceList;

       //! Lock to protect m_rMplayDeviceList
       Lock m_oLockDeviceList;

       trDiscovererCbs m_rDiscovererCbs;
       tfvDeviceListUpdate m_fvDeviceListUpdate;

       trMediaDeviceInfoCbs m_rMediaDeviceInfoCbs;

       //! member variable to store the CPW Enabled status
       t_Bool m_bCPWEnabledStatus;

       //! Role switch trigger status
       t_Bool m_bRoleSwitchTriggerStatus;
};

#endif /* COMPONENTS_PROJECTS_G3G_MPLAYAPPCONTROL_INC_SPI_TCLMPLAYDEVICELISTHANDLER_H_ */
