/*!
*******************************************************************************
* \file              spi_tclMPlayClientHandler.h
* \brief             DiPO Client handler class.
*******************************************************************************
\verbatim
PROJECT:        G3G
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    DiPO Client handler class for MPlay service
COPYRIGHT:      &copy; RBEI

HISTORY:
 Date       |  Author                      | Modifications
 08.1.2014  |  Shihabudheen P M            | Initial Version
 02.04.2014 | Hari Priya E R               | Implemented Timer for triggering
                                             transfer of GPS Data
 15.04.2014 | Ramya Muthy                  | Implemented Location data transfer
                                             and removed timer implementation.
 13.06.2014 | Ramya Murthy                 | Implementation for MPlay FI extn 
                                             for location info.
 31.07.2014 | Ramya Murthy                 | Changes for dynamic registration of properties
 05.11.2014 | Ramya Murthy                 | Implementation for Application metadata.
 05.11.2014 | Ramya Murthy                 | Implementation of revised CarPlay media concept 
 14.12.2015 | SHITANSHU SHEKHAR(RBEI/ECP2) | Implemented interfaces as per new Media player FI:
                                           | vOnDiPONowPlayingStatus(), vOnDiPOPlayBackStatus()
                                           | vOnDiPOPlayBackShuffleModeStatus(),
                                           | vOnDiPOPlayBackRepeatModeStatus(),
                                           | enGetPhoneCallState(), enGetPhoneCallDirection(),
                                           | enGetCallStateUpdateService(),
                                           | enGetCallStateUpdateDisconnectReason(),
                                           | vOnDiPOCallStateStatus(), enGetRegistrationStatus()
                                           | vOnDiPOCommunicationsStatus()
 01.03.2017 |  Shiva Kumar G               | Adaptations for CarPlay Wireless

\endverbatim
******************************************************************************/
#ifndef SPI_TCLMPLAY_CLIENT_HANDLER_H_
#define SPI_TCLMPLAY_CLIENT_HANDLER_H_

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include <cstring>
#include <vector>

#include "Timer.h"
#include "DiPOTypes.h"
#include "SPITypes.h"
#include "spi_tclClientHandlerBase.h"
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclAppMngrDefines.h"
#include "MsgContext.h"
#include "spi_TBTTypes.h"
#include "spi_tclMplayFItypes.h"
#include "spi_tclMplayDeviceListHandler.h"


/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
typedef mplay_fi_tcl_e8_AudioError  mplay_tFiAudError;
typedef mplay_fi_tcl_e8_DistanceDisplayUnits  mplay_tFiDistanceDisplayUnits;

typedef struct
{
   tenKeyCode enKeyCode;
   mplay_fi_tcl_e8_DiPOPlaypackAction::tenType enActionType;
}trIAPKeyCode;

//! Forward declarations
class spi_tclMainApp;
class ahl_tclBaseOneThreadClientHandler;
class ahl_tclBaseOneThreadApp;
class spi_tclRespInterface;
//class mplay_appcontrolfi_tclMsgDiPODeviceConnectionsStatus;
//class mplay_appcontrolfi_tclMsgDiPORoleSwitchRequestMethodResult;
class spi_tclMplayResp;

/****************************************************************************/
/*!
* \class DiPO_tclSPIClientHandler
* \brief DiPO clinet handler Implementation
*
* DiPO_tclSpiClientHandler is implement the client handler for DiPO based
* services provided by media player fi. It mainly dealt with connection
* establishment and management
****************************************************************************/
class spi_tclMPlayClientHandler : public ahl_tclBaseOneThreadClientHandler, public spi_tclClientHandlerBase
{
public:

   /***************************************************************************
   *********************************PUBLIC************************************
   ***************************************************************************/

  /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::spi_tclMPlayClientHandler(..)
   ** ***************************************************************************/
  /*!
   * \fn     spi_tclMPlayClientHandler(ahl_tclBaseOneThreadApp* poMainApp)
   * \brief  Parameterized Constructor
   * \sa
   ***************************************************************************/
   spi_tclMPlayClientHandler(ahl_tclBaseOneThreadApp* poMainApp);

   /***************************************************************************
   ** FUNCTION:  virtual spi_tclMPlayClientHandler::~spi_tclMPlayClientHandler()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclMPlayClientHandler()
   * \brief   Destructor
   * \sa
   **************************************************************************/
   virtual ~spi_tclMPlayClientHandler();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterForProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForProperties()
   * \brief   Registers for all interested properties to respective service
   *          Mandatory interface to be implemented.
   *          Dummy function to satisfy requirements of spi_tclClientHandlerBase
   * \sa      vUnregisterForProperties()
   **************************************************************************/
   virtual t_Void vRegisterForProperties(){}

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterForDiscovererProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForDiscovererProperties()
   * \brief   Registers for all interested properties to respective service
   *          Mandatory interface to be implemented.
   * \sa      vUnregisterForProperties()
   **************************************************************************/
   virtual t_Void vRegisterForDiscovererProperties();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterForNavigationProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForNavigationProperties()
   * \brief   Registers for all interested properties to respective service
   *          Mandatory interface to be implemented.
   * \sa      vUnregisterForProperties()
   **************************************************************************/
   virtual t_Void vRegisterForNavigationProperties();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterForNativeTransportProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForNativeTransportProperties()
   * \brief   Registers for all interested properties to respective service
   *          Mandatory interface to be implemented.
   * \sa      vUnregisterForProperties()
   **************************************************************************/
   virtual t_Void vRegisterForNativeTransportProperties();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterForMetadataProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForMetadataProperties()
   * \brief   Registers for all interested properties to Metadata
   * \sa      vRegisterForMetadataProperties()
   **************************************************************************/
   virtual t_Void vRegisterForMetadataProperties();
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterIAPCallbacks()
   **************************************************************************/
   /*!
   * \fn      vRegisterIAPCallbacks()
   * \brief   Registers for all callbacks properties to respective service
   * \sa      vUnregisterForProperties()
   **************************************************************************/
    t_Void vRegisterIAPCallbacks(trIAPDevInfoCallbacks rIAPDevInfoCallbacks,
               tfvBTProfileCallback fvBTProfileCallback = NULL);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vUnregisterForProperties()
   **************************************************************************/
   /*!
   * \fn      vUnregisterForProperties()
   * \brief   Un-registers all subscribed properties to respective service
   *          Mandatory interface to be implemented.
   * \sa      vRegisterForProperties()
   **************************************************************************/
   virtual t_Void vUnregisterForProperties();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vUnregisterForMetadataProperties()
   **************************************************************************/
   /*!
   * \fn      vUnregisterForMetadataProperties()
   * \brief   Un-registers all subscribed properties to Metadata
   * \sa      vUnregisterForMetadataProperties()
   **************************************************************************/
   virtual t_Void vUnregisterForMetadataProperties();
   /* Call function declarations */

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bDiPORoleSwitchRequest(t_U8..
   ***************************************************************************/
   /*!
   * \fn      bDiPORoleSwitchRequest(t_U8 u8DeviceTag, )
   * \brief   Function to activate the dipo device
   * \param   u32DeviceID: [IN] unique identifier of the device
   * \param   enDiPORoleStatus : [IN] Role to perform role switch
   * \sa
   **************************************************************************/
  // t_Bool bDiPORoleSwitchRequest(t_U32 u32DeviceID, tenDeviceConnectionReq enDevConnReq);
   t_Bool bRoleSwitchRequest(t_U32 u32DeviceID, tenDeviceConnectionReq enDevConnReq,
            trEAPAppInfo rEAPAppInfo, tenDeviceCategory endDeviceCat = e8DEV_TYPE_DIPO);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bDiPOGetAlbumArtInfo(t_String sAlbumArt)
   ***************************************************************************/
   /*!
   * \fn      bDiPOGetAlbumArtInfo(t_String sAlbumArt)
   * \brief   Function to get album art info
   * \param   sAlbumArt : [IN] Album art
   * \retval  bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bDiPOGetAlbumArtInfo(t_String sAlbumArt);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bDiPOGetAlbumArt(t_String sUrl)
   ***************************************************************************/
   /*!
   * \fn     bDiPOGetAlbumArt(t_String sUrl)
   * \brief  Function to get album art
   * \param  sAlbumArt : [IN] Album art
   * \retval bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bGetDiPOAlbumArt(t_String szAlbumArtUrl, const trUserContext& corfrUsrCntxt);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bTransferGPSData(...)
   ***************************************************************************/
   /*!
   * \fn     bTransferGPSData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo, t_String szGPGGAData,
   *            t_String szGPRMCData, t_String szGPGSVData, t_String szGPHDTData)
   * \brief  Function to transfer GPS date to device.
   * \param  u32DeviceHandle : [IN] Unique device handle of active device
   * \param  szDeviceSerialNo : [IN] Unique serial number of active device
   * \param  szGPGGAData : [IN] GPGGA data
   * \param  szGPRMCData : [IN] GPRMS data
   * \param  szGPGSVData : [IN] GPGSV data
   * \param  szGPHDTData : [IN] GPHDT data
   * \retval bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bTransferGPSData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo,
         t_String szGPGGAData, t_String szGPRMCData,
         t_String szGPGSVData, t_String szGPHDTData);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bTransferDrData(...)
   ***************************************************************************/
   /*!
   * \fn     bTransferDrData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo, t_String szPASCDData,
   *            t_String szPAGCDData, t_String szPAACDData)
   * \brief  Function to transfer DR date to device.
   * \param  u32DeviceHandle : [IN] Unique device handle of active device
   * \param  szDeviceSerialNo : [IN] Unique serial number of active device
   * \param  szPASCDData : [IN] PASCD data
   * \param  szPAGCDData : [IN] PAGCD data
   * \param  szPAACDData : [IN] PAGCD data
   * \retval bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bTransferDrData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo,
         t_String szPASCDData, t_String szPAGCDData, t_String szPAACDData);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bRequestAudioDevice()
   ***************************************************************************/
   /*!
   * \fn     bRequestAudioDevice(t_U32 u32DeviceHandle)
   * \brief  Function to get audio device name
   * \retval bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bRequestAudioDevice(t_U32 u32DeviceHandle);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bReleaseAudioDevice()
   ***************************************************************************/
   /*!
   * \fn     bReleaseAudioDevice(t_U32 u32DeviceHandle)
   * \brief  Function to release audio device
   * \retval bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bReleaseAudioDevice(t_U32 u32DeviceHandle);


   // For test - Remove later
   t_Void vDiPORoleSwitchResultTest();

   // For test - Remove later
   t_Void vTestMediaPlayer();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \brief  Registers callbacks for device connection and disconnection updates
   ***************************************************************************/
   t_Void vRegisterCallbacks(const trDiscovererCbs &rfrDiscovererCbs);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \brief  Registers callbacks for role switch updates
   ***************************************************************************/
   t_Void vRegisterCallbacks(const trDeviceSwitcherCbs &rfrDeviceSwitcherCbs);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \brief  Registers callbacks for devicelist updates
   ***************************************************************************/
   t_Void vRegisterCallbacks(const tfvDeviceListUpdate &rfrDeviceListUpdate);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterMetaDataCallbacks()
   ***************************************************************************/
   /*!
   * \brief  Registers callbacks for MetaData updates
   ***************************************************************************/
   t_Void vRegisterMetaDataCallbacks(const trMetaDataCbs &rfrMetaDataCbs);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterDipoLocationCallbacks()
   ***************************************************************************/
   /*!
   * \brief  Registers callbacks for DipoLocationInfo updates
   ***************************************************************************/
   t_Void vRegisterDipoLocationCallbacks(const trDipoLocationInfoCallbacks &rfrDipoLocationInfoCallbacks);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vUnregisterCallbacks()
   ***************************************************************************/
   /*!
   * \brief  Registers callbacks for device connection and disconnection updates
   ***************************************************************************/
   t_Void vUnregisterCallbacks(const trDiscovererCbs &rfrDiscovererCbs);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vUnregisterCallbacks()
   ***************************************************************************/
   /*!
   * \brief  Registers callbacks for role switch updates
   ***************************************************************************/
   t_Void vUnregisterCallbacks(const trDeviceSwitcherCbs &rfrDeviceSwitcherCbs);


   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bLaunchApp(const t_U32 cou32DevId, const tenDeviceCategory enDevcat...)
   ***************************************************************************/
   /*!
   * \fn     bLaunchApp(const t_U32 cou32DevId , tenDeviceCategory enDevcat,trEAPAppInfo rEAPAppInfo)
   * \brief  Function to connect and launch a particular app
   * \retval bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bLaunchApp(const t_U32 cou32DevId, const tenDeviceCategory enDevcat,trEAPAppInfo rEAPAppInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vStartiAP2OverCarPlay()
   ***************************************************************************/
   /*!
   * \fn     t_Void vStartiAP2OverCarPlay(t_String szBTMACAddress)
   * \brief  Function to trigger request to initiate iAP2OverCarPlay with a device.
   * \retval t_Void
   **************************************************************************/
   t_Void vStartiAP2OverCarPlay(t_String szBTMACAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::bSendPlayBackCtrlEvent()
   ***************************************************************************/
   /*!
   * \fn      bSendPlayBackCtrlEvent()
   * \brief   This method is used to send media playback control events
   * \retval  bool : true if succees, false otherwise
   **************************************************************************/
   t_Bool bSendPlayBackCtrlEvent(const t_U32 cou32DevId, tenKeyCode enKeyCode, tenKeyMode enKeyMode);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterGetSelectedDevSerialCb()
   **************************************************************************/
   /*!
   * \fn      vRegisterGetSelectedDevSerialCb()
   * \brief   Registers for all callbacks properties to respective service
   * \sa      vUnregisterForProperties()
   **************************************************************************/
    t_Void vRegisterGetSelectedDevSerialCb(tfvGetSelectedDevSerialCallback fvGetSelectedDevSerialCallback);

    /***************************************************************************
    ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRegisterTBTCallback()
    **************************************************************************/
    /*!
    * \fn      vRegisterTBTCallback()
    * \brief   Registers for all callbacks properties to respective service
    **************************************************************************/
     t_Void vRegisterTBTCallback(const trTBTCallbacks &corfrTBTCallbacks);

     /***************************************************************************
     ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vDipoStartRouteGuidanceUpdate()
     **************************************************************************/
     /*!
     * \fn      vDipoStartRouteGuidanceUpdate()
     * \brief   Function to start route guidance
     * \param   corfszDeviceSerialNumber: USB Serial Number of the connected device
     * \sa      vDipoStopRouteGuidance()
     **************************************************************************/
      t_Void vDipoStartRouteGuidanceUpdate(const t_String& corfszDeviceSerialNumber);

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vDipoStopRouteGuidanceUpdate()
      **************************************************************************/
      /*!
      * \fn      vDipoStopRouteGuidanceUpdate()
      * \brief   Function to stop route guidance
      * \param   corfszDeviceSerialNumber: USB Serial Number of the connected device
      * \sa      vDipoStartRouteGuidance()
      **************************************************************************/
       t_Void vDipoStopRouteGuidanceUpdate(const t_String& corfszDeviceSerialNumber);

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf()
   ***************************************************************************/
   /*!
    * \fn     bOnRegRouteGuidanceDisplayConf(
    *          const std::vector<trDiPORouteGuidanceDisplayComponent>& corfvecRouteGuidanceDisplayComponent)
    * \brief  Interface to send device list details
    * \param  [IN] corfvecRouteGuidanceDisplayComponent : registered display component list
    * \retval t_Bool  TRUE- Enabled FALSE-Disabled
    **************************************************************************/
   t_Bool bOnRegRouteGuidanceDisplayConf(const std::vector<trDiPORouteGuidanceDisplayComponent>& corfvecRouteGuidanceDisplayComponent);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates
    ***************************************************************************/
   /*!
    * \fn     bOnStartRouteGuidanceUpdates(
    *          const trStartRouteGuidanceUpdates& corfStartRouteGuidanceUpdates)
    * \brief  Interface to start route guidance updates
    * \param  [IN] trStartRouteGuidanceUpdates : start route guidance parameters
    * \retval t_Bool  TRUE- Enabled FALSE-Disabled
    **************************************************************************/
    t_Bool bOnStartRouteGuidanceUpdates(trStartRouteGuidanceUpdates& corfStartRouteGuidanceUpdates);

    /***************************************************************************
     ** FUNCTION: t_Void spi_tclMPlayClientHandler::bOnStopRouteGuidanceUpdates
     ***************************************************************************/
    /*!
     * \fn     bOnStopRouteGuidanceUpdates(
     *          const trStopRouteGuidanceUpdates& corfStopRouteGuidanceUpdates)
     * \brief  Interface to stop route guidance updates
     * \param  [IN] trStartRouteGuidanceUpdates : stop route guidance parameters
     * \retval t_Bool  TRUE- Enabled FALSE-Disabled
     **************************************************************************/
     t_Bool bOnStopRouteGuidanceUpdates(trStopRouteGuidanceUpdates& corfStopRouteGuidanceUpdates);


       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
       ***************************************************************************/
       /*!
       * \brief  Registers callbacks for Media Device Info
       * \param  corMediaDeviceInfoCbs: Media Device Info callbacks
       ***************************************************************************/
       t_Void vRegisterCallbacks(const trMediaDeviceInfoCbs& corMediaDeviceInfoCbs);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vSetFeatureSupport()
       ***************************************************************************/
       /*!
       * \fn     vSetFeatureSupport(t_Bool bCPWEnabledStatus)
       * \brief  Function to Set the KDS CPW Enabled Status
       * \param  corfrSpiFeatureSupport : [IN] Feature Support
       * \retval None
       **************************************************************************/
       t_Void vSetFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupport);
	   
       /***************************************************************************
       ** FUNCTION: t_Void spi_tclMPlayClientHandler::vRestoreSettings()
       ***************************************************************************/
       /*!
       * \fn     vRestoreSettings()
       * \brief  Interface to restore settings in SPI.
       * \param  None
       * \sa
       **************************************************************************/
       t_Void vRestoreSettings();
       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnMock_DipoRouteGuidanceManeuverStatus(amt_tclServiceData ..
       ***************************************************************************/
       /*!
       * \fn     t_Void vOnMock_DipoRouteGuidanceManeuverStatus(tenDiPoManeuverType enManeuverType,
       *         tenDiPoDrivingSide enDrivingSide,
       *         std::vector<t_S16> vecJunctionElementAngle,
       *         t_S16 s16JunctionElementExitAngle)
       * \brief  Method to mock Dipo Route Guidance Maneuver Property update from HMI through TTFIs commands
       * \retval none
       **************************************************************************/
       t_Void vOnMock_DipoRouteGuidanceManeuverStatus(tenDiPoManeuverType enManeuverType,tenDiPoDrivingSide enDrivingSide,std::vector<t_S16> vecJunctionElementAngle,t_S16 s16JunctionElementExitAngle);

       /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bSetCallControlsConfig()
       ***************************************************************************/
       /*!
       * \fn     t_Void bSetCallControlsConfig() const
       * \brief  Method to set the call controls configuration
       * \param  Key codes set by clients
       * \retval t_Bool
       **************************************************************************/
       t_Bool bSetCallControlConfig(const std::set<tenKeyCode> &rsetKeyCodes);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMPlayClientHandler::bDiPOAcceptCall
       ***************************************************************************/
       /*!
       * \fn     bDiPOAcceptCall(const t_String& corfszDeviceSerialNumber,
       *          const tenPhoneCallAcceptAction coenCallAcceptAction) const
       * \brief  Interface to accept phone call on AcceptCall softkey press
       * \param  [IN] corfszDeviceSerialNumber : Device USB Serial Number
       * \param  [IN] coenCallAcceptAction     : Phone call accept action
       * \retval t_Bool  TRUE - Success FALSE - Failure
       **************************************************************************/
       t_Bool bDiPOAcceptCall(const t_String& corfszDeviceSerialNumber, const tenPhoneCallAcceptAction coenCallAcceptAction) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMPlayClientHandler::bDiPOEndCall
       ***************************************************************************/
       /*!
       * \fn     bDiPOEndCall(const t_String& corfszDeviceSerialNumber,
       *          const tenPhoneCallEndAction coenCallEndAction) const
       * \brief  Interface to end phone call on EndCall softkey press
       * \param  [IN] corfszDeviceSerialNumber : Device USB Serial Number
       * \param  [IN] coenCallEndAction        : Phone call end action
       * \retval t_Bool  TRUE - Success FALSE - Failure
       **************************************************************************/
       t_Bool bDiPOEndCall(const t_String& corfszDeviceSerialNumber, const tenPhoneCallEndAction coenCallEndAction) const;


       /***************************************************************************
       ** FUNCTION: t_Void spi_tclMPlayClientHandler::vPrepareForRestoreSettings()
       ***************************************************************************/
       /*!
       * \fn     vPrepareForRestoreSettings()
       * \brief  Interface to Prepare for restore settings in SPI.
       * \param  None
       * \sa
       **************************************************************************/
       t_Void vPrepareForRestoreSettings();

       /***************************************************************************
       ** FUNCTION: t_Void spi_tclMPlayClientHandler::vRestoreSettingsComplete()
       ***************************************************************************/
       /*!
       * \fn     vRestoreSettingsComplete()
       * \param  None
       * \sa
       **************************************************************************/
       t_Void vRestoreSettingsComplete();

       /***************************************************************************
        ** FUNCTION: t_Void spi_tclMPlayClientHandler::vDeleteDevices
        ***************************************************************************/
       /*!
        * \fn     vDeleteDevices
        * \brief  Deleting the Device
        * \param   corfrvecDeletedDevices : List of devices to be Deleted
        **************************************************************************/
       t_Void vDeleteDevices(const std::vector<t_U32> &corfrvecDeletedDevices);

        /***************************************************************************
        ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vTriggerNowPlayingStatus
        ***************************************************************************/
        /*!
        * \fn     t_Void vTriggerNowPlayingStatus
        * \brief  Method to trigger for now playing status update from MP
        * \retval None
        **************************************************************************/
        t_Void vTriggerNowPlayingStatus() const;

   /*************************************************************************
   ****************************END OF PUBLIC*********************************
   *************************************************************************/

protected:

   /*************************************************************************
   ****************************START OF PROTECTED****************************
   *************************************************************************/

   /***************************************************************************
   * Overriding ahl_tclBaseOneThreadClientHandler methods.
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnServiceAvailable()
   ***************************************************************************/
   /*!
   * \fn      vOnServiceAvailable()
   * \brief   This function is called by the framework if the service of our server
   *          becomes available, e.g. server has been started.
   * \sa
   **************************************************************************/
   virtual t_Void vOnServiceAvailable();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnServiceUnavailable()
   ***************************************************************************/
   /*!
   * \fn      vOnServiceUnavailable()
   * \brief   This function is called by the framework if the service of our server
   *          becomes unavailable, e.g. server has been shut down.
   * \sa
   **************************************************************************/
   virtual t_Void vOnServiceUnavailable();

   /*************************************************************************
   ****************************END OF PROTECTED******************************
   *************************************************************************/

private:

   /*************************************************************************
   ****************************START OF PRIVATE******************************
   *************************************************************************/

   /***************************************************************************
   * Handler function declarations.
   **************************************************************************/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPODeviecConnectionsStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPODeviecConnectionsStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for DiPO Device Connection
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPODeviceConnectionsStatus(amt_tclServiceData* poMessage /*mplay_appcontrolfi_tclMsgDiPODeviceConnectionsStatus &oDevConnectionStatus*/);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPlaytimeStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOPlaytimeStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for Play time status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOPlaytimeStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPONowPlayingStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPONowPlayingStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for Meta Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPONowPlayingStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOPlayBackStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for Meta Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOPlayBackStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackShuffleModeStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOPlayBackShuffleModeStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for Meta Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOPlayBackShuffleModeStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackRepeatModeStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOPlayBackRepeatModeStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for Meta Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOPlayBackRepeatModeStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::enGetPhoneCallState()
   ***************************************************************************/
   /*!
   * \fn      enGetPhoneCallState(mplay_fi_tcl_e8_DiPOCallStateUpdateStatus &rfrenPhoneCallState)
   * \brief   Call back function for Phone Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   tenPhoneCallState enGetPhoneCallState(
         mplay_fi_tcl_e8_DiPOCallStateUpdateStatus &rfrenPhoneCallState);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::enGetPhoneCallDirection()
   ***************************************************************************/
   /*!
   * \fn      enGetPhoneCallDirection(mplay_fi_tcl_e8_DiPOCallStateUpdateDirection &rfenPhoneCallDirection)
   * \brief   Call back function for Phone Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   tenPhoneCallDirection enGetPhoneCallDirection(
         mplay_fi_tcl_e8_DiPOCallStateUpdateDirection &rfenPhoneCallDirection);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::enGetCallStateUpdateService()
   ***************************************************************************/
   /*!
   * \fn      enGetCallStateUpdateService(mplay_fi_tcl_e8_DiPOCallStateUpdateService &rfenCallStateUpdateService)
   * \brief   Call back function for Phone Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   tenCallStateUpdateService enGetCallStateUpdateService(
         mplay_fi_tcl_e8_DiPOCallStateUpdateService &rfenCallStateUpdateService);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::enGetCallStateUpdateService()
   ***************************************************************************/
   /*!
   * \fn      enGetCallStateUpdateService(mplay_fi_tcl_e8_DiPOCallStateUpdateDisconnectReason
   *           &rfenCallStateUpdateDisconnectReason)
   * \brief   Call back function for Phone Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   tenCallStateUpdateDisconnectReason enGetCallStateUpdateDisconnectReason(
         mplay_fi_tcl_e8_DiPOCallStateUpdateDisconnectReason &rfenCallStateUpdateDisconnectReason);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOCallStateStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOCallStateStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for Phone Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOCallStateStatus(amt_tclServiceData* poMessage);
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPoCallDuration()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPoCallDuration(amt_tclServiceData* poMessage)
   * \brief   Call back function for Phone Call Duration
   * \param   poMessage : [IN] pointer to message
   * \sa
   **************************************************************************/

   t_Void vOnDiPoCallDuration(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::enGetRegistrationStatus()
   ***************************************************************************/
   /*!
   * \fn      enGetRegistrationStatus(mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus
   *           &rfenRegistrationStatus)
   * \brief   Call back function for Phone Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   tenRegistrationStatus enGetRegistrationStatus(
         mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus &rfenRegistrationStatus);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOCommunicationsStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOCommunicationsStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for Phone Data status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOCommunicationsStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOActiveDevice..
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOActiveDevice(amt_tclServiceData* poMessage)
   * \brief
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOActiveDevice(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPORoleSwitchResult..
   ***************************************************************************/
   /*!
   * \fn      vOnDiPORoleSwitchResult(amt_tclServiceData* poMessage)
   * \brief
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPORoleSwitchResult(amt_tclServiceData* poMessage /*mplay_appcontrolfi_tclMsgDiPORoleSwitchRequestMethodResult &oRoleSwitchResult*/);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbu..
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOGetMediaObjectAlbumArtInfo(amt_tclServiceData* poMessage)
   * \brief
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOGetMediaObjectAlbumArtInfo(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArt..
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOGetMediaObjectAlbumArt(amt_tclServiceData* poMessage)
   * \brief
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOGetMediaObjectAlbumArt(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOTransferGPSData(..
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOTransferGPSData(amt_tclServiceData* poMessage)
   * \brief
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOTransferGPSData(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOTransferDRData..
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOTransferDRData(amt_tclServiceData* poMessage)
   * \brief
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOTransferDRData(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::vOnStatusDiPOLocationInfo(amt_tclServiceData)
   ***************************************************************************/
   /*!
   * \fn      vOnStatusDiPOLocationInfo(amt_tclServiceData* poMessage)
   * \brief   Method to handle DipoLocationInfo Status message from MediaPlayer
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnStatusDiPOLocationInfo(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOReqAudioDevice(amt_tclServiceData)
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOReqAudioDevice(amt_tclServiceData* poMessage)
   * \brief   Method to handle DiPOAudioDevice MethodResult message from MediaPlayer
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOReqAudioDevice(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPORelAudioDevice(amt_tclServiceData)
   ***************************************************************************/
   /*!
   * \fn      vOnDiPORelAudioDevice(amt_tclServiceData* poMessage)
   * \brief   Method to handle DiPOReleaseAudioDevice MethodResult message from MediaPlayer
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPORelAudioDevice(amt_tclServiceData* poMessage) const;
   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::vOnNativeTransportStartStatus(amt_tclServiceData)
   ***************************************************************************/
   /*!
   * \fn      vOnNativeTransportStartStatus(amt_tclServiceData* poMessage)
   * \brief   Method to handle Native Transport Start Property update from MP
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnNativeTransportStartStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::vOnNativeTransportStopStatus(amt_tclServiceData)
   ***************************************************************************/
   /*!
   * \fn      vOnNativeTransportStopStatus(amt_tclServiceData* poMessage)
   * \brief   Method to handle Native Transport Stop Property update from MP
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnNativeTransportStopStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::vOnConnectedBTProfilesUpdate(amt_tclServiceData)
   ***************************************************************************/
   /*!
   * \fn      vOnConnectedBTProfilesUpdate(amt_tclServiceData* poMessage)
   * \brief   Method to handle Connected Bluetooth Profiles Property update from MP
   * \param   poMessage : [IN] poniter to message
   * \sa
   **************************************************************************/
   t_Void vOnConnectedBTProfilesUpdate(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnConnectResult(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     vOnConnectResult(const t_U32 cou32DevId)
   * \brief  Function to handle result of connect method start
   * \retval none
   **************************************************************************/
   t_Void vOnConnectResult(amt_tclServiceData* poMessage) const;


   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnStartiAP2OverCarPlayResult(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnStartiAP2OverCarPlayResult(amt_tclServiceData* poMessage) const
   * \brief  Function to handle result of iAP2OverCarPlay method start
   * \retval none
   **************************************************************************/
   t_Void vOnStartiAP2OverCarPlayResult(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnDipoRouteGuidanceStatus(amt_tclServiceData* poMessage) const
   * \brief  Method to handle Dipo Route Guidance Property update from MP
   * \retval none
   **************************************************************************/
   t_Void vOnDipoRouteGuidanceStatus(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnDipoRouteGuidanceManeuverStatus(amt_tclServiceData* poMessage) const
   * \brief  Method to handle Dipo Route Guidance Maneuver Property update from MP
   * \retval none
   **************************************************************************/
   t_Void vOnDipoRouteGuidanceManeuverStatus(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoGPRMCDataStatusValuesStatus(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnDipoGPRMCDataStatusValuesStatus(amt_tclServiceData* poMessage) const
   * \brief  Method to handle Dipo GPRMCDataStatusValues Property update from MP
   * \retval none
   **************************************************************************/
   t_Void vOnDipoGPRMCDataStatusValuesStatus(amt_tclServiceData* poMessage) const;
 
  /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoCommErrorStatus(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnDipoCommErrorStatus(amt_tclServiceData* poMessage) const
   * \brief  Method to handle Dipo communication error(for rolwswitch errors) from MP
   * \retval none
   **************************************************************************/
   t_Void vOnDipoCommErrorStatus(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   * Other function declarations.
   **************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::spi_tclMPlayClientHandler(..)
   ***************************************************************************/
   /*!
   * \fn     spi_tclMPlayClientHandler()
   * \brief  Default Constructor
   * \sa
   **************************************************************************/
   spi_tclMPlayClientHandler();

   /***************************************************************************
   ** FUNCTION:  spi_tclMPlayClientHandler::spi_tclMPlayClientHandler(..)
   ***************************************************************************/
   /*!
   * \fn     spi_tclMPlayClientHandler(DiPO_tclMainApp* poMainApp,
   *            spi_tclMplayResp* poConnIntf)
   * \brief  Parameterized Constructor
   * \param  poMainApp : [IN] Pointer to main app.
   * \param  poConnIntf : [IN] Pointer to Dipo connection interface
   * \sa
   **************************************************************************/
   spi_tclMPlayClientHandler(ahl_tclBaseOneThreadApp* poMainApp, spi_tclMplayResp* poConnIntf);

   /***************************************************************************
   ** FUNCTION:  virtual spi_tclMPlayClientHandler::spi_tclMPlayClientHandler& ..
   ***************************************************************************/
   /*!
   * \fn      spi_tclMPlayClientHandler& operator=(const ..)
   * \brief   Assignment operator will not be implemented. So any attempt to
   *          use the same will cought by the compiler
   * \param   oClientHandler :[IN] reference to the client handler
   * \sa
   **************************************************************************/
   spi_tclMPlayClientHandler& operator=(
             const spi_tclMPlayClientHandler& rfoClientHandler);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vPopulateMessageContext(..
   ***************************************************************************/
   /*!
   * \fn     vPopulateMessageContext(trMsgContext &rMsgContext)
   * \brief  Function to populate the message context
   * \param  rMsgContext : [OUT] Message context
   **************************************************************************/
   t_Void vPopulateMessageContext(trMsgContext &rMsgContext) const;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vPopulateDipoVersion(..
   ***************************************************************************/
   /*!
   * \fn     vPopulateDipoVersion(t_String szSource, t_U32 &u32XValue,....
   * \brief  Function to populate the DiPo version
   * \param  szSource : [IN] Source string from the device
   * \param  u32XValue : [OUT] Major version
   * \param  u32YValue : [OUT] minor version
   * \param  u32ZValue : [OUT] patch version
   **************************************************************************/
   t_Void vPopulateDipoVersion(t_String szSource, t_U32 &u32XValue,
      t_U32 &u32YValue, t_U32 &u32ZValue) const;

   /***************************************************************************
   ** FUNCTION:  tenAudioError spi_tclMPlayClientHandler::enGetAudioError(...)
   ***************************************************************************/
   /*!
   * \fn     enGetAudioError(const mplay_tFiAudError& rfcpAudioError)
   * \brief  Function to convert MPlay AudioError to SPI AudioError code
   * \param  rfcpAudioError : [IN] Mplay FI AudioError object
   **************************************************************************/
   tenAudioError enGetAudioError(const mplay_tFiAudError& rfcpAudioError) const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPowerStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnDiPOPowerStatus(amt_tclServiceData* poMessage)
   * \brief   Call back function for DiPO Power status
   * \param   poMessage : [IN] ponter to message
   * \sa
   **************************************************************************/
   t_Void vOnDiPOPowerStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  tenTBTDistanceUnits spi_tclMPlayClientHandler::enGetTBTDistanceUnits(...)
   ***************************************************************************/
   /*!
   * \fn     enGetTBTDistanceUnits(const mplay_tFiDistanceDisplayUnits& rfcpDistanceUnits)
   * \brief  Function to convert MPlay Distance units to SPI TBT Distance units
   * \param  rfcpDistanceUnits : [IN] Mplay FI Distance units
   **************************************************************************/
   tenTBTDistanceUnits enGetTBTDistanceUnits(const mplay_tFiDistanceDisplayUnits& rfcpDistanceUnits) const;

   /***************************************************************************
    ** FUNCTION:  t_U8 spi_tclMPlayClientHandler::u8GetMPDeviceTag(...)
    ***************************************************************************/
   /*!
   * \fn      u8GetMPDeviceTag(t_U32 u32DeviceHandle)
   * \brief   To read MP device Tag for provided SPI device handle
   * \param   u32DeviceHandle : [IN] SPI device handle
   **************************************************************************/
   t_U8 u8GetMPDeviceTag(t_U32 u32DeviceHandle);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnDipoRouteGuidanceStatus(amt_tclServiceData* poMessage) const
   * \brief  Method to handle Dipo Route Guidance2 Property update from MP
   * \retval none
   **************************************************************************/
   t_Void vOnDipoRouteGuidanceStatus2(amt_tclServiceData* poMessage) const;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus(amt_tclServiceData ..
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnDipoRouteGuidanceManeuverStatus(amt_tclServiceData* poMessage) const
   * \brief  Method to handle Dipo Route Guidance Maneuver2 Property update from MP
   * \retval none
   **************************************************************************/
   t_Void vOnDipoRouteGuidanceManeuverStatus2(amt_tclServiceData* poMessage) const;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclMPlayClientHandler::vPopulatePhoneCallData
    ***************************************************************************/
   /*!
    * \fn     vPopulatePhoneCallData(amt_tclServiceData* poMessage)
    * \brief  Method to populate m_rAppPhoneCallData based on call status received
    *         from MP
    * \retval none
    **************************************************************************/
   t_Void vPopulatePhoneCallData(mplay_fi_tcl_DiPOCallStateUpdate& rCallStateUpdate);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclMPlayClientHandler::vOnDipoDeviceTimeUpdate
    ***************************************************************************/
   /*!
    * \fn     vOnDipoDeviceTimeUpdate(amt_tclServiceData* poMessage)
    * \brief  Method to handle device time updates from MP
    * \retval none
    **************************************************************************/
   t_Void vOnDipoDeviceTimeUpdate(amt_tclServiceData* poMessage) const;

    //! Main application pointer
   ahl_tclBaseOneThreadApp* m_poMainApp;

   //! DiPo media metadata
   trAppMediaMetaData m_rAppMediaMetaData;

   //! DiPo  media playtime
   trAppMediaPlaytime m_rAppMediaPlaytime;

   // Phone Metadata
   trAppPhoneData m_rAppPhoneData;

   //Phonecall data
   trAppPhoneCallData m_rAppPhoneCallData;

   // DiPo Connection manager interface
   spi_tclMplayResp *m_poMplayResp;

   //!Device Tag value
   t_U32 m_u32DeviceTag;

   //! Device selection request status
   tenDeviceConnectionReq m_enDevConnReq;

   trDiscovererCbs m_rDiscovererCbs;

   trDeviceSwitcherCbs m_rDeviceSwitcherCbs;

   //Protect the usage in future
   trIAPDevInfoCallbacks m_rIAPDevInfoCallbacks;

   tfvBTProfileCallback m_fvBTProfileCallback;

   tfvGetSelectedDevSerialCallback m_fvGetSelectedDevSerialCallback;

   trTBTCallbacks m_rTBTCallbacks;

   //! CarPlay media album art image data
   trAppMediaAlbumArt m_rAppMediaAlbumArt;

   //! Message context for sending the Album art result
   MsgContext m_oAlbumArtMsgContext;

   //! Command counter for Album art request
   static t_U32 su32RequestID;
   
   trMetaDataCbs m_rMetaDataCbs;

   trDipoLocationInfoCallbacks m_rDipoLocationInfoCallbacks;

   //! Register/Unregister status variable.
   // This is added to check if registration /unregistration is already done.So that other updates are not posted.
     t_Bool m_bIsMetaDataPropertyRegistered;

   //! Key Bit mask map
   std::map<tenKeyCode, mplay_fi_tcl_e8_DiPOPlaypackAction::tenType> m_KeyActionMap;

   spi_tclMplayDeviceListHandler* m_poMplayDevList;

   tfvDeviceListUpdate m_fvDeviceListUpdate;

   //! Indicates whether MediaPlayer service is available or not
   t_Bool m_bIsMPServiceAvailable;

   //! Queue of pending route guidance component registrations
   std::queue<mplay_appcontrolfi_tclMsgDiPOSetRouteGuidanceDisplayComponentInfoMethodStart> m_queueRouteGuidanceComponentReg;

   std::queue<mplay_appcontrolfi_tclMsgDiPOSetCallControlsConfigurationMethodStart> m_qCallControlsConfig;
   
   //! Lock for m_queueRouteGuidanceComponentReg queue
   Lock m_queueRouteGuidanceComponentRegLock;

   //! Lock for m_qCallControlsConfig
   Lock m_qCallControlsConfigLock;

   //! Lock for m_rAppMediaMetaData
   Lock m_rAppMediaMetaDataLock;

   //! Lock for m_rAppPhoneData
   Lock m_rAppPhoneDataLock;

   //!Lock for m_rAppPhoneCallData
   Lock m_rAppPhoneCallDataLock;

   /*************************************************************************
   ****************************END OF PRIVATE*********************************
   *************************************************************************/

   /***************************************************************************
   * Message map definition macro
   ***************************************************************************/
   DECLARE_MSG_MAP(spi_tclMPlayClientHandler)
};

#endif // SPI_TCLMPLAY_CLIENT_HANDLER_H_
