/*!
*******************************************************************************
* \file              spi_tclBluetoothClient.cpp
* \brief             Bluetooth Settings Client handler class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Bluetooth Settings Client handler class
COPYRIGHT:      &copy; RBEI

HISTORY:
 Date       |  Author                           | Modifications
 21.02.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 21.05.2014 |  Ramya Murthy (RBEI/ECP2)         | Revised logic for storing DeviceList
                                                  and validation of service responses.
 04.06.2014 |  Ramya Murthy (RBEI/ECP2)         | VehicleBTAddress implementation
 11.09.2014 |  Shihabudheen P M (RBEI/ECP2)     | Modified for writing/reading BT address 
                                                  to Datapool    
 22.09.2014 |  Ramya Murthy (RBEI/ECP2)         | Implemented locks for DeviceList and
                                                  ChngDevRequestsList maps
 28.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Implementation for setting BluetoothAudioSource
                                                  (Fix for SUZUKI-18263)
 27.05.2015 |  Tejaswini H B(RBEI/ECP2)         | Added Lint comments to suppress C++11 Errors
 19.05.2015 |  Ramya Murthy (RBEI/ECP2)         | Adaptation to DeviceListExtended property
 08.07.2015 |  Ramya Murthy (RBEI/ECP2)         | Added check to ensure BluetoothAudioSource.Set(ValidDevHandle)
                                                  is used to enable A2DP profile
 18.08.2015 |  Ramya Murthy (RBEI/ECP2)         | Adaptation to new BTSettings interfaces for AndroidAuto
 
 13.02.2018 | Ramesh S M (RBEI/ECO2)            | Implementation of Out of Band Pairing
 07.08.2018 | Dundamma S B (RBEI/ECO2)          | Service ID: u16ServiceID member is removed and "MOST_BTSETFI_C_U16_SERVICE_ID" from connectivity FI is added in constructor
 01.12.2018 | Dundamma S B (RBEI/ECO2)          | Callback trVehicleBTAddressCallback is used to write VehicleBTAddress to SPIDtapool inside function vOnVehicleBTAddressStatus


\endverbatim
******************************************************************************/

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include <algorithm>

#include "XFiObjHandler.h"
#include "spi_tclBluetoothClient.h"
#include "spi_LoopbackTypes.h"
#include "Datapool.h"

//For message dispatcher
#include "FIMsgDispatch.h"
using namespace shl::msgHandler;

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BLUETOOTH
      #include "trcGenProj/Header/spi_tclBluetoothClient.cpp.trc.h"
   #endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
//! BT Settings types
typedef most_fi_tcl_BTSetDeviceListExtendedResultItem          btset_DeviceListExtItem;
typedef most_fi_tcl_e8_BTSetDeviceStatus                       btset_tFiDevStatusType;
typedef most_fi_tcl_e8_BTSetDeviceStatus::tenType              btset_tenDevStatusType;

typedef most_fi_tcl_e8_BTSetDeviceDisconnectedReason::tenType  btset_tenDevDisconnReasonType;

typedef most_fi_tcl_e8_BTSetPairingResponseType::tenType       btset_tenBTPairingResponseType;
typedef most_fi_tcl_e8_BTTechnology::tenType                   btset_tenBTTechnology;
typedef most_fi_tcl_e8_BTCommunicationChannel::tenType         btset_tenBTCommChannel;
typedef most_fi_tcl_e8_BTLimitationAction::tenType             btset_tenBTLimAction;
typedef most_fi_tcl_e8_BTLimitationActionState::tenType        btset_tenBTLimActionState;
typedef most_fi_tcl_BTLimitationModeListItem                   btset_BTLimModeListItem;
typedef most_fi_tcl_e8_BTResolutionType::tenType			   btset_tenResolutionType;
typedef most_fi_tcl_e8_BTConflictTrigger::tenType			   btset_tenConflictTrigger;
typedef most_fi_tcl_BTLimitationMode                           btset_BTLimitationMode;
typedef most_fi_tcl_BTConflictInfo                             btset_ConflictsInfo;

typedef most_btsetfi_tclMsgPairingResponseMethodStart             btset_tMSPairingResponse;
typedef most_btsetfi_tclMsgSetBluetoothLimitationModeMethodStart  btset_tMSSetBTLimMode;
typedef most_btsetfi_tclMsgResolveConflictsMethodStart         tMSResolveConflict;
typedef most_btsetfi_tclMsgReplaceBluetoothLimitationModeMethodStart btset_tMSReplaceBTLimMode;

typedef XFiObjHandler<most_btsetfi_tclMsgDeviceListExtendedStatus>       btset_tXFiStDeviceListExt;
typedef XFiObjHandler<most_btsetfi_tclMsgPairingStatusStatus>            btset_tXFiStPairingStatus;
typedef XFiObjHandler<most_btsetfi_tclMsgVehicleBTAddressStatus>         btset_tXFiStVehicleBTAddr;
typedef XFiObjHandler<most_btsetfi_tclMsgSwitchBTLocalModeError>         btset_tXFiErrSwitchBTLocMode;
typedef XFiObjHandler<most_btsetfi_tclMsgPairingRequestStatus>           btset_tXFiStPairingRequest;
typedef XFiObjHandler<most_btsetfi_tclMsgVehicleBTFriendlyNamesetStatus>    btset_tXFiStVehicleBTFriendlyName;


typedef XFiObjHandler<most_btsetfi_tclMsgBluetoothLimitationModeStatus>             btset_tXFiStBTLimMode;
typedef XFiObjHandler<most_btsetfi_tclMsgSetBluetoothLimitationModeMethodResult>    btset_tXFiMRSetBTLimMode;
typedef XFiObjHandler<most_btsetfi_tclMsgSetBluetoothLimitationModeError>           btset_tXFiErrSetBTLimMode;
typedef XFiObjHandler<most_btsetfi_tclMsgResolveConflictsMethodResult>              btset_tXFiResolveConflitsMethodResult;
typedef XFiObjHandler<most_btsetfi_tclMsgResolveConflictsError>                     btset_tXFiResolveConflitsError;
typedef XFiObjHandler<most_btsetfi_tclMsgReplaceBluetoothLimitationModeMethodResult>   btset_tXFiMRReplaceBTLimMode;
typedef XFiObjHandler<most_btsetfi_tclMsgReplaceBluetoothLimitationModeError>          btset_tXFiErrReplaceBTLimMode;
typedef XFiObjHandler<most_btsetfi_tclMsgConflictsDetectedStatus>           btset_tXFiStConflictsDetected;
typedef XFiObjHandler<most_btsetfi_tclMsgServiceAvailableStatus>           btset_tXFiStServiceAvailable;

//! BT Settings FBlock DeviceList vector typedef
typedef std::vector<btset_DeviceListExtItem, std::allocator<btset_DeviceListExtItem> >::iterator
         btset_tFiDeviceListExtItr;
typedef std::vector<btset_BTLimModeListItem, std::allocator<btset_BTLimModeListItem> >::iterator
         btset_BTLimModeListItr;
//! Conflicts Information
typedef std::vector<btset_ConflictsInfo, std::allocator<btset_ConflictsInfo> >::iterator
         btset_ConflictsListItr;

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/
#define BTSETTINGS_FI_MAJOR_VERSION  MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION
#define BTSETTINGS_FI_MINOR_VERSION  MOST_BTSETFI_C_U16_SERVICE_MINORVERSION
//#define BTSETTINGS_FI_PATCH_VERSION  0                                         //Commented to suppress Lint Fix

//! Bluetooth Settings FBlock types
#define BTDEV_DELETED                 (btset_tFiDevStatusType::FI_EN_E8DEVICE_DELETED)
#define CHAR_SET_UTF8                 (most_fi_tcl_String::FI_EN_UTF8)

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const t_U8 cou8InvalidDeviceHandle = 0;


/******************************************************************************
** CCA MESSAGE MAP
******************************************************************************/
BEGIN_MSG_MAP(spi_tclBluetoothClient, ahl_tclBaseWork)

   /* -------------------------------Methods.---------------------------------*/
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnStatusDeviceListExtended)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_PAIRINGSTATUS,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnStatusPairingStatus)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_VEHICLEBTADDRESS,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnVehicleBTAddressStatus)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_PAIRINGREQUEST,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnStatusPairingRequest)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAMESET,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnStatusVehicleBTFriendlyName)


   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_BLUETOOTHLIMITATIONMODE,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnStatusBTLimitationMode)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_SETBLUETOOTHLIMITATIONMODE,
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnMRSetBTLimitationMode)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_SETBLUETOOTHLIMITATIONMODE,
   AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnErrorSetBTLimitationMode)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_CONFLICTSDETECTED,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnConflictsDetectedStatus)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_RESOLVECONFLICTS,
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnMRResolveConflicts)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_RESOLVECONFLICTS,
   AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnErrorResolveConflicts)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_REPLACEBLUETOOTHLIMITATIONMODE,
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnMRReplaceBTLimitationMode)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_REPLACEBLUETOOTHLIMITATIONMODE,
   AMT_C_U8_CCAMSG_OPCODE_ERROR, vOnErrorReplaceBTLimitationMode)
   ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_SERVICEAVAILABLE,
   AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnStatusServiceAvailable)

   /* -------------------------------Methods.---------------------------------*/

END_MSG_MAP();


/***************************************************************************
*********************************PUBLIC*************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::spi_tclBluetoothClient(...)
**************************************************************************/
spi_tclBluetoothClient::spi_tclBluetoothClient(ahl_tclBaseOneThreadApp* poMainAppl)
     : ahl_tclBaseOneThreadClientHandler(
         poMainAppl,                           /* Application Pointer */
         MOST_BTSETFI_C_U16_SERVICE_ID,        /* ID of used Service */
         BTSETTINGS_FI_MAJOR_VERSION,          /* MajorVersion of used Service */
         BTSETTINGS_FI_MINOR_VERSION),         /* MinorVersion of used Service */
         m_poMainAppl(poMainAppl),
         m_bIsBTServiceAvailable(false),
         m_enConflictState(e8CONFLICTS_STATE_INVALID),
         m_bOOBConflictProcessed(false),
         m_enBTServiceAvailableStatus(e8SERVICE_UNAVAILABLE)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient() entered "));
   NORMAL_M_ASSERT(nullptr != m_poMainAppl);
}//! end of spi_tclBluetoothClient()

/***************************************************************************
** FUNCTION:  virtual spi_tclBluetoothClient::~spi_tclBluetoothClient()
**************************************************************************/
spi_tclBluetoothClient::~spi_tclBluetoothClient()
{
   ETG_TRACE_USR1(("~spi_tclBluetoothClient() entered "));


   m_oDevLstLock.s16Lock();
   m_mapBTDeviceList.clear();
   m_oDevLstLock.vUnlock();
   
   m_poMainAppl = nullptr;

}//! end of spi_tclBluetoothClient()

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnServiceAvailable()
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnServiceAvailable()
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnServiceAvailable() entered "));

   m_oBTAvailLock.s16Lock();
   m_bIsBTServiceAvailable = true;
   m_oBTAvailLock.vUnlock();

}//! end of spi_tclBluetoothClient()

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnServiceUnavailable()
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnServiceUnavailable()
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnServiceUnavailable() "));

   m_oBTAvailLock.s16Lock();
   m_bIsBTServiceAvailable = false;
   m_oBTAvailLock.vUnlock();

}//! end of spi_tclBluetoothClient()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterForProperties()
**************************************************************************/
t_Void spi_tclBluetoothClient::vRegisterForProperties()
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vRegisterForProperties() "));

   //! Register for interested properties
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_PAIRINGSTATUS);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLEBTADDRESS);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_PAIRINGREQUEST);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAMESET);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_BLUETOOTHLIMITATIONMODE);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_CONFLICTSDETECTED);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_SERVICEAVAILABLE);
}//! end of vRegisterForProperties()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vUnregisterForProperties()
**************************************************************************/
t_Void spi_tclBluetoothClient::vUnregisterForProperties()
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vUnregisterForProperties() "));

   //@TODO: Currently this function is unused. If required, this function MUST be called from ExtIntf only!!!
   //! Unregister subscribed properties
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED);
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_PAIRINGSTATUS);
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLEBTADDRESS);
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_PAIRINGREQUEST);
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLEBTFRIENDLYNAMESET);
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_BLUETOOTHLIMITATIONMODE);
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_CONFLICTSDETECTED);
   vRemoveAutoRegisterForProperty(MOST_BTSETFI_C_U16_SERVICEAVAILABLE);
}//! end of vUnregisterForProperties()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclBluetoothClient::vRegisterCallbacks(trBluetoothCallbacks rBTRespCallbacks)
{
   //! Store the callbacks
   m_rBluetoothCallbacks = rBTRespCallbacks;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclBluetoothClient::vRegisterOOBTCallbacks(trOOBTPairingCallbacks rOOBTRespCallbacks)
{
   //! Store the callbacks
   m_rOOBTCallbacks = rOOBTRespCallbacks;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothPolicyBase::vRegisterPairingInfoCallbacks()
***************************************************************************/
t_Void spi_tclBluetoothClient::vRegisterPairingInfoCallbacks(
      trBluetoothPairingCallbacks rBTPairInfoCallbacks)
{
   m_rBTPairInfoCallbacks = rBTPairInfoCallbacks;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclBluetoothClient::vRegisterDeviceNameCallback(
      trBTDeviceNameCbInfo rBTDeviceNameCbInfo)
{
   //! Store callback info
   m_rBTDeviceNameCbInfo = rBTDeviceNameCbInfo;
}

/***************************************************************************
 ** FUNCTION: spi_tclBluetoothClient::vRegisterBTMacAddressCallbacks
 ***************************************************************************/
t_Void spi_tclBluetoothClient::vRegisterBTMacAddressCallbacks(trVehicleBTAddressCallback &rVehicleBTAddress)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vRegisterBTMacAddressCallbacks() entered"));
   m_rVehicleBTAddress = rVehicleBTAddress;
}


/***************************************************************************
** FUNCTION:  t_Bool spi_tclBluetoothClient::bGetPairingStatus(const...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bGetPairingStatus(const t_String& rfcoszDeviceBTAddress)
{
   //! If device exists in BT DeviceList map, device is already paired.
   //! Else pairing is required.
   t_Bool bDevicePaired =
         (cou8InvalidDeviceHandle != u8GetBTDeviceHandle(rfcoszDeviceBTAddress));

   ETG_TRACE_USR2(("spi_tclBluetoothClient::bGetPairingStatus() left with: %u (for BTAddress = %s) ",
         ETG_ENUM(BOOL, bDevicePaired),
         rfcoszDeviceBTAddress.c_str()));

   return bDevicePaired;

}//! end of bGetPairingStatus()

/***************************************************************************
** FUNCTION:  t_U32 spi_tclBluetoothClient::u32GetBTDeviceHandle(t_String...)
***************************************************************************/
t_U32 spi_tclBluetoothClient::u32GetBTDeviceHandle(
      const t_String& rfcoszDeviceBTAddress)
{
   //! Retrieve and send BT device handle
   return u8GetBTDeviceHandle(rfcoszDeviceBTAddress);
}//! end of u32GetBTDeviceHandle()

/***************************************************************************
** FUNCTION:  t_String spi_tclBluetoothClient::szGetBTDeviceName()
***************************************************************************/
t_String spi_tclBluetoothClient::szGetBTDeviceName(const t_String& rfcszDeviceBTAddress)
{
   t_String szDeviceName = "";

   if (true == IS_VALID_BT_ADDRESS(rfcszDeviceBTAddress))
   {
      m_oDevLstLock.s16Lock();

      //! Search for required device in BT DeviceList Map.
      for (tBTDevListMapConstItr DevLstMapCoItr = m_mapBTDeviceList.begin();
           DevLstMapCoItr != m_mapBTDeviceList.end();
           ++DevLstMapCoItr)
      {
         if (rfcszDeviceBTAddress == (DevLstMapCoItr->second).szBTAddress)
         {
            szDeviceName = (DevLstMapCoItr->second).szDeviceName;
            break;
         }
      } //for (tBTDevListMapConstItr DevLstMapCoItr = ...)

      m_oDevLstLock.vUnlock();
   }

   return szDeviceName;
}

/***************************************************************************`
** FUNCTION:  t_Bool spi_tclBluetoothClient::bGetBTServiceAvailability(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bGetBTServiceAvailability()
{
   m_oBTAvailLock.s16Lock();
   t_Bool bIsBTServiceAvailable = m_bIsBTServiceAvailable;
   m_oBTAvailLock.vUnlock();

   ETG_TRACE_USR1(("spi_tclBluetoothClient::bGetBTServiceAvailability: %d ", ETG_ENUM(BOOL, bIsBTServiceAvailable)));
   return bIsBTServiceAvailable;
}//! end of bGetBTServiceAvailability()

/***************************************************************************
** FUNCTION:  t_String spi_tclBluetoothClient::szGetBTDeviceName()
***************************************************************************/
t_String spi_tclBluetoothClient::szGetVehicleBTFriendlyName()
{
	ETG_TRACE_USR1(("spi_tclBluetoothClient::szGetVehicleBTFriendlyName returns Vhicle BT friendly name = %s ", m_szVehicleBTFriendlyName.c_str()));
	return m_szVehicleBTFriendlyName;
}

/***************************************************************************`
** FUNCTION:  t_Bool spi_tclBluetoothClient::bSetBTLimitationMode(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bSetBTLimitationMode(const t_String& rfcszBTAddress,
      const t_String& rfcszBTDeviceName,
      tenBTTechnology enTechnology,
      tenBTCommunicationChannel enCommChannel,
      tenBTLimitationAction enAction)
{
   btset_tMSSetBTLimMode oMSSetBTLimMode;
   t_Bool bRetVal = oMSSetBTLimMode.sBdAddress.bSet(rfcszBTAddress.c_str(), CHAR_SET_UTF8);
   t_Bool bRetDeviceName = oMSSetBTLimMode.sBdDeviceName.bSet(rfcszBTDeviceName.c_str(), CHAR_SET_UTF8);

   ETG_TRACE_USR4(("spi_tclBluetoothClient::bSetBTLimitationMode(): bSet returns value: %d",
         ETG_ENUM(BOOL, bRetVal)));

   ETG_TRACE_USR4(("spi_tclBluetoothClient::bSetBTLimitationMode(): bSet returns value: %d",
         ETG_ENUM(BOOL, bRetDeviceName)));

//This is a workaround for AIVI Bugs NCG3D-82250. Issuse is AAP does not auto pair for the first connection
//For AAP ,phone can send pairing request multiple times to recover from pairing/connection error and HU should send response to every request.So SetBTLimitationMode should be sent to BT Service even though it is already active so that BT Service can also recover if needed. 	 
   if(enTechnology != e8BT_TECH_ANDROIDAUTO)
   {
      t_Bool bIsSetBTLimRequired = vHandleLimitationMode(rfcszBTAddress, rfcszBTDeviceName, enTechnology, enCommChannel, enAction);

      if (false == bIsSetBTLimRequired)
      {
         ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::bSetBTLimitationMode - SetBTLimMode not required."));
         return true;
      }
   } 
   oMSSetBTLimMode.e8Technology.enType = static_cast<btset_tenBTTechnology>(enTechnology);
   oMSSetBTLimMode.e8CommunicationChannel.enType = static_cast<btset_tenBTCommChannel>(enCommChannel);
   oMSSetBTLimMode.e8Action.enType = static_cast<btset_tenBTLimAction>(enAction);

   t_Bool bSendSuccess = false;
   m_oBTAvailLock.s16Lock();
   // SetLimitationMode with Deactivate action can be sent even in restricted mode (i.e. either e8SERVICE_AVAILABLE_RESTRICTED or e8SERVICE_AVAILABLE state) 
   if (((e8BT_LIM_ACTION_DEACTIVATE == enAction) || (e8BT_LIM_ACTION_SPM_STATE_OFF == enAction)) &&
        (true == m_bIsBTServiceAvailable) && (e8SERVICE_UNAVAILABLE != m_enBTServiceAvailableStatus))
   {
      bSendSuccess = bPostMethodStart(oMSSetBTLimMode, 0);
   }
   // For other actions like Prepare, Preactivate and Activate, BT SPM should be in NORMAL mode i.e. e8SERVICE_AVAILABLE state
   else if((true == m_bIsBTServiceAvailable) && (e8SERVICE_AVAILABLE == m_enBTServiceAvailableStatus))
   {
      bSendSuccess = bPostMethodStart(oMSSetBTLimMode, 0);
   }
   else
   {
      ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::bSetBTLimitationMode- BT service unavailable"));
   }
   m_oBTAvailLock.vUnlock();

   ETG_TRACE_USR4(("spi_tclBluetoothClient::bSetBTLimitationMode() left with "
         "bSendSuccess = %u (for Technology = %d, CommChannel = %d, Action = %d, BTAddress = %s ",
         ETG_ENUM(BOOL, bSendSuccess),
         ETG_ENUM(BTSET_TECHNOLOGY, enTechnology),
         ETG_ENUM(BTSET_COMM_CHN, enCommChannel),
         ETG_ENUM(BTSET_LIMIT_ACTION, enAction),
         rfcszBTAddress.c_str()));

   return bSendSuccess;
}//! end of bSetBTLimitationMode()

/***************************************************************************`
** FUNCTION:  t_Bool spi_tclBluetoothClient::bSetBTPairingResponse(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bSetBTPairingResponse(tenPairingResponseType enBTPairingResponse)
{
   btset_tMSPairingResponse oMSPairingResponse;
   oMSPairingResponse.e8PairingResponseType.enType = static_cast<btset_tenBTPairingResponseType>(enBTPairingResponse);

   t_Bool bSendSuccess = bPostMethodStart(oMSPairingResponse, 0);

   ETG_TRACE_USR4(("spi_tclBluetoothClient::bSetBTPairingResponse : bSendSuceess = [%d], BT Pairing Response Type = [%d]",
          ETG_ENUM(BOOL, bSendSuccess),
          ETG_ENUM(BTSET_PAIRING_RESPONSE_TYPE, enBTPairingResponse)));

   return bSendSuccess;
}
/***************************************************************************`
** FUNCTION:  t_Bool spi_tclBluetoothClient::bReplaceBTLimitationMode(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bReplaceBTLimitationMode(
        trBTLimitationModeInfo rCurrentBTLimitationMode,
        trBTLimitationActionInfo rNewBTLimitationMode)
{
   btset_tMSReplaceBTLimMode oMSReplaceBTLimMode;

   oMSReplaceBTLimMode.oLimitationModeToBeReplaced.sBdAddress.bSet(rCurrentBTLimitationMode.szBTAddress.c_str(), CHAR_SET_UTF8);
   oMSReplaceBTLimMode.oLimitationModeToBeReplaced.sBdDeviceName.bSet(rCurrentBTLimitationMode.szBTDeviceName.c_str(), CHAR_SET_UTF8);
   oMSReplaceBTLimMode.oLimitationModeToBeReplaced.e8Technology.enType = static_cast<btset_tenBTTechnology>(rCurrentBTLimitationMode.enTechnology);
   oMSReplaceBTLimMode.oLimitationModeToBeReplaced.e8CommunicationChannel.enType = static_cast<btset_tenBTCommChannel>(rCurrentBTLimitationMode.enCommChannel);
   oMSReplaceBTLimMode.oLimitationModeToBeReplaced.e8ActionState.enType = static_cast<btset_tenBTLimActionState>(rCurrentBTLimitationMode.enActionState);

   oMSReplaceBTLimMode.oNewLimitationMode.sBdAddress.bSet(rNewBTLimitationMode.szBTAddress.c_str(), CHAR_SET_UTF8);
   oMSReplaceBTLimMode.oNewLimitationMode.sBdDeviceName.bSet(rNewBTLimitationMode.szBTDeviceName.c_str(), CHAR_SET_UTF8);
   oMSReplaceBTLimMode.oNewLimitationMode.e8Technology.enType = static_cast<btset_tenBTTechnology>(rNewBTLimitationMode.enTechnology);
   oMSReplaceBTLimMode.oNewLimitationMode.e8CommunicationChannel.enType = static_cast<btset_tenBTCommChannel>(rNewBTLimitationMode.enCommChannel);
   oMSReplaceBTLimMode.oNewLimitationMode.e8Action.enType = static_cast<btset_tenBTLimAction>(rNewBTLimitationMode.enAction);

   ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::bReplaceBTLimitationMode - Current Limitation Mode Info:"
       "Technology = %d, Communication channel = %d, Action State = %d, BT Address = %s",
       ETG_ENUM(BTSET_TECHNOLOGY, rCurrentBTLimitationMode.enTechnology), ETG_ENUM(BTSET_COMM_CHN, rCurrentBTLimitationMode.enCommChannel),
       ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurrentBTLimitationMode.enActionState), rCurrentBTLimitationMode.szBTAddress.c_str()));

   ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::bReplaceBTLimitationMode - New Limitation Mode Info:"
       "Technology = %d, Communication channel = %d, Action State = %d, BT Address = %s",
       ETG_ENUM(BTSET_TECHNOLOGY, rNewBTLimitationMode.enTechnology), ETG_ENUM(BTSET_COMM_CHN, rNewBTLimitationMode.enCommChannel),
       ETG_ENUM(BTSET_LIMIT_ACTION, rNewBTLimitationMode.enAction),rNewBTLimitationMode.szBTAddress.c_str()));

   ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::bReplaceBTLimitationMode - New Limitation Mode Info:"
          "BTDeviceName = %s",rNewBTLimitationMode.szBTDeviceName.c_str()));

   t_Bool bResult = bPostMethodStart(oMSReplaceBTLimMode, 0);

   ETG_TRACE_USR4(("spi_tclBluetoothClient::bReplaceBTLimitationMode() left with result = %u", ETG_ENUM(BOOL, bResult)));

   return bResult;
}//! end of bReplaceBTLimitationMode()

/***************************************************************************
*********************************PRIVATE************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnStatusDeviceListExtended(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnStatusDeviceListExtended(amt_tclServiceData* poMessage)
{
   //! Extract the MethodResult msg details
   btset_tXFiStDeviceListExt oXFiDevListExtStatus(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);

   if (true == oXFiDevListExtStatus.bIsValid())
   {
      tU8 u8ListChangeDevHandle = oXFiDevListExtStatus.oDeviceListChange.u8DeviceHandle;
      btset_tenDevStatusType enListChangeDevStatus = oXFiDevListExtStatus.oDeviceListChange.e8DeviceStatus.enType;
      btset_tenDevDisconnReasonType enListChangeDevDisconnReason =
               oXFiDevListExtStatus.oDeviceListChange.e8DeviceDisconnectedReason.enType;

      //! Print Device List details
      ETG_TRACE_USR4(("spi_tclBluetoothClient::vOnStatusDeviceListExtended: NumPairedDevices = %u,"
         "DeviceListChange DeviceHandle = 0x%x, DeviceStatus = %u, DisconnectionReason = %u ", oXFiDevListExtStatus.u8NumPairedDevices, u8ListChangeDevHandle, ETG_ENUM(BTSET_DEVICE_STATUS,
               enListChangeDevStatus), ETG_ENUM(BTSET_DEVICEDISCONN_REASON, enListChangeDevDisconnReason)));

      //Check if the number of paired devices is zero.
      if (0 == oXFiDevListExtStatus.u8NumPairedDevices)
      {
         ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnStatusDeviceListExtended: Number of paired devices is zero"));
         vClearBTDeviceList();
      }

      else
      {
         //! Handle device deletion first, since deleted device will not be part of device list.
         if (BTDEV_DELETED == enListChangeDevStatus)
         {
            vRemoveFromDeviceList(u8ListChangeDevHandle);
         }
         //! Validate each device in list
         for (btset_tFiDeviceListExtItr LstItr = oXFiDevListExtStatus.oDeviceListExtendedResult.oItems.begin(); LstItr
                  != oXFiDevListExtStatus.oDeviceListExtendedResult.oItems.end(); ++LstItr)
         {
            //!---------------------------------------------------------------------------------------
            // @Note:
            // 1. BTSettings FBlock sends DeviceList property update when a device is connected,
            // disconnected or deleted with DeviceListChange as CONNECTED_OSD/DISCONNECTED/DELETED
            // respectively.
            // DeviceListChange "NO_CHANGE" will (probably) only be sent during property registration.
            // 2. BTSettings FBlock does not send DeviceList property update when a device is paired.
            // (That info can be checked with PairingStatus update)
            //!---------------------------------------------------------------------------------------
            ETG_TRACE_USR4(("spi_tclBluetoothClient::vOnStatusDeviceListExtended: Bt Service type HFP : %d "
                  "A2DP : %d ", ETG_ENUM(BOOL,(*LstItr).oDeviceProfileStatus.bHFP),
                  ETG_ENUM(BOOL,(*LstItr).oDeviceProfileStatus.bAVP)));

            tU8 u8LstItemDevHandle = (*LstItr).u8DeviceHandle;
            tBool bIsNewDevice = (FALSE == bIsDeviceValid(u8LstItemDevHandle));

            // @Note: OutgoingSourceDeviceStatus is used since SPI is interested in only HFP profile connection
            tBool bLstItemDevConnStatus = (*LstItr).bOutgoingSourceDeviceStatus;

            t_String szLstItemDeviceName = "";
            GET_STRINGDATA_FROM_FI_STRINGOBJ((*LstItr).sDeviceName, CHAR_SET_UTF8, szLstItemDeviceName);

            //! If device already exists in DeviceList Map, update its details in the map.
            if (FALSE == bIsNewDevice)
            {
                  vUpdateDeviceName(u8LstItemDevHandle, szLstItemDeviceName);

               if ((u8LstItemDevHandle == u8ListChangeDevHandle) && (BTDEV_DELETED != enListChangeDevStatus))
               {
                  vUpdateDeviceConnStatus(u8LstItemDevHandle, bLstItemDevConnStatus, 
                  (*LstItr).oDeviceProfileStatus.bHFP,
                  (*LstItr).oDeviceProfileStatus.bAVP);
               }
            }//if (FALSE == bIsNewDevice)
            //! If there is a new device in list, add it to DeviceList map.
            else
            {
               t_String szLstItemBTAddress = "";
               GET_STRINGDATA_FROM_FI_STRINGOBJ((*LstItr).sDeviceAddress, CHAR_SET_UTF8, szLstItemBTAddress);
               std::transform(szLstItemBTAddress.begin(),
                        szLstItemBTAddress.end(),
                        szLstItemBTAddress.begin(),
                        ::toupper);

               //! Print details of new device in list.
               ETG_TRACE_USR4(("spi_tclBluetoothClient::vOnStatusDeviceListExtended: DeviceListResult info: "
                  "DeviceHandle = 0x%x, DateTimeStamp = %s ", u8LstItemDevHandle, szGetDateTimestamp((*LstItr).oDateTimeStamp).c_str()));

               vAddToDeviceList(u8LstItemDevHandle, bLstItemDevConnStatus, szLstItemDeviceName, 
                    szLstItemBTAddress,(*LstItr).oDeviceProfileStatus.bHFP,(*LstItr).oDeviceProfileStatus.bAVP);
               // @Note: Need to add device here (and not only on PairingStatus update), due to below scenarios:
               // a) Startup scenario (Since during startup only DeviceList update would be received.
               // PairingStatus will not be sent for already paired devices)
               // b) In the event that DeviceList update (for device connection) is received
               // before PairingStatus update is received (for a newly paired device).

               //! Notify connected device
               if (true == bLstItemDevConnStatus)
               {
                  vNotifyBTConnectionChanged(szLstItemBTAddress, e8BT_RESULT_CONNECTED);
               }

               vNotifyBTDeviceName(szLstItemBTAddress, szLstItemDeviceName);
               
               vNotifyBTProfileStatus();
            }
         }//for (btset_tFiDeviceListExtItr LstItr = ...)
      }//else
   }//if (true == oXFiDevListExtStatus.bIsValid())//else if (TRUE == bIsNewDevice)
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnStatusDeviceListExtended: Invalid message received! "));
   }

}//! end of vOnStatusDeviceListExtended()


/**************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnStatusPairingStatus(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::
      vOnStatusPairingStatus(amt_tclServiceData* poMessage)
{
   btset_tXFiStPairingStatus oXFiPairingStatus(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);

   if (true == oXFiPairingStatus.bIsValid())
   {
      tenBTSetPairingStatus enPairingStatus = static_cast<tenBTSetPairingStatus>(
            oXFiPairingStatus.e8PairingStatusType.enType);

      //! Print Pairing Status details
      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnStatusPairingStatus() entered: "
            "DeviceHandle = 0x%x, PairingStatusType = %u ",
            oXFiPairingStatus.u8DeviceHandle,
            ETG_ENUM(BTSET_PAIRING_STATUS, enPairingStatus)));

      t_String szBTDeviceAddress = szGetBTDeviceAddress(oXFiPairingStatus.u8DeviceHandle);

      if(NULL != m_rBluetoothCallbacks.fvOnBTPairingStatus)
      {
         m_rBluetoothCallbacks.fvOnBTPairingStatus(enPairingStatus, szBTDeviceAddress);
      }
   } //if (true == oXFiPairingStatus.bIsValid())
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnStatusPairingStatus: Invalid message received! "));
   }

}//! end of vOnStatusPairingStatus()

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnVehicleBTAddressStatus(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::
   vOnVehicleBTAddressStatus(amt_tclServiceData* poMessage)
{
   //! Extract the status msg
   btset_tXFiStVehicleBTAddr oXFiVehBTAddrMsg(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   tU16 u16ServiceID = CCA_C_U16_SRV_SMARTPHONEINTEGRATION;
   //! Retrieve Vehicle BT Address
   if (true == oXFiVehBTAddrMsg.bIsValid())
   {
      t_String szVehicleBTAddr = "";
      GET_STRINGDATA_FROM_FI_STRINGOBJ(
            oXFiVehBTAddrMsg.sVehicleBTAddress, CHAR_SET_UTF8, szVehicleBTAddr);

      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnVehicleBTAddressStatus() entered: ServiceID = %d Vehicle BTAddress = %s ",
            u16ServiceID, szVehicleBTAddr.c_str()));

      //! Store the address
      if ((false == szVehicleBTAddr.empty()) && (nullptr != m_poMainAppl))
      {
    	  //Writing the data to SPI datapool for future use
    	  if (NULL != m_rVehicleBTAddress.fvOnVehicleBTAddress)
      	   {
      	      (m_rVehicleBTAddress.fvOnVehicleBTAddress)(szVehicleBTAddr);
      	      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnVehicleBTAddressStatus() BT MAC Address from Callback: %s",szVehicleBTAddr.c_str()));
      	   }

          tLbOnVehicleBTAdressUpdate
            oBtAddressUpdate(m_poMainAppl->u16GetAppId(), // Source AppID
                           m_poMainAppl->u16GetAppId(),
                           0,
                           0,
                           u16ServiceID,
                           SPI_C_U16_IFID_VEHICLE_BTADDRESS_UPDATE,
                           AMT_C_U8_CCAMSG_OPCODE_STATUS,
                           static_cast<t_U32>((strlen(szVehicleBTAddr.c_str()) + 1)));

          //! Set the data & post message
          oBtAddressUpdate.vSetData((const tChar*)(szVehicleBTAddr.c_str()));

          if (true == oBtAddressUpdate.bIsValid())
          {
             if (AIL_EN_N_NO_ERROR != m_poMainAppl->enPostMessage(&oBtAddressUpdate, TRUE))
             {
                ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnVehicleBTAddressStatus() Loopback message posting failed! "));
             }
          } //if (true == oAudioDevLbMsg.bIsValid())
          else
          {
              ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnVehicleBTAddressStatus(): Loopback message creation failed! "));
          }
      }
   } //if (true == oXFiVehBTAddrMsg.bIsValid())
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnVehicleBTAddressStatus: Invalid message received! "));
   }

}//! end of vOnVehicleBTAddressStatus()

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnMRSetBTLimitationMode(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnMRSetBTLimitationMode(amt_tclServiceData* poMessage)
{
   //! Extract the msg
   btset_tXFiMRSetBTLimMode oXFiSetBTLimMode(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   if (true == oXFiSetBTLimMode.bIsValid())
   {
      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnMRSetBTLimitationMode() entered: Result = %d ",
            ETG_ENUM(BOOL, oXFiSetBTLimMode.bResult)));

      t_Bool bSetBTLimitationModeResult = oXFiSetBTLimMode.bResult;
      if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
    	  m_rBluetoothCallbacks.fvOnBTLimitationModeResult(bSetBTLimitationModeResult);
      }
   }
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnMRSetBTLimitationMode: Invalid message received! "));
   }
}//! end of vOnMRSetBTLimitationMode()

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnErrorSetBTLimitationMode(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnErrorSetBTLimitationMode(amt_tclServiceData* poMessage)
{
   //! Extract the msg
   btset_tXFiErrSetBTLimMode oXFiSetBTLimModeErr(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   if (true == oXFiSetBTLimModeErr.bIsValid())
   {
      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnErrorSetBTLimitationMode() entered "));
      if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
    	  m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   }
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnErrorSetBTLimitationMode: Invalid message received! "));
   }
}//! end of vOnErrorSetBTLimitationMode()

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnMRReplaceBTLimitationMode(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnMRReplaceBTLimitationMode(amt_tclServiceData* poMessage)
{
   //! Extract the msg
   btset_tXFiMRReplaceBTLimMode oXFiReplaceBTLimMode(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   if (true == oXFiReplaceBTLimMode.bIsValid())
   {
      ETG_TRACE_USR1(("[DESC}: spi_tclBluetoothClient::vOnMRReplaceBTLimitationMode() entered: Result = %d",
            ETG_ENUM(BOOL, oXFiReplaceBTLimMode.bResult)));

      t_Bool bReplaceBTLimitationModeResult = oXFiReplaceBTLimMode.bResult;
      if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
        m_rBluetoothCallbacks.fvOnBTLimitationModeResult(bReplaceBTLimitationModeResult);
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]: spi_tclBluetoothClient::vOnMRReplaceBTLimitationMode: Invalid message received!"));
   }
}//! end of vOnMRReplaceBTLimitationMode()

/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnErrorReplaceBTLimitationMode(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnErrorReplaceBTLimitationMode(amt_tclServiceData* poMessage)
{
   //! Extract the msg
   btset_tXFiErrReplaceBTLimMode oXFiReplaceBTLimModeErr(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   if (true == oXFiReplaceBTLimModeErr.bIsValid())
   {
      ETG_TRACE_USR1(("[DESC]: spi_tclBluetoothClient::vOnErrorReplaceBTLimitationMode() entered"));
      if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
        m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]: spi_tclBluetoothClient::vOnErrorReplaceBTLimitationMode: Invalid message received!"));
   }
}//! end of vOnErrorSetBTLimitationMode()


/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnStatusPairingRequest(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnStatusPairingRequest(amt_tclServiceData* poMessage)
{
   btset_tXFiStPairingRequest oXFiPairingReq(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   if (true == oXFiPairingReq.bIsValid())
   {
      //! Extract info from msg.
      trBTPairingRequestInfo rPairingReqInfo;
      t_String szRemoteDevName;

      GET_STRINGDATA_FROM_FI_STRINGOBJ(
            oXFiPairingReq.sRemoteDeviceAddress, CHAR_SET_UTF8, rPairingReqInfo.szRemoteDevBTAddr);
      std::transform(rPairingReqInfo.szRemoteDevBTAddr.begin(), rPairingReqInfo.szRemoteDevBTAddr.end(),
            rPairingReqInfo.szRemoteDevBTAddr.begin(), ::toupper);

      GET_STRINGDATA_FROM_FI_STRINGOBJ(
            oXFiPairingReq.sPinNumber, CHAR_SET_UTF8, rPairingReqInfo.szPairingPin);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(
            oXFiPairingReq.sRemoteDeviceAddress, CHAR_SET_UTF8, szRemoteDevName);

      rPairingReqInfo.enPairingMethod = static_cast<tenBTSetPairingMethod>(
            oXFiPairingReq.e8PairingType.enType);
      tBool bPairingInitiatedFromSytem =
            (most_fi_tcl_e8_BTSetOrigin::FI_EN_E8SYSTEM == oXFiPairingReq.e8Origin.enType);

      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnStatusPairingRequest(): PairingInitiatedFromSytem = %d, RemoteDevBTAddr = %s ",
            ETG_ENUM(BOOL, bPairingInitiatedFromSytem), rPairingReqInfo.szRemoteDevBTAddr.c_str()));
      ETG_TRACE_USR2((" vOnStatusPairingRequest(): PairingType = %d, PairingPin = %s ",
            ETG_ENUM(BTSET_PAIRING_METHOD, rPairingReqInfo.enPairingMethod), rPairingReqInfo.szPairingPin.c_str()));
      ETG_TRACE_USR2((" vOnStatusPairingRequest(): RemoteDevName = %s ", szRemoteDevName.c_str()));

      if ((false == rPairingReqInfo.szPairingPin.empty()) &&
            (IS_VALID_BT_ADDRESS(rPairingReqInfo.szRemoteDevBTAddr)) &&
            (NULL != m_rBTPairInfoCallbacks.fvOnBTPairingInfo))
      {
         m_rBTPairInfoCallbacks.fvOnBTPairingInfo(rPairingReqInfo);
      }

   } //if (true == oXFiPairingReq.bIsValid())
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnStatusPairingRequest: Invalid message received! "));
   }
}//! end of vOnStatusPairingRequest()



/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnStatusVehicleBTFriendlyName(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnStatusVehicleBTFriendlyName(amt_tclServiceData* poMessage)
{
	btset_tXFiStVehicleBTFriendlyName oXFiVehicleBTFriendlyName(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   if (true == oXFiVehicleBTFriendlyName.bIsValid())
   {
	   GET_STRINGDATA_FROM_FI_STRINGOBJ(
			   oXFiVehicleBTFriendlyName.sVehicleBTFriendlyName, CHAR_SET_UTF8, m_szVehicleBTFriendlyName);
	   ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnStatusVehicleBTFriendlyName(): Vehicle BTFriendly name = %s ",
			   m_szVehicleBTFriendlyName.c_str()));
   }
}//! end of vOnStatusVehicleBTFriendlyName()


/***************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vOnStatusBTLimitationMode(amt_tclServiceData...
**************************************************************************/
tVoid spi_tclBluetoothClient::vOnStatusBTLimitationMode(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnStatusBTLimitationMode() entered"));

   btset_tXFiStBTLimMode oXFiBTLimMode(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);
   if (true == oXFiBTLimMode.bIsValid())
   {
      ETG_TRACE_USR2(("[PARAM]::vOnStatusBTLimitationMode: Number of items in list = %d ", oXFiBTLimMode.oList.oItems.size()));
   
      //! Validate each item in list
      for (btset_BTLimModeListItr LstItr = oXFiBTLimMode.oList.oItems.begin();
           LstItr != oXFiBTLimMode.oList.oItems.end();
           ++LstItr)
      {
         t_String szBTAddress = "";
         GET_STRINGDATA_FROM_FI_STRINGOBJ((*LstItr).sBdAddress, CHAR_SET_UTF8, szBTAddress);
         std::transform(szBTAddress.begin(), szBTAddress.end(), szBTAddress.begin(), ::toupper);

         t_String szBTDeviceName = "";
         GET_STRINGDATA_FROM_FI_STRINGOBJ((*LstItr).sBdDeviceName, CHAR_SET_UTF8, szBTDeviceName);
         std::transform(szBTDeviceName.begin(), szBTDeviceName.end(), szBTDeviceName.begin(), ::toupper);

         m_oCurBTLimModeLock.s16Lock();
         m_rCurrentBTLimModeInfo.szBTAddress = szBTAddress;

         m_rCurrentBTLimModeInfo.szBTDeviceName = szBTDeviceName;

         m_rCurrentBTLimModeInfo.enTechnology = static_cast<tenBTTechnology>((*LstItr).e8Technology.enType);
         m_rCurrentBTLimModeInfo.enCommChannel = static_cast<tenBTCommunicationChannel>((*LstItr).e8CommunicationChannel.enType);

         //! SPI should not differentiate between the Communication channels
      /*   if((e8BT_COMM_CHN_WIFI_2_4GHZ == m_rCurrentBTLimModeInfo.enCommChannel) || (e8BT_COMM_CHN_WIFI_5GHZ == m_rCurrentBTLimModeInfo.enCommChannel))
         {
            m_rCurrentBTLimModeInfo.enCommChannel = e8BT_COMM_CHN_WIFI;
         }*/

         m_rCurrentBTLimModeInfo.enActionState = static_cast<tenBTLimitationActionState>((*LstItr).e8ActionState.enType);

         ETG_TRACE_USR2(("[PARAM]::vOnStatusBTLimitationMode: "
               "Technology = %d, CommChannel = %d, ActionState = %d, BTAddress = %s ",
               ETG_ENUM(BTSET_TECHNOLOGY, m_rCurrentBTLimModeInfo.enTechnology),
               ETG_ENUM(BTSET_COMM_CHN, m_rCurrentBTLimModeInfo.enCommChannel),
               ETG_ENUM(BTSET_LIMIT_ACTION_STATE, m_rCurrentBTLimModeInfo.enActionState),
               szBTAddress.c_str()));

         if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationMode)
         {
            m_rBluetoothCallbacks.fvOnBTLimitationMode(m_rCurrentBTLimModeInfo);
         }
         m_oCurBTLimModeLock.vUnlock();
      }
   } //if (true == oXFiBTLimMode.bIsValid())
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnStatusBTLimitationMode: Invalid message received! "));
   }
}//! end of vOnStatusBTLimitationMode()

/**************************************************************************
** FUNCTION:  tBool spi_tclBluetoothClient::bPostMethodStart(const...)
**************************************************************************/
tBool spi_tclBluetoothClient::bPostMethodStart(
      const btset_FiMsgBase& rfcooBTSetMS, t_U32 u32CmdCounter) const
{
   tBool bSuccess = false;

   if (nullptr != m_poMainAppl)
   {
      //! Create Msg context
      trMsgContext rMsgCtxt;
      rMsgCtxt.rUserContext.u32SrcAppID  = CCA_C_U16_APP_SMARTPHONEINTEGRATION;
      rMsgCtxt.rUserContext.u32DestAppID = u16GetServerAppID();
      rMsgCtxt.rUserContext.u32RegID     = u16GetRegID();
      rMsgCtxt.rUserContext.u32CmdCtr    = u32CmdCounter;

      //!Post BT settings MethodStart
      FIMsgDispatch oMsgDispatcher(m_poMainAppl);
      bSuccess = oMsgDispatcher.bSendMessage(rfcooBTSetMS, rMsgCtxt, BTSETTINGS_FI_MAJOR_VERSION);
   }

   ETG_TRACE_USR2(("spi_tclBluetoothClient::bPostMethodStart() left with: "
         "Message post success = %u (for FID = 0x%x) ",
         ETG_ENUM(BOOL, bSuccess),
         rfcooBTSetMS.u16GetFunctionID()));
   return bSuccess;

}//! end of bPostMethodResult()

/***************************************************************************
** FUNCTION:  tU8 spi_tclBluetoothClient::u8GetBTDeviceHandle(const t_String&...)
***************************************************************************/
tU8 spi_tclBluetoothClient::u8GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress)
{
   tU8 u8BTDevHandle = cou8InvalidDeviceHandle;

   //! Search for Device with matching BTAddress in DeviceList Map.
   if (true == IS_VALID_BT_ADDRESS(rfcoszDeviceBTAddress))
   {
      m_oDevLstLock.s16Lock();

      for (tBTDevListMapConstItr DevLstMapCoItr = m_mapBTDeviceList.begin();
          DevLstMapCoItr != m_mapBTDeviceList.end();
          ++DevLstMapCoItr)
      {
         if (rfcoszDeviceBTAddress == ((DevLstMapCoItr->second).szBTAddress))
         {
            u8BTDevHandle = DevLstMapCoItr->first;
            break;
         }
      } //for (tBTDevListMapConstItr DevLstMapCoItr = ...)

      m_oDevLstLock.vUnlock();

      ETG_TRACE_USR4(("spi_tclBluetoothClient::u8GetBTDeviceHandle() left with: "
            "BTDeviceHandle = 0x%x (for BTAddress = %s) ",
            u8BTDevHandle, rfcoszDeviceBTAddress.c_str()));
   } //if (coszBlankBTAddress != rfcoszDeviceBTAddress)
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::u8GetBTDeviceHandle: Invalid BT Address!"));
   }
   return u8BTDevHandle;

}//! end of u32GetBTDeviceHandle()

/**************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vAddToDeviceList(tU8 u8DeviceHandle,...)
**************************************************************************/
tVoid spi_tclBluetoothClient::vAddToDeviceList(tU8 u8DeviceHandle,
      tBool bDeviceConnStatus, const t_String& rfcszDeviceName,
      const t_String& rfcszBTAddress, t_Bool bIsHFPConn, t_Bool bIsA2DPConn)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vAddToDeviceList() entered: "
         "BT DeviceHandle = 0x%x, ConnectionStatus = %u, DeviceName = %s ",
         u8DeviceHandle, ETG_ENUM(BOOL, bDeviceConnStatus), rfcszDeviceName.c_str()));
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vAddToDeviceList() entered: "
         "BT Address = %s ", rfcszBTAddress.c_str()));

   tBTDevListMapConstItr DevLstMapCoItr; //not used

   //! If device does not already exist in DeviceList map, add it and
   //! send GetDeviceInfo request to obtain BT Address.
   if (FALSE == bIsDeviceValid(u8DeviceHandle))
   {
      trBluetoothDeviceInfo rDevInfo;
      rDevInfo.bDeviceConnected = bDeviceConnStatus;
      rDevInfo.szDeviceName = rfcszDeviceName.c_str();
      rDevInfo.szBTAddress = rfcszBTAddress.c_str();
      rDevInfo.bIsHFPConn = bIsHFPConn;
      rDevInfo.bIsA2DPConn = bIsA2DPConn;

      //@Note: m_oDevLstLock should be used only after bIsDeviceValid() call, as it uses same lock.
      m_oDevLstLock.s16Lock();
      m_mapBTDeviceList.insert(std::pair<tU8, trBluetoothDeviceInfo>(u8DeviceHandle, rDevInfo));
      //@Note: BT address will be updated on GetDeviceInfo response, and
      //Connection status will be updated on DeviceList updates.
      m_oDevLstLock.vUnlock();

   } //if (FALSE == bIsDeviceValid(u8DeviceHandle))
   else
   {
      ETG_TRACE_ERR((" vAddToDeviceList: Device already exists/Invalid handle! "));
   }
}//! end of vAddToDeviceList()

/**************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vRemoveFromDeviceList(tU8 u8DeviceHandle)
**************************************************************************/
tVoid spi_tclBluetoothClient::vRemoveFromDeviceList(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vRemoveFromDeviceList() entered: BT DeviceHandle = 0x%x ", u8DeviceHandle));

   //! Remove device from DeviceList map.
   if (TRUE == bIsDeviceValid(u8DeviceHandle))
   {
      //@Note: m_oDevLstLock should be used only after bIsDeviceValid() call, as it uses same lock.
      m_oDevLstLock.s16Lock();

      tBTDevListMapItr DevLstMapItr = m_mapBTDeviceList.find(u8DeviceHandle);
      if (m_mapBTDeviceList.end() != DevLstMapItr)
      {
         m_mapBTDeviceList.erase(DevLstMapItr);
      }

      m_oDevLstLock.vUnlock();
      vNotifyBTProfileStatus();
   }//if (TRUE == bIsDeviceValid(u8DeviceHandle))

}//! end of vRemoveFromDeviceList()

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vClearBTDeviceList()
***************************************************************************/
t_Void spi_tclBluetoothClient::vClearBTDeviceList()
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vClearBTDeviceList()) entered "));

   m_oDevLstLock.s16Lock();
   //Clear the list after lock is acquired
   m_mapBTDeviceList.clear();
   m_oDevLstLock.vUnlock();
   vNotifyBTProfileStatus();
}

/**************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vUpdateDeviceConnStatus(tU8...)
**************************************************************************/
tVoid spi_tclBluetoothClient::vUpdateDeviceConnStatus(tU8 u8DeviceHandle,
      tBool bDeviceConnectionStatus,
      tBool bIsHFPConn, tBool bIsA2DPConn)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vUpdateDeviceConnStatus() entered: "
         "BT DeviceHandle = 0x%x, DeviceConnectionStatus = %u ",
         u8DeviceHandle, ETG_ENUM(BOOL, bDeviceConnectionStatus)));

   if (TRUE == bIsDeviceValid(u8DeviceHandle))
   {
      //@Note: m_oDevLstLock should be used only after bIsDeviceValid() call, as it uses same lock.
      m_oDevLstLock.s16Lock();

      tBTDevListMapItr DevLstMapItr = m_mapBTDeviceList.find(u8DeviceHandle);
      if (m_mapBTDeviceList.end() != DevLstMapItr)
      {
         t_String szBTAddress = (DevLstMapItr->second).szBTAddress;

         t_Bool bConnStatusChanged = ((DevLstMapItr->second).bDeviceConnected !=
               static_cast<t_Bool> (bDeviceConnectionStatus));
         ETG_TRACE_USR2(("[DESC]::vUpdateDeviceConnStatus: Connection status has changed = %d ",
               ETG_ENUM(BOOL, bConnStatusChanged)));

         //! Update connection status of device in map
         (DevLstMapItr->second).bDeviceConnected = static_cast<t_Bool> (bDeviceConnectionStatus);
         (DevLstMapItr->second).bIsHFPConn = static_cast<t_Bool> (bIsHFPConn);
         (DevLstMapItr->second).bIsA2DPConn = static_cast<t_Bool> (bIsA2DPConn);
         ETG_TRACE_USR1(("spi_tclBluetoothClient::vUpdateDeviceConnStatus() entered: "
         "BT HFP = %d, BT A2DP = %d , BT address = %s",
         ETG_ENUM(BOOL,(DevLstMapItr->second).bIsHFPConn), ETG_ENUM(BOOL, (DevLstMapItr->second).bIsA2DPConn),
         szBTAddress.c_str()));

         if (true == bConnStatusChanged)
         {
            //! Notify connection/disconnection change
            tenBTConnectionResult enConnResult = (bDeviceConnectionStatus) ?
                  (e8BT_RESULT_CONNECTED) : (e8BT_RESULT_DISCONNECTED);
            vNotifyBTConnectionChanged(szBTAddress, enConnResult);
         }
         vNotifyBTProfileStatus();
      }//if (m_mapBTDeviceList.end() != DevLstMapItr)
      m_oDevLstLock.vUnlock();
   }//if (TRUE == bIsDeviceValid(u8DeviceHandle))

}//! end of vUpdateDeviceConnStatus()

/**************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vUpdateDeviceName(tU8...)
**************************************************************************/
tVoid spi_tclBluetoothClient::vUpdateDeviceName(tU8 u8DeviceHandle,
      const t_String& rfcszDeviceName)
{
   if ((TRUE == bIsDeviceValid(u8DeviceHandle)) && (false == rfcszDeviceName.empty()))
   {
      //@Note: m_oDevLstLock should be used only after bIsDeviceValid() call, as it uses same lock.
      m_oDevLstLock.s16Lock();

      tBTDevListMapItr DevLstMapItr = m_mapBTDeviceList.find(u8DeviceHandle);
      if (m_mapBTDeviceList.end() != DevLstMapItr)
      {
         t_Bool bDevNameChanged = ((DevLstMapItr->second).szDeviceName != rfcszDeviceName);

         if (true == bDevNameChanged)
         {
            ETG_TRACE_USR2(("[DESC]::vUpdateDeviceName: Device name of DeviceHandle = 0x%x has changed to %s ",
                  u8DeviceHandle, rfcszDeviceName.c_str()));
            t_String szBTAddress = (DevLstMapItr->second).szBTAddress;
            //! Update device details in map
            (DevLstMapItr->second).szDeviceName = rfcszDeviceName.c_str();
            //! Notify device name
            vNotifyBTDeviceName(szBTAddress, rfcszDeviceName);
         }
      }//if (m_mapBTDeviceList.end() != DevLstMapItr)

      m_oDevLstLock.vUnlock();
   }//if (TRUE == bIsDeviceValid(u8DeviceHandle))
}//! end of vUpdateDeviceName()


/***************************************************************************
** FUNCTION:  tBool spi_tclBluetoothClient::bIsDeviceValid(tU8...)
***************************************************************************/
tBool spi_tclBluetoothClient::bIsDeviceValid(tU8 u8DeviceHandle)
{
   //! Search in DeviceList map for requested device
   tBool bDeviceFound = FALSE;
   if (IS_VALID_DEVHANDLE(u8DeviceHandle))
   {
      m_oDevLstLock.s16Lock();
      bDeviceFound = (m_mapBTDeviceList.end() != m_mapBTDeviceList.find(u8DeviceHandle));
      m_oDevLstLock.vUnlock();

      if (FALSE == bDeviceFound)
      {
         ETG_TRACE_ERR(("spi_tclBluetoothClient::bIsDeviceValid: DeviceHandle 0x%x not found in DeviceList! ",
            u8DeviceHandle));
      }
   }//if (IS_VALID_DEVHANDLE(u8DeviceHandle))
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::bIsDeviceValid: Invalid DeviceHandle 0x%x! ", u8DeviceHandle));
   }
   return bDeviceFound;

}//! end of bIsDeviceValid()

/**************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vNotifyBTConnectionChanged(...)
**************************************************************************/
tVoid spi_tclBluetoothClient::vNotifyBTConnectionChanged(
      const t_String& rfcoszDevBTAddress,
      tenBTConnectionResult enBTConnResult) const
{
	/*lint -esym(40,fvOnBTConnectionChanged)fvOnBTConnectionChanged Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vNotifyBTConnectionChanged() entered: "
         "BT ConnectionResultType = %u, Device BTAddress = %s ",
         ETG_ENUM(BT_CONNECTION_RESULT, enBTConnResult),
         rfcoszDevBTAddress.c_str()));

   if (NULL != m_rBluetoothCallbacks.fvOnBTConnectionChanged)
   {
      //! Notify BT Manager with error info
      m_rBluetoothCallbacks.fvOnBTConnectionChanged(rfcoszDevBTAddress, enBTConnResult);
   } //if (NULL != m_rBluetoothCallbacks.fvOnBTConnectionResult)

}//! end of vNotifyBTConnectionChanged()

/**************************************************************************
** FUNCTION:  spi_tclBluetoothClient::vNotifyBTDeviceName(...)
**************************************************************************/
tVoid spi_tclBluetoothClient::vNotifyBTDeviceName(t_String szBTAddress,
      t_String szBTDeviceName) const
{
   /*lint -esym(40,fvOnBTDeviceNameUpdate)fvOnBTDeviceNameUpdate Undeclared identifier */
   if ((szBTAddress == m_rBTDeviceNameCbInfo.szBTAddress) &&
         (NULL != m_rBTDeviceNameCbInfo.fvOnBTDeviceNameUpdate))
   {
      m_rBTDeviceNameCbInfo.fvOnBTDeviceNameUpdate(szBTAddress, szBTDeviceName);
   }
}//! end of vNotifyBTDeviceName()

/**************************************************************************
** FUNCTION:  tVoid spi_tclBluetoothClient::szGetDateTimestamp(...)
**************************************************************************/
t_String spi_tclBluetoothClient::
      szGetDateTimestamp(const btset_tDateTimestamp& rfcooDateTimeStamp) const
{
   //! Extract Date Stamp
   t_String szYear;
   GET_STRINGDATA_FROM_FI_STRINGOBJ(
         rfcooDateTimeStamp.oDateStamp.sCldrYear, CHAR_SET_UTF8, szYear);
   t_String szMonth;
   GET_STRINGDATA_FROM_FI_STRINGOBJ(
         rfcooDateTimeStamp.oDateStamp.sCldrMonth, CHAR_SET_UTF8, szMonth);
   t_String szDay;
   GET_STRINGDATA_FROM_FI_STRINGOBJ(
         rfcooDateTimeStamp.oDateStamp.sCldrDay, CHAR_SET_UTF8, szDay);

   //! Extract Time Stamp
   t_String szHour;
   GET_STRINGDATA_FROM_FI_STRINGOBJ(
         rfcooDateTimeStamp.oTimeStamp.sHours, CHAR_SET_UTF8, szHour);
   t_String szMinute;
   GET_STRINGDATA_FROM_FI_STRINGOBJ(
         rfcooDateTimeStamp.oTimeStamp.sMinutes, CHAR_SET_UTF8, szMinute);
   t_String szSeconds;
   GET_STRINGDATA_FROM_FI_STRINGOBJ(
         rfcooDateTimeStamp.oTimeStamp.sSeconds, CHAR_SET_UTF8, szSeconds);

   //! Create DateTimeStamp
   t_String szDateTimeStamp =
         szYear + "/" + szMonth + "/" + szDay + " " +
         szHour + ":" + szMinute + ":" + szSeconds;

   return szDateTimeStamp;

}//! end of szGetDateTimestamp()

/**************************************************************************
** FUNCTION:  t_String spi_tclBluetoothClient::szGetBTDeviceAddress(...)
**************************************************************************/
t_String spi_tclBluetoothClient::szGetBTDeviceAddress(tU8 u8DeviceHandle)
{
   t_String szBTAddress;

   m_oDevLstLock.s16Lock();
   tBTDevListMapItr DevLstMapItr =  m_mapBTDeviceList.find(u8DeviceHandle);
   if (m_mapBTDeviceList.end() != DevLstMapItr)
   {
      szBTAddress = (DevLstMapItr->second).szBTAddress;
   }
   m_oDevLstLock.vUnlock();

   return szBTAddress;
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vGetBtMacAddress()
***************************************************************************/
t_Void spi_tclBluetoothClient::vGetBtMacAddress(const t_String& rfcszDevName,
                                                t_String& rfszMACAddress)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vSetDevicesDisconnected() entered "));
   if (!rfcszDevName.empty())
   {
      m_oDevLstLock.s16Lock();

      //! Search for required device in BT DeviceList Map.
      for (tBTDevListMapConstItr DevLstMapCoItr = m_mapBTDeviceList.begin();
           DevLstMapCoItr != m_mapBTDeviceList.end();
           DevLstMapCoItr++)
      {
         if (rfcszDevName == (DevLstMapCoItr->second).szDeviceName)
         {
            rfszMACAddress = (DevLstMapCoItr->second).szBTAddress;
            break;
         }
      }

      m_oDevLstLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclBluetoothClient::vOnConflictsDetectedStatus(amt_tclServiceData...
 **************************************************************************/
tVoid spi_tclBluetoothClient::
vOnConflictsDetectedStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnConflictsDetectedStatus: received Property update "));
   //! Extract the status msg
   btset_tXFiStConflictsDetected oXFiConflictsDetectedMsg(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);

   //! Retrieve ConflictsDetected Trigger type
   if (true == oXFiConflictsDetectedMsg.bIsValid())
   {
      tenconflictsDetected enConflictsDetected = static_cast<tenconflictsDetected>(
               oXFiConflictsDetectedMsg.e8ConflictTrigger.enType);
      ETG_TRACE_USR4(("spi_tclBluetoothClient::vOnConflictsDetectedStatus: Conflict Detected %d ",
               ETG_ENUM(CONFLICTS_DETECTED, enConflictsDetected)));

      ETG_TRACE_USR2(("[PARAM]::vOnConflictsDetectedStatus: Number of items in list = %d ", oXFiConflictsDetectedMsg.oBTConflictInfoList.oItems.size()));

      if(e8OOB_BT_PAIRING == enConflictsDetected)
      {
         //from the list top of the list value is obtained and used for further processing
         for (btset_ConflictsListItr LstItr = oXFiConflictsDetectedMsg.oBTConflictInfoList.oItems.begin();
                  LstItr != oXFiConflictsDetectedMsg.oBTConflictInfoList.oItems.end();
                  ++LstItr)
         {
             m_enConflictState = static_cast<tenConflictState>((*LstItr).e8ConflictState.enType);
             if((e8CONFLICT_STATE_NEW == m_enConflictState) && (false == m_bOOBConflictProcessed))
             {
                break;
             }
             else if((e8CONFLICT_STATE_SOLVED == m_enConflictState) && (true == m_bOOBConflictProcessed))
             {
                break;
             }
         }
         ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnConflictsDetectedStatus: Conflict State is %d ",
                  ETG_ENUM(CONFLICTS_STATE, m_enConflictState)));


         if(e8CONFLICT_STATE_NEW == m_enConflictState)
         {
            //Call ResloveConflictsMethodStart() - with PROCEED
            m_bOOBConflictProcessed = true;
            tMSResolveConflict oMSResolveConflict;
            tenResolutionType enResolutionType = e8RESOLUTION_TYPE_PROCEED;
            oMSResolveConflict.e8ConflictTrigger.enType = static_cast<btset_tenConflictTrigger>(enConflictsDetected);
            oMSResolveConflict.e8ResolutionType.enType = static_cast<btset_tenResolutionType>(enResolutionType);

            t_Bool bSendSuccess = bPostMethodStart(oMSResolveConflict, 0);

            ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnConflictsDetectedStatus: bSenSuccess %d ",ETG_ENUM(BOOL, bSendSuccess)));

            //invoke the PrepareSetup of WBL based on the Project Configuration - as of now it is policy based - through ExternalInterface
            if (NULL != m_rOOBTCallbacks.fvOnOOBTComplete)
            {
               m_rOOBTCallbacks.fvOnOOBTComplete();
            }
         }
         else if(e8CONFLICT_STATE_SOLVED == m_enConflictState)
         {
            //! Clearing the Details of the OOB Structure
            m_enConflictState =  e8CONFLICTS_STATE_INVALID;

            //! Set the flag to false, after the processing is done.
            m_bOOBConflictProcessed = false;
         }
         else
         {
            ETG_TRACE_USR4(("spi_tclBluetoothClient::vOnConflictsDetectedStatus: Conflict state is UNCHANGED"));
         }
      }
      else
      {
         ETG_TRACE_USR4(("spi_tclBluetoothClient::vOnConflictsDetectedStatus: Conflict Detected %d which is not SPI's interest",
                  ETG_ENUM(CONFLICTS_DETECTED, enConflictsDetected)));
      }
   }
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnConflictsDetectedStatus: Invalid message received! "));
   }

}//! end of vOnConflictsDetectedStatus()

/***************************************************************************
 ** FUNCTION:  spi_tclBluetoothClient::vOnMRResolveConflicts(amt_tclServiceData...
 **************************************************************************/
tVoid spi_tclBluetoothClient::
vOnMRResolveConflicts(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnMRResolveConflicts: received Method Result "));
   //! Extract the status msg
   btset_tXFiResolveConflitsMethodResult oXFiResolveConflictsMethodResultMsg(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);


   //! ResolveConflicts Method Result
   if (true == oXFiResolveConflictsMethodResultMsg.bIsValid())
   {
      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnMRResolveConflicts ResolveConflictsMethodResult is received with %d",
      ETG_ENUM(BOOL, oXFiResolveConflictsMethodResultMsg.bResult)));
   }
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnMRResolveConflicts: Invalid message received! "));
   }

}//! end of vOnResolveConflicts()

/***************************************************************************
 ** FUNCTION:  spi_tclBluetoothClient::vOnErrorResolveConflicts(amt_tclServiceData...
 **************************************************************************/
tVoid spi_tclBluetoothClient::
vOnErrorResolveConflicts(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnErrorResolveConflicts: received Method Result "));
   //! Extract the status msg
   btset_tXFiResolveConflitsError oXFiResolveConflictsErrorMsg(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);

   if (true == oXFiResolveConflictsErrorMsg.bIsValid())
   {
      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnErrorResolveConflicts() entered "));

   } //if (true == oXFiPairResponseErr.bIsValid())
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnErrorResolveConflicts: Invalid message received! "));
   }

}//! end of vOnResolveConflicts()

/***************************************************************************`
** FUNCTION:  t_Bool spi_tclBluetoothClient::vHandleLimitationMode(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::vHandleLimitationMode(
      const t_String& rfcszBTAddress,
      const t_String& rfcszBTDeviceName,
      tenBTTechnology enTechnology,
      tenBTCommunicationChannel enCommChannel,
      tenBTLimitationAction enAction)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vHandleLimitationMode() entered - Technology = %d, CommChannel = %d, Action = %d, BT Address = %s",
     ETG_ENUM(BTSET_TECHNOLOGY, enTechnology), ETG_ENUM(BTSET_COMM_CHN, enCommChannel), ETG_ENUM(BTSET_LIMIT_ACTION, enAction), rfcszBTAddress.c_str()));

   // Set the NewLimitationNode structure for sending ReplaceBTLimMode request if required
   trBTLimitationActionInfo rNewBTLimMode;
   rNewBTLimMode.szBTAddress = rfcszBTAddress;
   rNewBTLimMode.szBTDeviceName = rfcszBTDeviceName;
   rNewBTLimMode.enTechnology = enTechnology;
   rNewBTLimMode.enCommChannel = enCommChannel;
   rNewBTLimMode.enAction = enAction;

   t_Bool bRetVal = false;

   trBTLimitationModeInfo rCurBTLimModeInfo;
   vGetCurrentBTLimModeInfo(rCurBTLimModeInfo);
   ETG_TRACE_USR2(("vHandleLimitationMode: Current State = %d",ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
   switch(rCurBTLimModeInfo.enActionState)
   {
    case e8BT_LIM_ACTION_STATE_IDLE:
    case e8BT_LIM_ACTION_STATE_ERROR:
    case e8BT_LIM_ACTION_STATE_ERROR_USER_DENIED:
    case e8BT_LIM_ACTION_STATE_WAITING:
    {
       ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::vHandleLimitationMode - Current state is %d. Use SetBTLimMode",
         ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
       bRetVal = true;;
    }
    break;
   case e8BT_LIM_ACTION_STATE_PREPARING:
   case e8BT_LIM_ACTION_STATE_PREPARED:
   {
      bRetVal = bHandlePrepareStateBTLim(rNewBTLimMode);
   }
   break;

   case e8BT_LIM_ACTION_STATE_PREACTIVATING:
   case e8BT_LIM_ACTION_STATE_PREACTIVATED:
   {
      bRetVal = bHandlePreactivateStateBTLim(rNewBTLimMode);
   }
   break;

   case e8BT_LIM_ACTION_STATE_ACTIVATING:
   case e8BT_LIM_ACTION_STATE_ACTIVE:
   {
      ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::vHandleLimitationMode - calling bHandleActivateStateBTLim "));
      bRetVal = bHandleActivateStateBTLim(rNewBTLimMode);
   }
   break;

   case e8BT_LIM_ACTION_STATE_DEACTIVATING:
   {
      bRetVal = bHandleDactivateStateBTLim(rNewBTLimMode);
   }
   break;
   default:
      {
         ETG_TRACE_USR2(("[DESC]: spi_tclBluetoothClient::vHandleLimitationMode - Current BT Action state is invalid"));
         bRetVal = true;
      }
      break;

   } // switch(rCurBTLimModeInfo.enActionState)

   return bRetVal;
}

/***************************************************************************`
** FUNCTION:  t_Bool spi_tclBluetoothClient::bIsCurrentBTLimMode(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bIsCurrentBTLimMode(
        const t_String& rfcszBTAddress,
        const t_String& rfcszBTDeviceName,
        tenBTTechnology enTechnology,
        tenBTCommunicationChannel enCommChannel)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::bIsCurrentBTLimMode: New device details - Technology = %d, Communication channel = %d, BT Address = %s",
       ETG_ENUM(BTSET_TECHNOLOGY, enTechnology), ETG_ENUM(BTSET_COMM_CHN, enCommChannel), rfcszBTAddress.c_str()));
   SPI_INTENTIONALLY_UNUSED(rfcszBTDeviceName);
   trBTLimitationModeInfo rCurBTLimModeInfo;
   vGetCurrentBTLimModeInfo(rCurBTLimModeInfo);
   ETG_TRACE_USR1(("spi_tclBluetoothClient::bIsCurrentBTLimMode: Current device details - Technology = %d, Communication channel = %d, BT Address = %s",
       ETG_ENUM(BTSET_TECHNOLOGY, rCurBTLimModeInfo.enTechnology), ETG_ENUM(BTSET_COMM_CHN, rCurBTLimModeInfo.enCommChannel),
       rCurBTLimModeInfo.szBTAddress.c_str()));

   t_Bool bRetVal = false;
   if ((rfcszBTAddress == rCurBTLimModeInfo.szBTAddress) &&
       (enTechnology == rCurBTLimModeInfo.enTechnology))
   {
      switch(enCommChannel)
      {
         case e8BT_COMM_CHN_USB:
         {
            if (e8BT_COMM_CHN_USB == rCurBTLimModeInfo.enCommChannel)
               bRetVal = true;
         }
         break;
         case e8BT_COMM_CHN_WIFI_2_4GHZ:
         case e8BT_COMM_CHN_WIFI_5GHZ:
         case e8BT_COMM_CHN_WIFI:
         {
            if ((e8BT_COMM_CHN_WIFI_2_4GHZ == rCurBTLimModeInfo.enCommChannel) ||
                (e8BT_COMM_CHN_WIFI_5GHZ == rCurBTLimModeInfo.enCommChannel) ||
                (e8BT_COMM_CHN_WIFI == rCurBTLimModeInfo.enCommChannel))
                {
                   bRetVal = true;
                }
         }
         break;
         case e8BT_COMM_CHN_BLUETOOTH:
         {
            if(e8BT_COMM_CHN_BLUETOOTH == rCurBTLimModeInfo.enCommChannel)
               bRetVal = true;
         }
         break;
         default: bRetVal = false;
         break;
      } // switch(enCommChannel):o
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclBluetoothClient::bIsCurrentBTLimMode: request received for Different Device"));
   }
   return bRetVal;
}

/***************************************************************************`
** FUNCTION:  t_Bool spi_tclBluetoothClient::bHandlePrepareStateBTLim(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bHandlePrepareStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
{
   trBTLimitationModeInfo rCurBTLimModeInfo;
   vGetCurrentBTLimModeInfo(rCurBTLimModeInfo);
   if (e8BT_LIM_ACTION_PREPARE == rNewBtLimMode.enAction)
   {
      if ((true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel)) &&
          (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePrepareStateBTLim - %d request received during %d state for the same device and technology. "
           "Ignoring the request with success response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(true);
      }
      else
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePrepareStateBTLim - %d request received during %d state for different device/technology. "
           "Sending ReplaceBTLimMode", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         bReplaceBTLimitationMode(rCurBTLimModeInfo, rNewBtLimMode);
      }
   } // if (e8BT_LIM_ACTION_PREPARE == rBtLimModeToBeReplaced.enAction)
   else if ((e8BT_LIM_ACTION_PREACTIVATE == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE_WAIT == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE_INTERNAL == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_SPM_STATE_OFF == rNewBtLimMode.enAction))
   {
      if (true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePrepareStateBTLim - %d request received during %d for same device and technology. Use SetBTLimMode",
           ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction), ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         return true;
      }
      else if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePrepareStateBTLim - %d request received during %d state for different device/technology. "
           "Ignoring the request with error response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),
           ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   } // else if ((e8BT_LIM_ACTION_PREACTIVATE == rBtLimModeToBeReplaced.enAction)
   else if (e8BT_LIM_ACTION_ACTIVATE == rNewBtLimMode.enAction)
   {
      if (true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName, rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel))
      {
          ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePrepareStateBTLim - %d request received during %d. Use SetBTLimMode",
            ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
          return true;
      }
      else
      {
          ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePrepareStateBTLim - %d request received during %d state for different device/technology. "
            "Sending ReplaceBTLimMode", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
          bReplaceBTLimitationMode(rCurBTLimModeInfo, rNewBtLimMode);
      }
   } // else if (e8BT_LIM_ACTION_ACTIVATE == enAction)
   return false;
}

/***************************************************************************`
** FUNCTION:  t_Void spi_tclBluetoothClient::bHandlePreactivateStateBTLim(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bHandlePreactivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
{
   trBTLimitationModeInfo rCurBTLimModeInfo;
   vGetCurrentBTLimModeInfo(rCurBTLimModeInfo);
   if ((e8BT_LIM_ACTION_PREPARE == rNewBtLimMode.enAction) && (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult))
   {
      ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePreactivateStateBTLim - %d request received during %d state. "
        "Ignoring the request with error response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
      m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
   } // if (e8BT_LIM_ACTION_PREPARE == rBtLimModeToBeReplaced.enAction)
   else if (e8BT_LIM_ACTION_PREACTIVATE == rNewBtLimMode.enAction)
   {
      if ((true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel)) &&
          (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePreactivateStateBTLim - %d request received during %d state for the same device and technology. "
           "Ignoring the request with success response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(true);
      }
      else if(NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePreactivateStateBTLim - %d request received during %d state for the same device and technology. "
           "Ignoring the request with error response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   } // else if (e8BT_LIM_ACTION_PREACTIVATE == rBtLimModeToBeReplaced.enAction)
   else if ((e8BT_LIM_ACTION_DEACTIVATE == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE_WAIT == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE_INTERNAL == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_SPM_STATE_OFF == rNewBtLimMode.enAction))
   {
      if (true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePreactivateStateBTLim - %d request received during %d for same device and technology. Use SetBTLimMode",
           ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction), ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         return true;
      }
      else if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePreactivateStateBTLim - %d request received during %d state for different device/technology. "
           "Ignoring the request with error response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),
           ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   } // else if ((e8BT_LIM_ACTION_DEACTIVATE == enAction)
   else if (e8BT_LIM_ACTION_ACTIVATE == rNewBtLimMode.enAction)
   {
      if (true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePreactivateStateBTLim - %d request received during %d. Use SetBTLimMode",
           ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         return true;
      }
      else
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandlePreactivateStateBTLim - %d request received during %d state for different device/technology. "
           "Sending ReplaceBTLimMode", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         bReplaceBTLimitationMode(rCurBTLimModeInfo, rNewBtLimMode);
      }
   } // else if (e8BT_LIM_ACTION_ACTIVATE == enAction)
   return false;
}

/***************************************************************************`
** FUNCTION:  t_Void spi_tclBluetoothClient::bHandleActivateStateBTLim(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bHandleActivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
{
   trBTLimitationModeInfo rCurBTLimModeInfo;
   vGetCurrentBTLimModeInfo(rCurBTLimModeInfo);
   ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleActivateStateBTLim entered with - %d request received during %d",
            ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction), ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));

   if ((e8BT_LIM_ACTION_DEACTIVATE == rNewBtLimMode.enAction) ||
       (e8BT_LIM_ACTION_DEACTIVATE_WAIT == rNewBtLimMode.enAction) ||
       (e8BT_LIM_ACTION_DEACTIVATE_INTERNAL == rNewBtLimMode.enAction) ||
       (e8BT_LIM_ACTION_SPM_STATE_OFF == rNewBtLimMode.enAction) ||
	   (e8BT_LIM_ACTION_PREPARE == rNewBtLimMode.enAction))
   {
      if (true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleActivateStateBTLim - %d request received during %d for same device and technology. Use SetBTLimMode",
           ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction), ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         return true;
      }
      //! We should not compare Technology as there can be switch between different technologies too.
	  //! TODO - Communication channels check has to removed after Discussion with BT component as SPI should not differentate b/n the Channels
      else if ((e8BT_LIM_ACTION_PREPARE == rNewBtLimMode.enAction) &&(rNewBtLimMode.szBTAddress != rCurBTLimModeInfo.szBTAddress) &&
               (rNewBtLimMode.enCommChannel != rCurBTLimModeInfo.enCommChannel))
      {

         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleActivateStateBTLim - %d request received during %d state for different device, same transport and same Technology. "
           "Sending ReplaceBTLimMode", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         bReplaceBTLimitationMode(rCurBTLimModeInfo, rNewBtLimMode);
      }
      else if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleActivateStateBTLim - %d request received during %d state for different device/technology. "
           "Ignoring the request with error response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),
           ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   } // else if ((e8BT_LIM_ACTION_DEACTIVATE == enAction)
   else if (e8BT_LIM_ACTION_ACTIVATE == rNewBtLimMode.enAction)
   {
      if ((true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel)) && (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleActivateStateBTLim - %d request received during %d for same device and technology. "
           "Ignoring the request with success response",ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(true);
      }
      else if ((rNewBtLimMode.szBTAddress == rCurBTLimModeInfo.szBTAddress) &&
               (rNewBtLimMode.enTechnology == rCurBTLimModeInfo.enTechnology) &&
               (rNewBtLimMode.enCommChannel != rCurBTLimModeInfo.enCommChannel))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleActivateStateBTLim - %d request received during %d state for same device and different transport. "
           "Sending ReplaceBTLimMode", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         bReplaceBTLimitationMode(rCurBTLimModeInfo, rNewBtLimMode);
      }
      else if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleActivateStateBTLim - %d request received during %d state for different device/technology. "
           "Ignoring the request with error response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),
           ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   } // else if (e8BT_LIM_ACTION_ACTIVATE == enAction)
   return false;
}

/***************************************************************************`
** FUNCTION:  t_Void spi_tclBluetoothClient::bHandleDactivateStateBTLim(...)
***************************************************************************/
t_Bool spi_tclBluetoothClient::bHandleDactivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
{
   trBTLimitationModeInfo rCurBTLimModeInfo;
   vGetCurrentBTLimModeInfo(rCurBTLimModeInfo);
   if (e8BT_LIM_ACTION_PREPARE == rNewBtLimMode.enAction)
   {
      if ((true == bIsCurrentBTLimMode(rNewBtLimMode.szBTAddress,rNewBtLimMode.szBTDeviceName,rNewBtLimMode.enTechnology,rNewBtLimMode.enCommChannel)) &&
          (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult))
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleDactivateStateBTLim - %d request received during %d for same device and technology. "
          "Ignoring the request with error response",ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
      else
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleDactivateStateBTLim - %d request received during %d state for different device/technology. "
          "Sending ReplaceBTLimMode", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         bReplaceBTLimitationMode(rCurBTLimModeInfo, rNewBtLimMode);
      }
   } // if (e8BT_LIM_ACTION_PREPARE == enAction)
   else if (e8BT_LIM_ACTION_ACTIVATE == rNewBtLimMode.enAction)
   {
      ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleDactivateStateBTLim - %d request received during %d state. "
        "Sending ReplaceBTLimMode", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
      bReplaceBTLimitationMode(rCurBTLimModeInfo, rNewBtLimMode);
   } // else if (e8BT_LIM_ACTION_ACTIVATE == rBtLimModeToBeReplaced.enAction)
   else if ((e8BT_LIM_ACTION_PREACTIVATE == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE_WAIT == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_DEACTIVATE_INTERNAL == rNewBtLimMode.enAction) ||
            (e8BT_LIM_ACTION_SPM_STATE_OFF == rNewBtLimMode.enAction))
   {
      if (NULL != m_rBluetoothCallbacks.fvOnBTLimitationModeResult)
      {
         ETG_TRACE_USR2(("spi_tclBluetoothClient::bHandleDactivateStateBTLim - %d request received during %d state. "
          "Ignoring the request with error response", ETG_ENUM(BTSET_LIMIT_ACTION, rNewBtLimMode.enAction),
           ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rCurBTLimModeInfo.enActionState)));
         m_rBluetoothCallbacks.fvOnBTLimitationModeResult(false);
      }
   } // else if ((e8BT_LIM_ACTION_PREACTIVATE == enAction)
   return false;
}

/***************************************************************************`
** FUNCTION:  t_Void spi_tclBluetoothClient::vGetCurrentBTLimModeInfo()
***************************************************************************/
t_Void spi_tclBluetoothClient::vGetCurrentBTLimModeInfo(trBTLimitationModeInfo& rfrBTLimitationModeInfo)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vGetCurrentBTLimModeInfo entered"));

   m_oCurBTLimModeLock.s16Lock();
   rfrBTLimitationModeInfo = m_rCurrentBTLimModeInfo;
   m_oCurBTLimModeLock.vUnlock();
}

/***************************************************************************`
** FUNCTION:  t_Void spi_tclBluetoothClient::vNotifyBTProfileStatus()
***************************************************************************/
t_Void spi_tclBluetoothClient::vNotifyBTProfileStatus()
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vNotifyBTProfileStatus() entered: "));

   if (NULL != m_rBluetoothCallbacks.fvOnBTProfileStatus)
   {
      //! Notify BT Manager with error info
      m_rBluetoothCallbacks.fvOnBTProfileStatus();
   } //if (NULL != m_rBluetoothCallbacks.fvOnBTConnectionResult)
}

/***************************************************************************`
** FUNCTION:  t_Void spi_tclBluetoothClient::vGetBTProfileStatus()
***************************************************************************/
t_Void spi_tclBluetoothClient::vGetBTProfileStatus(t_U8 u8DeviceHandle,
                             t_Bool& rfbIsHFPConn, t_Bool& rfbIsA2DPConn)
{
    ETG_TRACE_USR1(("spi_tclBluetoothClient::vGetBTProfileStatus() entered "));
    
    m_oCurBTLimModeLock.s16Lock();
    tBTDevListMapItr DevLstMapItr = m_mapBTDeviceList.find(u8DeviceHandle);
    if (m_mapBTDeviceList.end() != DevLstMapItr)
    {
        rfbIsHFPConn = (DevLstMapItr->second).bIsHFPConn;
        rfbIsA2DPConn = (DevLstMapItr->second).bIsA2DPConn;
    }
    m_oCurBTLimModeLock.vUnlock();
    
    ETG_TRACE_USR1(("spi_tclBluetoothClient::vGetBTProfileStatus() entered: "
    "BT HFP = %d, BT A2DP = %d",
    ETG_ENUM(BOOL,rfbIsHFPConn), ETG_ENUM(BOOL,rfbIsA2DPConn)));
}

/**************************************************************************
 ** FUNCTION:  spi_tclBluetoothClient::vOnStatusServiceAvailable(amt_tclServiceData...
 **************************************************************************/
t_Void spi_tclBluetoothClient::vOnStatusServiceAvailable(amt_tclServiceData* poMessage)
{
   btset_tXFiStServiceAvailable oXFiServiceAvailable(*poMessage, BTSETTINGS_FI_MAJOR_VERSION);

   if (true == oXFiServiceAvailable.bIsValid())
   {
      tenBTServiceAvailableStatus enBTServiceAvailableStatus = static_cast<tenBTServiceAvailableStatus>(
               oXFiServiceAvailable.e8ServiceAvailable.enType);

      //! Print BT Service Availability status
      ETG_TRACE_USR1(("spi_tclBluetoothClient::vOnStatusServiceAvailable() entered: "
               "Bt Service Available status = %d ", ETG_CENUM(tenBTServiceAvailableStatus, enBTServiceAvailableStatus)));

      m_oBTAvailLock.s16Lock();
      // Check if BT service is available and BT SPM state is in NORMAL mode so that BT service can process the requests such as SetBTLimitationMode.
      // If the BT service is available but not in NORMAL mode, then BT service will be in BLOCK mode which can process only Deactivate, Block/Unblock requests.
      if ((true == m_bIsBTServiceAvailable) && (e8SERVICE_AVAILABLE != m_enBTServiceAvailableStatus) && (e8SERVICE_AVAILABLE == enBTServiceAvailableStatus))
      {
         vPostBTServiceAvailability(true);
      }
      else if ((e8SERVICE_AVAILABLE == m_enBTServiceAvailableStatus) && (e8SERVICE_AVAILABLE != enBTServiceAvailableStatus))
      {
         vPostBTServiceAvailability(false);
      }
      m_enBTServiceAvailableStatus = enBTServiceAvailableStatus;
      m_oBTAvailLock.vUnlock();
   } //if (true == oXFiServiceAvailable.bIsValid())
   else
   {
      ETG_TRACE_ERR(("spi_tclBluetoothClient::vOnStatusServiceAvailable: Invalid message received! "));
   }

}//! end of vOnStatusServiceAvailable()

/**************************************************************************
 ** FUNCTION:  spi_tclBluetoothClient::vPostBTServiceAvailability(t_Bool ...)
 **************************************************************************/
t_Void spi_tclBluetoothClient::vPostBTServiceAvailability(t_Bool bStatus)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vPostBTServiceAvailability() entered with bStatus - %d", ETG_ENUM(BOOL, bStatus)));
   if (NULL != m_rBluetoothCallbacks.fvOnBTServiceAvailability)
   {
      // Notify the registered clients that BT service is not available to process any requests
      m_rBluetoothCallbacks.fvOnBTServiceAvailability(bStatus);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBluetoothClient::vTriggerBTNameUpdate()
***************************************************************************/
t_Void spi_tclBluetoothClient::vTriggerBTNameUpdate(const t_String& rfcszAAPBTAddress)
{
   ETG_TRACE_USR1(("spi_tclBluetoothClient::vTriggerBTNameUpdate() entered with BTAddress - %s", rfcszAAPBTAddress.c_str()));
   t_String szDeviceName = "";

   if (true == IS_VALID_BT_ADDRESS(rfcszAAPBTAddress))
   {
      m_oDevLstLock.s16Lock();
      //! Search for required device in BT DeviceList Map.
      for (auto BTDeviceListMapElem : m_mapBTDeviceList)
      {
         if (rfcszAAPBTAddress == (BTDeviceListMapElem.second.szBTAddress))
         {
            szDeviceName = BTDeviceListMapElem.second.szDeviceName;
            break;
         }
      }
      m_oDevLstLock.vUnlock();
   }
   ETG_TRACE_USR2(("spi_tclBluetoothClient::vTriggerBTNameUpdate() left with device name - %s", szDeviceName.c_str()));
   vNotifyBTDeviceName(rfcszAAPBTAddress, szDeviceName);
}

//lint –restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
