/*!
*******************************************************************************
* \file             spi_tclAudioPolicy.cpp
* \brief            Gen3 Audio Policy class for PSA 
*******************************************************************************
\verbatim
PROJECT:        Gen3 Projects
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Gen3 Audio Policy class for PSA 
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                                  | Modifications
29.10.2013 |  Hari Priya E R(RBEI/ECP2)               | Initial Version
02.12.2016 |  Ramya Murthy(RBEI/ECP2)                 | Improvements to release audio resource early, 
                                                        taken from GM (Fix for SUZUKI-23851)
02.05.2016 |  Ramya Murthy(RBEI/ECP2)                 | Fix to do a take on SA_OFF, only if CarPlay music 
                                                        is playing (PSARCCB-8609)
07.06.2018 |  Ramya Murthy(RBEI/ECP2)                 | Fix for setting source availability before ARL 
                                                        service is available (NCG3D-62108)

\endverbatim
******************************************************************************/

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include <ahl_if.h>

#include "spi_tclCmdInterface.h"
#include "spi_tclAudioPolicy.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclAudioPolicy.cpp.trc.h"
#endif

/******************************************************************************
 | defines
 |----------------------------------------------------------------------------*/
#define CHAR_SET_UTF8  (midw_fi_tclString::FI_EN_UTF8)
#define SRCACT_ON      (midw_fi_tcl_e8_SrcActivity::FI_EN_ON)
#define SRCACT_OFF     (midw_fi_tcl_e8_SrcActivity::FI_EN_OFF)
#define SRCACT_PAUSE   (midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE)

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
static const t_U8 scou8Shift3Bytes = 24;
static const t_U8 scou8Shift2Bytes = 16;
static const t_U8 scou8StereoMixSrcNum = 44;

//! Stores Projection device speech app state
static tenSpeechAppState senDeviceSpeechAppState = e8SPI_SPEECH_UNKNOWN;
static arl_tenActivity senEntSrcActivity = ARL_EN_ISRC_ACT_OFF;
static t_Bool sbIsSrcSetToAvailable = false;

/***************************************************************************
** FUNCTION:  spi_tclAudioPolicy::spi_tclAudioPolicy
(spi_tclCmdInterface* poSpiCmdIntf,ahl_tclBaseOneThreadApp* poMainApp);
***************************************************************************/
spi_tclAudioPolicy::spi_tclAudioPolicy(spi_tclCmdInterface* poSpiCmdIntf, ahl_tclBaseOneThreadApp* poMainApp):
      arl_tclISource((ahl_tclBaseOneThreadApp*) poMainApp),
      m_poSpiCmdIntf(poSpiCmdIntf),
      m_bSpeechChnRequested(false),
      m_bIsARLServiceAvailable(false),
      m_bIsSrcAvailRequestPending(false),
      m_enSrcAvailReason(e8REASON_SAMEMEDIA)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::spi_tclAudioPolicy"));
   SPI_NORMAL_ASSERT((NULL == poMainApp) || (NULL == m_poSpiCmdIntf));
}

/***************************************************************************
** FUNCTION:  spi_tclAudioPolicy::~spi_tclAudioPolicy();
***************************************************************************/
spi_tclAudioPolicy::~spi_tclAudioPolicy()
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::~spi_tclAudioPolicy"));
   senEntSrcActivity = ARL_EN_ISRC_ACT_OFF;
   m_poSpiCmdIntf = NULL;
}

/***************************************************************************
 ** FUNCTION:  bool spi_tclAudioPolicy::bRequestAudioActivation(tU8)
 ***************************************************************************/
tBool spi_tclAudioPolicy::bRequestAudioActivation(tU8 u8SourceNum)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bRequestAudioActivation entered for SrcNum %d ", u8SourceNum));

   tBool bRetVal = false;
   arl_tenSource enSrcNum = static_cast<arl_tenSource> (u8SourceNum);

   //! Update Device Speech app state
   senDeviceSpeechAppState = (ARL_SRC_SPI_VR == enSrcNum) ? (e8SPI_SPEECH_SPEAKING) : senDeviceSpeechAppState;

   m_oSpeechChnReqLock.s16Lock();
   m_bSpeechChnRequested = (ARL_SRC_SPI_VR == enSrcNum) ? (true) : m_bSpeechChnRequested;
   m_oSpeechChnReqLock.vUnlock();

   //! Source should be set to available only for the first time when device is plugged in.
   //! If AutoPlay feature is enabled and source is set to available, it is considered as user action
   //! and has highest priority. This will remove the other source requests from the audio stack.
   //! Fix for PSARCC30-1169 & NCG3D-21391
   if ((ARL_SRC_SPI_MAIN == enSrcNum) && (false == sbIsSrcSetToAvailable))
   {
      bRetVal = bSetSrcAvailability(u8SourceNum, e8SRC_PLAYABLE, e8REASON_NEWMEDIA);
      ETG_TRACE_USR4(("[DESC]::bRequestAudioActivation: bSetSrcAvailability returns value: %d", ETG_ENUM(BOOL, bRetVal)));
   }

   bRetVal = bSetAudioRouteRequest(enSrcNum, ARL_EN_ISRC_ACT_ON);
   ETG_TRACE_USR4(("[DESC]::bRequestAudioActivation: bSetAudioRouteRequest returns value: %d", ETG_ENUM(BOOL, bRetVal)));

   return bRetVal;

}//spi_tclAudioPolicy::bRequestAudioActivation(tU8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  bool spi_tclAudioPolicy::bRequestAudioDeactivation(tU8, tenSrcAvailabilityReason)
 ***************************************************************************/
tBool spi_tclAudioPolicy::bRequestAudioDeactivation(tU8 u8SourceNum, tenSrcAvailabilityReason enSrcAvlReason)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bRequestAudioDeactivation entered for SrcNum %d ", u8SourceNum));

   tBool bRetVal = false;
   arl_tenSource enSrcNum = static_cast<arl_tenSource> (u8SourceNum);

   //! Update Device Speech app state
   senDeviceSpeechAppState = (ARL_SRC_SPI_VR == enSrcNum) ? (e8SPI_SPEECH_END) : (senDeviceSpeechAppState);

   //! SetRouteRequestOFF for SPI_ENTERTAIN source will be ignored by Audio Manager. Source should be set to Unavailable.
   //! Fix for NCG3D-11719.
   if ((ARL_SRC_SPI_MAIN == enSrcNum) && (true == sbIsSrcSetToAvailable))
   {
       bRetVal = bSetSrcAvailability(enSrcNum, e8SRC_NOT_AVAILABLE, enSrcAvlReason);
       ETG_TRACE_USR4(("[DESC]::bRequestAudioDeactivation: bSetSrcAvailability, enSrcAvlReason(%d) returns value: %d", ETG_ENUM(AVAIL_REASON,enSrcAvlReason), ETG_ENUM(BOOL, bRetVal)));
   }
   else
   {
      bRetVal = bSetAudioRouteRequest(enSrcNum, ARL_EN_ISRC_ACT_OFF);
      ETG_TRACE_USR4(("[DESC]::bRequestAudioDeactivation: bSetAudioRouteRequest returns value: %d", ETG_ENUM(BOOL, bRetVal)));
   }

   return bRetVal;

}//spi_tclAudioPolicy::bRequestAudioDeactivation(tU8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  bool spi_tclAudioPolicy::bPauseAudioActivity(tU8)
 ***************************************************************************/
tBool spi_tclAudioPolicy::bPauseAudioActivity(tU8 u8SourceNum)
{
   //SPI_INTENTIONALLY_UNUSED(u8SourceNum);
   tBool bRetVal = true;
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bPauseAudioActivity entered %d ", u8SourceNum));

   return bRetVal;

}//spi_tclAudioPolicy::bPauseAudioActivity(tU8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  void spi_tclAudioPolicy::vStartSourceActivityResult(tU8,tBool)
 ***************************************************************************/
void spi_tclAudioPolicy::vStartSourceActivityResult(tU8 u8SourceNum,
      tBool bError)
{
   //SPI_INTENTIONALLY_UNUSED(bError);
   ETG_TRACE_USR1(("spi_tclAudioPolicy::vStartSourceActivityResult entered %d %d ", u8SourceNum, bError));
   arl_tSrcActivity oSrcActivity;

   arl_tenSource enSrcNum = static_cast<arl_tenSource> (u8SourceNum);
   oSrcActivity.enType = SRCACT_ON;

   //arl_tclISource::vSourceActivityResult(enSrcNum, rfcoSrcActivity);
   vSourceActivityResult(enSrcNum, oSrcActivity);

}//spi_tclAudioPolicy::vStartSourceActivityResult(tU8 tBool)

/***************************************************************************
 ** FUNCTION:  void spi_tclAudioPolicy::vStopSourceActivityResult(tU8,tBool)
 ***************************************************************************/
void spi_tclAudioPolicy::vStopSourceActivityResult(tU8 u8SourceNum,
      tBool bError)
{
   //SPI_INTENTIONALLY_UNUSED(bError);
   ETG_TRACE_USR1(("spi_tclAudioPolicy::vStopSourceActivityResult entered %d %d ", u8SourceNum, bError));

   arl_tSrcActivity oSrcActivity;

   arl_tenSource enSrcNum = static_cast<arl_tenSource> (u8SourceNum);
   oSrcActivity.enType = SRCACT_OFF;

   //arl_tclISource::vSourceActivityResult(enSrcNum, rfcoSrcActivity);
   vSourceActivityResult(enSrcNum, oSrcActivity);

}//spi_tclAudioPolicy::vStopSourceActivityResult(tU8 tBool)

/***************************************************************************
 ** FUNCTION:  void spi_tclAudioPolicy::vPauseSourceActivityResult(tU8,tBool)
 ***************************************************************************/
void spi_tclAudioPolicy::vPauseSourceActivityResult(tU8 u8SourceNum,
      tBool bError)
{
   //SPI_INTENTIONALLY_UNUSED(bError);
   ETG_TRACE_USR1(("spi_tclAudioPolicy::vPauseSourceActivityResult entered %d %d ", u8SourceNum, bError));

   arl_tSrcActivity oSrcActivity;

   arl_tenSource enSrcNum = static_cast<arl_tenSource> (u8SourceNum);
   oSrcActivity.enType = SRCACT_PAUSE;

   //arl_tclISource::vSourceActivityResult(enSrcNum, rfcoSrcActivity);
   vSourceActivityResult(enSrcNum, oSrcActivity);

}//spi_tclAudioPolicy::vPauseSourceActivityResult(tU8 tBool)

/***************************************************************************
 ** FUNCTION:  tVoid spi_tclAudioPolicy::bSetSourceAvailability(tU8,tenSrcAvailability)
 ***************************************************************************/
tBool spi_tclAudioPolicy::bSetSrcAvailability(tU8 u8SourceNum,
         tenSrcAvailability enAvailability, tenSrcAvailabilityReason enAvailReason)
{
   m_oServiceStateLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bSetSrcAvailability entered: "
            "SrcNum = %d, Src availability = %d , Src Availability reason = %d",
            u8SourceNum, ETG_ENUM(SRC_AVAILABILITY, enAvailability), ETG_ENUM(AVAIL_REASON, enAvailReason)));

   tBool bRetVal = false;

   //! Fix for NCG3D-62108: Set source availability only after ARL source becomes available
   if ((false == m_bIsARLServiceAvailable) && (ARL_SRC_SPI_MAIN == u8SourceNum))
   {
      ETG_TRACE_USR4(("[DESC]::bSetSrcAvailability: Source %d request not sent. Will be sent once once ARL service is available",
               ETG_ENUM(SRC_AVAILABILITY, enAvailability)));
      m_bIsSrcAvailRequestPending = (e8SRC_PLAYABLE == enAvailability);
      m_enSrcAvailReason = enAvailReason;
   }
   else
   {
      //! Source should be set to available only for the first time when device is plugged in.
      //! If AutoPlay feature is enabled and source is set to available, it is considered as user action
      //! and has highest priority. This will remove the other source requests from the audio stack.
      //! Fix for PSARCC30-1169 & NCG3D-21391
      sbIsSrcSetToAvailable = ((ARL_SRC_SPI_MAIN == u8SourceNum) && (e8SRC_PLAYABLE == enAvailability));

      /*Sub Source ID(Param 4) is 0 which is the default value.*/
      bRetVal = bSetSourceAvailability(static_cast<arl_tenSrcAvailability>(enAvailability),
               static_cast<arl_tenAvailabilityReason>(enAvailReason), static_cast<arl_tenSource> (u8SourceNum));

      ETG_TRACE_USR2(("[PARAM]::bSetSrcAvailability: sbIsSrcSetToAvailable = %d", ETG_ENUM(BOOL, sbIsSrcSetToAvailable)));
   }
   m_oServiceStateLock.vUnlock();
   
   return bRetVal;

}//spi_tclAudioPolicy::bSetSourceAvailability(tU8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  tVoid spi_tclAudioPolicy::vSetServiceAvailable()
 ***************************************************************************/
t_Void spi_tclAudioPolicy::vSetServiceAvailable(tBool bAvail)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::vSetServiceAvailable entered  bAvail = %d ", bAvail));
   vSetSrcAvailable(bAvail);
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudioPolicy::bSetAudioDucking()
***************************************************************************/
t_Bool spi_tclAudioPolicy::bSetAudioDucking(const t_U8 cou8SrcNum, const t_U16 cou16RampDuration,
      const t_U8 cou8VolumeindB,
      const tenDuckingType coenDuckingType)
{
   SPI_INTENTIONALLY_UNUSED(cou8SrcNum);

   t_U32 u32DuckingType = coenDuckingType;
   t_U32 u32VolumeLevel = cou8VolumeindB;
   t_Bool bRetVal =false;
   //! Populate ducking value where the first two byte (from LSB) represents Ramp duration
   //! third byte represents Volume level in dB, fourth byte represents Ducking Type
   t_U32 u32DuckingVal = (u32DuckingType << scou8Shift3Bytes) | (u32VolumeLevel
            << scou8Shift2Bytes) | (cou16RampDuration);

   if (scou8StereoMixSrcNum == cou8SrcNum)
   {
       bRetVal = bSetAudioProperty(ARL_SRC_SPI_ML_INFO,
               ARL_SRC_IPOD_DUCKING, u32DuckingVal);
   }
   else
   {
       bRetVal = bSetAudioProperty(ARL_SRC_SPI_MIX,
               ARL_SRC_IPOD_DUCKING, u32DuckingVal);
   }
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bSetAudioDucking left with u32DuckingVal = 0x%x, "
         "SetAudioProperty result = %d ", u32DuckingVal, ETG_ENUM(BOOL, bRetVal)));

   return true;
}

/***************************************************************************
** FUNCTION:  tBool spi_tclAudioPolicy::bOnSrcActivity(
arl_tenSource enSrcNum, const arl_tSrcActivity& rfcoSrcActivity)
 ***************************************************************************/
tBool spi_tclAudioPolicy::bOnSrcActivity(arl_tenSource enSrcNum,
      const arl_tSrcActivity& rfcoSrcActivity)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bOnSrcActivity entered %d %d ", enSrcNum, rfcoSrcActivity.enType));
   ETG_TRACE_USR4(("[PARAM]::bOnSrcActivity: bIsARLServiceAvailable = %d ", ETG_ENUM(BOOL, m_bIsARLServiceAvailable)));

   tU8 u8SrcNum = static_cast<tU8> (enSrcNum);
   tBool bRetVal = (NULL != m_poSpiCmdIntf);
   arl_tenActivity enSrcActivity = ARL_EN_ISRC_ACT_OFF;

   m_oSpeechChnReqLock.s16Lock();
   m_bSpeechChnRequested = (ARL_SRC_SPI_VR == enSrcNum)? (false) : m_bSpeechChnRequested;
   t_Bool bIsSpeechChnRequested = m_bSpeechChnRequested;
   m_oSpeechChnReqLock.vUnlock();

   // Pointer NULL check kept for LINT Compliance
   if (NULL != m_poSpiCmdIntf)
   {
      switch (rfcoSrcActivity.enType)
      {
         case SRCACT_ON:
         {
            ETG_TRACE_USR4((" Source Activity ON received for Src Num: %d", u8SrcNum));
            m_poSpiCmdIntf->vOnStartSourceActivity(u8SrcNum);
            enSrcActivity = ARL_EN_ISRC_ACT_ON;
         }
            break;

         case SRCACT_PAUSE:
         {
            ETG_TRACE_USR4((" Source Activity PAUSE received for Src Num: %d", u8SrcNum));
            m_poSpiCmdIntf->vOnPauseSourceActivity(u8SrcNum);
            enSrcActivity = ARL_EN_ISRC_ACT_PAUSE;
         }
            break;

         case SRCACT_OFF:
         {
            ETG_TRACE_USR4((" Source Activity OFF received for Src Num: %d", u8SrcNum));
            m_poSpiCmdIntf->vOnStopSourceActivity(u8SrcNum);
            enSrcActivity = ARL_EN_ISRC_ACT_OFF;
         }
            break;

         default:
         {
            ETG_TRACE_USR4((" Unknown Source Activity received for Src Num: %d", u8SrcNum));
            bRetVal = false;
         }
            break;

      }//switch(rfcoSrcActivity.enType)

      t_U32 u32SelDevHandle = m_poSpiCmdIntf->u32GetSelectedDeviceHandle();

      bool isSpeechActive = (e8SPI_SPEECH_SPEAKING == senDeviceSpeechAppState) || (bIsSpeechChnRequested);
      if ((ARL_SRC_SPI_MAIN == enSrcNum) && (false == isSpeechActive))
      {
         switch (enSrcActivity)
         {
            case ARL_EN_ISRC_ACT_OFF:
            case ARL_EN_ISRC_ACT_PAUSE:
            {
               //! Set audio context to MAIN if HU has taken the main audio, else to UNKNOWN.
               //! (will be set to correct state once AudioContext update is received)
               //! Note: SPI should take/borrow audio focus to MD asap, to stop it from streaming music
               tenAudioContext enAudioContext = (ARL_EN_ISRC_ACT_OFF == enSrcActivity) ?
                     (e8SPI_AUDIO_MAIN) : (e8SPI_AUDIO_UNKNOWN);
               m_poSpiCmdIntf->vSetAccessoryAudioContext(u32SelDevHandle,
                     enAudioContext, true, corEmptyUsrContext, e8DEV_TYPE_ANDROIDAUTO, false);
               m_poSpiCmdIntf->vSetAccessoryAudioContext(u32SelDevHandle,
                     enAudioContext, true, corEmptyUsrContext, e8DEV_TYPE_CARLIFE, false);
            }
            break;
            case ARL_EN_ISRC_ACT_ON:
            {
               //Source activity pause to on scenario : Unborrow the previous set, UNKNOWN context(This is to handle the 
               //scenarios where there is no audio context update received between SA PAUSE TO SA ON)
               if (ARL_EN_ISRC_ACT_PAUSE == senEntSrcActivity)
               {
                  m_poSpiCmdIntf->vSetAccessoryAudioContext(u32SelDevHandle,
                        e8SPI_AUDIO_UNKNOWN, false, corEmptyUsrContext, e8DEV_TYPE_ANDROIDAUTO, false);
                  m_poSpiCmdIntf->vSetAccessoryAudioContext(u32SelDevHandle,
                        e8SPI_AUDIO_UNKNOWN, false, corEmptyUsrContext, e8DEV_TYPE_CARLIFE, false);
               }
            }
            break;
            default:
               ETG_TRACE_ERR(("[ERR]::bOnSrcActivity: Invalid Source Activity %d ", enSrcActivity));
            break;
         }
      } // if ((ARL_SRC_SPI_MAIN == enSrcNum) && (false == isSpeechActive))


      //! If SA_OFF is received while CarPlay media is playing, take the audio resource so that
      //! audio device is released asap.
      t_Bool bIsCarPlayMainAudioStopped =
            (e8DEV_TYPE_DIPO == m_poSpiCmdIntf->enGetDeviceCategory(u32SelDevHandle)) &&
            (ARL_SRC_SPI_MAIN == enSrcNum) &&
            (ARL_EN_ISRC_ACT_ON == senEntSrcActivity) &&
            (ARL_EN_ISRC_ACT_OFF == enSrcActivity);

      m_oServiceStateLock.s16Lock();
      t_Bool bIsARLServiceAvailable = m_bIsARLServiceAvailable;
      m_oServiceStateLock.vUnlock();

      if ((true == bIsCarPlayMainAudioStopped) && (true == bIsARLServiceAvailable))
      {
         m_poSpiCmdIntf->vSetAccessoryAudioContext(u32SelDevHandle,
                  e8SPI_AUDIO_MAIN, true, corEmptyUsrContext, e8DEV_TYPE_DIPO, false);
      }

      //! Store the latest entertainment source status
      senEntSrcActivity = (ARL_SRC_SPI_MAIN == enSrcNum) ? (enSrcActivity) : (senEntSrcActivity);

   }//if(NULL != m_poSpiCmdIntf)

   //! Fix for NCG3D-116064: Application is no longer responsible for maintaining
   //! route request status since there are timing issues observed when application
   //! requests deactivation and activation in quick succession. It will be handled by ARL.
   /*ETG_TRACE_USR4((" Audio Route Request %d is sent for Src %d ", enSrcActivity, enSrcNum));
   bSetAudioRouteRequest(enSrcNum, enSrcActivity);*/

   return bRetVal;

} //spi_tclAudioPolicy::bOnSrcActivityStart(arl_tenSource,const arl_tSrcActivity&)

/***************************************************************************
** FUNCTION:  tBool spi_tclAudioPolicy::bOnAllocate(arl_tenSource enSrcNum,
                        const arl_tAllocRouteResult& rfcoAllocRoute)
***************************************************************************/
tBool spi_tclAudioPolicy::bOnAllocate(arl_tenSource enSrcNum,
                             const arl_tAllocRouteResult& rfcoAllocRoute)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bOnAllocate entered %d ", enSrcNum));
   tBool bRetVal = true;
   trAudSrcInfo rSrcInfo;

   if (NULL != m_poSpiCmdIntf)
   {
      tU8 u8SrcNum = static_cast<tU8> (enSrcNum);

      t_U8 u8InListSize = static_cast <t_U8>(rfcoAllocRoute.listInputDev.strALSADev.size());
      t_U8 u8OutListSize = static_cast <t_U8>(rfcoAllocRoute.listOutputDev.strALSADev.size());

      ETG_TRACE_USR4((" Audio Device List Size: Input [%d], Output [%d]", u8InListSize, u8OutListSize)); 

      /* In case of Main Audio and Mix Audio source , we get only one ALSA device names.*/
      if ((1 == u8OutListSize)&& (0 == u8InListSize))
      {
         GET_STRINGDATA_FROM_FI_STRINGOBJ((rfcoAllocRoute.listOutputDev.strALSADev.front()), CHAR_SET_UTF8, rSrcInfo.rMainAudDevNames.szOutputDev);
         ETG_TRACE_USR4(("bOnAllocate: ALSAOutDev : %s ", rSrcInfo.rMainAudDevNames.szOutputDev.c_str()));
      }
      /* In case of Phone Audio and VR Audio source , we get only four ALSA device names(2-ECNR and 2- Main Audio).*/
      else if((3 == u8OutListSize) && (3 == u8InListSize))
      {
         GET_STRINGDATA_FROM_FI_STRINGOBJ((rfcoAllocRoute.listInputDev.strALSADev.at(0)), CHAR_SET_UTF8, rSrcInfo.rMainAudDevNames.szInputDev);
         ETG_TRACE_USR4(("bOnAllocate: ALSAInDev : %s ", rSrcInfo.rMainAudDevNames.szInputDev.c_str()));

         GET_STRINGDATA_FROM_FI_STRINGOBJ((rfcoAllocRoute.listInputDev.strALSADev.at(1)), CHAR_SET_UTF8, rSrcInfo.rEcnrAudDevNames.szInputDev);
         ETG_TRACE_USR4(("bOnAllocate: ECNR ALSAInDev : %s ", rSrcInfo.rEcnrAudDevNames.szInputDev.c_str()));

         GET_STRINGDATA_FROM_FI_STRINGOBJ((rfcoAllocRoute.listOutputDev.strALSADev.at(1)), CHAR_SET_UTF8, rSrcInfo.rEcnrAudDevNames.szOutputDev);
         ETG_TRACE_USR4(("bOnAllocate: ECNR ALSAOutDev : %s ", rSrcInfo.rEcnrAudDevNames.szOutputDev.c_str()));

         GET_STRINGDATA_FROM_FI_STRINGOBJ((rfcoAllocRoute.listOutputDev.strALSADev.at(2)), CHAR_SET_UTF8, rSrcInfo.rMainAudDevNames.szOutputDev);
         ETG_TRACE_USR4(("bOnAllocate: ALSAOutDev : %s ", rSrcInfo.rMainAudDevNames.szOutputDev.c_str()));
      }
      else
      {
         ETG_TRACE_ERR(("bOnAllocate: Invalid ALSA Device Names "));
         bRetVal = false;
      }
      bRetVal = (true == bRetVal)? m_poSpiCmdIntf->bOnRouteAllocateResult(u8SrcNum, rSrcInfo) : bRetVal;

   }//if(NULL != m_poSpiCmdIntf)
   return bRetVal;
}

/************************************************************************************
 ** FUNCTION:  tBool spi_tclAudioPolicy::bOnDeAllocate(arl_tenSource enSrcNum)
 *************************************************************************************/
tBool spi_tclAudioPolicy::bOnDeAllocate(arl_tenSource enSrcNum)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bOnDeAllocate entered %d ", enSrcNum));

   tBool bRetVal = (NULL != m_poSpiCmdIntf);
   tU8 u8SrcNum = static_cast<tU8> (enSrcNum);

   if (NULL != m_poSpiCmdIntf)
   {
      m_poSpiCmdIntf->vOnRouteDeAllocateResult(u8SrcNum);
   }//if(NULL != m_poSpiCmdIntf)

   return bRetVal;
}

/***************************************************************************
** FUNCTION:  tVoid spi_tclAudioPolicy::vOnError()
 ***************************************************************************/
tVoid spi_tclAudioPolicy::vOnError(tU8 u8SrcNum, arl_tenISourceError cenError)
{
   tenAudioError enAudiError = e8_AUDIOERROR_UNKNOWN;

   if (NULL != m_poSpiCmdIntf)
   {
      switch (cenError)
      {
         case ARL_EN_ISRC_REQAVACT_ERR:
         {
            enAudiError = e8_AUDIOERROR_AVACTIVATION;
         }
            break;

         case ARL_EN_ISRC_REQAVDEACT_ERR:
         {
            enAudiError = e8_AUDIOERROR_AVDEACTIVATION;
         }
            break;

         case ARL_EN_ISRC_ALLOC_ERR:
         {
            enAudiError = e8_AUDIOERROR_ALLOCATE;
         }
            break;

         case ARL_EN_ISRC_DEALLOC_ERR:
         {
            enAudiError = e8_AUDIOERROR_DEALLOCATE;
         }
            break;

         case ARL_EN_ISRC_SRCACT_ON_ERR:
         {
            enAudiError = e8_AUDIOERROR_STARTSOURCEACT;
         }
            break;

         case ARL_EN_ISRC_SRCACT_OFF_ERR:
         {
            enAudiError = e8_AUDIOERROR_STOPSOURCEACT;
         }
            break;

         default:
         {
            enAudiError = e8_AUDIOERROR_GENERIC;
         }
            break;
      }//switch(cenError)

      ETG_TRACE_USR4(("spi_tclAudioPolicy::vOnError: Audio Error for Source Number %d Error = %d",
               u8SrcNum, enAudiError));

      //! Store audio source as inactive in case of activation errors
      t_Bool bIsSourceInactive = ((e8_AUDIOERROR_AVACTIVATION == enAudiError) ||
            (e8_AUDIOERROR_ALLOCATE == enAudiError) ||  (e8_AUDIOERROR_STARTSOURCEACT == enAudiError));
      if (true == bIsSourceInactive)
      {
         arl_tenSource enSrcNum = static_cast<arl_tenSource> (u8SrcNum);

         m_oSpeechChnReqLock.s16Lock();
         m_bSpeechChnRequested = (ARL_SRC_SPI_VR == enSrcNum)? (false) : m_bSpeechChnRequested;
         t_Bool bIsSpeechChnRequested = m_bSpeechChnRequested;
         m_oSpeechChnReqLock.vUnlock();

         m_poSpiCmdIntf->vOnStopSourceActivity(u8SrcNum);

         //! Set audio context as invalid (will be set to correct state once AudioContext update is received)
         //! Note: SPI should take/borrow audio focus to MD asap, to stop it from streaming music.
         if ((ARL_SRC_SPI_MAIN == enSrcNum) && (e8SPI_SPEECH_SPEAKING != senDeviceSpeechAppState) && (false == bIsSpeechChnRequested))
         {
            m_poSpiCmdIntf->vSetAccessoryAudioContext(m_poSpiCmdIntf->u32GetSelectedDeviceHandle(),
                  e8SPI_AUDIO_UNKNOWN, true, corEmptyUsrContext, e8DEV_TYPE_ANDROIDAUTO);
         }
      }//if (true == bIsSourceInactive)

      m_poSpiCmdIntf->vOnAudioError(u8SrcNum,enAudiError);
   }//if (NULL != m_poSpiCmdIntf)

}// spi_tclAudioPolicy::vOnError(tU8 u8SrcNum,arl_tenISourceError cenError)


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudioPolicy::vOnDeviceAppStateChange(tU8)
 ***************************************************************************/
t_Void spi_tclAudioPolicy::vOnDeviceAppStateChange(tenSpeechAppState enSpeechAppState)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::vOnDeviceAppStateChange entered "));

   //! Store Device speech app state
   senDeviceSpeechAppState = enSpeechAppState;
}

/***************************************************************************
 ** FUNCTION:  tBool spi_tclAudioPolicy::bSetSourceMute()
 ***************************************************************************/
tBool spi_tclAudioPolicy::bSetSourceMute(tU8 u8SourceNum, t_Bool bIsMute)
{
   ETG_TRACE_USR1(("spi_tclAudioPolicy::bSetSourceMute entered with SrcNum = %d, MuteRequired = %d",
            u8SourceNum, ETG_ENUM(BOOL, bIsMute)));

   /*
   @Notes:  
   Mute State: Mute state to be set only for Media source. This is applicable for specific source. 
               if a MUTE is enabled on a specific source, this must be DEMUTED for audio.
			   
			   This is required to avoid audio glitched when there is no device is connected. 
			   So this should be set whenever SPI source is active and SPI device is not connected.
			   
   Volume Lock: Volume lock is used to lock the volume change. This is to prevent User from changing the volume when there is no SPI device connected and
               SPI session is not active. 
			   
			   Volume lock is overall setting. If we enable Volume lock for SPI entertain doesn't disable when source is allocated for FM/native VR/SPI VR,
			   User will not be able to change the volume. Hence volume lock should be remove as soon as the source is dealloacted to different source.
			   
   */

   // Demute the audio source (should be muted only on device disconnection)
   arl_tenMuteAction enMuteAction = ARL_EN_MUTEACTION_DEMUTE;

   // Reset the volume lock by default (should be locked only on device disconnection)
   arl_tenSourcePropType enPropType = ARL_SRC_VOLUME_LOCK;
   tS32 s32PropValue = 0; // EN_INT_VOLUME_LOCK_MODE_OFF

   if (true == bIsMute)
   {
      // Mute the audio source to avoid any glitches.
      enMuteAction = ARL_EN_MUTEACTION_MUTE;

      // Set the volume lock so that volume change is disabled.
      s32PropValue = 1; // EN_INT_VOLUME_LOCK_MODE_TOTAL
   }

   tBool bResult = false;
   if (m_rAudConfig.bMuteEnabled)
   {
      ETG_TRACE_USR4(("[PARAM]:: bSetSourceMute - SetMuteState called with source number = %d, mute action = %d",
               u8SourceNum, ETG_ENUM(MUTE_ACTION, enMuteAction)));
      bResult = bSetMuteState(static_cast<arl_tenSource> (u8SourceNum), enMuteAction);
   }
   if (m_rAudConfig.bVolLockEnabled)
   {
      ETG_TRACE_USR4(("[PARAM]:: bSetSourceMute - SetAudioProperty called with source number = %d, volume lock mode = %d",
               u8SourceNum, ETG_ENUM(VOL_LOCK_MODE, s32PropValue)));
      bResult = bSetAudioProperty(static_cast<arl_tenSource> (u8SourceNum), enPropType, s32PropValue);
   }
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudioPolicy::vOnNewAppStateNotify(..
 ***************************************************************************/
tVoid spi_tclAudioPolicy::vOnNewAppStateNotify(tU32 u32OldAppState,tU32 u32AppState)
{
   m_oServiceStateLock.s16Lock();
	ETG_TRACE_USR1(("spi_tclAudioPolicy::vOnNewAppStateNotify(): OldAppState:%d, NewAppState:%d", u32OldAppState,u32AppState));
	if (AMT_C_U32_STATE_OFF == u32AppState)
	{
		ETG_TRACE_USR4(("[DESC]::vOnNewAppStateNotify():Off state received"));
		//Service will become unavailable before the system state changes to OFF
		m_bIsARLServiceAvailable = false;
	}
   m_oServiceStateLock.vUnlock();
	
	arl_tclISource::vOnNewAppStateNotify(u32OldAppState,u32AppState);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudioPolicy::vOnServiceAvailable(..
 ***************************************************************************/
tVoid spi_tclAudioPolicy::vOnServiceAvailable(tVoid)
{
   m_oServiceStateLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclAudioPolicy::vOnServiceAvailable() entered"));
   //In system state OFF, service will not be available.
   m_bIsARLServiceAvailable = true;

   ETG_TRACE_USR4(("[PARAM]::vOnServiceAvailable: SrcAvailRequestPending = %d", ETG_ENUM(BOOL, m_bIsSrcAvailRequestPending)));

   if (m_bIsSrcAvailRequestPending)
   {
      m_bIsSrcAvailRequestPending = false;
      bSetSrcAvailability(ARL_SRC_SPI_MAIN, e8SRC_PLAYABLE, m_enSrcAvailReason);
   }
   m_oServiceStateLock.vUnlock();
}
 
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudioPolicy::vOnServiceUnavailable(..
 ***************************************************************************/
tVoid spi_tclAudioPolicy::vOnServiceUnavailable(tVoid)
{
   m_oServiceStateLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclAudioPolicy::vOnServiceUnavailable() entered"));
   m_bIsARLServiceAvailable = false;
   m_oServiceStateLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudioPolicy::vSetAudioConfig(...)
 ***************************************************************************/
t_Void spi_tclAudioPolicy::vSetAudioConfig(const trAudioConfig& corfConfig)
{
   //! Store audio config
   m_rAudConfig = corfConfig;
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
