/***********************************************************************/
/*!
 * \file  spi_tclMySPINTAWrapper.h
 * \brief  Implementation of the Class spi_tclMySPINTAWrapper
 *************************************************************************
 \verbatim


 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification

 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "mySPINTypes.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_MYSPINWRAPPER
#include "trcGenProj/Header/spi_tclMySPINTAWrapper.cpp.trc.h"
#endif
#endif

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
static const t_U32 scou32VEHICLE_DATA_KEY_IS_MOVING = 2;
static const t_U32 scou32VEHICLE_DATA_KEY_IS_NIGHT = 3;

#define BUFFER_MAX_LENGTH 10
/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
#include "spi_tclMySPINTAWrapper.h"

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINTAWrapper::spi_tclMySPINTAWrapper()
 ***************************************************************************/
spi_tclMySPINTAWrapper::spi_tclMySPINTAWrapper() :
   m_poMySPINInstance(NULL), m_bIsInitialized(false),m_rMySPINContext(NULL),
   m_MySPINCapabilities(0)
{
   ETG_TRACE_USR1(("Creating Wrapper for mySPIN"));

}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINTAWrapper::~spi_tclMySPINTAWrapper()
 ***************************************************************************/
spi_tclMySPINTAWrapper::~spi_tclMySPINTAWrapper()
{
   ETG_TRACE_USR1(("Destroying Wrapper for mySPIN"));
   m_poMySPINInstance = NULL;
   //make lint happy
   m_rMySPINContext = NULL;
   //Clear Connected device list
   m_mapConnectedDevices.clear();
   m_MySPINCapabilities = 0;
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vInitMySPIN()
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vInitMySPIN()
{
   if (false == m_bIsInitialized)
   {
      MSPIN_ERROR initStatus = MSPIN_Init();

      m_bIsInitialized = (MSPIN_SUCCESS == initStatus);
      ETG_TRACE_USR4(("[PARAM]:mySPIN initialization Status : %d", ETG_ENUM(MYSPIN_ERROR, initStatus)));
   }
}

/***************************************************************************
 ** FUNCTION:  MSPIN_Instance_t* spi_tclMySPINTAWrapper::getMySPINInstance(...)
 ***************************************************************************/
MSPIN_Instance_t* spi_tclMySPINTAWrapper::getMySPINInstance(t_U32 u32DeviceHandle)
{
   if (NULL == m_poMySPINInstance)
   {
      ETG_TRACE_USR1(("Create a new mySPIN instance"));
      m_rMySPINContext = new trMySPINContext;
      if (NULL != m_rMySPINContext)
      {
         m_rMySPINContext->u32DeviceHandle = u32DeviceHandle;
         m_poMySPINInstance = MSPIN_CreateInstance(m_rMySPINContext);
      }
      // TODO create a MAP to store different instances to support multiple devices
      // simultaneously.
   }

   return (m_poMySPINInstance);
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vRegisterSessionCbs(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vRegisterSessionCbs(t_U32 u32DeviceHandle, trMySPINSessionCbs rMySPINSessionCbs)
{
   // Get the mySPIN Instance and register for respective call backs

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      // Register for session related callbacks now
      MSPIN_SetAppTransitionStatusCallback(poMySPINInstance, rMySPINSessionCbs.vAppTransitionCb);
      MSPIN_SetAppInactiveCallback(poMySPINInstance, rMySPINSessionCbs.vAppInactiveCb);
      MSPIN_SetInitiatePhoneCallCallback(poMySPINInstance, rMySPINSessionCbs.vInitiatePhoneCallCb);
      MSPIN_SetPhoneCallStatusCallback(poMySPINInstance, rMySPINSessionCbs.vPhoneCallStatusCallback);
      MSPIN_SetCustomDataStringCallback(poMySPINInstance, rMySPINSessionCbs.vCustomDataStringCallback);
      //MSPIN_SetCustomIntCallback(poMySPINInstance,rMySPINSessionCbs.vCustomDataIntCallback);
      MSPIN_SetErrorCallback(poMySPINInstance, rMySPINSessionCbs.vErrorCb);
      //Move it later
      
      MSPIN_SetVoiceSupport(poMySPINInstance,
               MSPIN_VOICESESSION_FULL_SCO_SUPPORT,
               MSPIN_VOICESESSION_ONLY_MIC_AND_SPEAKER_TOGETHER);
      MSPIN_EnablePTTCustomKey(poMySPINInstance, true);  
      MSPIN_EnableCustomKeys(poMySPINInstance, MSPIN_CUSTOMDATA_KEYPOROFILE_6KEYS_AND_ROTARY);
      MSPIN_SetPTTAvailableCallback(poMySPINInstance, rMySPINSessionCbs.vPTTAvailableCallback);
      MSPIN_SetVoiceSessionRequestCallback(poMySPINInstance, rMySPINSessionCbs.vVoiceSessionCallback);
      
      m_MySPINCapabilities |= MSPIN_CLIENTCAPABILITIES_INITIATEPHONECALL;
      MSPIN_SetCapabilities(poMySPINInstance, m_MySPINCapabilities);

      ETG_TRACE_USR2(("Session call backs registered"));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vRegisterVideoCbs(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vRegisterVideoCbs(t_U32 u32DeviceHandle, trMySPINVideoCbs rMySPINVideoCbs)
{
   // Get the mySPIN Instance and register for respective call backs

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      // Register for video related call backs now
      MSPIN_SetFirstFrameRenderedCallback(poMySPINInstance, rMySPINVideoCbs.vFirstFrameRenderedCb);

      MSPIN_SetLayerManagerInitializedCallback(poMySPINInstance, rMySPINVideoCbs.vLayerManagerInitCb);

      MSPIN_SetLayerManagerDeinitializedCallback(poMySPINInstance, rMySPINVideoCbs.vLayerManagerDeinitCb);
	  
	  MSPIN_EnableTouchFiltering(poMySPINInstance, true);

      ETG_TRACE_USR1(("Video call backs registered"));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vRegisterVehicleDataCbs(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vRegisterVehicleDataCbs(t_U32 u32DeviceHandle, trVehicleDataCbs rVehicleDataCbs,t_Bool bIsNativeNavigationEnabled)
{
   // Get the mySPIN Instance and register for respective call backs

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      // Register for Vehicle Data related call backs now
      MSPIN_SetVehicleDataRequestCallback(poMySPINInstance, rVehicleDataCbs.vVehicleDataRequestCb);

      if(bIsNativeNavigationEnabled)
      {
          m_MySPINCapabilities |= MSPIN_CLIENTCAPABILITIES_POSITIONINFORMATION;
      }

      //m_MySPINCapabilities |= MSPIN_CLIENTCAPABILITIES_NAVIGATETO;

      MSPIN_SetNavigateToCallback(poMySPINInstance, rVehicleDataCbs.vNavigateToCallback);
      ETG_TRACE_USR1(("Vehicle Data call backs registered"));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vRegisterAudioCbs(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vRegisterAudioCbs(t_U32 u32DeviceHandle, trMYSPINAudioCbs rMYSPINAudioCbs)
{
   // Get the mySPIN Instance and register for respective call backs

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      MSPIN_SetAudioRequestCallback (poMySPINInstance,rMYSPINAudioCbs.vAudioRequestCb);

      ETG_TRACE_USR2(("[DESC]:Set client capabilities to handle Audio"));
      m_MySPINCapabilities |= MSPIN_CLIENTCAPABILITIES_AUDIOHANDLING;
   }
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSetAccessoryPropertiesAoap(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetAccessoryPropertiesAoap(t_U32 u32DeviceHandle, const trMySPINAccInfo& corfrAccInfo)
{
   ETG_TRACE_USR1(("Entering vSetAccessoryPropertiesAoap"));

   // Get the mySPIN Instance
   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      // Set the accessory properties which are required to
      // perform switch of the device
      MSPIN_AccessoryPropertiesAoap_t rAccProp;
      rAccProp.vendor = corfrAccInfo.szVendorName.c_str();
      rAccProp.model = corfrAccInfo.szModel.c_str();
      rAccProp.description = corfrAccInfo.szDescription.c_str();
      rAccProp.version = corfrAccInfo.szVersion.c_str();
      rAccProp.uri = corfrAccInfo.szUri.c_str();
      rAccProp.serial = corfrAccInfo.szSerial.c_str();

      MSPIN_SetAccessoryPropertiesAoap(poMySPINInstance, &rAccProp);

      ETG_TRACE_USR1(("mySPIN AOAP Accessory Properties set"));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSetFrameProperties(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetFrameProperties(t_U32 u32DeviceHandle, t_U32 u32LayerId, t_U32 u32SurfaceId,
         t_U32 u32WidthInPixels, t_U32 u32HeightInPixels, t_U32 u32WidthInMm, t_U32 u32HeightInMm,
         MSPIN_PIXEL_FORMAT enPixelFormat, MSPIN_FRAME_COMPRESSION enPixelCompression, t_U32 u32PreferredCompression)
{
   ETG_TRACE_USR1(("Entering vSetFrameProperties"));
   // Get the mySPIN Instance
   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      MSPIN_FrameProperties_t rFrameProp;
      rFrameProp.width = u32WidthInPixels;
      rFrameProp.height = u32HeightInPixels;
      rFrameProp.physicalWidth = u32WidthInMm;
      rFrameProp.physicalHeight = u32HeightInMm;
      rFrameProp.format = enPixelFormat;
      rFrameProp.compression = enPixelCompression;
      rFrameProp.preferredCompression = static_cast<MSPIN_FRAME_COMPRESSION> (u32PreferredCompression);
      rFrameProp.layerId = u32LayerId;
      rFrameProp.surfaceId = u32SurfaceId;

      MSPIN_SetFrameProperties(poMySPINInstance, &rFrameProp);
      ETG_TRACE_USR1(("mySPIN Frame Properties are set"));
   }
}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::enConnectAoap(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::enConnectAoap(t_U32 u32DeviceHandle, const trMySPINDevInfo& corfrDevInfo,
         t_Bool& rfbAudioSupport)
{

   ETG_TRACE_USR1(("ConnectAOAP - START"));

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;

   if (NULL != poMySPINInstance)
   {
      MSPIN_ERROR connectStatus = MSPIN_ConnectAoap(poMySPINInstance,
               corfrDevInfo.u32VendorID,
               corfrDevInfo.u32ProductID,
               corfrDevInfo.szSerial.c_str(),
               &rfbAudioSupport);
      ETG_TRACE_USR4(("[PARAM]::enConnectAoap-ConnectAoap Status %d", ETG_ENUM(MYSPIN_ERROR, connectStatus)));

      enErrorCode = vSetMySPINError(connectStatus);
	  
      //Maintain a list of successfully connected devices
      ETG_TRACE_USR4(("ConnectAOAP Status %d", ETG_ENUM(MYSPIN_ERROR, connectStatus)));

      // Save connected devices.
      if (MSPIN_SUCCESS == connectStatus)
      {
         m_mapConnectedDevices[u32DeviceHandle] = corfrDevInfo;

         //get the capability of the phone
         t_U32 u32ServerCaps = MSPIN_GetServerCapabilities(poMySPINInstance);

         ETG_TRACE_USR4(("######### CAPABILITIES ##########[%d]",u32ServerCaps));

         if (MSPIN_SERVERCAPABILITIES_AUDIOHANDLING == (u32ServerCaps & MSPIN_SERVERCAPABILITIES_AUDIOHANDLING))
         {
            rfbAudioSupport = true;
         }

      }
   }
   return enErrorCode;
}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::enConnectiAP2(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::enConnectiAP2(t_U32 u32DeviceHandle, const trMySPINDevInfo& corfrDevInfo,
         const trMySPINAccInfo& corfrAccInfo, t_Bool isHostMode)
{
   ETG_TRACE_USR1(("enConnectiAP2 - START"));

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;

   if (NULL != poMySPINInstance)
   {
      t_U32 u32VendorID = 0x05ac;
      t_U32 u32ProductID = 0x128a;

      MSPIN_ERROR connectStatus = MSPIN_ConnectiAP2(poMySPINInstance,
               u32VendorID,
               u32ProductID,
               corfrDevInfo.szSerial.c_str(),
               corfrAccInfo.szWriteDevice.c_str(),
               corfrAccInfo.szReadDevice.c_str(),
               isHostMode);

      ETG_TRACE_USR4(("[PARAM]::enConnectiAP2-ConnectiAP2 Status %d", ETG_ENUM(MYSPIN_ERROR, connectStatus)));

      // Save connected devices.
      if (MSPIN_SUCCESS == connectStatus)
      {
         m_mapConnectedDevices[u32DeviceHandle] = corfrDevInfo;
         //get the capability of the phone
         t_U32 u32ServerCaps = MSPIN_GetServerCapabilities(poMySPINInstance);
         ETG_TRACE_USR4(("######### CAPABILITIES ##########[%d]",u32ServerCaps));
         if (MSPIN_SERVERCAPABILITIES_AUDIOHANDLING == (u32ServerCaps & MSPIN_SERVERCAPABILITIES_AUDIOHANDLING))
         {
            ETG_TRACE_USR4(("######### Advacned Audio Capable ##########[%d]",u32ServerCaps));
         }
      }

      enErrorCode = vSetMySPINError(connectStatus);
   }
   return enErrorCode;
}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::enEAPSessionStart(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::enEAPSessionStart(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("Start EAP Native Transport Session"));

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;

   t_Bool bIsConnectedDevice = bIsDeviceConnected(u32DeviceHandle);

   if ((NULL != poMySPINInstance) && (true == bIsConnectedDevice))
   {
      MSPIN_ERROR eapSessionStatus = MSPIN_EAPSessionStart(poMySPINInstance);

      ETG_TRACE_USR2(("EAP Session Start Status %d", ETG_ENUM(MYSPIN_ERROR, eapSessionStatus)));
      enErrorCode = vSetMySPINError(eapSessionStatus);
   }
   return enErrorCode;
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vEAPSessionStop(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vEAPSessionStop(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("Stop EAP Native Transport Session"));

   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

      if ((NULL != poMySPINInstance))
      {
         MSPIN_EAPSessionStop(poMySPINInstance);
         ETG_TRACE_USR2(("EAP Native Transport Session Stopped"));
      }
   }
}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::enStartVideoPlayback(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::enStartVideoPlayback(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("Start Video Play-back from Phone"));

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;
   t_Bool bIsConnectedDevice = bIsDeviceConnected(u32DeviceHandle);

   if (!bIsConnectedDevice)
   	   return enErrorCode;

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if ((NULL != poMySPINInstance) && (true == bIsConnectedDevice))
   {
      MSPIN_ERROR connectStatus = MSPIN_ResumeConnection(poMySPINInstance); //Todo - remove this later.
      //TODO Add a ETG_ENUM here.
      ETG_TRACE_USR2(("Start Video Play-back , Status %d",connectStatus));

      //enErrorCode = vSetMySPINError(connectStatus);
      enErrorCode = e8NO_ERRORS; //Todo - change this later. Hardcoded for now , since TA is returning ErrorCode 4 on this.
   }

   return enErrorCode;
}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::enStopVideoPlayback(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::enStopVideoPlayback(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("Stop Video Play-back from Phone"));

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;
   t_Bool bIsConnectedDevice = bIsDeviceConnected(u32DeviceHandle);

   if (!bIsConnectedDevice)
   	   return enErrorCode;

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if ((NULL != poMySPINInstance) && (true == bIsConnectedDevice))
   {
      // Suspend connection, this would suspend frame buffer update
      MSPIN_ERROR connectStatus = MSPIN_SuspendConnection(poMySPINInstance); //Todo - remove this later.
      //TODO Add a ETG_ENUM here.l
      ETG_TRACE_USR2(("Stop Video Play-back , Status %d",connectStatus));

      //enErrorCode = vSetMySPINError(connectStatus);
      enErrorCode = e8NO_ERRORS;//Todo - change this later. Hardcoded for now , since TA is returning ErrorCode 4 on this.
   }

   return enErrorCode;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vPumpEvents(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vPumpEvents(t_U32 u32DevID, t_S32 s32TimeOut)
{
   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DevID);

   if (NULL != poMySPINInstance)
   {
      MSPIN_PumpEvents(poMySPINInstance, s32TimeOut);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSetWaylandTouch(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetWaylandTouch(t_U32 u32DeviceHandle, t_Bool bIsWayland)
{
   ETG_TRACE_USR2(("[DESC]:Touch handling set to wayland-%d", ETG_ENUM(BOOL, bIsWayland)));

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      //Set touch handling to wayland.
      MSPIN_SetWaylandTouch(poMySPINInstance, bIsWayland);
   }
}
/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::enDisconnectAOAP(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::enDisconnectAOAP(t_U32 u32DeviceHandle)
{
   t_Bool bIsConnectedDevice = bIsDeviceConnected(u32DeviceHandle);
   ETG_TRACE_USR1(("AOAP Device %s connected", bIsConnectedDevice ? "is" : "not"));

   if(bIsConnectedDevice)
   {
	 ETG_TRACE_USR1(("Disconnecting AOAP Device"));
	 MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
	 if (NULL != poMySPINInstance)
	 {
	    m_mapConnectedDevices.erase(u32DeviceHandle);
	    m_MySPINCapabilities = 0;
	    // Delete mySPIN Instance
	    MSPIN_DeleteInstance(&poMySPINInstance);
		m_poMySPINInstance = NULL;
		delete m_rMySPINContext;
		m_rMySPINContext = NULL;
		ETG_TRACE_USR2(("MYSPIN Instance deleted."));

	 }
   }
   return e8NO_ERRORS;
}
/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::enDisconnectiAP2(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::enDisconnectiAP2(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("Disconnecting IAP2 Device"));
   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

   if (NULL != poMySPINInstance)
   {
      MSPIN_DisconnectiAP2(poMySPINInstance);
      ETG_TRACE_USR2(("Device disconnected"));

      t_Bool bIsConnectedDevice = bIsDeviceConnected(u32DeviceHandle);

      if (true == bIsConnectedDevice)
      {
         m_mapConnectedDevices.erase(u32DeviceHandle);
      }
      m_MySPINCapabilities = 0;
      // Delete mySPIN Instance
      MSPIN_DeleteInstance(&poMySPINInstance);
      // Delete the current context
      delete m_rMySPINContext;
      m_rMySPINContext = NULL;
      m_poMySPINInstance = NULL;
      ETG_TRACE_USR2(("MYSPIN Instance deleted."));
   }
   return e8NO_ERRORS;
}
/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINTAWrapper::vSetMySPINError(...)
 ***************************************************************************/
tenErrorCode spi_tclMySPINTAWrapper::vSetMySPINError(MSPIN_ERROR mySPINError)
{
   ETG_TRACE_USR2(("[DESC]:Set mySPIN Error-%d", ETG_ENUM(MYSPIN_ERROR, mySPINError)));
   tenErrorCode enErrorCode;

   // Map mySPIN errors to SPI error types
   switch (mySPINError)
   {
      case MSPIN_SUCCESS:
      {
         enErrorCode = e8NO_ERRORS;
      }
         break;

      case MSPIN_ERROR_RECEIVER_START_FAILED:
      {
         // Case when the remote launch of the Launcher app failed.
         enErrorCode = e8REMOTE_APP_LAUNCH_ERROR;
      }
         break;

      case MSPIN_ERROR_CONNECTION_TIMEOUT:
      case MSPIN_ERROR_PROTOCOL_SETUP:
      {
         // Case when there is timeout in connection.
         enErrorCode = e8CONNECT_TIME_OUT;
      }
         break;

      default:
      {
         //Consider other error codes as Internal error.
         //Since SPI need not expose internal errors to HMI etc.
         enErrorCode = e8INTERNAL_ERROR;
      }
         break;
   }

   return enErrorCode;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINTAWrapper::bIsDeviceConnected(...)
 ***************************************************************************/
t_Bool spi_tclMySPINTAWrapper::bIsDeviceConnected(t_U32 u32DeviceHandle)
{

   std::map<t_U32, trMySPINDevInfo>::iterator itDeviceMap = m_mapConnectedDevices.find(u32DeviceHandle);
   return ((itDeviceMap != m_mapConnectedDevices.end()) ? true : false);

}

/***************************************************************************
 ** FUNCTION: t_U32 spi_tclMySPINTAWrapper::u32GetSupportedKeys(...)
 ***************************************************************************/
t_U32 spi_tclMySPINTAWrapper::u32GetSupportedKeys(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("Fetching supported keys"));

   t_U32 u32SupportedKeys = 0;

   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      if ((NULL != poMySPINInstance))
      {
         u32SupportedKeys = MSPIN_GetAvailableKeys(poMySPINInstance);
         ETG_TRACE_USR4(("[PARAM]::u32GetSupportedKeys-Supported keys are [%d]", u32SupportedKeys));
      }
   }
   return u32SupportedKeys;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSendKeyEvent(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSendKeyEvent(t_U32 u32DeviceHandle, tenKeyMode enKeyMode,
         tenMySPINKeyCode enMySPINKeyCode)
{
   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);

      if (NULL != poMySPINInstance)
      {
         ETG_TRACE_USR2(("[DESC]::Send to TA Key Event [%d]", ETG_ENUM(MYSPIN_SOFTKEY_CODE, enMySPINKeyCode)));

         MSPIN_KeyEvent(poMySPINInstance, static_cast<MSPIN_KEY> (enMySPINKeyCode), enKeyMode);
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSetMovingStatus(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetMovingStatus(t_U32 u32DeviceHandle, t_Bool bIsMoving)
{
   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      if (NULL != poMySPINInstance)
      {
         ETG_TRACE_USR2(("[DESC]::Send to TA Moving Status [%d]", ETG_ENUM(BOOL, bIsMoving)));
         t_String szJsonString;
         szJsonString = bIsMoving ? "true" : "false";
         szJsonString = "{\"value\":" + szJsonString + "}";
         vSendVehicleData(u32DeviceHandle,scou32VEHICLE_DATA_KEY_IS_MOVING,szJsonString);
      }
   }
}

/***************************************************************************
 ** FUNCTION: spi_tclMySPINTAWrapper::vSetNightMode(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetNightMode(t_U32 u32DeviceHandle, t_Bool bIsNight)
{
   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      if (NULL != poMySPINInstance)
      {
         ETG_TRACE_USR2(("[DESC]::Send to TA Night Mode [%d]", ETG_ENUM(BOOL, bIsNight)));
         t_String szJsonString;
         szJsonString = bIsNight ? "true" : "false";
         szJsonString = "{\"value\":" + szJsonString + "}";
         vSendVehicleData(u32DeviceHandle,scou32VEHICLE_DATA_KEY_IS_NIGHT,szJsonString);
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSetVehicleBTMACAddress(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetVehicleBTMACAddress(t_U32 u32DeviceHandle, t_String szVehicleBTAddress)
{
   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      if (NULL != poMySPINInstance)
      {
         ETG_TRACE_USR2(("[DESC]::Send to TA Vehicle BT MAC Address for [%d]", u32DeviceHandle));
         MSPIN_SendCustomString(poMySPINInstance, MSPIN_HOSTMSGSTRTYPE_BTMACADDRESS, szVehicleBTAddress.c_str());
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSetVehicleManufactureInfo(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetVehicleManufactureInfo(const t_U32 u32DeviceHandle, t_U16 u16VehicleType, t_String szVehicleManufacturerName, t_String szVehicleManufacturerScope)
{
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      t_Char allianceVehicleType[BUFFER_MAX_LENGTH];
      memset(allianceVehicleType, 0, BUFFER_MAX_LENGTH);
      snprintf(allianceVehicleType, BUFFER_MAX_LENGTH, "%0x", u16VehicleType);
      if (NULL != poMySPINInstance)
      {
         ETG_TRACE_USR2(("[DESC]::Send to TA alliance vehicle type [%s]",allianceVehicleType));
         ETG_TRACE_USR2(("[DESC]::Send to TA Vehicle manufacturer name [%s]", szVehicleManufacturerName.c_str()));
         MSPIN_SendCustomString(poMySPINInstance, MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_A, allianceVehicleType);
         MSPIN_SendCustomString(poMySPINInstance, MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_D, szVehicleManufacturerName.c_str());
         MSPIN_SendCustomString(poMySPINInstance, MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_E, szVehicleManufacturerScope.c_str());
      }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINTAWrapper::vUpdateVoiceSessionStatus(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vUpdateVoiceSessionStatus(t_U32 u32DeviceHandle,
         tenMySPINVoiceSessionStatus enMySPINVoiceSessionStatus,
         tenMySPINVoiceSessionConstraint enMySPINVoiceSessionConstraint)
{
   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      if (NULL != poMySPINInstance)
      {
         ETG_TRACE_USR2(("Send Voice Status as [%d]", ETG_ENUM(MYSPIN_VOICESESSION_STATUS, enMySPINVoiceSessionStatus)));
         MSPIN_UpdateVoiceSessionStatus(poMySPINInstance,
                  static_cast<MSPIN_VOICESESSION_STATUS> (enMySPINVoiceSessionStatus),
                  static_cast<MSPIN_VOICESESSION_STATUS_CONSTRAINT> (enMySPINVoiceSessionConstraint));
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSendPositionData(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSendPositionData(t_U32 u32DeviceHandle, t_String szNmeaSentence)
{
   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      if (NULL != poMySPINInstance)
      {
         ETG_TRACE_USR2(("Send Position Data for Device [%d]", u32DeviceHandle));
         MSPIN_SendPosition(poMySPINInstance, const_cast<char*> (szNmeaSentence.c_str()));
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSendVehicleData(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSendVehicleData(t_U32 u32DeviceHandle, t_U32 u32Type, t_String szJsonString)
{
   if (true == bIsDeviceConnected(u32DeviceHandle))
   {
      MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
      if (NULL != poMySPINInstance)
      {
         MSPIN_STRINGENCODING enStringEncoding = MSPIN_STRINGENCODING_UTF8;

         ETG_TRACE_USR2(("Send Vehicle Data for Device [%d]", u32DeviceHandle));

         MSPIN_ERROR enError = MSPIN_SetVehicleData(poMySPINInstance,
                  u32Type,
                  enStringEncoding,
                  const_cast<char*> (szJsonString.c_str()));
         ETG_TRACE_ERR(("Error sending vehicle data [%d]", enError));
      }
   }
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINTAWrapper::vSetAudioStatus(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSetAudioStatus(t_U32 u32DeviceHandle, t_U32 u32requestID, tenAudioResponse enAudioResponse)
{
   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
   ETG_TRACE_USR2(("Set Audio Status for device [%d] u32requestID [%d] enAudioResponse[%d]", u32DeviceHandle, u32requestID,
                    enAudioResponse));
   //TODO Add check
   if (NULL != poMySPINInstance)
   {
      MSPIN_SetAudioStatus(poMySPINInstance, u32requestID, static_cast<MSPIN_AUDIOSTATUS>(enAudioResponse), MSPIN_AUDIORESULTCODE_NOERROR );
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINTAWrapper::vSendCustomKeyEvent(...)
 ***************************************************************************/
t_Void spi_tclMySPINTAWrapper::vSendCustomKeyEvent(t_U32 u32DeviceHandle, 
                                                   tenMySPINCustomKeyMode enCustomKeyMode, 
                                                   tenMySPINCustomKeyCode enMySPINCustomKeyCode)
{

   MSPIN_Instance_t* poMySPINInstance = getMySPINInstance(u32DeviceHandle);
   if (NULL != poMySPINInstance)
   {
       ETG_TRACE_USR4(("[PARAM]:spi_tclMySPINTAWrapper::vSendCustomKeyEvent() found key code %d",enMySPINCustomKeyCode));
       ETG_TRACE_USR4(("[PARAM]:spi_tclMySPINTAWrapper::vSendCustomKeyEvent() found key mode %d",enCustomKeyMode));
       MSPIN_SendCustomKey(poMySPINInstance,static_cast<MSPIN_CUSTOMDATA_KEYTYPE>(enCustomKeyMode),
                                static_cast<MSPIN_CUSTOMDATA_KEYCODE>(enMySPINCustomKeyCode));
   }
}
