/***********************************************************************/
/*!
 * \file  spi_tclMySPINCmdSession.h
 * \brief  Implementation of the Class spi_tclMySPINCmdSession
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         gri2kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 12.04.2016  | Chaitra Srinivasa     | Trace message cleanup
 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "mySPINTypes.h"
#include "spi_tclMySPINTAWrapper.h"
#include "spi_tclMySPINCmdSession.h"
#include "spi_tclMySPINMsgQInterface.h"
#include "spi_tclMySPINSessionDispatcher.h"
#include "spi_tclMySPINTADataIntf.h"
#include "spi_tclMySPINTAConnIntf.h"
#include "spi_tclMySPINManager.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_MYSPINWRAPPER
#include "trcGenProj/Header/spi_tclMySPINCmdSession.cpp.trc.h"
#endif
#endif

using namespace std;
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 ** FUNCTION: spi_tclMySPINCmdSession::spi_tclMySPINCmdSession()
 ***************************************************************************/
spi_tclMySPINCmdSession::spi_tclMySPINCmdSession() :
   m_enBTVoiceSessionStatus(e8BTVOICESESSION_STATUS_NONE),m_bAdvAudioSupported(false),
   m_enMySPINError(e8MSPIN_ERROR_UNKNOWN)
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdSession::spi_tclMySPINCmdSession()"));

   spi_tclMySPINTADataIntf oTADataIntf;
   spi_tclMySPINTAConnIntf* poMySPINTAConnIntf = oTADataIntf.poGetTAConnIntfInst();

   if (NULL != poMySPINTAConnIntf)
   {
      poMySPINTAConnIntf->vInitMySPIN();
   }
}

/***************************************************************************
 ** FUNCTION: spi_tclMySPINCmdSession::~spi_tclMySPINCmdSession()
 ***************************************************************************/
spi_tclMySPINCmdSession::~spi_tclMySPINCmdSession()
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdSession::~spi_tclMySPINCmdSession()"));
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINCmdSession::bInitialize()
 ***************************************************************************/
t_Bool spi_tclMySPINCmdSession::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdSession::bInitialize()"));
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINCmdSession::vUnInitialize()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdSession::vUnInitialize()"));
}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINCmdSession::enInitMySPINSession()
 ***************************************************************************/
tenErrorCode spi_tclMySPINCmdSession::enInitMySPINSession(t_U32 u32DevID, tenDeviceSubCategory enDevSubCat,
         const trMySPINDevInfo& corfrDevInfo, const trMySPINAccInfo& corfrAccInfo, t_Bool& rfbDevAudioSupport)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::enInitMySPINSession: DevID-0x%x Device Type-%d", u32DevID, ETG_ENUM(MSPIN_DEV_TYPE,
            enDevSubCat)));

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;

   spi_tclMySPINTADataIntf oTADataIntf;
   spi_tclMySPINTAConnIntf* poMySPINTAConnIntf = oTADataIntf.poGetTAConnIntfInst();
   
   if (NULL != poMySPINTAConnIntf)
   {
      trMySPINSessionCbs rSessionCbs;
      memset(&rSessionCbs, 0, sizeof(rSessionCbs));
      rSessionCbs.vAppInactiveCb = &spi_tclMySPINCmdSession::vAppInactiveCallback;
      rSessionCbs.vAppTransitionCb = &spi_tclMySPINCmdSession::vAppTransitionStatusCallback;
      rSessionCbs.vErrorCb = &spi_tclMySPINCmdSession::vMyspinErrorCallback;
      rSessionCbs.vPhoneCallStatusCallback = &spi_tclMySPINCmdSession::PhoneCallStatusCallback;
      rSessionCbs.vInitiatePhoneCallCb = &spi_tclMySPINCmdSession::vInitiatePhoneCallCallback;
      rSessionCbs.vCustomDataIntCallback = &spi_tclMySPINCmdSession::vCustomDataIntCallback;
      rSessionCbs.vCustomDataStringCallback = &spi_tclMySPINCmdSession::vCustomDataStringCallback;
      rSessionCbs.vPTTAvailableCallback = &spi_tclMySPINCmdSession::vPTTAvailableCallback;
      rSessionCbs.vVoiceSessionCallback = &spi_tclMySPINCmdSession::vVoiceSessionCallback;

      poMySPINTAConnIntf->vRegisterSessionCbs(u32DevID, rSessionCbs);

      switch (enDevSubCat)
      {
         case e8DEVTYPE_AOAP:
         {
            poMySPINTAConnIntf->vSetAccessoryPropertiesAoap(u32DevID, corfrAccInfo);
            enErrorCode = poMySPINTAConnIntf->enConnectAoap(u32DevID, corfrDevInfo, rfbDevAudioSupport);
            m_bAdvAudioSupported = rfbDevAudioSupport;

         }
            break;
         case e8DEVTYPE_IAP:
         {
            enErrorCode = poMySPINTAConnIntf->enConnectiAP2(u32DevID, corfrDevInfo, corfrAccInfo, true);
            m_bAdvAudioSupported = rfbDevAudioSupport;
         }
            break;
         case e8DEVTYPE_UNKNWON:
         default:
         {
            ETG_TRACE_ERR(("[ERR]::Unknown Device Type"));
         }
            break;
      }
   }

   return enErrorCode;

}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINCmdSession::enUninitMySPINSession()
 ***************************************************************************/
tenErrorCode spi_tclMySPINCmdSession::enUninitMySPINSession(t_U32 u32DevID, tenDeviceSubCategory enDevSubCat)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::enUninitMySPINSession:DevID-0x%x Device Type-%d", u32DevID, ETG_ENUM(MSPIN_DEV_SUBCAT,
            enDevSubCat)));

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;

   spi_tclMySPINTADataIntf oTADataIntf;
   spi_tclMySPINTAConnIntf* poMySPINTAConnIntf = oTADataIntf.poGetTAConnIntfInst();
   m_enBTVoiceSessionStatus = e8BTVOICESESSION_STATUS_NONE;
   m_enMySPINError = e8MSPIN_ERROR_UNKNOWN;
   
   if (NULL != poMySPINTAConnIntf)
   {
      switch (enDevSubCat)
      {
         case e8DEVTYPE_AOAP:
         {
            enErrorCode = poMySPINTAConnIntf->enDisconnectAOAP(u32DevID);
         }
            break;
         case e8DEVTYPE_IAP:
         {
            enErrorCode = poMySPINTAConnIntf->enDisconnectiAP2(u32DevID);
         }
            break;
         case e8DEVTYPE_UNKNWON:
         default:
         {
            ETG_TRACE_ERR(("[ERR]:Unknown Device Type"));
         }
            break;
      }
   }

   return enErrorCode;

}

/***************************************************************************
 ** FUNCTION: tenErrorCode spi_tclMySPINCmdSession::enSessionStart()
 ***************************************************************************/
tenErrorCode spi_tclMySPINCmdSession::enSessionStart(t_U32 u32DevID, tenDeviceSubCategory enDevSubCat)
{
   ETG_TRACE_USR2(("[DESC]::spi_tclMySPINCmdSession::enSessionStart:DevID-0x%x Device Type-%d", u32DevID, ETG_ENUM(MSPIN_DEV_SUBCAT,
            enDevSubCat)));

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;

   spi_tclMySPINTADataIntf oTADataIntf;
   spi_tclMySPINTAConnIntf* poMySPINTAConnIntf = oTADataIntf.poGetTAConnIntfInst();
   
   if (NULL != poMySPINTAConnIntf)
   {
      switch (enDevSubCat)
      {
         case e8DEVTYPE_IAP:
         {
            enErrorCode = poMySPINTAConnIntf->enEAPSessionStart(u32DevID);
            // On proper session start check for servver capabilities
         }
            break;
         case e8DEVTYPE_AOAP:
         case e8DEVTYPE_UNKNWON:
         default:
         {
            ETG_TRACE_ERR(("[ERR]:Unknown Device Type"));
         }
            break;
      }
   }

   return enErrorCode;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vSessionStop()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vSessionStop(t_U32 u32DevID, tenDeviceSubCategory enDevSubCat)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::vSessionStop:DevID-0x%x Device Type-%d", u32DevID, ETG_ENUM(MSPIN_DEV_SUBCAT,
            enDevSubCat)));

   spi_tclMySPINTADataIntf oTADataIntf;
   spi_tclMySPINTAConnIntf* poMySPINTAConnIntf = oTADataIntf.poGetTAConnIntfInst();
   
   if (NULL != poMySPINTAConnIntf)
   {
      switch (enDevSubCat)
      {
         case e8DEVTYPE_IAP:
         {
            poMySPINTAConnIntf->vEAPSessionStop(u32DevID);
         }
            break;
         case e8DEVTYPE_AOAP:
         case e8DEVTYPE_UNKNWON:
         default:
         {
            ETG_TRACE_ERR(("[ERR]:Unknown Device Type"));
         }
            break;
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vAppInactiveCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vAppInactiveCallback(t_Void *poContext, t_Bool bIsInActive)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::vAppInactiveCallback: Is App Inactive-%d", ETG_ENUM(BOOL,
            bIsInActive)));
   // Post a message , If the App Is inactive Session should be suspended.

   if (NULL != poContext)
   {
      MySPINAppInactiveMsg oMySPINAppInactiveMsg;
      trMySPINContext* rMySPINContext = (trMySPINContext*) poContext;
      oMySPINAppInactiveMsg.vSetDeviceHandle(rMySPINContext->u32DeviceHandle);
      oMySPINAppInactiveMsg.m_bIsInActive = bIsInActive;
      spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();
      ETG_TRACE_USR4(("[PARAM]::vAppInactiveCallback-For mySPIN Device with Handle :[0x%x]", oMySPINAppInactiveMsg.u32GetDeviceHandle()));

      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oMySPINAppInactiveMsg, sizeof(oMySPINAppInactiveMsg));
      }
   }

}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vAppTransitionStatusCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vAppTransitionStatusCallback(t_Void *poContext, t_Bool bIsActive)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::vAppTransitionStatusCallback: Is Active-%d", ETG_ENUM(BOOL,
            bIsActive)));

   // Post a message , If the 3rd Party App is in transition Session should be suspended.
   if (NULL != poContext)
   {
      MySPINAppTransitionMsg oMySPINAppTransitionMsg;
      trMySPINContext* rMySPINContext = static_cast<trMySPINContext*> (poContext);
      oMySPINAppTransitionMsg.vSetDeviceHandle(rMySPINContext->u32DeviceHandle);
      oMySPINAppTransitionMsg.m_bIsActive = bIsActive;
      spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();
      ETG_TRACE_USR4(("[PARAM]::vAppTransitionStatusCallback-For mySPIN Device with Handle :[0x%x]", oMySPINAppTransitionMsg.u32GetDeviceHandle()));

      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oMySPINAppTransitionMsg, sizeof(oMySPINAppTransitionMsg));
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vCustomDataIntCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vCustomDataIntCallback(t_Void *poContext, MSPIN_PHONEMSGINTTYPE enPhoneStringType,
         t_S64 s64Data)
{
   SPI_INTENTIONALLY_UNUSED(poContext)
   SPI_INTENTIONALLY_UNUSED(enPhoneStringType)
   SPI_INTENTIONALLY_UNUSED(s64Data)
   ETG_TRACE_USR1(("spi_tclMySPINCmdSession::vCustomDataIntCallback()"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINCmdSession::vCustomDataStringCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vCustomDataStringCallback(t_Void *poContext, MSPIN_PHONEMSGSTRTYPE enPhoneStringType,
         MSPIN_STRINGENCODING enEncoding, t_Char *pczData)
{
   SPI_INTENTIONALLY_UNUSED(enEncoding)
   ETG_TRACE_USR2(("[DESC]:Received custom data string call-back for [%d]", enPhoneStringType));
   ETG_TRACE_USR2(("[DESC]:Received custom data string with data [%s]", pczData));
   switch (enPhoneStringType)
   {
      case MSPIN_PHONEMSGSTRTYPE_BTNAME:
      {
         //Send a message to mySPIN Vehicle Data
         MySPINBTProfileMsg oMySPINBTProfileMsg;

         //The expected string would be of the format [HFP,A2DP]BTFriendlyName
         t_String szData(pczData);
         //Step1 : Find the first "[" and "]"

         size_t tStartBracePos = szData.find_first_of("[");
         size_t tCloseBracePos = szData.find_first_of("]");

         //Step2 : Extract the Profiles supported which are comma seperated
         t_String szProfileStr = szData.substr(tStartBracePos + 1, tCloseBracePos - 1);

         //Step3 : Parse the string to get the individual profiles
         vector<t_String> vecTokens;
         size_t tStart = 0, tEnd = 0;
         while ((tEnd = szProfileStr.find(',', tStart)) != std::string::npos)
         {
            vecTokens.push_back(szProfileStr.substr(tStart, tEnd - tStart));
            tStart = tEnd + 1;
         }
         vecTokens.push_back(szProfileStr.substr(tStart));

         //Step 4 : Search for the Profiles
         std::vector<t_String>::iterator itrHFP = find(vecTokens.begin(), vecTokens.end(), "HFP");
         oMySPINBTProfileMsg.m_bIsHFPConnected = (itrHFP != vecTokens.end());
         ETG_TRACE_USR2(("HFP enabled [%d]", ETG_ENUM(BOOL, oMySPINBTProfileMsg.m_bIsHFPConnected)));

         std::vector<t_String>::iterator itrA2DP = find(vecTokens.begin(), vecTokens.end(), "A2DP");
         oMySPINBTProfileMsg.m_bIsA2DPConnected = (itrA2DP != vecTokens.end());
         ETG_TRACE_USR2(("A2DP enabled [%d]", ETG_ENUM(BOOL, oMySPINBTProfileMsg.m_bIsA2DPConnected)));

         //Step 5 : Extract the BT Friendly Name
         if((szData.substr(tCloseBracePos + 1).length()) < MAX_LENGTH)
         {
             strncpy(oMySPINBTProfileMsg.m_szFriendlyName, szData.substr(tCloseBracePos + 1).c_str(),MAX_LENGTH-1);
             ETG_TRACE_USR2(("Friendly name [%s]", oMySPINBTProfileMsg.m_szFriendlyName));
         }
         
         //Set the state of mySPIN Session
         spi_tclMySPINTADataIntf oTADataIntf;
         spi_tclMySPINTAConnIntf* poMySPINTAConnIntf = oTADataIntf.poGetTAConnIntfInst();
   
         if (poMySPINTAConnIntf)
         {
            trMySPINContext* rMySPINContext = (trMySPINContext*) poContext;

            //Get the device handle
            oMySPINBTProfileMsg.vSetDeviceHandle(rMySPINContext->u32DeviceHandle);

            rMySPINContext->bIsBTConnected = (oMySPINBTProfileMsg.m_bIsA2DPConnected
                     && oMySPINBTProfileMsg.m_bIsHFPConnected);
         }

         spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();

         if (NULL != poMsgQinterface)
         {
            poMsgQinterface->bWriteMsgToQ(&oMySPINBTProfileMsg, sizeof(oMySPINBTProfileMsg));
         }
      }
         break;

      default:
      {
         //not a valid string type.
         ETG_TRACE_ERR(("[ERROR] invalid phone message"));
      }
         break;

   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINCmdSession::vInitiatePhoneCallCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vInitiatePhoneCallCallback(t_Void *poContext, t_Char *pczNumberString,
         t_Char *pczDisplayString)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::vInitiatePhoneCallCallback() [%s]", pczNumberString));
   ETG_TRACE_USR2(("[DESC]:Display String [%s]", pczDisplayString));
   // Post a message when the a phone call needs to be initiated
   if (NULL != poContext)
   {
      MySPINInitiateCallMsg oMySPINInitiateCallMsg;
      if(strlen(pczNumberString) < MAX_LENGTH)
      {
          strncpy(oMySPINInitiateCallMsg.m_szTelephoneNumber, pczNumberString, MAX_LENGTH-1);
      }
      if(strlen(pczDisplayString) < MAX_LENGTH)
      {  
          strncpy(oMySPINInitiateCallMsg.m_szDisplayContent, pczDisplayString, MAX_LENGTH-1);
      }

      spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();
      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oMySPINInitiateCallMsg, sizeof(oMySPINInitiateCallMsg));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINCmdSession::vMyspinErrorCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vMyspinErrorCallback(t_Void *poContext, MSPIN_ERROR enErrorCode)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::vMyspinErrorCallback: Error - %d", ETG_ENUM(MYSPIN_ERROR,
            enErrorCode)));

   if (NULL != poContext)
   {
      //Process the request only
      MySPINSessionErrorMsg oMySPINSessionErrorMsg;
      oMySPINSessionErrorMsg.enErrorCode = static_cast<tenMySPINError>(enErrorCode);
      
      spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
      if(NULL != poMySPINMngr)
      {
          spi_tclMySPINCmdSession *poMyspinCmdSession = poMySPINMngr->poGetSessionInstance();
          if(NULL != poMyspinCmdSession)
          {
              poMyspinCmdSession->m_enMySPINError = static_cast<tenMySPINError>(enErrorCode);
              ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::vMyspinErrorCallback: Error - %d", poMyspinCmdSession->m_enMySPINError))
          }
      }

      trMySPINContext* rMySPINContext = static_cast<trMySPINContext*> (poContext);
      oMySPINSessionErrorMsg.vSetDeviceHandle(rMySPINContext->u32DeviceHandle);

      spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();

      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oMySPINSessionErrorMsg, sizeof(oMySPINSessionErrorMsg));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINCmdSession::PhoneCallStatusCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::PhoneCallStatusCallback(t_Void *poContext, t_Bool bIsActive)
{
   SPI_INTENTIONALLY_UNUSED(poContext)
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::PhoneCallStatusCallback: Is call Active-%d", ETG_ENUM(BOOL,
            bIsActive)));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINCmdSession::vPTTAvailableCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vPTTAvailableCallback(t_Void *poContext, t_Bool bIsPttAvail)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclMySPINCmdSession::vPTTAvailableCallback: Is Ptt available-%d", 
               ETG_ENUM(BOOL, bIsPttAvail)));
               
   if (NULL != poContext)
   {
      //Process the request only
      MySPINPTTAvailable oMySPINPTTAvailable;

      trMySPINContext* rMySPINContext = (trMySPINContext*) poContext;
      oMySPINPTTAvailable.m_bIsPTTAvail = bIsPttAvail;

      spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();

      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oMySPINPTTAvailable, sizeof(oMySPINPTTAvailable));
      }
   }               
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vVoiceSessionCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vVoiceSessionCallback(t_Void *poContext, MSPIN_VOICESESSION_REQUESTTYPE requestType)
{
   ETG_TRACE_USR2(("[DESC]:Received Voice Session Request [%d]", ETG_ENUM(MYSPIN_VOICESESSION_REQUESTTYPE, requestType)));

   if (NULL != poContext)
   {
      //Process the request only
      MySPINVoiceSessionMsg oMySPINVoiceSessionMsg;
      oMySPINVoiceSessionMsg.m_enMySPINVoiceSessionReq = static_cast<tenMySPINVoiceSessionReq> (requestType);

      spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();

      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oMySPINVoiceSessionMsg, sizeof(oMySPINVoiceSessionMsg));
      }
   }

}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vUpdateVoiceSessionStatus()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vUpdateVoiceSessionStatus(t_U32 u32DevID,
         tenBTVoiceSessionStatus enBTVoiceSessionStatus)
{
   //Get the response from HMI or internal updates and do basic validations required.
   ETG_TRACE_USR2(("[DESC]:Update Voice Session status as [%d]", ETG_ENUM(MYSPIN_BT_VOICESESSION_STATUS,
            enBTVoiceSessionStatus)));
   spi_tclMySPINTADataIntf oTADataIntf;
   spi_tclMySPINTAConnIntf* poMySPINTAConnIntf = oTADataIntf.poGetTAConnIntfInst();
   
   if((NULL != poMySPINTAConnIntf) && ((e8MSPIN_ERROR_PROTOCOL_STOP == m_enMySPINError) 
       || (e8MSPIN_ERROR_INITIALIZATION_ABORT == m_enMySPINError)))
   {
      m_enBTVoiceSessionStatus = enBTVoiceSessionStatus;

      // Only update the TA when there is a change in the state compared to previous state.
      switch (enBTVoiceSessionStatus)
      {
         case e8BTVOICESESSION_STATUS_UNAVAILABLE:
         case e8BTVOICESESSION_STATUS_REQUEST_DENIED_NOHFP:
         case e8BTVOICESESSION_STATUS_REQUEST_DENIED_VOICEALREADYACTIVE:
         {
            poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                     e8VOICESESSION_STATUS_UNAVAILABLE,
                     e8VOICESESSION_STATUS_CONSTRAINT_NONE);
            MySPINBTStateUpdateMsg oMySPINBTStateUpdateMsg;
            spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();
            if (NULL != poMsgQinterface)
            {
               poMsgQinterface->bWriteMsgToQ(&oMySPINBTStateUpdateMsg, sizeof(oMySPINBTStateUpdateMsg));
            }

         }
            break;
         case e8BTVOICESESSION_STATUS_REQUEST_ACCEPTED:
         case e8BTVOICESESSION_STATUS_ACTIVE:
         {
            poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                     static_cast<tenMySPINVoiceSessionStatus> (enBTVoiceSessionStatus),
                     e8VOICESESSION_STATUS_CONSTRAINT_NONE);
         }
            break;
         case e8BTVOICESESSION_STATUS_IDLE: 
         {
                poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                    e8VOICESESSION_STATUS_IDLE,
                    e8VOICESESSION_STATUS_CONSTRAINT_NONE);
         }
            break;
         case e8BTVOICESESSION_STATUS_REQUEST_DENIED:
         {
             poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                      tenMySPINVoiceSessionStatus::e8VOICESESSION_STATUS_IDLE,
                      e8VOICESESSION_STATUS_CONSTRAINT_NONE);
         }
            break;
         default:
         {
            ETG_TRACE_ERR(("spi_tclMySPINCmdSession::vUpdateVoiceSessionStatus:Invalid Voice Session Update"));
         }
            break;
      }    
      m_enMySPINError = e8MSPIN_ERROR_UNKNOWN;
   }
   
   else if ((NULL != poMySPINTAConnIntf) && (enBTVoiceSessionStatus != m_enBTVoiceSessionStatus))
   {
      m_enBTVoiceSessionStatus = enBTVoiceSessionStatus;

      // Only update the TA when there is a change in the state compared to previous state.
      switch (enBTVoiceSessionStatus)
      {
         case e8BTVOICESESSION_STATUS_UNAVAILABLE:
         case e8BTVOICESESSION_STATUS_REQUEST_DENIED_NOHFP:
         case e8BTVOICESESSION_STATUS_REQUEST_DENIED_VOICEALREADYACTIVE:
         {
            poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                     e8VOICESESSION_STATUS_UNAVAILABLE,
                     e8VOICESESSION_STATUS_CONSTRAINT_NONE);
            MySPINBTStateUpdateMsg oMySPINBTStateUpdateMsg;
            spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();
            if (NULL != poMsgQinterface)
            {
               poMsgQinterface->bWriteMsgToQ(&oMySPINBTStateUpdateMsg, sizeof(oMySPINBTStateUpdateMsg));
            }

         }
            break;
         case e8BTVOICESESSION_STATUS_REQUEST_ACCEPTED:
         case e8BTVOICESESSION_STATUS_ACTIVE:
         {
            poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                     static_cast<tenMySPINVoiceSessionStatus> (enBTVoiceSessionStatus),
                     e8VOICESESSION_STATUS_CONSTRAINT_NONE);
         }
            break;
         case e8BTVOICESESSION_STATUS_IDLE: 
         {
                poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                    e8VOICESESSION_STATUS_IDLE,
                    e8VOICESESSION_STATUS_CONSTRAINT_NONE);
         }
            break;
         case e8BTVOICESESSION_STATUS_REQUEST_DENIED:
         {
             poMySPINTAConnIntf->vUpdateVoiceSessionStatus(u32DevID,
                      tenMySPINVoiceSessionStatus::e8VOICESESSION_STATUS_IDLE,
                      e8VOICESESSION_STATUS_CONSTRAINT_NONE);
         }
            break;
         default:
         {
            ETG_TRACE_ERR(("spi_tclMySPINCmdSession::vUpdateVoiceSessionStatus:Invalid Voice Session Update"));
         }
            break;
      }

      ETG_TRACE_USR2(("vUpdateVoiceSessionStatus: m_enBTVoiceSessionStatus [%d]", ETG_ENUM(MYSPIN_BT_VOICESESSION_STATUS,
            m_enBTVoiceSessionStatus)));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vVoiceSessionCallback()
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vGetAppCapability(tenAppCapability& enAppCapability)
{
   ETG_TRACE_USR2(("[DESC]:Set app capability for audio as [%d]", ETG_ENUM(BOOL,
            m_bAdvAudioSupported)));
   enAppCapability = (m_bAdvAudioSupported) ? e8_CAPABILITY_AUDIO_HANDLING: e8_CAPABILITY_NONE;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdSession::vSendVREndResponse(...
 ***************************************************************************/
t_Void spi_tclMySPINCmdSession::vSendVREndResponse(t_U32 u32DeviceHandle,
                                                 tenBTVoiceSessionStatus enBTVoiceSessionStatus,
                                                 tenMspinVRStatus enVRState)
{
    ETG_TRACE_USR1(("spi_tclMySPINCmdSession::vSendVREndResponse entered VR state =%d",
                  ETG_ENUM(MYSPIN_VR_STATUS,enVRState)));
    SPI_INTENTIONALLY_UNUSED(enVRState);
    vUpdateVoiceSessionStatus(u32DeviceHandle, enBTVoiceSessionStatus);
}
