/***********************************************************************/
/*!
 * \file  spi_tclMySPINDbusHandler.cpp
 * \brief  Implementation of the Class spi_tclMySPINDbusHandler
 *************************************************************************
 \verbatim


 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 12.04.2016  | Chaitra Srinivasa     | Trace message cleanup
 \endverbatim
 *************************************************************************/
/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "mySPINTypes.h"
#include "mySPIN_Dbus_Introspection.h"
#include "FileHandler.h"
#include <algorithm>
#include <unistd.h>

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_MYSPINWRAPPER
#include "trcGenProj/Header/spi_tclMySPINDbusHandler.cpp.trc.h"
#endif
#endif

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

static t_String szBTALPS[] = {"com.alps.bt","/com/alps/bt","com.alps.bt.Audio",
		                      "com.alps.bt.Audio","com.alps.bt.Audiovisual","/com/alps/bt/dev"};
static t_String szBTEVO[] = {"org.bluez","/org/bluez/hci1","org.bluez.evo.AVP",
		                     "org.freedesktop.DBus.Properties","org.bluez.MediaPlayer","/org/bluez/hci1/dev"};

static const t_U8 scou8KEY_EVENT_RELEASE = 0;
static const t_U8 scou8KEY_EVENT_PRESS = 1;

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
#include "spi_tclMySPINDbusHandler.h"

/***************************************************************************
 ** FUNCTION: spi_tclMySPINDbusHandler::spi_tclMySPINDbusHandler()
 ***************************************************************************/
spi_tclMySPINDbusHandler::spi_tclMySPINDbusHandler() :
   m_poALPSBTDeamonProxy(NULL), m_poAVProxy(NULL), m_poSystemDBusConn(NULL), m_poSessionDBusConn(NULL),m_szPlayerInstance(NULL),
   m_poMPProxy(NULL),m_bBTEvolutionAvailability(false)
{
   ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::spi_tclMySPINDbusHandler():Creating mySPIN DBUS HANDLER"));
   m_rMySpinAudioConfig.u32bitRate = 0;
   m_rMySpinAudioConfig.u32sampleRate = 0;
   m_rMySpinAudioConfig.u8Channels = 2;
   m_rMySpinAudioConfig.u8SampleWidth = 16;
   m_rMySpinAudioConfig.u8SampleDepth = 16;
   m_rMySpinAudioConfig.bIsSigned = true;
   m_rMySpinAudioConfig.bHasIndianNess = true;
   m_szServiceName.clear();
   m_szBaseObjectPath.clear();
   m_szAudioStringInterface.clear();
   m_szAudioInterface.clear();
   m_szAVPInterface.clear();
   m_szDeviceAddr.clear();


}

/***************************************************************************
 ** FUNCTION: spi_tclMySPINDbusHandler::~spi_tclMySPINDbusHandler()
 ***************************************************************************/
spi_tclMySPINDbusHandler::~spi_tclMySPINDbusHandler()
{
   g_object_unref(m_poALPSBTDeamonProxy);
   g_object_unref(m_poAVProxy);
   g_object_unref(m_poMPProxy);
   g_object_unref(m_poSystemDBusConn);
   g_object_unref(m_poSessionDBusConn);
   m_szPlayerInstance = NULL;
   m_poALPSBTDeamonProxy = NULL;
   m_poSystemDBusConn = NULL;
   m_poAVProxy = NULL;
   m_poSessionDBusConn = NULL;

}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINDbusHandler::vInitialize()
 ***************************************************************************/
t_Void spi_tclMySPINDbusHandler::vInitialize()
{
   ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::vInitialize:Initializing DBUS Handler for mySPIN"));
   // Read DBus configuration based on BT Stack
   vLoadDBusConfiguration();

   //Setup connections with the system and session bus.
   GError *poDBusError = NULL;

   //Get the connection for System Bus on DBUS
   m_poSystemDBusConn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &poDBusError);

   if (NULL == poDBusError)
   {
      ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::vInitialize:System Bus connection established"));
   }
   else
   {
      ETG_TRACE_USR4(("[PARAM]:vInitialize-Failed to establish connection with System bus : [%s]", poDBusError->message));
      g_error_free(poDBusError);
   }

}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINDbusHandler::vLoadDBusConfiguration()
 ***************************************************************************/
t_Void spi_tclMySPINDbusHandler::vLoadDBusConfiguration()
{
	//Test for existence for this file.
	spi::io::FileHandler oFile(NULL,spi::io::SPI_EN_RDONLY);
	if(oFile.s32Access("/opt/bosch/connectivity/bluetoothservice/bluetoothservice_out.out") == -1)
	   m_bBTEvolutionAvailability = true;

	ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::vLoadDBusConfiguration- BT %s Stack loaded",m_bBTEvolutionAvailability ? "Evolution":"ALPS"));

	if(m_bBTEvolutionAvailability)
	{
       m_szServiceName = szBTEVO[BTDAEMON_SRV_NAME];
       m_szBaseObjectPath = szBTEVO[BTDAEMON_BASEOBJECT_PATH];
       m_szAudioStringInterface = szBTEVO[BTDAEMON_AUDIO_STRING_INTERFACE];
       m_szAudioInterface = szBTEVO[BTDAEMON_AUDIO_INTERFACE];
       m_szAVPInterface = szBTEVO[BTDAEMON_AVP_INTERFACE];
       m_szDeviceAddr = szBTEVO[BTDAEMON_DEVICE_ADDR];
	}
	else
	{
       m_szServiceName = szBTALPS[BTDAEMON_SRV_NAME];
       m_szBaseObjectPath = szBTALPS[BTDAEMON_BASEOBJECT_PATH];
       m_szAudioStringInterface = szBTALPS[BTDAEMON_AUDIO_STRING_INTERFACE];
       m_szAudioInterface = szBTALPS[BTDAEMON_AUDIO_INTERFACE];
       m_szAVPInterface = szBTALPS[BTDAEMON_AVP_INTERFACE];
       m_szDeviceAddr = szBTALPS[BTDAEMON_DEVICE_ADDR];
	}
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINDbusHandler::vUnInitialize()
 ***************************************************************************/
t_Void spi_tclMySPINDbusHandler::vUnInitialize()
{
   //Unregister for any signals registered
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINDbusHandler::vSetConfiguration()
 ***************************************************************************/
t_Void spi_tclMySPINDbusHandler::vSetConfiguration(trMySpinAudioConfig rMySpinAudioConfig)
{
   // Set the configurations
   m_rMySpinAudioConfig = rMySpinAudioConfig;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINDbusHandler::vGetConfiguration()
 ***************************************************************************/
t_Void spi_tclMySPINDbusHandler::vGetConfiguration(trMySpinAudioConfig& rMySpinAudioConfig)
{
   rMySpinAudioConfig = m_rMySpinAudioConfig;
}
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINDbusHandler::bStartA2DPAudioPipe()
 ***************************************************************************/
t_Bool spi_tclMySPINDbusHandler::bStartA2DPAudioPipe(t_String szDevBTMacAdd)
{
   t_Bool bIsPipeStarted = m_bBTEvolutionAvailability ? bStartBTEVOA2DPAudioPipe(szDevBTMacAdd) : \
		   	   	   	   	   	   	   	   	   	   	   	   	bStartBTALPSA2DPAudioPipe(szDevBTMacAdd);

   return bIsPipeStarted;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINDbusHandler::bStartBTALPSA2DPAudioPipe()
 ***************************************************************************/
t_Bool spi_tclMySPINDbusHandler::bStartBTALPSA2DPAudioPipe(t_String szDevBTMacAdd)
{
   ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bStartA2DPAudioPipe:Start A2DP Pipe"));

   t_Bool bIsPipeStarted = false;
   t_String strSub;
   t_U8 u8Length = szDevBTMacAdd.length();
   char szBTaddr[u8Length];
   strcpy(szBTaddr, szDevBTMacAdd.c_str());
   t_U8 hexValue = 0x00;
   t_U8 au8Address[6] = { 0, 0, 0, 0, 0, 0 };
   t_U8 u8Cnt = 0;

   DBusGProxy* poProxy = poGetDBusProxy(DBUS_PROXY_FOR_BTALPS_AUDIO);

   if (NULL != poProxy)
   {
      for (t_U8 it = 0; it < szDevBTMacAdd.length(); it += 2)
      {
         // convert the string two characters at a time
         strSub = "0x" + szDevBTMacAdd.substr(it, 2);

         hexValue = (uint8_t) strtoul(strSub.c_str(), NULL, 0);
         au8Address[u8Cnt] = hexValue;
         u8Cnt++;
      }

      GArray aBTAddr;
      aBTAddr.data = (gchar*) au8Address;
      aBTAddr.len = 6;

      t_Char* szInputDevice;
      guchar szcodecType;
      guint u32bitRate;
      GError* poError = NULL;

      bIsPipeStarted = com_alps_bt_Audio_start_audio_pipe(poProxy,
               &aBTAddr,
               &szInputDevice,
               &szcodecType,
               &u32bitRate,
               &poError);
      ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe-Start A2DP Pipe [%d]", ETG_ENUM(BOOL, bIsPipeStarted)));

      if (true == bIsPipeStarted)
      {
         // Now the method has returned, so read the out parameters
         t_String szInputDev(szInputDevice);
         ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe-Input device from BT Stack to start A2DP [%s]", szInputDev.c_str()));

         //populate available configurations.
         trMySpinAudioConfig rMySpinAudioConfig;
         rMySpinAudioConfig.szInputDev = szInputDev;
         rMySpinAudioConfig.szEncoding = szcodecType;
         rMySpinAudioConfig.u32bitRate = u32bitRate;
         rMySpinAudioConfig.szAddress = szDevBTMacAdd;

         vSetConfiguration(rMySpinAudioConfig);
      }
      else
      {
         ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe-Could not start audio pipe [%s]", poError->message));
         // free error
         g_error_free(poError);
      }
   }
   return bIsPipeStarted;

}
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINDbusHandler::bStartBTEVOA2DPAudioPipe()
 ***************************************************************************/
t_Bool spi_tclMySPINDbusHandler::bStartBTEVOA2DPAudioPipe(t_String szDevBTMacAdd)
{
   ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bStartA2DPAudioPipe:Start A2DP Pipe"));

   t_Bool bIsPipeStarted = false;

   //Done for evolution stack
   m_rMySpinAudioConfig.szAddress = szDevBTMacAdd;

   DBusGProxy* poProxy = poGetDBusProxy(DBUS_PROXY_FOR_BTALPS_AUDIO);

   if (NULL != poProxy)
   {
      guint u32bitRate = 0;
      GError* poError = NULL;

      GHashTable* poTable = NULL;
      t_String szInterfaceName =  m_szAudioStringInterface;
      bIsPipeStarted = org_freedesktop_DBus_Properties_get_all(poProxy,szInterfaceName.c_str(),&poTable,&poError);
      const t_Char* szInputDevice = NULL;
      const t_Char* szcodecType = NULL;
      GValue* result;
	  
      ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe-Start A2DP Pipe [%d]", ETG_ENUM(BOOL, bIsPipeStarted)));
      if(bIsPipeStarted)
      {	  
    	  guint u32Size = g_hash_table_size(poTable);
    	  ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe size of hash table [%d]",u32Size));

    	  if(u32Size>0)
    	  {
    		  result = (GValue*) g_hash_table_lookup(poTable,"CodecBitrate");
    		  u32bitRate = g_value_get_uint(result);
    		  result = (GValue*)g_hash_table_lookup(poTable,"CodecType");
    		  szcodecType = g_value_get_string(result);
    		  result = (GValue*)g_hash_table_lookup(poTable,"CodecPipeName");
    		  szInputDevice = g_value_get_string(result);
    	  }

         ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe-Input device from BT Stack to start A2DP [%s]", szInputDevice));
         ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe Bit Rate [%d]",u32bitRate));
         ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe Codec Type [%s]",szcodecType));

         //populate available configurations.
         trMySpinAudioConfig rMySpinAudioConfig;
         rMySpinAudioConfig.szInputDev = szInputDevice != NULL ? szInputDevice : "";
         rMySpinAudioConfig.szEncoding = szcodecType != NULL ? szcodecType : "";
         rMySpinAudioConfig.u32bitRate = u32bitRate;
         rMySpinAudioConfig.szAddress = szDevBTMacAdd.empty() ? "" : szDevBTMacAdd;

         vSetConfiguration(rMySpinAudioConfig);
         g_hash_table_destroy(poTable);
      }
      else
      {
         ETG_TRACE_USR4(("[PARAM]::bStartA2DPAudioPipe-Could not start audio pipe [%s]", poError->message));
         // free error
         g_error_free(poError);
      }
   }
   return bIsPipeStarted;

}
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINDbusHandler::bStopPlayBackFromSlot()
 ***************************************************************************/
t_Bool spi_tclMySPINDbusHandler::bStopA2DPAudioPipe()
{
   gboolean bStopPipeStatus = true;
   if(!m_bBTEvolutionAvailability)
   {
	   //Get the dbus proxy for ALPS BT Stack
	   DBusGProxy* poProxy = poGetDBusProxy(DBUS_PROXY_FOR_BTALPS_AUDIO);
	   GError* poError = NULL;
	   if (NULL != poProxy)
	   {
		  bStopPipeStatus = com_alps_bt_Audio_stop_audio_pipe(poProxy, &poError);

		  if ((true == bStopPipeStatus) && (NULL == poError))
		  {
			 ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bStopA2DPAudioPipe:Stopped A2DP Audio Pipe"));
		  }
		  else
		  {
			 ETG_TRACE_USR4(("[PARAM]::bStopA2DPAudioPipe-Failed to stop A2DP pipe with error [%s] ", poError->message));
			 // free error
			 g_error_free(poError);
		  }
	   }
   }
   return bStopPipeStatus;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINDbusHandler::bSendAVPCtrlCmd()
 ***************************************************************************/
t_Bool spi_tclMySPINDbusHandler::bSendAVPCtrlCmd(t_U8 u8AVPcmd, t_U8 szEventType)
{
   GError* poError = NULL;
   t_U8 u8Res = 0;
   gboolean bSendCmdStatus = 0;
   if(!m_bBTEvolutionAvailability)
   {
	   //Get the dbus proxy for ALPS BT Stack for AVP
	   DBusGProxy* poProxy = poGetDBusProxy(DBUS_PROXY_FOR_AVP);

	   t_U8 u8RetAVPCmd = 0;
	   t_U8 retEventType = 0;
	   u8Res = ~(u8Res & 0); //Set all bits to 1
	   /* Equivalent dbus command line
		root@mx6q:~#  dbus-send  --system --print-reply --dest=com.alps.bt /com/alps/bt/dev_90_e7_c4_f1_08_44 com.alps.bt.Audiovisual.AvpCtrlCmd byte:1 byte:0
		[  942.669573] method return sender=:1.19 -> dest=:1.41 reply_serial=2
		byte 1
		byte 0
		byte 0
		*/
	   bSendCmdStatus = com_alps_bt_Audiovisual_avp_ctrl_cmd(poProxy,
				u8AVPcmd,
				szEventType,
				&u8RetAVPCmd,
				&retEventType,
				&u8Res,
				&poError);
   }
   else
   {
	   /*root@mx6q:~#  dbus-send  --system --print-reply --dest=org.bluez /org/bluez/hci1/dev_C0_EE_FB_31_48_AD/player0 \
		* org.bluez.MediaPlayer.Play
	   [ 1109.343854] method return sender=:1.29 -> dest=:1.47 reply_serial=2*/

	   bSendCmdStatus = bGetMediaPlayerPath();
	   if(bSendCmdStatus)
	   {
		   DBusGProxy* poProxy = poGetDBusProxy(DBUS_PROXY_FOR_AVP);
		   if(scou8KEY_EVENT_RELEASE == szEventType)
		   {
		       if(e8_CMD_FORWARD == u8AVPcmd || e8_CMD_REWIND == u8AVPcmd)
		       {
		           /*in case of key event release for FFWD and FREV, we need to send play in order to make bt_stack stop the
		           FFWD or FREV operation*/
		           u8AVPcmd = e8_CMD_PLAY;
		       }
		       else
		       {
		           /*in case of key event release, we need to send e8_CMD_UNUSED else we end up doing the same operation twice*/
		           u8AVPcmd = e8_CMD_UNUSED;
		       }
		   }
		   ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bSendAVPCtrlCmd:Forwarding request to BT STACK for AVPCtrlCmd = %d KeyMode = %d",
		                   u8AVPcmd,szEventType));
		   switch(u8AVPcmd)
		   {
		   case e8_CMD_PLAY:
			   bSendCmdStatus = org_bluez_MediaPlayer_play(poProxy,&poError);
			   break;
		   case e8_CMD_PAUSE:
			   bSendCmdStatus = org_bluez_MediaPlayer_pause(poProxy,&poError);
			   break;
		   case e8_CMD_STOP:
			   bSendCmdStatus = org_bluez_MediaPlayer_stop(poProxy,&poError);
			   break;
		   case e8_CMD_NEXTTRACK:
			   bSendCmdStatus = org_bluez_MediaPlayer_next(poProxy,&poError);
			   break;
		   case e8_CMD_PREVTRACK:
			   bSendCmdStatus = org_bluez_MediaPlayer_previous(poProxy,&poError);
			   break;
		   case e8_CMD_FORWARD:
		       bSendCmdStatus = org_bluez_MediaPlayer_fast_forward(poProxy,&poError);
		       break;
		   case e8_CMD_REWIND:
		       bSendCmdStatus = org_bluez_MediaPlayer_rewind(poProxy,&poError);
		       break;
		   default:
			   ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::Key not supported"));
			   break;
		   }
	   }
   }
   if ((true == bSendCmdStatus) && (NULL == poError) && (0 == u8Res))
   {
      ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bSendAVPCtrlCmd:AVP Ctrl Command Sent successfully"));
   }
   else if(NULL != poError)
   {
      ETG_TRACE_USR4(("[PARAM]:bSendAVPCtrlCmd-Failed to send AVP Ctrl command with error [%s] ", poError->message));
      // free error
      g_error_free(poError);
   }

   return bSendCmdStatus;
}
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINDbusHandler::bGetMediaPlayerPath()
 ***************************************************************************/
t_Bool spi_tclMySPINDbusHandler::bGetMediaPlayerPath()
{
   GPtrArray* poArray;
   GError* poError = NULL;
   gboolean bSendCmdStatus;
   DBusGProxy* poProxy = poGetDBusProxy(DBUS_PROXY_FOR_AVP_PLAYER);
   bSendCmdStatus = org_bluez_evo_AVP_get_players(poProxy,&poArray,&poError);

   if(bSendCmdStatus && NULL != poArray && poArray->len > 0)
   {
       ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bGetMediaPlayerPath: Message sent Successfully"));
       gchar* path = (gchar*)g_ptr_array_index(poArray, 0);
       m_rMySpinAudioConfig.szPlayerAddress = path;
       ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bGetMediaPlayerPath: Device address %s",path));
      g_ptr_array_free (poArray, TRUE);
   }

   if(NULL != poError)
   {
	   ETG_TRACE_USR4(("[PARAM]:bGetMediaPlayerPath-Failed to send GetPlayers command with error [%s] ", poError->message));
	   g_error_free(poError);
   }

   return bSendCmdStatus;
}
/***************************************************************************
 ** FUNCTION: DBusGProxy* spi_tclMySPINDbusHandler::poGetDBusProxy()
 ***************************************************************************/
DBusGProxy* spi_tclMySPINDbusHandler::poGetDBusProxy(tenProxyType enProxyType)
{
   // Utility to get the DBUS proxies.
   DBusGProxy* poProxy = NULL;
   ETG_TRACE_USR2(("[DESC]:Fetching the DBUS proxy for [%d]", ETG_ENUM(DBUS_PROXY_TYPE, enProxyType)));

   switch (enProxyType)
   {
      case DBUS_PROXY_FOR_BTALPS_AUDIO:
      {
    	 t_String szDevicename = szGetDevicePath();
         if (NULL == m_poALPSBTDeamonProxy)
         {
            //If the proxy is not created, create it.
            m_poALPSBTDeamonProxy = dbus_g_proxy_new_for_name(m_poSystemDBusConn,
                     m_szServiceName.c_str(),
                     szDevicename.c_str(),
            		 m_szAudioInterface.c_str());

            if (NULL == m_poALPSBTDeamonProxy)
            {
               ETG_TRACE_USR4(("[PARAM]:poGetDBusProxy-Failed to create Proxy for BT ALPS audio"));
            }
         }
         poProxy = m_poALPSBTDeamonProxy;
      }
         break;

      case DBUS_PROXY_FOR_AVP:
      {
         if (NULL == m_poAVProxy)
         {
            //Device player path to be sent as ex - /org/bluez/hci1/dev_C0_EE_FB_31_48_AD/player0
            m_poAVProxy = dbus_g_proxy_new_for_name(m_poSystemDBusConn,
            		 m_szServiceName.c_str(),
            		 m_rMySpinAudioConfig.szPlayerAddress.c_str(),
            		 m_szAVPInterface.c_str());

            if (NULL == m_poAVProxy)
            {
               ETG_TRACE_ERR(("[ERR]:Failed to create Proxy for BT ALPS AVP"));
            }
         }
         poProxy = m_poAVProxy;
      }
         break;
      case DBUS_PROXY_FOR_AVP_PLAYER:
      {
    	  if (NULL == m_poMPProxy)
    	  {
    		  t_String szDevicename = szGetDevicePath();
			  m_poMPProxy = dbus_g_proxy_new_for_name(m_poSystemDBusConn,
					 m_szServiceName.c_str(),
					 szDevicename.c_str(),
					 m_szAudioStringInterface.c_str());

			  if (NULL == m_poMPProxy)
			  {
				 ETG_TRACE_ERR(("[ERR]:Failed to create Proxy for BT Media Player"));
			  }
    	  }
    	  poProxy = m_poMPProxy;
      }
         break;
      default:
      {
         ETG_TRACE_ERR(("[ERR]:Could not fetch DBUS Proxy !"));
      }
         break;
   }
   return poProxy;
}
/***************************************************************************
 ** FUNCTION: t_String spi_tclMySPINDbusHandler::szGetDevicePath()
 ***************************************************************************/
t_String spi_tclMySPINDbusHandler::szGetDevicePath()
{
   // create the device path from device name in the
   // format /com/alps/bt/dev_ma_ca_dd_re_ss

   t_String szDeviceAddress = m_szDeviceAddr;
   t_String szUnderscore = "_";
   t_String szBTAddr = m_rMySpinAudioConfig.szAddress;

   t_U8 u8AddrLen = m_rMySpinAudioConfig.szAddress.length();
   t_U8 u8Pos = 0;

   m_bBTEvolutionAvailability ? std::transform(szBTAddr.begin(), szBTAddr.end(), szBTAddr.begin(), ::toupper) : \
                                std::transform(szBTAddr.begin(), szBTAddr.end(), szBTAddr.begin(), ::tolower);

   while (u8Pos < u8AddrLen)
   {
	   szDeviceAddress += szUnderscore + szBTAddr.substr(u8Pos, 2);
      u8Pos += 2;
   }

   ETG_TRACE_USR4(("[PARAM]::szGetDevicePath:Device Path for Command Control [%s]", szDeviceAddress.c_str()));

   return szDeviceAddress;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclMySPINDbusHandler::szGetDevicePath()
 ***************************************************************************/
t_Void spi_tclMySPINDbusHandler::vClearDBUSProxies()
{
    ETG_TRACE_USR4(("spi_tclMySPINDbusHandler::vClearDBUSProxies entered"));
    m_poALPSBTDeamonProxy = NULL;
    m_poAVProxy = NULL;
    m_poMPProxy = NULL;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINDbusHandler::bGetStreamingStatus()
 ***************************************************************************/
t_Bool spi_tclMySPINDbusHandler::bGetA2DPStreamingStatus(t_String szDevBTMacAdd)
{
   ETG_TRACE_USR1(("spi_tclMySPINDbusHandler::bGetA2DPStreamingStatus entered"));

   t_Bool bStreamingStatus = false;

   m_rMySpinAudioConfig.szAddress = szDevBTMacAdd;

   DBusGProxy* poProxy = poGetDBusProxy(DBUS_PROXY_FOR_BTALPS_AUDIO);

   if (NULL != poProxy)
   {
      gboolean bIsStreaming = false;
      t_Bool bResult = false;
      GError* poError = NULL;

      GHashTable* poTable = NULL;
      t_String szInterfaceName =  m_szAudioStringInterface;
      bResult = org_freedesktop_DBus_Properties_get_all(poProxy,szInterfaceName.c_str(),&poTable,&poError);
      GValue* result;

      ETG_TRACE_USR4(("[PARAM]::bGetA2DPStreamingStatus-DBUS call status [%d]", ETG_ENUM(BOOL, bResult)));
      if(bResult)
      {
          guint u32Size = g_hash_table_size(poTable);
          ETG_TRACE_USR4(("[PARAM]::bGetA2DPStreamingStatus size of hash table [%d]",u32Size));

          if(u32Size>0)
          {
              result = (GValue*) g_hash_table_lookup(poTable,"Streaming");
              bIsStreaming = g_value_get_boolean(result);
              if(bIsStreaming)
                  bStreamingStatus = true;
          }
          g_hash_table_destroy(poTable);
      }
      else
      {
         ETG_TRACE_USR4(("[PARAM]::bGetA2DPStreamingStatus-Could not fetch streaming status [%s]", poError->message));
         // free error
         g_error_free(poError);
      }

      ETG_TRACE_USR4(("[PARAM]::bGetA2DPStreamingStatus Streaming [%d]",bStreamingStatus));
   }
   return bStreamingStatus;
}
