/*!
 *******************************************************************************
 * \file             spi_tclMySPINConnection.h
 * \brief            Base Connection class for MySPIN
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 02.11.2015 |  Shiva Kumar Gurija          | Initial Version
 05.04.2016 |  Chaitra Srinivasa           | Adding comments
 \endverbatim
 ******************************************************************************/
#ifndef SPI_TCLMYSPINCONNECTION_H_
#define SPI_TCLMYSPINCONNECTION_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <functional> //! C++11
#include "mySPINTypes.h"
#include "spi_tclConnSettingsIntf.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

//! Callback signatures definitions: To be registered by the Creator of this class object
typedef std::function<void(t_U32, const trMySPINDeviceInfo&)> tfvMySPINDeviceConnection;
typedef std::function<void(t_U32, tenDeviceCategory, tenDeviceSubCategory)> tfvMySPINDeviceDisconnection;
typedef std::function<void(t_U32, tenDeviceSubCategory, tenErrorCode)> tfvMySPINSelectDeviceResult;
typedef std::function<void(t_U32, tenErrorCode)> tfvMySPINSelectDeviceError;

struct trMySPINConnCbs
{
      //!Inform the select device result
      tfvMySPINSelectDeviceResult fvSelectDeviceResult;

      //! Sets error flag to prevent further automatic selection
      tfvMySPINSelectDeviceError fvSelectDeviceError;

      trMySPINConnCbs() :
         fvSelectDeviceResult(NULL), fvSelectDeviceError(NULL)
      {
      }
};

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/*!
 * \class spi_tclMySPINConnection
 * \brief Base class for Connection classes. Provides basic connection
 *         interface to be implemented by derived classes
 */

class spi_tclMySPINConnection
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::spi_tclMySPINConnection
       ***************************************************************************/
      /*!
       * \fn     spi_tclMySPINConnection()
       * \brief  Default Constructor
       * \sa      ~spi_tclMySPINConnection()
       **************************************************************************/
      spi_tclMySPINConnection():m_poConnSettings(NULL)
      {
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::~spi_tclMySPINConnection
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclMySPINConnection()
       * \brief  virtual Destructor
       * \sa     spi_tclMySPINConnection()
       **************************************************************************/
      virtual ~spi_tclMySPINConnection()
      {
         m_poConnSettings = NULL;
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks()
       * \brief  interface for the creator class to register for the required
       *        callbacks.
       *         Mandatory interface
       * \param corfrMySPINConnCbs : reference to the callback structure
       *        populated by the caller
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vRegisterCallbacks(const trMySPINConnCbs &corfrMySPINConnCbs) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::bInitializeConnection
       ***************************************************************************/
      /*!
       * \fn     bInitializeConnection()
       * \brief  required initializations.
       *         Mandatory interface
       * \retval returns true on successful initialization and false on failure
       **************************************************************************/
      virtual t_Bool bInitializeConnection() = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::vUnInitializeConnection
       ***************************************************************************/
      /*!
       * \fn     vUnInitializeConnection()
       * \brief  Uninitialization of sdk's etc
       *         Mandatory interface
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vUnInitializeConnection() = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::vOnLoadSettings
       ***************************************************************************/
      /*!
       * \fn     vOnLoadSettings()
       * \brief  Called on loadsettings
       *         Optional interface
       * \param corfrheadUnitInfo : Head unit information
       * \param corfrVehicleInfo : Vehicle info
       * \param u16VehicleType : Vehicle type
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vOnLoadSettings(const trHeadUnitInfo &corfrheadUnitInfo,
                             const trVehicleInfo& corfrVehicleInfo,
                             t_U16 u16VehicleType)
      {
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::vOnSaveSettings
       ***************************************************************************/
      /*!
       * \fn     vOnSaveSettings()
       * \brief  Called on savesettings
       *         Optional interface
       * \retval none
       **************************************************************************/
      virtual t_Void vOnSaveSettings()
      {
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::vOnSelectDevice
       ***************************************************************************/
      /*!
       * \fn     vOnSelectDevice
       * \brief  Called when a device is selected by the user. Optional interface
       * \param  cou32DevID : Uniquely identifies the target Device.
       * \param  enDevConnType   : Identifies the Connection Type.
       * \param  enDevSelectType : Identifies the Connection Request.
       * \param  corUsrCntxt      : User Context Details.
       * \param  corfrMySPINDevInfo : Identifies MySPIN device information
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vOnSelectDevice(const t_U32 cou32DevID, tenDeviceConnectionType enDevConnType,
               tenDeviceConnectionReq enDevSelectType, const trUserContext corUsrCntxt,
               const trMySPINDevInfo& corfrMySPINDevInfo)= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINConnection::vOnSelectDeviceResult()
       ***************************************************************************/
      /*!
       * \fn      t_Void vOnSelectDeviceResult
       * \brief   To perform the actions that are required, after the select device is
       *           successful/failed
       * \param    cou32DevID  : [IN] Unique device ID
       * \param    coenConnReq : [IN] Identifies the Connection Request.
       * \param    coenRespCode: [IN] Response code. Success/Failure
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vOnSelectDeviceResult(const t_U32 cou32DevID, const tenDeviceConnectionReq coenConnReq,
               const tenResponseCode coenRespCode)= 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::vSetDeviceSelectionState
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceSelectionState()
       * \brief  Sets the device Selection State
       * \retval returns true on successful detection and false on failure
       **************************************************************************/
      virtual t_Void vSetDeviceSelectionState(t_Bool bSelectionInProgress)
      {
      }

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINConnection::vSetConnsettingsInstance(spi_tclConnSettingsIntf* poConnSettingsIntf)
       ***************************************************************************/
       /*!
       * \fn      t_Void vSetConnsettingsInstance()
       * \brief   Method to set configuration manager instance.
       * \param   poConnSettingsIntf  : pointer to ConnsettingsIntf.
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetConnsettingsInstance(spi_tclConnSettingsIntf* poConnSettingsIntf)
      {
         if( NULL != poConnSettingsIntf)
         {
            m_poConnSettings = poConnSettingsIntf;
         }
      }
      
      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnMngr::vSetSelectionProgressState
       **************************************************************************/
      /*!
       * \fn      vSetSelectionProgressState()
       * \brief   set selection state to myspin Connmanger
       * \retval  t_Void
       *************************************************************************/
      virtual t_Void vOnSetSelectionProgressState(tenDeviceSelectionState /*enDeviceSelectionState*/)
      {
      }

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINConnection::vMySPINSessionErrorCb
       ***************************************************************************/
      /*!
       * \fn      vMySPINSessionErrorCb
       * \brief   Informs the mySPIN Error from Phone
       * \param   enErrorCode : Voice Session Request
       * \sa
       **************************************************************************/
      virtual t_Void vMySPINSessionErrorCb(tenMySPINError enErrorCode)
      {
      }
   protected:
      spi_tclConnSettingsIntf* m_poConnSettings;

   private:

      /***************************************************************************
       *********************************PRIVATE************************************
       ***************************************************************************/

};
/*! } */
#endif // SPI_TCLMYSPINCONNECTION_H_
