/*!
 *******************************************************************************
 * \file         spi_tclMySPINBluetooth.h
 * \brief        MySpin Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    MySpin Bluetooth handler class for SPI
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                | Modification
 02.11.2015  | tch5kor | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclBluetoothIntf.h"
#include "spi_tclMySPINBluetooth.h"
#include "spi_tclMySPINManager.h"
#include "spi_tclDeviceIDDataIntf.h"
#include <algorithm>
#include "RespRegister.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BLUETOOTH
#include "trcGenProj/Header/spi_tclMySPINBluetooth.cpp.trc.h"
#endif
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
static spi_tclMySPINManager* spoMySPINMngr = NULL;
static t_Bool sbBTLimModeRequested = false;
/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINBluetooth::spi_tclMySPINBluetooth();
 ***************************************************************************/
spi_tclMySPINBluetooth::spi_tclMySPINBluetooth(spi_tclBluetoothIntf* poBTInterface) :
   m_cpoBTInterface(poBTInterface), m_u8CurrentProfiles(0xFF),
   m_bIsHFPConn(false), m_bIsA2DPConn(false),
   m_enHFPStatus(e8MSPIN_HFP_UNKNOWN),
   m_enA2DPStatus(e8MSPIN_A2DP_UNKNOWN),
   m_enPendingBTLimAction(e8BT_LIM_ACTION_UNKNOWN),
   m_enMySPINError(e8MSPIN_ERROR_UNKNOWN)
{
   ETG_TRACE_USR1(("spi_tclMySPINBluetooth() entered "));
   SPI_NORMAL_ASSERT(NULL == m_cpoBTInterface);
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINBluetooth::~spi_tclMySPINBluetooth();
 ***************************************************************************/
spi_tclMySPINBluetooth::~spi_tclMySPINBluetooth()
{
   ETG_TRACE_USR1(("~spi_tclMySPINBluetooth() entered "));
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINBluetooth::bInitialise();
 ***************************************************************************/
t_Bool spi_tclMySPINBluetooth::bInitialise()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclMySPINBluetooth::bInitialise() entered "));

   spoMySPINMngr = spi_tclMySPINManager::getInstance();
   SPI_NORMAL_ASSERT(NULL == spoMySPINMngr);

   t_Bool bret = false;
   if (NULL != spoMySPINMngr)
   {
       bret = spoMySPINMngr->bRegisterObject((spi_tclMySPINRespSession*) this);
       ETG_TRACE_USR2(("[DESC]Registered to Session Response [%d]", ETG_ENUM(BOOL, bret)));
   }

   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
       bret = pRespRegister->bRegisterObject((spi_tclExtRespBluetooth*)this);
   }
   return bret;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPISelectDeviceRequest(
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq)
   ETG_TRACE_USR1(("spi_tclMySPINBluetooth::vOnSPISelectDeviceRequest() entered "));

   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
      pRespRegister->bRegisterObject((spi_tclExtRespNativeTransport*)this);
   }

   if (NULL != m_cpoBTInterface)
   {
      m_cpoBTInterface->vSendSelectDeviceResult(true);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPISelectDeviceResponse(
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq, tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("spi_tclMySPINBluetooth::vOnSPISelectDeviceResponse() entered "));
   tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
   spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);
   //! For Myspin iPhones BTAddress is not known so this parameter can be an empty string.
   t_String szBTAddress;

   spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = NULL;
   if(NULL != m_cpoBTInterface)
   {
       poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
   }

   if ((e8SUCCESS == enRespCode) && (NULL != poExtCmdBluetoothIntf) )
   {
      su32CurSelectedDevId = corfrSelectReq.m_u32DeviceHandle;

      if(e8DEVTYPE_AOAP == enDeviceSubCategory)
      {
         //Get the vehicle BT MAC Address
         t_String szVehBTAddress;
         m_poBluetoothSettings->vGetVehicleBTAddress(szVehBTAddress);

         if (false == szVehBTAddress.empty())
         {
            //Add delimiters to the BT Address since IVI Core requires them to be
            //C-string of six two-digit hex values with colon delimiter.

              t_U8 u8Index = szVehBTAddress.length();

              while (u8Index > 2)
              {
                 u8Index = u8Index - 2;
                 szVehBTAddress.insert(u8Index, 1, ':');
              }
              std::transform(szVehBTAddress.begin(), szVehBTAddress.end(), szVehBTAddress.begin(), ::toupper);

              if (NULL != spoMySPINMngr)
              {
                  spi_tclMySPINCmdVehicleData* poCmdvehData = spoMySPINMngr->poGetVehDataInstance();
                  if (NULL != poCmdvehData)
                  {
                     poCmdvehData->vSetVehicleBTMACAddress(corfrSelectReq.m_u32DeviceHandle, szVehBTAddress);
                  }
              }
         }
      }
      else
      {
         //Check if any messages have piled up from Media player
         std::map<t_U32,trBTDeviceInfo>::iterator itrBTDevInfo;
         itrBTDevInfo = m_rBTDeviceInfo.find(corfrSelectReq.m_u32DeviceHandle);

         if (m_rBTDeviceInfo.end() != itrBTDevInfo)
         {
            vBTProfileInfoCb(corfrSelectReq.m_u32DeviceHandle,
                     itrBTDevInfo->second.bIsHFP,
                     itrBTDevInfo->second.bIsA2DP,
                     itrBTDevInfo->second.szFriendlyName);

            //Clear the device from map
            m_rBTDeviceInfo.erase(itrBTDevInfo);
         }
         if(NULL != m_cpoBTInterface)
         {
            m_szFriendlyName =  m_cpoBTInterface->szGetBTDeviceName();
         }
         
         vPostBTProfileStatusUpdateMsg();

         //Start BT limitation mode
         spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = NULL;
         if (NULL != m_cpoBTInterface)
         {
            poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
            if (NULL != poExtCmdBluetoothIntf)
            {
               sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
                     szBTAddress,m_szFriendlyName,
                     e8BT_TECH_MYSPIN_IOS,
                     e8BT_COMM_CHN_USB,
                     e8BT_LIM_ACTION_ACTIVATE);

               if(false == sbBTLimModeRequested)
               {
                   ETG_TRACE_USR2(("BT service unavailable. Will request BT Lim mode later"));
                   m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
                   m_enPendingBTLimAction = e8BT_LIM_ACTION_ACTIVATE;
               }
            }
         }
      }
   }
   else
   {
       m_enMySPINError = e8MSPIN_ERROR_UNKNOWN;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPIDeselectDeviceRequest(
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclMySPINBluetooth::vOnSPIDeselectDeviceRequest() entered"));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   //! For Myspin iPhones BTAddress is not known so this parameter can be an empty string.
   t_String szBTAddress;
   if (NULL != m_cpoBTInterface)
   {
      tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
      spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      if((NULL != poExtCmdBluetoothIntf) && (e8DEVTYPE_IAP == enDeviceSubCategory) && (sbBTLimModeRequested))
      {
          sbBTLimModeRequested = false;
          
          if (false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
                szBTAddress,
                m_cpoBTInterface->szGetBTDeviceName(),
                e8BT_TECH_MYSPIN_IOS,
                e8BT_COMM_CHN_USB,
                e8BT_LIM_ACTION_DEACTIVATE))
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
             m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
          }
      }

      su32CurSelectedDevId = 0;
      m_u8CurrentProfiles = 0xFF;    //clean current BT profile after phone deselected.
      m_szFriendlyName.clear();
      m_rBTDeviceInfo.clear();
      // Nothing else to be done. Simply send success result.
      m_cpoBTInterface->vSendSelectDeviceResult(true);
      m_enMySPINError = e8MSPIN_ERROR_UNKNOWN;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPIDeselectDeviceResponse(
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode)
{
   SPI_INTENTIONALLY_UNUSED(enRespCode);
   ETG_TRACE_USR1(("spi_tclMySPINBluetooth::vOnSPIDeselectDeviceResponse() entered"));
   if(e8DEVCONNREQ_DESELECT == corfrSelectReq.m_enDevConnReq)
   {
      if(NULL != m_cpoBTInterface)
      {
         m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
      }
      m_enHFPStatus = e8MSPIN_HFP_UNKNOWN;
      m_enA2DPStatus = e8MSPIN_A2DP_UNKNOWN;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vBTProfileInfoCb(t_U32...)
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vBTProfileInfoCb(const t_U32 cou32DeviceHandle, t_Bool bIsHFP, t_Bool bIsA2DP,
         t_String szFriendlyName)
{
   ETG_TRACE_USR2(("spi_tclMySPINBluetooth::vBTProfileInfoCb() entered for device[%d]",cou32DeviceHandle));
   ETG_TRACE_USR2(("spi_tclMySPINBluetooth::vBTProfileInfoCb() szFriendlyName [%s]",szFriendlyName.c_str()));
   
   if (su32CurSelectedDevId == cou32DeviceHandle)
   {

          //Inform Audio When BT is connected/disconnected in case of android phones
          tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
          spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);
          
          t_U32 u32ConnBTDevHandle = 0;
          t_String szMacAddress;
          
          if(e8DEVTYPE_AOAP == enDeviceSubCategory)
          {
              m_szFriendlyName = szFriendlyName;

              vFetchBTDetails(u32ConnBTDevHandle,szMacAddress);

              vUpdateCmdAudio(u32ConnBTDevHandle,szMacAddress);

              vSendDeviceSwitchEvent(u32ConnBTDevHandle,e8DEVTYPE_AOAP);
          }
          else if(e8DEVTYPE_IAP == enDeviceSubCategory)
          {
             vFetchBTDetails(u32ConnBTDevHandle,szMacAddress);
             // Nothing else to be done. Simply send success result.
             vSendDeviceSwitchEvent(u32ConnBTDevHandle,e8DEVTYPE_IAP);
          }

   }//if ((NULL != m_cpoBTInterface) && (su32CurSelectedDevId == cou32DeviceHandle ))
   else
   {
      trBTDeviceInfo rBTDeviceInfo;
      rBTDeviceInfo.bIsA2DP = m_bIsA2DPConn;
      rBTDeviceInfo.bIsHFP = m_bIsHFPConn;
      rBTDeviceInfo.szFriendlyName = szFriendlyName;
      
      //Push in to a map which will be processed on select device result.
      m_rBTDeviceInfo[cou32DeviceHandle] = rBTDeviceInfo;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vPostBTProfileInfo(...)
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vPostBTProfileInfo(const t_U32 cou32DeviceHandle, t_Bool bIsHFP, t_Bool bIsA2DP, t_String szFriendlyName)
{
   ETG_TRACE_USR2(("spi_tclMySPINBluetooth::vPostBTProfileInfo() entered for [%d]",cou32DeviceHandle));
   vBTProfileInfoCb(cou32DeviceHandle,bIsHFP,bIsA2DP,szFriendlyName);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vPostBTProfileStatusUpdateMsg(...)
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vPostBTProfileStatusUpdateMsg()
{
    tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
    spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);
    
    if(e8DEVTYPE_UNKNWON != enDeviceSubCategory)
    {
        t_U32 u32ConnBTDevHandle = 0;
        t_String szMacAddress;
        
        vFetchBTDetails(u32ConnBTDevHandle,szMacAddress);

        if(e8DEVTYPE_AOAP == enDeviceSubCategory)
        {
           vUpdateCmdAudio(u32ConnBTDevHandle,szMacAddress);
           vSendDeviceSwitchEvent(u32ConnBTDevHandle,e8DEVTYPE_AOAP);
        }
        else if(e8DEVTYPE_IAP == enDeviceSubCategory)
        {
           // Nothing else to be done. Simply send success result.
           vSendDeviceSwitchEvent(u32ConnBTDevHandle,e8DEVTYPE_IAP);
        }
    }
}   

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vFetchBTDetails(...)
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vFetchBTDetails(t_U32 &rfu32ConnBTDevHandle, t_String& rfszMacAddress)
{     
     if ((!m_szFriendlyName.empty()) && (NULL != m_cpoBTInterface) && (0 != su32CurSelectedDevId))
     {
        t_Bool bIsHFPConn = false;
        t_Bool bIsA2DPConn = false;
        
        spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
        if(NULL != poExtCmdBluetoothIntf)
        {
            poExtCmdBluetoothIntf->vGetBtMacAddress(m_szFriendlyName, rfszMacAddress);
            rfu32ConnBTDevHandle = poExtCmdBluetoothIntf->u32GetBTDeviceHandle(rfszMacAddress);
            if(0 == rfu32ConnBTDevHandle)
            {
                m_enHFPStatus = e8MSPIN_HFP_UNKNOWN;
                m_enA2DPStatus = e8MSPIN_A2DP_UNKNOWN;
            }
            
            ETG_TRACE_USR2(("spi_tclMySPINBluetooth::vFetchBTDetails() BT device handle [%d]",rfu32ConnBTDevHandle));
            ETG_TRACE_USR2(("spi_tclMySPINBluetooth::vFetchBTDetails() m_enHFPStatus: [%d], m_enA2DPStatus: [%d]",m_enHFPStatus, m_enA2DPStatus));
            
            poExtCmdBluetoothIntf->vGetBTProfileStatus(rfu32ConnBTDevHandle,
                    bIsHFPConn, bIsA2DPConn);
            
            if((m_enHFPStatus == e8MSPIN_HFP_UNKNOWN) && (m_enA2DPStatus == e8MSPIN_A2DP_UNKNOWN))
            {
                m_enHFPStatus = (bIsHFPConn == true ) ? e8MSPIN_HFP_CONNECTED : e8MSPIN_HFP_DISCONNECTED;
                m_enA2DPStatus = (bIsA2DPConn == true) ? e8MSPIN_A2DP_CONNECTED : e8MSPIN_A2DP_DISCONNECTED;
                vSendVoiceSessionStatus();
            }
            else
            {
                if((m_enHFPStatus==e8MSPIN_HFP_DISCONNECTED)&&(bIsHFPConn==true))
                {
                    m_enHFPStatus = e8MSPIN_HFP_CONNECTED;
                    vSendVoiceSessionStatus();
                }
                if((m_enA2DPStatus==e8MSPIN_A2DP_DISCONNECTED)&&(bIsA2DPConn==true))
                {
                    m_enA2DPStatus = e8MSPIN_A2DP_CONNECTED;
                }
                if((m_enHFPStatus==e8MSPIN_HFP_CONNECTED)&&(bIsHFPConn==false))
                {
                    m_enHFPStatus = e8MSPIN_HFP_DISCONNECTED;
                    vSendVoiceSessionStatus();
                }
                if((m_enA2DPStatus==e8MSPIN_A2DP_CONNECTED)&&(bIsA2DPConn==false))
                {
                    m_enA2DPStatus = e8MSPIN_A2DP_DISCONNECTED;
                }
            }
        }
        spi_tclMySPINManager::vSetBTConnInfo(su32CurSelectedDevId,bIsHFPConn,rfszMacAddress);
     }
     else
     {
         ETG_TRACE_USR2(("spi_tclMySPINBluetooth::vFetchBTDetails() BT friendly name empty"));
         m_enHFPStatus = e8MSPIN_HFP_UNKNOWN;
         m_enA2DPStatus = e8MSPIN_A2DP_UNKNOWN;
     }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vUpdateCmdAudio(...)
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vUpdateCmdAudio(t_U32 u32ConnBTDevHandle,t_String szMacAddress)
{
     spi_tclMySPINCmdAudio* poCmdAudio = spoMySPINMngr->poGetAudioInstance();

     ETG_TRACE_USR1(("spi_tclMySPINBluetooth::vUpdateCmdAudio entered"));

     if ((NULL != poCmdAudio) && (0 != u32ConnBTDevHandle))
     {
          if(e8MSPIN_A2DP_CONNECTED == m_enA2DPStatus)
          {
              //If device is connected over A2DP, inform CmdAudio about it so that it can play music
              poCmdAudio->vSetBtMacAddress(su32CurSelectedDevId, szMacAddress);
          }
          t_Bool bA2DPStatus = false;
          if(e8MSPIN_A2DP_CONNECTED == m_enA2DPStatus)
          {
              bA2DPStatus = true;
          }
          else if(e8MSPIN_A2DP_DISCONNECTED == m_enA2DPStatus)
          {
              bA2DPStatus = false;
          } 
          poCmdAudio->vSetBluetoothStatus(su32CurSelectedDevId, bA2DPStatus, m_enA2DPStatus);
          
          //Update BT device name in device list
          if(NULL != m_cpoBTInterface)
          {
             m_cpoBTInterface->vSendBTDeviceInfo(su32CurSelectedDevId, m_szFriendlyName,szMacAddress);
          }

     }
}   

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vSendDeviceSwitchEvent(...)
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vSendDeviceSwitchEvent(t_U32 u32ConnBTDevHandle,tenDeviceSubCategory enDeviceSubCategory)
{
    if((NULL != m_cpoBTInterface) && (0 != su32CurSelectedDevId))
    {
        t_U8 u8Profiles = 0;
        t_Bool bIsSameDevice = false;
 
        if(e8MSPIN_HFP_CONNECTED == m_enHFPStatus)
        {
            u8Profiles |= 1 << 0;
        }
        else if(e8MSPIN_HFP_DISCONNECTED == m_enHFPStatus)
        {
            u8Profiles &= ~(1 << 0);
        }

        if(e8MSPIN_A2DP_CONNECTED == m_enA2DPStatus)
        {
            u8Profiles |= 1 << 1;
        }
        else if(e8MSPIN_A2DP_DISCONNECTED == m_enA2DPStatus)
        {
            u8Profiles &= ~(1 << 1);
        } 
        
        if(enDeviceSubCategory == e8DEVTYPE_AOAP)
        {

           if((e8MSPIN_HFP_CONNECTED == m_enHFPStatus) && (e8MSPIN_A2DP_CONNECTED == m_enA2DPStatus))
           {
               bIsSameDevice = true;
           }
           else if((e8MSPIN_HFP_DISCONNECTED == m_enHFPStatus) && (e8MSPIN_A2DP_DISCONNECTED == m_enA2DPStatus))
           {
               bIsSameDevice = false;
           }
        }
        else
        {
           if(e8MSPIN_HFP_CONNECTED == m_enHFPStatus)
           {
               bIsSameDevice = true;
           }
           else if(e8MSPIN_HFP_DISCONNECTED == m_enHFPStatus)
           {
               bIsSameDevice = false;
           }
        } 
 
        if(u8Profiles != m_u8CurrentProfiles)
        {
            m_cpoBTInterface->vSendDeviceSwitchEvent(u32ConnBTDevHandle, su32CurSelectedDevId, e8NO_CHANGE, u8Profiles, bIsSameDevice, false);
            m_u8CurrentProfiles = u8Profiles;
        }
    }
}  

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINBluetooth::vSendVoiceSessionStatus()
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vSendVoiceSessionStatus()
{
    spi_tclMySPINCmdSession* poCmdSession = spoMySPINMngr->poGetSessionInstance();
    if(NULL != poCmdSession)
    {
        ETG_TRACE_USR1(("spi_tclMySPINBluetooth::vSendVoiceSessionStatus BT status: [%d]",ETG_ENUM(MYSPIN_HFP_STATUS, m_enHFPStatus)));
        if(e8MSPIN_HFP_CONNECTED == m_enHFPStatus)
        {
            poCmdSession->vUpdateVoiceSessionStatus(su32CurSelectedDevId,
                    e8BTVOICESESSION_STATUS_IDLE);
        }
        else
        {
            poCmdSession->vUpdateVoiceSessionStatus(su32CurSelectedDevId,
              e8BTVOICESESSION_STATUS_UNAVAILABLE);
        }
    }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vPostBTServiceAvailabilityMsg()
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable)
{
   ETG_TRACE_USR1(("spi_tclMySPINBluetooth::vPostBTServiceAvailabilityMsg entered bServiceAvailable = %d",ETG_ENUM(BOOL,bServiceAvailable)));
   //! For Carlfie iPhones BTAddress is not known so this parameter can be an empty string.
   t_String szBTAddress;
   if( (true == bServiceAvailable) &&
       (NULL != m_cpoBTInterface) &&
       (e8DEVICE_CHANGE_DELAYED == m_cpoBTInterface->enGetSelectedDevStatus()) &&
       (e8DEV_TYPE_MYSPIN == m_cpoBTInterface->enGetSelectedDeviceCategory()) &&
       (e8BT_LIM_ACTION_UNKNOWN != m_enPendingBTLimAction))
   {
       spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
       if(NULL != poExtCmdBluetoothIntf)
       {
          sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
              szBTAddress,m_cpoBTInterface->szGetBTDeviceName(),
              e8BT_TECH_MYSPIN_IOS,
              e8BT_COMM_CHN_USB,
              m_enPendingBTLimAction);
              
          if (true == sbBTLimModeRequested)
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
          }
          else
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
          }
       }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINBluetooth::vMySPINSessionErrorCb()
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vMySPINSessionErrorCb(const t_U32 cou32DeviceHandle, tenMySPINError enErrorCode)
{
   //Get Device Subcategory
   tenDeviceSubCategory enDeviceSubCategory;
   spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

   ETG_TRACE_USR2(("[DESC]::Received Error - [%d] , Device SubCategory [%d]", ETG_ENUM(MSPIN_ERROR_CODE, enErrorCode), ETG_ENUM(MSPIN_DEV_SUBCAT,
            enDeviceSubCategory)));

   if (((e8MSPIN_ERROR_PROTOCOL_STOP == enErrorCode) || (e8MSPIN_ERROR_INITIALIZATION_ABORT == enErrorCode)) 
       && (e8DEVTYPE_IAP == enDeviceSubCategory))
   {
       m_enMySPINError = enErrorCode;
   }
}

/***************************************************************************
 ** FUNCTION: spi_tclMySPINBluetooth::vPostNativeTransportStartResult()
 ***************************************************************************/
t_Void spi_tclMySPINBluetooth::vPostNativeTransportStartResult(t_U32 u32DeviceId)
{
    if(((e8MSPIN_ERROR_PROTOCOL_STOP == m_enMySPINError) || (e8MSPIN_ERROR_INITIALIZATION_ABORT == m_enMySPINError))
        && (su32CurSelectedDevId == u32DeviceId))
    {
        vSendVoiceSessionStatus();
    }
}
//lint �restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
