/*!
 *******************************************************************************
 * \file             spi_tclMySPINAudio.h
 * \brief            Implements the Audio functionality for MySpin using
 interface to Real VNC SDK through VNC Wrapper.
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for MySpin
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                | Modification
 02.11.2015  | tch5kor               | Initial Version
 12.04.2016  | Chaitra Srinivasa     | Trace message cleanup
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "spi_tclMySPINAudio.h"
#include "spi_tclMySPINManager.h"
#include "spi_tclMySPINRespSession.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclMySPINAudio.cpp.trc.h"
#endif

static t_U32 su32SelectedDevice = 0;
static const t_U8 ONLY_ONE_ACTIVE_REQUEST = 1;

static const t_U8 scou8AudioUnduck      = 1;
static const t_U8 scou8AudioDuckDefdB   = 41; 
static const t_U16 scou16AudioRampDefDuration  = 1000;  //In msec

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::spi_tclMySPINAudio()
 ***************************************************************************/
spi_tclMySPINAudio::spi_tclMySPINAudio() : m_enDuckingType(e8_DUCKINGTYPE_UNKNOWN)
{
   ETG_TRACE_USR1((" spi_tclMySPINAudio::spi_tclMySPINAudio() "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::~spi_tclMySPINAudio()
 ***************************************************************************/
spi_tclMySPINAudio::~spi_tclMySPINAudio()
{
   ETG_TRACE_USR1((" spi_tclMySPINAudio::~spi_tclMySPINAudio()"));
   m_oActiveAudioDirLock.s16Lock();
   m_mapActiveAudioDir.clear();
   m_oActiveAudioDirLock.vUnlock();
   m_enDuckingType = e8_DUCKINGTYPE_UNKNOWN;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINAudio::bInitialise();
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bInitialise()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclMySPINAudio::bInitialise() entered "));

   t_Bool bRet = false;
   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
   if (NULL != poMySPINMngr)
   {
      bRet = poMySPINMngr->bRegisterObject((spi_tclMySPINRespSession*) this);
      ETG_TRACE_USR2(("[DESC]:Registered to Audio Callbacks from mySPIN Session [%d]", ETG_ENUM(BOOL, bRet)));
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks)
{
   ETG_TRACE_USR1((" spi_tclMySPINAudio::vRegisterCallbacks "));
   m_rAudCallbacks = rfrAudCallbacks;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vSelectDevice(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq)
{
   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
   ETG_TRACE_USR2(("[DESC]::vSelectDevice:Dev-0x%x", cou32DevId));

   //! Call the registered callbacks
   if ((NULL != m_rAudCallbacks.fvSelectDeviceResp))
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(cou32DevId, coenConnReq, true);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::bSelectAudioDevice()
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bSelectAudioDevice(t_U32 u32DeviceId)
{
   ETG_TRACE_USR2(("[DESC]::bSelectAudioDevice:Dev-0x%x", u32DeviceId));

   su32SelectedDevice = u32DeviceId;

   //Check if SPI supports advanced audio

   if (NULL != m_poAudioSettings)
   {
      t_Bool bIsAdvAudioSup = m_poAudioSettings->bGetMySIPNAdvAudioSupport();
      if (true == bIsAdvAudioSup)
      {
         spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

         if (NULL != poMySPINMngr)
         {
            spi_tclMySPINCmdAudio* poCmdAudio = poMySPINMngr->poGetAudioInstance();
            if (NULL != poCmdAudio)
            {
               trmySPINAudioCbs rmySPINAudioCbs ;
               rmySPINAudioCbs.fvRequestLaunchAudioCb = std::bind(&spi_tclMySPINAudio::vRequestLaunchAudioCmd,
                        this,
                        std::placeholders::_1);
               rmySPINAudioCbs.fvRequestTerminateAudioCb = std::bind(&spi_tclMySPINAudio::vRequestTerminateAudioCmd,
                        this,
                        std::placeholders::_1);
                        
               trmySPINAudioDuckCb rmySPINAudioDuckCb;
               rmySPINAudioDuckCb.fvAudioDuckingCb = std::bind(&spi_tclMySPINAudio::vAudioDucking,
                        this,
                        std::placeholders::_1);

               poCmdAudio->vRegisterAudioCbs(u32DeviceId,rmySPINAudioCbs,rmySPINAudioDuckCb);
            }
         }
      }
   }
   m_oActiveAudioDirLock.s16Lock();
   m_mapActiveAudioDir.clear();
   m_oActiveAudioDirLock.vUnlock();
   if ((NULL != m_rAudCallbacks.fvSelectDeviceResp))
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_SELECT, true);
   }

  return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::vDeselectAudioDevice()
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vDeselectAudioDevice(t_U32 u32DeviceId)
{
   ETG_TRACE_USR2(("[FUNC]spi_tclMySPINAudio::vDeselectAudioDevice:Dev-0x%x", u32DeviceId));

   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

   t_Bool bStopState = true;
   if (NULL != poMySPINMngr)
   {
      spi_tclMySPINCmdAudio* poCmdAudio = poMySPINMngr->poGetAudioInstance();

      if (NULL != poCmdAudio)
      {
          m_oActiveAudioDirLock.s16Lock();
         for (std::map<tenAudioDir, trAudioState>::iterator itrActiveAud = m_mapActiveAudioDir.begin(); itrActiveAud
            != m_mapActiveAudioDir.end(); ++itrActiveAud)
         {
             if(true == itrActiveAud->second.isActive)
             {
                 tenAudioStreamError enAudioStreamError = poCmdAudio->enStopAudioStreaming(itrActiveAud->first);
                 ETG_TRACE_USR2(("[DESC]spi_tclMySPINAudio::vDeselectAudioDevice: Stop Audio Stream [%d] dir [%d] ",
                           ETG_ENUM(MYSPIN_AUDIO_STREAM_ERROR, enAudioStreamError), ETG_ENUM(SPI_AUDIO_DIRECTION, itrActiveAud->first)));
                 bStopState = (e8NOERROR == enAudioStreamError);
                 if (e8AUD_STEREO_MIX_OUT == itrActiveAud->first)
                 {
                     vAudioDucking(e8_DUCKINGTYPE_UNDUCK);
                 }//if(NULL != m_rAudCallbacks.fbSetAudioDucking)
                 if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
                 {
                     (m_rAudCallbacks.fvTerminateAudioReq)(u32DeviceId, itrActiveAud->first, e8REASON_NOMEDIA);
                 }
             }
             vSetAudioSrcState(itrActiveAud->first, false); //set Main Audio source inactive/FALSE independent of Audio stop streaming state.
             poCmdAudio->bSendAudioCmdResponse(itrActiveAud->first,false,e8_REQUEST_INVALID,e8SPI_AUDIO_UNKNOWN);
             tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
             spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);
             poCmdAudio->vUnInitialize(enDeviceSubCategory, itrActiveAud->first);
         }
         m_oActiveAudioDirLock.vUnlock();
         
         su32SelectedDevice = 0;
         poCmdAudio->vClearBTDevices(u32DeviceId);
         poCmdAudio->vCleanup(); 
      }

   }
   

   if ((NULL != m_rAudCallbacks.fvSelectDeviceResp))
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_DESELECT, bStopState);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::bInitializeAudioPlayback()
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bInitializeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir,
      tenAudioSamplingRate enSamplingRate,tenAudioSamplingRate enNativeSamplingRate)
{
   SPI_INTENTIONALLY_UNUSED(enSamplingRate)
   ETG_TRACE_USR2(("[DESC]::bInitializeAudioPlayback:Dev-0x%x for Direction [%d]", u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir)));

   //TODO Check if there was a voice session.
   //spi_tclMySPINManager *poMySPINManager = spi_tclMySPINManager::getInstance();

   //TODO - handling of last mode for MySpin to be implemented. Set the last mode as native in datapool
   if ((e8AUD_MAIN_OUT == enAudDir) && (NULL != m_poAudioSettings))
   {
      t_Bool bRetValue = false;
      bRetValue = m_poAudioSettings->bWriteAudioLastMode(false);
      ETG_TRACE_USR2(("[DESC]::bInitializeAudioPlayback - Audio last mode is cleared in datapool: %d",
               ETG_ENUM(BOOL, bRetValue)));
   }
   // Commenting for future use
      // //! Initialize ECNR
   // if (((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir)) && 
      // (NULL != m_rAudCallbacks.fvInitializeAudioIn) && (NULL != m_rAudCallbacks.fvSetAlsaDevice))
   // {
      // (m_rAudCallbacks.fvInitializeAudioIn)(enAudDir, enGetAudioInDataSet(enAudDir, enSamplingRate));
      // (m_rAudCallbacks.fvSetAlsaDevice)(enAudDir);
   // }


   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir)
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir)
{
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);

   ETG_TRACE_USR1(("spi_tclDiPoAudio::bFinalizeAudioPlayback entered "));

   // Commenting for future use
   // //! Uninitialize ECNR
   // if (((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir))
         // && (NULL != m_rAudCallbacks.fvUninitializeAudioIn))
   // {
      // (m_rAudCallbacks.fvUninitializeAudioIn)(enAudDir);
   // }

   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::bStartAudio()
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bStartAudio(t_U32 u32DeviceId, t_String szAudioDev, tenAudioDir enAudioDir)
{
   ETG_TRACE_USR2(("[DESC]::Start Audio for mySPIN for : Dev-0x%x, Audio Dir [%d],Device [%s]", u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudioDir), szAudioDev.c_str()));

   //Callback to handle cases for Audio
   tvBluetoothConStat fvBTConStatus = std::bind(&spi_tclMySPINAudio::vBTDisconnectedCB, this, std::placeholders::_1);

   t_Bool bIsAudioStarted = false;
   tenAudioStreamError enAudioStreamError = e8NOERROR;
   // This routine is triggered when ever we receive OnStartSourceActivity
   // Note: Save the Audio device name and Audio Direction which is required to start the
   //       Audio pipe.
   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

   if (NULL != poMySPINMngr)
   {
      spi_tclMySPINCmdAudio* poCmdAudio = poMySPINMngr->poGetAudioInstance();

      if ((NULL != poCmdAudio) && (false == bIsAudioSrcActive(enAudioDir)))
      {
         ETG_TRACE_USR1(("spi_tclMySPINAudio::bStartAudio:No active source"));
         tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
         spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

         bIsAudioStarted = poCmdAudio->bInitialize(u32DeviceId, enDeviceSubCategory, enAudioDir, fvBTConStatus);

         if (true == bIsAudioStarted)
         {
            // Currently Myspin VR is handled by BT. Commenting for future use
            // if ((e8AUD_VR_IN == enAudioDir) && (NULL != m_rAudCallbacks.fvStartAudioIn))
            // {
               // (m_rAudCallbacks.fvStartAudioIn)(enAudioDir);
               // vSendVoiceSessionStatus(e8BTVOICESESSION_STATUS_REQUEST_ACCEPTED);
            // }

            enAudioStreamError = poCmdAudio->enStartAudioStreaming(enAudioDir, szAudioDev);
            ETG_TRACE_USR2(("[DESC]::Start Audio Stream [%d]", ETG_ENUM(MYSPIN_AUDIO_STREAM_ERROR, enAudioStreamError)));

            vSetAudioSrcState(enAudioDir, (e8NOERROR == enAudioStreamError));
            if (e8AUD_STEREO_MIX_OUT == enAudioDir)
            {
                vAudioDucking(e8_DUCKINGTYPE_DUCK);
            }//if(e8AUD_STEREO_MIX_OUT == enAudioDir)
                
            // if ((e8AUD_VR_IN == enAudioDir) && (e8NOERROR == enAudioStreamError))
            // {
               // vSendVoiceSessionStatus(e8BTVOICESESSION_STATUS_ACTIVE);
            // }
         }
         if((false == bIsAudioStarted || e8NOERROR != enAudioStreamError)/* && (e8AUD_VR_IN != enAudioDir)*/)
         {
             //some error occurred. respond to phone request with close
             poCmdAudio->bSendAudioCmdResponse(enAudioDir,false,e8_REQUEST_INVALID,e8SPI_AUDIO_UNKNOWN);
         }
      }
      // Invoke the response to Audio
      if ((NULL != m_rAudCallbacks.fvStartAudioResp) && (e8NOERROR == enAudioStreamError))
      {
         (m_rAudCallbacks.fvStartAudioResp)(enAudioDir, bIsAudioStarted);
      }

      if ((NULL != m_rAudCallbacks.fvTerminateAudioReq) && (e8NOERROR != enAudioStreamError)) //if audio streaming couldn't be started, deallocate the audio channel
      {
         // if (e8AUD_VR_IN == enAudioDir)
         // {
             // vSendVoiceSessionStatus(e8BTVOICESESSION_STATUS_REQUEST_DENIED);
             // (m_rAudCallbacks.fvTerminateAudioReq)(u32DeviceId, enAudioDir);
             // vSendVoiceSessionStatus(e8BTVOICESESSION_STATUS_IDLE);
         // }
         // else
         // {
             (m_rAudCallbacks.fvTerminateAudioReq)(u32DeviceId, enAudioDir, e8REASON_NOMEDIA);
         // }
      }
   }
   return bIsAudioStarted;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::bStartAudio()
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
        t_String szInputAudioDev, tenAudioDir enAudDir)
{
    ETG_TRACE_USR2(("spi_tclMySPINAudio::bStartAudio u32DeviceId = %d",u32DeviceId));
    ETG_TRACE_USR2(("spi_tclMySPINAudio::bStartAudio szOutputAudioDev = %s",szOutputAudioDev.c_str()));
    ETG_TRACE_USR2(("spi_tclMySPINAudio::bStartAudio szInputAudioDev = %s",szInputAudioDev.c_str()));
    ETG_TRACE_USR2(("spi_tclMySPINAudio::bStartAudio enAudDir = %d",ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir)));
    t_Bool bIsAudioStarted = false;
    bIsAudioStarted = bStartAudio(u32DeviceId,szInputAudioDev,enAudDir);
    return bIsAudioStarted;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudio::vStopAudio()
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused)
{
   ETG_TRACE_USR2(("[FUNC]spi_tclMySPINAudio::vStopAudio: Stop Audio Stream for Device [%d] for Audio Dir [%d]", u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir)));
   // This routine is triggered when ever we receive OnSopSourceActivity
   // Note: Save the Audio device name and Audio Direction which is required to start the
   //       Audio pipe.
   SPI_INTENTIONALLY_UNUSED(bIsPaused);

   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

   if (NULL != poMySPINMngr)
   {
      spi_tclMySPINCmdAudio* poCmdAudio = poMySPINMngr->poGetAudioInstance();

      if ((NULL != poCmdAudio) && (true == bIsAudioSrcActive(enAudDir)))
      {
         tenAudioStreamError enAudioStreamError = poCmdAudio->enStopAudioStreaming(enAudDir);
         ETG_TRACE_USR2(("[DESC]::Stop Audio Stream [%d]", ETG_ENUM(MYSPIN_AUDIO_STREAM_ERROR, enAudioStreamError)));
         
         // if ((e8AUD_VR_IN == enAudDir) && (NULL != m_rAudCallbacks.fvStopAudioIn))
         // {
            // (m_rAudCallbacks.fvStopAudioIn)(enAudDir);
         // }       
         
         if (e8AUD_STEREO_MIX_OUT == enAudDir)
         {
             vAudioDucking(e8_DUCKINGTYPE_UNDUCK);
         }//if(NULL != m_rAudCallbacks.fbSetAudioDucking)

         tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
         spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);
         poCmdAudio->vUnInitialize(enDeviceSubCategory, enAudDir);

         vSetAudioSrcState(enAudDir, false);
      }

   }
   // Invoke the response to Audio
   if (NULL != m_rAudCallbacks.fvStopAudioResp)
   {
      (m_rAudCallbacks.fvStopAudioResp)(enAudDir, true);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINAudio::enGetAudioType()
 ***************************************************************************/
tenAudioTransType spi_tclMySPINAudio::enGetAudioTransportType()
{
   //Check the device sub category and inform which audio type would be used.
   tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
   spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

   tenAudioTransType enAudioTransType = e8AUDIOTRANSPORT_UNKNWON;
   enAudioTransType = (e8DEVTYPE_AOAP == enDeviceSubCategory) ? (e8AUDIOTRANSPORT_BT) : (e8AUDIOTRANSPORT_USB);

   ETG_TRACE_USR4(("[PARAM]::enGetAudioTransportType-Audio transport type [%d]", enAudioTransType));

   return enAudioTransType;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINAudio::bIsAudioSrcActive()
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bIsAudioSrcActive(tenAudioDir enAudDir)
{
   trAudioState rAudioState;
   std::map<tenAudioDir, trAudioState>::iterator itrAudioDir;
   m_oActiveAudioDirLock.s16Lock();
   itrAudioDir = m_mapActiveAudioDir.find(enAudDir);

   if (m_mapActiveAudioDir.end() != itrAudioDir)
   {
      rAudioState = itrAudioDir->second;
   }
   m_oActiveAudioDirLock.vUnlock();
   ETG_TRACE_USR4(("[PARAM]::bIsAudioSrcActive-Source Status for Dir [%d] is [%d]", ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir), ETG_ENUM(BOOL, rAudioState.isActive)));
   return rAudioState.isActive;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINAudio::vSetAudioSrcState()
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vSetAudioSrcState(tenAudioDir enAudDir, t_Bool bIsAudioActive)
{
   ETG_TRACE_USR1(("[DESC]::Set Source Status for Dir [%d] as [%d]", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir), ETG_ENUM(BOOL,
            bIsAudioActive)));

   if(true == bIsAudioActive)
   {
       spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
       if (NULL != poMySPINMngr)
       {
           spi_tclMySPINCmdAudio * poCmdAudio = poMySPINMngr->poGetAudioInstance();
           if (NULL != poCmdAudio)
           {
              // Currently Myspin VR is handled by BT. Commenting for future use
              if((true == poCmdAudio->bSendAudioCmdResponse(enAudDir,bIsAudioActive,e8_REQUEST_INVALID,e8SPI_AUDIO_UNKNOWN)) /*|| (e8AUD_VR_IN == enAudDir)*/)
              {
                  //If atleast one phone request was responded, then only set to true.
                  /*We should set isActive to true only when atleast one of the phone request was served.
                   * Consider a case when mySPIN audio is being played and user plugs in USB and then unplugs it again.
                   * We will get SrcActivity ON from audio, but we don't have any request from the phone with us now.*/
                  m_oActiveAudioDirLock.s16Lock();
                  m_mapActiveAudioDir[enAudDir].isActive = bIsAudioActive;
                  m_oActiveAudioDirLock.vUnlock();
              }
           }
       }
   }
   else
   { 
       spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
       if(NULL != poMySPINMngr)
       {
            spi_tclMySPINCmdAudio *poCmdAudio = poMySPINMngr->poGetAudioInstance();
            if((NULL != poCmdAudio) && (e8AUD_MAIN_OUT != enAudDir))
            {
                poCmdAudio->bSendAudioCmdResponse(enAudDir,bIsAudioActive,e8_REQUEST_INVALID,e8SPI_AUDIO_UNKNOWN);
            }
            m_oActiveAudioDirLock.s16Lock();
            m_mapActiveAudioDir[enAudDir].isActive = bIsAudioActive;
            m_oActiveAudioDirLock.vUnlock();
       }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vSendVoiceSessionStatus(t_U32...
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vSendVoiceSessionStatus(tenBTVoiceSessionStatus enBTVoiceSessionStatus)
{
   spi_tclMySPINManager* poMySPINManager = spi_tclMySPINManager::getInstance();

   if (NULL != poMySPINManager)
   {
      spi_tclMySPINCmdSession* poMySPINCmdSession = poMySPINManager->poGetSessionInstance();

      if (NULL != poMySPINCmdSession)
      {
         poMySPINCmdSession->vUpdateVoiceSessionStatus(su32SelectedDevice, enBTVoiceSessionStatus);
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::bGetAudioCapability(..
 ***************************************************************************/
t_Bool spi_tclMySPINAudio::bGetAudioCapability(t_U32 u32DeviceId)
{
   t_Bool bAudioCapable = true;

   tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
   spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

   if (e8DEVTYPE_AOAP == enDeviceSubCategory)
   {
      spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

      if (NULL != poMySPINMngr)
      {
         spi_tclMySPINCmdAudio * poCmdAudio = poMySPINMngr->poGetAudioInstance();

         if (NULL != poCmdAudio)
         {
            bAudioCapable = poCmdAudio->bGetBTConnectionStatus(u32DeviceId);
         }
      }

      ETG_TRACE_USR4(("[PARAM]::bGetAudioCapability-mySPIN Android Devices Audio Capable [%d]", ETG_ENUM(BOOL,
               bAudioCapable)));
   }

   return bAudioCapable;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vBTDisconnectedCB(..
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vBTDisconnectedCB(t_U32 u32DeviceId)
{
   ETG_TRACE_USR2(("[DESC]::Bluetooth Disconnected for[%d]", u32DeviceId));

   m_oActiveAudioDirLock.s16Lock();
   for (std::map<tenAudioDir, trAudioState>::iterator itrActiveAud = m_mapActiveAudioDir.begin(); itrActiveAud
            != m_mapActiveAudioDir.end(); ++itrActiveAud)
   {
      trAudioState  rAudioState = itrActiveAud->second;
      if (true == rAudioState.isActive)
      {
         ETG_TRACE_USR2(("Main Audio Active"));
         spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
         if (NULL != poMySPINMngr)
         {
             spi_tclMySPINCmdAudio * poCmdAudio = poMySPINMngr->poGetAudioInstance();
             if ((NULL != poCmdAudio) /*&& (e8AUD_VR_IN != itrActiveAud->first) */)
             {
                 poCmdAudio->bSendAudioCmdResponse(e8AUD_INVALID,false,e8_REQUEST_CLOSE,e8SPI_AUDIO_UNKNOWN);
                 if(e8AUD_STEREO_MIX_OUT == itrActiveAud->first)
                 {
                     vAudioDucking(e8_DUCKINGTYPE_UNDUCK);
                 }
             }
         }
         //! Since main audio can not be active terminate the channel
         if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
         {
            ETG_TRACE_USR1(("spi_tclMySPINAudio::vBTDisconnectedCB:Terminate Audio Now"));
            (m_rAudCallbacks.fvTerminateAudioReq)(u32DeviceId, itrActiveAud->first, e8REASON_NOMEDIA);

         }
      }
   }
   m_oActiveAudioDirLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vRequestTerminateAudioCmd()
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vRequestTerminateAudioCmd(trMySPINAudioCmd rMySPINAudioCmd)
{
   ETG_TRACE_USR2(("spi_tclMySPINAudio::vRequestTerminateAudioCmd entered"));
   if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
   {
       m_rAudCallbacks.fvTerminateAudioReq(rMySPINAudioCmd.u32DeviceHandle, rMySPINAudioCmd.enAudioDir, e8REASON_NOMEDIA);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vRequestLaunchAudioCmd()
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vRequestLaunchAudioCmd(trMySPINAudioCmd rMySPINAudioCmd)
{
   ETG_TRACE_USR2(("spi_tclMySPINAudio::vRequestLaunchAudioCmd entered"));

   tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
   spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);
   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
   if (NULL != poMySPINMngr)
   {
       spi_tclMySPINCmdAudio * poCmdAudio = poMySPINMngr->poGetAudioInstance();
       if (NULL != poCmdAudio)
       {
           t_Bool bIsA2DPConnected = false;
           if(e8DEVTYPE_AOAP == enDeviceSubCategory)
           {
               bIsA2DPConnected = poCmdAudio->bGetBTConnectionStatus(rMySPINAudioCmd.u32DeviceHandle);
           }
           if((e8DEVTYPE_AOAP == enDeviceSubCategory && true == bIsA2DPConnected) ||
              (e8DEVTYPE_IAP == enDeviceSubCategory) )
           {
               if (NULL != m_rAudCallbacks.fvLaunchAudioReq)
               {
                  trAudSampleRate rAudioSampleRate;
				  rAudioSampleRate.enSampleRate = e8AUD_SAMPLERATE_DEFAULT;
                   m_rAudCallbacks.fvLaunchAudioReq(rMySPINAudioCmd.u32DeviceHandle,
                         e8DEV_TYPE_MYSPIN,
                         rMySPINAudioCmd.enAudioDir,
                         rAudioSampleRate);
                         
               }
           }
           else if((e8DEVTYPE_AOAP == enDeviceSubCategory) && (false == bIsA2DPConnected))
           {
               //We shall respond to phone request so that music can be played through phone speakers
               if(e8AUD_MAIN_OUT == rMySPINAudioCmd.enAudioDir)
               {
                    poCmdAudio->bSendAudioCmdResponse(e8AUD_INVALID,false,e8_REQUEST_OPEN,e8SPI_AUDIO_UNKNOWN);
               }
               else if(e8AUD_STEREO_MIX_OUT == rMySPINAudioCmd.enAudioDir)
               {
                    poCmdAudio->bSendAudioCmdResponse(e8AUD_INVALID,false,e8_REQUEST_REJECT,e8SPI_AUDIO_UNKNOWN);
               }
           }
       }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vVRStreamErrorCb(t_U32...
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vVRStreamErrorCb(t_U32 u32DevHandle, tenMySPINVRStreamError enError)
{
    ETG_TRACE_USR2(("spi_tclMySPINAudio::vVRStreamErrorCb entered u32DevHandle=%d, enError=%d",u32DevHandle,enError));
    SPI_INTENTIONALLY_UNUSED(enError)
    if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
    {
        //close the VR session due to error condition
        (m_rAudCallbacks.fvTerminateAudioReq)(u32DevHandle, e8AUD_VR_IN, e8REASON_NOMEDIA);
    }
}

// /***************************************************************************
 // ** FUNCTION:  spi_tclMySPINAudio::enGetAudioDataSet(...)
 // ***************************************************************************/
// tenAudioInDataSet spi_tclMySPINAudio::enGetAudioInDataSet(tenAudioDir enAudioDir,
      // tenAudioSamplingRate enAudioSamplingRate)
// {
   // tenAudioInDataSet enAudioDataSet = e32_AUDIOIN_DATASET_UNKNOWN;
   // switch (enAudioDir)
   // {
      // case e8AUD_PHONE_IN:
         // enAudioDataSet = (e8AUD_SAMPLERATE_8KHZ == enAudioSamplingRate) ?
            // (e32_AUDIOIN_DATASET_PHONE_NB) : (e32_AUDIOIN_DATASET_PHONE_WB);
         // break;
      // case e8AUD_VR_IN:
         // enAudioDataSet = (e8AUD_SAMPLERATE_8KHZ == enAudioSamplingRate) ?
            // (e32_AUDIOIN_DATASET_VR_NB): (e32_AUDIOIN_DATASET_VR_WB);
         // break;
      // case e8AUD_DEFAULT:
         // //! Default audio type is without ECNR
         // enAudioDataSet = e32_AUDIOIN_DATASET_VR_MIC_ONLY;
         // break;
      // default:
         // ETG_TRACE_ERR(("Invalid audio direction type %d ", enAudioDir));
         // break;
   // }//switch (enAudioDir)

   // return enAudioDataSet;
// }//! end of spi_tclDiPoAudioResourceMngr::enGetAudioInDataSet()

//This function is retained for future use
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vAudioDucking(t_U32...
 ***************************************************************************/
t_Void spi_tclMySPINAudio::vAudioDucking(tenDuckingType enDuckingType)
{
    ETG_TRACE_USR2(("spi_tclMySPINAudio::vAudioDucking entered with value [%d]", enDuckingType));
    if ((NULL != m_rAudCallbacks.fbSetAudioDucking) && (m_enDuckingType != enDuckingType))
    {
        if(e8_DUCKINGTYPE_DUCK == enDuckingType)
        {
            m_rAudCallbacks.fbSetAudioDucking(scou16AudioRampDefDuration, scou8AudioDuckDefdB,
                e8_DUCKINGTYPE_DUCK, e8DEV_TYPE_MYSPIN);
        }
        else
        {
            m_rAudCallbacks.fbSetAudioDucking(scou16AudioRampDefDuration, scou8AudioUnduck,
                e8_DUCKINGTYPE_UNDUCK, e8DEV_TYPE_MYSPIN);
        }
        m_enDuckingType = enDuckingType;
    }//if(NULL != m_rAudCallbacks.fbSetAudioDucking)
}

//lint -restore

///////////////////////////////////////////////////EOF///////////////////////////////////////////////////////////////
