/*!
 *******************************************************************************
 * \file             spi_tclMySPINAudio.h
 * \brief
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for MySpin
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                | Modification
 02.11.2015  | tch5kor               | Initial Version
 05.04.2016  | Chaitra Srinivasa     | Adding comments
 \endverbatim

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLMYSPINAUDIO_H
#define SPI_TCLMYSPINAUDIO_H

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "Lock.h"
#include "spi_tclAudioDevBase.h"
#include "mySPINTypes.h"
#include "spi_tclMySPINRespAudio.h"
/**
 * This class implements the Audio for MySpin
 */

class spi_tclMySPINAudio: public spi_tclAudioDevBase, public spi_tclMySPINRespAudio
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINAudio::spi_tclMySPINAudio();
       ***************************************************************************/
      /*!
       * \fn      spi_tclMySPINAudio()
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclMySPINAudio();

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINAudio::~spi_tclMySPINAudio();
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclMySPINAudio()
       * \brief   Virtual Destructor
       **************************************************************************/
      virtual ~spi_tclMySPINAudio();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINAudio::bInitialise();
       ***************************************************************************/
      /*!
       * \fn      bInitialise()
       * \brief   Method to initialise the service handler. (Performs initialisations which
       *          are not device specific.)
       *          Optional interface to be implemented.
       * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
       * \sa      bUninitialise()
       ***************************************************************************/
      t_Bool bInitialise() override;

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINAudio::vRegisterCallbacks();
       ***************************************************************************/
      /*!
       * \fn      vRegisterCallbacks()
       * \brief   Register callbacks for audio
       * \param   [rfrAudCallbacks]: Used to hold the callbacks to be registered by the
       *                          creator of the class object
       * \retval t_Void
       **************************************************************************/
      t_Void vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINAudio::vSelectDevice()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Void vSelectDevice
       * \brief   To update the changes in the audio of the selected device
       * \param    cou32DevId : [IN] Unique Device Id
       * \param   coenConnReq : [IN] connected/disconnected
       * \retval  t_Void
       **************************************************************************/
      t_Void vSelectDevice(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINAudio::vStopAudio()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Void vStopAudio
       * \brief   To stop the audio that is currently playing
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       * \param   [enAudDir]   : Specify the Audio Direction.
       * \param   [bIsPaused]  : Indicates if source is paused or stopped
       * \retval  t_Void
       **************************************************************************/
      t_Void vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINAudio::bSelectAudioDevice()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bSelectAudioDevice
       * \brief   To select the audio device
       * \param   u32DeviceId : [IN] Unique Device Id
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSelectAudioDevice(t_U32 u32DeviceId);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINAudio::vDeselectAudioDevice()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool vDeselectAudioDevice
       * \brief   To deselect the audio device
       * \param   u32DeviceId : [IN] Unique Device Id
       * \retval  t_Void
       **************************************************************************/
      t_Void vDeselectAudioDevice(t_U32 u32DeviceId);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINAudio::bIsAudioLinkSupported()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bIsAudioLinkSupported
       * \brief   To deselect the audio device
       * \param   u32DeviceId : [IN] Unique Device Id
       * \param   enLink : [IN] used to identify the audio streaming link
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bIsAudioLinkSupported(t_U32 u32DeviceId, tenAudioLink enLink)
      {
          return true;
      }

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINAudio::vOnAudioError()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Void vOnAudioError
       * \brief   To deselect the audio device
       * \param   enAudDir : [IN] Used to identify the audio stream direction
       * \param   enAudioError : [IN] used to define the audio errors
       * \retval  t_Void
       **************************************************************************/
      t_Void vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError)
      {
      }

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINAudio::bStartAudio()
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bStartAudio
       * \brief   To Start audio streaming
       * \param    cou32DevId : [IN] Unique Device Id
       * \param   coenConnReq : [IN] connected/disconnected
       * \retval  t_Void
       **************************************************************************/
      t_Bool bStartAudio(t_U32 u32DeviceId, t_String szAudioDev, tenAudioDir enAudioDir);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINAudio::bStartAudio(t_U32,t_String, t_String,
       **                   tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
       *          t_String szInputAudioDev, tenAudioDir enAudDir)
       * \brief   Overloaded method to handle audio stream for Phone and VR.
       *          Start Streaming of Audio from the CE Device to the Audio Output
       *          Device assigned by the Audio Manager for the Source.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       *          [szOutputAudioDev]: Output ALSA Audio Device
       *          [szInputAudioDev] : Input ALSA Audio Device
       *          [enAudDir]        : Specify the Audio Direction(Phone or VR Audio).
       * \retval  Bool value
       **************************************************************************/
      t_Bool bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
              t_String szInputAudioDev, tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINAudio::bInitializeAudioPlayback(t_U32)
       ***************************************************************************/
      /*!
       * \fn      bInitializeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
       * \brief   Perform necessary actions to prepare for an Audio Playback.
       *          Function will be called prior to a Play Command from Audio Manager.
       *          Optional Interface to be implemented by Device Class.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device
       * \param   [enAudDir]: Audio route being allocated
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bInitializeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir,
            tenAudioSamplingRate enSamplingRate,tenAudioSamplingRate enNativeSamplingRate);

      /***************************************************************************
       ** FUNCTION: tenAudioTransType spi_tclMySPINAudio::enGetAudioTransportType()
       ***************************************************************************/
      /*!
       * \fn     tenAudioTransType enGetAudioTransportType()
       * \brief  Interface to get Audio transport type
       * \param  None
       * \retval  tenAudioTransType
       **************************************************************************/
      tenAudioTransType enGetAudioTransportType();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINAudio::vSendVoiceSessionStatus(tenBTVoi...
       ***************************************************************************/
      /*!
       * \fn     vSendVoiceSessionStatus(tenBTVoiceSessionStatus enBTVoiceSessionStatus)
       * \brief  Used to send the voice session status
       * \param  [enBTVoiceSessionStatus]: Used to store the status of a
       *                                  voice session via BT
       * \retval t_Void
       **************************************************************************/
      t_Void vSendVoiceSessionStatus(tenBTVoiceSessionStatus enBTVoiceSessionStatus);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINAudio::bGetAudioCapability(...
       ***************************************************************************/
      /*!
       * \fn     bGetAudioCapability(t_U32 u32DeviceId)
       * \brief  Used to get the audio capabiltity
       * \param  [u32DeviceId]: Unique device Id
       * \retval t_Void
       **************************************************************************/
      t_Bool bGetAudioCapability(t_U32 u32DeviceId);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINAudio::vBTDisconnectedCB(...
       ***************************************************************************/
      /*!
       * \fn     vBTDisconnectedCB(t_U32 u32DeviceId)
       * \brief  Used to disconnect the BT connection
       * \param  [u32DeviceId]: Unique device Id
       * \retval t_Void
       **************************************************************************/
      t_Void vBTDisconnectedCB(t_U32 u32DeviceId);
      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINAudio::bFinalizeAudioPlayback(t_U32,tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      bFinalizeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
       * \brief   Perform necessary actions on completion of Audio Playback.
       *          Function will be called after to a Stop Command from Audio Manager.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
	   * \param   [enAudDir]: Audio route being deallocated
       * \retval  Bool value
       **************************************************************************/	  
      t_Bool bFinalizeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir);

     /***************************************************************************
      ** FUNCTION: t_Void spi_tclMySPINAudio::vRequestLaunchAudioCmd(...
      ***************************************************************************/
     /*!
      * \fn     vRequestLaunchAudioCmd(trMySPINAudioCmd rMySPINAudioCmd)
      * \brief  Method to handle audio commands from phone
      * \param  [rMySPINAudioCmd]: Structure containing audio command information
      * \retval t_Void
      **************************************************************************/
     t_Void vRequestLaunchAudioCmd(trMySPINAudioCmd rMySPINAudioCmd);

     /***************************************************************************
      ** FUNCTION: t_Void spi_tclMySPINAudio::vRequestTerminateAudioCmd(...
      ***************************************************************************/
     /*!
      * \fn     vRequestTerminateAudioCmd(trMySPINAudioCmd rMySPINAudioCmd)
      * \brief  Method to handle audio commands from phone
      * \param  [rMySPINAudioCmd]: Structure containing audio command information
      * \retval t_Void
      **************************************************************************/
     t_Void vRequestTerminateAudioCmd(trMySPINAudioCmd rMySPINAudioCmd);
     
     /***************************************************************************
      ** FUNCTION: t_Void spi_tclMySPINAudio::vAudioDucking(...
      ***************************************************************************/
     /*!
      * \fn     vAudioDucking(tenDuckingType enDuckingType)
      * \brief  Method to request audio ducking
      * \param  enDuckingType : Ducking type(duck/unduck)
      * \retval t_Void
      **************************************************************************/     
     t_Void vAudioDucking(tenDuckingType enDuckingType);     

   protected:
     /***************************************************************************
      ** FUNCTION: t_Void spi_tclMySPINAudio::vVRStreamErrorCb(...
      ***************************************************************************/
     /*!
      * \fn     vVRStreamErrorCb(t_U32 u32DeviceId)
      * \brief  Callback for VR Stream error from phone
      * \param  [enError]: Error number
      * \retval t_Void
      **************************************************************************/
     t_Void vVRStreamErrorCb(t_U32 u32DevHandle, tenMySPINVRStreamError enError);

   private:

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclMySPINAudio::bIsAudioSrcActive()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsAudioSrcActive()
       * \brief  Interface to get Audio transport type
       * \param  None
       * \param  enAudDir : Audio direction
       * \retval t_Bool
       **************************************************************************/
      t_Bool bIsAudioSrcActive(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINAudio::vSetAudioSrcState()
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetAudioSrcState()
       * \brief  Interface to get Audio transport type
       * \param  None
       * \param  enAudDir : Audio direction
	   * \param  bIsAudioActive : Parameter which says whether audio is active or not
       * \retval t_Void
       **************************************************************************/
      t_Void vSetAudioSrcState(tenAudioDir enAudDir, t_Bool bIsAudioActive);

      // Commenting for future use
      // /***************************************************************************
       // ** FUNCTION: tenAudioInDataSet spi_tclMySPINAudio::enGetAudioInDataSet()
       // ***************************************************************************/
      // /*!
       // * \fn     tenAudioInDataSet enGetAudioInDataSet()
       // * \brief  Interface to get Audio transport type
       // * \param  None
       // * \param  enAudioDir : Audio direction
	   // * \param  enAudioSamplingRate : Audio sampling rate
       // * \retval tenAudioInDataSet
       // **************************************************************************/	  
	  // tenAudioInDataSet enGetAudioInDataSet(tenAudioDir enAudioDir, tenAudioSamplingRate enAudioSamplingRate);

      //! Structure object for Function pointers .
      //! This will be used by Audio Manager to register for response callbacks from  Audio
      trAudioCallbacks m_rAudCallbacks;

      //Map to store currently active audio direction
      std::map<tenAudioDir, trAudioState> m_mapActiveAudioDir;

      Lock m_oActiveAudioDirLock;

      tenDuckingType m_enDuckingType;
};
#endif // SPI_TCLMYSPINAUDIO_H
