
/***********************************************************************/
/*!
* \file    spi_tclOnCarVideoResourceMngr.cpp
* \brief   OnCar Video Resource Manager
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    OnCar Video resource manager
AUTHOR:         Unmukt Jain
COPYRIGHT:      &copy; RBEI

HISTORY:
Date         | Author                | Modification
20.06.2018   | Unmukt Jain           | Initial Version

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "spi_tclResorceMngrDefines.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclOnCarCmdVideo.h"
#include "spi_tclOnCarResourceMngr.h"
#include "spi_tclOnCarVideoResourceMngr.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
      #include "trcGenProj/Header/spi_tclOnCarVideoResourceMngr.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static spi_tclOnCarManager* spoOnCarMngr = NULL;
static spi_tclOnCarCmdVideo* spoCmdVideo = NULL;

#define ONCAR_VIDEO_HU_REQ_CNTXT
static const trOnCarAccVideoFocusNoti sacoAccVideoFocusNoti[]=
#include "spi_tclOnCarVideoContext.cfg"
#undef ONCAR_VIDEO_HU_REQ_CNTXT

#define ONCAR_VIDEO_HNDL_HU_REQSTATE
static const trOnCarVideoFocusState  sacoVideoFocusState[]=
#include "spi_tclOnCarVideoContext.cfg"
#undef ONCAR_VIDEO_HNDL_HU_REQSTATE

#define ONCAR_VIDEO_MD_FOCUS_REQ
static const trOnCarMDVideoFocusReq  sacoMDVideoFocusReq[]=
#include "spi_tclOnCarVideoContext.cfg"
#undef ONCAR_VIDEO_MD_FOCUS_REQ

/***************************************************************************
** FUNCTION:  spi_tclOnCarVideoResourceMngr::spi_tclOnCarVideoResourceMngr()
***************************************************************************/
spi_tclOnCarVideoResourceMngr::spi_tclOnCarVideoResourceMngr(
   spi_tclOnCarResourceMngr* poOnCarRsrcMngr):m_cpoOnCarRsrcMngr(poOnCarRsrcMngr),
   m_enCurMDFocusState(e8_ONCAR_VIDEOFOCUSSTATE_LOSS),
   m_enGrantConstraint(e8DIPO_CONSTRAINT_NA),
   m_enPrevTransferType(e8DIPO_TRANSFERTYPE_NA),
   m_enCurAccDispCntxt(e8DISPLAY_CONTEXT_NATIVE),
   m_enPlaybackState(e8VID_PB_STATE_STOPPED),
   m_u32SelectedDeviceID(0),
   m_bInitialMDVideoFocus(false)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr() entered "));
   //nothing to do
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarVideoResourceMngr::~spi_tclAAPVideoResourceMngr()
***************************************************************************/
spi_tclOnCarVideoResourceMngr::~spi_tclOnCarVideoResourceMngr()
{
   ETG_TRACE_USR1(("~spi_tclOnCarVideoResourceMngr() entered"));

   spoOnCarMngr        = NULL;
   spoCmdVideo         = NULL;
   //nothing to do
}
/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bInitialize()
***************************************************************************/
t_Bool spi_tclOnCarVideoResourceMngr::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::bInitialize() entered"));

   t_Bool bRet = false;

   spoOnCarMngr = spi_tclOnCarManager::getInstance();
   SPI_NORMAL_ASSERT(NULL == spoOnCarMngr);

   if (NULL != spoOnCarMngr)
   {
      spoCmdVideo = spoOnCarMngr->poGetVideoInstance();
      bRet = (spoCmdVideo == NULL) ? false : true;
   }//if (NULL != spoAAPMngr)

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vUninitialize()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vUnInitialize()
{
   //Nothing to do
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayContext()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayContext(const t_U32 cou32DevId,
                                                                t_Bool bDisplayFlag,
                                                                tenDisplayContext enDisplayContext)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayContext:Device ID-0x%x Accessory Takes Focus-%d AccDisplayContext-%d"
      , cou32DevId,ETG_ENUM(BOOL,bDisplayFlag),ETG_ENUM(DISPLAY_CONTEXT,enDisplayContext)));

   // do not process the update, if the DisplayFlag is false for a context, which is not active currently.
   if(  ( true == bDisplayFlag ) ||
      ( (false == bDisplayFlag) && (m_enCurAccDispCntxt == enDisplayContext) )  )
   {
      if( true ==  bDisplayFlag )
      {
         m_enCurAccDispCntxt =  enDisplayContext;
      } //if( true ==  bDisplayFlag )
      else
      {
         m_enCurAccDispCntxt = e8DISPLAY_CONTEXT_NATIVE;
      }

      if(0 != m_u32SelectedDeviceID )
      {
          tenOnCarVideoFocusState enCurAccFocusState = e8_ONCAR_VIDEOFOCUSSTATE_UNKNOWN;
          tenOnCarVideoFocusState enNewMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_UNKNOWN;

         m_oCurMDFocusStateLock.s16Lock();
         if(  ( true == bGetAccVideoFocusState(enDisplayContext,bDisplayFlag,enCurAccFocusState) )  &&
            ( true == bGetUpdatedMDFocusState(m_enCurMDFocusState,enCurAccFocusState,enNewMDFocusState) )  )
         {
            switch(m_enCurMDFocusState)
            {
            case e8_ONCAR_VIDEOFOCUSSTATE_GAIN:
               {
                  //HMI does a TAKE or BORROW, when the MD has the screen, SetVideoFocus to Native
                  if( ( e8_ONCAR_VIDEOFOCUSSTATE_GAIN == enCurAccFocusState ) ||
                     (e8_ONCAR_VIDEOFOCUSSTATE_GAIN_TRANSIENT == enCurAccFocusState) )
                  {
                     vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_NATIVE);
                  }//if( (e8AAP_VIDEOFOCUSSTATE_GAIN == enCurAccFocusState )
               }//case e8AAP_VIDEOFOCUSSTATE_GAIN:
               break;
            case e8_ONCAR_VIDEOFOCUSSTATE_LOSS_TRANSIENT:
               {
                  //HMI does a UNBORROW, SetVideoFocus to PROJECTED on the Phone.
                  if(e8_ONCAR_VIDEOFOCUSSTATE_LOSS_TRANSIENT == enCurAccFocusState)
                  {
                     vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_PROJECTED);
                  }//if(e8AAP_VIDEOFOCUSSTATE_LOSS_TRANSIENT == enCurAccFocusState)
               }//case e8AAP_VIDEOFOCUSSTATE_LOSS_TRANSIENT:
               break;
            default:
               {
                  ETG_TRACE_ERR(("[ERR]:vSetAccessoryDisplayContext:default state"));
               }
            }//switch(m_enCurMDFocusState)

            //upadte the MD focus state
            m_enCurMDFocusState = enNewMDFocusState;
         }//if(  ( true == bGetAccVide
         else
         {
            ETG_TRACE_ERR(("[ERR]:vSetAccessoryDisplayContext: Element not found"));
         }
         m_oCurMDFocusStateLock.vUnlock();
      }//if(0 != m_u32SelectedDeviceID )

      //Populate TAKE Constraint, even when device is not active.
      tenOnCarMDVideoFocusReqResp enRespType = e8_ONCAR_VIDEOFOCUS_DENY;
      if( true == bGetAccRespType(enRespType) )
      {
          m_enGrantConstraint = ( e8_ONCAR_VIDEOFOCUS_GRANT == enRespType) ? e8DIPO_CONSTRAINT_ANYTIME : e8DIPO_CONSTRAINT_NEVER ;
      }

   } //if(  ( true == bDisplayFlag ) ||
   else
   {
      ETG_TRACE_USR4(("spi_tclAAPVideoResourceMngr::vSetAccessoryDisplayContext: Current active Context is %d ",
         ETG_ENUM(DISPLAY_CONTEXT,m_enCurAccDispCntxt)));
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bGetAccVideoFocusState()
***************************************************************************/
t_Bool spi_tclOnCarVideoResourceMngr::bGetAccVideoFocusState(
   tenDisplayContext enAccDispCntxt,
   t_Bool bDisplayFlag,
   tenOnCarVideoFocusState& rfenVideoFocusState)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::bGetAccVideoFocusState: AccDisplayContext-%d Accessory Takes Focus-%d",
      ETG_ENUM(DISPLAY_CONTEXT,enAccDispCntxt),ETG_ENUM(BOOL,bDisplayFlag)));

   t_Bool bRet=false;

   t_U32 u32ContSize = (sizeof(sacoAccVideoFocusNoti))/(sizeof(trOnCarAccVideoFocusNoti));
   for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)
   {
      // Check for the matching Accessory context entry
      if(enAccDispCntxt == sacoAccVideoFocusNoti[u8Index].enAccDispCntxt)
      {
         bRet = true;

         rfenVideoFocusState = (true == bDisplayFlag)? (sacoAccVideoFocusNoti[u8Index].enAccFocusReqType):
            (sacoAccVideoFocusNoti[u8Index].enAccFocusRelType);

         ETG_TRACE_USR4(("[DESC]:Derived Accessory Focus State based on Accessory display context-%d",
            ETG_ENUM(VIDEOFOCUS_STATE,rfenVideoFocusState)));

         // stop the process once search hits at the desired value
         break;
      } //if(enAccDispCntxt == sacoAccVideoFocusNoti[u8Index].enAccDispCntxt)
   } //for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bGetUpdatedMDFocusState()
***************************************************************************/
t_Bool spi_tclOnCarVideoResourceMngr::bGetUpdatedMDFocusState(tenOnCarVideoFocusState enCurMDFocusState,
                                                              tenOnCarVideoFocusState enCurAccFocusState,
                                                              tenOnCarVideoFocusState& rfenUpdatedMDFocusState)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::bGetUpdatedMDFocusState:CurMDFocusState-%d" \
      "Derived Accessory Focus State-%d ",ETG_ENUM(VIDEOFOCUS_STATE,enCurMDFocusState),
      ETG_ENUM(VIDEOFOCUS_STATE,enCurAccFocusState)));

   t_Bool bRet=false;

   t_U32 u32ContSize = (sizeof(sacoVideoFocusState))/(sizeof(trOnCarVideoFocusState));
   for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)
   {
      if( (enCurMDFocusState == sacoVideoFocusState[u8Index].enCurMDFocusState) &&
         (enCurAccFocusState == sacoVideoFocusState[u8Index].enReqFocusState) )
      {
         bRet = true;

         rfenUpdatedMDFocusState = sacoVideoFocusState[u8Index].enUpdatedMDFocusState;

         ETG_TRACE_USR4(("[PARAM]:bGetUpdatedMDFocusState:ResultedMDFocusState-%d"
            ,ETG_ENUM(VIDEOFOCUS_STATE,rfenUpdatedMDFocusState)));

         // stop the process once search hits at the desired value
         break;
      } //if( (enCurMDFocusState == sacoVideoFocusState[u8Index]
   } //for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bGetAccRespType()
***************************************************************************/
t_Bool spi_tclOnCarVideoResourceMngr::bGetAccRespType(tenOnCarMDVideoFocusReqResp& rfenRespType)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::bGetAccRespType:Current AccDispCntxt-%d Current MDFocusState-%d" ,
      ETG_ENUM(DISPLAY_CONTEXT,m_enCurAccDispCntxt),ETG_ENUM(VIDEOFOCUS_STATE,m_enCurMDFocusState)));

   t_Bool bRet=false;

   t_U32 u32ContSize = (sizeof(sacoMDVideoFocusReq))/(sizeof(trOnCarMDVideoFocusReq));
   for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)
   {
      if(m_enCurAccDispCntxt == sacoMDVideoFocusReq[u8Index].enAccDispCntxt)
      {
         bRet = true;

         m_oCurMDFocusStateLock.s16Lock();
         rfenRespType = (e8_ONCAR_VIDEOFOCUSSTATE_LOSS == m_enCurMDFocusState) ?
            sacoMDVideoFocusReq[u8Index].enMDFocusReqResp_StateLoss:
            sacoMDVideoFocusReq[u8Index].enMDFocusReqResp_StateLossTransient;
         m_oCurMDFocusStateLock.vUnlock();

         ETG_TRACE_USR4(("[PARAM]:bGetAccRespType:Accesspry response for MD Video focus request-%d"
            ,ETG_ENUM(VIDEOFOCUS_RESP_TYPE,rfenRespType)));

         // stop the process once search hits at the desired value
         break;
      } //if( (enCurMDFocusState == sacoMDVideoFocusReq[u8Index]
   } //for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vSetVideoFocus()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vSetVideoFocus(tenOnCarVideoFocus enVideoFocus)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vSetVideoFocus:VideoFocusMode-%d",
      ETG_ENUM(VIDEOFOCUS_MODE,enVideoFocus)));

   //PROJECTED && STOPPED => SetVideoFocus to PROJECTED & State to PB_START_RQSTD
   //NATIVE && STARTED => SetVideoFocus to NATIVE & State to PB_STOP_RQSTD

   t_Bool bSendVideoFocusReq = false;

   m_oVideoPlaybackStateLock.s16Lock();

   ETG_TRACE_USR4(("[PARAM]:vSetVideoFocus:Current Video Playback state - %d",ETG_ENUM(AAP_VID_PB_STATE,m_enPlaybackState)));

   /*
   Request is to set the Video Focus with PROJECTED and the play back state is STOPPED, Then send the request to phone
   and set the state to request is in progress.
   Request is received when the Playback state
   is STOP in progress => delay processing of the  request. That will be processed, when the PlaybackStopCb comes
                          There processing happens, based on the current MD Focus state and it is set
                          upon receiving the SetAccessoryDisplay context update & Launch App requests.
   is STARTED or START in progress => no need to process the update. ignore it.
   */
   if((e8_ONCAR_VIDEOFOCUS_PROJECTED == enVideoFocus) && (e8VID_PB_STATE_STOPPED == m_enPlaybackState))
   {
      bSendVideoFocusReq = true;
      m_enPlaybackState = e8VID_PB_STATE_START_RQSTD;
   }
   /*
   Request is to set the Video Focus with NATIVE and the play back state is STARTED. Then send the request to phone
   and set the state to request is in progress.
   Request is received when the Playback state
   is START is in progress => delay processing of the request. That will be processed, when the PlaybackStartCb comes.
                              There processing happens, based on the current MD Focus state and it is set
                              upon receiving the SetAccessoryDisplay context update & Launch App requests.
   is STOPPED or STOP in progress => no need to process the update. ignore it.
   */
   else if((e8_ONCAR_VIDEOFOCUS_NATIVE == enVideoFocus)&&(e8VID_PB_STATE_STARTED == m_enPlaybackState))
   {
      bSendVideoFocusReq = true;
      m_enPlaybackState = e8VID_PB_STATE_STOP_RQSTD;
   }//else if((e8VIDEOFOCUS_NATIVE == enVideoFocus)&&(
   else
   {
      ETG_TRACE_USR4(("[DESC]:Ignore the request. Some other request is still being processed (or) no need to send request"));
   }
   m_oVideoPlaybackStateLock.vUnlock();

   if((true == bSendVideoFocusReq)&&(NULL != spoCmdVideo))
   {
      spoCmdVideo->vSetVideoFocus(enVideoFocus);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vPostOnCarStartVideoPlaybackMsg ()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vPostOnCarStartVideoPlaybackMsg()
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vPostOnCarStartVideoPlaybackMsg"));
   m_oVideoPlaybackStateLock.s16Lock();
   m_enPlaybackState = e8VID_PB_STATE_STARTED;
   m_oVideoPlaybackStateLock.vUnlock();

   m_oCurMDFocusStateLock.s16Lock();
   if( (e8_ONCAR_VIDEOFOCUSSTATE_LOSS == m_enCurMDFocusState) || (e8_ONCAR_VIDEOFOCUSSTATE_LOSS_TRANSIENT == m_enCurMDFocusState) )
   {
      //MD was in GAIN state, when the SetVideoFocus with PROJECTED is requested.
      //Due to some user interaction,MD's state changed to LOSS and this has happened, when the last request is being
      //processed. So request the Phone to release the Video focus.
      ETG_TRACE_USR2(("[DESC]:Accessory has taken Video Focus due to user interactions. Release the Video Focus"));
      vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_NATIVE);
   }
   //Enable Projection screen
   else if(NULL!= spoCmdVideo && NULL != m_cpoOnCarRsrcMngr)
   {
       spoCmdVideo->acknowledgeStartVideoRequest();
       m_cpoOnCarRsrcMngr->vUpdateDeviceDisplayCntxt(true,tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
   }//else if(NULL != m_poAAPRsrcMngr)

   m_oCurMDFocusStateLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vPostOnCarStopVideoPlaybackMsg()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vPostOnCarStopVideoPlaybackMsg()
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vPostOnCarStopVideoPlaybackMsg"));

   m_oVideoPlaybackStateLock.s16Lock();
   m_enPlaybackState = e8VID_PB_STATE_STOPPED;
   m_oVideoPlaybackStateLock.vUnlock();

   m_oCurMDFocusStateLock.s16Lock();
   if( (e8_ONCAR_VIDEOFOCUSSTATE_GAIN == m_enCurMDFocusState) || (e8_ONCAR_VIDEOFOCUSSTATE_GAIN_TRANSIENT == m_enCurMDFocusState) )
   {
      //MD was in LOSS/LOSS_TRANSIENT state, when the SetVideoFocus with NATIVE request is sent.
      //Due to user interactions, HU released the Video Focus, before the response is received for the last request.
      //In this case, request Phone to take the Video Focus.
      ETG_TRACE_USR2(("[DESC]:Accessory has released the Video Focus due to user interactions. Request Phone for projection"));
      vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_PROJECTED);
   }
   //Disable Projection screen
   else if(NULL != spoCmdVideo && NULL != m_cpoOnCarRsrcMngr)
   {
       spoCmdVideo->acknowledgeStopVideoRequest();
       m_cpoOnCarRsrcMngr->vUpdateDeviceDisplayCntxt(false,tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
   }//else if(NULL != m_poAAPRsrcMngr)

   m_oCurMDFocusStateLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vVideoFocusCallback()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vPostOnCarRequestVideoFocusMsg(const tenOnCarVideoFocus enOnCarVideoFocus,
                                                          const tenOnCarVideoFocusReason enOnCarVideoFocusReason)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vPostOnCarRequestVideoFocusMsg: VideoFocusMode-%d VideoFocusReason-%d",
      ETG_ENUM(VIDEOFOCUS_MODE,enOnCarVideoFocus),enOnCarVideoFocusReason));

   tenDisplayContextReason enDisplayContextReason;

   switch(enOnCarVideoFocusReason)
   {
      case e8_ONCAR_VIDEOFOCUS_REASON_SCREEN_OFF:
      {
         enDisplayContextReason = tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_PHONE_SCREEN_OFF;
      }
      break;
      case e8_ONCAR_VIDEOFOCUS_REASON_OEM_ICON_PRESSED:
      {
         enDisplayContextReason = tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_HOME_SCREEN;
      }
      break;
      case e8_ONCAR_VIDEOFOCUS_REASON_SHOW_NATIVE_BT_SCREEN:
      {
    	 enDisplayContextReason = tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_SHOW_NATIVE_BT_SCREEN;
      }
      break;
      default:
      {
         enDisplayContextReason = tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN;
      }
      break;
   }

   ETG_TRACE_USR2(("[PARAM] spi_tclOnCarVideoResourceMngr:: DisplayContextReason set to-%d ",ETG_ENUM(DISPLAY_CONTEXT_REASON,enDisplayContextReason)));

   if(enOnCarVideoFocus == e8_ONCAR_VIDEOFOCUS_NATIVE)
   {
       m_oCurMDFocusStateLock.s16Lock();
       m_enCurMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_LOSS;
       m_oCurMDFocusStateLock.vUnlock();

      //Switch to Native UI, whenever the MD requests
      vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_NATIVE);
      //Switch to Native UI
      if(NULL != m_cpoOnCarRsrcMngr)
      {
          m_cpoOnCarRsrcMngr->vUpdateDeviceDisplayCntxt(false,enDisplayContextReason);
      }//if(NULL != m_poAAPRsrcMngr)
   }//if(enVideoFocusMode == e8VIDEOFOCUS_NATIVE)
   else
   {
      ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vVideoFocusCallback: Constraint = %d",
              m_enGrantConstraint));
      if((e8DIPO_CONSTRAINT_NEVER == m_enGrantConstraint) || 
         (e8DIPO_CONSTRAINT_NA == m_enGrantConstraint))
      {
         vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_NATIVE);
         //Already Native UI is enabled and Phone is requesting for Focus
         //no need to send DeviceDisplayContext with true to HMI
      }
      else
      {
         //Video Focus can be granted to Phone in the current context
         m_oCurMDFocusStateLock.s16Lock();
         m_enCurMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_GAIN ;
         m_oCurMDFocusStateLock.vUnlock();
         vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_PROJECTED);
      }
   } //else
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vOnSPISelectDeviceResult()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID,
                                                             tenDeviceConnectionReq enDevConnReq,
                                                             tenResponseCode enRespCode,
                                                             tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vOnSPISelectDeviceResult: Result of -%d ",
      ETG_ENUM(CONNECTION_REQ,enDevConnReq)));

   SPI_INTENTIONALLY_UNUSED(enErrorCode);

   if((e8DEVCONNREQ_SELECT == enDevConnReq) && (e8SUCCESS == enRespCode))
   {
      m_u32SelectedDeviceID = u32DevID;

      m_oVideoSetupLock.s16Lock();
      m_bInitialMDVideoFocus = false;
      m_oVideoSetupLock.vUnlock();
   }

   if(e8DEVCONNREQ_DESELECT == enDevConnReq)
   {
      m_u32SelectedDeviceID = 0;

      m_oVideoSetupLock.s16Lock();
      m_bInitialMDVideoFocus = false;
      m_oVideoSetupLock.vUnlock();

      m_oCurMDFocusStateLock.s16Lock();
      m_enCurMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_LOSS;
      m_oCurMDFocusStateLock.vUnlock();

      m_oVideoPlaybackStateLock.s16Lock();
      //Set the Video Focus request state to default value, on device de selection
      m_enPlaybackState = e8VID_PB_STATE_STOPPED;
      m_oVideoPlaybackStateLock.vUnlock();

      if(NULL != m_cpoOnCarRsrcMngr)
      {
          m_cpoOnCarRsrcMngr->vUpdateDeviceDisplayCntxt(false,tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
      }//else if(NULL != m_poAAPRsrcMngr)
      //! Whenever device is activated/de-activated default the user interaction flag to false.
      if (NULL != m_cpoOnCarRsrcMngr)
      {
         m_cpoOnCarRsrcMngr->vDevAuthAndAccessInfoCb(m_u32SelectedDeviceID, e8_HANDSET_INTERACTION_UNKNOWN);
      }
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayMode(t_U32...
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayMode(const t_U32 cou32DeviceHandle,
                                                             const trDisplayContext corDisplayContext,
                                                             const trDisplayConstraint corDisplayConstraint,
                                                             const tenDisplayInfo coenDisplayInfo)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayMode: Resource Mngt handled by %d",
      ETG_ENUM(DISPLAY_INFO,coenDisplayInfo)));

   SPI_INTENTIONALLY_UNUSED(corDisplayConstraint);

   if(e8_DISPLAY_CONTEXT == coenDisplayInfo)
   {
      vSetAccessoryDisplayContext(cou32DeviceHandle,
         corDisplayContext.bDisplayFlag,
         corDisplayContext.enDisplayContext);
   }//if(e8_DISPLAY_CONTEXT == coenDisplayInfo)
   if(e8_DISPLAY_CONSTRAINT == coenDisplayInfo)
   {
      if((e8DIPO_CONSTRAINT_NEVER ==corDisplayConstraint.enTakeConstraint)||(e8DIPO_CONSTRAINT_NEVER == corDisplayConstraint.enBorrowConstraint))
      {  // Considered Borrow constraint also as in the cases for Emergency e.g RVC, Borrow can be set to Never where take can still be userinitiated/Anytime
         m_enGrantConstraint = e8DIPO_CONSTRAINT_NEVER;
      }
      else
      {
         m_enGrantConstraint = e8DIPO_CONSTRAINT_ANYTIME;
      }

      if( 0 != m_u32SelectedDeviceID )
      {
         switch(corDisplayConstraint.enTransferType)
         {
         case e8DIPO_TRANSFERTYPE_TAKE:
            {
               //Take has higher priority, process take even after borrow
               m_oCurMDFocusStateLock.s16Lock();
               m_enCurMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_LOSS;
               m_oCurMDFocusStateLock.vUnlock();
               m_enPrevTransferType = corDisplayConstraint.enTransferType;
               vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_NATIVE);
            }
            break;

         case e8DIPO_TRANSFERTYPE_BORROW:
            //after take, if borrow comes - continue with take only
            //else proceed with borrow
            if(e8DIPO_TRANSFERTYPE_TAKE != m_enPrevTransferType)
            {
               m_oCurMDFocusStateLock.s16Lock();
               m_enCurMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_LOSS_TRANSIENT;
               m_oCurMDFocusStateLock.vUnlock();
               m_enPrevTransferType = corDisplayConstraint.enTransferType;
               vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_NATIVE);
            }
            break;

         case e8DIPO_TRANSFERTYPE_UNTAKE:
            {
               //Nothing to do. HMI has to do Launch App
            }
            break;

         case e8DIPO_TRANSFERTYPE_UNBORROW:
            // check for previous state - If the last state was Borrow then only process
            if( e8DIPO_TRANSFERTYPE_BORROW == m_enPrevTransferType)
            {
               m_oCurMDFocusStateLock.s16Lock();
               m_enCurMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_GAIN_TRANSIENT;
               m_oCurMDFocusStateLock.vUnlock();
               m_enPrevTransferType = corDisplayConstraint.enTransferType;
               vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_PROJECTED);
            }
            break;
         default:
            ETG_TRACE_ERR(("[ERR]::Accessory Display Context:default state"));
         }//switch(corDisplayConstraint.enTransferType)
      }//if( 0 != m_u32SelectedDeviceID )

   }//if(e8_DISPLAY_CONSTRAINT == coenDisplayInfo)

   //If HMI is handling display context, SPI has to use the interfaces DeviceVideoFocusRequest & SetDeviceVideoFocus.

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vRequestProjUI()
***************************************************************************/
t_Void spi_tclOnCarVideoResourceMngr::vRequestProjUI(t_U32 u32DevId,t_U32 u32AppId)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideoResourceMngr::vRequestProjUI:Device ID-0x%x AppID-0x%x",
      u32DevId,u32AppId));

   m_oVideoSetupLock.s16Lock();
   // Request for Projected view by providing focus to MD only if video setup is completed
   if(NULL != spoCmdVideo)
   {
      //Get the accessory response based on the current display context, for the launch app request.
      //If the RVC is ACTIVE, Ignore the Launch App request. [Long press of MENU button by User, when RVC is ACTIVE]
      if(true == bProcessRequestProjUI())
      {
         //Set the Playback state to GAIN
         m_oCurMDFocusStateLock.s16Lock();
         m_enCurMDFocusState = e8_ONCAR_VIDEOFOCUSSTATE_GAIN;
         m_oCurMDFocusStateLock.vUnlock();

         //Send Projection request to Phone
         m_oVideoPlaybackStateLock.s16Lock();
         ETG_TRACE_USR4(("[PARAM]:vRequestProjUI:Current Video Playback state - %d",ETG_ENUM(AAP_VID_PB_STATE,m_enPlaybackState)));

         /*
         if the Video Focus is already with MD & user has launched app, then directly request HMI to enable the Projection layer.
         if not, send VideoFocus request to PHONE and set the Video Focus state to START request in progress.
         */
         if (e8VID_PB_STATE_STARTED == m_enPlaybackState)
         {
            ETG_TRACE_USR4(("[DESC]:MD already has the Video Focus. Directly enable projection Layer"));
            if(NULL != m_cpoOnCarRsrcMngr)
            {
                m_cpoOnCarRsrcMngr->vUpdateDeviceDisplayCntxt(true,tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
            }
         }//if (e8VID_PB_STATE_STARTED == m_enPlaybackState)
         else
         {
            vSetVideoFocus(e8_ONCAR_VIDEOFOCUS_PROJECTED);
         }

         m_oVideoPlaybackStateLock.vUnlock();
      }
      else
      {
         ETG_TRACE_ERR(("[ERR]: Do not process the Launch app request in the current display context"));
      }

   }//if (NULL != spoCmdVideo)
   else
   {
      ETG_TRACE_USR2(("[DESC]: Ignoring Projection UI requested before Video Setup"));
   }
   m_oVideoSetupLock.vUnlock();

   //return true;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bProcessRequestProjUI()
***************************************************************************/
t_Bool spi_tclOnCarVideoResourceMngr::bProcessRequestProjUI()
{
   t_Bool bRet=false;

   t_U8 u8ContSize = (sizeof(sacoMDVideoFocusReq))/(sizeof(trOnCarMDVideoFocusReq));
   for(t_U8 u8Index =0; u8Index < u8ContSize; u8Index++)
   {
      if(m_enCurAccDispCntxt == sacoMDVideoFocusReq[u8Index].enAccDispCntxt)
      {
         bRet = (e8_ONCAR_VIDEOFOCUS_GRANT == sacoMDVideoFocusReq[u8Index].enMDFocusReqResp_StateLoss);

         ETG_TRACE_USR4(("[DESC]:Accessory response for Start Video focus request by User:Display Context-%d Result:%d",
            ETG_ENUM(DISPLAY_CONTEXT,m_enCurAccDispCntxt),ETG_ENUM(BOOL,bRet)));

         // stop the process once search hits at the desired value
         break;
      } //if( (enCurMDFocusState == sacoMDVideoFocusReq[u8Index]
   } //for(t_U8 u8Index =0; u8Index < u8ContSize; u8Index++)

   return bRet;
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>

