/***********************************************************************/
/*!
 * \file    spi_tclOnCarAudioResourceMngr.cpp
 * \brief   OnCar Audio Resource Manager
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    OnCar Audio resource manager
 AUTHOR:         jun5kor
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date         | Author                | Modification
 22.03.2018   | Unmukt Jain           | Initial Version
 20.8.2018    | Ashwini Savadi        | Audio Resource Manager adaptation for
                                        Media.
 10.10.2018   | Ashwini Savadi        | Audio Resource Manager implementation for
                                        Guidance


 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/

#include "spi_tclOnCarAudioResourceMngr.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclOnCarCmdAudio.h"
static spi_tclOnCarCmdAudio* spoCmdAudio = NULL;
static spi_tclOnCarManager* spoOnCarMngr = NULL;

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclOnCarAudioResourceMngr.cpp.trc.h"
#endif
#endif

//! Initialise table for Device audio focus requests
#define ONCAR_AUDIO_FOCUS_RESPONSE
static trOnCarAudioResponseContext rOnCarAudioRespContext[] =
#include "spi_tclOnCarAudioContext.cfg"
#undef ONCAR_AUDIO_FOCUS_RESPONSE

//! Initialise table for audio focus notifications to Device
#define ONCAR_AUDIO_FOCUS_NOTIFICATION
static trOnCarAudioNotifContext rOnCarAudioNotifContext[] =
#include "spi_tclOnCarAudioContext.cfg"
#undef ONCAR_AUDIO_FOCUS_NOTIFICATION
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/


/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudioResourceMngr::spi_tclOnCarAudioResourceMngr()
 ***************************************************************************/
spi_tclOnCarAudioResourceMngr::spi_tclOnCarAudioResourceMngr(spi_tclOnCarResourceMngr* poOnCarRsrcMngr):
m_enCurAccFocusState(e8_ON_CAR_ACC_FOCUS_STATE_GAINED),
m_bCurCtxtFlag(false),
m_enCurAudioCntxt(e8SPI_AUDIO_UNKNOWN),
m_enCurAccAudContextType(e8_ONCAR_AUDIO_CTXT_MAIN),
m_bIsMediaActive(false),
m_bIsGuidanceActive(false)
{
   ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr() entered"));
}
/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudioResourceMngr::spi_tclOnCarAudioResourceMngr()
 ***************************************************************************/
spi_tclOnCarAudioResourceMngr::spi_tclOnCarAudioResourceMngr():
m_enCurAccFocusState(e8_ON_CAR_ACC_FOCUS_STATE_GAINED),
m_bCurCtxtFlag(false),
m_enCurAudioCntxt(e8SPI_AUDIO_UNKNOWN),
m_enCurAccAudContextType(e8_ONCAR_AUDIO_CTXT_MAIN),
m_bIsMediaActive(false),
m_bIsGuidanceActive(false)
{
   ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr() entered"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarAudioResourceMngr::~spi_tclOnCarAudioResourceMngr()
 ***************************************************************************/
spi_tclOnCarAudioResourceMngr::~spi_tclOnCarAudioResourceMngr()
{
   ETG_TRACE_USR1(("~spi_tclOnCarAudioResourceMngr() entered"));
   spoCmdAudio = NULL;
   spoOnCarMngr = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarAudioResourceMngr::bInitialize()
 ***************************************************************************/
t_Bool spi_tclOnCarAudioResourceMngr::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr::bInitialize() entered"));

   t_Bool bRet = false;
    spoOnCarMngr = spi_tclOnCarManager::getInstance();
    SPI_NORMAL_ASSERT(NULL == spoOnCarMngr);
    if (NULL != spoOnCarMngr)
    {
        spoCmdAudio = spoOnCarMngr->poGetAudioInstance();
        bRet = (spoCmdAudio == NULL) ? false : true;
    }//if (NULL != spoOnCarMngr)
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarAudioResourceMngr::vUninitialize()
 ***************************************************************************/
t_Void spi_tclOnCarAudioResourceMngr::vUnInitialize()
{
    m_bIsMediaActive = false;
    m_bIsGuidanceActive = false;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarAudioResourceMngr::vSetAccessoryAudioContext()
***************************************************************************/
t_Void spi_tclOnCarAudioResourceMngr::vSetAccessoryAudioContext(const t_U32 cou32DevId,
         const tenAudioContext coenAudioCntxt, t_Bool bReqFlag)
{
    m_oAudioRsrcMngrCbLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr::vSetAccessoryAudioContext() entered coenAudioCntxt = %d, bReqFlag = %d",
                    ETG_ENUM(AUDIO_CONTEXT,coenAudioCntxt),ETG_ENUM(BOOL,bReqFlag)));
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   tenOnCarAccessoryAudioFocusState enCurAccFocusState = enGetCurrentAccFocusState();

   //! Process audio context update only if there is a change in audio context or flag.
   if((coenAudioCntxt != m_enCurAudioCntxt) || (bReqFlag != m_bCurCtxtFlag))
   {
       tenOnCarDeviceAudioFocusState enNewDevAudFocusState = e8_ON_CAR_FOCUS_STATE_LOSS;
       tenOnCarAccessoryAudioFocusState enNewAccAudFocusState = enCurAccFocusState;
       tenOnCarAudioResourceOwner enAudResourceOwner = (true == bReqFlag)?
               e8_ONCAR_AUDIO_OWNER_CAR : e8_ONCAR_AUDIO_OWNER_DEVICE;
       //! Evaluate audio context (Main/Transient/Mix)
       tenOnCarAudioContextType enAudContextType = e8_ONCAR_AUDIO_CTXT_MAIN; //Initialised to avoid Lint warning
       t_Bool bValidAudioContext = true;
       switch(coenAudioCntxt)
       {
           case e8SPI_AUDIO_MAIN:
           case e8SPI_AUDIO_INTERNET_APP:
           case e8SPI_AUDIO_MAIN_DEFAULT:
           case e8SPI_AUDIO_PROJECTION:
           {
               enAudContextType = e8_ONCAR_AUDIO_CTXT_MAIN;
               vSetCurrentAccAudioCtxtType(e8_ONCAR_AUDIO_CTXT_MAIN);
           }
           break;
           case e8SPI_AUDIO_SPEECH_REC:
           case e8SPI_AUDIO_ADVISOR_PHONE:
           case e8SPI_AUDIO_EMER_PHONE:
           case e8SPI_AUDIO_PHONE:
           case e8SPI_AUDIO_INCOM_TONE:
           case e8SPI_AUDIO_SYNC_MSG:
           case e8SPI_AUDIO_ASYNC_MSG:
           case s8SPI_AUDIO_TRAFFIC:
           case e8SPI_AUDIO_LVM:
           case e8SPI_AUDIO_CUE:
           case e8SPI_AUDIO_SLEEP_MODE:
           case e8SPI_AUDIO_STANDBY_MODE:
           {
               enAudContextType = e8_ONCAR_AUDIO_CTXT_TRANSIENT;
               (bReqFlag) ? vSetCurrentAccAudioCtxtType(e8_ONCAR_AUDIO_CTXT_TRANSIENT):
                       vSetCurrentAccAudioCtxtType(e8_ONCAR_AUDIO_CTXT_MAIN);
           }
           break;
           case e8SPI_AUDIO_MIX_ALERT_MSG:
           {
               enAudContextType = e8_ONCAR_AUDIO_CTXT_TRANSIENT_MIX;
               (bReqFlag) ? vSetCurrentAccAudioCtxtType(e8_ONCAR_AUDIO_CTXT_TRANSIENT_MIX) :
                       vSetCurrentAccAudioCtxtType(e8_ONCAR_AUDIO_CTXT_MAIN);
           }
           break;
           case e8SPI_AUDIO_NONE:
           {
              enAudContextType = e8_ONCAR_AUDIO_CTXT_MAIN;
              vSetCurrentAccAudioCtxtType(e8_ONCAR_AUDIO_CTXT_MAIN);
              //Note: This is used to reset native audio context to main when there is no channel active.
              //Audio focus should not be updated to phone in this case.
              bValidAudioContext = false;
           }
           break;
           case e8SPI_AUDIO_MUTE:
           {
               vSetCurrentAccAudioCtxtType(e8_ONCAR_AUDIO_CTXT_MAIN);
               bValidAudioContext = false;
               (bReqFlag) ? (enNewDevAudFocusState = e8_ON_CAR_FOCUS_STATE_MUTE) : 
                            (enNewDevAudFocusState = e8_ON_CAR_FOCUS_STATE_DEMUTE);
               if(NULL != spoCmdAudio)
               {
                   if(true == m_bIsMediaActive)
                   {
                        spoCmdAudio->vSetAudioFocus(e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA, enNewDevAudFocusState);
                   }
               }
               (bReqFlag) ? vSetCurrentAccFocusState(e8_ON_CAR_ACC_FOCUS_STATE_GAINED) : 
                     vSetCurrentAccFocusState(e8_ON_CAR_ACC_FOCUS_STATE_LOST);
           }
           break;
           case e8SPI_AUDIO_EMER_MSG:
           case e8SPI_AUDIO_SHORT_MIX_ALERT_MSG:
           case e8SPI_AUDIO_ALERT_TONE:
           default:
               bValidAudioContext = false;
               break;
       }
       if(bValidAudioContext &&
            (true == bGetAudioFocusNotification(enAudResourceOwner, enAudContextType,
                                 enCurAccFocusState, enNewDevAudFocusState, enNewAccAudFocusState)))
       {
           vSetCurrentAccFocusState(enNewAccAudFocusState);
           if(NULL != spoCmdAudio)
           {
               if(true == m_bIsMediaActive)
               {
                   spoCmdAudio->vSetAudioFocus(e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA, enNewDevAudFocusState);
               }
               else if(true == m_bIsGuidanceActive)
               {
                   spoCmdAudio->vSetAudioFocus(e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE, enNewDevAudFocusState);
               }
               if(e8_ON_CAR_FOCUS_STATE_LOSS == enNewDevAudFocusState)
               {
                   m_bIsMediaActive = false;
               }
           }
       }
       else
       {
           ETG_TRACE_USR3(("spi_tclOnCarAudioResourceMngr::vSetAccessoryAudioContext: No change in audio state "));
       }
       //! Store current states
       m_enCurAudioCntxt = coenAudioCntxt;
       m_bCurCtxtFlag = bReqFlag;
   }
   m_oAudioRsrcMngrCbLock.vUnlock();
}

/***************************************************************************
** FUNCTION: spi_tclOnCarAudioResourceMngr::vPostaudioFocusRequestCallback()
***************************************************************************/
t_Void spi_tclOnCarAudioResourceMngr::vPostaudioFocusRequestCallback(const tenOnCarAudioStreamType enStreamType,
                    const tenOnCarAudioFocusRequest enDevFocusRequest)
{
    ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr::vPostaudioFocusRequestCallback :: ReqType :%d ",
          ETG_ENUM(DEVICE_AUDIOFOCUS_REQ, enDevFocusRequest)));
    tenOnCarAccessoryAudioFocusState enNewAccAudFocusState = e8_ON_CAR_ACC_FOCUS_STATE_GAINED;
    tenOnCarDeviceAudioFocusState enNewDevAudFocusState = e8_ON_CAR_FOCUS_STATE_LOSS;
    m_oAudioRsrcMngrCbLock.s16Lock();
    tenOnCarDeviceAudioFocusRequest enDevAudioFocusRequest = e8_ONCAR_DEVICE_AUDIOFOCUS_REQUEST_UNKNOWN;
    switch(enStreamType)
    {
        case e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA:
        {
            if(e8_ONCAR_AUDIOFOCUS_REQUEST_GAIN == enDevFocusRequest)
            {
                ETG_TRACE_USR1(("MEDIA : GAIN Request "));
                enDevAudioFocusRequest = e8_ONCAR_AUDIOFOCUS_REQUEST_MEDIA_GAIN;
                m_bIsMediaActive = true;
            }
            else if(e8_ONCAR_AUDIOFOCUS_REQUEST_RELEASE == enDevFocusRequest)
            {
                ETG_TRACE_USR1(("MEDIA : RELEASE Request "));
                enDevAudioFocusRequest = e8_ONCAR_AUDIOFOCUS_REQUEST_MEDIA_RELEASE;
                m_bIsMediaActive = false;
            }
            else
            {
                ETG_TRACE_ERR(("MEDIA : UNKNOWN Request "));
                m_bIsMediaActive = false;
            }
        }
        break;
        case e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE:
        {
            if(e8_ONCAR_AUDIOFOCUS_REQUEST_GAIN == enDevFocusRequest)
            {
                ETG_TRACE_USR1(("GUIDANCE : GAIN Request "));
                enDevAudioFocusRequest = e8_ONCAR_AUDIOFOCUS_REQUEST_GUIDANCE_GAIN;
                m_bIsGuidanceActive = true;
            }
            else if(e8_ONCAR_AUDIOFOCUS_REQUEST_RELEASE == enDevFocusRequest)
            {
                ETG_TRACE_USR1(("GUIDANCE : RELEASE Request "));
                enDevAudioFocusRequest = e8_ONCAR_AUDIOFOCUS_REQUEST_GUIDANCE_RELEASE;
                m_bIsGuidanceActive = false;
            }
            else
            {
                ETG_TRACE_ERR(("GUIDANCE : UNKNOWN Request"));
                m_bIsGuidanceActive = false;
            }
        }
        break;
        default:
        {
            ETG_TRACE_ERR(("Invalid Stream "));
        }
        break;
    }
    if(false == bGetAudioFocusResponse(enDevAudioFocusRequest, enGetCurrentAccFocusState(), enGetCurrentAccAudioCtxtType(),
            enNewDevAudFocusState, enNewAccAudFocusState))
    {
        ETG_TRACE_USR3(("spi_tclOnCarAudioResourceMngr::vPostaudioFocusRequestCallback: No response found in map. Sending default. "));
        enNewDevAudFocusState = e8_ON_CAR_FOCUS_STATE_LOSS;
        enNewAccAudFocusState = e8_ON_CAR_ACC_FOCUS_STATE_GAINED;
    }

    //! Store latest accessory focus state
    vSetCurrentAccFocusState(enNewAccAudFocusState);
    
    if((e8_ONCAR_AUDIOFOCUS_REQUEST_GAIN == enDevFocusRequest) && 
     (e8_ON_CAR_FOCUS_STATE_LOSS == enNewDevAudFocusState))
    {
        if(e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA == enStreamType)
        {
            m_bIsMediaActive = false;
        }
        else if(e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE == enStreamType)
        {
            m_bIsGuidanceActive = false;
        }
    }
    
    if(NULL != spoCmdAudio)
    {
        spoCmdAudio->vSetAudioFocus(enStreamType, enNewDevAudFocusState);
    }
    m_oAudioRsrcMngrCbLock.vUnlock();
}

/***************************************************************************
** FUNCTION: spi_tclOnCarAudioResourceMngr::bGetAudioFocusNotification()
***************************************************************************/
t_Bool spi_tclOnCarAudioResourceMngr::bGetAudioFocusNotification(
        tenOnCarAudioResourceOwner enAudResourceOwner,
        tenOnCarAudioContextType enAudContextType,
        tenOnCarAccessoryAudioFocusState enCurAccAudFocusState,
        tenOnCarDeviceAudioFocusState& rfeNewDevAudFocusState,
        tenOnCarAccessoryAudioFocusState& rfenNewAccAudFocusState)
{
   t_Bool bRetVal = false;
   t_U32 u32ContainerSize = (sizeof(rOnCarAudioNotifContext))/(sizeof(trOnCarAudioNotifContext));
   //! Iterate through table to find match based on current audio resource owner,
   //! audio context type and accessory audio focus.
   for(t_U8 u8index = 0 ; u8index < u32ContainerSize ; u8index++)
   {
       if (
          (enAudResourceOwner == rOnCarAudioNotifContext[u8index].enAudResourceOwner)
          &&
          (enAudContextType == rOnCarAudioNotifContext[u8index].enAudContextType)
          &&
          (enCurAccAudFocusState == rOnCarAudioNotifContext[u8index].enCurAccAudFocusState)
          )
       {
           bRetVal = true;
           rfeNewDevAudFocusState = rOnCarAudioNotifContext[u8index].eNewDevAudFocusState;
           rfenNewAccAudFocusState = rOnCarAudioNotifContext[u8index].enNewAccAudFocusState;
           // stop the process one search hits at the desired value
           break;
       }
   }
   
   ETG_TRACE_USR2(("spi_tclOnCarAudioResourceMngr::bGetAudioFocusNotification responded with :: "
              "Success: %d, NewDevAudFocusState :%d, NewAccAudFocusState : %d "
              "(for CurAudResourceOwner : %d, CurAudContextType : %d, CurAccAudFocusState : %d)",
              ETG_ENUM(BOOL, bRetVal),
              ETG_ENUM(ONCAR_DEVICE_AUDIOFOCUS, rfeNewDevAudFocusState),
              ETG_ENUM(NATIVE_AUDIOFOCUS, rfenNewAccAudFocusState),
              ETG_ENUM(AUDIO_RSRC_OWNER, enAudResourceOwner),
              ETG_ENUM(AUDIO_CONTEXT_TYPE, enAudContextType),
              ETG_ENUM(NATIVE_AUDIOFOCUS, enCurAccAudFocusState)));
   return bRetVal;
}

/***************************************************************************
** FUNCTION: spi_tclOnCarAudioResourceMngr::bGetAudioFocusResponse()
***************************************************************************/
t_Bool spi_tclOnCarAudioResourceMngr::bGetAudioFocusResponse(
        tenOnCarDeviceAudioFocusRequest enDevAudFocusRequest,
        tenOnCarAccessoryAudioFocusState enCurAccAudFocusState,
         tenOnCarAudioContextType enCurAccAudContext,
         tenOnCarDeviceAudioFocusState& rfeNewDevAudFocusState,
         tenOnCarAccessoryAudioFocusState& rfenNewAccAudFocusState)
{
   t_Bool bRetVal = false;
   t_U32 u32ContainerSize = (sizeof(rOnCarAudioRespContext))/(sizeof(trOnCarAudioResponseContext));
   //@Note: Currently GAIN_TRANSIENT and GAIN_TRANSIENT_MAY_DUCK are treated as same.
   //Hence config tabel contains only one entry for both types to avoid duplicate entries.
   //if (e8_ONCAR_AUDIOFOCUS_REQUEST_GAIN_TRANSIENT_MAY_DUCK == enDevAudFocusRequest)
  // {
   //   enDevAudFocusRequest = e8_ONCAR_AUDIOFOCUS_REQUEST_GAIN_TRANSIENT;
 //  }
   //! Iterate through table to find match based on audio focus request,
   //! current accessory audio focus and accessory transient audio state
   for(t_U8 u8index = 0; u8index < u32ContainerSize; u8index++)
   {
       ETG_TRACE_USR2(("spi_tclOnCarAudioResourceMngr::bGetAudioFocusResponse responded with :: "
                     "Success: %d, NewDevAudFocusState :%d, NewAccAudFocusState : %d "
                     "(for DevAudFocusRequest : %d, CurAccAudFocusState : %d, CurAccAudContext : %d)",
                     ETG_ENUM(BOOL, bRetVal),
                     ETG_ENUM(ONCAR_DEVICE_AUDIOFOCUS, rfeNewDevAudFocusState),
                     ETG_ENUM(NATIVE_AUDIOFOCUS, rfenNewAccAudFocusState),
                     ETG_ENUM(DEVICE_AUDIOFOCUS_REQ, enDevAudFocusRequest),
                     ETG_ENUM(NATIVE_AUDIOFOCUS, enCurAccAudFocusState),
                     ETG_ENUM(AUDIO_CONTEXT_TYPE, enCurAccAudContext)));
      if (
         (enDevAudFocusRequest == rOnCarAudioRespContext[u8index].enDevAudFocusRequest)
         &&
         (enCurAccAudFocusState == rOnCarAudioRespContext[u8index].enCurAccAudFocusState)
         &&
         (enCurAccAudContext == rOnCarAudioRespContext[u8index].enCurAccAudContext)
         )
      {
         bRetVal = true;
         rfeNewDevAudFocusState = rOnCarAudioRespContext[u8index].eNewDevAudFocusState;
         rfenNewAccAudFocusState = rOnCarAudioRespContext[u8index].enNewAccAudFocusState;
         // stop the process one search hits at the desired value
         break;
      }
   }

   ETG_TRACE_USR2(("spi_tclOnCarAudioResourceMngr::bGetAudioFocusResponse responded with :: "
              "Success: %d, NewDevAudFocusState :%d, NewAccAudFocusState : %d "
              "(for DevAudFocusRequest : %d, CurAccAudFocusState : %d, CurAccAudContext : %d)",
              ETG_ENUM(BOOL, bRetVal),
              ETG_ENUM(ONCAR_DEVICE_AUDIOFOCUS, rfeNewDevAudFocusState),
              ETG_ENUM(NATIVE_AUDIOFOCUS, rfenNewAccAudFocusState),
              ETG_ENUM(DEVICE_AUDIOFOCUS_REQ, enDevAudFocusRequest),
              ETG_ENUM(NATIVE_AUDIOFOCUS, enCurAccAudFocusState),
              ETG_ENUM(AUDIO_CONTEXT_TYPE, enCurAccAudContext)));
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarAudioResourceMngr::vOnSPISelectDeviceResult()
***************************************************************************/
t_Void spi_tclOnCarAudioResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID, tenDeviceConnectionReq enDevConnReq,
                                                               tenResponseCode enRespCode,tenErrorCode enErrorCode)
{
   SPI_INTENTIONALLY_UNUSED(u32DevID);
   SPI_INTENTIONALLY_UNUSED(enDevConnReq);
   SPI_INTENTIONALLY_UNUSED(enRespCode);
   SPI_INTENTIONALLY_UNUSED(enErrorCode);
   ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr::vOnSPISelectDeviceResult() entered"));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarAudioResourceMngr::vSetCurrentAccFocusState()
***************************************************************************/
t_Void spi_tclOnCarAudioResourceMngr::vSetCurrentAccFocusState(tenOnCarAccessoryAudioFocusState enNewAccAudFocusState)
{
   ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr::vSetCurrentAccFocusState: %d ",
         ETG_ENUM(NATIVE_AUDIOFOCUS, enNewAccAudFocusState)));
   //! Store current accessory audio focus state
   m_enCurAccFocusState = enNewAccAudFocusState;
}

/***************************************************************************
** FUNCTION:  tenAccessoryAudioFocusState tenOnCarAccessoryAudioFocusState::enGetCurrentAccFocusState()
***************************************************************************/
tenOnCarAccessoryAudioFocusState spi_tclOnCarAudioResourceMngr::enGetCurrentAccFocusState()
{
   //! Read current accessory audio focus state
   ETG_TRACE_USR2(("spi_tclOnCarAudioResourceMngr::enGetCurrentAccFocusState left with: %d ",
         ETG_ENUM(NATIVE_AUDIOFOCUS, m_enCurAccFocusState)));
   return m_enCurAccFocusState;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarAudioResourceMngr::vSetCurrentAccAudioCtxtType()
***************************************************************************/
t_Void spi_tclOnCarAudioResourceMngr::vSetCurrentAccAudioCtxtType(
        tenOnCarAudioContextType enNewAccAudioCtxtType)
{
   ETG_TRACE_USR1(("spi_tclOnCarAudioResourceMngr::vSetCurrentAccAudioCtxtType: %d ",
         ETG_ENUM(AUDIO_CONTEXT_TYPE, enNewAccAudioCtxtType)));
   //! Set current accessory audio context type
   m_enCurAccAudContextType = enNewAccAudioCtxtType;
}

/***************************************************************************
** FUNCTION:  tenAudioContextType spi_tclOnCarAudioResourceMngr::enGetCurrentAccAudioCtxtType()
***************************************************************************/
tenOnCarAudioContextType spi_tclOnCarAudioResourceMngr::enGetCurrentAccAudioCtxtType()
{
   //! Read current accessory audio context type
   ETG_TRACE_USR2(("spi_tclOnCarAudioResourceMngr::enGetCurrentAccAudioCtxtType left with: %d ",
         ETG_ENUM(AUDIO_CONTEXT_TYPE, m_enCurAccAudContextType)));
   return m_enCurAccAudContextType;
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>

