/*!
 *******************************************************************************
 * \file              spi_tclOnCarCmdSensor.cpp
 * \brief             Sensor wrapper for OnCar
 *******************************************************************************
 \verbatim
 PROJECT        :   Gen3
 SW-COMPONENT   :   Smart Phone Integration
 DESCRIPTION    :   Sensor wrapper for OnCar
 COPYRIGHT      :   &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 26.04.2018  | Rishav Sardar         | Initial Version
 
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclOnCarCmdSensor.h"
#include "spi_tclOnCarDataIntf.h"
#include <math.h>

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_ONCARWRAPPER
      #include "trcGenProj/Header/spi_tclOnCarCmdSensor.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| defines and macros 
|----------------------------------------------------------------------------*/
#define dPItoDegree (180/3.14159265358979323846)         //used for converting pi to decimal degrees
static const t_S32 scos32Failed = -1;


//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::spi_tclOnCarCmdSensor()
***************************************************************************/
spi_tclOnCarCmdSensor::spi_tclOnCarCmdSensor():m_poOnCarSensorEndpoint(NULL),m_s32speed(0)
{
   ETG_TRACE_USR1((" spi_tclOnCarCmdSensor::spi_tclOnCarCmdSensor() entered "));
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::~spi_tclOnCarCmdSensor()
***************************************************************************/
spi_tclOnCarCmdSensor::~spi_tclOnCarCmdSensor()
{
   ETG_TRACE_USR1((" spi_tclOnCarCmdSensor::~spi_tclOnCarCmdSensor() entered "));
   m_oSensorEndPointLock.s16Lock();
   //RELEASE_MEM(m_poOnCarSensorEndpoint);
   m_s32speed = 0;
   m_oSensorEndPointLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::bInitializeSensorEndpoint()
***************************************************************************/
t_Bool spi_tclOnCarCmdSensor::bInitializeSensorEndpoint(const trDataServiceConfigData& rfrDataServiceConfigData)
{
   SPI_INTENTIONALLY_UNUSED(rfrDataServiceConfigData);
   
   ETG_TRACE_USR1(("spi_tclOnCarCmdSensor::bInitializeSensorEndpoint() entered "));

   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   OnCarAPI *poOnCarAPI = NULL;
   t_Bool bRetVal = false;

   if(NULL != poOnCarDataIntf)
   {
	   m_poOnCarSensorEndpoint = poOnCarDataIntf->poGetSensorEndpointInstance();
	   poOnCarAPI = poOnCarDataIntf->poGetOnCarAPIInstance();
   }//if(NULL != poOnCarDataIntf)

   if((m_poOnCarSensorEndpoint != NULL) && (NULL != poOnCarAPI))
   {
	   m_poOnCarSensorEndpoint->init();
       bRetVal = poOnCarAPI->registerService(m_poOnCarSensorEndpoint);
       ETG_TRACE_USR2(("[DESC]:spi_tclOnCarCmdSensor::bInitializeSensorEndpoint(): Initialization Result : %d ",
                     ETG_ENUM(BOOL,bRetVal)));
   }// ((m_poOnCarSensorEndpoint != NUL) && (NULL != poOnCarAPI))
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclOnCarCmdSensor::bInitializeSensor(): Invalid pointer encountered! "));
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::bUnInitializeSensorEndpoint()
***************************************************************************/
t_Void spi_tclOnCarCmdSensor::bUnInitializeSensorEndpoint()
{
   ETG_TRACE_USR1(("spi_tclOnCarCmdSensor::bUnInitializeSensorEndpoint() entered "));
   m_oSensorEndPointLock.s16Lock();
   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   if(NULL != m_poOnCarSensorEndpoint)
   {
	   m_poOnCarSensorEndpoint->deinit();
	   poOnCarDataIntf->vDestroyEndpointInstance(m_poOnCarSensorEndpoint,e8_ONCAR_OBJECTFACTORY_SENSOR_ENDPOINT);
	   m_poOnCarSensorEndpoint = NULL;
   }// ((m_poOnCarSensorEndpoint != NUL) && (NULL != poOnCarAPI))
   m_s32speed = 0;
   m_oSensorEndPointLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::vReportLocationData(...)
***************************************************************************/
t_Void spi_tclOnCarCmdSensor::vReportLocationData(const trSensorData& rfrcSensorGpsData)
{
   ETG_TRACE_USR1((" spi_tclOnCarCmdSensor:::vReportGpsData() entered "));
   t_Bool bIsGPSFixAvailable = (e8GNSSQUALITY_NOFIX != rfrcSensorGpsData.enGnssQuality);
   if(true == bIsGPSFixAvailable)
   {
      t_Double dLatitude = static_cast<t_Double>(fabs(((rfrcSensorGpsData.dLatitude)*(dPItoDegree))));            // unit: degree
      t_Double dLongitude = static_cast<t_Double>(fabs(((rfrcSensorGpsData.dLongitude)*(dPItoDegree))));          // unit: degree
      t_Float fHeading = static_cast<t_Float>(fmod((((rfrcSensorGpsData.fHeading)*(dPItoDegree)) + 360.0), 360.0));   // unit: degree, range: [0,360)

      //!fabs converts negative values to positive. Retain the negative values of location.
      dLatitude  = (0 > rfrcSensorGpsData.dLatitude)?((-1)*(dLatitude)):(dLatitude);
      dLongitude = (0 > rfrcSensorGpsData.dLongitude)?((-1)*(dLongitude)):(dLongitude);

      ETG_TRACE_USR4(("[PARAM]::spi_tclOnCarCmdSensor::vReportGpsData(): Raw data "
            "Latitude = %f, Longitude = %f, Heading = %f, Speed = %f, "
            "Altitude = %f, Accuracy = %f, PosixTime = 0x%x,  ",
            dLatitude, dLongitude, fHeading, rfrcSensorGpsData.fSpeed,
            rfrcSensorGpsData.fAltitude, rfrcSensorGpsData.fAccuracy, rfrcSensorGpsData.PosixTime));

      t_U64 timestamp = (rfrcSensorGpsData.PosixTime) * 1000000000;

      t_S32 latitudeE7 = static_cast<t_S32>(dLatitude * 10000000);
      t_S32 longitudeE7 = static_cast<t_S32>(dLongitude * 10000000);

      t_Bool hasBearing = true;
      t_S32 bearingE6 = static_cast<t_S32>((fHeading) * 1000000);
      hasBearing = ((0 == bearingE6) || (0 == m_s32speed))?(false):(true);

      t_Bool hasAccuracy = true;
      t_S32 accuracyE3 =  static_cast<t_S32>(rfrcSensorGpsData.fAccuracy * 1000);

      t_Bool hasAltitude = true;
      t_S32 altitudeE2 = static_cast<t_S32>((rfrcSensorGpsData.fAltitude) * 100);

      t_Bool hasSpeed = true;
      t_S32 speedE3 = static_cast<t_S32>((rfrcSensorGpsData.fSpeed) * 1000);

      ETG_TRACE_USR4(("[PARAM]::spi_tclOnCarCmdSensor::vReportGpsData(): Calculated data "
        "latitudeE7 = %d, longitudeE7 = %d, hasBearing= %d, bearingE6 = %d, speedE3 = %d, "
        "altitudeE2 = %d, accuracyE3 = %d, timestamp = %d",
        latitudeE7, longitudeE7, ETG_ENUM(BOOL,hasBearing), bearingE6, speedE3,
        altitudeE2, accuracyE3, timestamp));

      m_oSensorEndPointLock.s16Lock();
      if(NULL != m_poOnCarSensorEndpoint)
      {
    	    m_poOnCarSensorEndpoint->reportLocationData(
            timestamp, latitudeE7, longitudeE7,
            hasAccuracy, accuracyE3, hasAltitude, altitudeE2,
            hasSpeed, speedE3, hasBearing, bearingE6);
      }//if (NULL != m_poOnCarSensorEndpoint)
      m_oSensorEndPointLock.vUnlock();
   }//(true == bIsGPSFixAvailable)
   else if (false == bIsGPSFixAvailable)
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclOnCarCmdSensor::vReportGpsData(): No GPS Fix available "));
   }
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::vReportSpeedData()
***************************************************************************/
t_Void spi_tclOnCarCmdSensor::vReportSpeedData(t_S32 s32speedE3)
{
	ETG_TRACE_USR1((" spi_tclOnCarCmdSensor:::vReportGpsData() entered "));
   m_s32speed = s32speedE3;
   m_oSensorEndPointLock.s16Lock();
   if (NULL != m_poOnCarSensorEndpoint)
   {
	   m_poOnCarSensorEndpoint->reportSpeedData(s32speedE3);
   }
   m_oSensorEndPointLock.vUnlock();
   ETG_TRACE_USR4(("[PARAM]:spi_tclOnCarCmdSensor::vReportSpeedData() Speed = %d ",s32speedE3));
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::vReportDayNightMode(...)
***************************************************************************/
t_Void spi_tclOnCarCmdSensor::vReportDayNightMode(tenOnCarDayNightMode enMode)
{
	ETG_TRACE_USR1((" spi_tclOnCarCmdSensor:::vReportDayNightMode() entered "));
   m_oSensorEndPointLock.s16Lock();
   if (NULL != m_poOnCarSensorEndpoint)
   {
	   m_poOnCarSensorEndpoint->reportDayNightMode(static_cast<DAYNIGHT_MODE>(enMode));
   }
   m_oSensorEndPointLock.vUnlock();
   ETG_TRACE_USR4(("[PARAM]:spi_tclOnCarCmdSensor::vReportDayNightMode() DayNightMode = %d",enMode));
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::vSetDriveModeRestriction(...)
***************************************************************************/
t_Void spi_tclOnCarCmdSensor::vSetDriveModeRestriction(t_S32 s32RestrictionInfo)
{
	ETG_TRACE_USR1((" spi_tclOnCarCmdSensor:::setDriveModeRestriction() entered "));
   m_oSensorEndPointLock.s16Lock();
   //! Send Drive status data
   if (NULL != m_poOnCarSensorEndpoint)
   {
	   t_U16 driveModeRestrictions = static_cast<t_U16>(s32RestrictionInfo);
	   //m_poOnCarSensorEndpoint->setDriveModeRestriction(driveModeRestrictions);
   }
   m_oSensorEndPointLock.vUnlock();
   ETG_TRACE_USR4(("[PARAM]:spi_tclOnCarCmdSensor::vSetDriveModeRestriction()(RestrictionInfo = %d)", s32RestrictionInfo));
}

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdSensor::setDriveModeRestriction(...)
***************************************************************************/
t_Void spi_tclOnCarCmdSensor::vReportVehicleDrivingState(tenOnCarDrivingState enState)
{
   ETG_TRACE_USR1((" spi_tclOnCarCmdSensor::vReportVehicleDrivingState() entered "));
   m_oSensorEndPointLock.s16Lock();
   if(NULL != m_poOnCarSensorEndpoint)
   {
	   m_poOnCarSensorEndpoint->reportVehicleDrivingState(static_cast<DRIVING_STATE>(enState));
   }
   m_oSensorEndPointLock.vUnlock();
   ETG_TRACE_USR4(("[PARAM]:spi_tclOnCarCmdSensor::vReportVehicleDrivingState() Driving Status = %d",enState));
}


//lint restore
