/*!
 *******************************************************************************
 * \file              spi_tclOnCarCmdInput.cpp
 * \brief             Input wrapper for OnCar
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Input wrapper for OnCar
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 21.05.2018 |  Rishav Sardar               | Initial Version

 \endverbatim
 ******************************************************************************/
/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclOnCarCmdInput.h"
#include "spi_tclOnCarDataIntf.h"
#include "OnCarAPI.h"


//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_ONCARWRAPPER
      #include "trcGenProj/Header/spi_tclOnCarCmdInput.cpp.trc.h"
   #endif
#endif

static trOnCarKeyCode aKeyCode[] =
{
    { e32INVALID_KEY, e8_ONCAR_INPUT_KEYCODE_INVALID },
    { e32DEV_BACKWARD, e8_ONCAR_INPUT_KEYCODE_BACK },
    { e32MULTIMEDIA_NEXT, e8_ONCAR_INPUT_KEYCODE_MEDIA_NEXT },
    { e32MULTIMEDIA_PREVIOUS, e8_ONCAR_INPUT_KEYCODE_MEDIA_PREVIOUS },
    { e32MULTIMEDIA_FORWARD, e8_ONCAR_INPUT_KEYCODE_MEDIA_FORWARD },
    { e32MULTIMEDIA_REWIND, e8_ONCAR_INPUT_KEYCODE_MEDIA_REWIND },
    { e32MULTIMEDIA_STOP, e8_ONCAR_INPUT_KEYCODE_MEDIA_STOP},
    { e32MULTIMEDIA_PLAY, e8_ONCAR_INPUT_KEYCODE_MEDIA_PLAY},
    { e32MULTIMEDIA_PAUSE, e8_ONCAR_INPUT_KEYCODE_MEDIA_PAUSE,},
    { e32APP_KEYCODE_MEDIA, e8_ONCAR_INPUT_KEYCODE_MEDIA},
    { e32APP_KEYCODE_NAVIGATION, e8_ONCAR_INPUT_KEYCODE_NAVIGATION},
    { e32APP_KEYCODE_TELEPHONY, e8_ONCAR_INPUT_KEYCODE_TEL},
    { e32DEV_PHONE_CALL, e8_ONCAR_INPUT_KEYCODE_PHONE_CALL},
    { e32DEV_PHONE_END, e8_ONCAR_INPUT_KEYCODE_PHONE_END},
    { e32TWODKNOB0_SHIFT_UP, e8_ONCAR_INPUT_KEYCODE_SHIFT_UP},
    { e32TWODKNOB0_SHIFT_DOWN, e8_ONCAR_INPUT_KEYCODE_SHIFT_DOWN},
    { e32TWODKNOB0_SHIFT_LEFT, e8_ONCAR_INPUT_KEYCODE_SHIFT_LEFT},
    { e32TWODKNOB0_SHIFT_RIGHT, e8_ONCAR_INPUT_KEYCODE_SHIFT_RIGHT},
    { e32DEV_HOME, e8_ONCAR_INPUT_KEYCODE_HOME},
    { e32DEV_OK, e8_ONCAR_INPUT_KEYCODE_ENTER}
};

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::spi_tclOnCarCmdInput()
***************************************************************************/
spi_tclOnCarCmdInput::spi_tclOnCarCmdInput(): m_poOnCarInputEndpoint(NULL)
{
  ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarCmdInput::spi_tclOnCarCmdInput() entered "));
  ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarCmdInput::spi_tclOnCarCmdInput() left "));
}//!end of spi_tclOnCarCmdInput()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::~spi_tclOnCarCmdInput()
***************************************************************************/
spi_tclOnCarCmdInput::~spi_tclOnCarCmdInput()
{
  ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarCmdInput::~spi_tclOnCarCmdInput() entered "));

  m_spoInputCbs = nullptr;
  ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarCmdInput::~spi_tclOnCarCmdInput() left "));
}//!end of ~spi_tclOnCarCmdInput()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::vReportkey()
***************************************************************************/
t_Void spi_tclOnCarCmdInput::vReportkey(t_U32 u32DeviceHandle,tenKeyMode enKeyMode,
        tenKeyCode enKeyCode)
{
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vReportkey() entered : Dev-0x%x KeyMode = %d KeyCode = %d",
                  u32DeviceHandle,enKeyMode,enKeyCode));
  SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);
  t_Bool bIsDown = false;
  m_oInputEndpointLock.s16Lock();
  if(m_poOnCarInputEndpoint)
  {
      tenOnCarInputKeyCodes enOnCarKeyCode = e8_ONCAR_INPUT_KEYCODE_INVALID;
      enOnCarKeyCode = (enGetKeyCode(enKeyCode));
      if(e8_ONCAR_INPUT_KEYCODE_INVALID != enOnCarKeyCode)
      {
          ETG_TRACE_USR2(("[DESC]:spi_tclOnCarCmdInput::vReportkey() : Received Key Event with : Key type = %d , key event = %d",
                          enKeyCode,enKeyMode));
          bIsDown = (e8KEY_PRESS == enKeyMode) ? (true):(false);
          m_poOnCarInputEndpoint->reportKey(static_cast<INPUT_KEYCODE>(enOnCarKeyCode),bIsDown);
      }//if(e8_ONCAR_INPUT_KEYCODE_INVALID != enOnCarKeyCode)
      else
      {
          ETG_TRACE_USR2(("[ERR]:spi_tclOnCarCmdInput::vReportkey Invalid key code"));
      }
  }// if(m_poOnCarInputEndpoint)
  m_oInputEndpointLock.vUnlock();
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vReportkey() left "));
}//!end of vReportkey()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::vReportTouch()
***************************************************************************/
t_Void spi_tclOnCarCmdInput::vReportTouch(t_U32 u32DeviceHandle,trTouchData &rfrTouchData,
                                        trScalingAttributes rScalingAttributes)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vReportTouch() entered : Dev-0x%x",u32DeviceHandle));
   SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);
   SPI_INTENTIONALLY_UNUSED(rfrTouchData);
   SPI_INTENTIONALLY_UNUSED(rScalingAttributes);
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vReportTouch() left "));
}//!end of vReportTouch()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::bInitializeInputEndpoint()
***************************************************************************/
t_Bool spi_tclOnCarCmdInput::bInitializeInputEndpoint(const trOnCarInputConfig &rOnCarInputConfig)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() entered "));
   t_Bool bInitStatus = false;
   vSetOnCarCodes();
   InputConfig oConfig;
   oConfig.DisplayHeight = rOnCarInputConfig.u32DisplayHeight;
   oConfig.DisplayWidth = rOnCarInputConfig.u32DisplayWidth;
   oConfig.TouchHeight = rOnCarInputConfig.u16TouchHeight;
   oConfig.TouchLayerID = rOnCarInputConfig.u16TouchLayerID;
   oConfig.TouchSurfaceID = rOnCarInputConfig.u16TouchSurfaceID;
   oConfig.TouchWidth = rOnCarInputConfig.u16TouchWidth;
   oConfig.eventType = static_cast <TOUCH_EVENT_TYPE> (rOnCarInputConfig.enOnCarTouchEventType);
   oConfig.screenType = static_cast <TOUCH_SCREEN_TYPE> (rOnCarInputConfig.enOnCarTouchScreenType);
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() :  TouchLayerID:%d ",oConfig.TouchLayerID));
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() :  TouchSurfaceID:%d ",oConfig.TouchSurfaceID));
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() :  TouchHeigh:%d ",oConfig.TouchHeight));
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() :  TouchWidth:%d ",oConfig.TouchWidth));
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() :  Event Type:%d ",oConfig.eventType));
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() :  Screen Type:%d ",oConfig.screenType));

   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   OnCarAPI *poOnCarAPI = NULL;
   if(poOnCarDataIntf)
   {
       m_poOnCarInputEndpoint = poOnCarDataIntf->poGetInputEndpointInstance();
       poOnCarAPI = poOnCarDataIntf->poGetOnCarAPIInstance();
   }//if(poOnCarDataIntf)
   m_oInputEndpointLock.s16Lock();
   if((m_poOnCarInputEndpoint) && (poOnCarAPI))
   {
       m_poOnCarInputEndpoint->setInputConfig(oConfig);
       m_spoInputCbs = std::make_shared<spi_tclOnCarInputCbs>();
       if(nullptr != m_spoInputCbs)
       {
           m_poOnCarInputEndpoint->registerCallbacks(m_spoInputCbs);
       } //if(NULL != m_spoInputCbs)
       m_poOnCarInputEndpoint->init();
       bInitStatus = poOnCarAPI->registerService(m_poOnCarInputEndpoint);
   } //if((m_poOnCarInputEndpoint) && (poOnCarAPI))
   m_oInputEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::bInitializeInputEndpoint() left with result:%d \n",bInitStatus));
  return bInitStatus;
}//!end of bInitializeInputEndpoint()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::vDestroyInputEnpointInstance()
***************************************************************************/
t_Void spi_tclOnCarCmdInput::vDestroyInputEnpointInstance()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vDestroyInputEnpointInstance() entered "));
   m_oInputEndpointLock.s16Lock();
   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   if((poOnCarDataIntf) && (m_poOnCarInputEndpoint))
   {
       poOnCarDataIntf->vDestroyEndpointInstance(m_poOnCarInputEndpoint,e8_ONCAR_OBJECTFACTORY_INPUT_ENDPOINT);
   } //((poOnCarDataIntf) && (m_poOnCarInputEndpoint))
   m_poOnCarInputEndpoint = NULL;
   m_spoInputCbs.reset();
   m_oInputEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vDestroyInputEnpointInstance() left "));
}//!end of vDestroyInputEnpointInstance()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::vUnInitializeInputEndpoint()
***************************************************************************/
t_Void spi_tclOnCarCmdInput::vUnInitializeInputEndpoint()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vUnInitializeInputEndpoint() entered "));
   m_oInputEndpointLock.s16Lock();
   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   if((poOnCarDataIntf) && (m_poOnCarInputEndpoint))
   {
       m_poOnCarInputEndpoint->deinit();
   } //((poOnCarDataIntf) && (m_poOnCarInputEndpoint))
   m_oInputEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vUnInitializeInputEndpoint() left "));
}//!end of vUnInitializeInputEndpoint()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::enGetKeycode()
***************************************************************************/
tenOnCarInputKeyCodes spi_tclOnCarCmdInput::enGetKeyCode(tenKeyCode enSpiKeyCode)
{
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::enGetKeyCode() entered : SpiKeyCode = %d",enSpiKeyCode));
  tenOnCarInputKeyCodes enOnCarKeyCode = e8_ONCAR_INPUT_KEYCODE_INVALID;
  std::map<tenKeyCode,tenOnCarInputKeyCodes>::const_iterator itrKeyCode;

  itrKeyCode = m_keyCodeMap.find(enSpiKeyCode);
  if (m_keyCodeMap.end() != itrKeyCode)
  {
      enOnCarKeyCode = itrKeyCode->second;
  }//if (m_keyCodeMap.end() != itrKeyCode)
  ETG_TRACE_USR2(("[DESC]:spi_tclOnCarCmdInput::enGetKeyCode : Get key code value : Key code = %d ", enOnCarKeyCode));
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::enGetKeyCode() left "));
  return enOnCarKeyCode;
}//!end of enGetKeyCode()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::vSetOnCarCodes()
***************************************************************************/
t_Void spi_tclOnCarCmdInput::vSetOnCarCodes()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vSetOnCarCodes() entered "));
   //! populate mapping between SPI and OnCar key-codes
   for (t_U32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(aKeyCode)) / sizeof(trOnCarKeyCode)); ++u32KeyIndex)
   {
      m_keyCodeMap.insert(std::pair<tenKeyCode, tenOnCarInputKeyCodes>(
        aKeyCode[u32KeyIndex].enKeyCode, aKeyCode[u32KeyIndex].enOnCarInputKeyCode));
   }
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vSetOnCarCodes() left "));
}//!end of vSetOnCarCodes()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdInput::vReportKnobkey()
***************************************************************************/
t_Void spi_tclOnCarCmdInput::vReportKnobkey(t_U32 u32DeviceHandle, t_S32 s32DeltaCnts)
{
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vReportKnobkey() entered : Dev-0x%x DeltaCnts-0x%x",
                  u32DeviceHandle,s32DeltaCnts));
  SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);
  ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdInput::vReportKnobkey() left "));
}//!end of vReportKnobkey()
