/***************************************************************************/
/*!
* \file  spi_tclOnCarCmdBluetooth.cpp
* \brief Interface to interact with OnCar BT Endpoint
****************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Interface to interact with OnCar BT Endpoint
AUTHOR:         jun5kor
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
13.02.2018  | Unmukt Jain           | Initial Version
12.03.2018  | Rishav Sardar         | Bluetooth Endpoint Implementation

\endverbatim
*****************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <algorithm>
#include "SPITypes.h"
#include "OnCarTypes.h"
#include "spi_tclOnCarDataIntf.h"
#include "spi_tclOnCarCmdBluetooth.h"
#include "OnCarAPI.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_ONCARWRAPPER
      #include "trcGenProj/Header/spi_tclOnCarCmdBluetooth.cpp.trc.h"
   #endif
#endif
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/***************************************************************************
*********************************PUBLIC*************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdBluetooth::spi_tclOnCarCmdBluetooth()
***************************************************************************/

spi_tclOnCarCmdBluetooth::spi_tclOnCarCmdBluetooth() : 
    m_poOnCarBluetoothEndpoint(NULL)
{
   ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarCmdBluetooth::spi_tclOnCarCmdBluetooth() entered "));
   ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarCmdBluetooth::spi_tclOnCarCmdBluetooth() left "));
}//!end of spi_tclOnCarCmdBluetooth()

/***************************************************************************
** FUNCTION:  spi_tclOnCarCmdBluetooth::~spi_tclOnCarCmdBluetooth()
***************************************************************************/
spi_tclOnCarCmdBluetooth::~spi_tclOnCarCmdBluetooth()
{
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarCmdBluetooth::~spi_tclOnCarCmdBluetooth() entered "));
   m_oEndpointLock.s16Lock();
   m_spoBluetoothCbs = nullptr;
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarCmdBluetooth::~spi_tclOnCarCmdBluetooth() left "));
}//!end of ~spi_tclOnCarCmdBluetooth()

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint()
***************************************************************************/
t_Bool spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint(const t_String& rfcoszVehicleBTAddress,
        tenOnCarBTPairingMethod enBTPairingMethod)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint() entered : OnCarBTPairingMethod = %d"
                   "VehicleBTAddress = %s",enBTPairingMethod,rfcoszVehicleBTAddress.c_str()));
   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   OnCarAPI *poOnCarAPI = NULL;
   if(poOnCarDataIntf)
   {
       m_poOnCarBluetoothEndpoint = poOnCarDataIntf->poGetBluetoothEndpointInstance();
       poOnCarAPI = poOnCarDataIntf->poGetOnCarAPIInstance();
   }//if(poOnCarDataIntf)

   //! Convert Vehicle BT Address to MAC Address
   t_String szVehMacAddr = szConvertToMacAddress(rfcoszVehicleBTAddress);
   t_Bool bRegSuccess = false;
   
   m_oEndpointLock.s16Lock();
   if ((m_poOnCarBluetoothEndpoint) && (false == szVehMacAddr.empty()) && (poOnCarAPI))
   {
      ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint() : Creating BT Endpoint "));
      SPI_NORMAL_ASSERT(NULL == m_poOnCarBluetoothEndpoint);

      //! Register callbacks, Vehicle BT Address & Supported Pairing method to endpoint
      m_poOnCarBluetoothEndpoint->setHUBluetoothAddress(szVehMacAddr);
      m_poOnCarBluetoothEndpoint->setHUPairingMethod(static_cast<BTPAIRING_METHOD>(enBTPairingMethod));
       m_spoBluetoothCbs = std::make_shared<spi_tclOnCarBluetoothCbs>();
      if(m_spoBluetoothCbs != nullptr)
      {
          m_poOnCarBluetoothEndpoint->registerCallbacks(m_spoBluetoothCbs);
      }//if(m_spoBluetoothCbs != nullptr)
      m_poOnCarBluetoothEndpoint->init();
      bRegSuccess = poOnCarAPI->registerService(m_poOnCarBluetoothEndpoint);
      ETG_TRACE_USR2(("[DESC]:spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint() :"
                      "Bluetooth Endpoint intialization result: "
                      "Registration success %d (for PairMethod = %d, VehMacAddr = %s)",
            ETG_ENUM(BOOL, bRegSuccess),
            ETG_ENUM(ONCAR_BTPAIRING_METHOD, enBTPairingMethod),
            szVehMacAddr.c_str()));
   }//if ((m_poOnCarBluetoothEndpoint) && (false == szVehMacAddr.empty()) && (poOnCarAPI))
   else if (true == szVehMacAddr.empty())
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint() : BT Endpoint not created due to empty Vehicle MAC Address! "));
   }// else if (true == szVehMacAddr.empty())
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint() : Invalid pointer encountered! "));
   }
   m_oEndpointLock.vUnlock();

   return bRegSuccess;
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint() left with bRegSuccess = %d",bRegSuccess));
}//!end of bInitialiseBTEndpoint()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vUninitialiseBTEndpoint()
***************************************************************************/
t_Void spi_tclOnCarCmdBluetooth::vUninitialiseBTEndpoint()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vUninitialiseBTEndpoint() entered "));
   m_oEndpointLock.s16Lock();
   if(m_poOnCarBluetoothEndpoint)
   {
       m_poOnCarBluetoothEndpoint->deinit();
   }//if(m_poOnCarBluetoothEndpoint)
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vUninitialiseBTEndpoint() left "));
}//!end of vUninitialiseBTEndpoint()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vDestroyBtEndpointInstance()
***************************************************************************/
t_Void spi_tclOnCarCmdBluetooth::vDestroyBtEndpointInstance()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vDestroyBtEndpointInstance() entered "));
   m_oEndpointLock.s16Lock();
   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   if((m_poOnCarBluetoothEndpoint) && (poOnCarDataIntf))
   {
       poOnCarDataIntf->vDestroyEndpointInstance(m_poOnCarBluetoothEndpoint,e8_ONCAR_OBJECTFACTORY_BLUETOOTH_ENDPOINT);
       m_poOnCarBluetoothEndpoint = NULL;
   }//if((m_poOnCarBluetoothEndpoint) && (poOnCarDataIntf))
   m_spoBluetoothCbs = nullptr;
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vUninitialiseBTEndpoint() left "));
}//!end of vDestroyBtEndpointInstance()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vSendBTPairingResponse()
***************************************************************************/
t_Void spi_tclOnCarCmdBluetooth::vSendBTPairingResponse(t_Bool bReadyToPair, t_Bool bOnCarDevicePaired,
                                                        tenOnCarBTPairingState enOnCarBTPairingState)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vSendBTPairingResponse() entered"));
   ETG_TRACE_USR2(("[DESC]:spi_tclOnCarCmdBluetooth::vSendBTPairingResponse() : "
         "IsHUReadyToPair = %d, IsOnCarDevicePairedToHU = %d ",
         ETG_ENUM(BOOL, bReadyToPair), ETG_ENUM(BOOL, bOnCarDevicePaired)));
   m_oEndpointLock.s16Lock();
   if (m_poOnCarBluetoothEndpoint)
   {
      if(false == bOnCarDevicePaired)
      {
         if(e8_ONCAR_BTPAIRING_DELAYED_PAIRING == enOnCarBTPairingState)
         {
            m_poOnCarBluetoothEndpoint->btPairingResponse(static_cast<BTPAIRING_STATE> (e8_ONCAR_BTPAIRING_DELAYED_PAIRING));
         }
         else
         {
            m_poOnCarBluetoothEndpoint->btPairingResponse(static_cast<BTPAIRING_STATE> (e8_ONCAR_BTPAIRING_NOT_PAIRED));
         }
      }
      else
      {
         m_poOnCarBluetoothEndpoint->btPairingResponse(static_cast<BTPAIRING_STATE> (e8_ONCAR_BTPAIRING_ALREADY_PAIRED));
      }//if(false == bOnCarDevicePaired)
   }//if (m_poOnCarBluetoothEndpoint)
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vSendBTPairingResponse() left"));
}//!end of vSendBTPairingResponse()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vSendAuthenticationData()
***************************************************************************/
t_Void spi_tclOnCarCmdBluetooth::vSendBTPairingPIN(const t_String& rfcoszBTPairingPIN)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vSendBTPairingPIN() entered : BTPairingPIN = %s ",
                   rfcoszBTPairingPIN.c_str()));
   m_oEndpointLock.s16Lock();
    if (m_poOnCarBluetoothEndpoint)
   {
      m_poOnCarBluetoothEndpoint->sendBTPairingPIN(rfcoszBTPairingPIN);
   }///if (m_poOnCarBluetoothEndpoint)
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::vSendBTPairingPIN() left"));
}//!end of vSendBTPairingPIN()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::szConvertToMacAddress(...)
***************************************************************************/
t_String spi_tclOnCarCmdBluetooth::szConvertToMacAddress(const t_String& rfcoszBTAddress)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarCmdBluetooth::szConvertToMacAddress() entered : BTAddress = %s",
                   rfcoszBTAddress.c_str()));
   //! Initialize string with the BT address
   t_String szMacAddress;

   if (false == rfcoszBTAddress.empty())
   {
      szMacAddress = rfcoszBTAddress.c_str();
      // Insert the ":" charecter in the address string begin from second position and after the two characters.
      //! (Example: If szBTMACAddress is "28E14CDF3072", format string as "28:E1:4C:DF:30:72"
      const t_U8 cou8Init = 2;
      const t_U8 cou8Iterator = 3;
      for(t_U8 u8Index = cou8Init; u8Index < static_cast<t_U8>(szMacAddress.length()) ; u8Index=static_cast<t_U8>(u8Index+cou8Iterator))
      {
         szMacAddress.insert(u8Index,":");
      }
      std::transform(szMacAddress.begin(), szMacAddress.end(), szMacAddress.begin(), ::toupper);
   }//if (false == rfcoszBTAddress.empty())
   ETG_TRACE_USR1(("[DESC]:spi_tclOnCarCmdBluetooth::szConvertToMacAddress() left with szMacAddress = %s",szMacAddress.c_str()));
   return szMacAddress;
}//!end of szConvertToMacAddress()

// <EOF>
