/***********************************************************************/
/*!
* \file  spi_tclOnCarCmdBluetooth.h
* \brief Interface to interact with OnCar BT Endpoint
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Interface to interact with  OnCar BT Endpoint
AUTHOR:         jun5kor
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
13.02.2018  | Unmukt Jain           | Initial Version

\endverbatim
*************************************************************************/

#ifndef _SPI_TCLONCARCMDBLUETOOTH_H_
#define _SPI_TCLONCARCMDBLUETOOTH_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "BaseTypes.h"
#include "SPI_OnCarTypes.h"
#include "spi_tclOnCarBluetoothCbs.h"
#include "Lock.h"

//class OnCarAPI;

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/* Forward declaration */

/******************************************************************************/
/*!
* \class spi_tclOnCarCmdBluetooth
* \brief Interface to interact with Bluetooth Endpoint
*
* It provides an interface to interact with OnCar API.
* It is responsible for creation & initialization of Bluetooth Endpoint.
*******************************************************************************/
class spi_tclOnCarCmdBluetooth
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarCmdBluetooth::spi_tclOnCarCmdBluetooth()
   ***************************************************************************/
   /*!
   * \fn      spi_tclOnCarCmdBluetooth()
   * \brief   Default Constructor
   * \sa      ~spi_tclOnCarCmdBluetooth()
   ***************************************************************************/
    spi_tclOnCarCmdBluetooth();

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarCmdBluetooth::~spi_tclOnCarCmdBluetooth()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclOnCarCmdBluetooth()
   * \brief   Destructor
   * \sa      spi_tclOnCarCmdBluetooth()
   ***************************************************************************/
   ~spi_tclOnCarCmdBluetooth();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclOnCarCmdBluetooth::bInitialiseBTEndpoint()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bInitialiseBTEndpoint()
   * \brief   Creates and initialises an instance of BT Endpoint
   * \retval  t_Bool  :  True if the BT Endpoint is initialised, else False
   * \sa      vUninitialiseBTEndpoint()
   ***************************************************************************/
   t_Bool bInitialiseBTEndpoint(const t_String& rfcoszVehicleBTAddress,
           tenOnCarBTPairingMethod enBTPairingMethod);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vUninitialiseBTEndpoint()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUninitialiseBTEndpoint()
   * \brief   Uninitialises an instance of BT Endpoint
   * \retval  t_Void
   * \sa      bInitialiseBTEndpoint()
   ***************************************************************************/
   t_Void vUninitialiseBTEndpoint();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vSendBTPairingResponse()
   ***************************************************************************/
   /*!
   * \fn      vSendBTPairingResponse(t_Bool bReadyToPair, t_Bool bOnCarDevicePaired)
   * \brief   Interface to send Pairing response to BT Endpoint
   * \param   bReadyToPair [IN] : True - if HU is ready to pair AAP device, else False.
   * \param   bAAPDevicePaired [IN] : True - if HU is paired with AAP device, else False.
   * \sa      None
   ***************************************************************************/
   t_Void vSendBTPairingResponse(t_Bool bReadyToPair, t_Bool bOnCarDevicePaired,
                                 tenOnCarBTPairingState enOnCarBTPairingState);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vSendAuthenticationData()
   ***************************************************************************/
   /*!
   * \fn      vSendAuthenticationData(const t_String& rfcoszAuthData)
   * \brief   Interface to send pairing authentication data to BT Endpoint
   * \param   rfcoszAuthData [IN] : Pairing authentication data
   * \sa      None
   ***************************************************************************/
   t_Void vSendBTPairingPIN(const t_String& rfcoszBTPairingPIN);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::vDestroyBtEndpointInstance()
   ***************************************************************************/
   /*!
   * \fn      t_Void vDestroyBtEndpointInstance()
   * \brief   Destroys an instance of BT Endpoint
   * \retval  t_Void
   * \sa      None
   ***************************************************************************/
   t_Void vDestroyBtEndpointInstance();

private:
   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION: spi_tclOnCarCmdBluetooth(const spi_tclOnCarCmdBluetooth &rfcoCmdBluetooth)
    ***************************************************************************/
   /*!
    * \fn      spi_tclOnCarCmdBluetooth(const spi_tclOnCarCmdBluetooth &rfcoCmdBluetooth)
    * \brief   Copy constructor not implemented hence made private
    **************************************************************************/
   spi_tclOnCarCmdBluetooth(const spi_tclOnCarCmdBluetooth &rfcoCmdBluetooth);

   /***************************************************************************
    ** FUNCTION: const spi_tclOnCarCmdBluetooth & operator=(
    **                      const spi_tclOnCarCmdBluetooth &rfcoCmdBluetooth);
    ***************************************************************************/
   /*!
    * \fn      const spi_tclOnCarCmdBluetooth & operator=(
    *             const spi_tclOnCarCmdBluetooth &rfcoCmdBluetooth);
    * \brief   assignment operator not implemented hence made private
    **************************************************************************/
   const spi_tclOnCarCmdBluetooth & operator=(
            const spi_tclOnCarCmdBluetooth &rfcoCmdBluetooth);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarCmdBluetooth::szConvertToMacAddress(...)
   ***************************************************************************/
   /*!
   * \fn      szConvertToMacAddress(const t_String& rfcoszBTAddress)
   * \brief   Converts a BT address to MAC address
   * \param   [IN] rfcoszBTAddress: BT address of a device
   * \retval  t_String: MAC Address
   * \sa      None
   ***************************************************************************/
   t_String szConvertToMacAddress(const t_String& rfcoszBTAddress);

   //! Bluetooth Endpoint pointer
   OnCarBluetoothEndpoint* m_poOnCarBluetoothEndpoint;

   //! shared pointer for OnCar BT callbacks
   std::shared_ptr<IOnCarBluetoothCallbacks> m_spoBluetoothCbs;

   //! Lock object for BT endpoint
   Lock  m_oEndpointLock;

};

#endif // _SPI_TCLONCARCMDBLUETOOTH_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
