/*!
 *******************************************************************************
 * \file              spi_tclOnCarCmdAudio.cpp
 * \brief             Audio Endpoint Wrapper for OnCar
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Endpoint Wrapper for OnCar
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 19.04.2018 |  Rishav Sardar       | Initial Version

 \endverbatim
 ******************************************************************************/
/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclOnCarCmdAudio.h"
#include "spi_tclOnCarDataIntf.h"
#include "OnCarAPI.h"
#include<vector>

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclOnCarCmdAudio.cpp.trc.h"
#endif
#endif

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarCmdAudio::spi_tclOnCarCmdAudio();
 ***************************************************************************/
spi_tclOnCarCmdAudio::spi_tclOnCarCmdAudio():m_poOnCarAudioEndpoint(NULL)
{
   ETG_TRACE_USR1(("[Constructor]:spi_tclOnCarCmdAudio::spi_tclOnCarCmdAudio Entered"));
}

/***************************************************************************
 ** FUNCTION:  virtual spi_tclOnCarCmdAudio::~spi_tclOnCarCmdAudio()
 ***************************************************************************/
spi_tclOnCarCmdAudio::~spi_tclOnCarCmdAudio()
{
   ETG_TRACE_USR1(("[Destructor]:spi_tclOnCarCmdAudio::~spi_tclOnCarCmdAudio() entered "));
   m_oEndpointLock.s16Lock();
   m_spoAudioCbs = nullptr;
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("[Destructor]:spi_tclOnCarCmdAudio::~spi_tclOnCarCmdAudio() left "));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdAudio::vSetAudioConfig()
***************************************************************************/
t_Void spi_tclOnCarCmdAudio::vSetAudioConfig(const trOnCarAudioConfig& corfrOnCarAudioConfig )
{
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vSetAudioConfig entered "));
   if(NULL != m_poOnCarAudioEndpoint)
   {
      AudioConfig oAudioConfig;
      oAudioConfig.threadPriority = corfrOnCarAudioConfig.u16threadPriority;
      oAudioConfig.audioPipeline = corfrOnCarAudioConfig.szAudioPipeline;
      oAudioConfig.type = static_cast <AUDIOSTREAM_TYPE>(corfrOnCarAudioConfig.enOnCarAudStreamtype);

      ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vSetAudioConfig:threadPriority-%d",oAudioConfig.threadPriority));
      ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vSetAudioConfig:audioPipeline-%s",oAudioConfig.audioPipeline.c_str()));
      ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vSetAudioConfig:type-%d",oAudioConfig.type));

      m_poOnCarAudioEndpoint->setAudioConfig(oAudioConfig);
   }//if (NULL != m_poOnCarVideoEndpoint)
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclOnCarCmdAudio::vSetAudioConfig- Audio sink is NULL"));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarCmdAudio::vsetAudioFocus
 ***************************************************************************/
t_Void spi_tclOnCarCmdAudio::vSetAudioFocus(const tenOnCarAudioStreamType enStreamType, const tenOnCarDeviceAudioFocusState enAudioFocusState)
{
    ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vsetAudioFocus Entered stream: %d, dev audio focus: %d", 
              ETG_ENUM(ONCAR_AUDSTREAM_TYPE, enStreamType), ETG_ENUM(ONCAR_DEVICE_AUDIOFOCUS, enAudioFocusState)));

   if(NULL != m_poOnCarAudioEndpoint)
   {
      AUDIOSTREAM_TYPE streamtype = static_cast <AUDIOSTREAM_TYPE>(enStreamType);
      AUDIOFOCUS_STATE state = static_cast <AUDIOFOCUS_STATE>(enAudioFocusState);
      m_poOnCarAudioEndpoint->setAudioFocus(streamtype,state);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarCmdAudio::bInitialize()
***************************************************************************/
t_Bool spi_tclOnCarCmdAudio::bInitialize(const tvectrOnCarAudioConfig& covtrOnCarAudioConfig)
{
   m_oEndpointLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::bInitialize() entered"));
  
   t_Bool bRegSuccess = false;
   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   OnCarAPI *poOnCarAPI = NULL;
   if(poOnCarDataIntf)
   {
      m_poOnCarAudioEndpoint = poOnCarDataIntf->poGetAudioEndpointInstance();
      poOnCarAPI = poOnCarDataIntf->poGetOnCarAPIInstance();
   }
   if((m_poOnCarAudioEndpoint)&& (poOnCarAPI))
   {
       std::vector<trOnCarAudioConfig>::const_iterator itrOnCarAudioConfig;
       for (itrOnCarAudioConfig = covtrOnCarAudioConfig.begin(); itrOnCarAudioConfig!= covtrOnCarAudioConfig.end(); ++itrOnCarAudioConfig)
       {
           vSetAudioConfig(*itrOnCarAudioConfig);
       }
       m_spoAudioCbs = std::make_shared<spi_tclOnCarAudioCbs>();
       if(m_spoAudioCbs)
       {
          m_poOnCarAudioEndpoint->registerCallbacks(m_spoAudioCbs);
          m_poOnCarAudioEndpoint->init();
          bRegSuccess = poOnCarAPI->registerService(m_poOnCarAudioEndpoint);
          ETG_TRACE_USR2(("[DESC]:Audio Endpoint intialization result: "
                "Registration success %d",ETG_ENUM(BOOL, bRegSuccess)));
       }
   }// if (NULL == m_poOnCarAudioEndpoint)
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::bInitialize() returned with bRetval=%d",bRegSuccess));
   return bRegSuccess;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdAudio::vUninitialize()
***************************************************************************/
t_Void spi_tclOnCarCmdAudio::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vUninitialize() entered "));
   m_oEndpointLock.s16Lock();
   if(NULL != m_poOnCarAudioEndpoint)
   {
      m_poOnCarAudioEndpoint->deinit();
   }
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vUninitialize() left "));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdAudio::vDestroyAudioEnpointInstance()
***************************************************************************/
t_Void spi_tclOnCarCmdAudio::vDestroyAudioEndpointInstance()
{
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vDestroyAudioEnpointInstance() entered "));
   spi_tclOnCarDataIntf *poOnCarDataIntf = spi_tclOnCarDataIntf::getInstance();
   m_oEndpointLock.s16Lock();
   if(NULL != m_poOnCarAudioEndpoint && NULL != poOnCarDataIntf)
   {
      poOnCarDataIntf->vDestroyEndpointInstance(m_poOnCarAudioEndpoint,e8_ONCAR_OBJECTFACTORY_AUDIO_ENDPOINT);
      m_poOnCarAudioEndpoint = NULL;
   }
   m_spoAudioCbs = nullptr;
   m_oEndpointLock.vUnlock();
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vDestroyAudioEnpointInstance() left "));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdAudio::vAcknowledgeStartAudioRequest
**                                        (tenOnCarAudioStreamType enOnCarAudStreamType)
***************************************************************************/
t_Void spi_tclOnCarCmdAudio::vAcknowledgeStartAudioRequest(tenOnCarAudioStreamType enOnCarAudStreamType)
{
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::acknowledgeStartAudioRequest entered "));
   m_oEndpointLock.s16Lock();
   if(NULL != m_poOnCarAudioEndpoint)
   {
      AUDIOSTREAM_TYPE type = static_cast<AUDIOSTREAM_TYPE> (enOnCarAudStreamType);
      m_poOnCarAudioEndpoint->acknowledgeStartAudioRequest(type);
   }
   m_oEndpointLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdAudio::vAcknowledgeStopAudioRequest
**                                         (tenOnCarAudioStreamType enOnCarAudStreamType)
***************************************************************************/
t_Void spi_tclOnCarCmdAudio::vAcknowledgeStopAudioRequest(tenOnCarAudioStreamType enOnCarAudStreamType)
{
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::acknowledgeStopAudioRequest entered "));
   m_oEndpointLock.s16Lock();
   if(NULL != m_poOnCarAudioEndpoint)
   {
      AUDIOSTREAM_TYPE type = static_cast<AUDIOSTREAM_TYPE> (enOnCarAudStreamType);
      m_poOnCarAudioEndpoint->acknowledgeStopAudioRequest(type);
   }
   m_oEndpointLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarCmdAudio::vStopALSAStreaming
**                                         (const tenOnCarAudioStreamType enStreamType)
***************************************************************************/
t_Void spi_tclOnCarCmdAudio::vStopALSAStreaming(const tenOnCarAudioStreamType enStreamType)
{
   ETG_TRACE_USR1(("spi_tclOnCarCmdAudio::vStopALSAStreaming Entered stream: %d", 
              ETG_ENUM(ONCAR_AUDSTREAM_TYPE, enStreamType)));
   
   if(NULL != m_poOnCarAudioEndpoint)
   {
      AUDIOSTREAM_TYPE streamtype = static_cast <AUDIOSTREAM_TYPE>(enStreamType);
      m_poOnCarAudioEndpoint->stopALSAStreaming(streamtype);
   }
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>
